        SUBROUTINE FILTER(X,V,N,M,ISW,P1,P2,P3,P4,WORK,IVAR,IFAIL)
*
*
*  Subroutine to filter either by a gaussian of standard deviation
*  P1 (ISW=1), by a Wiener smoothing filter (ISW=2), by a high pass
*  filter (ISW=3), by both a high
*  pass filter and by the Wiener filter (ISW=4) or by a low-pass filter
*  with a cut-off at P1 rolling off at 40dB/decade. For the Wiener filter
*  the original series is assumed to have a gaussian a.c.f.
*  of standard deviation P1, and the ratio of signal power
*  (at zero wavenumber) to white noise is P2
*  P2 = (SUM of signal**2)/(SUM of noise**2)
*
* The high pass filter is a cosine bell starting at zero at P3 and
* reaching unity at P4
*
*  IFAIL = 0 all is ok, IFAIL .NE. 0, filtering has failed
*
*  In all cases the spectrum is apodised for the top and bottom 10%
*  cosine bell function
*
*********************************************************************
*
* Inputs:
*         REAL*4 X(N) array to be filtered
*         REAL*4 V(N) variance array to be filtered
*         INTEGER*4 N    number of elements in array
*         INTEGER*4 M    number in array before padding with zeroes.
*         INTEGER*4 ISW  control integer. See above.
*         REAL*4 P1   standard deviation of gaussian filter (ISW=1)
*                     or of acf of spectrum for Wiener filter (ISW=2)
*                     or cut-off wavenumber (ISW=5)
*         REAL*4 P2   Ratio of signal power to white noise (ISW=2,4)
*         REAL*4 P3,P4 Lower and upper wavenumbers for high pass filter
*                      a cosine bell is used in between (ISW=3,4)
*         REAL*8 WORK(2*N) workspace array. It need only be N if the
*                variances are not being filtered
*         INTEGER*4 IVAR = 0 X filtered only
*                        = 1 X and V filtered
*
* Outputs:
*         REAL*4 X(N) filtered array, V(N) filtered variances
*         INTEGER*4 IFAIL error return
*
**********************************************************************
*
*  History:
*  First written by CRJ 28/03/85
*  Modified by TRM 22/01/86
*
*************************************************
*
        REAL*4 X(N),V(N),P1,P2,P3,P4,ALPHA,EXPTRM,SMTH
        REAL*4 F,F1,F2,F3,NORM,SWITCH
        REAL*8 WORK(1), XYZ, XYZ1
        INTEGER*4 END,N,ISW,M,IVAR
*
*  Apodize the top and bottom 10% of the spectrum
*
        PI = 4.*ATAN(1.)
        F  = 0.1*REAL(M)
        F1 = PI/F
        F2 = 0.5*F
        F3 = 0.95*REAL(M)
*
        DO I=1,INT(F)+1
          X(I) = X(I)*0.5*(1.+SIN(F1*(REAL(I-1)-F2)))
        END DO
        DO I=M-INT(F),M
          X(I) = X(I)*0.5*(1.+SIN(F1*(F3-REAL(I))))
        END DO
        DO I = 1, N
          WORK(I) = DBLE(X(I))
        END DO
        IF(IVAR.EQ.1) THEN
          DO I=1,INT(F)+1
            V(I) = V(I)*0.25*(1.+SIN(F1*(REAL(I-1)-F2)))**2
          END DO
          DO I=M-INT(F),M
            V(I) = V(I)*0.25*(1.+SIN(F1*(F3-REAL(I))))**2
          END DO
*
* Set variance filter to unity to start with
*
          DO I = 1, N/2+1
            WORK(N+I) = 1.D0
          END DO
          DO I = N/2+2, N
            WORK(N+I) = 0.D0
          END DO
        END IF
*
* FFT
*
        IFAIL = 1
        CALL C06EAF(WORK,N,IFAIL)
        IF(IFAIL.NE.0) THEN
          PRINT *,'NAG C06EAF failed. IFAIL = ',IFAIL
          RETURN
        END IF
*
        END = N + 2
        ALPHA = 2.*PI/REAL(N)
        EXPTRM = ALPHA*P1
        IF(ISW.EQ.1) THEN
*
* Gaussian filter
*
          DO I = 2, N/2
            SMTH = EXPTRM*REAL(I-1)
            SMTH = EXP(-0.5*SMTH*SMTH)
            WORK(I) = WORK(I)*DBLE(SMTH)
            WORK(END-I) = WORK(END-I)*DBLE(SMTH)
            IF(IVAR.EQ.1) THEN
              J = N + I
              WORK(J) = DBLE(SMTH)*WORK(J)
            END IF
          END DO
          SMTH = EXPTRM*REAL(N/2)
          SMTH = EXP(-0.5*SMTH*SMTH)
          WORK(N/2+1) = WORK(N/2+1)*DBLE(SMTH)
          IF(IVAR.EQ.1) WORK(N+N/2+1) = DBLE(SMTH)*WORK(N+N/2+1)
        ELSE IF(ISW.EQ.2 .OR. ISW.EQ.4) THEN
*
* Wiener filter
*
          NORM = (P2 + 1.)/P2
          SWITCH = SQRT(-ALOG(1.E-4/P2))
          DO I = 2, N/2
            SMTH = EXPTRM*REAL(I-1)
            IF(SMTH.GT.SWITCH)THEN
              SMTH = 1.E-4
            ELSE
              SMTH = EXP(-SMTH*SMTH)
              SMTH = NORM/(1.+1./(P2*SMTH))
            END IF
            WORK(I) = WORK(I)*DBLE(SMTH)
            WORK(END-I) = WORK(END-I)*DBLE(SMTH)
            IF(IVAR.EQ.1) THEN
              J = N + I
              WORK(J) = DBLE(SMTH)*WORK(J)
            END IF
          END DO
          SMTH = EXPTRM*REAL(N/2)
          IF(SMTH.GT.SWITCH)THEN
            SMTH = 1.E-4
          ELSE
            SMTH = EXP(-SMTH*SMTH)
            SMTH = NORM/(1.+1./(P2*SMTH))
          END IF
          WORK(N/2+1) = WORK(N/2+1)*DBLE(SMTH)
          IF(IVAR.EQ.1) WORK(N+N/2+1) = DBLE(SMTH)*WORK(N+N/2+1)
        ELSE IF(ISW.EQ.5) THEN
*
* 40dB/decade filter
*
          DO I = 2, N/2
            SMTH = (REAL(I-1)/P1)**4
            SMTH = 1./(1.+SMTH)
            WORK(I) = WORK(I)*DBLE(SMTH)
            WORK(END-I) = WORK(END-I)*DBLE(SMTH)
            IF(IVAR.EQ.1) THEN
              J = N + I
              WORK(J) = DBLE(SMTH)*WORK(J)
            END IF
          END DO
          SMTH = (REAL(N/2)/P1)**4
          SMTH = 1./(1.+SMTH)
          WORK(N/2+1) = WORK(N/2+1)*DBLE(SMTH)
          IF(IVAR.EQ.1) WORK(N+N/2+1) = DBLE(SMTH)*WORK(N+N/2+1)
        END IF
        IF(ISW.EQ.3 .OR. ISW.EQ.4) THEN
*
* High pass cosine bell filter
*
          IF(P4.LE.P3 .OR. P3.LT.0. .OR. P4.GT.REAL(N/2)) THEN
            IFAIL = 1
            RETURN
          END IF
          F = PI/(P4-P3)
          WORK(1) = 0.D0
          DO I = 2, INT(P4) + 1
            XX = REAL(I-1)
            IF(XX.LT.P3) THEN
              SMTH = 0.
            ELSE IF(XX.GT.P4) THEN
              SMTH = 1.
            ELSE
              SMTH = 0.5 + 0.5*COSD(F*(XX-P4))
            END IF
            WORK(I)     = DBLE(SMTH)*WORK(I)
            WORK(END-I) = DBLE(SMTH)*WORK(END-I)
            IF(IVAR.EQ.1) THEN
              J = N + I
              WORK(J) = DBLE(SMTH)*WORK(J)
            END IF
          END DO
        END IF
*
* Retrieve spectrum
*
        IFAIL = 1
        CALL C06GBF(WORK,N,IFAIL)
        IF(IFAIL.NE.0) THEN
          PRINT *,'NAG C06GBF failed. IFAIL = ',IFAIL
          RETURN
        END IF
        IFAIL = 1
        CALL C06EBF(WORK,N,IFAIL)
        IF(IFAIL.NE.0) THEN
          PRINT *,'NAG C06EBF failed. IFAIL = ',IFAIL
          RETURN
        END IF
        DO I = 1, N
          X(I) = SNGL(WORK(I))
        END DO
*
        IF(IVAR.EQ.1) THEN
*
* FFT of the filter applied
*
          IFAIL = 1
          CALL C06EBF(WORK(N+1),N,IFAIL)
          IF(IFAIL.NE.0) THEN
            PRINT *,'NAG C06EBF failed. IFAIL = ',IFAIL
            RETURN
          END IF
*
* Take modulus squared
*
          XYZ1 = 1.D0/SQRT(DBLE(N))
          DO I = N+1, 2*N
            XYZ = WORK(I)
            WORK(I) = XYZ1*XYZ*XYZ
          END DO
*
* Invert to get filter to apply to
* the variance array transform. The imaginary parts
* should be zero since the result is the autocorrelation
* of the original real filter
*
          IFAIL = 1
          CALL C06EAF(WORK(N+1),N,IFAIL)
          IF(IFAIL.NE.0) THEN
            PRINT *,'NAG C06EAF failed. IFAIL = ',IFAIL
            RETURN
          END IF
*
          DO I = 1, N
            WORK(I) = DBLE(V(I))
          END DO
*
* FFT of variances
*
          IFAIL = 1
          CALL C06EAF(WORK,N,IFAIL)
          IF(IFAIL.NE.0) THEN
            PRINT *,'NAG C06EAF failed. IFAIL = ',IFAIL
            RETURN
          END IF
*
* Multiply filter and variance transform
*
          WORK(1) = WORK(1)*WORK(N+1)
          L = N/2 + 1
          IF(MOD(N,2).EQ.0) THEN
            WORK(L) = WORK(L)*WORK(N+L)
          ELSE
            WORK(L)     = WORK(L)*WORK(N+L)
            WORK(END-L) = WORK(END-L)*WORK(N+L)
          END IF
          DO I = 2, N/2
            J = END - I
            XYZ = WORK(N+I)
            WORK(I) = WORK(I)*XYZ
            WORK(J) = WORK(J)*XYZ
          END DO
          IFAIL = 1
          CALL C06GBF(WORK,N,IFAIL)
          IF(IFAIL.NE.0) THEN
            PRINT *,'NAG C06GBF failed. IFAIL = ',IFAIL
            RETURN
          END IF
          IFAIL = 1
          CALL C06EBF(WORK,N,IFAIL)
          IF(IFAIL.NE.0) THEN
            PRINT *,'NAG C06EBF failed. IFAIL = ',IFAIL
            RETURN
          END IF
          DO I = 1, N
            V(I) = SNGL(WORK(I))
          END DO
        END IF
*
        RETURN
        END


        SUBROUTINE VWFILTER( RAWDATA, NPIX, NFILT, FILT, DATASIG )
*
*  Variable width linear filter for 1-dimensional data
*
*  Inputs:
*       RAWDATA = UNFILTERED DATA ARRAY
*       NPIX    = NUMBER OF PIXELS
*       NFILT   = MAXIMUM LENGTH OF FILTER (PIXELS)
*  Outputs:
*       DATA    = FILTERED DATA
*       DATASIG = UNCERTAINTY IN FILTERED DATA (1-SIGMA)
*
*  AUG 1984 BY KEITH HORNE AT IOA
*
        PARAMETER (MAXFILT = 999) ! (must be an odd integer)
        PARAMETER (THRESH = 2.5)  ! Threshold for filter width reduction
        REAL*4 RAWDATA(1), FILT(1), DATASIG(1)
        REAL*8 SUM

        PI = ATAN2(1.,1.)*4.
*
*  INTRODUCTION
*
        PRINT *,'Variable width mean filtering'
*
*  Test inputs
*
      IF( NPIX.LE.0 ) THEN
        PRINT *,'** INVALID NUMBER OF PIXELS.', NPIX
  999   PRINT *,'** ''VWFILTER'' ABORTED.'
        RETURN
      ELSE IF( NFILT.LE.0 ) THEN
        PRINT *,'** INVALID FILTER WIDTH.', NFILT
        GOTO 999
      END IF
*
*  Determine largest window half-width to be examined
*
        NHALF= (NFILT-1)/2
        LONG = MIN( MAXFILT, 2*NHALF + 1 )
        LONGHALF = (LONG-1)/2
        IF( NFILT.NE.LONG ) PRINT *,'** FILTER WIDTH', NFILT,
     *  ' REDUCED TO', LONG,' PIXELS.'
*
        PRINT '(A, I7, A, I5, A, F6.2 )',
     *  ' Data pixels =', NPIX,
     *  ' Window width=', LONG,
     *  ' Sigma clip=', THRESH
        PRINT *,' '      ! This line over-written within the loop
        IFILTLO = LONG
        IFILTHI = 0
        FILTSUM = 0.
*
*  Loop through the data pixels
*
        NREPORT = MAX(1, NPIX/20 )
      DO 200 IPIX = 1,NPIX
*
*  Estimate sigma from data fluctuations.
*
        NSUM = 0
        SUM = 0.D0
      DO I = MAX( IPIX-LONGHALF, 1 ), MIN( IPIX+LONGHALF, NPIX )
        IM = MAX( I-1, 1)
        IP = MIN( I+1, NPIX)
        NSUM = NSUM + 1
        SUM = SUM + ABS( RAWDATA(I) - 0.5*( RAWDATA(IM) + RAWDATA(IP)) )
      END DO
        XSIG = SUM/NSUM
*
*  Compute SIGMAX, approximately the upper THRESH-sigma level
*  for the largest sample in a sequence of NPIX random variables,
*  boosted to correct for uncertainty due to use of only NSUM
*  samples to estimate the value of XSIG.
*
        ROOT = SQRT( FLOAT(NSUM) )
        SIGMAX = THRESH * (ROOT+1.)/ROOT + ALOG( FLOAT(NPIX) )/PI
*
*  Loop thru possible window halfwidths (largest to smallest)
*
      DO IHALF = LONGHALF, 0, -1
*
*  Loop thru possible window offsets
*
        ioffset = 0
*
*  Pixel limits for this data segment
*
        I1 = MAX( IPIX + IOFFSET - IHALF, 1 )
        I2 = MIN( IPIX + IOFFSET + IHALF, NPIX )
*
*  Fit sloped line to data segment
*
        NSUM = 0
        SUMY = 0.D0
      DO I = I1, I2
        NSUM = NSUM + 1
        SUMY = SUMY + RAWDATA(I)
      END DO
        XBAR = SUMY / NSUM
        SLOPE = 0.
        ROOT = SQRT( FLOAT(NSUM) )
*
*  Scan data segment for significant outliers
*
      IF( XSIG.GT.0. ) THEN
      DO I = I1, I2
        PREDATA = XBAR + SLOPE * (I-IPIX)
        SIGMA = (RAWDATA(I) - PREDATA ) / XSIG
        IF( ABS(SIGMA) .GT. SIGMAX ) GOTO 100   ! Outlier found
      END DO
      END IF
*
        GOTO 150
*
*  Try next window offset
*
  100 CONTINUE
*
*  Try next shorter window
*
      END DO
*
*  Adopt current mean value
*
150   FILT(IPIX) = XBAR
        DATASIG(IPIX) = XSIG/ ROOT
*
*  Report progress on terminal
*
        IFILTLO = MIN(IFILTLO, NSUM )
        IFILTHI = MAX(IFILTHI, NSUM )
        FILTSUM = FILTSUM + NSUM
      IF( MOD(IPIX,NREPORT).EQ.0 ) THEN
        PRINT '( A, I6, A, I4, A, F6.1, A, I4 )',
     *  '+Thru pixel', IPIX,
     *  ' Filter pixel width Min=', IFILTLO,
     *  ' Mean=', FILTSUM/IPIX,
     *  ' Max=', IFILTHI
      END IF
*
*  End of loop thru pixels
*
  200 END DO
*
        RETURN
        END


