        DOUBLE PRECISION FUNCTION BRENT(AX, BX, CX, F, TOL, XMIN)

* Given a function F, and given a bracketing triplet of abscissas AX,BX,CX
* such that BX is between AX and CX, and F(BX) is less than both F(AX)
* and F(CX), this routine isolates the minimum to a fractional precision
* of about TOL using Brent's method. The abscissa of the minimum is returned
* as XMIN, and the minimum function value is returned as BRENT, the returned
* function value

        IMPLICIT  NONE
        INTEGER ITMAX, ITER
        DOUBLE PRECISION CGOLD
        DOUBLE PRECISION TOL, XMIN, W, X, E, FX, FV, FW, FU, U
        DOUBLE PRECISION XM, TOL1, TOL2, R, Q, P, A, D, ETEMP
        DOUBLE PRECISION ZEPS, AX, CX, B, V, BX, F

* Maximum allowed number of iterations, golden ratio and a small
* number to prevent trying for a fractional accuracy in a minimum
* which is zero

        PARAMETER (ITMAX=100, CGOLD=.3819660,ZEPS=1.0D-10)

        A = MIN(AX, CX)
        B = MAX(AX, CX)
        V = BX
        W = V
        X = V
        E = 0.
        FX = F(X)
        FV = FX
        FW = FX
        DO ITER = 1, ITMAX
          XM = 0.5*(A+B)
          TOL1 = TOL*ABS(X)+ZEPS
          TOL2 = 2.*TOL1
          IF(ABS(X-XM).LE.(TOL2-.5*(B-A))) GOTO 3
          IF(ABS(E).GT.TOL1) THEN
            R = (X-W)*(FX-FV)
            Q = (X-V)*(FX-FW)
            P = (X-V)*Q - (X-W)*R
            Q = 2.*(Q-R)
            IF(Q.GT.0.) P = - P
            Q = ABS(Q)
            ETEMP = E
            E = D
            IF(ABS(P).GE.ABS(.5*Q*ETEMP).OR.P.LE.Q*(A-X).OR.
     &          P.GE.Q*(B-X)) GOTO 1

* The above conditions determine the acceptability of the parabolic
* fit. Here it is OK.
            D = P/Q
            U = X+D
            IF(U-A.LT.TOL2 .OR. B-U.LT.TOL2) D=SIGN(TOL1, XM-X)
            GOTO 2
          END IF
1         IF(X.GE.XM) THEN
            E=A-X
          ELSE
            E=B-X
          END IF
          D = CGOLD*E
2         IF(ABS(D).GE.TOL1) THEN
            U = X+D
          ELSE
            U=X+SIGN(TOL1,D)
          END IF
          FU = F(U)
          IF(FU.LE.FX) THEN
            IF(U.GE.X) THEN
              A=X
            ELSE
              B=X
            END IF
            V=W
            FV=FW
            W=X
            FW=FX
            X=U
            FX=FU
          ELSE
            IF(U.LT.X) THEN
              A=U
            ELSE
              B=U
            END IF
            IF(FU.LE.FW .OR. W.EQ.X) THEN
              V=W
              FV=FW
              W=U
              FW=FU
            ELSE IF(FU.LE.FV .OR. V.EQ.X .OR. V.EQ.W) THEN
              V = U
              FV = FU
            END IF
          END IF
        END DO
        WRITE(*,*) 'Brent exceeded maximum iteration'
3       XMIN = X
        BRENT = FX
        RETURN
        END

        DOUBLE PRECISION FUNCTION DBRENT(AX, BX, CX, F, DF, TOL, XMIN)

* Given a function F, and its derivative function DF, and given a bracketing
* triplet of abscissas AX,BX,CX such that BX is between AX and CX, and F(BX)
* is less than both F(AX) and F(CX), this routine isolates the minimum to a
* fractional precision of about TOL using a modification of Brent's method
* that uses derivatives. The abscissa of the minimum is returned as XMIN, and
* the minimum function value is returned as DBRENT, the returned function value

        IMPLICIT NONE
        INTEGER ITMAX, ITER
        DOUBLE PRECISION ZEPS, A, B, AX, BX, CX, V, W, X
        DOUBLE PRECISION E, FX, FV, FW, DX, DV, DF, DW, XM
        DOUBLE PRECISION TOL, TOL1, TOL2, D1, D2, U1, U2
        DOUBLE PRECISION OLDE, D, U, FU, DU, F, XMIN
        PARAMETER (ITMAX=100,ZEPS=1.0D-10)
        LOGICAL OK1, OK2

        A = MIN(AX, CX)
        B = MAX(AX, CX)
        V = BX
        W = V
        X = V
        E = 0.
        FX = F(X)
        FV = FX
        FW = FX
        DX = DF(X)
        DV = DX
        DW = DX
        DO ITER = 1, ITMAX
          XM = 0.5*(A+B)
          TOL1 = TOL*ABS(X)+ZEPS
          TOL2 = 2.*TOL1
          IF(ABS(X-XM).LE.(TOL2-.5*(B-A))) GOTO 3
          IF(ABS(E).GT.TOL1) THEN
            D1=2.*(B-A)
            D2=D1
            IF(DW.NE.DX) D1=(W-X)*DX/(DX-DW)
            IF(DV.NE.DX) D2=(V-X)*DX/(DX-DV)
            U1=X+D1
            U2=X+D2
            OK1=((A-U1)*(U1-B).GT.0.) .AND.(DX*D1.LE.0.)
            OK2=((A-U2)*(U2-B).GT.0.) .AND.(DX*D2.LE.0.)
            OLDE=E
            E=D
            IF(.NOT.(OK1.OR.OK2)) THEN
              GOTO 1
            ELSE IF(OK1.AND.OK2) THEN
              IF(ABS(D1).LT.ABS(D2)) THEN
                D = D1
              ELSE
                D=D2
              END IF
            ELSE IF(OK1) THEN
              D=D1
            ELSE
              D=D2
            END IF
            IF(ABS(D).GT.ABS(0.5*OLDE)) GOTO 1
            U = X+D
            IF(U-A.LT.TOL2 .OR. B-U.LT.TOL2) D=SIGN(TOL1,XM-X)
            GOTO 2
          END IF
1         IF(DX.GE.0.) THEN
            E=A-X
          ELSE
            E=B-X
          END IF
          D = 0.5*E
2         IF(ABS(D).GE.TOL1) THEN
            U = X+D
            FU=F(U)
          ELSE
            U=X+SIGN(TOL1,D)
            FU=F(U)
            IF(FU.GT.FX) GOTO 3
          END IF
          DU = DF(U)
          IF(FU.LE.FX) THEN
            IF(U.GE.X) THEN
              A=X
            ELSE
              B=X
            END IF
            V=W
            FV=FW
            DV=DW
            W=X
            FW=FX
            DW=DX
            X=U
            FX=FU
            DX=DU
          ELSE
            IF(U.LT.X) THEN
              A=U
            ELSE
              B=U
            END IF
            IF(FU.LE.FW .OR. W.EQ.X) THEN
              V=W
              FV=FW
              DV=DW
              W=U
              FW=FU
              DW=DU
            ELSE IF(FU.LE.FV .OR. V.EQ.X .OR. V.EQ.W) THEN
              V=U
              FV=FU
              DV=DU
            END IF
          END IF
        END DO
        WRITE(*,*) 'DBRENT exceeded maximum iteration'
3       XMIN = X
        DBRENT = FX
        RETURN
        END

      DOUBLE PRECISION FUNCTION ZBRENT(FUNC,X1,X2,TOL)
*
      IMPLICIT NONE 
      INTEGER ITMAX, ITER
      DOUBLE PRECISION X1, X2, EPS, TOL, FUNC, A, B
      DOUBLE PRECISION FA, FB, FC, C, D, E, TOL1
      DOUBLE PRECISION XM, P, Q, R, S
      PARAMETER (ITMAX=100,EPS=3.D-8)
*
      ZBRENT = 0.
      A = X1
      B = X2
      FA = FUNC(A)
      FB = FUNC(B)
      IF(FA*FB.GT.0.) THEN
        WRITE(*,*) 'Root must be bracketed for ZBRENT'
        RETURN
      END IF
      FC = FB
      DO ITER=1, ITMAX
        IF(FB*FC.GT.0.) THEN
          C = A
          FC = FA
          D = B-A
          E=D
        END IF
        IF(ABS(FC).LT.ABS(FB)) THEN
          A = B
          B = C
          C = A
          FA = FB
          FB = FC
          FC = FA
        END IF
        TOL1 = 2.*EPS*ABS(B)+0.5*TOL
        XM = (C-B)/2.
        IF(ABS(XM).LE.TOL1 .OR. FB.EQ.0.) THEN
          ZBRENT = B
          RETURN
        END IF
        IF(ABS(E).GE.TOL1 .AND. ABS(FA).GT.ABS(FB)) THEN
          S = FB/FA
          IF(A.EQ.C) THEN
            P = 2.*XM*S
            Q = 1.-S
          ELSE
            Q = FA/FC
            R = FB/FC
            P = S*(2.*XM*Q*(Q-R)-(B-A)*(R-1.))
            Q = (Q-1.)*(R-1.)*(S-1.)
          END IF
          IF(P.GT.0.) Q = - Q
          P = ABS(P)
          IF(2.*P.LT. MIN(3.*XM*Q-ABS(TOL1*Q),ABS(E*Q))) THEN
            E = D
            D = P/Q
          ELSE
            D = XM
            E = D
          END IF
        ELSE
          D = XM
          E = D
        END IF
        A = B
        FA = FB
        IF(ABS(D) .GT. TOL1) THEN
          B = B + D
        ELSE
          B = B + SIGN(TOL1, XM)
        END IF
        FB = FUNC(B)
      END DO
      WRITE(*,*) 'ZBRENT exceeded maximum iterations.'
      RETURN
      END

