*XCOR
*  XCOR N1 N2 N3 N4 N5 SLO SHI TAPER FILE OPTION [F1] [F2] SET NAME -- 
*     Cross-correlates spectra. See below for more details.
*     
*  Parameters:
*     N1   -- First object spectrum
*     N2   -- Last object spectrum
*     N3   -- First template spectrum
*     N4   -- Last template spectrum
*     N5   -- First slot for cross-correlations. They are stored
*             as spectra with 5000 A corresponding to 0 km/s.
*
*     SLO  -- First shift in pixels to compute. 
*     SHI  -- Last shift in pixels to compute. A search will be
*             made for a REAL header parameter called 'XCOR_SHIFT'
*             and if found SLO and SHI will be assumed relative to 
*             this. If an arc is available, it is assumed that 
*             'XCOR_SHIFT' has units of 'km/s', otherwise pixels.
*             This should help pick out the right peak in cases of
*             low signal-to-noise especially when binary motion is
*             significant. The value of XCOR_SHIFT is converted to 
*             the nearest equivalent integer pixel shift and then
*             cross-correlation is carried out from SHIFT+SLO to
*             SHIFT+SHI. The range SLO to SHI can then be reduced 
*             to an amount consistent with the need to bracket the 
*             maximum.
*            
*     TAPER - Fraction to taper at ends of spectra. This reduces
*             end effects.
*     FILE  - ASCII file to store results (blank to ignore)
*     
*     SET   - Yes to set mask of pixels in object (not template)
*     NAME  - Name of header item to store radial velocity in.
*             (uncertainty will go to 'NAME_err')
*     
*  Carries out standard computation of cross-correlation, interpolating
*  over regions masked out of analysis. Maximum cross-correlation located
*  by parabolic approx to three points at maximum. Purely statistical
*  uncertainty computed. 
*     
*  Scaling is probably best done by first fitting a constant to a region
*  of the spectra of both templates and targets and then dividing this 
*  through. Note that a spline or higher order poly is not desireable
*  because it might change the relative line strengths at different 
*  wavelengths. Following normalisation then one should subtract a fit
*  to the continuum. Another possible method is to apply a band-pass
*  filter with 'bfilt', especially to filter out longer term variations.
*     
*  The velocities that xcor produces are only accurate if prior to
*  using it you have rebinned both targets and templates onto an identical
*  velocity scale with 'vbin'.
*     
*  Related commands: vbin, bfilt, back
*     
*XCOR
      SUBROUTINE XCOR(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, FLUX, 
     &     MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, 
     &     NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, 
     &     NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &     MXREAL, SLOTS1, SLOTS2, MXSLOTS, MASK, WORK1, WORK2, 
     &     WORK3, WORK4, WORK5, WORK6, MXWORK, CLOBBER, IFAIL)
*     
*  Cross-correlate spectra
*     
*  Arguments:
*     
*  >  C*(*) SPLIT(MXSPLIT)    -- Command parameters. See below.
*  >  I     NSPLIT            -- Actual number of entries in SPLIT. Can be
*                                zero in which case defaults are used.
*  >  I     MXSPLIT           -- Dimension of SPLIT in calling routine.
*  =  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
*  =  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
*  =  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
*  >  I     MXBUFF            -- Size of spectrum buffers in calling routine
*  >  I     MXSPEC            -- Maximum number of spectra allowed by size
*                                of header buffers. 
*  >  I     MAXPX            -- Maximum number of pixels/spectrum
*  >  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
*  <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
*  <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
*  >  I     MXARC            -- Maximum number of arc coefficients/spectrum
*  <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
*  <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
*  <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
*  <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
*  <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
*  <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
*  <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
*  <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
*  <  I     NCHAR(MXSPEC)  -- Number of character header items.
*  <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
*  <  I     NINTR(MXSPEC)  -- Number of integer items.
*  <  I     NREAL(MXSPEC)  -- Number of real items.
*  >  I     MXCHAR    -- Maximum number of character header items/spec
*  >  I     MXDOUB    -- Maximum number of double precsion header items/spec
*  >  I     MXINTR    -- Maximum number of integer header items/spec
*  >  I     MXREAL    -- Maximum number of real headers items/spec
*     I     SLOTS1(MXSLOTS)     -- Workspace for list of slots
*     I     SLOTS2(MXSLOTS)     -- Workspace for list of slots
*  <  I     IFAIL            -- Error return.
*     
      IMPLICIT NONE
C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX
      INTEGER MXCHAR, MXDOUB, MXREAL, MXINTR, MXARC
      INTEGER NPIX(MXSPEC), NARC(MXSPEC)
      LOGICAL CLOBBER
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Slot lists
C     
      INTEGER MXSLOTS, NSLOTS1, NSLOTS2, SXC1, SXC2
      INTEGER SLOTS1(MXSLOTS), SLOTS2(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST1, LIST1(2,MXLIST)
      INTEGER NLIST2, LIST2(2,MXLIST)
C     
C     Mask arrays
C     
      INTEGER MXMASK, MXWORK
      LOGICAL MASK(MXWORK)
      PARAMETER (MXMASK=50)
      INTEGER NPMASK, NWMASK, NVMASK
      INTEGER PMASK(2,MXMASK)
      REAL WMASK(2,MXMASK)
      REAL VMASK(3,MXMASK)      
C     
C     Work arrays
C     
      REAL WORK1(MXWORK), WORK2(MXWORK)
      REAL WORK3(MXWORK), WORK4(MXWORK)
      REAL WORK5(MXWORK), WORK6(MXWORK)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
C     Local variables
C     
      INTEGER I, J,  N, NSHIFT
      REAL GET_FLX, GET_ERF
      INTEGER NS1, NS2, JMAX, NOUT, NCOM
      DOUBLE PRECISION ANGST, VEARTH
      DOUBLE PRECISION SUM1, SUM2
      REAL VLIGHT, VAVE, W1, W2, VS, PS, RMAX, SHIFT
      REAL TAPER, PI, FAC, Z1, Z2, SIG, Y1, Y2, Y3, WGT
      CHARACTER*64 FILE, SET*3
      INTEGER SLOT1, SLOT2, SLOT3, SLOT4, SLOT5,  IFAIL
      INTEGER  MAXSPEC, NF, NL, NLIM, NORM, NOW
      LOGICAL SAMECI, DEFAULT, WARN
C     
C     PFLM variables for adding RV to header
C     
      CHARACTER*12 NAME
      LOGICAL HADD
C     
C     Functions
C     
      REAL CGS
C     
      DATA SLOT1, SLOT2, SLOT3, SLOT4, SLOT5/1,1,2,2,3/
      DATA NS1, NS2/-10,10/
      DATA SET/'y'/
      DATA TAPER/0.03/
C     
C     PFLM Initial values of variables for adding RV to header
C     
      DATA NAME, HADD /'NULL',.FALSE./
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      NCOM    = 0
      IFAIL   = 0
      CALL INTR_IN('First slot to cross-correlate', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, 
     &     IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to cross-correlate', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         WRITE(*,*) 'Enter list of spectra to X-correlate'
         CALL GETLIS(LIST1, NLIST1, MXLIST, MAXSPEC, SLOTS1, 
     &        NSLOTS1, MXSLOTS)
         IF(NSLOTS1.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST1, NLIST1, MXLIST, SLOTS1, 
     &        NSLOTS1, MXSLOTS)
      END IF
      CALL INTR_IN('First template slot', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT3, 0, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      SLOT4 = MAX(SLOT3, SLOT4)
      CALL INTR_IN('Last template slot', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT4, SLOT3, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT3.EQ.0 .AND. SLOT4.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT3.EQ.0 .AND. SLOT4.EQ.0) THEN
         WRITE(*,*) 'Enter list of template spectra'
         CALL GETLIS(LIST2, NLIST2, MXLIST, MAXSPEC, SLOTS2, 
     &        NSLOTS2, MXSLOTS)
         IF(NSLOTS1.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT3, SLOT4, LIST2, NLIST2, MXLIST, SLOTS2, 
     &        NSLOTS2, MXSLOTS)
      END IF
      IF(NSLOTS2.NE.1 .AND. NSLOTS2.NE.NSLOTS1) THEN
         WRITE(*,*) 'Must have either a single template spectrum'
         WRITE(*,*) 'or same number as objects'
         GOTO 999
      END IF
      CALL INTR_IN('First X-corr slot (0 to ignore)', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT5, 0, MAXSPEC, IFAIL)
      IF(SLOT5+NSLOTS1-1.GT.MAXSPEC) THEN
         WRITE(*,*) 'Have not allowed enough room for X-corrs'
         GOTO 999
      END IF
      CALL INTR_IN('First shift (pixels)', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NS1, -1000000, 1000000, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      NS2 = MAX(NS1, NS2)
      CALL INTR_IN('Last shift (pixels)', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NS2, NS1, 1000000, IFAIL)
      CALL REAL_IN('Taper fraction', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, TAPER, 0., 0.5, IFAIL)
      FILE = ' '
      CALL CHAR_IN('File for output (<CR> terminal only)', 
     &     SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, FILE, IFAIL)

      IF(SLOT5+2*NSLOTS1-1.GT.MAXSPEC) THEN
         WRITE(*,*) 'Have not allowed enough room for X-corrs'
         GOTO 999
      END IF
C     
C     Mask
C     
      CALL CHAR_IN('Set mask?', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &     DEFAULT, SET, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SAMECI(SET,'YES')) THEN
         WRITE(*,*) 'Mask regions not wanted in X-corr.'
         CALL MASKIT(PMASK, WMASK, VMASK, NPMASK, NWMASK, 
     &        NVMASK, MXMASK, COUNTS, ERRORS, FLUX, MXBUFF, MAXPX,
     &        NPIX, MXSPEC, NARC, ARC, MXARC, NMCHAR, NMDOUB, NMINTR,
     &        NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB,
     &        NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, WORK1, 
     &        WORK2, WORK3, MXWORK, MASK)
      END IF
      IF(FILE.NE.' ') THEN
         IF(CLOBBER) THEN
            OPEN(UNIT=47,FILE=FILE,STATUS='UNKNOWN',IOSTAT=IFAIL)
         ELSE
            OPEN(UNIT=47,FILE=FILE,STATUS='NEW',IOSTAT=IFAIL)
         END IF
         IF(IFAIL.NE.0) THEN
            WRITE(*,*) 'Could not open ',FILE
            GOTO 999
         END IF
      END IF
C     
C     Header item
C     
      CALL CHAR_IN('Header item name (''NULL'' to ignore)', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, NAME, IFAIL)
      IF (SAMECI(NAME,'NULL')) THEN
         WRITE(*,*) 'Measured RV will not be added to header'
         HADD = .FALSE.
      ELSE
         WRITE(*,*) 'Measured RV will be written to header item ',NAME
         WRITE(*,*) '  and error will be written to header item ',
     +        NAME//'_err'
         HADD =  .TRUE.
      ENDIF
      IF(IFAIL.NE.0) GOTO 999
C     
C     Now go through spectra
C     
      VLIGHT = CGS('C')/1.E5
      NOUT   = SLOT5 - 1
      PI     = 4.*ATAN(1.)
      DO I = 1, NSLOTS1
         SXC1 = SLOTS1(I)
         SXC2 = SLOTS2(MIN(I,NSLOTS2))
         IF( NPIX(SXC1).LE.0) THEN 
            WRITE(*,*) 'Object slot ',SXC1,' empty.'
         ELSE IF( NPIX(SXC2).LE.0) THEN 
            WRITE(*,*) 'Template slot ',SXC2,' empty.'
         ELSE IF(NPIX(SXC1).NE.NPIX(SXC2)) THEN
            WRITE(*,*) 'Object ',SXC1,', template ',SXC2,
     &           ' have different'
            WRITE(*,*) 'numbers of pixels.'
         ELSE
            IF(NARC(SXC1).NE.-2 .OR. NARC(SXC2).NE.-2) THEN
               WRITE(*,*) 'Object and/or template not on a logarithmic'
               WRITE(*,*) 'scale. A mean velocity/pixel will be used'
               WRITE(*,*) 'but it may not be a good approximation'
            END IF
            WARN = .FALSE.
            IF(NARC(SXC1).NE.NARC(SXC2)) THEN
               WARN = .TRUE.
            ELSE IF(NARC(SXC1).NE.0) THEN
               DO J = 1, ABS(NARC(SXC1))
                  WARN = WARN .OR. 
     &                 ABS(ARC(J,SXC1)-ARC(J,SXC2))
     &                 .GT.1.D-5*ARC(J,SXC1)
               END DO
            END IF
            IF(WARN) THEN
               WRITE(*,*) 
     &              'Object and template are on different wavelength'
               WRITE(*,*) 'so the results may be meaningless.'
            END IF
C     
C     Load data
C     
            DO J = 1, NPIX(SXC1)
               WORK1(J) = GET_FLX(COUNTS(J,SXC1), FLUX(J,SXC1))
               WORK2(J) = GET_ERF(COUNTS(J,SXC1), ERRORS(J,SXC1),
     &              FLUX(J,SXC1))
               WORK3(J) = GET_FLX(COUNTS(J,SXC2), FLUX(J,SXC2))
            END DO
C     
C     Load arc
C     
            CALL HGETD('Vearth', VEARTH, SXC1, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
            IFAIL = 0
C     
C     Get mask (assumed to be the same for every object and template)
C     
            CALL APPMASK(MASK,NPIX(SXC1),ARC(1,SXC1),NARC(SXC1),
     &           MXARC,PMASK,NPMASK,WMASK,NWMASK,VMASK,NVMASK,MXMASK,
     &           .TRUE.,VEARTH,IFAIL)
C     
C     Apply mask by making error array negative
C     
            NF = 0
            NL = 0
            NORM = 0
            DO J = 1, NPIX(SXC1)
               IF(MASK(J) .OR. ERRORS(J,SXC1).LE.0.) THEN
                  WORK2(J) = -ABS(WORK2(J))
               ELSE 
                  IF(NF.EQ.0) NF = J
                  NL = J
                  NORM = NORM + 1
               END IF
            END DO
C     
C     Taper front and back ends
C     
            IF(TAPER.GT.0.) THEN
               NLIM = NF + INT(TAPER*REAL(NPIX(SXC1)))
               NLIM = MIN(NLIM, NL)
               DO J = NF, NLIM
                  FAC = (1.-COS(PI*REAL(J-NF)/
     &                 REAL(NPIX(SXC1))/TAPER))/2.
                  WORK1(J) = FAC*WORK1(J)
                  WORK3(J) = FAC*WORK3(J)
               END DO
               NLIM = NL - INT(TAPER*REAL(NPIX(SXC1)))
               NLIM = MAX(NLIM, NF)
               DO J = NLIM, NL
                  FAC = (1.-COS(PI*REAL(J-NL)/
     &                 REAL(NPIX(SXC1))/TAPER))/2.
                  WORK1(J) = FAC*WORK1(J)
                  WORK3(J) = FAC*WORK3(J)
               END DO
            END IF
C
C     Search for XCOR_SHIFT parameter which gives an offset to
C     the shifts over which the cross-correlation is computed.
C     
            CALL HGETR('XCOR_SHIFT', SHIFT, SXC1, MXSPEC, 
     &           NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
            IF(IFAIL.NE.0) THEN
               SHIFT = 0.
               IFAIL = 0
            END IF
            IF(NARC(SXC1).NE.0) THEN
               W1 = REAL(ANGST(1., NPIX(SXC1), NARC(SXC1), ARC(1,SXC1), 
     &              MXARC))
               W2 = REAL(ANGST(REAL(NPIX(SXC1)), NPIX(SXC1), NARC(SXC1),
     &              ARC(1,SXC1), MXARC))
               VAVE = VLIGHT*(EXP(LOG(W2/W1)/REAL(NPIX(SXC1)-1))-1.)
               NSHIFT = NINT(SHIFT/VAVE)
            ELSE
               NSHIFT = NINT(SHIFT)
            END IF
            
            DO N = NS1+NSHIFT, NS2+NSHIFT
               SUM1 = 0.D0
               SUM2 = 0.D0
               NOW  = 0
               DO J = MAX(NF,NF+N), MIN(NL, NL+N)
                  IF(WORK2(J).GT.1.E-10) THEN
                     WGT  = 1./WORK2(J)**2
                     SUM1 = SUM1 + WGT*WORK3(J-N)**2
                     SUM2 = SUM2 + WGT*WORK1(J)*WORK3(J-N)
                     NOW  = NOW + 1
                  END IF
               END DO
               IF(SUM1.GT.0.D0) THEN
                  FAC = REAL(NORM)/REAL(NOW)
                  WORK4(N-NS1-NSHIFT+1) = REAL(FAC*SUM2)
                  WORK5(N-NS1-NSHIFT+1) = REAL(FAC*SQRT(SUM1))
                  WORK6(N-NS1-NSHIFT+1) = REAL(FAC*SUM1)
               ELSE
                  WORK4(N-NS1-NSHIFT+1) =  0.D0
                  WORK5(N-NS1-NSHIFT+1) = -1.D0
                  WORK6(N-NS1-NSHIFT+1) =  0.D0
               END IF
            END DO

            RMAX = -1.E30
            DO J = 1, NS2-NS1+1
               IF(WORK4(J).GT.RMAX) THEN
                  RMAX = WORK4(J)
                  JMAX = J
               END IF
            END DO
            WRITE(*,*) ' '
            WRITE(*,*) 'Object ',SXC1,
     &           ' X-correlated with template ',SXC2
            IF(JMAX.EQ.1 .OR. JMAX.EQ.NS2-NS1+1) THEN
               WRITE(*,*) 'Maximum X-corr occurs at end of shift range'
               PS = REAL(JMAX-1+NS1+NSHIFT)
            ELSE
C     
C     Parabolic interpolation to refine position.
C     
               Y1 = WORK4(JMAX-1)
               Y2 = WORK4(JMAX  )
               Y3 = WORK4(JMAX+1)

               Z1   = Y3 - Y1
               Z2   = Y1 + Y3 - 2.*Y2
               PS   = REAL(JMAX+NS1+NSHIFT-1)-Z1/Z2/2.
               SIG  = 1./SQRT(-Z2/2.)

            END IF

            IF(NARC(SXC1).NE.0) THEN
               VS   = VAVE*PS
            ELSE
               WRITE(*,*) 'No arc cal; velocity shift undetermined.' 
               IF (HADD) WRITE(*,*) 'Cannot add header item ',NAME
               VS = 0.
            END IF     
            IF(SLOT5.GT.0) THEN
C     
C     Store X-correlation as a fake spectrum with zero lag = 5000 A
C     and a velocity scale. Other headers come from object spectrum 
C     
               NOUT = NOUT + 1
C     
C     Set headers equal to those of input spectrum 
C     
               CALL SET_HEAD(NOUT, SXC1, MXSPEC, NPIX, ARC, NARC, 
     &              MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &              HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, 
     &              NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, IFAIL)
C     
C     Store data
C     
               NPIX(NOUT)  = NS2-NS1+1
               DO J = 1, NPIX(NOUT)
                  COUNTS(J,NOUT) = WORK4(J)
                  ERRORS(J,NOUT) = WORK5(J)
                  IF(COUNTS(J,NOUT).EQ.0) THEN
                     FLUX(J,NOUT) = 1.
                  ELSE
                     FLUX(J,NOUT) = WORK4(J)
                  END IF
               END DO
C     
C     Set arc if possible
C     
               IF(NARC(SXC1).NE.0) THEN
                  NARC(NOUT)  = -2
                  ARC(2,NOUT) = DBLE(NPIX(NOUT))*VAVE/VLIGHT
                  ARC(1,NOUT) = LOG(5.D3)-ARC(2,NOUT)*
     &                 REAL(1-NS1-NSHIFT)/REAL(NPIX(NOUT))
C     
C     Set Vearth = 0 
C     
                  CALL HSETD('Vearth',0.D0,NOUT, MXSPEC, NMDOUB,
     &                 HDDOUB, NDOUB, MXDOUB, IFAIL)
               ELSE
                  NARC(NOUT) = 0
               END IF
C     
            END IF
            WRITE(*,*) 'Object shifted by ',PS,'+/-',SIG,' pixels.'
            IF(NARC(SXC1).NE.0) THEN
               WRITE(*,*) 'Equivalent to ',VS,
     &              '+/-',ABS(VAVE)*SIG,' km/s.'
               IF (HADD) THEN
                  CALL HSETR(NAME,VS,SXC1, MXSPEC, NMREAL, 
     &                 HDREAL, NREAL, MXREAL, IFAIL)
                  CALL HSETR(NAME//'_err', ABS(VAVE)*SIG, SXC1, 
     &                 MXSPEC, NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
               END IF
            END IF
            IF(FILE.NE.' ') 
     &           WRITE(47,'(2(1X,F9.3),2(1X,F9.4),1X,G10.3)',ERR=999) 
     &           VS, ABS(VAVE)*SIG, PS, SIG, RMAX
            IF(SLOT5.GT.0)
     &           WRITE(*,*) 'X-correlation stored in slot ',NOUT
         END IF
      END DO
 999  CLOSE(UNIT=47)
      RETURN
      END



