*RVEL
* RVEL N1 N2 FILE W0 METHOD WIDTH FWHM SEP -- Measures radial velocities
*                                        of lines.
*
* Parameters:
*
*            N1, N2 -- Range of spectra
*            W0     -- Rest wavelength of line
*            FILE   -- File to store results
*            METHOD -- Measurement method. 
*                      Core -- Cross-correlation with a single gaussian
*                      Wing -- Cross-correlation with two gaussians
*            WIDTH  -- Search width in km/s. Routine will not allow
*                      velocity to shift outside this range
*            FWHM   -- Full width at half maximum of gaussian (or gaussians)
*                      used for cross-correlation (km/s)
*            SEP    -- Separation (total, not half) in km/s between gaussians in WING case.
*
*
* This routine impliments standard radial velocity measurement of Young
* and Schneider. Uncertainties are computed by propagation of errors
* starting with uncertainties on data. Statistical only therefore.
*RVEL
      SUBROUTINE RADVEL(SPLIT, NSPLIT, MXSPLIT, WORK1, WORK2, MXWORK, 
     &COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, 
     &MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR,
     &MXREAL, NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, 
     &NMSREAL, VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, 
     &MXSDOUB, MXSINTR, MXSREAL, SLOTS, MXSLOTS, CLOBBER, IFAIL)
*
* Routine for measuring radial velocities by cross-correlation with
* a gaussian or double gaussian.
* allowing selection criteria to be applied.
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* >  R     WORK1(MXWORK)    -- Work arrays 
* >  R     WORK2(MXWORK)
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
* >  C*(*) NMSCHAR(MXSCHAR)   -- Names of character selection items.
* >  C*(*) VSCHAR(MXSCHAR)    -- Values of character selection items.
* >  C*(*) NMSDOUB(MXSDOUB)   -- Names of double precision selection items.
* >  D     VSDOUB(2,MXSDOUB)  -- Values of double precision selection items.
* >  C*(*) NMSINTR(MXSINTR)   -- Names of integer selection items.
* >  I     VSINTR(2,MXSINTR)  -- Values of integer selection items.
* >  C*(*) NMSREAL(MXSREAL)   -- Names of real selection items.
* >  R     VSREAL(2,MXSREAL)  -- Values of real selection items.
* >  I     NSCHAR             -- Number of character selection items.
* >  I     NSDOUB             -- Number of double precsion selection items.
* >  I     NSINTR             -- Number of integer selection items.
* >  I     NSREAL             -- Number of real selection items.
* >  I     MXSCHAR            -- Dimension of NMSCHAR in calling routine.
* >  I     MXSDOUB            -- Dimension of NMSDOUB in calling routine.
* >  I     MXSINTR            -- Dimension of NMSINTR in calling routine.
* >  I     MXSREAL            -- Dimension of NMSREAL in calling routine.   
*                                incremented and returned by routine.
* >  I     SLOTS(MXSLOTS)       -- List of slots
* >  I     MXSLOTS             -- Maximum number in list
* <  I     IFAIL              -- Error return.
*
      IMPLICIT NONE

*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL, MXSCHAR, MXSDOUB, MXSINTR
      INTEGER MXSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
      LOGICAL CLOBBER
*
* Plot parameters
*
      INTEGER MXWORK 
      REAL WORK1(MXWORK), WORK2(MXWORK)
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Search parameters, names then values.
*
      CHARACTER*(*) NMSCHAR(MXSCHAR)
      CHARACTER*(*) NMSDOUB(MXSDOUB)
      CHARACTER*(*) NMSINTR(MXSINTR)
      CHARACTER*(*) NMSREAL(MXSREAL)
*
      CHARACTER*(*) VSCHAR(MXSCHAR)
      DOUBLE PRECISION VSDOUB(2,MXSDOUB)
      INTEGER VSINTR(2,MXSINTR)
      REAL VSREAL(2,MXSREAL)
*
* Slot lists
*
      INTEGER MXSLOTS, NSLOTS, SLOT
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
*
* Local variables
*
      DOUBLE PRECISION COREFIT, WINGFIT
      EXTERNAL COREFIT, WINGFIT
      INTEGER I, J
      REAL GET_FLX, GET_ERF
      DOUBLE PRECISION ANGST, VEARTH,  PIXL, DISPA, PSIG, PSEP
      DOUBLE PRECISION PIXCEN, PIXSIG, W0, VLIGHT, PHASE, HJD
      DOUBLE PRECISION WNEW
      REAL DISP, VEL, VERR, RPIX
      CHARACTER*64 FILENAME
      CHARACTER*5 METHOD
      INTEGER SLOT1, SLOT2,  IFAIL,  NCOM, MAXSPEC,  ILO, IHI
      LOGICAL SELECT, DEFAULT, SAMECI
      REAL CGS, WIDTH, PLO, PHI, FWHM, SEP, SIGMA
*
* Functions
*
      DATA SLOT1, SLOT2, W0/1,1,5D3/
      DATA WIDTH, FWHM, SEP/1.E4, 200., 1000./
      DATA METHOD/'C'/
      SAVE SLOT1, SLOT2, NLIST, LIST, FILENAME, W0
      SAVE METHOD, WIDTH
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
*
      CALL INTR_IN('First slot to measure', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to measure', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST.GT.0) THEN
            CALL SETSLOT(LIST, NLIST, MXLIST, SLOTS, NSLOTS, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra to measure'
            CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &           NSLOTS, MXSLOTS)
         END IF
         IF(NSLOTS.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &        NSLOTS, MXSLOTS)
      END IF
*     
      FILENAME = ' '
      CALL CHAR_IN('File to store results (<CR> to ignore)', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, FILENAME, IFAIL)
      IF(FILENAME.NE.' ') THEN
         IF(CLOBBER) THEN
            OPEN(UNIT=32,FILE=FILENAME,STATUS='UNKNOWN',IOSTAT=IFAIL)
         ELSE
            OPEN(UNIT=32,FILE=FILENAME,STATUS='NEW',IOSTAT=IFAIL)
         END IF
         IF(IFAIL.NE.0) THEN
            WRITE(*,*) 'Could not open ',FILENAME
            GOTO 999
         END IF
      END IF
      CALL DOUB_IN('Rest wavelength', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, W0, 0.D0, 1.D5, IFAIL)
      CALL CHAR_IN('C(ore) or W(ing)?', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, METHOD, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(.NOT.SAMECI(METHOD,'C') .AND. .NOT.SAMECI(METHOD,'W')) THEN
         WRITE(*,*) 'Must be either C or W for core or wing'
         WRITE(*,*) 'method'
         GOTO 999
      END IF
*     
      CALL REAL_IN('Search width', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, WIDTH, 0., 1.E5, IFAIL)
      CALL REAL_IN('Gaussian FWHM', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, FWHM, 0., 1.E4, IFAIL)
      SIGMA = FWHM/2.3548
*     
      IF(SAMECI(METHOD,'W')) THEN
         CALL REAL_IN('Gaussian separation', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, SEP, 0., 5.E4, IFAIL)
      END IF
      IF(IFAIL.NE.0) GOTO 999
*
* Loop through spectra, measuring wavelength.
*
      VLIGHT = CGS('C')/1.E5
      DO I = 1, NSLOTS
*
        SLOT = SLOTS(I)
        IF(NPIX(SLOT).LE.0) THEN
          WRITE(*,*) 'No spectrum in slot ',SLOT
          GOTO 100
        END IF
        IF(NARC(SLOT).EQ.0) THEN
          WRITE(*,*) 'No arc coefficients in slot ',SLOT
          GOTO 100
        END IF
        IF(.NOT.SELECT(SLOT, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &    HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &    MXCHAR, MXDOUB, MXINTR, MXREAL, MXSPEC, NMSCHAR, VSCHAR, 
     &    NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, VSREAL, 
     &    NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, 
     &    MXSINTR, MXSREAL)) THEN
          WRITE(*,*) 'Skipped slot ',SLOT
          GOTO 100
        END IF
*
        CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &       NDOUB, MXDOUB, IFAIL)  
*
* Load up arrays
*
        DO J = 1, NPIX(SLOT)
           IF(ERRORS(J,SLOT).GT.0.) THEN
*     
*     Load Y array
*     
              WORK1(J) = GET_FLX(COUNTS(J,SLOT), FLUX(J,SLOT))
*     
*     Load errors array
*     
              WORK2(J) = GET_ERF(COUNTS(J,SLOT), ERRORS(J,SLOT), 
     &             FLUX(J,SLOT))
           ELSE
              WORK1(J) = 0.
              WORK2(J) = 1.E10
           END IF
        END DO         
*     
        PIXCEN = PIXL(W0, NPIX(SLOT), NARC(SLOT), ARC(1,SLOT), MXARC)
        DISP   = REAL(DISPA(REAL(PIXCEN), NPIX(SLOT), NARC(SLOT),
     &       ARC(1,SLOT), MXARC))
        PIXSIG = WIDTH/VLIGHT*W0/ABS(DISP)
        PSIG   = SIGMA/VLIGHT*W0/ABS(DISP)
        PSEP   = SEP  /VLIGHT*W0/ABS(DISP)
        PLO = REAL(PIXCEN - PIXSIG/2.)
        PHI = REAL(PIXCEN + PIXSIG/2.)
*     
*     Measure position
*     
        IF(SAMECI(METHOD,'C')) THEN
           ILO = NINT(PLO-4.*PSIG)
           IHI = NINT(PHI+4.*PSIG)
           ILO = MAX(1, MIN(NPIX(SLOT), ILO))
           IHI = MAX(1, MIN(NPIX(SLOT), IHI))
           CALL LINE_POS( NPIX(SLOT), WORK1, WORK2, PIXCEN, PIXSIG, 
     &          PLO, PHI, ILO, IHI, COREFIT, PSIG, 0., 0., 0.)
        ELSE IF(SAMECI(METHOD,'W')) THEN
           ILO = NINT(PLO-4.*PSIG-PSEP/2.)
           IHI = NINT(PHI+4.*PSIG+PSEP/2.)
           ILO = MAX(1, MIN(NPIX(SLOT), ILO))
           IHI = MAX(1, MIN(NPIX(SLOT), IHI))
           CALL LINE_POS( NPIX(SLOT), WORK1, WORK2, PIXCEN, PIXSIG, 
     &          PLO, PHI, ILO, IHI, WINGFIT, PSIG, PSEP, 0., 0.)
        END IF
*     
*     Convert to velocity
*
        RPIX = REAL(PIXCEN)
        WNEW = ANGST(RPIX, NPIX(SLOT), NARC(SLOT), 
     &       ARC(1,SLOT), MXARC)
        VEL  = REAL(VLIGHT*(WNEW-W0)/W0 - VEARTH)
        VERR = REAL(VLIGHT*PIXSIG*ABS(DISP)/W0)
        CALL HGETD('Orbital phase', PHASE, SLOT, MXSPEC, NMDOUB, 
     &       HDDOUB, NDOUB, MXDOUB, IFAIL)
        CALL HGETD('HJD', HJD, SLOT, MXSPEC, NMDOUB, HDDOUB,
     &       NDOUB, MXDOUB, IFAIL)
        WRITE(*,*) 'Slot, Vel, Verr ',SLOT, VEL, VERR
        IF(FILENAME.NE.' ') THEN
           WRITE(32,'(2(1X,1PG21.14),2(1X,1PG11.4))',IOSTAT=IFAIL) 
     &          HJD, PHASE, VEL, VERR
           IF(IFAIL.NE.0) WRITE(*,*) 'Could not write to disk file'
        END IF
 100    CONTINUE
      END DO
      IF(FILENAME.NE.' ') THEN
         CLOSE(UNIT=32)
         WRITE(*,*) 'Results written to ',FILENAME
      END IF
      RETURN
 999  IFAIL = 1
      CLOSE(UNIT=32)
      RETURN
      END
      
      SUBROUTINE LINE_POS( NPIX, DATA, SIGMA, PIXCEN, PIXSIG, PLO, PHI,
     &     ILO, IHI, PROFILE, P1, P2, P3, P4 )
* 
* Adapted from Keith Horne's LINEPOSN with help from Numerical Recipes
* to make more robust
*
*  Measures the pixel positions of spectral lines by maximizing the
*  cross-correlation between the spectrum data and a template line profile.
*
*  The first two derivatives of the template line profile are evaluated
*  by the USER-WRITTEN function PROFILE( PIXEL, IDERIV, P1, P2, ... ), where
*
*      PIXEL      = PIXEL OFFSET FROM CENTER OF LINE PROFILE
*      IDERIV     = 1 IF FIRST DERIVATIVE NEEDED
*                   2 IF SECOND DERIVATIVE IS NEEDED
*      P1,P2...= PARAMETERS DETERMINING FORM OF LINE PROFILE
*
*  The formal uncertainty in the line position is also evaluated.
*
*  Inputs:
*
*      NPTS       = Number of data points
*      DATA       = Spectrum data
*      SIGMA      = 1-sigma uncertainties
*      PIXCEN     = Initial guess at pixel position of spectrum line
*      PIXSIG       undefined 
*      PLO, PHI   = Range to search for root in 
*      ILO, IHI   = pixel range to evaluate cross-correlation over
*      
*      PROFILE    = Function to evaluate template profile, its first and
*            second derivatives with respect to pixel position
*      P1,P2...= PARAMETERS USED BY PROFILE( PIXEL, IDERIV, P1, P2,...)
*
*  Output:
*
*      PIXCEN      = Position of spectrum line in pixels
*      PIXSIG      = Formal 1-sigma uncertainty in line position
*
*  Sept 1984 by Keith Horne at IOA.
*
      IMPLICIT NONE
      INTEGER NPIX, I, J
      REAL PLO, PHI
      REAL DATA(NPIX), SIGMA(NPIX)
      DOUBLE PRECISION SUM, SUM2, PROFILE, PIXCEN, PIXSIG
      EXTERNAL PROFILE
      DOUBLE PRECISION EPS, PCEN, OFFSET, P1, P2, P3, P4, C1, C2
      DOUBLE PRECISION  ADD, TEMP, DC, DCOLD, CL, CH
      INTEGER MAXITER, ILO, IHI, NSEG, NADD, IP, IN
      PARAMETER (NSEG=16)
      DOUBLE PRECISION C(NSEG), V(NSEG)
      LOGICAL PL, NL
      DATA EPS/1.D-6/      
      DATA MAXITER/100/
*
*     Search for root by subdividing interval into NSEG segments.
*     Bounce back and forth from centre to make roots close to start
*     the most likely ones to find.
*
      PL = .FALSE.
      NL = .FALSE.
      C1 = PLO
      C2 = PHI
      J  = NSEG/2+MOD(NSEG,2)
      NADD = 0
      DO I = 1, NSEG
        J = J + NADD
        NADD = I
        IF(MOD(NADD,2).EQ.0) NADD = - NADD
        C(I) = C1 + (C2-C1)*DBLE(J-1)/DBLE(NSEG-1)
      END DO
      DO J = 1, NSEG
        SUM = 0.D0
        PCEN = C(J)
        DO I = ILO, IHI
          OFFSET = DBLE(I) - PCEN
          SUM    = SUM  + DATA(I) * PROFILE( OFFSET, 1, P1, P2, P3, P4 )
        END DO
        V(J) = SUM
        IF(V(J).GT.0.D0) THEN
          PL = .TRUE.
          IP = J
        ELSE
          NL = .TRUE.
          IN = J
        END IF
        IF(NL .AND. PL) GOTO 500
      END DO
      IF(NL .AND. PL) GOTO 500
*
*     Give up
*
      WRITE(*,*) 'LINE_POS could not bracket root'
      PIXSIG = -1.D3
      RETURN
*
*     Root found
*
500   CONTINUE

      CL = C(IN)
      CH = C(IP)

*
* initialise
*
      PCEN  = (CL+CH)/2.D0
      DCOLD = ABS(CH-CL)
      DC    = DCOLD
      SUM   = 0.D0
      SUM2  = 0.D0
      DO I = ILO, IHI
         OFFSET = DBLE(I) - PCEN
         SUM    = SUM  + DATA(I) * PROFILE( OFFSET, 1, P1, P2, P3, P4 )
         SUM2   = SUM2 + DATA(I) * PROFILE( OFFSET, 2, P1, P2, P3, P4 )
      END DO
*     
      DO J = 1, MAXITER

*
*     piece adapted from 'rtsafe'
*
         IF(((CH-PCEN)*SUM2-SUM)*((CL-PCEN)*SUM2-SUM).GE.0.D0
     &        .OR. ABS(2.D0*SUM).GT.ABS(DCOLD*SUM2)) THEN

* conservative binary chop

            DCOLD = DC
            DC    = (CH-CL)/2.D0
            PCEN  = CL + DC
            IF(CL.EQ.PCEN) GOTO 1000
         ELSE

* bold Newton-Raphson

            DCOLD = DC
            DC    = - SUM/SUM2
            TEMP  = PCEN
            PCEN  = PCEN - DC
            IF(TEMP.EQ.PCEN) GOTO 1000
         END IF

         IF(ABS(DC).LT.EPS) GOTO 1000

* rejig the limits

         SUM  = 0.D0
         SUM2 = 0.D0
         DO I = ILO, IHI
            OFFSET = DBLE(I) - PCEN
            SUM    = SUM  + DATA(I)*PROFILE( OFFSET, 1, P1, P2, P3, P4 )
            SUM2   = SUM2 + DATA(I)*PROFILE( OFFSET, 2, P1, P2, P3, P4 )
         END DO
         IF(SUM.LT.0.D0) THEN
            CL = PCEN
         ELSE
            CH = PCEN
         END IF
 
      END DO
      WRITE(*,*) 'LINE_POS exceeded maximum iterations'
*
*     Evaluate formal uncertainty on position
*

1000  CONTINUE

      PIXCEN = PCEN
      SUM = 0.D0
      SUM2= 0.D0
      DO I = ILO, IHI
        OFFSET = DBLE(I) - PIXCEN
        ADD    = SIGMA(I) * PROFILE( OFFSET, 1, P1, P2, P3, P4 )
        SUM    = SUM  + ADD * ADD
        SUM2   = SUM2 + DATA(I) * PROFILE( OFFSET, 2, P1, P2, P3, P4 )
      END DO
      PIXSIG = SQRT( SUM ) 
      IF( SUM2.NE.0.D0 ) PIXSIG = PIXSIG / ABS( SUM2 )

      RETURN
      END
