COPTSUB
C OPTSUB N1 N2 N3 N4 N5 FWHM SET -- Optimum subtraction of spectra
C
C OPTSUB subtracts constants times one set of spectra from another
C set, adjusting the constants to minimise the residual scatter
C between the spectra (excluding masked regions). The scatter is
C measured by carrying out the subtraction and then computing the
C Chi**2 between this and a smoothed version of itself. The routine
C also reports a formal error and the coefficients A,B,C of
C Chi**2 = A*F**2+B*F+C where F is the constant. It is assumed that
C the noise is dominated by the spectra which will have the templates
C subtracted from them.
C
C  Parameters:
C             N1   -- First of spectra to be subtracted from
C             N2   -- Last of spectra to be subtracted from
C             N3   -- First of spectra to subtract
C             N4   -- Last of spectra to subtract
C             N5   -- First slot for results
C             FWHM -- FWHM of gaussian smoothing to be applied for Chi**2
C                     in pixels
C             SET  -- Yes to mask bad regions from being used to compute
C                     the optimum factor.
COPTSUB
      SUBROUTINE OPTSUB(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &     FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, 
     &     NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &     HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &     MXDOUB, MXINTR, MXREAL, SLOTS1, SLOTS2, MXSLOTS, 
     &     WORK1, WORK2, WORK3, WORK4, WORK5, WORK6, MASK, 
     &     MXWORK, IFAIL)
C     
C     Subtracts spectra from each other attempting to optimise the residual
C     or variance of the residual in user defined regions.
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     I     SLOTS1(MXSLOTS)
C     I     SLOTS2(MXSLOTS)
C     I     MXSLOTS
C     R     WORK1(MXWORK)
C     R     WORK2(MXWORK)
C     R     WORK3(MXWORK)
C     R     WORK4(MXWORK)
C     R     WORK5(MXWORK)
C     R     WORK6(MXWORK)
C     L     MASK(MXWORK)
C     I     MXWORK
C     <  I     IFAIL              -- Error return.
C     
      IMPLICIT NONE

C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Slot lists
C     
      INTEGER MXSLOTS, NSLOTS1, NSLOTS2
      INTEGER SLOTS1(MXSLOTS), SLOTS2(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST1, LIST1(2,MXLIST)
      INTEGER NLIST2, LIST2(2,MXLIST)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
C     Work arrays
C     
      INTEGER MXWORK
      LOGICAL MASK(MXWORK)
      REAL WORK1(MXWORK), WORK2(MXWORK), WORK3(MXWORK)
      REAL WORK4(MXWORK), WORK5(MXWORK), WORK6(MXWORK)
C     
C     Mask arrays
C     
      INTEGER MXMASK
      PARAMETER (MXMASK=50)
      INTEGER NPMASK, NWMASK, NVMASK
      INTEGER PMASK(2,MXMASK)
      REAL WMASK(2,MXMASK)
      REAL VMASK(3,MXMASK)
C
C     Local variables
C
      INTEGER I, J, K, L,  N,  NS
      REAL CFRAT, GET_FLX, GET_ERF
      CHARACTER*100 SET*3
      INTEGER SLOT1, SLOT2, SLOT3, SLOT4, SLOT5
      INTEGER  IFAIL, SL1, SL2, SL3, NDOF
      INTEGER  MAXSPEC,  NCOM
      INTEGER    NF, NL, NSM, N1, N2
      REAL FLX1, FLX2, FLX3, RAT1,   RATIO
      REAL ERR1, ERR2,  CMIN, FWHM, EFAC, D, T, W
      LOGICAL  SAMECI,   DEFAULT
      DOUBLE PRECISION VEARTH, SUM1, SUM2, SUM3, SUM
C
C     Functions
C
C
      DATA SLOT1, SLOT2, SLOT3, SLOT4, SLOT5/1,1,2,2,3/
      DATA NPMASK, NWMASK, NVMASK/0,0,0/
      DATA FWHM, SET/15., 'Y'/
C     
C     
C***********************************************************************
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
      CALL INTR_IN('First spectrum to process', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last spectrum to process', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST1.GT.0) THEN
            CALL SETSLOT(LIST1, NLIST1, MXLIST, SLOTS1, 
     &           NSLOTS1, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra to process'
            CALL GETLIS(LIST1, NLIST1, MXLIST, MAXSPEC, SLOTS1, 
     &           NSLOTS1, MXSLOTS)
         END IF
         IF(NSLOTS1.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST1, NLIST1, MXLIST, SLOTS1, 
     &        NSLOTS1, MXSLOTS)
      END IF
C     
      CALL INTR_IN('First spectrum to subtract', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT3, 0, MAXSPEC, IFAIL)
      SLOT4 = MIN(SLOT3 + SLOT2 - SLOT1, MAXSPEC)
      CALL INTR_IN('Last spectrum to subtract', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT4, SLOT3, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT3.EQ.0 .AND. SLOT4.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT3.EQ.0 .AND. SLOT4.EQ.0) THEN
         IF(DEFAULT .AND. NLIST2.NE.0) THEN
            CALL SETSLOT(LIST2, NLIST2, MXLIST, SLOTS2, 
     &           NSLOTS2, MXSLOTS)
         END IF
         IF(.NOT.DEFAULT .OR. NLIST2.EQ.0 .OR. 
     &        (NSLOTS2.NE.1 .AND. NSLOTS1.NE.NSLOTS2)) THEN
            WRITE(*,*) 'Enter list of spectra to subtract'
            CALL GETLIS(LIST2, NLIST2, MXLIST, MAXSPEC, SLOTS2, 
     &           NSLOTS2, MXSLOTS)
         END IF
         IF(NSLOTS2.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT3, SLOT4, LIST2, NLIST2, MXLIST, SLOTS2, 
     &        NSLOTS2, MXSLOTS)
      END IF
      IF(NSLOTS2.NE.1 .AND. NSLOTS1.NE.NSLOTS2) THEN
         WRITE(*,*) 'Must have equal numbers of spectra or just'
         WRITE(*,*) '1 in the second group'
         GOTO 999
      END IF
C     
      CALL INTR_IN('First output slot', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT5, 1, MAXSPEC, IFAIL)
      CALL REAL_IN('FWHM of smoothing (<1 to ignore)', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, FWHM, 0., 500., IFAIL)
C     
      CALL CHAR_IN('Set mask?', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &     DEFAULT, SET, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SAMECI(SET,'YES')) THEN
         WRITE(*,*) 'Mask unwanted regions'
         CALL MASKIT(PMASK, WMASK, VMASK, NPMASK, NWMASK, 
     &        NVMASK, MXMASK, COUNTS, ERRORS, FLUX, MXBUFF, MAXPX,
     &        NPIX, MXSPEC, NARC, ARC, MXARC, NMCHAR, NMDOUB, NMINTR,
     &        NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB,
     &        NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, WORK1, 
     &        WORK2, WORK3, MXWORK, MASK)
      END IF
C     
C     Compute smoothing array
C     
      IF(FWHM.GT.1.) THEN
         NSM = INT(3.*FWHM)+1
         EFAC = 1./2./(FWHM/2.3548)**2
         SUM = 0.D0
         DO J = 1, NSM
            WORK6(J) = EXP(-EFAC*REAL(J-1)**2)
            IF(J.EQ.1) THEN
               SUM = SUM + WORK6(J)
            ELSE
               SUM = SUM + 2.*WORK6(J)
            END IF
         END DO
         DO J = 1, NSM
            WORK6(J) = REAL(WORK6(J)/SUM)
         END DO
      END IF
C     
C**************************************************************************
C     
C     Now subtract spectra
C     
      NS = 0
      DO I = 1, NSLOTS1
         SL1 = SLOTS1(I)
         IF(NSLOTS2.EQ.1) THEN
            SL2 = SLOTS2(1)
         ELSE
            SL2 = SLOTS2(I)
         END IF
         IF( NPIX(SL1).LE.0 .OR. NPIX(SL2).LE.0 .OR.
     &        NPIX(SL1).NE.NPIX(SL2)) THEN
            WRITE(*,*) 'Slot ',SL1,' and/or ',SL2,' empty or'
            WRITE(*,*) 'incompatible.'
            WRITE(*,*) SL1,' - ',SL2,' skipped.'
         ELSE
C     
C     Slot number of output
C     
            SL3 = SLOT5 + NS
            NS = NS + 1
C     
C     Set headers of output to headers of first spectrum
C     
            CALL SET_HEAD(SL3, SL1, MXSPEC, NPIX, ARC, NARC, MXARC, 
     &           NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
C     
C     Get mask
C     
            CALL HGETD('Vearth', VEARTH, SL1, MXSPEC, 
     &           NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
            IFAIL = 0
C     
            CALL APPMASK(MASK,NPIX(SL1),ARC(1,SL1),
     &           NARC(SL1),MXARC,PMASK,NPMASK,WMASK,NWMASK,
     &           VMASK,NVMASK,MXMASK,.TRUE.,VEARTH,IFAIL)
C     
C     Load data, errors and template
C     
            DO J = 1, NPIX(SL1)
               WORK1(J) = GET_FLX(COUNTS(J,SL1), FLUX(J,SL1))
               WORK2(J) = GET_ERF(COUNTS(J,SL1), ERRORS(J,SL1), 
     &              FLUX(J,SL1))
               WORK3(J) = GET_FLX(COUNTS(J,SL2), FLUX(J,SL2))
            END DO
C     
C     Interpolate over unmasked regions
C     
            CALL REPLACE(WORK1, WORK2, MASK, .TRUE., NPIX(SL1), 
     &           NF, NL, IFAIL)
            CALL REPLACE(WORK3, WORK2, MASK, .TRUE., NPIX(SL1), 
     &           NF, NL, IFAIL)
C     
C     Compute smoothed versions
C     
            IF(FWHM.GT.1.) THEN
               DO J = 1, NPIX(SL1)
                  N1 = J - NSM + 1
                  N2 = J + NSM - 1
                  SUM1 = 0.D0
                  SUM2 = 0.D0
                  DO N = N1, N2
                     K = MAX(1,MIN(N,NPIX(SL1)))
                     SUM1 = SUM1 + WORK6(ABS(N-J)+1)*WORK1(K)
                     SUM2 = SUM2 + WORK6(ABS(N-J)+1)*WORK3(K)
                  END DO
                  WORK4(J) = REAL(SUM1)
                  WORK5(J) = REAL(SUM2)
               END DO
            END IF
C     
C     Compute best factor
C     
            SUM1 = 0.D0
            SUM2 = 0.D0
            SUM3 = 0.D0
            NDOF = 0
            DO J = 1, NPIX(SL1)
               IF(WORK2(J).GT.0. .AND. .NOT.MASK(J)) THEN
                  NDOF = NDOF + 1
                  IF(FWHM.GT.1.) THEN
                     D = WORK1(J)-WORK4(J)
                     T = WORK3(J)-WORK5(J)
                  ELSE
                     D = WORK1(J)
                     T = WORK3(J)
                  END IF
                  W = 1./WORK2(J)**2
                  SUM1 = SUM1 + W*D*T
                  SUM2 = SUM2 + W*T*T
                  SUM3 = SUM3 + W*D*D
               END IF
            END DO
            RATIO = REAL(SUM1/SUM2)
            CMIN  = REAL(SUM3 - SUM1*SUM1/SUM2)
C     
C     Report optimum factor and chi-squared and 3 coefficients of
C     quadratic representing Chi**2
C     
            WRITE(*,*) 'Optimum factor = ',RATIO,' +/- ',
     &           SNGL(1./SQRT(SUM2))
            WRITE(*,*) 'Chi**2 = ',CMIN,', deg of free = ',NDOF
            WRITE(*,*) 'A,B,C = ',SNGL(SUM2),-2.*SNGL(SUM1),SNGL(SUM3)
C     
C     Carry out subtraction
C     
            DO L = 1, NPIX(SL1)
               IF(ERRORS(L,SL1).GT.0. .AND. ERRORS(L,SL2).GT.0) THEN
                  RAT1 = CFRAT(COUNTS(L,SL1),   FLUX(L,SL1))
                  FLX1 = GET_FLX(COUNTS(L,SL1), FLUX(L,SL1))
                  ERR1 = GET_ERF(COUNTS(L,SL1), ERRORS(L,SL1), 
     &                 FLUX(L,SL1))
                  FLX2 = GET_FLX(COUNTS(L,SL2), FLUX(L,SL2))
                  ERR2 = GET_ERF(COUNTS(L,SL2), ERRORS(L,SL2),
     &                 FLUX(L,SL2))
C     
                  FLX3 = FLX1 - RATIO*FLX2

                  FLUX(L,SL3)   = FLX3
                  COUNTS(L,SL3) = RAT1*FLX3
                  ERRORS(L,SL3) = RAT1*SQRT(ERR1**2+(RATIO*ERR2)**2)
                  IF(COUNTS(L,SL3).EQ.0.) FLUX(L,SL3) = RAT1
               ELSE
                  ERRORS(L,SL3) = -1.
               END IF
            END DO                                 
            WRITE(*,*) 'Slot ',SL3,'=',SL1,'-',RATIO,'*',SL2
         END IF
      END DO
 999  RETURN
      END
