CLNDF
C LNDF File N1 -- Reads in NDF 1D data
C
C Parameters:
C
C            File -- Name of an NDF file (without trailing .sdf) to read in or name 
C                    of a file containing a list of NDF files (without trailing 
C                    .sdf s) to be read in. It assumes first a list and then a 
C                    specific name if the list file cannot be read properly.
C
C            N1   -- First slot to read data into
C
C Recognises 'Angstroms' or 'Microns' as wavelength scales and a variety
C of different flux scales. 
C
C LNDF is the NDF version of the old routine LFIG but should look
C identical to the user. While this can read most 1D NDFs in, it will
C only handle headers in the 'pamela' and 'molly' extensions. (i.e.
C it can read NDFs dumped by WNDF).
C
C Related command: WNDF
C
CLNDF
      SUBROUTINE NDFRED(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &     FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, 
     &     NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &     HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &     MXREAL, WORK1, WORK2, WORK3, MXWORK, IFAIL)
C     
C     
C     
C     Authors:
C     TRM: Tom Marsh (Warwick)
C     
C     History:
C     06-May-1994 TRM
C     Added more possible flux units
C     
C     
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     D     WORK1(MXWORK)
C     D     WORK2(MXWORK)
C     D     WORK2(MXWORK)
C     I     MXWORK
C     IFAIL)
C     
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'DAT_PAR'
      INCLUDE 'PRM_PAR'
      INCLUDE 'CNF_PAR'
C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
      INTEGER MXWORK
      DOUBLE PRECISION WORK1(MXWORK), WORK2(MXWORK)
      DOUBLE PRECISION WORK3(MXWORK)
C     
C     Local variables
C     
      CHARACTER*64 FILENAME, FILE
      CHARACTER*100 STRING
      INTEGER SLOT1, SLOT,  IFAIL, IOPEN, STATUS
      INTEGER I,  MAXSPEC, SPECT
      LOGICAL    DEFAULT, FLAM, ESAMECI, ERROR, AXIS, THERE
      LOGICAL VALID
      DOUBLE PRECISION ANGST, W, DVAL
      REAL  VLIGHT, CGS, FAC, FCON, RVAL
      INTEGER  NCOM, IVAL, DAY, MONTH, YEAR
      INTEGER HOURS, MINUTES, SECONDS, MILLISECONDS
      INTEGER DPTR, EL, EPTR, XPTR, PLACE, NX, NDIM
      CHARACTER*32 AUNITS, FUNITS

C     
C     Functions
C     
      INTEGER LENSTR
C     
      DATA FILENAME/'FIG.LIS'/
      DATA SLOT1/1/
C     
      STATUS  = SAI__OK
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL CHAR_IN('List of files', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, FILENAME, IFAIL)
      CALL INTR_IN('First slot to load into', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT1, 1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
      SLOT = SLOT1 - 1
      OPEN(UNIT=47,FILE=FILENAME,STATUS='OLD',IOSTAT=IOPEN)
 100  CONTINUE
      IF(IOPEN.NE.0) THEN
         FILE = FILENAME
      ELSE
         READ(47,'(A)',IOSTAT=IFAIL) FILE
         IF(IFAIL.LT.0) THEN
            WRITE(*,*) 'End-of-file reached'
            WRITE(*,*) SLOT-SLOT1+1,' slots loaded.'
            IFAIL = 0
            GOTO 999
         ELSE IF(IFAIL.GT.0) THEN
            WRITE(*,*) 'Error reading file list'
            WRITE(*,*) SLOT-SLOT1+1,' slots loaded.'
            GOTO 999
         END IF
      END IF   
C     
C     Open NDF file for reading
C     
      I = INDEX(FILE,'.sdf')
      IF(I.GT.0) FILE(I:) = ' '
      CALL NDF_OPEN(DAT__ROOT,FILE,'READ','OLD',SPECT,PLACE,STATUS)
      CALL NDF_DIM(SPECT,1,NX,NDIM,STATUS)
      IF(STATUS.NE.SAI__OK) GOTO 999          
      IF(NX.GT.MAXPX) THEN
         STATUS = SAI__ERROR
         WRITE(*,*) 'Too many pixels for current maximum'
         WRITE(*,*) NX, ' versus ',MAXPX
         GOTO 999
      END IF
C     
      SLOT = SLOT + 1
      NPIX(SLOT) = NX
C     
C     Load data
C     
      CALL NDF_MAP(SPECT,'Data','_REAL','READ',DPTR,EL,STATUS)
      CALL LOAD_DATA(%VAL(CNF_PVAL(DPTR)), NPIX(SLOT), COUNTS(1,SLOT),
     &     FLUX(1,SLOT),STATUS)
C     
C     Load errors if they exist
C     
      CALL NDF_STATE(SPECT,'Variance',ERROR,STATUS)
      IF(ERROR) THEN
         CALL NDF_MAP(SPECT,'Error','_REAL','READ',EPTR,EL,STATUS)
         CALL STERR(ERRORS(1,SLOT), %VAL(CNF_PVAL(EPTR)), 
     :              NPIX(SLOT),STATUS)
      ELSE
         WRITE(*,*) 'No errors found. Will be fudged'
         IF(STATUS.EQ.SAI__OK) 
     &        CALL SIGFAKE(NPIX(SLOT), COUNTS(1,SLOT), ERRORS(1,SLOT))
      END IF
C     
C     Eliminate bad pixels
C     
      DO I = 1, NPIX(SLOT)
         IF(COUNTS(I,SLOT).EQ.VAL__BADR .OR. 
     &        ERRORS(I,SLOT).EQ.VAL__BADR) THEN
            COUNTS(I,SLOT) = 0.
            ERRORS(I,SLOT) = -1.
         END IF
      END DO
C     
C     Look for axis data
C     
      CALL NDF_ASTAT(SPECT,'Centre',1,AXIS,STATUS)
      IF(AXIS) THEN
         CALL NDF_AMAP(SPECT,'Centre',1,'_REAL','READ',XPTR,EL,STATUS)
         CALL FIT_ARC(%VAL(CNF_PVAL(XPTR)), 
     :                NPIX(SLOT), ARC(1,SLOT), NARC(SLOT),
     &        MXARC, WORK1, WORK2, WORK3, MXWORK, STATUS)
         AUNITS = ' '
         CALL NDF_ACGET(SPECT,'Units',1,AUNITS,STATUS)
         IF(.NOT.ESAMECI(AUNITS,'ANGSTROMS') .AND.
     &        .NOT.ESAMECI(AUNITS,'MICRONS')) THEN
            WRITE(*,*) 'Unrecognised axis units = ',AUNITS
            WRITE(*,*) 'Assuming pixel scale instead.'
            NARC(SLOT) = 0
         ELSE IF(ESAMECI(AUNITS,'MICRONS')) THEN
C     
C     Convert to Angstroms
C     
            DO I = 1, ABS(NARC(SLOT))
               ARC(I,SLOT) = 1.D4*ARC(I,SLOT)
            END DO
         END IF 
      ELSE
         WRITE(*,*) 'No axis data. Pixel scale assumed.'
         NARC(SLOT) = 0
      END IF
C     
C     Get flux units
C     
      FUNITS = ' '
      CALL NDF_CGET(SPECT,'Units', FUNITS, STATUS)
      IF(ESAMECI(FUNITS,'ERGS/CM**2/S/A')) THEN
         FCON = 1.
         FLAM = .TRUE.
      ELSE IF(ESAMECI(FUNITS,'W/(M**2*MICRON)')) THEN
         FCON = 0.1
         FLAM = .TRUE.
      ELSE IF(ESAMECI(FUNITS,'JANSKYS')) THEN
         FCON = 1000.
         FLAM = .FALSE.
      ELSE IF(ESAMECI(FUNITS,'MILLI-JANSKYS')) THEN
         FCON = 1.
         FLAM = .FALSE.
      ELSE IF(ESAMECI(FUNITS,'MILLIJANSKYS')) THEN
         FCON = 1.
         FLAM = .FALSE.
      ELSE IF(ESAMECI(FUNITS,'MJY')) THEN
         FCON = 1.
         FLAM = .FALSE.
      ELSE IF(ESAMECI(FUNITS,'COUNTS')) THEN
         FCON = 1.
         FLAM = .FALSE.
      ELSE 
         WRITE(*,*) 'Unrecognised flux units: ',FUNITS
         WRITE(*,*) 'Following are known:'
         WRITE(*,*) ' '
         WRITE(*,*) 'ergs/c**2/s/A'
         WRITE(*,*) 'W/(m**2*micron)'
         WRITE(*,*) 'Janskys'
         WRITE(*,*) 'milli-Janskys'
         WRITE(*,*) 'milliJanskys'
         WRITE(*,*) 'mJy'
         WRITE(*,*) 'counts'
         WRITE(*,*) ' '
         WRITE(*,*) 'Will assume counts by default.'
         FCON = 1.
         FLAM = .FALSE.
      END IF
      IF(FLAM) THEN
         IF(NARC(SLOT).EQ.0) THEN
            WRITE(*,*) 'Cannot convert to mJy without wavelengths'
            GOTO 999
         END IF
C     
C     Convert from FLAMBDA to FNU (assumes ergs/s/cm**2/A to MJY, FCON
C     used to provide ratios for other changes).
C     
         VLIGHT = CGS('C')/1.E18
         DO I = 1, NPIX(SLOT)
            IF(ERRORS(I,SLOT).GT.0.) THEN
               W = ANGST(REAL(I),NPIX(SLOT),NARC(SLOT),
     &              ARC(1,SLOT),MXARC)
               FAC = REAL(W**2/VLIGHT)
               IF(COUNTS(I,SLOT).NE.0.) FLUX(I,SLOT) = FAC*FLUX(I,SLOT)
               COUNTS(I,SLOT) = FAC*COUNTS(I,SLOT)
               ERRORS(I,SLOT) = FAC*ERRORS(I,SLOT)
            END IF
         END DO
      END IF
C     
C     Scale fluxes
C     
      DO I = 1, NPIX(SLOT)
         IF(ERRORS(I,SLOT).GT.0.) THEN
            IF(COUNTS(I,SLOT).NE.0.) FLUX(I,SLOT) = FCON*FLUX(I,SLOT)
            COUNTS(I,SLOT) = FCON*COUNTS(I,SLOT)
            ERRORS(I,SLOT) = FCON*ERRORS(I,SLOT)
         END IF
      END DO
C     
C     Look for standard header items from PAMELA format files.
C     First reset headers
C     
      NCHAR(SLOT) = 0
      NDOUB(SLOT) = 0
      NINTR(SLOT) = 0
      NREAL(SLOT) = 0
      CALL NDF_XSTAT(SPECT,'Pamela',THERE,STATUS)
      IF(THERE) THEN
C     
C     Object name
C     
         STRING = 'XYZXYZ'
         CALL NDF_XGT0C(SPECT,'Pamela','OBJECT',STRING,STATUS)
         IF(STRING.NE.'XYZXYZ')
     &        CALL HSETC('Object',STRING(:32), SLOT, MXSPEC, NMCHAR, 
     &        HDCHAR, NCHAR, MXCHAR, IFAIL)
C     
C     Run number
C     
         IVAL = -9999
         CALL NDF_XGT0I(SPECT,'Pamela','NUMRUN',IVAL,STATUS)
         IF(IVAL.NE.-9999) 
     &        CALL HSETI('Record', IVAL, SLOT, MXSPEC, NMINTR, 
     &        HDINTR, NINTR, MXINTR, IFAIL)
C     
C     Night number
C     
         IVAL = -9999
         CALL NDF_XGT0I(SPECT,'Pamela','NIGHT',IVAL,STATUS)
         IF(IVAL.NE.-9999) 
     &        CALL HSETI('Night', IVAL, SLOT, MXSPEC, NMINTR, 
     &        HDINTR, NINTR, MXINTR, IFAIL)
C     
C     Date
C     
         STRING = 'XYZXYZ'
         CALL NDF_XGT0C(SPECT,'Pamela','DATE',STRING,STATUS)
         IF(STRING.NE.'XYZXYZ') THEN
            READ(STRING(:10),'(I2,1X,I2,1X,I4)',IOSTAT=IFAIL) 
     &           DAY, MONTH, YEAR
            IF(IFAIL.EQ.0) THEN
               CALL HSETI('Day', DAY, SLOT, MXSPEC, NMINTR, 
     &              HDINTR, NINTR, MXINTR, IFAIL)
               CALL HSETI('Month', MONTH, SLOT, MXSPEC, NMINTR, 
     &              HDINTR, NINTR, MXINTR, IFAIL)
               IF(YEAR.LT.100) THEN
                  IF(YEAR.GT.40) THEN
                     YEAR = YEAR + 1900
                  ELSE
                     YEAR = YEAR + 2000
                  END IF
               END IF
               CALL HSETI('Year', YEAR, SLOT, MXSPEC, NMINTR, 
     &              HDINTR, NINTR, MXINTR, IFAIL)
            END IF
         END IF
C     
C     UT
C     
         STRING = 'XYZXYZ'
         CALL NDF_XGT0C(SPECT,'Pamela','UTC',STRING,STATUS)
         IF(STRING.NE.'XYZXYZ') THEN
            READ(STRING,'(I2,1X,I2,1X,I2,1X,I3)',IOSTAT=IFAIL)
     &           HOURS,MINUTES,SECONDS,MILLISECONDS
            IF(IFAIL.EQ.0) THEN
               DVAL = DBLE(HOURS)+DBLE(MINUTES)/6.D1+
     &              DBLE(SECONDS)/3.6D3+DBLE(MILLISECONDS)/3.6D6
               CALL HSETD('UTC', DVAL, SLOT, MXSPEC, NMDOUB, 
     &              HDDOUB, NDOUB, MXDOUB, IFAIL)
            END IF
         END IF
C     
C     Exposure time
C     
         RVAL = -1.E10
         CALL NDF_XGT0R(SPECT,'Pamela','TIME',RVAL,STATUS)
         IF(RVAL.NE.-1.E10) 
     &        CALL HSETR('Dwell', RVAL, SLOT, MXSPEC, NMREAL, 
     &        HDREAL, NREAL, MXREAL, IFAIL)
C     
C     RA
C     
         DVAL = -1.D10
         CALL NDF_XGT0D(SPECT,'Pamela','RA',DVAL,STATUS)
         IF(DVAL.NE.-1.E10) 
     &        CALL HSETD('RA', DVAL, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Dec
C     
         DVAL = -1.D10
         CALL NDF_XGT0D(SPECT,'Pamela','DEC',DVAL,STATUS)
         IF(DVAL.NE.-1.E10) 
     &        CALL HSETD('Dec', DVAL, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Equinox
C     
         DVAL = -1.D10
         CALL NDF_XGT0D(SPECT,'Pamela','EQUINOX',DVAL,STATUS)
         IF(DVAL.NE.-1.E10) 
     &        CALL HSETD('Equinox', DVAL, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Extraction position
C     
         RVAL = -1.E10
         CALL NDF_XGT0R(SPECT,'Pamela','CPOS',RVAL,STATUS)
         IF(RVAL.NE.-1.E10) 
     &        CALL HSETR('Extract position', RVAL, SLOT, 
     &        MXSPEC, NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
C     
C     RJD
C     
         DVAL = -1.D10
         CALL NDF_XGT0D(SPECT,'Pamela','JD',DVAL,STATUS)
         IF(DVAL.NE.-1.E10) 
     &        CALL HSETD('RJD', DVAL, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)

C     
C     Waveplate & wollastron prism angle from VLT
C     
         DVAL = -1.D10
         CALL NDF_XGT0D(SPECT,'Pamela','VLT_FORS1.ANGLE',DVAL,
     &        STATUS)
         IF(DVAL.NE.-1.D10) 
     &        CALL HSETD('VLT-FORS1-ANGLE', DVAL, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Order number for fibre spectra (from LT data March 2010)
C     
         IVAL = -9999
         CALL NDF_XGT0I(SPECT,'Pamela','FIBRE',IVAL,STATUS)
         IF(IVAL.NE.-9999) 
     &        CALL HSETI('Fibre', IVAL, SLOT, MXSPEC, NMINTR, 
     &        HDINTR, NINTR, MXINTR, IFAIL)
C     
      ELSE
C     
C     If pamela not there try out molly extension instead 
C     
         CALL NDF_XSTAT(SPECT,'Molly',THERE,STATUS)
         IF(THERE) THEN
C     
C     Object name
C     
            STRING ='XYZXYZ'
            CALL NDF_XGT0C(SPECT,'Molly','OBJECT',STRING,STATUS)
            IF(STRING.NE.'XYZXYZ')
     &           CALL HSETC('Object',STRING(:32), SLOT, MXSPEC, NMCHAR, 
     &           HDCHAR, NCHAR, MXCHAR, IFAIL)
C     
C     UT at centre of exposure
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','UTC',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('UTC', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Equinox
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','EQUINOX',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Equinox', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
            
C     
C     RA
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','RA',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('RA', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
            
C     
C     Dec
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','DEC',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Dec', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Longitude
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','LONGITUDE',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Longitude', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Latitude
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','LATITUDE',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Latitude', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Height
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','Height',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Height', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     RJD
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','RJD',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('RJD', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     HJD
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','HJD',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('HJD', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Delay
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','DELAY',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Delay', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     VEARTH
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','VEARTH',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Vearth', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Sidereal time
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','SIDEREAL_TIME',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Sidereal time', DVAL, SLOT, MXSPEC, NMDOUB,
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Airmass
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','AIRMASS',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Airmass', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Hour angle
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','HOUR_ANGLE',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Hour angle', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Altitude
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','ALTITUDE',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Altitude', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Azimuth
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','Azimuth',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Airmass', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Orbital phase
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','ORBITAL_PHASE',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('Orbital phase', DVAL, SLOT, MXSPEC, NMDOUB,
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     ZeroO
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','ZEROO',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('ZeroO', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     PeriodO
C     
            DVAL = -1.D10
            CALL NDF_XGT0D(SPECT,'Molly','PERIODO',DVAL,STATUS)
            IF(DVAL.NE.-1.D10) 
     &           CALL HSETD('PeriodO', DVAL, SLOT, MXSPEC, NMDOUB, 
     &           HDDOUB, NDOUB, MXDOUB, IFAIL)
C     
C     Run number
C     
            IVAL = -9999
            CALL NDF_XGT0I(SPECT,'Molly','RECORD',IVAL,STATUS)
            IF(IVAL.NE.-9999) 
     &           CALL HSETI('Record', IVAL, SLOT, MXSPEC, NMINTR, 
     &           HDINTR, NINTR, MXINTR, IFAIL)
C     
C     Day 
C     
            IVAL = -9999
            CALL NDF_XGT0I(SPECT,'Molly','DAY',IVAL,STATUS)
            IF(IVAL.NE.-9999) 
     &           CALL HSETI('Day', IVAL, SLOT, MXSPEC, NMINTR, 
     &           HDINTR, NINTR, MXINTR, IFAIL)
C     
C     Month
C     
            IVAL = -9999
            CALL NDF_XGT0I(SPECT,'Molly','MONTH',IVAL,STATUS)
            IF(IVAL.NE.-9999) 
     &           CALL HSETI('Month', IVAL, SLOT, MXSPEC, NMINTR, 
     &           HDINTR, NINTR, MXINTR, IFAIL)
C     
C     Year
C     
            IVAL = -9999
            CALL NDF_XGT0I(SPECT,'Molly','YEAR',IVAL,STATUS)
            IF(IVAL.NE.-9999) 
     &           CALL HSETI('Year', IVAL, SLOT, MXSPEC, NMINTR, 
     &           HDINTR, NINTR, MXINTR, IFAIL)
C     
C     Night number
C     
            IVAL = -9999
            CALL NDF_XGT0I(SPECT,'Molly','NIGHT',IVAL,STATUS)
            IF(IVAL.NE.-9999) 
     &           CALL HSETI('Night', IVAL, SLOT, MXSPEC, NMINTR, 
     &           HDINTR, NINTR, MXINTR, IFAIL)
C     
C     Exposure time
C     
            RVAL = -1.E10
            CALL NDF_XGT0R(SPECT,'Molly','DWELL',RVAL,STATUS)
            IF(RVAL.NE.-1.E10) 
     &           CALL HSETR('Dwell', RVAL, SLOT, MXSPEC, NMREAL, 
     &           HDREAL, NREAL, MXREAL, IFAIL)
         END IF
      END IF

      IF(STATUS.NE.0) GOTO 999
C     
      WRITE(*,*) 'Loaded ',FILE(:LENSTR(FILE)),' into slot ',SLOT
      CALL SL_INF(SLOT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &     NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, 
     &     NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, IFAIL)
      WRITE(*,*) STRING(:LENSTR(STRING))
C     
      CALL NDF_ANNUL(SPECT, STATUS)
      IF(IOPEN.EQ.0) GOTO 100
      IFAIL = 0
C     
C     Clear up
C     
 999  CLOSE(UNIT=47)
      CALL NDF_VALID(SPECT, VALID, STATUS)
      IF(VALID) CALL NDF_ANNUL(SPECT, STATUS)
      RETURN
      END

      SUBROUTINE LOAD_DATA(FDAT, NX, COUNTS, FLUX, STATUS)
C     
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER NX, I, STATUS
      REAL FDAT(NX), COUNTS(NX), FLUX(NX)
C     
      IF(STATUS.NE.SAI__OK) RETURN
      DO I = 1, NX
         COUNTS(I) = FDAT(I)
         IF(COUNTS(I).EQ.0.) THEN
            FLUX(I) = 1.
         ELSE
            FLUX(I) = COUNTS(I)
         END IF
      END DO
      RETURN
      END 

      SUBROUTINE FIT_ARC(WAVE, NX, ARC, NARC, MXARC, 
     &     WORK1, WORK2, WORK3, MXWORK, STATUS)
C     
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER MXARC, NARC, NX, NPOLY, I, MXWORK, STATUS
      REAL WAVE(NX)
      DOUBLE PRECISION ARC(MXARC), POLY
      DOUBLE PRECISION WORK1(MXWORK), WORK2(MXWORK), WORK3(MXWORK)
      INTEGER NMAX
      PARAMETER (NMAX=20)
      DOUBLE PRECISION XM(NMAX,2*NMAX+3), CHISQ, FRACTION, RMS
C     
C     Fit arc, assuming an Angstrom wavelength scale.
C     
      IF(STATUS.NE.SAI__OK) RETURN
      IF(ABS(WAVE(1)-1.).GT.0.1) THEN
C     
         FRACTION = 1.D-2
C     
C     Load fitting arrays
C     
         DO I = 1, NX
            WORK1(I) = DBLE(I)/DBLE(NX)
            WORK2(I) = WAVE(I)
            WORK3(I) = 1.
         END DO
         DO NPOLY = 2, MIN(NX, NMAX)
            CALL LEASQ(WORK1, WORK2, WORK3, NX, NPOLY, ARC, CHISQ,
     &           XM, 1)
            RMS = 0.D0
            DO I = 1, NX
               RMS = RMS + (WORK2(I)-POLY(ARC,NPOLY,WORK1(I)))**2
            END DO
            RMS = SQRT(RMS/DBLE(MAX(1,NX-NPOLY)))
            RMS = RMS*DBLE(NX)/ABS(WORK2(NX)-WORK2(1))
            IF(RMS.LT.FRACTION) GOTO 100
         END DO
         WRITE(*,*) 'Could not fit X scale; '//
     &        'will assume pixel scale instead'
         NARC = 0
         RETURN
 100     WRITE(*,'(A,F7.5,A,I2)')
     &        ' Acceptable fit, RMS = ',RMS,' pixels for NARC = ',NPOLY
         NARC = NPOLY
      ELSE
         NARC = 0
      END IF
      RETURN
      END

      SUBROUTINE STERR(ERRORS, SIG, NX, STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER I, NX, STATUS
      REAL ERRORS(NX), SIG(NX)
      IF(STATUS.NE.SAI__OK) RETURN
      DO I = 1, NX
         ERRORS(I) = SIG(I)
      END DO
      RETURN
      END
