*MGFIT
* MGFIT N1 N2 N3 Infile Outfile THRLO THRHI NSPLIT FIT RESET SET 
*                     Fits multiple gaussian components to spectra 
*                     allowing many different constraints on parameters.
*
* *** NB. MGFIT has a reject cycle loop. If this is set, then on exit the 
*         spectra will have masked points. This allows one to examine
*         which parts have been masked, but you may want to keep an
*         unaltered copy of the spectra as well.
*
* mgfit uses Marquardt's method of minimisation. This involves a mixture of
* steepest descent and using a second-derivative matrix to get to the minimum.
* In difficult cases (many variables) you will see it coming up with the same 
* value of chi**2 several times in a row as it moves more towards steepest
* descent having found that the quadratic method is not doing well. This is 
* normal behaviour. You should however try to give it as reasonable a start 
* as possible in such cases because it is not guaranteed to work. Identical 
* chi**2 are a sign of degeneracy or near degeneracy in the model variables 
* being fit (see below) so you may want to check whether you can re-specify 
* your model.
*
* This method is pretty poor at finding the global minimum, and you may have 
* to hold its hand a lot on its way to it. It has the advantage that once 
* there, one can get uncertainties from it.
*
*         
* Parameters:
*
*       N1, N2  -- Range of slots to fit.
*
*       N3      -- First slot to store fits. Useful for comparing model
*                  with data, 0 to ignore.
*
*       Infile  -- Input file defining the model to be used; see below.
*
*       Outfile -- Output file with fitted parameters.
*
*       THRLO, THRHI -- Lower and upper sigma reject thresholds. Pixels will 
*                  be rejected and the fit repeated until no more are rejected.
*                  Thresholds measured in units of RMS of fit. See note above.
*                 
*       NSPLIT  -- Number to split each exposure in as a means of accounting
*                  for binary motion during an exposure. In this case the
*                  model will consist of a trapezoidal integration over the
*                  exposure. Set = 1 for no account for smearing. This slows
*                  done the fits since every gaussian and derivatives have to
*                  computed NSPLIT times over.
*
*       FIT     -- There are two fit methods:
*
*                    'I' -- The variables are fitted separately for
*                           each spectrum.
*
*                    'A' -- The variable parameters are fitted to all
*                            the spectra at once.
*
*       RESET   -- Yes to reset to the starting values provided in the input
*                  file, No to use the previous spectrum's final fitted
*                  values.
*
*       SET     -- yes to set bad pixel mask (if you know that some
*                  parts will not be fitted, it is better to reject
*                  them a priori than let the routine try to do it for
*                  you.)
*
* Input files:
*
* Because of the complexity of the possible models, file input is used.
* This section now describes the format required. It does so with a
* series of examples of increasing complexity. If you are new to this you
* may want to adopt a similar approach to the fitting.
*
* Fitting a single gaussian plus a polynomial.
*
* ------------------------------- 
* poly: 6500. $const $grad
* gaussian: 6562.76 $off $fwhm 2.
*
* $const = 1.
* $grad  = 0.
* $off   = 0.
* $fwhm  = 25.
* ------------------------------
*
* WARNING: make sure you include the '.'s in the numbers. I don't know
* whether this is a feature or a bug, but I can't be bothered to fix it.
*
* The polynomial is indicated by starting "poly:". This must be exactly
* this with no leading blanks. It is then defined by a pivot wavelength 
* and coefficients. In this case 6500 is a fixed value whereas $const 
* $grad are variables (the constant and gradient terms of the poly) or
* at least potentially variable -- see below for how they can be fixed.
*
* The gaussian is similarly specified starting with "gaussian:". In this
* case the parameters are (1) the central wavelength (angstroms), 
* (2) a velocity offset (km/s), (3) a full width half max (Angstroms) and
* (4) the peak height in mJy. In the example given, only the offset and
* FWHM are variable.
*
* The variable names are case sensitive and must always begin with a $.
* Each variable must be given an initial value as in "$fwhm = 25.". 
* The initial value lines can have no leading blanks. The names can be
* 8 characters at most, excluding the $.
*
*
* FIXING "VARIABLES"
*
* One often wants to switch which parameters vary or not. This can be done
* by adding an "F" for fixed at the end of its initialisation line, e.g.:
*
* $fwhm =  25. F
*
* In fact in complicated cases it is likely that you would want most of the 
* variables to be held fixed in this manner while you optimise subsets of them. 
* This should be emphasized: the routine cannot magically find a solution
* if you give it too poor a start, and in complex cases you will need to "hold
* its hand".
*
* 
* "VARIABLES" VERSUS "PARAMETER DEFINITIONS"
*
* In the example above each parameter defining the model is associated
* with just one variable. However, it is useful to distinguish between 
* "variables" and "parameter definitions". For example consider the
* following:
*
* ------------------------------- 
* poly: 6500. $const $grad
* gaussian: 6562.76 $off $fwhm $area/$fwhm
*
* $const = 1.
* $grad  = 0.
* $off   = 0.
* $fwhm  = 25.
* $area  = 20.
* ------------------------------
*
* The last item of the gaussian represents its height and is here
* "$area/$fwhm$". $area is the variable that will be fitted here
* and the height of a gaussian is proportional to its area divided
* by its width as specified. "$area/$fwhm" is a "parameter definition"
* while $area and $fwhm are the "variables". Parameter definitions
* tell mgfit how to compute the value of a parameter from the available
* variables and can consist of series of terms joined by the *, /, + or 
* -. Brackets are not supported at the moment and there must be no 
* blanks. Thus "2.5*$fwhm+10.-$height*$area" is legal, while
* "2.*($fwhm+10.)" is not.
*
* The differentiation between the variables and the parameters required
* to specify the gaussian permits great flexibility in fitting. Here is
* another example:
*
* ----------------------------------------
* gaussian: 6562 $off 6562*$vfwhm/$c $haght
* gaussian: 4861 $off 4861*$vfwhm/$c $hbhgt
* $off = 0.
* $vfwhm = 1000.
* $hahgt = 2.
* $hbhgt = 1.
* $c = 2.9979e5 F
* ----------------------------------------
*
* This fits one gaussian to Halpha and another to Hbeta. They are
* forced to have the same width in velocity (units of km/s, $vfwhm) 
* by the constructs 6562*$vfwhm/$c etc. They are allowed different
* heights but have the same velocity offset from their central
* wavelengths ($off). The use of one variable in many different
* components is a good way of reducing the chances of degeneracy
* in the fits.
*
*
* DEGENERACY
*
* With flexibility comes the room for error and it is very easy to 
* set up a file with degeneracy in the variables which may cause
* mgfit to bomb out, although not crash molly (I hope). e.g.
*
* ---------------------------------
* gaussian: 6562 $off1+$off2 10. 1.
* $off1 = 0.
* $off2 = 10.
* ---------------------------------
*
* is clearly a degenerate case. In general it is not easy to check 
* for these beforehand and so I have not tried to. It is up to you
* to be sensible when creating the fit file. What happens during
* minimisation is that a matrix with near zero determinant is found
* and you will get a message about a routine called GAUSSJ failing.
* This will cause mgfit to either move on to the next spectrum or
* stop altogether. If this happens, check your model. It may also
* be a result of starting too far from the solution; you can use the
* no variable option (see next) to check for this.
*
*
* NO VARIABLES
*
* A file like this:
*
* ---------------------------------
* gaussian: 6562 0. 10. $height
* $height = 1. F
* --------------------------------- 
*
* has no variables and no fit will be performed. However, if you have 
* specified an output slot, it will compute a model and store it there. 
* This is a useful way to develop initial models in hard cases where if 
* your initial guess is too far off, the routine fails to find a minimum. 
* It also allows you to compute your model for arbitrary spectra.
*
*
* SPECTRUM-TO-SPECTRUM VARIATIONS: ACCESSING HEADER PARAMETERS
*
* You may want to fit a model to all spectra at once but account for
* some sort of variation from spectrum-to-spectrum. For instance suppose
* you were fitting a line gaussian in shape but with a height that
* you suspected varied sinusoidally with time. Then a file like so could
* be what you want:
*
* ---------------------------------
* gaussian: 6562 0. 10. $const+$hnorm*soff
* $const = 1.
* $hnorm = 1.
* --------------------------------- 
*
* The new element here is "soff" (NB it has no $). This will be interpreted 
* as a header parameter that must be of type double. It is up to you
* to have set this for every spectrum. That is, every spectrum will need
* to have a double precision header item called 'soff' with a value set. 
* Note that case DOES matter here and you cannot have blanks in its name. 
* In this case soff could have been set to the sine of the HJDs for example.
*
* 
* RESOLUTION PARAMETER
*
* As of June 1999, molly now recognises a parameter representing the
* FWHM resolution of a spectrum as in:
*
* ---------------------------------
* gaussian: 6562 0. $fwhm $height
* resolution: FWHM
* $fwhm  = 2.
* $height = 1.
* --------------------------------- 
*
* The meaning of this is as follows: suppose you have several spectra of
* the same object, taken with different resolutions. Then by specifying
* a resolution, in this case as a header parameter FWHM, the program
* tries to find an "underlying model" representing the lines. i.e. it
* will fit a single $fwhm and $height, blurr them by the appropriate
* resolution and then fit to the data. The resolution is added in 
* in quadrature so that the fitted FWHM = SQRT(FWHM**2+$fwhm**2) and the
* fitted height = $height*FWHM/SQRT(FWHM**2+$fwhm**2).
*
* Another use for the resolution parameter is to provide a lower limit to 
* the fitted widths. This can be useful in preventing narrow spikes being
* fitted to cosmic rays for instance.
*
* The resolution parameter only has any meaning if you have some gaussian
* components.
*
*
* ORBITAL FITS
*
* mgfit allows you to specify orbital parameters for each gaussian.
* Thus:
*
* ---------------------------------
* gaussian: 6562 $gamma 10. 1. $kx $ky 2450000 $period
* $gamma  = 0.
* $kx     = 100.
* $ky     = 150.
* $period = 0.1
* --------------------------------- 
*     
* will fit a fixed gaussian moving on an orbit of the form:
*
* V = gamma - Kx*cos(2*PI*phase) + Ky*sin(2*PI*phase)
*
* with the phase computed from the ephemeris which can be fitted
* just like the other parameters. (Note here that the zero point
* of the ephemeris is fixed here to prevent degeneracy. See above.)
*
* ---------------------------------
* gaussian: 6562 $gamma 10. 1. $kx $ky $kx2 $ky2 2450000 $period
* $gamma  = 0.
* $kx     = 100.
* $ky     = 150.
* $kx2    = 10.
* $ky2    = -5.
* $period = 0.1
* ---------------------------------
*
* will also fit a second harmonic:
*
* V = gamma - Kx*cos(2*PI*phase) + Ky*sin(2*PI*phase)
*           - Kx2*cos(4*PI*phase) + Ky2*sin(4*PI*phase)
*
* MISCELLANY
*
* The central wavelength of the gaussians is also a possible variable,
* thus:
*
* gaussian: $wave 0. 10. 1.
*
* is perfectly legal. The only thing that cannot be fitted is the
* polynomial pivot wavelength; I just felt it was unlikely ever to be
* much use to fit this one.
*
* Here is a way of specifying a multigaussian fit for a fixed ephemeris:
*
* gaussian: 6562 $off1 $fwhm1 $height1 $kx1 $ky1 ZeroO PeriodO
* gaussian: 6562 $off2 $fwhm2 $height2 $kx2 $ky2 ZeroO PeriodO
*
* etc, where ZeroO and PeriodO is the orbital ephemeris set using
* "Phase"
*
* Any line that cannot be translated into any of the above is
* regarded as a comment.
*
*
* PROBLEMS
*
* Mind you have a carriage return at the end of the last line of
* the input file otherwise it will go unrecognised.
*
*MGFIT
      SUBROUTINE MGASFIT(SPLIT, NSPLIT, MXSPLIT, CLOBBER, MUTE, IFAIL)
* 
* Arguments (most passed via common blocks in molly.coms)
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  I     IFAIL              -- Error return.
*
      IMPLICIT NONE
      INCLUDE 'molly.coms'
C
      LOGICAL CLOBBER, MUTE
C
C     Command parameters
C
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C
C     Mask arrays
C
      INTEGER MXMASK
      PARAMETER (MXMASK=50)
      INTEGER NPMASK, NWMASK, NVMASK
      INTEGER PMASK(2,MXMASK)
      REAL WMASK(2,MXMASK)
      REAL VMASK(3,MXMASK)
C
C     Slot lists
C
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
C
C     Local variables
C
      INTEGER SLOT1, SLOT2, SLOT3, NCOM, IFAIL, LENSTR
      INTEGER E1, E2, E3, E4, E, LAST, MAXSPEC, I, NLINE
      INTEGER LS, L, LADD, NSTR, ISTSTR, LE, S, B, OFF, NG
      INTEGER NOK, J, NSLOTS, SLOT, OUT, NREJT, NREJ, NCYCLE
      INTEGER NFIT, OFF1, OFF2, L1, L2, K, NTRY, EQW
      REAL THRLO, THRHI, GET_FLX, GET_ERF, CFRAT, CFR
      REAL F, R, DWELL, CGS
      CHARACTER*256 STRING
      CHARACTER*64 INFILE, OUTFILE
      CHARACTER*3 SET, RESET
      CHARACTER*1 ALL, REPLY
      LOGICAL DEFAULT, SAMECI, SELECT, LMATCH, CORV
      DOUBLE PRECISION VEARTH, VLIGHT, VFAC, ALAMBDA, AOLD
      DOUBLE PRECISION CHISQ, YONLY, STEP, CHOLD, SUM, HJD
      DOUBLE PRECISION DX, ANGST, DY, FIT, RMS, DEV, DSIG
C     
C     Include file for gaussian fitting routine
C
      INCLUDE 'gsfit.inc'
C
C     Variable arrays
C
      INTEGER NVAR
      DOUBLE PRECISION INITVAR(MAXVAR), VAR(MAXVAR)
      LOGICAL FVAR(MAXVAR)
C
      SAVE SLOT1, SLOT2, SLOT3, INFILE, SET, RESET, THRLO, THRHI
      SAVE ALL, NLIST
      DATA SLOT1, SLOT2, SLOT3, NLIST/1,1,0,0/
      DATA INFILE/'fit.dat'/
      DATA SET, RESET/'yes','yes'/
      DATA THRLO, THRHI/-4.,4./
      DATA NSBINT/1/
      DATA ALL/'A'/
      DATA NPMASK, NWMASK, NVMASK/0, 0, 0/
C
C
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL   = 0
      NCOM    = 0
C
      CALL INTR_IN('First slot to fit', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to fit', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST.GT.0) THEN
            CALL SETSLOT(LIST, NLIST, MXLIST, ISLOT, NSLOTS, MXSLOT)
         ELSE
            WRITE(*,*) 'Enter list of spectra to fit'
            CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, ISLOT, 
     &           NSLOTS, MXSLOT)
         END IF
         IF(NSLOTS.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, ISLOT, 
     &        NSLOTS, MXSLOT)
      END IF
C     
      CALL INTR_IN('First slot to store fits (0 to ignore)', 
     &     SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT3, 0, MAXSPEC, 
     &     IFAIL)
C     
      CALL CHAR_IN('Input file defining fit', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, INFILE, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
C     Open fit file. 
C     
      OPEN(UNIT=41, FILE=INFILE, STATUS='OLD', IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN
         WRITE(*,*) 'Could not open ',INFILE
         GOTO 999
      END IF
C     
C     Initialise action array and other variables
C     
      DO I = 1, MAXPAR
         ACTION(I) = 'UNDEF'
      END DO
      NPOLY  = 0
      NGAUSS = 0
      NLINE  = 0
      NVAR   = 0
      NCON   = 0
      EPHEM  = .FALSE.
      RESSPEC = .FALSE.
C     
C     Load fit file directly into ACTION array which contains strings
C     defining how the value of each parameter is to be found.
C     
      DO WHILE(IFAIL.EQ.0)
         READ(41,'(A)',IOSTAT=IFAIL) STRING
         NLINE = NLINE + 1
         IF(IFAIL.EQ.0) THEN
            LS = LEN(STRING)
            IF(STRING(:5).EQ.'poly:') THEN
C     
C     Polynomial. (1 at most allowed)
C     Specified by a pivot wavelength and then coefficients.
C     
               IF(NPOLY.GT.0) THEN
                  WRITE(*,*) 'Second poly encountered when'//
     &                 ' one is already loaded.'
                  GOTO 999
               END IF
               L     = INDEX(STRING,':') + 1
               LADD  = 1
               NSTR  = 0
               DO WHILE(LADD.GT.0 .AND. L.LE.LS)
                  LADD = ISTSTR(STRING(L:))
                  IF(LADD.GT.0) THEN
                     L  = L + LADD - 1
                     LE = INDEX(STRING(L:),' ')
                     IF(LE.EQ.0) THEN
                        LE = LS 
                     ELSE
                        LE = LE + L - 2
                     END IF
                     NSTR = NSTR + 1
                     IF(NSTR.GT.MAXPOLY+1) THEN
                        WRITE(*,*) 'Too many poly coefficients.'
                        WRITE(*,*) 'Max available = ',MAXPOLY
                        GOTO 999
                     END IF
                     IF(LE-L+1.GT.MAXACT) THEN
                        WRITE(*,*) 'Line ',NLINE,' string ',
     &                       STRING(L:LE),
     &                       ' too large for ACTION array.'
                        GOTO 999
                     END IF
                     IF(NSTR.EQ.1) THEN
                        READ(STRING(L:LE),*,IOSTAT=IFAIL) PIVOT
                        IF(IFAIL.NE.0) THEN
                           WRITE(*,*) 'Failed to read poly pivot.'
                           GOTO 999
                        END IF
                     ELSE
                        ACTION(NSTR-1) = STRING(L:LE)
                     END IF
                     L = LE + 1
                  END IF
               END DO
               NPOLY = NSTR - 1
               IF(NPOLY.LE.0) THEN
                  WRITE(*,*) 
     &                 'Poly found but no coefficients set.'
                  GOTO 999
               END IF
C     
            ELSE IF(STRING(:9).EQ.'gaussian:') THEN
C     
C     Set gaussians.
C     
               NGAUSS = NGAUSS + 1
               IF(NGAUSS.GT.MAXGAUSS) THEN
                  WRITE(*,*) 'Line ',NLINE,' attempting to set'//
     &                 ' too many gaussians.'
                  WRITE(*,*) 'Max available = ',MAXGAUSS
                  GOTO 999
               END IF
               L     = INDEX(STRING,':') + 1
               LADD  = 1
               NSTR  = 0
               DO WHILE(LADD.GT.0 .AND. L.LE.LS)
                  LADD = ISTSTR(STRING(L:))
                  IF(LADD.GT.0) THEN
                     L  = L + LADD - 1
                     LE = INDEX(STRING(L:),' ')
                     IF(LE.EQ.0) THEN
                        LE = LS 
                     ELSE
                        LE = LE + L - 2
                     END IF
                     NSTR = NSTR + 1
                     IF(NSTR.GT.8+MAXEPH) THEN
                        WRITE(*,*) 'Line ',NLINE
                        WRITE(*,*) 'Too many definitions for gaussian.'
                        WRITE(*,*) 'Maximum of ',8+MAXEPH,
     &                       ' per gaussian.'
                        GOTO 999
                     END IF
                     IF(LE-L+1.GT.MAXACT) THEN
                        WRITE(*,*) 'Line ',NLINE,' string ',
     &                       STRING(L:LE),
     &                       ' too large for ACTION array.'
                        GOTO 999
                     END IF
                     ACTION(MAXPOLY+1+(8+MAXEPH)*(NGAUSS-1)+NSTR) 
     &                    = STRING(L:LE)
                     L = LE + 1
                  END IF
               END DO
               IF(NSTR.EQ.4) THEN
                  NEPH(NGAUSS) = 0
               ELSE IF(NSTR.GE.10 .AND. NSTR.LE.8+MAXEPH) THEN
                  NEPH(NGAUSS) = NSTR-8
                  EPHEM = .TRUE.
               ELSE IF(NSTR.GE.8 .AND. NSTR.LE.6+MAXEPH) THEN
C
C     For backwards compatibility purposes with old first harmonic
C     only fits.
C
                  NEPH(NGAUSS) = NSTR-6
                  DO I = 1, NEPH(NGAUSS)
                     ACTION(MAXPOLY+1+(8+MAXEPH)*(NGAUSS-1)+8+I)
     &                    = ACTION(MAXPOLY+1+(8+MAXEPH)*(NGAUSS-1)+6+I)
                  END DO
                  ACTION(MAXPOLY+1+(8+MAXEPH)*(NGAUSS-1)+7) = '0.'
                  ACTION(MAXPOLY+1+(8+MAXEPH)*(NGAUSS-1)+8) = '0.'
                  EPHEM = .TRUE.
               ELSE
                  WRITE(*,*) 'Line ',NLINE,' gaussian number ',NGAUSS
                  WRITE(*,*) 'Invalid number of parameter'//
     &                 ' definitions encountered = ',NSTR
                  GOTO 999
               END IF
C
            ELSE IF(STRING(:11).EQ.'resolution:') THEN
C
C     FWHM resolution 
C
               L     = INDEX(STRING,':') + 1
               LADD  = 1
               NSTR  = 0
               DO WHILE(LADD.GT.0 .AND. L.LE.LS)
                  LADD = ISTSTR(STRING(L:))
                  IF(LADD.GT.0) THEN
                     L  = L + LADD - 1
                     LE = INDEX(STRING(L:),' ')
                     IF(LE.EQ.0) THEN
                        LE = LS 
                     ELSE
                        LE = LE + L - 2
                     END IF
                     NSTR = NSTR + 1
                     IF(NSTR.GT.1) THEN
                        WRITE(*,*) 'Line ',NLINE
                        WRITE(*,*) 
     &                       'Too many (>1) definitions for resolution.'
                        GOTO 999
                     END IF
                     IF(LE-L+1.GT.MAXACT) THEN
                        WRITE(*,*) 'Line ',NLINE,' string ',
     &                       STRING(L:LE),
     &                       ' too large for ACTION array.'
                        GOTO 999
                     END IF
                     ACTION(MAXPOLY+1) = STRING(L:LE)
                     L = LE + 1
                     RESSPEC = .TRUE.
                  END IF
               END DO
               IF(NSTR.EQ.0) THEN
                  WRITE(*,*) 
     &                 'Line ',NLINE,' has no definition of resolution'
                  GOTO 999
               END IF
C     
            ELSE IF(STRING(:1).EQ.'$') THEN
C     
C     Variable initialisation line. Expected to be of form
C     "$fwhm = 13.63" or "$fwhm = 13.63    F"
C     
               EQW = INDEX(STRING,'=')
               IF(EQW.EQ.0) THEN
                  WRITE(*,*) 'Line ',NLINE
                  WRITE(*,*) 'Variable initialisation line found'//
     &                 ' but no = sign located.'
                  GOTO 999
               END IF
C     
C     If there is an F after the = then this "variable" must be fixed.
C     
               L    = INDEX(STRING(EQW+1:),'F')
               CORV = L.GT.0
               IF(CORV .AND. NVAR.EQ.MAXCON) THEN
                  WRITE(*,*) 'Line ',NLINE
                  WRITE(*,*) 'Too many constants. Max = ',MAXCON
                  GOTO 999
               ELSE IF(.NOT.CORV .AND. NVAR.EQ.MAXVAR) THEN
                  WRITE(*,*) 'Line ',NLINE
                  WRITE(*,*) 'Too many variables. Max = ',MAXVAR
                  GOTO 999
               END IF
               B    = INDEX(STRING,' ')
               IF(B.GT.0 .AND. B.LT.EQW) THEN
                  E = B-1
               ELSE
                  E = EQW-1
               END IF
C     
C     STRING(2:E) now contains name of variable
C     
               IF(E.GT.MAXCVAR+1) THEN
                  WRITE(*,*) 'Variable name too long on line ',
     &                 NLINE
                  GOTO 999
               END IF
C     
C     Check for repetition
C     
               IF(NVAR.GT.0) THEN
                  LMATCH = .FALSE.
                  I     = 0
                  DO WHILE(.NOT.LMATCH .AND. I.LT.NVAR)
                     I = I + 1
                     LMATCH = STRING(2:E).EQ.
     &                    NMVAR(I)(:LENSTR(NMVAR(I)))
                  END DO
                  IF(LMATCH) THEN
                     WRITE(*,*) 'Line ',NLINE
                     WRITE(*,*) STRING(:LENSTR(STRING))
                     WRITE(*,*) 'Repeated variable name.'
                     GOTO 999
                  END IF
               END IF
               IF(NCON.GT.0) THEN
                  LMATCH = .FALSE.
                  I     = 0
                  DO WHILE(.NOT.LMATCH .AND. I.LT.NCON)
                     I = I + 1
                     LMATCH = STRING(2:E).EQ.
     &                    NMCON(I)(:LENSTR(NMCON(I)))
                  END DO
                  IF(LMATCH) THEN
                     WRITE(*,*) 'Line ',NLINE
                     WRITE(*,*) STRING(:LENSTR(STRING))
                     WRITE(*,*) 'Repeated variable name.'
                     GOTO 999
                  END IF
               END IF
C     
C     All ok, squirrel it away
C     
               IF(CORV) THEN
                  NCON = NCON + 1
                  NMCON(NCON) = STRING(2:E)
                  READ(STRING(EQW+1:EQW+L-1),*,IOSTAT=IFAIL) CON(NCON)
               ELSE
                  NVAR = NVAR + 1
                  NMVAR(NVAR) = STRING(2:E)
                  READ(STRING(EQW+1:),*,IOSTAT=IFAIL) INITVAR(NVAR)
               END IF
               IF(IFAIL.NE.0) THEN
                  WRITE(*,*) 'Failed to translate initial value'
     &                 //' in line ',NLINE
                  GOTO 999
               END IF
            END IF
         END IF
      END DO
      CLOSE(UNIT=41)
      IF(IFAIL.GT.0) THEN
         WRITE(*,*) 'Error reading ',INFILE
         GOTO 999
      END IF
C
C     Report progress
C
      IF(NPOLY.GT.0) THEN
         WRITE(*,*) 'Read ',NPOLY,' poly coefficient definitions.'
      ELSE
         WRITE(*,*) 'No polynomial will be fitted.'
      END IF
      IF(NGAUSS.GT.0) THEN
         WRITE(*,*) 'Read ',NGAUSS,' gaussian definitions.'
      ELSE
         WRITE(*,*) 'No gaussians will be fitted.'
      END IF
      IF(RESSPEC) THEN
         WRITE(*,*) 'FWHM resolution has been specified'
      ELSE
         WRITE(*,*) 'FWHM resolution has not been specified'
      END IF
      IF(NVAR.EQ.0 .AND. SLOT3.GT.0) THEN
         WRITE(*,*) 'No variables, will just compute models.'
      ELSE IF(NVAR.EQ.0) THEN
         WRITE(*,*) 'No variables and no model fits wanted.'
         WRITE(*,*) 'Aborting.'
         GOTO 999
      END IF
C
C     Now some simple checks.
C
      IF(NPOLY.EQ.0 .AND. NGAUSS.EQ.0) THEN
         WRITE(*,*) 'If you are not fitting a poly or a'//
     &        ' gaussian you are doing nothing.'
         GOTO 999
      END IF
C     
C     Every variable initialised must have an equivalent amongst 
C     the parameter definitions. At same time set VARPAR array
C     which indicates whether a parameter varies at all. Note
C     it does NOT matter if there are uninitialised constants.
C
      DO I = 1, NVAR
         FVAR(I) = .FALSE.
      END DO
      DO I = 1, MAXPAR
         VARPAR(I) = .FALSE.
         IF(ACTION(I).NE.'UNDEF') THEN
            OFF = 1
            S   = INDEX(ACTION(I)(OFF:),'$')
            DO WHILE(S.GT.0)
               LAST = LENSTR(ACTION(I)(OFF+S:))+1
               E1   = INDEX(ACTION(I)(OFF+S:),'+')
               IF(E1.EQ.0) E1 = LAST
               E2 = INDEX(ACTION(I)(OFF+S:),'-')
               IF(E2.EQ.0) E2 = LAST
               E3 = INDEX(ACTION(I)(OFF+S:),'/')
               IF(E3.EQ.0) E3 = LAST
               E4 = INDEX(ACTION(I)(OFF+S:),'*')
               IF(E4.EQ.0) E4 = LAST
               S = OFF+S
               E = S+MIN(E1,E2,E3,E4)-2
               IF(E-S+1.GT.MAXCVAR) THEN
                  WRITE(*,*) ' '
                  WRITE(*,*) '** ERROR ** '
                  IF(I.LE.MAXPOLY) THEN
                     WRITE(*,*) 
     &                    'Poly parameter definition: [',
     &                    ACTION(I)(:LENSTR(ACTION(I))),']'
                  ELSE IF(I.EQ.MAXPOLY+1) THEN
                     WRITE(*,*) 
     &                    'Resolution parameter definition: [',
     &                    ACTION(I)(:LENSTR(ACTION(I))),']'
                  ELSE
                     NG = (I-MAXPOLY-2)/(8+MAXEPH)+1
                     WRITE(*,*) 'Gaussian ',NG,
     &                    ' parameter definition: [',
     &                    ACTION(I)(:LENSTR(ACTION(I))),']'
                  END IF
                  WRITE(*,*) 'contains an invalid variable.'
                  WRITE(*,*) 'called: [',ACTION(I)(S:E),']'
                  GOTO 999
               END IF
C
C     Search for this variable
C
               LMATCH = .FALSE.
               J      = 0
               DO WHILE(.NOT.LMATCH .AND. J.LT.NVAR)
                  J = J + 1
                  LMATCH = ACTION(I)(S:E).EQ.
     &                 NMVAR(J)(:LENSTR(NMVAR(J)))
               END DO
               IF(LMATCH) THEN
                  VARPAR(I) = .TRUE.
                  FVAR(J)   = .TRUE.
               ELSE
                  J      = 0
                  DO WHILE(.NOT.LMATCH .AND. J.LT.NCON)
                     J = J + 1
                     LMATCH = ACTION(I)(S:E).EQ.
     &                    NMCON(J)(:LENSTR(NMCON(J)))
                  END DO
                  IF(.NOT.LMATCH) THEN
                     WRITE(*,*) ' '
                     WRITE(*,*) '** ERROR **'
                     IF(I.LE.MAXPOLY) THEN
                        WRITE(*,*) 
     &                       'Poly parameter definition: [',
     &                       ACTION(I)(:LENSTR(ACTION(I))),']'
                     ELSE IF(I.EQ.MAXPOLY+1) THEN
                        WRITE(*,*) 
     &                       'Resolution parameter definition: [',
     &                       ACTION(I)(:LENSTR(ACTION(I))),']'
                     ELSE
                        NG = (I-MAXPOLY-2)/(8+MAXEPH)+1
                        WRITE(*,*) 'Gaussian ',NG,
     &                       ' parameter definition: [',
     &                       ACTION(I)(:LENSTR(ACTION(I))),']'
                     END IF
                     WRITE(*,*) 'contains an uninitialised variable.'
                     WRITE(*,*) 'called: [',ACTION(I)(S:E),']'
                     GOTO 999
                  END IF
               END IF
               OFF = E + 1 
               S   = INDEX(ACTION(I)(OFF:),'$')
            END DO
         END IF
      END DO
      DO I = 1, NVAR
         IF(.NOT.FVAR(I)) THEN
            WRITE(*,*) 'Initialised variable = ',NMVAR(I)
            WRITE(*,*) 'not found in any parameter definition.'
            GOTO 999
         END IF
      END DO
C
C     All OK so far. Get remaining command parameters.
C
      IFAIL  = 0 
      IF(NVAR.GT.0) THEN
         OUTFILE = ' '
         CALL CHAR_IN('Output file for results? <CR> to ignore', 
     &        SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, OUTFILE, IFAIL)
         
         IF(OUTFILE.NE.' ') THEN
            IF(CLOBBER) THEN
               OPEN(UNIT=51,FILE=OUTFILE,STATUS='UNKNOWN',IOSTAT=IFAIL)
            ELSE
               OPEN(UNIT=51,FILE=OUTFILE,STATUS='NEW',IOSTAT=IFAIL)
            END IF
            IF(IFAIL.NE.0) THEN
               WRITE(*,*) 'Failed to open ',OUTFILE
               GOTO 999
            END IF
         END IF
         CALL REAL_IN('Lower sigma reject threshold', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, THRLO, -1.E20, 1.E20, IFAIL)
         CALL REAL_IN('Upper sigma reject threshold', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, THRHI, THRLO, 1.E20, IFAIL)
      END IF
      IF(EPHEM) THEN
         CALL INTR_IN('Number of sub-intervals/exposure', SPLIT, 
     &        NSPLIT, MXSPLIT, NCOM, DEFAULT, NSBINT, 1, MAXSMR, 
     &        IFAIL)
      ELSE
         NSBINT = 1
      END IF
      IF(IFAIL.NE.0) GOTO 999
      IF(NVAR.EQ.0) THEN
         ALL = 'I'
      ELSE IF(NSLOTS.EQ.1) THEN
         ALL = 'I'
      ELSE
         CALL CHAR_IN('Fit A(ll) or I(ndividually)?', 
     &        SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, ALL, IFAIL)
         IF(.NOT.SAMECI(ALL,'A') .AND. 
     &        .NOT.SAMECI(ALL,'I')) THEN
            WRITE(*,*) 'Must answer A or I.'
            GOTO 999
         END IF
      END IF
      IF(SAMECI(ALL,'I') .AND. NSLOTS.GT.1 .AND. NVAR.GT.0) THEN
         CALL CHAR_IN('Reset to initial values for each spectrum?', 
     &        SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, RESET, IFAIL)
         IF(.NOT.SAMECI(RESET,'YES') .AND. 
     &        .NOT.SAMECI(RESET,'NO')) THEN
            WRITE(*,*) 'Must answer yes or no'
            GOTO 999
         END IF
      ELSE
         RESET = 'yes'
      END IF
      IF(NVAR.GT.0) THEN
         CALL CHAR_IN('Set mask?', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &        DEFAULT, SET, IFAIL)
         IF(SAMECI(SET,'YES')) THEN
            WRITE(*,*) 'Mask regions not wanted in fit'
            CALL MASKIT(PMASK, WMASK, VMASK, NPMASK, NWMASK, 
     &           NVMASK, MXMASK, COUNTS, ERRORS, FLUX, MXBUFF, MAXPX,
     &           NPIX, MXSPEC, NARC, ARC, MXARC, NMCHAR, NMDOUB, NMINTR,
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB,
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, RWORK, 
     &           RWORK(1,2), RWORK(1,3), MXWORK, LWORK)
         END IF
      END IF
C     
C     Set initial values. Old ones are kept in case they
C     are required to reset starting values
C     
      DO I = 1, NVAR
         VAR(I) = INITVAR(I)
      END DO
C     
C     Accumulate a list of good spectra
C     
      NOK  = 0
      DO I = 1, NSLOTS
         SLOT = ISLOT(I,1)
         IF(NPIX(SLOT).LE.0) THEN
            WRITE(*,*) 'No spectrum in slot ',SLOT
         ELSE IF(NARC(SLOT).EQ.0) THEN
            WRITE(*,*) 'No arc coefficients in slot ',SLOT
         ELSE IF(.NOT.SELECT(SLOT, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &           HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, 
     &           NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, MXSPEC, 
     &           NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, 
     &           NMSREAL, VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, 
     &           MXSCHAR, MXSDOUB, MXSINTR, MXSREAL)) THEN
            WRITE(*,*) 'Skipped slot ',SLOT
         ELSE
C     
C     If an ephemeris is to be used, HJDs must exist.
C     
            IF(EPHEM) THEN
               CALL HGETD('HJD', HJD, SLOT, MXSPEC, 
     &              NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
               IF(IFAIL.NE.0) THEN
                  WRITE(*,*) 'No HJD in slot ',SLOT
                  WRITE(*,*) 'Needed for orbits; aborting.'
                  GOTO 999
               END IF
            END IF
C     
C     If smearing is to accounted for DWELL must exist too.
C     
            IF(NSBINT.GT.1) THEN
               CALL HGETR('DWELL', DWELL, SLOT, MXSPEC, 
     &              NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
               IF(IFAIL.NE.0) THEN
                  WRITE(*,*) 'No DWELL in slot ',SLOT
                  WRITE(*,*) 'Needed for sub-splitting; aborting.'
                  GOTO 999
               END IF
            END IF
            NOK = NOK + 1
            ISLOT(NOK,2) = SLOT
         END IF
      END DO
      IF(NOK.EQ.0) THEN
         WRITE(*,*) 'No valid spectra.'
         GOTO 999
      END IF
C     
C     Now carry out fits.
C     
      VLIGHT = CGS('C')/1.D5
      IF(SAMECI(ALL,'I')) THEN
         OUT = SLOT3 - 1
         DO I = 1, NOK
            SLOT = ISLOT(I,2)
            IF(OUTFILE.NE.' ') THEN
               WRITE(51,*,ERR=998) ' '
               WRITE(51,*,ERR=998) ' Slot ',SLOT
            END IF
C     
C     Retrieve initial values if specified.
C     
            IF(SAMECI(RESET,'YES')) THEN
               DO J = 1, NVAR
                  VAR(J) = INITVAR(J)
               END DO
            END IF
C     
C     Search for double precision header item Vearth. If not
C     found, assume = 0
C     
            CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, 
     &           NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
            IFAIL = 0
C     
C     Get mask
C     
            CALL APPMASK(LWORK,NPIX(SLOT),ARC(1,SLOT),NARC(SLOT),
     &           MXARC,PMASK,NPMASK,WMASK,NWMASK,VMASK,NVMASK,
     &           MXMASK,.FALSE.,VEARTH,IFAIL)   
            VFAC = 1.D0 - VEARTH/DBLE(VLIGHT)
C
C     Set parameter values in order to compute initial chi**2
C
            CALL SETPARS(SLOT, VAR, NVAR, MAXVAR, IFAIL)
            IF(IFAIL.NE.0) GOTO 999
C
            IF(NVAR.GT.0) THEN
C     
C     Evaluate initial chi**2
C     
               SUM  = 0.D0
               OFF  = MAXPX*(SLOT-1)
               DO J = 1, NPIX(SLOT)
                  OFF = OFF + 1
                  IF(LWORK(J,1) .AND. ERRORS(OFF).GT.0.) THEN
                     DX  = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                    NARC(SLOT), ARC(1,SLOT), MXARC)
                     DY  = GET_FLX(COUNTS(OFF), FLUX(OFF))
                     DSIG= GET_ERF(COUNTS(OFF), ERRORS(OFF), FLUX(OFF))
                     FIT  = YONLY(DX)
                     SUM = SUM + ((DY-FIT)/DSIG)**2
                  END IF
               END DO
               WRITE(*,*) '   Initial Chi**2 = ',SUM
               WRITE(*,*) ' '
C     
C     Cycle rejecting bad pixels
C     
               NREJT = 0
               NREJ  = 1
               NCYCLE = 1
               DO WHILE(NREJ.GT.0)
C     
C     Initialise and perform first fit.
C     
                  NTRY = 1
                  ALAMBDA = -1.D0
                  CALL MMRQMIN(VAR, NVAR, CHISQ, ALAMBDA, I, 1, 
     &                 PMASK, WMASK, VMASK, NPMASK, NWMASK, NVMASK, 
     &                 MXMASK, IFAIL)
                  IF(IFAIL.NE.0) GOTO 997
                  IF(.NOT.MUTE) 
     &                 WRITE(*,*) 'Cycle ',NCYCLE,', step ',NTRY,
     &                 ', Chi**2 = ',CHISQ
C     
C     Cycle until Chi**2 is not reduced any more and ALAMBDA does
C     not increase.
C     
                  STEP = 1.D0
                  AOLD = ALAMBDA-100.
                  DO WHILE(AOLD.LT.ALAMBDA .OR. STEP.GT.1.D-2)
                     CHOLD = CHISQ
                     AOLD  = ALAMBDA
                     NTRY  = NTRY + 1
                     CALL MMRQMIN(VAR, NVAR, CHISQ, ALAMBDA, I, 1, 
     &                    PMASK, WMASK, VMASK, NPMASK, NWMASK, NVMASK, 
     &                    MXMASK, IFAIL)
                     IF(IFAIL.NE.0) GOTO 997
                     IF(.NOT.MUTE)
     &                    WRITE(*,*) 'Cycle ',NCYCLE,', step ',NTRY,
     &                    ', Chi**2 = ',CHISQ
                     STEP = CHOLD - CHISQ
                     IF(MOD(NTRY,1000).EQ.0) THEN
                        WRITE(*,*) NTRY,' iterations.'
                        WRITE(*,'(A,$)') 
     &                       '<CR> to continue,'//
     &                       ' anything else to abort: '
                        READ(*,'(A)') REPLY
                        IF(REPLY.NE.' ') GOTO 999
                     END IF
                  END DO
C     
C     Set parameter values so that fits will be
C     evaluated correctly
C     
                  CALL SETPARS(SLOT, VAR, NVAR, MAXVAR, IFAIL)
                  IF(IFAIL.NE.0) GOTO 999
C     
C     Evaluate RMS of fit
C     
                  NFIT = 0
                  SUM  = 0.D0
                  OFF  = MAXPX*(SLOT-1)
                  DO J = 1, NPIX(SLOT)
                     OFF = OFF + 1
                     IF(LWORK(J,1) .AND. ERRORS(OFF).GT.0.) THEN
                        NFIT = NFIT + 1
                        DX  = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                       NARC(SLOT), ARC(1,SLOT), MXARC)
                        DY  = GET_FLX(COUNTS(OFF), FLUX(OFF))
                        DSIG= GET_ERF(COUNTS(OFF), 
     &                       ERRORS(OFF), FLUX(OFF))
                        FIT  = YONLY(DX)
                        SUM = SUM + ((DY-FIT)/DSIG)**2
                     END IF
                  END DO
                  IF(NFIT.LT.MAX(NVAR,2)) THEN
                     WRITE(*,*) 'Too few points to fit on slot ',SLOT
                     GOTO 999
                  END IF
                  RMS = SQRT(SUM/DBLE(NFIT-NVAR))
C     
C     Kick out bad pixels
C     
                  NREJ = 0
                  OFF  = MAXPX*(SLOT-1)
                  DO J = 1, NPIX(SLOT)
                     OFF = OFF + 1
                     IF(LWORK(J,1) .AND. ERRORS(OFF).GT.0.) THEN
                        DX  = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                       NARC(SLOT), ARC(1,SLOT), MXARC)
                        DY  = GET_FLX(COUNTS(OFF), FLUX(OFF))
                        DSIG= GET_ERF(COUNTS(OFF), 
     &                       ERRORS(OFF), FLUX(OFF))
                        FIT = YONLY(DX)
                        DEV = DY - FIT
                        IF(DEV.LT.DBLE(THRLO)*RMS*DSIG .OR.
     &                       DEV.GT.DBLE(THRHI)*RMS*DSIG) THEN
                           IF(.NOT.MUTE)
     &                          WRITE(*,*) 'Pixel ',J,' of slot ',SLOT,
     &                          ' deviated by ',
     &                          SNGL(DEV/RMS/DSIG),' sigma.'
                           IF(OUTFILE.NE.' ') THEN
                              WRITE(51,*,ERR=998) 'Pixel ',J,
     &                             ', W = ',
     &                             SNGL(DX),' deviated by ',
     &                             SNGL(DEV/RMS/DSIG),' sigma.'
                           END IF
                           NREJ = NREJ + 1
                           ERRORS(OFF) = - ERRORS(OFF)
                        END IF
                     END IF
                  END DO
                  NREJT = NREJT + NREJ
                  NCYCLE = NCYCLE + 1
               END DO
C     
C     Compute covariances
C     
               ALAMBDA = 0.D0
               CALL MMRQMIN(VAR, NVAR, CHISQ, ALAMBDA, I, 1, 
     &              PMASK, WMASK, VMASK, NPMASK, NWMASK, NVMASK, 
     &              MXMASK, IFAIL)
               IF(OUTFILE.NE.' ') THEN
                  WRITE(51,*,ERR=998) NFIT,' pixels fitted.'
                  WRITE(51,*,ERR=998) NREJT,
     &                 ' pixels rejected during fit.'
                  WRITE(51,*,ERR=998) 'Chi**2/D of F = ',
     &                 CHISQ/DBLE(NFIT-NVAR)
                  WRITE(51,*,ERR=998) ' '
                  WRITE(51,*,ERR=998) 'Fitted variable values'//
     &                 ' along with their'
                  WRITE(51,*,ERR=998) '1-param uncertainties are: '
                  WRITE(51,*,ERR=998) ' '
                  DO J = 1, NVAR
                     L = NVAR*(J-1)+J
                     WRITE(51,*,ERR=998) 'Variable: ',NMVAR(J),'=',
     &                    VAR(J),' +/- ',SQRT(DWORK(L,2))
                  END DO      
                  DO J = 1, NVAR-1
                     L  = NVAR*(J-1)
                     L1 = L+J
                     DO K = J+1, NVAR
                        L2 = NVAR*(K-1)+K
                        R = REAL(DWORK(L+K,2)/
     &                       SQRT(DWORK(L1,2)*DWORK(L2,2)))
                        WRITE(51,*,ERR=998) '<',
     &                       NMVAR(J)(:LENSTR(NMVAR(J))),'*',
     &                       NMVAR(K)(:LENSTR(NMVAR(K))),'> = ',R
                     END DO
                  END DO      
               END IF
               WRITE(*,*) ' Slot ',SLOT,', Chi**2 = ',CHISQ
               WRITE(*,*) NFIT,' pixels fitted.'
               WRITE(*,*) NREJT,' pixels rejected during fit.'
               WRITE(*,*) 'Chi**2/D of F = ',CHISQ/DBLE(NFIT-NVAR)
               WRITE(*,*) ' '
               WRITE(*,*) 'Fitted variable values along with their'
               WRITE(*,*) '1-sigma 1-param uncertainties are: '
               WRITE(*,*) ' '
               DO J = 1, NVAR
                  L = NVAR*(J-1)+J
                  WRITE(*,*,ERR=998) 'Variable: ',NMVAR(J),'=',
     &                 VAR(J),' +/- ',SQRT(DWORK(L,2))
               END DO      
            END IF
C     
C     If SLOT3 .GT. 0, store results there.
C     
            IF(SLOT3.GT.0) THEN
               OUT = OUT + 1
C     
C     Copy headers
C
               CALL SET_HEAD(OUT, SLOT, MXSPEC, NPIX, ARC, NARC, 
     &              MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &              HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, 
     &              NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, IFAIL)

               DO J = 1, NCHAR(out)
                  WRITE(*,*) 'C: ',NMCHAR(J,out),' ',HDCHAR(J,out)
               END DO
C     
C     Initialise spectrum
C     
               CFR  = 1.
               OFF1 = MAXPX*(SLOT-1)
               OFF2 = MAXPX*(OUT-1)
               DO J = 1, NPIX(OUT)
                  OFF1 = OFF1 + 1
                  OFF2 = OFF2 + 1
                  CFR  = CFRAT(COUNTS(OFF1),FLUX(OFF1))
                  ERRORS(OFF2) = 1.E-10
                  DX  = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                 NARC(SLOT), ARC(1,SLOT), MXARC)
                  F  = REAL(YONLY(DX))
                  COUNTS(OFF2) = CFR*F
                  IF(COUNTS(OFF2).EQ.0.) THEN
                     FLUX(OFF2) = CFR
                  ELSE
                     FLUX(OFF2) = F
                  END IF
               END DO
               IF(.NOT.MUTE) THEN
                  IF(NVAR.GT.0) THEN
                     WRITE(*,*) 'Fit to slot ',SLOT,
     &                    ' stored in slot ',OUT
                  ELSE
                     WRITE(*,*) 'Model equivalent to slot ',SLOT,
     &                    ' stored in slot ',OUT
                  END IF
               END IF
            END IF
 997        CONTINUE
         END DO    

      ELSE IF(SAMECI(ALL,'ALL')) THEN
C    
C     Fit all at once
C     
         IF(NVAR.GT.0) THEN
            SUM    = 0.D0
            DO I = 1, NOK
               SLOT = ISLOT(I,2)                  
C     
C     Set parameter values in order to compute initial chi**2
C     
               CALL SETPARS(SLOT, VAR, NVAR, MAXVAR, IFAIL)
               IF(IFAIL.NE.0) GOTO 999
C     
C     Search for double precision header item Vearth. If not
C     found, assume = 0
C     
               CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, 
     &              NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
               IFAIL = 0
C     
C     Get mask
C     
               CALL APPMASK(LWORK,NPIX(SLOT),ARC(1,SLOT),NARC(SLOT),
     &              MXARC,PMASK,NPMASK,WMASK,NWMASK,VMASK,NVMASK,
     &              MXMASK,.FALSE.,VEARTH,IFAIL)   
               VFAC = 1.D0 - VEARTH/DBLE(VLIGHT)
C
               OFF  = MAXPX*(SLOT-1)
               DO J = 1, NPIX(SLOT)
                  OFF = OFF + 1
                  IF(LWORK(J,1) .AND. ERRORS(OFF).GT.0.) THEN
                     DX  = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                    NARC(SLOT), ARC(1,SLOT), MXARC)
                     DY  = GET_FLX(COUNTS(OFF), FLUX(OFF))
                     DSIG= GET_ERF(COUNTS(OFF), 
     &                    ERRORS(OFF), FLUX(OFF))
                     FIT  = YONLY(DX)
                     SUM = SUM + ((DY-FIT)/DSIG)**2
                  END IF
               END DO
            END DO
            WRITE(*,*) '   Initial Chi**2 = ',SUM
            WRITE(*,*) ' '
C     
C     Cycle rejecting bad pixels
C     
            NREJT = 0
            NREJ  = 1
            NCYCLE = 1
            DO WHILE(NREJ.GT.0)
C     
C     Initialise and perform first fit.
C     
               NTRY = 1
               ALAMBDA = -1.D0
               CALL MMRQMIN(VAR, NVAR, CHISQ, ALAMBDA, 1, NOK, 
     &              PMASK, WMASK, VMASK, NPMASK, NWMASK, NVMASK, 
     &              MXMASK, IFAIL)
               IF(IFAIL.NE.0) GOTO 999
               IF(.NOT.MUTE)
     &          WRITE(*,*) 'Cycle ',NCYCLE,', step ',NTRY,
     &              ', Chi**2 = ',CHISQ
C     
C     Cycle until Chi**2 is not reduced any more and ALAMBDA does
C     not increase.
C     
               STEP = 1.D0
               AOLD = ALAMBDA-100.
               DO WHILE(AOLD.LT.ALAMBDA .OR. STEP.GT.1.D-2)
                  CHOLD = CHISQ
                  AOLD  = ALAMBDA
                  NTRY  = NTRY + 1
                  CALL MMRQMIN(VAR, NVAR, CHISQ, ALAMBDA, 1, NOK, 
     &                 PMASK, WMASK, VMASK, NPMASK, NWMASK, NVMASK, 
     &                 MXMASK, IFAIL)
                  IF(IFAIL.NE.0) GOTO 999
                  IF(.NOT.MUTE)
     &                 WRITE(*,*) 'Cycle ',NCYCLE,', step ',NTRY,
     &                 ', Chi**2 = ',CHISQ
                  STEP = CHOLD - CHISQ
                  IF(MOD(NTRY,1000).EQ.0) THEN
                     WRITE(*,*) NTRY,' iterations.'
                     WRITE(*,'(A,$)') 
     &                    '<CR> to continue, anything else to abort: '
                     READ(*,'(A)') REPLY
                     IF(REPLY.NE.' ') GOTO 999
                  END IF
               END DO
C
C     Evaluate RMS of fit
C     
               SUM  = 0.D0
               NFIT = 0
               DO I = 1, NOK
C     
C     Search for double precision header item Vearth. If not
C     found, assume = 0
C     
                  SLOT = ISLOT(I,2)                  
                  CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, 
     &                 NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                  IFAIL = 0
C     
C     Get mask
C     
                  CALL APPMASK(LWORK,NPIX(SLOT),ARC(1,SLOT),NARC(SLOT),
     &                 MXARC,PMASK,NPMASK,WMASK,NWMASK,VMASK,NVMASK,
     &                 MXMASK,.FALSE.,VEARTH,IFAIL)   
                  VFAC = 1.D0 - VEARTH/DBLE(VLIGHT)
C     
C     Set parameter values in order to compute initial chi**2
C     
                  CALL SETPARS(SLOT, VAR, NVAR, MAXVAR, IFAIL)
                  IF(IFAIL.NE.0) GOTO 999
                  OFF  = MAXPX*(SLOT-1)
                  DO J = 1, NPIX(SLOT)
                     OFF = OFF + 1
                     IF(LWORK(J,1) .AND. ERRORS(OFF).GT.0.) THEN
                        NFIT = NFIT + 1
                        DX   = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                       NARC(SLOT), ARC(1,SLOT), MXARC)
                        DY   = GET_FLX(COUNTS(OFF), FLUX(OFF))
                        DSIG = GET_ERF(COUNTS(OFF), 
     &                       ERRORS(OFF), FLUX(OFF))
                        FIT  = YONLY(DX)
                        SUM  = SUM + ((DY-FIT)/DSIG)**2
                     END IF
                  END DO
               END DO
               IF(NFIT.LT.MAX(NVAR,2)) THEN
                  WRITE(*,*) 'Too few points to fit'
                  GOTO 999
               END IF
               RMS = SQRT(SUM/DBLE(NFIT-NVAR))
C     
C     Kick out bad pixels
C     
               NREJ = 0
               DO I = 1, NOK
C     
C     Search for double precision header item Vearth. If not
C     found, assume = 0
C     
                  SLOT = ISLOT(I,2)                  
                  CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, 
     &                 NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                  IFAIL = 0
C     
C     Get mask
C     
                  CALL APPMASK(LWORK,NPIX(SLOT),ARC(1,SLOT),NARC(SLOT),
     &                 MXARC,PMASK,NPMASK,WMASK,NWMASK,VMASK,NVMASK,
     &                 MXMASK,.FALSE.,VEARTH,IFAIL)   
                  VFAC = 1.D0 - VEARTH/DBLE(VLIGHT)
C     
C     Set parameter values in order to compute initial chi**2
C     
                  CALL SETPARS(SLOT, VAR, NVAR, MAXVAR, IFAIL)
                  IF(IFAIL.NE.0) GOTO 999
                  OFF  = MAXPX*(SLOT-1)
                  DO J = 1, NPIX(SLOT)
                     OFF = OFF + 1
                     IF(LWORK(J,1) .AND. ERRORS(OFF).GT.0.) THEN
                        DX  = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                       NARC(SLOT), ARC(1,SLOT), MXARC)
                        DY  = GET_FLX(COUNTS(OFF), FLUX(OFF))
                        DSIG= GET_ERF(COUNTS(OFF), 
     &                       ERRORS(OFF), FLUX(OFF))
                        DEV = DY - YONLY(DX)
                        IF(DEV.LT.DBLE(THRLO)*RMS*DSIG .OR.
     &                       DEV.GT.DBLE(THRHI)*RMS*DSIG) THEN
                           IF(.NOT.MUTE)
     &                          WRITE(*,*) 'Pixel ',J,' of slot ',SLOT,
     &                          ' deviated by ',
     &                          SNGL(DEV/RMS/DSIG),' sigma.'
                           IF(OUTFILE.NE.' ') THEN
                              WRITE(51,*,ERR=998) 'Pixel ',J,
     &                             ', W = ',
     &                             SNGL(DX),' deviated by ',
     &                             SNGL(DEV/RMS/DSIG),' sigma.'
                           END IF
                           NREJ = NREJ + 1
                           ERRORS(OFF) = - ERRORS(OFF)
                        END IF
                     END IF
                  END DO
               END DO
               NREJT = NREJT + NREJ
               NCYCLE = NCYCLE + 1
            END DO
C
C     Compute covariances
C
            ALAMBDA = 0.D0
            CALL MMRQMIN(VAR, NVAR, CHISQ, ALAMBDA, 1, NOK, 
     &           PMASK, WMASK, VMASK, NPMASK, NWMASK, NVMASK, 
     &           MXMASK, IFAIL)
            IF(IFAIL.NE.0) GOTO 999
            IF(OUTFILE.NE.' ') THEN
               WRITE(51,*,ERR=998) NFIT,' pixels fitted.'
               WRITE(51,*,ERR=998) NREJT,
     &              ' pixels rejected during fit.'
               WRITE(51,*,ERR=998) 'Chi**2/D of F = ',
     &              CHISQ/DBLE(NFIT-NVAR)
               WRITE(51,*,ERR=998) ' '
               WRITE(51,*,ERR=998) 'Fitted variable values'//
     &              ' along with their'
               WRITE(51,*,ERR=998) '1-param uncertainties are: '
               WRITE(51,*,ERR=998) ' '
               DO J = 1, NVAR
                  L = NVAR*(J-1)+J
                  WRITE(51,*,ERR=998) 'Variable: ',NMVAR(J),'=',
     &                 VAR(J),' +/- ',SQRT(DWORK(L,2))
               END DO      
               DO J = 1, NVAR-1
                  L  = NVAR*(J-1)
                  L1 = L+J
                  DO K = J+1, NVAR
                     L2 = NVAR*(K-1)+K
                     R = REAL(DWORK(L+K,2)/
     &                    SQRT(DWORK(L1,2)*DWORK(L2,2)))
                     WRITE(51,*,ERR=998) '<',
     &                    NMVAR(J)(:LENSTR(NMVAR(J))),'*',
     &                    NMVAR(K)(:LENSTR(NMVAR(K))),'> = ',R
                  END DO
               END DO      
            END IF
            WRITE(*,*) ' '
            WRITE(*,*) 'Final Chi**2 = ',CHISQ
            WRITE(*,*) NFIT,' pixels fitted.'
            WRITE(*,*) NREJT,' pixels rejected during fit.'
            WRITE(*,*) 'Chi**2/D of F = ',CHISQ/DBLE(NFIT-NVAR)
            WRITE(*,*) ' '
            WRITE(*,*) 'Fitted variable values along with their'
            WRITE(*,*) '1-sigma 1-param uncertainties are: '
            WRITE(*,*) ' '
            DO J = 1, NVAR
               L = NVAR*(J-1)+J
               WRITE(*,*,ERR=998) 'Variable: ',NMVAR(J),'=',
     &              VAR(J),' +/- ',SQRT(DWORK(L,2))
            END DO      
         END IF
C     
C     If SLOT3 .GT. 0, store results there.
C     
         IF(SLOT3.GT.0) THEN
            OUT = SLOT3 - 1
            DO I = 1, NOK
C     
C     Search for double precision header item Vearth. If not
C     found, assume = 0
C     
               SLOT = ISLOT(I,2)                  
               OUT  = OUT + 1
C     
C     Copy headers
C
               CALL SET_HEAD(OUT, SLOT, MXSPEC, NPIX, ARC, NARC, 
     &              MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &              HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, 
     &              NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, IFAIL)
C
               CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, 
     &              NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
               IFAIL = 0
C     
C     Get mask
C     
               CALL APPMASK(LWORK,NPIX(SLOT),ARC(1,SLOT),NARC(SLOT),
     &              MXARC,PMASK,NPMASK,WMASK,NWMASK,VMASK,NVMASK,
     &              MXMASK,.FALSE.,VEARTH,IFAIL)   
               VFAC = 1.D0 - VEARTH/DBLE(VLIGHT)
C     
C     Set parameter values
C     
               CALL SETPARS(SLOT, VAR, NVAR, MAXVAR, IFAIL)
               IF(IFAIL.NE.0) GOTO 999
C     
C     Initialise spectrum
C     
               CFR  = 1.
               OFF1 = MAXPX*(SLOT-1)
               OFF2 = MAXPX*(OUT-1)
C
               DO J = 1, NPIX(OUT)
                  OFF1 = OFF1 + 1
                  OFF2 = OFF2 + 1
                  CFR  = CFRAT(COUNTS(OFF1),FLUX(OFF1))
                  ERRORS(OFF2) = 1.E-10
                  DX  = VFAC*ANGST(REAL(J), NPIX(SLOT), 
     &                 NARC(SLOT), ARC(1,SLOT), MXARC)
                  F  = REAL(YONLY(DX))
                  COUNTS(OFF2) = CFR*F
                  IF(COUNTS(OFF2).EQ.0.) THEN
                     FLUX(OFF2) = CFR
                  ELSE
                     FLUX(OFF2) = F
                  END IF
               END DO
               IF(.NOT.MUTE) THEN
                  IF(NVAR.GT.0) THEN
                     WRITE(*,*) 'Fit to slot ',SLOT,
     &                    ' stored in slot ',OUT
                  ELSE
                     WRITE(*,*) 'Model equivalent to slot ',SLOT,
     &                    ' stored in slot ',OUT
                  END IF
               END IF
            END DO
         END IF
      END IF
      CLOSE(UNIT=51)
      IFAIL = 0
      RETURN
 998  WRITE(*,*) 'Error writing output file'
 999  IFAIL = 1
      CLOSE(UNIT=41)
      CLOSE(UNIT=51)
      RETURN
      END


