*LRUBY
* LRUBY File N1 N2 NF OPSYS -- Reads in RUBY spectra. This is a UNIX version
*                              for reading VMS RUBY files.
*
* Parameters: 
*     
*  File -- Name of RUBY file
*  N1   -- Start slot of first group to read
*  N2   -- End slot of first group to read
*  NF   -- First slot of RUBY file to read
*  OPSYS-- 1 = ULTRIX or OSF, 2 = SUNos or Solaris
*
* Selection criteria apply. 
*LRUBY
      SUBROUTINE LRUBY(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, 
     &NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &MXREAL, NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, 
     &NMSREAL, VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, 
     &MXSDOUB, MXSINTR, MXSREAL, IFAIL)
*
* Master routine for reading RUBY into MOLLY buffers
* allowing selection criteria to be applied.
*
* Note that this routine uses the non-standard FORTRAN
* BYTE type
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
* >  C*(*) NMSCHAR(MXSCHAR)   -- Names of character selection items.
* >  C*(*) VSCHAR(MXSCHAR)    -- Values of character selection items.
* >  C*(*) NMSDOUB(MXSDOUB)   -- Names of double precision selection items.
* >  D     VSDOUB(2,MXSDOUB)  -- Values of double precision selection items.
* >  C*(*) NMSINTR(MXSINTR)   -- Names of integer selection items.
* >  I     VSINTR(2,MXSINTR)  -- Values of integer selection items.
* >  C*(*) NMSREAL(MXSREAL)   -- Names of real selection items.
* >  R     VSREAL(2,MXSREAL)  -- Values of real selection items.
* >  I     NSCHAR             -- Number of character selection items.
* >  I     NSDOUB             -- Number of double precsion selection items.
* >  I     NSINTR             -- Number of integer selection items.
* >  I     NSREAL             -- Number of real selection items.
* >  I     MXSCHAR            -- Dimension of NMSCHAR in calling routine.
* >  I     MXSDOUB            -- Dimension of NMSDOUB in calling routine.
* >  I     MXSINTR            -- Dimension of NMSINTR in calling routine.
* >  I     MXSREAL            -- Dimension of NMSREAL in calling routine.   
* <  I     IFAIL              -- Error return
*
      IMPLICIT NONE
*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL, MXSCHAR, MXSDOUB, MXSINTR
      INTEGER MXSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT, NCOM
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Search parameters, names then values.
*
      CHARACTER*(*) NMSCHAR(MXSCHAR)
      CHARACTER*(*) NMSDOUB(MXSDOUB)
      CHARACTER*(*) NMSINTR(MXSINTR)
      CHARACTER*(*) NMSREAL(MXSREAL)
*
      CHARACTER*(*) VSCHAR(MXSCHAR)
      DOUBLE PRECISION VSDOUB(2,MXSDOUB)
      INTEGER VSINTR(2,MXSINTR)
      REAL VSREAL(2,MXSREAL)
*
* Local variables
*
      CHARACTER*64 FILENAME
      INTEGER SLOT1, SLOT2, FSPEC, IFAIL
      INTEGER I,   MAXSPEC, OPSYS
      LOGICAL SELECT,   DEFAULT
*
* RUBY specific parameters
*
      INTEGER NSTORE, MAXBYTE
      PARAMETER (NSTORE  = 4)
      INTEGER  JCODE, NBYTE
      INTEGER NARR, NHEAD, NPIXBYTE
      PARAMETER (MAXBYTE = 40000)
      INTEGER*2 JDATA(MAXBYTE/2)
      INTEGER JFLAG, IFRAME, IFILE
      INTEGER RECL, NBYT
      COMMON/REC_LEN/RECL, NBYT
*
      DATA FILENAME/'spectrum.rub'/
      DATA SLOT1, SLOT2, FSPEC/1,1,1/
      DATA OPSYS/2/
      SAVE FILENAME, SLOT1, SLOT2, FSPEC, OPSYS
*
      RECL = 2042
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
      CALL CHAR_IN('File name', SPLIT, NSPLIT, MXSPLIT, NCOM,
     &     DEFAULT, FILENAME, IFAIL)
      CALL INTR_IN('First slot to read into', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT1, 1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 99
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to read into', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      CALL INTR_IN('Start from which spectrum in file', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, FSPEC, 1, 1000000, IFAIL)
      CALL INTR_IN('Operating system (1=ULTRIX/OSF, 2=Solaris)',
     &     SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, OPSYS, 1, 2, IFAIL)
      IF(IFAIL.NE.0) GOTO 99
*     
*     
*     Open file for reading
*     
*     Grab a data frame --------------------------------------------
*     
      NBYTE = MAXBYTE
      JFLAG = 0
      IFRAME = FSPEC - 1
      IFILE  = 1
      DO I = SLOT1, SLOT2
 10      IFRAME = IFRAME + 1
         CALL MOLSHEC(FILENAME, JDATA, MAXBYTE, IFILE, 
     &        IFRAME, NBYTE, JFLAG, OPSYS)
         IF(JFLAG.EQ.1) THEN
            WRITE(*,*) '** Error reading RUBY file **'
            GOTO 99
         ELSE IF(JFLAG.EQ.2) THEN
            WRITE(*,*) 'Reached end-of-file'
            IFAIL = 0
            GOTO 100
         ELSE IF(JFLAG.EQ.3) THEN
            WRITE(*,*) '** Failed to open RUBY file **'
            GOTO 99
         ELSE IF(JFLAG.EQ.4) THEN
            WRITE(*,*) 'I*2 buffer in LRUBY too small'
            WRITE(*,*) 'Required size ',NBYTE
            GOTO 99
         ELSE IF(JFLAG.NE.0) THEN
            WRITE(*,*) 'Unknown error has occurred'
            GOTO 99
         END IF
*
*     Check data frame format. Only standard RUBY read
*     
         CALL CHECKFMT( JDATA, NBYTE, JCODE, OPSYS)
         IF(JCODE.NE.9) THEN
            WRITE(*,*) 'Input data frame has invalid format.'
            GOTO 99
         END IF
*     
*     Extract header information (put in common blocks) 
*     
         CALL GETHEAD(I, JDATA, MAXBYTE, NBYTE, JCODE, NARR, 
     &        NHEAD, NPIXBYTE, MXSPEC, NPIX, ARC, NARC, MXARC, 
     &        NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB,
     &        HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL,
     &        MXCHAR, MXDOUB, MXINTR, MXREAL, OPSYS, IFAIL)
         IF(IFAIL.NE.0) GOTO 99
         IF(NPIX(I).GT.MAXPX) THEN
            WRITE(*,*) 'Too many points for MOLLY'
            WRITE(*,*) NPIX(I),' pixels in spectrum'
            WRITE(*,*) MAXPX,' maximum in MOLLY.'
            NPIX(I) = 0
            GOTO 99
         END IF
         IF(NARC(I).EQ.0) THEN
            WRITE(*,*) 'Only setup for spectra with arc'
            WRITE(*,*) 'coefficients.'
            NPIX(I) = 0
            GOTO 99
         ELSE IF(NARC(I).GT.MXARC) THEN
            WRITE(*,*) 'Spectrum ', I,' has ',NARC(I),
     &           ' arc coefficients'
            WRITE(*,*) 'compared with maximum value ',MXARC
            NPIX(I) = 0
            GOTO 99
         END IF
*     
*     Finished checks, now apply selection criteria.
*     
         IF(.NOT.SELECT(I, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &        HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &        MXCHAR, MXDOUB, MXINTR, MXREAL, MXSPEC, NMSCHAR, VSCHAR, 
     &        NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, VSREAL, NSCHAR,
     &        NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, MXSINTR, 
     &        MXSREAL)) THEN
            WRITE(*,*) 'Spectrum ',IFRAME,' skipped.'
            NPIX(I) = 0
            GOTO 10
         END IF
*     
         CALL MOLSET( JDATA, JDATA(NHEAD+1), JCODE,
     &        COUNTS(1,I), ERRORS(1,I), FLUX(1,I), NPIX(I),
     &        OPSYS, IFAIL)
         IF(IFAIL.NE.0) GOTO 99
         WRITE(*,*) 'Read spectrum ',IFRAME,' into slot ',I
      END DO
      JFLAG = 1
      CALL MOLSHEC(FILENAME, JDATA, MAXBYTE, IFILE, IFRAME, 
     &     NBYTE, JFLAG, OPSYS)
      IFAIL = 0
      RETURN
 99   IFAIL = 1
 100  JFLAG = 1
      CALL MOLSHEC(FILENAME, JDATA, MAXBYTE, IFILE, IFRAME, 
     &     NBYTE, JFLAG, OPSYS)
      RETURN
      END 
      
      
      SUBROUTINE MOLSHEC(FILE,JDATA,MAXBYTE,IFILE,IFRAME,
     &NBYTE,JFLAG,OPSYS)
*
*   'MOLSHEC' reads data frames from disk
*   JFLAG = 1 on input, file, will be closed
*
      IMPLICIT NONE
      INTEGER*2 JDATA(*)
      INTEGER MAXBYTE, IFILE, IFRAME, NBYTE, JFLAG
      INTEGER LFILE, LFRAME, JFRAME, JBYTE, IUNIT
      INTEGER IOPEN, JFILE, IFAIL, OPSYS
      LOGICAL SEG
      CHARACTER*(*) FILE
      INTEGER RECL, NBYT
      COMMON/REC_LEN/RECL, NBYT
      DATA IUNIT/31/
      DATA IOPEN/0/
      DATA LFILE,LFRAME/1,0/
*
* Initialize for disk input.
*
      IF(IOPEN.EQ.1 .AND. JFLAG.EQ.1) THEN
        CLOSE(UNIT=IUNIT)
        IOPEN = 0
        RETURN
      END IF
      IF (IOPEN.EQ.0) THEN
        OPEN (UNIT=IUNIT,STATUS='OLD',FILE=FILE,
     $  FORM='UNFORMATTED', ERR=70)
        IOPEN=1
      END IF
*
* Grab frame.
*
* Commented part is old VMS version
C40    READ(IUNIT,END=50,ERR=60) JFILE,JFRAME,JBYTE
*
* UNIX code starts here
*
40    IFAIL = 0
      SEG = .FALSE.
      CALL SKIPBYTES(IUNIT,2,IFAIL)               
      NBYT = 0
*
* Now read in parameters
*
      IF(OPSYS.EQ.1) THEN
        CALL I_VMS_DEC(IUNIT,1,JFILE,SEG,IFAIL)
        CALL I_VMS_DEC(IUNIT,1,JFRAME,SEG,IFAIL)
        CALL I_VMS_DEC(IUNIT,1,JBYTE,SEG,IFAIL)
      ELSE IF(OPSYS.EQ.2) THEN
        CALL I_VMS_SUN(IUNIT,1,JFILE,SEG,IFAIL)
        CALL I_VMS_SUN(IUNIT,1,JFRAME,SEG,IFAIL)
        CALL I_VMS_SUN(IUNIT,1,JBYTE,SEG,IFAIL)
      END IF
      IF(IFAIL.LT.0) THEN
        GOTO 60
      ELSE IF(IFAIL.GT.0) THEN
        GOTO 50
      END IF
*
* UNIX code ends here
*
      LFILE =JFILE
      LFRAME=JFRAME
      IF(JBYTE.GT.MAXBYTE) THEN
        JFLAG = 4
        CLOSE(UNIT=IUNIT)
        IOPEN = 0
        NBYTE = JBYTE
        RETURN
      END IF
      IF (JFILE.EQ.IFILE.AND.JFRAME.EQ.IFRAME) THEN
        CALL READER(IUNIT,JDATA,JBYTE,JFLAG,OPSYS)
        IF(JFLAG.EQ.0) NBYTE=JBYTE
        RETURN
      END IF
      CALL SKIPBYTES(IUNIT,14,IFAIL)
C      READ (IUNIT,END=50,ERR=60)
      GOTO 40
*
*  Handle end of file condition
*
50    JFLAG=2
      CLOSE(UNIT=IUNIT)
      IOPEN = 0
      RETURN
*
*  Handle I/O error condition
*
60    JFLAG=1 
      CLOSE(UNIT=IUNIT)
      IOPEN = 0
      RETURN
*
* Handle open file error
*
70    JFLAG = 3
      CLOSE(UNIT=IUNIT)
      IOPEN = 0
      RETURN
      END
      
      SUBROUTINE READER(IUNIT,JDATA,NBYTE,JFLAG,OPSYS)
*
*  A fast record reader (no implied DO loops)
*
      IMPLICIT NONE
      INTEGER IUNIT, NBYTE, JFLAG, IFAIL, OPSYS
      LOGICAL SEG
      BYTE JDATA(NBYTE)
      INTEGER RECL, NBYT, NI2
      COMMON/REC_LEN/RECL, NBYT
*
      JFLAG=0
      CALL SKIPBYTES(IUNIT,2,IFAIL)
      SEG = NBYTE.GT.RECL
      NBYT = 0
      NI2  = NBYTE/2
      IF(OPSYS.EQ.1) THEN
        CALL I2_VMS_DEC(IUNIT,NI2,JDATA,SEG,IFAIL)
      ELSE IF(OPSYS.EQ.2) THEN
        CALL I2_VMS_SUN(IUNIT,NI2,JDATA,SEG,IFAIL)
      END IF
      IF(IFAIL.LT.0) THEN
        GOTO 20
      ELSE IF(IFAIL.GT.0) THEN
        GOTO 10
      END IF
C      READ(IUNIT,END=20,ERR=10) JDATA
      RETURN
10    JFLAG=1
      RETURN
20    JFLAG=2
      RETURN
      END

      SUBROUTINE GETHEAD(ISPEC, LOLHEDR, MAXBYTE, JBYTE, JCODE, 
     &NARR, NHEAD, NPIXBYTE, MXSPEC, NPIX, ARC, NARC, MXARC, 
     &NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, 
     &MXINTR, MXREAL, OPSYS, IFAIL)
*
*  Extracts information from the Lolita-style header.
*  KDH program modified for 'MOLLY' by T.R. Marsh. I have
*  tried to get rid of the non-standard DECODE statements
*
* There are some tricky parts because of the need to swap bytes
* for different operating systems. Since ULTRIX and VMS store data
* in almost the same way the conversionis simpler than for VMS to Solaris
*
*  Input:
*
*      ISPEC      = Which spectrum slot
*      LOLHEDR    = Data frame (header, ephemeris strip, data arrays)
*
* Output:
*
*      JBYTE      = Number of bytes 
*      JCODE      = RUBY format code
*      NARR       = Number of arrays
*      NHEAD      = Number of header bytes
*      NPIXBYTE   = Number of bytes/pixel
*
*      Molly headers
*      I*4 IFAIL      = 0 IF OK, 1 IF FAILED
*
*
      IMPLICIT NONE
      INTEGER MXSPEC,  MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
      INTEGER NC, ND, NI, NR, ISPEC, IFAIL
      INTEGER I, J, MONTH, DAY, YEAR, HOURS, MINS, SECS
      INTEGER MSECS, OPSYS
      DOUBLE PRECISION EQUINOX, RA, DEC, RJD,  DELAY, VEARTH
      DOUBLE PRECISION SIDTIM, AIRMASS, GCLONG, GCLAT, ELEV
      DOUBLE PRECISION PHI, UTC, TIMADD
      INTEGER JCODE, NARR, NHEAD, NPIXBYTE, JBYTE, MAXBYTE
      INTEGER*2 LOLHEDR(MAXBYTE/2),IDUM(64)
      INTEGER*2 I2VAL(2),OBJECT(40)
      BYTE I1VAL(4), BSWAP
      INTEGER I4VAL
      CHARACTER*128 CDUM
      EQUIVALENCE (IDUM, CDUM)
      EQUIVALENCE (I4VAL, I2VAL)
      EQUIVALENCE (I4VAL, I1VAL)
*
* Initialise numbers of header items
*
      NC = 0
      ND = 0
      NI = 0
      NR = 0
      IFAIL = 0
*
*  Determine data frame format
*
      I2VAL(1) = LOLHEDR(13)
      IF(OPSYS.EQ.1) THEN
        JCODE = I1VAL(1)
      ELSE IF(OPSYS.EQ.2) THEN
        JCODE = I1VAL(2)
      END IF
      NPIXBYTE = 2
      IF( JCODE.EQ.9 ) NPIXBYTE = 4
      NPIX(ISPEC) = LOLHEDR(27)
      NARR = LOLHEDR(28)
      NARC(ISPEC) = LOLHEDR(26)
      NHEAD = ( JBYTE - NPIX(ISPEC)*NARR*NPIXBYTE ) / 2
*
*  Night number
*
      NI = NI + 1
      NMINTR(MIN(MXINTR, NI),ISPEC) = 'Night'
      HDINTR(MIN(MXINTR, NI),ISPEC) = LOLHEDR(14)
*
*  Record number
*
      NI = NI + 1
      NMINTR(MIN(MXINTR, NI),ISPEC) = 'Record'
      HDINTR(MIN(MXINTR, NI),ISPEC) = LOLHEDR(15)
*
*  Dwell
*
      IF(OPSYS.EQ.1) THEN
        I2VAL(1) = LOLHEDR(21)
        I2VAL(2) = LOLHEDR(22)
      ELSE
        I2VAL(1) = LOLHEDR(22)
        I2VAL(2) = LOLHEDR(21)
      END IF
      NR = NR + 1
      NMREAL(MIN(MXREAL, NR),ISPEC) = 'Dwell'
      HDREAL(MIN(MXREAL, NR),ISPEC) = FLOAT(I4VAL)/LOLHEDR(29)
*
*  Date
*
      IF(OPSYS.EQ.1) THEN
        DO I = 1, 3
          IDUM(I) = LOLHEDR(I)
        END DO
      ELSE IF(OPSYS.EQ.2) THEN
        DO I = 1, 3
          I2VAL(1) = LOLHEDR(I)
          BSWAP    = I1VAL(1)
          I1VAL(1) = I1VAL(2)
          I1VAL(2) = BSWAP
          IDUM(I)  = I2VAL(1)
        END DO
      END IF
*
      READ(CDUM,'(3I2)') MONTH, DAY, YEAR
C      DECODE(6,'(3I2)', LOLHEDR(1)) MONTH, DAY, YEAR
      IF(YEAR.LT.100) YEAR=YEAR+1900
      NI = NI + 1
      NMINTR(MIN(MXINTR,NI),ISPEC) = 'Day'
      HDINTR(MIN(MXINTR,NI),ISPEC) = DAY
      NI = NI + 1
      NMINTR(MIN(MXINTR,NI),ISPEC) = 'Month'
      HDINTR(MIN(MXINTR,NI),ISPEC) = MONTH
      NI = NI + 1
      NMINTR(MIN(MXINTR,NI),ISPEC) = 'Year'
      HDINTR(MIN(MXINTR,NI),ISPEC) = YEAR
*
* UT at centre of exposure
*
      HOURS = LOLHEDR(17)
      MINS  = LOLHEDR(18)
      SECS  = LOLHEDR(19) 
      MSECS  = LOLHEDR(20) 
      UTC = TIMADD(HOURS,MINS,SECS,MSECS)
      ND = ND + 1
      NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'UTC'
      HDDOUB(MIN(MXDOUB,ND),ISPEC) = UTC
*
*  Object name
*
      DO I=1,8
        I2VAL(1) = LOLHEDR(I+4)
        IF(OPSYS.EQ.1) THEN
          OBJECT(I+I-1) = I1VAL(1)
          OBJECT(I+I)   = I1VAL(2)
        ELSE IF(OPSYS.EQ.2) THEN
          OBJECT(I+I-1) = I1VAL(2)
          OBJECT(I+I)   = I1VAL(1)
        END IF
      END DO
      NC = NC + 1
      NMCHAR(MIN(MXCHAR, NC), ISPEC) = 'Object'
      WRITE(HDCHAR(MIN(MXCHAR, NC),ISPEC),'(16A2)') 
     &(OBJECT(I),I=1,16)
*
      IF(NHEAD.EQ.32 +8*IABS(NARC(ISPEC))+64) THEN
*
*  Get ephemeris information from the header 
*
        IF(OPSYS.EQ.1) THEN
          DO I = 1, 64
            IDUM(I)  = LOLHEDR(I+32)
          END DO
        ELSE IF(OPSYS.EQ.2) THEN
          DO I = 1, 64
            I2VAL(1) = LOLHEDR(I+32)
            BSWAP    = I1VAL(1)
            I1VAL(1) = I1VAL(2)
            I1VAL(2) = BSWAP
            IDUM(I)  = I2VAL(1)
          END DO
        END IF
*
        READ(CDUM,'(F8.3,4F12.8,F8.2,F16.8,2(F12.8,F8.4),F8.6)') 
     &      EQUINOX, RA, DEC, 
     &      GCLONG, GCLAT, ELEV,
     &      RJD, DELAY, VEARTH, SIDTIM,
     &      AIRMASS, PHI
C        DECODE( 128, 1210, LOLHEDR(33)) 
C     *      EQUINOX, RA, DEC, 
C     *      GCLONG, GCLAT, ELEV,
C     *      RJD, DELAY, VEARTH, SIDTIM,
C     *      AIRMASS, PHI
C1210    FORMAT(F8.3, 2F12.8,
C     *      2F12.8, F8.2,
C     *      F16.8, F12.8, F8.4, F12.8, 
C     *      F8.4, F8.6 )
*
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Equinox'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = EQUINOX
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'RA'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = RA
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Dec'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = DEC
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Longitude'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = GCLONG
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Latitude'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = GCLAT
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Height'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = ELEV
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'RJD'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = RJD
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'HJD'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = RJD-DELAY
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Delay'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = DELAY
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Vearth'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = VEARTH
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Sidereal Time'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = SIDTIM
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Airmass'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = AIRMASS
        ND = ND + 1
        NMDOUB(MIN(MXDOUB,ND),ISPEC) = 'Orbital phase'
        HDDOUB(MIN(MXDOUB,ND),ISPEC) = PHI
      ELSE
        IFAIL = 1
        WRITE(*,*) 'Header not set properly'
        RETURN
      END IF
      IF(NC.GT.MXCHAR) THEN
        WRITE(*,*) 'Too many CHARACTER header items ',NC
        IFAIL = 1
        RETURN
      END IF
      IF(ND.GT.MXDOUB) THEN
        WRITE(*,*) 'Too many DOUBLE PRECISION header items ',ND
        IFAIL = 1
        RETURN
      END IF
      IF(NI.GT.MXINTR) THEN
        WRITE(*,*) 'Too many INTEGER header items ',NI
        IFAIL = 1
        RETURN
      END IF
      IF(NR.GT.MXREAL) THEN
        WRITE(*,*) 'Too many REAL header items ',NR
        IFAIL = 1
        RETURN
      END IF
      NCHAR(ISPEC) = NC
      NDOUB(ISPEC) = ND
      NINTR(ISPEC) = NI
      NREAL(ISPEC) = NR
*
*  Decode arcfit coefficients 
*
      NC = ABS(NARC(ISPEC))
      DO I = 1, NC
        IF(OPSYS.EQ.1) THEN
          DO J = 1, 8
            IDUM(J) = LOLHEDR(NHEAD+8*(I-1-NC)+J)
          END DO
        ELSE IF(OPSYS.EQ.2) THEN
          DO J = 1, 8
            I2VAL(1) = LOLHEDR(NHEAD+8*(I-1-NC)+J)
            BSWAP    = I1VAL(1)
            I1VAL(1) = I1VAL(2)
            I1VAL(2) = BSWAP
            IDUM(J)  = I2VAL(1)
          END DO
        END IF
        READ(CDUM,'(D16.10)') ARC(I,ISPEC)
      END DO
C      ISTART = NHEAD - 8*NC + 1
C      DECODE( 16*NC, 1220, LOLHEDR(ISTART)) (ARC(I,ISPEC),I=1,NC)
C1220  FORMAT(<NC>D16.10)
      RETURN
      END

      SUBROUTINE MOLSET( LOLHEDR, JDATA, JCODE, COUNTS, 
     &ERRORS, FLUX, NPIX, OPSYS, IFAIL)
*
* Converts different RUBY/LOLITA style data formats to
* MOLLY internal format
*
*  Input:
*
*      LOLHEDR    = Lolita-style header from external format
*      JDATA      = Spectrum data in external format
*      JCODE      = Format code:
*                  Only handles
*                 = 9 for output from 'RUBY' in REAL*4 format
*
*  Output: 
*
*        COUNTS(NPIX)   -- Raw counts
*        ERRORS(NPIX)   -- 1-sigma errors
*        FLUX(NPIX)     -- MilliJansky fluxes
*        IFAIL          -- Error return
*
      IMPLICIT NONE
      INTEGER NPIX, JCODE, NBAD, MAXWK
      REAL COUNTS(NPIX), ERRORS(NPIX), FLUX(NPIX)
      PARAMETER (MAXWK=2500)
      REAL WORK(MAXWK)
      INTEGER*2 LOLHEDR(1)
      INTEGER*2 JDATA(*)
      INTEGER I, IFAIL, OPSYS
*
* JCODE = 9 REAL*4 'RUBY' OUTPUT 
*
      IF( JCODE.EQ.9 ) THEN
*
C900     NBYTES = NPIX*4
        IF(NPIX.GT.MAXWK) THEN
          WRITE(*,*) 'Work array in MOLSET too small'
          IFAIL = 1
          RETURN
        END IF
*
* UNIX code. Translate VMS REAL*4 numbers
*
        IF(OPSYS.EQ.1) THEN
          CALL VMS_DEC_REAL(JDATA          , COUNTS, NPIX )
          CALL VMS_DEC_REAL(JDATA(2*NPIX+1), ERRORS, NPIX )
          CALL VMS_DEC_REAL(JDATA(4*NPIX+1),   WORK, NPIX )
          CALL VMS_DEC_REAL(JDATA(6*NPIX+1),   FLUX, NPIX )
        ELSE IF(OPSYS.EQ.2) THEN
          CALL VMS_SUN_REAL(JDATA          , COUNTS, NPIX )
          CALL VMS_SUN_REAL(JDATA(2*NPIX+1), ERRORS, NPIX )
          CALL VMS_SUN_REAL(JDATA(4*NPIX+1),   WORK, NPIX )
          CALL VMS_SUN_REAL(JDATA(6*NPIX+1),   FLUX, NPIX )
        END IF
*
* End
*
C        CALL MBYTMOVE(JDATA          , COUNTS, NBYTES )
C        CALL MBYTMOVE(JDATA(2*NPIX+1), ERRORS, NBYTES )
C        CALL MBYTMOVE(JDATA(4*NPIX+1),   WORK, NBYTES )
C        CALL MBYTMOVE(JDATA(6*NPIX+1),   FLUX, NBYTES )
*
* Change from RUBY to MOLLY format.
*
        NBAD = 0
        DO I = 1, NPIX
          IF(COUNTS(I).EQ.0.)  THEN
            IF(WORK(I).EQ.0.) THEN
              NBAD = NBAD + 1
              WRITE(*,*) 'Pixel ',I,
     &        ' does not have valid RUBY format'
              WRITE(*,*) 'C,E,S,F = ',
     &        COUNTS(I),ERRORS(I),WORK(I),FLUX(I)
              FLUX(I)   = 1.
              COUNTS(I) = 0.
              ERRORS(I) = 1.
            ELSE
*
* In old versions this was incorrectly inverted. Bug spotted by
* Bill Welsh, Oct 1994
*
              FLUX(I) = WORK(I)/FLUX(I)
            END IF
          END IF
          IF(ERRORS(I).LE.0.) THEN
            ERRORS(I) = -1.
            FLUX(I)   = 0.
          ELSE IF(FLUX(I).EQ.0.) THEN
            ERRORS(I) = -1.
          END IF
        END DO
        IF(NBAD.GT.0) THEN
          WRITE(*,*) NBAD,' bad points masked by MOLSET.'
          WRITE(*,*) 'This problem has only ever been seen in HST'
          WRITE(*,*) 'and an HST specific correction has been made'
          WRITE(*,*) 'which may not be valid for any other RUBY'
          WRITE(*,*) 'data. You have been warned!'
        END IF
      END IF
      IFAIL = 0
      RETURN
      END

      SUBROUTINE MBYTMOVE(SOURCE,DEST,NBYTES)
*
*  MOVES NBYTES FROM SOURCE TO DEST
*
      IMPLICIT NONE
      INTEGER I, NBYTES
      BYTE SOURCE(NBYTES), DEST(NBYTES)
*
      DO I=1,NBYTES
        DEST(I) = SOURCE(I)
      END DO
      RETURN
      END

	SUBROUTINE CHECKFMT( JDATA, JBYTE, JCODE, OPSYS )
*
*  IDENTIFIES FORMAT OF RAW DATA FRAME
*
*  Inputs:
*      JDATA      = RAW DATA FRAME (HEADER, EPHEMERIS STRIP, DATA ARRAYS)
*      JBYTE      = NUMBER OF BYTES IN RAW DATA FRAME
*  Outputs:
*      JCODE   = DATA FORMAT CODE
*              =-1 FOR INVALID FORMATS
*              = 0 FOR RAW SHECTOGRAPH DATA
*              = 1 FOR 1D SPECTRA FROM LOLITA
*              = 2 FOR SIT NET COUNT DATA
*              = 3 FOR HIGH-SPEED IPCS DATA FROM FRAMER
*              = 4 FOR SIT LINEARIZED AB DATA
*              = 5 FOR 1D SPECTRA FROM PAMELA
*              = 7 FOR 1D SPECTRA CONVERTED TO RUBY FORMAT
*              = 8 FOR 1D SPECTRA FROM RUBY (OLD INTEGER FORMAT)
*              = 9 FOR 1D SPECTRA FROM RUBY (NEW REAL*4 FORMAT)
*              = 10 For FIGARO spectra (count data)
*              = 11 For FIGARO spectra (fnu data)
*              = 12 For FIGARO spectra (flambda data)
*
* Modified by TRM for FIGARO June 88
*
      INTEGER OPSYS
      INTEGER*2 JDATA(*)
      INTEGER*2 I2VAL
      BYTE I1VAL(2)
      EQUIVALENCE (I2VAL, I1VAL)
*
*   CHECK FOR CORRECT NUMBER OF BYTES
*
      IF ( (JBYTE/2)*2 .NE. JBYTE 
     *      .OR. JBYTE .LE. 64 ) THEN
        WRITE(*,*) '** JBYTE MUST BE EVEN AND 64 < JBYTE'
10      JCODE = -1
        RETURN
      END IF
C
C   IDENTIFY DATA FORMAT CODE -----------------------------------
C
      I2VAL = JDATA(13)
      IF(OPSYS.EQ.1) THEN
        JCODE = I1VAL(1)
      ELSE IF(OPSYS.EQ.2) THEN
        JCODE = I1VAL(2)
      END IF
*
      NBYTEPIX = 2
      IF(JCODE.EQ.0) THEN
        WRITE(*,*) 'Raw Shectograph data.'
*
*  FIX FAULTY HEADER DATA THAT MIGHT CAUSE A CRASH.
*
      IF(JDATA(30).NE.0) THEN
        WRITE(*,*) 'Invalid CFAC value',JDATA(30),' changed to 0'
      END IF
      IF(JDATA(16).NE.0.AND.JDATA(16).NE.1000) THEN
        WRITE(*,*) 'Invalid IAB value',JDATA(16),' changed to 1000'
      END IF
      IF(JDATA(26).NE.0) THEN
        WRITE(*,*) 'Invalid NARC value',JDATA(26),' changed to 0'
      END IF
      JDATA(30)=0
      JDATA(16)=1000
      JDATA(26)=0
*
      ELSE IF(JCODE.EQ.1) THEN
        WRITE(*,*) 'Data from ''LOLITA'''
*
      ELSE IF(JCODE.EQ.2) THEN
        WRITE(*,*) 'SIT net-count data'

      ELSE IF(JCODE.EQ.3) THEN
        WRITE(*,*) 'High-speed IPCS data from ''FRAMER'''
*
      ELSE IF(JCODE.EQ.4) THEN
        WRITE(*,*) 'SIT linearized AB data.  AB/DN =',JDATA(16)
*
      ELSE IF(JCODE.EQ.5) THEN
        WRITE(*,*) 'Data from ''PAMELA'''
*
      ELSE IF(JCODE.EQ.7) THEN
        WRITE(*,*) '''RUBY'' format data, but not from ''RUBY'''
*
      ELSE IF(JCODE.EQ.8) THEN
        WRITE(*,*) 'Data from ''RUBY'' in old INTEGER*2 format'
*
      ELSE IF(JCODE.EQ.9) THEN
        WRITE(*,*) 'Data from ''RUBY'' in REAL*4 format'
        NBYTEPIX = 4
      ELSE IF(JCODE.EQ.10) THEN
        WRITE(*,*) 'Count data from ''FIGARO'''
        NBYTEPIX = 4
      ELSE IF(JCODE.EQ.11) THEN
        WRITE(*,*) 'Fnu data from ''FIGARO'''
        NBYTEPIX = 4
      ELSE IF(JCODE.EQ.12) THEN
        WRITE(*,*) 'Flambda data from ''FIGARO'''
        NBYTEPIX = 4
      ELSE
        WRITE(*,*) 'Unknown data format code = ',JCODE
        GOTO 10
      END IF
*
      NPIX=JDATA(27)
      NARR=JDATA(28)
      NARC=JDATA(26)
      NHEAD = (JBYTE - NPIX*NARR*NBYTEPIX) / 2
      MNHEAD = 32
*
      IF( NPIX.LT.1 .OR. NARR.LT.1 .OR. NHEAD.LT.MNHEAD) THEN
        WRITE(*,*) 'Invalid frame dimensions in header.'
        WRITE(*,'(A,I5,A,I3,A,I4,A,I2)') 
     &' NPIX: ',NPIX,', NARR: ',NARR,', NHEAD: ',NHEAD,', NARC: ',NARC
        GOTO 10
      END IF
      RETURN
      END

      SUBROUTINE VMS_DEC_REAL(B,R,N)
*
* Translates bytes array B(4*N) into real array R(N) assuming that 
* the bytes are VMS REAL*4 numbers
*
* For VMS ---> Decstations, alphas
*
      INTEGER N, J, I
      REAL R(N), RVAL
      BYTE B(4*N),B1(4)
      EQUIVALENCE (RVAL,B1(1))
*
      DO I = 1, N
        J = 4*(I-1)
        B1(1) = B(J+3)
        B1(2) = B(J+4)
        B1(3) = B(J+1)
        B1(4) = B(J+2)
        R(I)  = RVAL/4.
      END DO
      RETURN
      END

      SUBROUTINE VMS_SUN_REAL(B,R,N)
*
* Translates bytes array B(4*N) into real array R(N) assuming that 
* the bytes are VMS REAL*4 numbers that have been swapped in pairs
* as a result of being read in as I*2 numbers
*
* For VMS ---> Suns
*
      INTEGER N, J, I
      REAL R(N), RVAL
      BYTE B(4*N),B1(4)
      EQUIVALENCE (RVAL,B1(1))
*
      DO I = 1, N
        J = 4*(I-1)
        B1(4) = B(J+4)
        B1(3) = B(J+3)
        B1(2) = B(J+2)
        B1(1) = B(J+1)
        R(I)  = RVAL/4.
      END DO
      RETURN
      END






