*LLIG
* LLIG FILE N1 CF CE CH FAC W0 BP -- Reads a light curve from a file as a 
*                       series of spectra with one point each.
*
* Parameters
*         FILE -- Name of ASCII file with data
*         N1   -- First slot for results
*         CF   -- Column number with fluxes
*         CE   -- Column number with errors
*         CH   -- Column with HJDs. 0 to ignore
*         FAC  -- Multiplier to convert fluxes/errors to mJy
*         W0   -- Central wavelength in A
*         BP   -- Band pass in A
*
* This is not meant to replace a dedicated light curve analysis package. It
* is for one thing grossly inefficient in terms of storage with many header
* parameters/per point. However, it is occasionally of use for consistency
* with spectrum analysis. If "Fluxes" are actually in counts, this routine
* wills et the counts correctly.
*LLIG
      SUBROUTINE LIGRED(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC,
     &NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &MXREAL, IFAIL)
*
* Reads in a light curve as spectra with one point each.
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
* >  I     IFAIL     -- O if read ok
*
      IMPLICIT NONE
*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
*
* Local variables
*
      CHARACTER*64 FILENAME
      CHARACTER*100 STRING
      INTEGER SLOT1,  IFAIL, LUNIT
      INTEGER I,   MAXSPEC
      INTEGER  NP, NCOM
      REAL  FFAC
      LOGICAL  DEFAULT
      DOUBLE PRECISION T(20)
      INTEGER  CW, CF, CH, CMAX, CE, SLOT
      REAL BPASS, WCEN
*
* Functions
*
      DATA FILENAME/'light.dat'/
      DATA SLOT1, CW, CF, CE, CH/1,1,2,3,4/
      DATA WCEN, BPASS, FFAC/5000., 100., 1./
      DATA LUNIT/23/
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL CHAR_IN('File name', SPLIT, NSPLIT, MXSPLIT, NCOM,
     &DEFAULT, FILENAME, IFAIL)
      CALL INTR_IN('First slot to load', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, SLOT1, 1, MAXSPEC, IFAIL)
      CALL INTR_IN('Flux column', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, CF, 1, 20, IFAIL)
      CALL INTR_IN('Errors column', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, CE, 1, 20, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(CE.EQ.CF) THEN
        WRITE(*,*) 'Can''t have same column for errors'
        WRITE(*,*) 'and fluxes'
        IFAIL = 1
        GOTO 999
      END IF
*
      CALL INTR_IN('HJD column', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, CH, 0, 20, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(CH.EQ.CE .OR. CH.EQ.CF) THEN
        WRITE(*,*) 'Can''t have same column for wavelengths'
        WRITE(*,*) 'and fluxes and HJDs'
        IFAIL = 0
        GOTO 999
      END IF
      CMAX = MAX(CW, CF)
      CMAX = MAX(CMAX, CE)
      CALL REAL_IN('Multiplier to mJy', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, FFAC, -1.E25, 1.E25, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      CALL REAL_IN('Central wavelength', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, WCEN, 1.E-6, 1.E6, IFAIL)
      CALL REAL_IN('Band pass width', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, BPASS, 1.E-8, 1.E6, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
*
***************************************************************************
*
* Open file for reading
*
      OPEN(UNIT=LUNIT,FILE=FILENAME,FORM='FORMATTED',
     &     STATUS='OLD',IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(*,*) 'Could not open ',FILENAME
        GOTO 999
      END IF
*
* Start to read file
*
      NP = 0
      SLOT = SLOT1 - 1
100   READ(LUNIT,'(A)',IOSTAT=IFAIL) STRING
*
      IF(IFAIL.EQ.0) THEN
        READ(STRING,*,IOSTAT=IFAIL) (T(I),I=1,CMAX)
        IF(IFAIL.EQ.0) THEN
          SLOT = SLOT + 1
          NPIX(SLOT) = 1
          NARC(SLOT) = 2
          ARC(2,SLOT) = BPASS
          ARC(1,SLOT) = WCEN - BPASS
          COUNTS(1,SLOT) = REAL(T(CF))       
          ERRORS(1,SLOT) = REAL(T(CE))       
          IF(T(CF).EQ.0.) THEN
            FLUX(1,SLOT) = FFAC
          ELSE
            FLUX(1,SLOT) = REAL(FFAC*T(CF))
          END IF
          IF(CH.GT.0) CALL HSETD('HJD', T(CH), SLOT, MXSPEC, 
     &    NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
          WRITE(*,*) 'Slot ',SLOT,', flux = ',FFAC*SNGL(T(CF)),
     &    '+/-',FFAC*SNGL(T(CE))
          NP = NP + 1
        END IF
        GOTO 100
      ELSE IF(IFAIL.GT.0) THEN
        WRITE(*,*) 'Error reading file'
        GOTO 999
      END IF  
      WRITE(*,*) NP,' points read'
      IF(NP.EQ.0) THEN
        WRITE(*,*) '** NO DATA **'
        GOTO 999
      END IF
      IFAIL = 0
999   CLOSE(UNIT=LUNIT)
      RETURN
      END
