*LASC
* LASC FILE N CW CF CE WUNITS FUNITS FRAC -- Reads an ASCII file as a spectrum
*
* Parameters
*       FILE  -- Name of ASCII file with data in column form
*       N     -- Slot to read it into
*       CW    -- Column with wavelengths in A
*       CF    -- Column with fluxes
*       CE    -- Column with errors 
*               0 to ignore, in which case they will be 
*                 set to make S/N = 10000.
*              <0 to make the routine estimate errors from fluctuations.
*
*       WUNITS-- Wavelength units 'A' or 'mu' (microns) 
*       FUNITS-- Flux units 'MJY' or 'FLAMBDA' (ergs/cm**2/s/A)
*       FRAC  -- The wavelength array is fitted with a series of polys
*                of increasing order until one is found which produces
*                a maximum discrepancy less than FRAC pixels.
*LASC
      SUBROUTINE LASC(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC,
     &NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &MXREAL, WORK1, WORK2, WORK3, MXWORK, IFAIL)
*
* Reads in ASCII file with wavelength, flux and error columns.
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
* >  I     IFAIL     -- O if read ok
*
      IMPLICIT NONE
*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Work arrays
*
      INTEGER MXWORK
      DOUBLE PRECISION WORK1(MXWORK), WORK2(MXWORK)
      DOUBLE PRECISION WORK3(MXWORK)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Local variables
*
      CHARACTER*64 FILENAME
      CHARACTER*16 FUNITS, WUNITS
      INTEGER SLOT1,  IFAIL, LUNIT
      INTEGER I, J,  MAXSPEC
      INTEGER  NP, NCOM
      REAL    WAV, FRACTION
      LOGICAL  DEFAULT, SAMECI
      REAL  VLIGHT, CGS, FAC, CFAC,  T(20)
      INTEGER  NPOLY, MPOLY, CW, CF, CE, CMAX, MAXPOLY
      PARAMETER (MAXPOLY=20)
      DOUBLE PRECISION FIT(MAXPOLY), XM(MAXPOLY,2*MAXPOLY+3)
      DOUBLE PRECISION DMAX, CHISQ, POLY, DEV
*
* Functions
*
      DATA FILENAME/'spectrum.mol'/
      DATA SLOT1, CW, CF, CE/1,1,2,3/
      DATA LUNIT, FRACTION/23, 5.E-3/
      DATA FUNITS, WUNITS/'MJY','A'/
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL CHAR_IN('File name', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, FILENAME, IFAIL)
      CALL INTR_IN('Slot to load', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, SLOT1, 1, MAXSPEC, IFAIL)
      CALL INTR_IN('Wavelength column', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, CW, 1, 20, IFAIL)
      CALL INTR_IN('Flux column', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, CF, 1, 20, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(CF.EQ.CW) THEN
        WRITE(*,*) 'Can''t have same column for wavelengths'
        WRITE(*,*) 'and fluxes'
        IFAIL = 1
        GOTO 999
      END IF
      CALL INTR_IN('Errors column', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, CE, -1000, 20, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(CE.EQ.CW .OR. CE.EQ.CF) THEN
        WRITE(*,*) 'Can''t have same column for wavelengths'
        WRITE(*,*) 'and fluxes and errors'
        IFAIL = 1
        GOTO 999
      END IF
      CMAX = MAX(CW, CF)
      CMAX = MAX(CMAX, CE)
*
      CALL CHAR_IN('Wavelength units (A, MU)', SPLIT, NSPLIT, 
     &MXSPLIT, NCOM, DEFAULT, WUNITS, IFAIL)
      IF(.NOT.SAMECI(WUNITS,'A') .AND. 
     &   .NOT.SAMECI(WUNITS,'MU')) THEN
        WRITE(*,*) 'Valid types:'
        WRITE(*,*) ' '
        WRITE(*,*) ' A   -- Angstroms'
        WRITE(*,*) ' MU  -- Microns'
        WUNITS = 'A'
        IFAIL = 1
        GOTO 999
      END IF
      CALL CHAR_IN('Flux units (MJY, FLAM)', SPLIT, NSPLIT, 
     &MXSPLIT, NCOM, DEFAULT, FUNITS, IFAIL)
      IF(.NOT.SAMECI(FUNITS,'MJY') .AND. 
     &   .NOT.SAMECI(FUNITS,'FLAM')) THEN
        WRITE(*,*) 'Valid types:'
        WRITE(*,*) ' '
        WRITE(*,*) ' MJY -- Millijanskys'
        WRITE(*,*) ' FLAM - Flambda ergs/cm**2/s'
        FUNITS = 'MJY'
        IFAIL = 1
        GOTO 999
      END IF
      CALL REAL_IN('Wavelength tolerance (pixels)', SPLIT, 
     &NSPLIT, MXSPLIT, NCOM, DEFAULT, FRACTION, 0., 1.E5, IFAIL)
*
* Open file for reading
*
      OPEN(UNIT=LUNIT,FILE=FILENAME,FORM='FORMATTED',
     &     STATUS='OLD',IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(*,*) 'Could not open ',FILENAME
        GOTO 999
      END IF
*
* Start to read file
*
      VLIGHT = CGS('C')
      CFAC = 1.E18/VLIGHT
      NP = 0
100   READ(LUNIT,*,IOSTAT=IFAIL) (T(I),I=1,CMAX)
*
      IF(IFAIL.LT.0 .OR. NP.EQ.MAXPX) THEN
        IF(IFAIL.EQ.0 .AND. NP.EQ.MAXPX) THEN
          WRITE(*,*) 'Maximum spectrum length of ',MAXPX
          WRITE(*,*) 'has been reached. Ignoring further data'
        END IF
*
* Have reached end of spectrum.
* Store number of pixels, fit wavelength scale,
* convert to mJy if need be.
*
        NPIX(SLOT1) = NP
        DO I = 1, NP
          WORK1(I) = WORK1(I)/DBLE(NP)
        END DO
        NPOLY = 1
        MPOLY = MIN(NP-1, MAXPOLY)
150     CONTINUE
        NPOLY = NPOLY + 1
        CALL LEASQ(WORK1, WORK2, WORK3, NP, NPOLY, 
     &  FIT, CHISQ, XM, 1)
        DMAX = -1.D0
        DO I = 1, NP
          DEV = ABS(WORK2(I)-POLY(FIT,NPOLY,WORK1(I)))
          IF(DEV.GT.DMAX) THEN
            DMAX = DEV
            J = I
          END IF
        END DO
        DMAX = DMAX*DBLE(NP)/ABS(WORK2(NP)-WORK2(1))
        WRITE(*,*) 'Slot ',SLOT1,', NPOLY = ',NPOLY,
     &  ', Max dev = ',SNGL(DMAX),' pixels.'
        IF(DMAX.GT.FRACTION .AND. NPOLY.LT.MPOLY) THEN
          WRITE(*,*) 'Max dev for wavelength = ',WORK2(J)
          GOTO 150
        END IF
        IF(DMAX.GT.FRACTION) THEN
          WRITE(*,*) 'Could not fit X scale'
          WRITE(*,*) 'Will assume pixel scale instead'
          NARC(SLOT1) = 0
        ELSE
          WRITE(*,*) 'Acceptable fit found'
          NARC(SLOT1) = NPOLY
          DO I = 1, NPOLY
            IF(SAMECI(WUNITS,'A')) THEN
              ARC(I,SLOT1) = FIT(I)
            ELSE IF(SAMECI(WUNITS,'MU')) THEN
              ARC(I,SLOT1) = 1.D4*FIT(I)
            END IF
          END DO
        END IF
        IF(SAMECI(FUNITS,'FLAM')) THEN
          DO J = 1, NP
            WAV = REAL(WORK2(J))
            FAC = CFAC*WAV**2
            FLUX(J,SLOT1)   = FAC*FLUX(J,SLOT1)
            ERRORS(J,SLOT1) = FAC*ERRORS(J,SLOT1)
          END DO
        END IF
*
* Fake up counts array
*
        DO J = 1, NP
          IF(FLUX(J,SLOT1).EQ.0) THEN
            FLUX(J,SLOT1) = 1.
            COUNTS(J,SLOT1) = 0.
          ELSE
            COUNTS(J,SLOT1) = FLUX(J,SLOT1)
          END IF
        END DO
*
* If no error column has been read, fake them up from
* variation in counts
*
        IF(CE.LT.0) THEN
          CALL SIGFAKE(NP, COUNTS(1,SLOT1), ERRORS(1,SLOT1))
        END IF
*
      ELSE IF(IFAIL.EQ.0) THEN
*
* Store another point, return for more.
* Skip repeated points
*
        IF(NP.EQ.0 .OR.T(CW).NE.WORK2(NP)) THEN
          NP = NP + 1
          WORK1(NP)     = DBLE(NP)
          WORK2(NP)     = T(CW)
          WORK3(NP)     = 1.D0
          FLUX(NP,SLOT1)  = T(CF)
          IF(CE.GT.0) THEN
            ERRORS(NP,SLOT1) = T(CE)
          ELSE
            ERRORS(NP,SLOT1)  = T(CF)/10000.
          END IF
        END IF
        GOTO 100
      ELSE IF(IFAIL.GT.0) THEN
        WRITE(*,*) 'Error during read'
        GOTO 999
      END IF
      IFAIL = 0
999   CLOSE(UNIT=LUNIT)
      RETURN
      END

