CHFIX
C HFIX N1 N2 TELESCOPE -- Computes HJD, heliocentric corrections etc. 
C                         Some of these are used by MOLLY routines (e.g. 
C                         Vearth), so apply HFIX as soon as possible in 
C                         your use of MOLLY. 
C
C NB: the following header items must be present for HFIX to operate:
C
C C Object -- Object name
C D UTC    -- UTC at middle of exposure
C I Day    -- Day of month
C I Month  -- Month 
C I Year   -- Year (as in 1994)
C
C (RJD can substitute for UTC and date)
C
C You will be prompted for object coordinates if they are not present.
C (but see TELESCOPE below for a special option for arcs)
C
C Parameters: 
C     
C  N1        -- Slot number of first spectrum to plot.
C  N2        -- Slot number of last spectrum to plot.
C  TELESCOPE -- Name of observing site/telescope. The program first looks 
C          for the longitude, latitude and height of the observing site in the
C          headers. If all are present, they will be used. If any are
C          not present, the site as set by the user will be used.
C
C          If you enter "SKIP", no attempt to derive coordinates will be made.
C          This is useful if you just want to set the RJD for arcs but can't be
C          bothered to get their coord right.
C
C No selection. 
C
C Format of object coordinates if read from a file:
C
C IP Peg
C 1950. 23 34 56 +18 06 06 24494056.8492 0.1578033
C Another star
C 1950. 00 03 45 -00 -06 05 2445000. 1.000
C
C Note the need for a negative sign on minutes of dec in second example
C because degrees are zero.
C
CHFIX
CPHASE
C  PHASE N1 N2 Name ZERO PERIOD PDOT -- Sets phases on spectra.
C
C  N1 -- First slot to set
C  N2 -- Last slot to set
C  Name- Name of phase e.g. 'Orbital phase'. Only the following 
C        names will be correctly averaged in other routines and so
C        only these are permitted inside 'phase'.
C
C        'Orbital phase', 'Spin phase', 'Beat phase', 'Hump phase',
C        'Phase1', 'Phase2', 'Phase3'. 
C
C         O,S,B,H are accepted as short forms for the first 4, and 
C         '1', '2' and '3' for the last 3. The name will be expanded
C         out in each case.
C
C  ZERO   -- Constant part of ephemeris (=phase 0)
C  PERIOD -- Period in days. 
C  PDOT   -- Quadratic term (i.e. the coefficient of phase**2 in the,
C            ephemeris, not precisely Pdot)
C
C The ephemeris used for the phases is set in the headers in case you
C forget what you did later. A suffix (O,S,B,H,1,2,3) is added to 
C distinguish between the various standard phases, so that ZeroO
C is the constant term of the orbital ephemeris.
C
CPHASE
      SUBROUTINE HEDFIX(SPLIT, NSPLIT, MXSPLIT, MXBUFF, MXSPEC, 
     &     MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, 
     &     NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &     NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, PHASE,
     &     SLOTS, MXSLOTS, MUTE, IFAIL)
C     
C Fixs incomplete headers so that have RA, Dec, HJD etc.
C Fairly interactive
C 
C Arguments:
C
C >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C                              zero in which case defaults are used.
C >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C >  I     MXSPEC           -- Maximum number of spectra allowed by size
C                              of header buffers. 
C >  I     MAXPX            -- Maximum number of pixels/spectrum
C <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C <  I     NCHAR(MXSPEC)  -- Number of character header items.
C <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C <  I     NINTR(MXSPEC)  -- Number of integer items.
C <  I     NREAL(MXSPEC)  -- Number of real items.
C >  I     MXCHAR    -- Maximum number of character header items/spec
C >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C >  I     MXINTR    -- Maximum number of integer header items/spec
C >  I     MXREAL    -- Maximum number of real headers items/spec
C >  L     PHASE     -- TRUE means fixes up phases only.
C    I     SLOTS(MXSLOTS) -- List buffer
C    I     MXSLOTS       
C >  I     IFAIL     -- Error return. 0 ok
C
      IMPLICIT NONE
C
C Integer parameters
C
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C
C Numbers of header items
C
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C
C Names of header items
C
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C
C Values of header items
C
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C
C Arc coefficients
C
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C
C Slot lists
C
      INTEGER MXSLOTS, NSLOTS, SLOT
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
C
C Command parameters
C
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C
      LOGICAL PHASE, MUTE
C
C Local variables
C
      CHARACTER*100 STRING
      CHARACTER*32 NAME, OBJECT
      INTEGER SLOT1, SLOT2, IFAIL, NCOM
      INTEGER I,   MAXSPEC, LENSTR
      LOGICAL  SAMECI, ESAMECI, DEFAULT
      LOGICAL POSOK, NOK
      DOUBLE PRECISION ZERO, PERIOD, PDOT, TEST
      DOUBLE PRECISION HJD1, PER1, PHS, PHSO
      INTEGER NR, N
C
      INTEGER YEAR, MONTH, DAY
      DOUBLE PRECISION RJD, HJD, UTC, RA, DEC, EQUINOX
      DOUBLE PRECISION LONG, LAT, HEIGHT, SIDTIM, HA
      CHARACTER*32 SITE, TELESCOPE
      DOUBLE PRECISION GLONG, GLAT, DELAY, VEARTH
      DOUBLE PRECISION EL, AZ, ELR, AIRMASS
      COMMON/FRUIT6/GLONG,GLAT
      COMMON/FRUIT7/RJD,HJD,DELAY,VEARTH
      COMMON/FRUIT8/SIDTIM,HA
      COMMON/FRUIT9/EL,AZ,ELR,AIRMASS
C
      DATA SLOT1, SLOT2/1,1/
      DATA ZERO, PERIOD, PDOT/2440000.0, 0.1000, 0./
      DATA NAME, TELESCOPE, SITE/'Orbital phase', 'INT', 'La Palma'/
C
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
C
      CALL INTR_IN('First slot to fix', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to fix', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST.GT.0) THEN
            CALL SETSLOT(LIST, NLIST, MXLIST, SLOTS, NSLOTS, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra to fix'
            CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &           NSLOTS, MXSLOTS)
         END IF
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &        NSLOTS, MXSLOTS)
      END IF
      IF(NSLOTS.EQ.0) GOTO 999
C
C     Special section for setting phases
C
      SLOT = SLOTS(1)
      IF(PHASE) THEN
        CALL CHAR_IN('Enter phase to compute', SPLIT, NSPLIT, MXSPLIT, 
     &  NCOM, DEFAULT, NAME, IFAIL)
C
C     Recognise and expand short forms
C
        IF(SAMECI(NAME,'Orbital phase')) NAME = 'Orbital phase'
        IF(SAMECI(NAME,'Spin phase'))    NAME = 'Spin phase'
        IF(SAMECI(NAME,'Beat phase'))    NAME = 'Beat phase'
        IF(SAMECI(NAME,'Hump phase'))    NAME = 'Hump phase'
        IF(SAMECI(NAME,'1'))             NAME = 'Phase1'
        IF(SAMECI(NAME,'2'))             NAME = 'Phase2'
        IF(SAMECI(NAME,'3'))             NAME = 'Phase3'
C
C     Check that name is one of recognised forms
C
        IF(.NOT.SAMECI(NAME,'Orbital phase') .AND.               
     &       .NOT.SAMECI(NAME,'Spin phase') .AND.               
     &       .NOT.SAMECI(NAME,'Beat phase') .AND.               
     &       .NOT.SAMECI(NAME,'Hump phase') .AND.               
     &       .NOT.SAMECI(NAME,'Phase1') .AND.               
     &       .NOT.SAMECI(NAME,'Phase2') .AND.               
     &       .NOT.SAMECI(NAME,'Phase3')) THEN
           WRITE(*,*) NAME,' is not one of the recognised'
           WRITE(*,*) 'phases. See help on phase for more info.'
           NAME  = 'Orbital phase'
           GOTO 999
        END IF
        
        CALL DOUB_IN('Zero (Heliocentric Julian date)', SPLIT, 
     &       NSPLIT, MXSPLIT, NCOM, DEFAULT, ZERO, 0.D0, 1.D7, IFAIL)
        CALL DOUB_IN('Period (days)', SPLIT, NSPLIT, MXSPLIT, 
     &       NCOM, DEFAULT, PERIOD, 0.D0, 1.D4, IFAIL)
        CALL DOUB_IN('Quadratic term (days)', SPLIT, NSPLIT, 
     &       MXSPLIT, NCOM, DEFAULT, PDOT, -1.D3, 1.D3, IFAIL)
        IF(IFAIL.NE.0) GOTO 999
        
      ELSE
         CALL CHAR_IN(
     &        'Telescope (''list'' for list, ''skip'' to ignore)', 
     &        SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, TELESCOPE, IFAIL)
         IF(.NOT. ESAMECI(TELESCOPE, 'SKIP')) 
     &        CALL SITEQ(TELESCOPE, SITE, LAT, LONG, HEIGHT)
      END IF
C
C     Loop through headers
C
      N = 0
      DO I = 1, NSLOTS
        SLOT = SLOTS(I)
C
C     For fixing the headers, each slot must have one of
C     
C     (1) DATE and UTC at centre of exposure.
C     (2) RJD at centre of exposure.
C     
C     RA, Dec etc will be prompted for if not found.
C     
C     For changing the phases, the HJD must be present.
C
        IF(NPIX(SLOT).LE.0) THEN
           WRITE(*,*) 'No data in slot ',SLOT
        ELSE IF(.NOT.PHASE) THEN
           NOK = .FALSE.
           CALL HGETD('UTC',UTC,SLOT,MXSPEC,NMDOUB,HDDOUB,
     &          NDOUB,MXDOUB,IFAIL)
           NOK = NOK .OR. IFAIL.NE.0
           CALL HGETI('YEAR',YEAR,SLOT,MXSPEC,NMINTR,HDINTR,
     &          NINTR,MXINTR,IFAIL)
           NOK = NOK .OR. IFAIL.NE.0
           CALL HGETI('MONTH',MONTH,SLOT,MXSPEC,NMINTR,HDINTR,
     &          NINTR,MXINTR,IFAIL)
           NOK = NOK .OR. IFAIL.NE.0
           CALL HGETI('DAY',DAY,SLOT,MXSPEC,NMINTR,HDINTR,
     &          NINTR,MXINTR,IFAIL)
           NOK = NOK .OR. IFAIL.NE.0
           
           IF(NOK) THEN
              CALL HGETD('RJD', RJD, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
              IF(IFAIL.NE.0) THEN
                 WRITE(*,*) 'No date and/or UT present and no'
                 WRITE(*,*) 'RJD present. Edit headers please.'
                 GOTO 999
              ELSE
                 CALL CALENDAR(RJD, MONTH, DAY, YEAR, UTC)
              END IF
           END IF
           N = N + 1
C
C     Check for positional information
C
           IF(.NOT. ESAMECI(TELESCOPE, 'SKIP')) THEN
              POSOK = .TRUE.
              CALL HGETD('RA', RA, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
              POSOK = POSOK .AND. IFAIL.EQ.0
              CALL HGETD('DEC', DEC, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
              POSOK = POSOK .AND. IFAIL.EQ.0
              CALL HGETD('EQUINOX', EQUINOX, SLOT, MXSPEC, NMDOUB, 
     &             HDDOUB, NDOUB, MXDOUB, IFAIL)
              POSOK = POSOK .AND. IFAIL.EQ.0
           END IF
C     
           CALL HGETC('OBJECT', OBJECT, SLOT, MXSPEC, NMCHAR, 
     &          HDCHAR, NCHAR, MXCHAR, IFAIL)
C     
C     Load position information. May also return orbital phase 
C     parameters
C     
           IF(ESAMECI(TELESCOPE, 'SKIP')) THEN
              LONG    = 0.
              LAT     = 85.
              HEIGHT  = 0.
              RA      = 0.
              DEC     = 80.
              EQUINOX = 2000
           ELSE IF(.NOT.POSOK) THEN
              CALL DECRA(OBJECT, EQUINOX, RA, DEC, HJD1, PER1, 
     &             MUTE, IFAIL)
              IF(IFAIL.NE.0) GOTO 999
           END IF
C     
C     Compute other header parameters
C     
           CALL FRUIT( LONG, LAT, HEIGHT, EQUINOX, RA, DEC,
     $          YEAR, MONTH, DAY, UTC)
C     
C     Set headers (put in a run number if none found)
C     
           CALL HGETI('Record', NR, SLOT, MXSPEC, NMINTR, HDINTR, 
     &          NINTR, MXINTR, IFAIL)
           IF(IFAIL.NE.0) THEN
              CALL HSETI('Record', N, SLOT, MXSPEC, NMINTR, HDINTR, 
     &             NINTR, MXINTR, IFAIL)
           END IF
           CALL HSETD('UTC', UTC, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &          NDOUB, MXDOUB, IFAIL)
           CALL HSETI('Day', DAY, SLOT, MXSPEC, NMINTR, HDINTR, 
     &          NINTR, MXINTR, IFAIL)
           CALL HSETI('Month', MONTH, SLOT, MXSPEC, NMINTR, HDINTR, 
     &          NINTR, MXINTR, IFAIL)
           CALL HSETI('Year', YEAR, SLOT, MXSPEC, NMINTR, HDINTR, 
     &          NINTR, MXINTR, IFAIL)
           CALL HSETD('RJD', RJD, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &          NDOUB, MXDOUB, IFAIL)

           IF(.NOT.ESAMECI(TELESCOPE, 'SKIP')) THEN
              CALL HSETD('Equinox', EQUINOX, SLOT, MXSPEC, NMDOUB, 
     &             HDDOUB, NDOUB, MXDOUB, IFAIL)
              CALL HSETD('RA', RA, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
              CALL HSETD('Dec', DEC, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
              CALL HSETD('Longitude', LONG, SLOT, MXSPEC, NMDOUB, 
     &             HDDOUB, NDOUB, MXDOUB, IFAIL)
              CALL HSETD('Latitude', LAT, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
              CALL HSETD('Height', HEIGHT, SLOT, MXSPEC, NMDOUB, HDDOUB,
     &             NDOUB, MXDOUB, IFAIL)
              CALL HSETC('Telescope', TELESCOPE, SLOT, MXSPEC, NMCHAR, 
     &             HDCHAR, NCHAR, MXCHAR, IFAIL)
              CALL HSETC('Site', SITE, SLOT, MXSPEC, NMCHAR, HDCHAR, 
     &             NCHAR, MXCHAR, IFAIL)
              CALL HSETD('HJD', HJD, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
              CALL HSETD('Delay', DELAY, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL)
C     
C     Earth velocity not reset if is found to be close to
C     zero since that indicates that the spectrum has been
C     rebinned earlier
C
              CALL HGETD('Vearth', TEST, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &             NDOUB, MXDOUB, IFAIL) 
              IF(IFAIL.NE.0 .OR. ABS(TEST).GT.1.D-6) THEN
                 CALL HSETD('Vearth', VEARTH, SLOT, MXSPEC, NMDOUB, 
     &                HDDOUB, NDOUB, MXDOUB, IFAIL)
              END IF
              CALL HSETC('Hel corr', 'Fruit II, 06/11/2009', SLOT, 
     &             MXSPEC, NMCHAR, HDCHAR, NCHAR, MXCHAR, IFAIL)
              CALL HSETD('Sidereal time', SIDTIM, SLOT, MXSPEC, NMDOUB, 
     &             HDDOUB, NDOUB, MXDOUB, IFAIL)
              CALL HSETD('Hour angle', HA, SLOT, MXSPEC, NMDOUB, HDDOUB,
     &             NDOUB, MXDOUB, IFAIL)
              CALL HSETD('Airmass', AIRMASS, SLOT, MXSPEC, NMDOUB, 
     &             HDDOUB, NDOUB, MXDOUB, IFAIL)
              CALL HSETR('Gal longitude', SNGL(GLONG), SLOT, MXSPEC, 
     &             NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
              CALL HSETR('Gal latitude', SNGL(GLAT), SLOT, MXSPEC, 
     &             NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
C     
C     Compute phase if possible
C     
              IF(PER1.GT.0. .AND. HJD1.GT.1.D6) THEN
                 PHS = (HJD-HJD1)/PER1
                 CALL HSETD('Orbital phase', PHS, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('ZeroO', HJD1, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('PeriodO', PER1, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
              END IF
           END IF
           IF(.NOT.MUTE) THEN
              CALL SL_INF(SLOT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &             NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB,
     &             NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING,
     &             IFAIL)
              WRITE(*,*) STRING(:LENSTR(STRING))
           END IF
        ELSE
C     
C     Set phases
C     
           CALL HGETD('HJD', HJD, SLOT, MXSPEC, NMDOUB, 
     &          HDDOUB, NDOUB, MXDOUB, IFAIL)
           IF(IFAIL.NE.0) THEN
              WRITE(*,*) 'No HJD for slot ',SLOT
              WRITE(*,*) 'Can''t compute phase'
           ELSE
C     
C     Iterate phase
C     
              PHSO = -1.E10
              PHS = (HJD-ZERO)/PERIOD
              DO WHILE(PERIOD*ABS(PHS-PHSO).GT.1.D-7)
                 PHSO = PHS
                 PHS  = (HJD-ZERO-(PDOT*PHS)*PHS)/PERIOD
              END DO
              IF(SAMECI(NAME,'Orbital phase')) THEN
                 CALL HSETD('ZeroO', ZERO, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('PeriodO', PERIOD, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 IF(PDOT.NE.0.) THEN
                    CALL HSETD('PdotO', PDOT, SLOT, MXSPEC, 
     &                   NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 END IF
              ELSE IF(SAMECI(NAME,'Spin phase')) THEN
                 CALL HSETD('ZeroS', ZERO, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('PeriodS', PERIOD, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 IF(PDOT.NE.0.) THEN
                    CALL HSETD('PdotS', PDOT, SLOT, MXSPEC, 
     &                   NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 END IF
              ELSE IF(SAMECI(NAME,'Beat phase')) THEN
                 CALL HSETD('ZeroB', ZERO, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('PeriodB', PERIOD, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 IF(PDOT.NE.0.) THEN
                    CALL HSETD('PdotB', PDOT, SLOT, MXSPEC, 
     &                   NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 END IF
              ELSE IF(SAMECI(NAME,'Hump phase')) THEN
                 CALL HSETD('ZeroH', ZERO, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('PeriodH', PERIOD, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 IF(PDOT.NE.0.) THEN
                    CALL HSETD('PdotH', PDOT, SLOT, MXSPEC, 
     &                   NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 END IF
              ELSE IF(ESAMECI(NAME,'Phase1')) THEN
                 CALL HSETD('Zero1', ZERO, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('Period1', PERIOD, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 IF(PDOT.NE.0.) THEN
                    CALL HSETD('Pdot1', PDOT, SLOT, MXSPEC, 
     &                   NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 END IF
              ELSE IF(ESAMECI(NAME,'Phase2')) THEN
                 CALL HSETD('Zero2', ZERO, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('Period2', PERIOD, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 IF(PDOT.NE.0.) THEN
                    CALL HSETD('Pdot2', PDOT, SLOT, MXSPEC, 
     &                   NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 END IF
              ELSE IF(ESAMECI(NAME,'Phase3')) THEN
                 CALL HSETD('Zero3', ZERO, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 CALL HSETD('Period3', PERIOD, SLOT, MXSPEC, 
     &                NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 IF(PDOT.NE.0.) THEN
                    CALL HSETD('Pdot3', PDOT, SLOT, MXSPEC, 
     &                   NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
                 END IF
              END IF                               
              CALL HSETD(NAME, PHS, SLOT, MXSPEC, 
     &             NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
           END IF
        END IF
      END DO
 999  RETURN
      END

      SUBROUTINE DECRA(OBJECT, EQUINOX, RA, DEC, HJD0, PERIOD, MUTE,
     &     IFAIL)
C     
C     Returns with positional information on OBJECT
C     
      IMPLICIT NONE
      CHARACTER*(*) OBJECT
      DOUBLE PRECISION EQUINOX, RA, DEC, HJD0, PERIOD
      INTEGER IFAIL, SEARC
      LOGICAL MUTE
C     
C     Positional information
C     
      INTEGER MAXSTAR
      PARAMETER (MAXSTAR=200)
      CHARACTER*32 STARS(MAXSTAR)
      DOUBLE PRECISION SRA(MAXSTAR), SDEC(MAXSTAR)
      DOUBLE PRECISION SEQUINOX(MAXSTAR) 
      DOUBLE PRECISION SHJD0(MAXSTAR),  SPERIOD(MAXSTAR)
      INTEGER RAH, RAM, DECD, DECM
      REAL RAS, DECS
C     
      CHARACTER*1 REPLY
      CHARACTER*100 STRING
      INTEGER LUNIT, NLOAD, I,  N, NALT
      DATA LUNIT, NLOAD/23, 0/
C     
 50   CONTINUE
      IF(NLOAD.EQ.0) THEN
         WRITE(*,*) 'No positional information loaded'
         WRITE(*,*) 'E(nter) by hand, L(oad) from file'
      ELSE
         N = SEARC(STARS,NLOAD,NLOAD,OBJECT)
         IF(N.EQ.0) THEN
            WRITE(*,*) 'No match found for object = [',OBJECT,']'
            WRITE(*,*) 
     &           'E(nter) by hand, L(oad) more, O(ther) name, C(lear)'
         ELSE IF(N.LT.0) THEN
            WRITE(*,*) 'More than one match found for object = [',
     &           OBJECT,']'
            WRITE(*,*) 
     &           'E(nter) by hand, L(oad) more, O(ther) name, C(lear)'
         ELSE
            RA     = SRA(N)
            DEC    = SDEC(N)
            EQUINOX  = SEQUINOX(N)
            HJD0   = SHJD0(N)
            PERIOD = SPERIOD(N)
            IF(.NOT.MUTE) THEN
               WRITE(*,*) 'Match found: ',STARS(N)
               WRITE(*,*) 'RA: ',SNGL(RA),', Dec: ',SNGL(DEC)
               WRITE(*,*) 'HJD0: ',HJD0,', period: ',PERIOD
            END IF
            GOTO 500
         END IF
      END IF
      READ(*,'(A)') REPLY
      CALL UPPER_CASE(REPLY)
      IF(REPLY.EQ.'L') THEN
         WRITE(*,*) 'What file contains data?'
         READ(*,'(A)') STRING
         OPEN(UNIT=LUNIT,FILE=STRING,STATUS='OLD',IOSTAT=IFAIL)
         IF(IFAIL.NE.0) THEN
            WRITE(*,*) 'Could not open file'
            GOTO 50
         END IF
 100     CONTINUE
         NLOAD = NLOAD + 1
         IF(NLOAD.GT.MAXSTAR) THEN
            NLOAD = MAXSTAR
            WRITE(*,*) 'List full'
            CLOSE(UNIT=LUNIT)
            GOTO 50
         END IF
         READ(LUNIT,'(A)',IOSTAT=IFAIL) STARS(NLOAD)
         WRITE(*,*) STARS(NLOAD)
         IF(IFAIL.LT.0) THEN
            WRITE(*,*) 'End of file reached'
            NLOAD = NLOAD - 1
            CLOSE(UNIT=LUNIT)
            GOTO 50
         ELSE IF(IFAIL.GT.0) THEN
            WRITE(*,*) 'Error reading name'
            NLOAD = NLOAD - 1
            CLOSE(UNIT=LUNIT)
            GOTO 50
         END IF
         READ(LUNIT,*,IOSTAT=IFAIL) SEQUINOX(NLOAD), RAH, RAM,
     &        RAS, DECD, DECM, DECS, SHJD0(NLOAD), SPERIOD(NLOAD)
         IF(IFAIL.EQ.0) THEN
            SRA(NLOAD) = DBLE(RAH)+(DBLE(RAM)+DBLE(RAS)/6.D1)/6.D1
            SDEC(NLOAD)= DBLE(ABS(DECD))+(DBLE(ABS(DECM))
     &           +DBLE(ABS(DECS))/6.D1)/6.D1
            IF(DECD.LT.0 .OR. (DECD.EQ.0 .AND. DECM.LT.0)
     &           .OR. (DECD.EQ.0 .AND. DECM.EQ.0 .AND. 
     &           DECS.LT.0.)) SDEC(NLOAD) = - SDEC(NLOAD)
         ELSE IF(IFAIL.LT.0) THEN
            WRITE(*,*) 'End of file reached'
            WRITE(*,*) ' '
            NLOAD = NLOAD - 1
            CLOSE(UNIT=LUNIT)
            GOTO 50
         ELSE
            WRITE(*,*) 'Error reading data'
            NLOAD = NLOAD - 1
            CLOSE(UNIT=LUNIT)
            GOTO 50
         END IF
         GOTO 100
      ELSE IF(REPLY.EQ.'E') THEN
         WRITE(*,*) 'Enter RA (HMS), Dec (DMS)'
         READ(*,*,ERR=999) RAH, RAM, RAS, DECD, DECM, DECS
         RA  = DBLE(RAH)+(DBLE(RAM)+DBLE(RAS)/6.D1)/6.D1
         DEC = DBLE(ABS(DECD))+(DBLE(ABS(DECM))
     &        +DBLE(ABS(DECS))/6.D1)/6.D1
         IF(DECD.LT.0 .OR. (DECD.EQ.0 .AND. DECM.LT.0)
     &        .OR. (DECD.EQ.0 .AND. DECM.EQ.0 .AND. 
     &        DECS.LT.0.)) DEC = - DEC
         WRITE(*,*) 'Enter Equinox, HJD0, Period'
         READ(*,*,ERR=999) EQUINOX, HJD0, PERIOD
         IF(NLOAD.EQ.MAXSTAR) THEN
            WRITE(*,*) 'List full; cannot store, but'
            WRITE(*,*) 'coordinates set correctly.'
            GOTO 500
         ELSE
            NLOAD = NLOAD + 1
            SEQUINOX(NLOAD)  = EQUINOX
            SRA(NLOAD)      = RA
            SDEC(NLOAD)     = DEC
            SHJD0(NLOAD)    = HJD0
            SPERIOD(NLOAD)  = PERIOD
         END IF
      ELSE IF(REPLY.EQ.'C' .AND. NLOAD.NE.0) THEN
         NLOAD = 0
         GOTO 50
      ELSE IF(REPLY.EQ.'O' .AND. NLOAD.NE.0) THEN
         DO I = 1, NLOAD/2
            WRITE(*,'(1X,I3,A,A,1X,I3,A,A)') I,') ',STARS(I),
     &           I+(NLOAD+1)/2,') ',STARS(I+(NLOAD+1)/2)
         END DO
         IF(2*(NLOAD/2).NE.NLOAD) 
     &        WRITE(*,'(1X,I3,A,A)') NLOAD/2+1,') ',STARS(NLOAD/2+1)

         WRITE(*,*) ' '
         WRITE(*,*) 'Enter number of alternative object.'
         READ(*,*,IOSTAT=IFAIL) NALT
         IF(IFAIL.NE.0 .OR. NALT.LT.1 .OR. NALT.GT.NLOAD) THEN
            WRITE(*,*) 'Invalid entry'
            GOTO 50
         END IF

         RA       = SRA(NALT)
         DEC      = SDEC(NALT)
         EQUINOX  = SEQUINOX(NALT)
         HJD0     = SHJD0(NALT)
         PERIOD   = SPERIOD(NALT)
         WRITE(*,*) 'RA: ',SNGL(RA),', Dec: ',SNGL(DEC), 
     &        ', Equinox: ',SNGL(EQUINOX)
         WRITE(*,*) 'HJD0: ',HJD0,', period: ',PERIOD
         IF(NLOAD.LT.MAXSTAR) THEN
            NLOAD           = NLOAD + 1
            STARS(NLOAD)    = OBJECT
            SRA(NLOAD)      = RA
            SDEC(NLOAD)     = DEC
            SEQUINOX(NLOAD) = EQUINOX
            SHJD0(NLOAD)    = HJD0
            SPERIOD(NLOAD)  = PERIOD
         END IF
         GOTO 500
      ELSE
         WRITE(*,*) 'Invalid reply'
         GOTO 999
      END IF
 500  CONTINUE             
      IFAIL = 0
      RETURN
 999  IFAIL = 1
      RETURN
      END
