      DOUBLE PRECISION FUNCTION ANGST(PIX, NPIX, NARC, ARC, MXARC)
C     
C     Returns the wavelength in Angstroms of a pixel position PIX
C     of a spectrum with NPIX pixels using wavelength coefficients 
C     ARC(MXARC). NARC is the number of coefficients which can be 
C     negative to indicate a logarithmic scale.
C     
C     > R  PIX   -- Pixel position
C     > I  NPIX  -- Number of pixels in spectrum
C     > I  NARC  -- Number of arc coefficients. Negative if logarithmic scale.
C     > D  ARC(MXARC) -- Arc coefficients.
C     > I  MXARC -- Maximum number of arc coefficients.
C     
      IMPLICIT NONE
      INTEGER NARC, MXARC, NPIX, NABS
      REAL PIX
      DOUBLE PRECISION ARC(MXARC), POLY, Z
C     
      Z = PIX/DBLE(NPIX)
      IF(NARC.LT.0) THEN
         NABS  = - NARC
         ANGST = EXP(POLY(ARC, NABS, Z))
      ELSE
         ANGST = POLY(ARC, NARC, Z)
      END IF
      RETURN
      END                                    

      SUBROUTINE ARCSHF(ARC1, NARC1, NPIX1, ARC2, NARC2, NPIX2,
     &     MXARC, SHIFT, SSKEW)
C     
C     Checks to see whether two MOLLY wavelength scales are just shifted 
C     versions of each other, in which case fast rebinning methods
C     are possible. If they are it returns the shift needed to get
C     from ARC1 to ARC2 consistent with the SSKEW parameter of REBIN
C     and SINCBIN
C     
C     Returns: LOGICAL SHIFT -- True if they are shifted
C     REAL SSKEW    -- Amount of shift
C     
      IMPLICIT NONE
      INTEGER NARC1, NARC2, MXARC, NPIX1, NPIX2
      INTEGER NARC, J, I
      DOUBLE PRECISION ARC1(MXARC), ARC2(MXARC), SKEW
      DOUBLE PRECISION FAC, ACC, POLY, Z, SUM
      REAL SSKEW
      LOGICAL SHIFT
C     
      SHIFT = .FALSE.
      IF(NARC1.NE.NARC2 .OR. NPIX1.NE.NPIX2) RETURN
      NARC = ABS(NARC1)
C     
C     Differences will be compared to 1/1000 of a typical pixel
C     
      Z = 1./DBLE(NPIX1)
      ACC = POLY(ARC1,NARC,1.D0)-POLY(ARC1,NARC,Z)
      ACC = 1.D-3*ABS(ACC)/DBLE(NPIX1)
      IF(ABS(ARC1(NARC)-ARC2(NARC)).GT.ACC) RETURN
C     
C     Scaled skew
C     
      SKEW = (ARC2(NARC-1)-ARC1(NARC-1))/DBLE(NARC-1)/ARC1(NARC)
C     
C     Final check
C     
      DO I = NARC-2,1,-1
         SUM = 0.D0
         FAC = 1.D0
         DO J = I, NARC
            SUM = SUM + FAC*ARC1(J) 
            FAC = FAC*DBLE(J)/DBLE(J-I+1)
         END DO
         IF(ABS(SUM-ARC2(I)).GT.ACC) RETURN
      END DO
C     
C     All tests passed
C     
      SHIFT = .TRUE.
      SSKEW = REAL(NPIX1*SKEW)
      RETURN
      END

      DOUBLE PRECISION FUNCTION DISPA(PIX, NPIX, NARC, ARC, MXARC)
C     
C     Returns the dispersion in A/pixel a pixel position PIX
C     of a spectrum with NPIX pixels using wavelength coefficients 
C     ARC(MXARC). NARC is the number of coefficients which can be 
C     negative to indicate a logarithmic scale.
C     
C     > R  PIX   -- Pixel position
C     > I  NPIX  -- Number of pixels in spectrum
C     > I  NARC  -- Number of arc coefficients. Negative if logarithmic scale.
C     > D  ARC(MXARC) -- Arc coefficients.
C     > I  MXARC -- Maximum number of arc coefficients.
C     
      IMPLICIT NONE
      INTEGER NARC, MXARC, NPIX, NABS, I
      REAL PIX
      DOUBLE PRECISION ARC(MXARC), POLY, Z
      INTEGER MAXARC
      PARAMETER (MAXARC=30)
      DOUBLE PRECISION DARC(MAXARC-1)
C     
      IF(NARC.GT.MAXARC) THEN
         WRITE(*,*) 'Buffer too small in DISPA'
         DISPA = 1.D0
         RETURN
      END IF
      Z = PIX/DBLE(NPIX)
      IF(NARC.LT.0) THEN
         NABS = - NARC
         DO I = 1, NABS-1
            DARC(I) = DBLE(I)*ARC(I+1)
         END DO
         DISPA = POLY(DARC, NABS-1, Z)*
     &        EXP(POLY(ARC, NABS, Z))/DBLE(NPIX)
      ELSE
         DO I = 1, NARC-1
            DARC(I) = DBLE(I)*ARC(I+1)
         END DO
         DISPA = POLY(DARC, NARC-1, Z)/DBLE(NPIX)
      END IF
      RETURN
      END                                    

      REAL FUNCTION CFRAT(COUNTS, FLUX)
C     
C     Gets number of counts/milli-Jansky of flux.
C     value COUNTS, FLUX 
C     
      IMPLICIT NONE
      REAL COUNTS, FLUX
C     
      IF(COUNTS.EQ.0.) THEN
         CFRAT = FLUX
      ELSE
         CFRAT = COUNTS/FLUX
      END IF
      RETURN
      END

      REAL FUNCTION GET_ERF(COUNTS, ERRORS, FLUX)
C     
C     Gets error on milli-Jansky flux value on a data point with
C     value COUNTS, ERRORS, FLUX 
C     
      IMPLICIT NONE
      REAL COUNTS, ERRORS, FLUX
C     
      IF(ERRORS.LE.0.) THEN
         IF(COUNTS.EQ.0. .AND. FLUX.NE.0) THEN
            GET_ERF = -ABS(ERRORS/FLUX)
         ELSE IF(COUNTS.NE.0.) THEN
            GET_ERF = -ABS(ERRORS/COUNTS*FLUX)
         ELSE
            GET_ERF = -1.
         END IF
      ELSE
         IF(COUNTS.EQ.0.) THEN
            GET_ERF = ERRORS/FLUX
         ELSE
            GET_ERF = ERRORS/COUNTS*FLUX
         END IF
      END IF
      RETURN
      END

      REAL FUNCTION SET_ERF(COUNTS, FLUX, FERR, RATIO)
C     
C     Sets errors on counts given counts, flux and error on flux.
C     value COUNTS, ERRORS, FLUX, and the ratio COUNTS/FLUX = RATIO
C     
C     This should be used to set the ERRORS value on a spectrum if
C     you have the COUNTS, FLUX and error on the FLUX. If you have
C     the error on the counts, then set ERRORS directly
C     
      IMPLICIT NONE
      REAL COUNTS, FLUX, FERR, RATIO
C     
      IF(COUNTS.EQ.0.) THEN
         SET_ERF = FERR*RATIO
      ELSE 
         SET_ERF = FERR/FLUX*COUNTS
      END IF
      IF(FERR.LE.0.) THEN
         SET_ERF = -ABS(SET_ERF)
      ELSE
         SET_ERF = ABS(SET_ERF)
      END IF
      RETURN
      END

      REAL FUNCTION GET_FLX(COUNTS, FLUX)
C     
C     Gets milli-Jansky flux value on a data point with
C     value COUNTS, FLUX 
C     
      IMPLICIT NONE
      REAL COUNTS, FLUX
C     
      IF(COUNTS.EQ.0.) THEN
         GET_FLX = 0.
      ELSE
         GET_FLX = FLUX
      END IF
      RETURN
      END

      REAL FUNCTION SET_FLX(COUNTS, FLUX, RATIO)
C     
C     Sets milli-Jansky flux value on a data point with
C     real counts and flux = COUNTS, FLUX, and ratio COUNTS/FLUX = RATIO 
C     
C     Use this to set the value of the FLUX value on a pixel
C     
      IMPLICIT NONE
      REAL COUNTS, FLUX, RATIO
C     
      IF(COUNTS.EQ.0.) THEN
         SET_FLX = RATIO
      ELSE
         SET_FLX = FLUX
      END IF
      RETURN
      END



      INTEGER FUNCTION INSPEC(PIX, SPEC, MAXPX)
C     
C     Gets index in COUNTS, ERRORS, FLUX arrays for pixel
C     PIX of spectrum SPEC. MAXPX number of pixels allocated
C     to each spectrum.
C     
      IMPLICIT NONE
      INTEGER PIX, SPEC, MAXPX
C     
      INSPEC = MAXPX*(SPEC-1)+PIX
      RETURN
      END

      DOUBLE PRECISION FUNCTION PIXL(WAVE, NPIX, NARC, ARC, MXARC)
C     
C     Returns the pixel position equivalent to a wavelength WAVE 
C     in a spectrum with NPIX pixels using wavelength coefficients 
C     ARC(MXARC). NARC is the number of coefficients which can be 
C     negative to indicate a logarithmic scale. Linear extrapolation
C     only is used beyond the ends of the array.
C     
C     > D  WAVE  -- Wavelength in Angstroms
C     > I  NPIX  -- Number of pixels in spectrum
C     > I  NARC  -- Number of arc coefficients. Negative if logarithmic scale.
C     > D  ARC(MXARC) -- Arc coefficients.
C     > I  MXARC -- Maximum number of arc coefficients.
C     
      IMPLICIT NONE
      INTEGER NARC, MXARC, NPIX, NABS
      INTEGER NDABS, NC, I, MAXARC
      DOUBLE PRECISION WAVE, ARC(MXARC), POLY, Z, W
      PARAMETER (MAXARC=39)
      DOUBLE PRECISION DARC(MAXARC), WLO, WHI
      DOUBLE PRECISION DISP, TEST, ZLO, ZHI, ZN
C     
      IF(MXARC.GT.MAXARC+1) THEN
         WRITE(*,*) 'Too many coefficients for PIXL'
         PIXL = 0.D0
         RETURN
      END IF
      IF(NARC.LT.0) THEN
         TEST = LOG(WAVE)
      ELSE
         TEST = WAVE
      END IF
      NABS  = ABS(NARC)
      NDABS = NABS - 1
      DO I = 1, NDABS
         DARC(I) = DBLE(I)*ARC(I+1)
      END DO
C     
C     Check that it falls within the correct range.
C     If not linearly extrapolate beyond end of array.
C     
      ZLO = 5.D-1/DBLE(NPIX)
      WLO = POLY(ARC, NABS, ZLO)
      ZHI = 1.+ZLO
      WHI = POLY(ARC, NABS, ZHI)
      IF(WLO.GT.WHI) THEN
         W = WLO
         Z = ZLO
         WLO = WHI
         ZLO = ZHI
         WHI = W
         ZHI = Z
      END IF
      IF(TEST.LT.WLO) THEN
         DISP = POLY(DARC, NDABS, ZLO)
         PIXL = DBLE(NPIX)*(ZLO + (TEST-WLO)/DISP)
         RETURN
      ELSE IF(WAVE.GT.WHI) THEN
         DISP = POLY(DARC, NDABS, ZHI)
         PIXL = DBLE(NPIX)*(ZHI + (TEST-WHI)/DISP)
         RETURN
      END IF
C     
C     Within range. Newton iteration.
C     
      Z = ZLO + (TEST-WLO)/(WHI-WLO)*(ZHI-ZLO)
      NC = 0
 10   DISP = POLY(DARC, NDABS, Z)
      W    = POLY(ARC, NABS, Z)
      ZN   = Z + (TEST-W)/DISP
      IF(ABS(ZN-Z)*DBLE(NPIX).GT.1.D-5) THEN
         Z = ZN
         NC = NC + 1
         IF(NC.GT.10) WRITE(*,*) '''PIXL'' stuck ',NC
         GOTO 10
      END IF
      PIXL = DBLE(NPIX)*ZN
      RETURN
      END

      REAL FUNCTION AB_MJY(MJY)
C     
C     Converts from milliJanskys to AB mags
C     
      REAL MJY
      IF(MJY.GT.0.) THEN
         AB_MJY =  16.4-2.5*LOG10(MJY)
      ELSE
         AB_MJY =  30.
      END IF      
      RETURN
      END

      REAL FUNCTION MJY_AB(AB)
C     
C     Converts from AB mags to milliJanskys 
C     
      REAL AB
      MJY_AB = 10.**((16.4-AB)/2.5)
      RETURN
      END

      DOUBLE PRECISION FUNCTION LGAUSS(X1, X2)
C     
C     Returns LOG of integral over gaussian from X1 to X2.
C     Assumes X2 > X1
C     
      INTEGER I
      DOUBLE PRECISION X1, X2, PDA_DERF, RTWOPI, SUM
      DOUBLE PRECISION FAC1, FAC2, Z1, Z2
      LOGICAL FIRST
      DATA FIRST/.TRUE./
C     
      IF(FIRST) THEN
         FIRST = .FALSE.
         RTWOPI = 1.D0/SQRT(8.*ATAN(1.D0))
      END IF
      IF(ABS(X1).LT.11 .OR. ABS(X2).LT.11) THEN
         IF(X1.GT.0.) THEN
            LGAUSS = LOG(0.5*(PDA_DERF(X2/SQRT(2.))-
     &           PDA_DERF(X1/SQRT(2.))))
         ELSE
            LGAUSS = LOG(0.5*(PDA_DERF(-X1/SQRT(2.))-
     &           PDA_DERF(-X2/SQRT(2.))))
         END IF
      ELSE IF(X1.LT.0. .AND. X2.GT.0.) THEN
         LGAUSS = 0.D0
      ELSE
         IF(X2.LT.0.) THEN
            Z1 = -X2
            Z2 = -X1
         ELSE
            Z1 = X1
            Z2 = X2
         END IF
         LGAUSS = -Z1**2/2.D0
         SUM = 0.D0
         FAC1 =  1.D0/Z1
         FAC2 =  1.D0/Z2*EXP((Z1**2-Z2**2)/2.D0)
         DO I = 1, 5
            SUM  = SUM + FAC1 - FAC2
            FAC1 = - DBLE(2*I-1)*FAC1/Z1**2
            FAC2 = - DBLE(2*I-1)*FAC2/Z2**2
         END DO                        
         LGAUSS = LGAUSS + LOG(RTWOPI*SUM)
      END IF
      RETURN
      END
