      SUBROUTINE BRCOFF(X,VSINI,EPS,METHOD,COF,IFAIL)
*
* Function to compute blurring array for rotationally broadening a spectrum.
* Returns a value proportional to the amount of a pixel that is blurred into
* a pixel X away. Not normalised.
*
* R*4 X     -- Displacement in pixels from pixel to be blurred
* R*4 VSINI -- Rotational broadening in pixels
* R*4 EPS   -- Limb darkening in pixels
* I*4 METHOD -- Computation method
*
*   METHOD = 1 -- Fast crude method which just computes the pdf of
*                 the limb darkened rotational profile. Adequate if
*                 VSINI >> 1, but only approximately correct otherwise.
*                 It tends to under-blurr for small VSINI (e.g. does not
*                 blurr at all for VSINI<1).
*
*   METHOD = 2 -- Slow method which computes integral of blurring function
*                 with interpolation function (sinc function).
*
* R*8 COF   -- Returned coefficient  
* I*4 IFAIL -- Error return from Romberg integration QROMB. Ok if = 0.
*
      IMPLICIT NONE
      INTEGER IFAIL, METHOD
      REAL X,VSINI,EPS,EPSOLD
      DOUBLE PRECISION BROT, LOW, HIGH, COF, ACC
      EXTERNAL BROT
      DOUBLE PRECISION X1,VSINI1,EPS1,PI,C1,C2,Z
      INCLUDE 'maxsinc.inc'
      COMMON/BROTC/X1,VSINI1,EPS1
      DATA EPSOLD/-1./
*
      IF(METHOD.EQ.1) THEN
        IF(EPS.NE.EPSOLD) THEN
          PI =  4.*ATAN(1.)
          C1 = PI*(1.-EPS/3.)
          C2 = PI*EPS/2./C1
          C1 = 2.*(1.-EPS)/C1
          EPSOLD = EPS
        END IF
        IF(ABS(X).LT.VSINI) THEN
          Z = 1.D0 - DBLE((X/VSINI)**2)
          COF = (C1*SQRT(Z)+C2*Z)/DBLE(VSINI)
        ELSE
          COF = 0.D0
        END IF
      ELSE
*
* Set common block parameters
*
        X1     = DBLE(X)
        VSINI1 = DBLE(VSINI)
        EPS1   = DBLE(EPS)
*
* Evaluate integration limits 
*
        LOW  = DBLE(MAX(-VSINI, X-REAL(MAXSINC)-0.5))
        HIGH = DBLE(MIN( VSINI, X+REAL(MAXSINC)+0.5))
        ACC  = 1.D-7/VSINI
        CALL QROMB(BROT,LOW,HIGH,COF,ACC,IFAIL)
      END IF
      RETURN
      END
      
      DOUBLE PRECISION FUNCTION BROT(U)
*
* Function to evaluate convolution integrand for stellar
* rotation profile. Common block parameters X, VSINI have 
* units of pixels.
*
      IMPLICIT NONE
      EXTERNAL SINCF
      DOUBLE PRECISION U, C1, C2, PI, Z, DIFF
      DOUBLE PRECISION X, VSINI, EPS, EPSOLD, SINCF
      COMMON/BROTC/X,VSINI,EPS
      DATA EPSOLD/-1.D0/
*
      IF(EPS.NE.EPSOLD) THEN
        PI =  4.D0*ATAN(1.D0)
        C1 = PI*(1.D0-EPS/3.D0)
        C2 = PI*EPS/2.D0/C1
        C1 = 2.D0*(1.D0-EPS)/C1
        EPSOLD = EPS
      END IF
*
      IF(ABS(U).LT.VSINI) THEN
        Z = 1.D0 - (U/VSINI)**2
        DIFF = X-U
        BROT = SINCF(DIFF)*(C1*SQRT(Z)+C2*Z)/VSINI
      ELSE
        BROT = 0.D0
      END IF
      RETURN
      END

      SUBROUTINE BSCOFF(X,WIDTH,METHOD,COF,IFAIL)
*
* Function to compute blurring array for a uniform smear of a spectrum
*
* R*4 X     -- Displacement in pixels from pixel to be blurred
* R*4 WIDTH -- The (total) smear width in pixels
* I*4 METHOD -- Computation method
*
*   METHOD = 1 -- Fast crude method which just computes the pdf of
*                 the smear profile (a rectangle). Adequate if
*                 WIDTH >> 1, but only approximately correct otherwise.
*                 It tends to under-blurr for small WIDTH (e.g. does not
*                 blurr at all for WIDTH < 2).
*
*   METHOD = 2 -- Slow method which computes integral of blurring function
*                 with interpolation function (sinc function).
*
*   
* R*8 COF   -- Returned coefficient
* I*4 IFAIL -- Error return from Romberg integration QROMB. Ok if = 0.
*
      IMPLICIT NONE
      INTEGER IFAIL, METHOD
      REAL X, WIDTH
      DOUBLE PRECISION BSMR, LOW, HIGH, ACC
      EXTERNAL BSMR
      DOUBLE PRECISION X1,WIDTH1,COF
      INCLUDE 'maxsinc.inc'
      COMMON/BSMRC/X1,WIDTH1
*
      IF(METHOD.EQ.1) THEN
        IF(ABS(X).LE.WIDTH/2.) THEN
          COF = 1.D0/DBLE(WIDTH)
        ELSE
          COF = 0.D0
        END IF
      ELSE
*
* Set common block parameters
*
        X1     = DBLE(X)
        WIDTH1 = DBLE(WIDTH)
*
* Evaluate integration limits 
*
        LOW  = DBLE(MAX(-WIDTH/2., X-REAL(MAXSINC)-0.5))
        HIGH = DBLE(MIN( WIDTH/2., X+REAL(MAXSINC)+0.5))
        ACC  = 1.D-7/WIDTH
        CALL QROMB(BSMR,LOW,HIGH,COF,ACC,IFAIL)
      END IF
      RETURN
      END
      
      DOUBLE PRECISION FUNCTION BSMR(U)
*
* Function to evaluate convolution integrand for uniform
* smear profile. Common block parameters X, WIDTH have 
* units of pixels. Uniform smear profile is 
*
      IMPLICIT NONE
      EXTERNAL SINCF
      DOUBLE PRECISION U, X, WIDTH
      DOUBLE PRECISION SINCF, DIFF
      COMMON/BSMRC/X,WIDTH
*
      IF(ABS(U).LE.WIDTH/2.D0) THEN
        DIFF = X-U
        BSMR = SINCF(DIFF)/WIDTH
      ELSE
        BSMR = 0.D0
      END IF
      RETURN
      END

      SUBROUTINE BACOFF(X,METHOD,COF,IFAIL)
*
* Function to compute blurring array for arbitrary tabulated profile
* rotationally broadening a spectrum. The tabulated profile should
* be in velocity along the X axis and represent the profile in y.
* It should start and end with y = 0.
*
* R*4 X      -- Displacement in pixels from pixel to be blurred
* I*4 METHOD -- Computation method
*
*   METHOD = 1 -- Fast crude method which just computes the pdf of
*                 the profile by linear interpolation. Adequate if 
*                 the profile extends over many pixels.
*
*   METHOD = 2 -- Slow method which computes integral of blurring function
*                 with interpolation function (sinc function).
*
* R*8 COF   -- Returned coefficient  
* I*4 IFAIL -- Error return from Romberg integration QROMB. Ok if = 0.
*
      IMPLICIT NONE
      INTEGER IFAIL, METHOD, JLO, I, NPROF
      REAL X, VAVE, RX
      DOUBLE PRECISION BARB, LOW, HIGH, COF, XX, VX, ACC
      EXTERNAL BARB
      COMMON /ARB/NPROF, VAVE, RX
      INCLUDE 'maxsinc.inc'
      INCLUDE 'molly.coms'
      SAVE JLO
      DATA JLO/1/
C
C     X and Y stored in slots 3 and 4 of DWORK. 
C
      XX = X
      VX = X*VAVE
      IF(METHOD.EQ.1) THEN
         IF(VX.LT.DWORK(1,3) .OR. VX.GT.DWORK(NPROF,3)) THEN
            COF = 0.D0
         ELSE
            CALL HUNTD(DWORK(1,3),NPROF,VX,JLO)
            COF = DWORK(JLO,4)*(DWORK(JLO+1,3)-VX) +
     &           DWORK(JLO+1,4)*(VX-DWORK(JLO,3))
            COF = COF/(DWORK(JLO+1,3)-DWORK(JLO,3))
         END IF
      ELSE
C
C     Evaluate integration limits 
C
         LOW  = MAX(DWORK(1,3)/DBLE(ABS(VAVE)), 
     &        XX-DBLE(MAXSINC)-5.D-1)
         HIGH = MIN(DWORK(NPROF,3)/DBLE(ABS(VAVE)), 
     &        XX+DBLE(MAXSINC)+5.D-1)
         RX = X
C
C     Stop QROMB when it reaches an uncertainty of 1.D-7 times highest value
C     of profile. This should be small compared to the uncertainty deriving
C     from the use of linear interpolation, and yet it should also prevent
C     QROMB trying too hard which can make the whole process agonisingly slow.
C
         ACC = ABS(DWORK(1,4))
         DO I = 2,NPROF
            ACC = MAX(ACC, ABS(DWORK(I,4)))
         END DO
         ACC = 1.D-7*ACC
         CALL QROMB(BARB,LOW,HIGH,COF,ACC,IFAIL)
      END IF
      RETURN
      END
      
      DOUBLE PRECISION FUNCTION BARB(U)
C
C     Function to evaluate convolution integrand for 
C     arbitrary tabulated profile.
C
      IMPLICIT NONE
      INTEGER JLO
      DOUBLE PRECISION U, V
      EXTERNAL SINCF
      DOUBLE PRECISION SINCF, DIFF
      INCLUDE 'molly.coms'
      INTEGER NPROF
      REAL VAVE, X
      COMMON /ARB/NPROF, VAVE, X
      DATA JLO/1/
C
      V   = U*VAVE
      IF(V.LT.DWORK(1,3) .OR. V.GT.DWORK(NPROF,3)) THEN
         BARB = 0.D0
      ELSE
         CALL HUNTD(DWORK(1,3),NPROF,V,JLO)
         BARB = DWORK(JLO,4)*(DWORK(JLO+1,3)-V) +
     &        DWORK(JLO+1,4)*(V-DWORK(JLO,3))
         BARB = BARB/(DWORK(JLO+1,3)-DWORK(JLO,3))
         DIFF = DBLE(X) - U
         BARB = BARB*SINCF(DIFF)
      END IF
      RETURN
      END




