*ACAL
* ACAL N1 N2 A1 A2 METHOD DA DT DX -- Applies arc calibration to a set of 
*                                     spectra
*
* Parameters:
*            N1, N2 -- The range of spectra to be wavelength calibrated
*            A1, A2 -- The range of arc spectra to use for calibration
*            METHOD -- N -- Nearest arc in time is used
*                      I -- Interpolate (linearly) between two nearest arcs,
*                           take nearest if arcs only on one side (in time)
*                      E -- Interpolate as above, but extrapolate if arc is
*                           only on one side
*
*            DA    -- Maximum displacement in minutes of arc on sky that is
*                     allowable between an arc and an object. This prevents
*                     an arc taken for one object being used to calibrate
*                     another (unless DA is made huge). If a position
*                     cannot be found in the arc's header, it will taken to
*                     be OK.
*
*            DT    -- Maximum time offset between exposure and arc in
*                     minutes. Allows one to eliminate possibility of
*                     one night's arc being applied to next.
*
*            DX    -- Maximum difference in extraction position in pixels.
*                     If the real header item 'Extract position' is set,
*                     this can be used to make sure that an arc has been
*                     extracted at the same position as the object.
*
* The run numbers of the arc or arcs used to derive the wavelength scale are
* written in a character header item called "Arc(s) used".
* 
* NB The arc and target MUST have the same number of pixels. If it never
*    seems to find any arcs, you may want to check this amongst other
*    things. If your spectra have the header parameter 'Fibre' defined 
*    representing a fibre number (integer) then both arc and object must match
*    exactly for an arc to be used.
*
* Related command: ARC, DRIFT
*
*ACAL
      SUBROUTINE ARCAL(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, FLUX, 
     &MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, 
     &NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS1, SLOTS2,
     &KEY, RJD, NLIST, MXSLOTS, IFAIL)
*
* Applies arc calibration. 
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)   -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)   -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)     -- Equivalent fluxes to COUNTS
* >  I     MXBUFF           -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
*    I     SLOTS1(MXSLOTS) -- List buffer.
*    I     SLOTS2(MXSLOTS) -- List buffer.
*    I     KEY(MXSLOTS)   -- Sorting buffer
*    D     RJD(MXSLOTS)   -- Buffer for storing RJDs
*    I     NLIST(MXSLOTS) -- buffer
*    I     MXSLOTS
* <  I     IFAIL              -- Error return.
*
      IMPLICIT NONE
C
C     Integer parameters
C
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C
C     Data arrays
C
      REAL COUNTS(MXBUFF), ERRORS(MXBUFF), FLUX(MXBUFF)
C
C     Numbers of header items
C
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C
C     Names of header items
C
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C
C     Values of header items
C
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C
C     Arc coefficients
C
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C
C     Command parameters
C
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C
C     Slot lists
C
      INTEGER MXSLOTS, NSLOTS1, NSLOTS2, SLOT, ASLOT
      INTEGER SLOTS1(MXSLOTS), SLOTS2(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST1, LIST1(2,MXLIST)
      INTEGER NLIST2, LIST2(2,MXLIST)
      INTEGER KEY(MXSLOTS), NLIST(MXSLOTS)
      DOUBLE PRECISION RJD(MXSLOTS), JD, JDA, DIFF, F1, F2
      DOUBLE PRECISION RAO, DECO, RAA, DECA, TWOPI
      CHARACTER*32 WHAT, METHOD*3
C
C     Local variables
C
      INTEGER I, J, K,   I1, I2, FIBREO, FIBREA
      CHARACTER*32 AUSED
      INTEGER SLOT1, SLOT2, SLOT3, SLOT4, NSAME, NCOM
      INTEGER   MAXSPEC, IPOS, SL1, SL2, NRUN1
      INTEGER   IFAIL, NNEW, NRUN2
      LOGICAL  SAME, OK, POS, PASS, EOBJ, DEFAULT, FOK
      REAL   DA, DT, DX
      REAL  CPA, CPO
C
C     Functions
C
      DATA SLOT1, SLOT2, SLOT3, SLOT4/1, 1, 2, 2/
      DATA METHOD/'I'/
      DATA DA, DT, DX/1., 60., 1.E-4/
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL INTR_IN('First slot to calibrate', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to calibrate', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST1.GT.0) THEN
            CALL SETSLOT(LIST1, NLIST1, MXLIST, SLOTS1, 
     &           NSLOTS1, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra to calibrate'
            CALL GETLIS(LIST1, NLIST1, MXLIST, MAXSPEC, SLOTS1, 
     &           NSLOTS1, MXSLOTS)
         END IF
         IF(NSLOTS1.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST1, NLIST1, MXLIST, SLOTS1, 
     &        NSLOTS1, MXSLOTS)
      END IF
*     
      CALL INTR_IN('First arc slot', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT3, 0, MAXSPEC, IFAIL)
      SLOT4 = MAX(SLOT3, SLOT4)
      CALL INTR_IN('Last arc slot', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT4, SLOT3, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT3.EQ.0 .AND. SLOT4.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT3.EQ.0 .AND. SLOT4.EQ.0) THEN
         IF(DEFAULT .AND. NLIST2.GT.0) THEN
            CALL SETSLOT(LIST2, NLIST2, MXLIST, SLOTS2, 
     &           NSLOTS2, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of arc spectra'
            CALL GETLIS(LIST2, NLIST2, MXLIST, MAXSPEC, SLOTS2, 
     &           NSLOTS2, MXSLOTS)
         END IF
         IF(NSLOTS2.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT3, SLOT4, LIST2, NLIST2, MXLIST, SLOTS2, 
     &        NSLOTS2, MXSLOTS)
      END IF
*     
      CALL CHAR_IN('Method of applying arc (N,I,E)', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, METHOD, IFAIL)
      CALL UPPER_CASE(METHOD)
      IF(METHOD.NE.'N' .AND. METHOD.NE.'I' .AND. METHOD.NE.'E') THEN
         WRITE(*,*) 'Invalid method. Possibilities are:'
         WRITE(*,*) ' '
         WRITE(*,*) ' N -- Nearest arc used'
         WRITE(*,*) ' I -- Interpolate, nearest when not bracketed'
         WRITE(*,*) ' E -- Interpolate and extrapolate'
         WRITE(*,*) ' '
         METHOD = 'I'
         IFAIL = 1
         GOTO 999
      END IF
      CALL REAL_IN('Maximum separation (mins of arc)', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, DA, 0., 1.E10, IFAIL)
      CALL REAL_IN('Maximum time diff (mins)', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, DT, 0., 1.E10, IFAIL)
      CALL REAL_IN('Maximum extract posn diff (pixels)', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, DX, 0., 1.E10, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
C     Loop through arc spectra looking for JD
C
      WRITE(*,*) ' '
      WRITE(*,*) 'Checking arc spectra'
      WRITE(*,*) ' '
      NSAME = 0
      IPOS = 1
      DO I = 1, NSLOTS2
        SLOT = SLOTS2(I)
        IF(NPIX(SLOT).GT.0) THEN
          CALL HGETC('TYPE', WHAT, SLOT, MXSPEC, NMCHAR, HDCHAR, 
     &          NCHAR, MXCHAR, IFAIL)
          IF(IFAIL.NE.0 .OR. .NOT.SAME(WHAT,'ARC')) THEN
             WRITE(*,*) 'Slot ',SLOT,' does not seem to be an arc.'
          END IF
          IF(NARC(SLOT).EQ.0) THEN
             WRITE(*,*) 'Slot ',SLOT,' has no wavelength calibration.'
             GOTO 999
          END IF
          CALL HGETD('RJD', JD, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &         NDOUB, MXDOUB, IFAIL)
          IF(IFAIL.EQ.0) THEN
             WRITE(*,*) 'Arc in slot ',SLOT,', JD = ',JD
          ELSE
             WRITE(*,*) 'No JD found for arc in slot ',SLOT
             GOTO 999
          END IF
          IF(NSAME.EQ.0) THEN
             NSAME = NARC(SLOT)
          ELSE IF(NSAME.NE.NARC(SLOT)) THEN
             WRITE(*,*) 'Incompatible numbers of arc coefficients'
             WRITE(*,*) 'First = ',NSAME,', this = ',NARC(SLOT)
             GOTO 999
          END IF
        ELSE
           WRITE(*,*) 'Slot ',SLOT,' is empty and will be ignored.'
        END IF
      END DO               
C
C     Now loop through spectra
C
      TWOPI = 8.D0*ATAN(1.D0)
      DO I = 1, NSLOTS1
         SLOT = SLOTS1(I)
         IF(NPIX(SLOT).GT.0) THEN
            WRITE(*,*) ' '
            CALL HGETC('TYPE', WHAT, SLOT, MXSPEC, NMCHAR, HDCHAR, 
     &           NCHAR, MXCHAR, IFAIL)
            IF(IFAIL.EQ.0 .AND. SAME(WHAT,'ARC')) THEN
               WRITE(*,*) 'Spectrum ',SLOT,
     &              ' is an arc and cannot be written over'
               GOTO 200
            END IF
            CALL HGETD('RJD', JD, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            IF(IFAIL.EQ.0) THEN
               WRITE(*,*) 'Slot ',SLOT,', JD = ',JD
            ELSE
               WRITE(*,*) 'No JD found for slot ',SLOT
               GOTO 200
            END IF
            CALL HGETR('Extract position', CPO, SLOT, MXSPEC, 
     &           NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
            EOBJ = IFAIL.EQ.0
C
C     Determine revised list of arcs which satisfy restriction on
C     position, time and extraction difference. If position and extract
C     position are undetermined, arc is allowed.
C
          CALL HGETD('RA', RAO, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
          OK = IFAIL.EQ.0
          CALL HGETD('DEC', DECO, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &         NDOUB, MXDOUB, IFAIL)
          OK = OK .AND. IFAIL.EQ.0
          
          CALL HGETI('Fibre', FIBREO, SLOT, MXSPEC, NMINTR, HDINTR, 
     &         NINTR, MXINTR, IFAIL)
          FOK = IFAIL.EQ.0
          NNEW = 0
          DO J=1, NSLOTS2
C
C     Check whether times and extract positions are ok
C
             ASLOT = SLOTS2(J)
             IF(NPIX(ASLOT).EQ.NPIX(SLOT)) THEN
                CALL HGETD('RJD', JDA, ASLOT, MXSPEC, NMDOUB, 
     &               HDDOUB, NDOUB, MXDOUB, IFAIL)
                PASS  = IFAIL.EQ.0 .AND. 1.44E3*ABS(JD-JDA).LT.DT 
                IF(EOBJ) THEN
                   CALL HGETR('Extract position', CPA, ASLOT, 
     &                  MXSPEC, NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
                   IF(IFAIL.EQ.0) PASS = PASS 
     &                  .AND. ABS(CPA-CPO).LT.DX
                END IF
C
C     Check positions if necessary
C
                IF(OK) THEN
                   POS  = .TRUE.
                   CALL HGETD('RA', RAA, ASLOT, MXSPEC, NMDOUB, 
     &                  HDDOUB, NDOUB, MXDOUB, IFAIL)
                   POS = POS .AND. IFAIL.EQ.0
                   CALL HGETD('DEC', DECA, ASLOT, MXSPEC, NMDOUB, 
     &                  HDDOUB, NDOUB, MXDOUB, IFAIL)
                   POS = POS .AND. IFAIL.EQ.0
                   IF(POS) THEN
                      DIFF = 6.D1*SQRT((DECA-DECO)**2+
     &                     (COS(TWOPI*DECO/3.6D2)*(RAA-RAO)*1.5D1)**2)
                      PASS = PASS .AND. DIFF.LE.DA
                   END IF
                END IF

                IF(FOK) THEN
                   CALL HGETI('Fibre', FIBREA, SLOT, MXSPEC, NMINTR, 
     &                  HDINTR, NINTR, MXINTR, IFAIL)
                   IF(IFAIL.EQ.0) PASS = PASS .AND. FIBREA.EQ.FIBREO
                END IF
C     
C     Include if arc has passed all tests
C
                IF(PASS) THEN
                   NNEW = NNEW + 1
                   RJD(NNEW) = JDA
                   NLIST(NNEW) = ASLOT
                END IF
             END IF
          END DO
C
          IF(NNEW.LT.1) THEN
             WRITE(*,*) 'NO ACCEPTABLE ARCS FOR SLOT ',SLOT
             GOTO 200
          ELSE
             WRITE(*,*) NNEW,' arcs found for slot ',SLOT
             DO K = 1, (NNEW-1)/8+1
                WRITE(*,*) (NLIST(8*(K-1)+J),J=1,MIN(NNEW-8*(K-1),8))
             END DO
          END IF
C     
C     Sort into ascending order
C
          IF(METHOD.EQ.'I' .OR. METHOD.EQ.'E')
     &         CALL DHEAPSORT(NNEW, RJD, KEY)
          IF(METHOD.EQ.'I' .OR. METHOD.EQ.'E') THEN
C
C     Search for position of JD compared to arcs.
C
             DO J = 1, NNEW
                IF(RJD(KEY(J)).GT.JD) THEN
                   IPOS = J
                   GOTO 150
                END IF
             END DO
             IPOS = NNEW + 1
 150         CONTINUE
             IF(((IPOS.EQ.1 .OR. IPOS.EQ.NNEW+1) .AND.
     &            METHOD.EQ.'I') .OR. NNEW.EQ.1) THEN
                IPOS = MAX(1, MIN(NNEW, IPOS))
                WRITE(*,'(1X,A,I4,A,F14.6)') 
     &               'Nearest arc slot ',NLIST(IPOS),
     &               ', JD = ',RJD(IPOS)
                NARC(SLOT) = NARC(NLIST(IPOS))
                DO J = 1, ABS(NARC(NLIST(IPOS)))
                   ARC(J,SLOT) = ARC(J,NLIST(IPOS))
                END DO
                AUSED = ' '
                CALL HGETI('Record', NRUN1, NLIST(IPOS), MXSPEC, 
     &               NMINTR, HDINTR, NINTR, MXINTR, IFAIL)          
                WRITE(AUSED,'(1X,I7)') NRUN1
             ELSE
C
C     Interpolation will be used.
C
                IF(IPOS.EQ.1) THEN
                   I1 = 1
                   I2 = 2
                ELSE IF(IPOS.EQ.NNEW+1) THEN
                   I1 = NNEW-1
                   I2 = NNEW
                ELSE
                   I1 = IPOS-1
                   I2 = IPOS
                END IF
                SL1 = KEY(I1)
                SL2 = KEY(I2)
                WRITE(*,*) 'Arc scale will be interpolated between'
                WRITE(*,'(1X,A,I4,A,F14.6,A,I4,A,F14.6)') 
     &               'Slot ',NLIST(SL1),', JD = ',RJD(SL1),
     &               ' and slot ',NLIST(SL2),', JD = ',RJD(SL2)
                F1 = (RJD(SL2)-JD)/(RJD(SL2)-RJD(SL1))
                F2 = 1.D0 - F1             
                SL1 = NLIST(SL1) 
                SL2 = NLIST(SL2) 
                NARC(SLOT) = NARC(SL1)
                DO J = 1, ABS(NARC(SL1))
                   ARC(J,SLOT) = F1*ARC(J,SL1)+F2*ARC(J,SL2)
                END DO            
                AUSED = ' '
                CALL HGETI('Record', NRUN1, SL1, MXSPEC, 
     &               NMINTR, HDINTR, NINTR, MXINTR, IFAIL)          
                CALL HGETI('Record', NRUN2, SL2, MXSPEC, 
     &               NMINTR, HDINTR, NINTR, MXINTR, IFAIL)          
                WRITE(AUSED,'(1X,I7,A1,I7)') NRUN1, ',', NRUN2          
             END IF
             CALL HSETC('Arc(s) used', AUSED, SLOT, MXSPEC, NMCHAR, 
     &            HDCHAR, NCHAR, MXCHAR, IFAIL)
          ELSE IF(METHOD.EQ.'N') THEN
             DIFF = 1.D30
             DO J = 1, NNEW
                IF(ABS(RJD(J)-JD).LT.DIFF) THEN
                   DIFF = ABS(RJD(J)-JD)
                   IPOS = J
                END IF
             END DO
             WRITE(*,'(1X,A,I4,A,F14.6)') 
     &            'Nearest arc slot ',NLIST(IPOS),', JD = ',RJD(IPOS)
             NARC(SLOT) = NARC(NLIST(IPOS))
             DO J = 1, ABS(NARC(NLIST(IPOS)))
                ARC(J,SLOT) = ARC(J,NLIST(IPOS))
             END DO
             AUSED = ' '
             CALL HGETI('Record', NRUN1, NLIST(IPOS), MXSPEC, 
     &            NMINTR, HDINTR, NINTR, MXINTR, IFAIL)          
             WRITE(AUSED,'(1X,I7)') NRUN1
             CALL HSETC('Arc(s) used', AUSED, SLOT, MXSPEC, NMCHAR, 
     &            HDCHAR, NCHAR, MXCHAR, IFAIL)
          END IF
       ELSE
          WRITE(*,*) 'Slot ',SLOT,' is empty and will be skipped.'
       END IF
 200   CONTINUE
      END DO
 999  RETURN
      END
      
