#!/usr/local/bin/perl -w
#
# Perl script finds all molly files starting from the current
# directory downwards, reads them in and sets real header items 
# containing some basic information about the data. This is to 
# facilitate setting up a database.
#
# Entries set are:
#
# 'Min wave'    -- Smallest wavelength
# 'Max wave'    -- Largest wavelength
# 'Dispersion'  -- A/pix
# 'Mean counts' -- Mean counts
# 'Mean errors' -- Mean count error
# 'Mean flux'   -- Mean flux
#
# Invoke as 'analdat.pl' in whatever directory you want to process.
# The files are rweritten to the same name as the old ones plus .new.
# This is for safety. If you are happy everything is working then
# you should rename the files (e.g. using 'transfer.pl')
#
# Written by T.R.Marsh Aug 1998.

use lib "/home/sousun/trm/code/molly/perl";

use PDL;
use Molly;
use Gsubs;

# Get list of molly files starting from the current diectory.

@molly_file = findmol('.');

# Go through them setting target types.

foreach(@molly_file){

    # Ignore temp, junk and crap files

    next if (/temp\d*\.mol$/ || /junk\d*\.mol$/ || /crap\d*\.mol$/);

    $input  = $_;
    $output = "$_.new";

    print "\nSetting file $input\n";

    # if failure anywhere, remove output.

    if(!danal($input, $output)){
	unlink $output;
    }
}

exit;

# Subroutine that reads one file, looks for object names, tries to
# identify their types

sub danal{
    my ($infile, $outfile) = @_;
    my $nspec = 0;
    my %noid = ();
    my %noname = ();
    my ($spec,$hdr,$wave,$wmin,$wmax,$mflux,$mcerr,$mcnts);
    my ($npix,$disp,$cerr);

    # Open files

    if(!open(INPUT, $infile)){
	print "Failed to open $infile for input\n";
	return 0;
    }
    if(!open(OUTPUT, ">$outfile")){
	print "Failed to open $outfile for output\n";
	close(INPUT);
	return 0;
    }

    while(cmol(*INPUT)){
	$spec = rmol(*INPUT);	
	$hdr  = $spec->gethdr;
	$nspec++;
	$wave = $spec->gethwav;
	$npix = $$hdr{'First record'}[2]; 
	$wmin = $wave->min;
	$wmax = $wave->max;
	$spec->areal('Min wave',$wmin);
	$spec->areal('Max wave',$wmax);
	$disp  = ($wmax - $wmin)/($npix-1);
	$spec->areal('Dispersion',$disp);
	$mcnts = $spec->getcnts->avg;
	$mflux = $spec->getflux->avg;
	$cerr  = $spec->getcerr;
	$mcerr = sqrt(avg($cerr*$cerr));
	$spec->areal('Mean counts',$mcnts);
	$spec->areal('Mean errors',$mcerr);
	$spec->areal('Mean flux',$mflux);

	# Write out modified spectra
	
	$spec->wmol(*OUTPUT);
    }
    close(INPUT);
    close(OUTPUT);

    return 1;
}


