# This provides a straighforward way of reading molly
# files into Perl. Molly.pm provides a superior PDL
# based method.
#
#read_molly
#
# read_molly($filehandle, \$npix, \$narc, \%header, \@arc, \@counts, \@errors, 
#            \@flux)
#
# Subroutine for reading a molly spectrum, assuming currently positioned
# at start of headers. Returns associative header array, counts, errors
# and fluxes.
#
#read_molly

sub read_molly{

    local($filehandle, *npix, *narc, *header, *arc,
	  *counts, *errors, *flux) = @_;
    local($fcode,$units,$nchar,$ndoub,$nintr,$nreal);
    local($buffer, @data);
    local($nhead,$template,$nbytes,$ndata,$ncoeff);
    local(@header_values,@header_names);
    
    @arc           = ();
    @data          = ();
    @counts        = ();
    @errors        = ();
    @flux          = ();
    @header_values = ();
    @header_names  = ();
    %header        = ();
#
# Read and interpret first line of header
#

    if(read($filehandle, $buffer, 52)) {
	
	($fcode,$units,$npix,$narc,$nchar,$ndoub,$nintr,$nreal) =
	    unpack("x4ia16i7", $buffer);

#
# Read and interpret header item names
#
	$nhead = $nchar + $ndoub + $nintr + $nreal;
	read($filehandle, $buffer, 16*$nhead+8);
	$template = "x4";
	for (1..$nhead) {
	    $template .= "A16";
	}
	@header_names = unpack($template, $buffer);
	foreach(@header_names){
	    s/ *$//;
	}

#
# Read and interpret header item values
#

	$nbytes = 32*$nchar + 8*$ndoub + 4*$nintr + 4*$nreal + 8;
	read($filehandle, $buffer, $nbytes);
	$template = "x4";
	for (1..$nchar) {
	    $template .= "A32";
	}
	$template .= "d$ndoub i$nintr f$nreal";
	@header_values = unpack($template, $buffer);
 
#
# Convert header to an associative array
#
	
	for (0..$#header_names) {
	    $header{$header_names[$_]} = $header_values[$_];
	}

#
# Read arc coefficients
#

	$nbytes = $narc < 0 ? -8*$narc + 8 : 8*$narc + 8;
	$ncoeff = $narc < 0 ? -$narc : $narc;
	
	read($filehandle, $buffer, $nbytes);
	$template = "x4d$ncoeff";
	@arc = unpack($template, $buffer);

#
# Read data
#

	$nbytes = 12*$npix + 8;
	$ndata  =  4*$npix;
	read($filehandle, $buffer, $nbytes);
	$template = "x4f$ndata";
	@data = unpack($template, $buffer);
	@counts = @data[0..$npix-1];
	@errors = @data[$npix..2*$npix-1];
	@flux   = @data[2*$npix..3*$npix-1];
    }
}				

#skip_molly
#
# skip_molly($filehandle, $nskip)
#
# Subroutine for skipping a molly spectrum, assuming currently positioned
# at start of headers. 
#
#skip_molly

sub skip_molly{			

    local($filehandle, $nskip) = @_;
    local($fcode,$units,$npix,$narc,$nchar,$ndoub,$nintr,$nreal);
    local($nbytes, $buffer);

    for (1..$nskip) {
#
# Read and interpret first line of header
#

	if(read($filehandle, $buffer, 52)) {
	    
	    ($fcode,$units,$npix,$narc,$nchar,$ndoub,$nintr,
	     $nreal) = unpack("x4ia16i7", $buffer);
			
#
# Compute number of bytes to skip
#
	
	    $nbytes  = 16*($nchar + $ndoub + $nintr + $nreal)+8;
	    $nbytes += 32*$nchar + 8*$ndoub + 4*$nintr + 4*$nreal + 8;
	    $nbytes += $narc < 0 ? -8*$narc + 8 : 8*$narc + 8;
	    $nbytes += 12*$npix + 8;

	    read($filehandle, $buffer, $nbytes);
	}
    }
}

#calculate_wavelength
#
# @wave = calculate_wavelength(\@arc, $narc, $npix)
#
# Calculates wavelength scale equivalent to molly arc coefficients. No
# correction for the Earth's velocity is applied.
#
#calculate_wavelength

sub calculate_wavelength{

    local(*arc,$narc,$npix) = @_;
    local($i,$j,$fac,$z);
    local(@wave) = ();
	
    for $i (0..$npix-1) {
	$wave[$i] = 0.;
	$z = ($i+1)/$npix;
	if($narc > 0) {
	    $fac = 1.;
	    for $j (0..$narc-1) {
		$wave[$i] += $arc[$j]*$fac;
		$fac  *= $z;
	    }
	}
	elsif($narc < 0) {		
	    $fac = 1.;
	    for $j (0..-$narc-1) {
		$wave[$i] += $arc[$j]*$fac;
		$fac  *= $z;
	    }
	    $wave[$i] = exp($wave[$i]);
	}
	else {
	    $wave[$i] = $i + 1;
	}
	
    }
    @wave;
}				

#remove_earth_velocity
#
# remove_earth_velocity(\@wave, \%header)
#
# Looks up Earth velocity and corrects wavelengths for it.
#
#remove_earth_velocity

sub remove_earth_velocity{

    local(*wave, *header) = @_;
    local($i,$j,$fac,$z);

    $corr_factor = 1. - $header{'Vearth'}/2.997925e5;
    
    for (@wave) {
	$_ *= $corr_factor;
    }
}

#correct_fluxes
#
# correct_fluxes(\@counts, \@errors, \@flux, \@ferr)
#
# Checks for 0 count pixels and corrects fluxes. Also generates
# array of flux errors.
#
#correct_fluxes

sub correct_fluxes{

    local(*counts, *errors, *flux, *ferr) = @_;
    local($i);
    
    @ferr = ();
    for $i (0..$#flux) {
	if($counts[$i] == 0.) {
	    $ferr[$i] = $errors[$i]/$flux[$i];
	    $flux[$i] = 0.;
	}
	else {
	    $ferr[$i] = $flux[$i]*$errors[$i]/$counts[$i];
	}
	$ferr[$i] = $ferr[$i]*$errors[$i] > 0 ? $ferr[$i] : -$ferr[$i];
    }
}

#wave_to_kms
#
# @velocity = wave_to_kms(\@wave, $wzero)
#
# Converts from wavelength to velocity in km/s given a central
# wavelength corresponding to zero velocity which must have the
# same units as the wavelength array.
#
#wave_to_kms

sub wave_to_kms{

    local(*wave, $wzero) = @_;
    local($i,$vlight);
    local(@velocity) = ();
    
    $vlight = 2.997925e5;

    for $i (0..$#wave) {
	$velocity[$i] = $vlight*($wave[$i]-$wzero)/$wzero;
    }
    @velocity;
}


1;












