# TRM 15/04/2002
#
# Perl subroutine for generating mgfit files. Meant for calling from other scripts. 
# The basic idea is that it tries to do the fittin automatically, including re-fits
# to account for any velocity changes between spectra.
#
# Arguments:
#
#  1) File name (with or without .mol)
#  2) where to find molly
#  3) continuum fit mask file
#  4) spline or poly
#  5) number of coefficients
#  6) mgfit mask file
#  7) mgfit fit file (output)
#  8) number of poly coeffs for mgfit
#  9) pivot wavelength
# 10) wavelength of line
# 11) number of gaussians
# 12) wavelength of line
# 13) number of gaussians
# etc
#
# The routine carries out a series of 7 fits as follows:
# 
# 1) Default FWHMs and heights, get overall mean velocity.
# 2) Single velocity, fit heights
# 3) Single velocity, fit FWHMs
# 4) Single velocity, fit FWHMs and heights
# 5) Fit individual velocities, fixed FWHM, heights
# 6) Shift out velocities, refine FWHMs and heights
# 7) Fit individual velocities, fixed FWHM, heights. 
#
# It does it in this way in an attempt to be robust, although
# this is by no means always the case. In difficult case, you
# may create a file called 'mgfit.defs' which contains a series
# of lines as follows:
#
# 4861.327 25 -0.12
#
# i.e. the rest wavelength, FWHM and height of a gauusian. These
# will be used to define default values for the fit. Blank lines and
# one starting with # will be ignored.
#
# A fair few output files are generated, which can be used to assess
# problems. These are:
#
# zzz_fit.1    -- log of first fit# zzz_mgfit.1  -- initial fit
#
# etc 7 times over.
# 
# zzz_hjds.dat     -- the hjds
# zzz_number.log   -- to determine number of spectra
# zzz_velocities.1 -- file of velocities used to shift out to refine FWHMs etc.

sub mgfit{

    -w '.' or die "mgfit error: working directory cannot be written to!\n";

# interpret arguments, check them as well.

    my $molly_file   = shift @_;
    if($molly_file !~ /\.mol$/){
	$molly_file = "${molly_file}.mol";
    }
    -e $molly_file or
	die "mgfit error: molly file = $molly_file does not exist!\n";
    -r $molly_file or
	die "mgfit error: molly file = $molly_file is not readable!\n";

    my $molly   = shift @_;
    -e $molly or
	die "mgfit error: molly executable = $molly does not exist!\n";
    -x $molly or
	die "mgfit error: molly executable = $molly is not executable!\n";

    my $contfit_mask = shift @_;
    -e $contfit_mask or
	die "mgfit error: continuum fit mask file = $contfit_mask does not exist!\n";
    -r $contfit_mask or
	die "mgfit error: continuum fit mask file = $contfit_mask is not readable!\n";

    my $fit_type     = shift @_;
    ($fit_type eq 'spline' || $fit_type eq 'poly') or 
	die "mgfit error: continuum fit type must be either 'spline' or 'poly'\n";

    my $contfit_ncoeff   = shift @_;
    ($contfit_ncoeff > 0) or 
	die "mgfit error: number of continuum fit coefficients must be > 0\n";
    
    my $mgfit_mask = shift @_;
    -e $mgfit_mask or
	die "mgfit error: mgfit mask file = $mgfit_mask does not exist!\n";
    -r $mgfit_mask or
	die "mgfit error: mgfit mask file = $mgfit_mask is not readable!\n";

    my $mgfit_fit = shift @_;

    my $mgfit_npoly   = shift @_;
    ($mgfit_npoly > 0) or 
	die "mgfit error: number of mgfit poly coefficients must be > 0\n";

    my $pivot_wave  = shift @_;
    ($pivot_wave > 100 && $pivot_wave < 100000) or 
	die "mgfit error: pivot wavelength = $pivot_wave must be between 100 and 100000 A!\n";

    my $nline = 0;
    my ($defs,$new,$wref,$ngauss);
    my (@wave, @ngauss, @fwhm, @height);

    if(-e "mgfit.defs"){

# default file exists, which overrides anay other arguments

	$defs = 1;
	open(MGDEFS, "mgfit.defs") or
	    die "Could not open mgfit.defs for reading\n";
	while(<MGDEFS>){
	    if(!/^\#/ && !/^\s*$/){
		($wave,$fwhm,$height) = split;
		($wave > 0 && $fwhm > 0 && $height != 0.) or
		    die "Invalid default parameters: $wave $fwhm $height\n";
		$new = 1;
		for($i=0; $i<$nline; $i++){
		    if ($wave == $wave[$i]){
			$new = 0;
			$wref = $i;
		    }
		}
		if($new){
		    $wave[$nline]      = $wave;
		    $fwhm[$nline][0]   = $fwhm;
		    $height[$nline][0] = $height;
		    $ngauss[$nline]++;
		    $nline++;
		}else{
		    $fwhm[$wref][$ngauss[$wref]]   = $fwhm;
		    $height[$wref][$ngauss[$wref]] = $height;
		    $ngauss[$wref]++;
		}
	    }
	}
	close(MGDEFS);
	print STDERR "Default gaussian parameters loaded.\n";
    }else{
	$defs = 0;
	while(@_){
	    $wave[$nline] = shift @_;
	    ($wave[$nline] > 100. && $wave[$nline] < 100000.) or
		die "mgfit error: wavelength = $wave[$nline] must be between 100 and 100000 A!\n";
	    scalar(@_) or
		die "Each wavelength specified requires a number of gaussians too!\n"; 
	    $ngauss[$nline] = shift @_;
	    ($ngauss[$nline] > 0 && $ngauss[$nline] < 10) or
		die "mgfit error: ngauss = $ngauss[$nline] must be from 1 to 9!\n";
	    $nline++;
	}
	$nline or die "mgfit error: no wavelengths were specified!\n";
    }

# OK, finally we can start. To save lots of unnecessary output, first
# work out number of spectra.

    my $file = 'zzz_number.log';
    open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
    print MOLLY "confirm\n";
    print MOLLY "mxpix 3000 sure\n";
    print MOLLY "load $molly_file 1 500 1\n";  
    print MOLLY "quit\ny\n";
    close(MOLLY);

    my $nspec = 0;
    open(FILE, $file) or die "Could not open $file\n";
    while(<FILE>){
	if(/^\s*Read\s*(\d+)\sspectr\S*\s*from/){
	    $nspec = $1;
	}
    }
    close(FILE);
    ($nspec > 0) or die "mgfit error: no spectra found!\n";

# OK now get on with it. First generate default fit file,

    my($i,$j,$n,@poly_arg);
    $poly_arg[0] = 1.;
    $poly_arg[1] = ' ';
    for($i=1;$i<$mgfit_npoly;$i++){
	$poly_arg[2*$i]   = 0.;
	$poly_arg[2*$i+1] = ' ';
    }
    my @wave_arg;
    $n = 0;
    my $start;
    for($i=0;$i<$nline;$i++){
	$wave_arg[$n++]   = $wave[$i];
	$wave_arg[$n++]   = $ngauss[$i];
	$start = 2.;
	for($j=0;$j<$ngauss[$i];$j++){
	    if($defs){
		$wave_arg[$n++] = $fwhm[$i][$j];
	    }else{
		$wave_arg[$n++] = $start;
		$start *= 4.;
	    }
	    $wave_arg[$n++] = 'F';
	    if($defs){
		$wave_arg[$n++] = $height[$i][$j];
	    }else{
		$wave_arg[$n++] = -0.1;
	    }
	    $wave_arg[$n++] = 'F';
	}
    }

    mgfit_gen('zzz_mgfit.1',$mgfit_npoly,$pivot_wave,@poly_arg,0.,' ',@wave_arg);

# OK first fit.
	    
    $file = 'zzz_fit.1';
    open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
    print MOLLY "confirm\n";
    print MOLLY "mxpix 3000 sure\n";
    print MOLLY "load $molly_file 1 $nspec 1\n";  
    if($fit_type eq 'spline'){
	print MOLLY "sfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }else{
	print MOLLY "pfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }
    print MOLLY "load\n";
    print MOLLY "$contfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "div 1 $nspec ",$nspec+1," ",2*$nspec," 1\n";

# OK, spectra now divided by continuum, mgfit them.

    print MOLLY "mgfit 1 $nspec 0 zzz_mgfit.1 \" \" -6 6 A yes\n";
    print MOLLY "load\n";
    print MOLLY "$mgfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "quit\ny\n";
    close(MOLLY);

# Extract mean velocity

    my $vel;
    my $found = 0;
    open(FILE, "$file") or die "Could not open $file!\n";
    while(<FILE>){
	if(/^\s*Variable: vel\s*=\s*(\S*)/){
	    $found = 1;
	    $vel   = $1;
	}elsif(/GAUSSJ/){
	    die "Singular matrix encountered on first fit. Look at $file.\n";
	}
    }
    close(FILE);
    $found or die "Could not find mean velocity in $file!\n";

    print STDERR "First fit completed. Mean velocity = $vel\n";

# Now, if defaults, we skip straight to the fourth fit, otherwise we 
# press on trying to fix FWHM and heights automatically.

    if(!$defs){

# Re-generate the fit file (allow gaussians to vary, fix poly apart from constant)

	$poly_arg[0] = 1.;
	$poly_arg[1] = ' ';
	for($i=1;$i<$mgfit_npoly;$i++){
	    $poly_arg[2*$i]   = 0.;
	    $poly_arg[2*$i+1] = 'F';
	}
	$n = 0;
	for($i=0;$i<$nline;$i++){
	    $wave_arg[$n++]   = $wave[$i];
	    $wave_arg[$n++]   = $ngauss[$i];
	    $start = 2.;
	    for($j=0;$j<$ngauss[$i];$j++){
		$wave_arg[$n++] = $start;
		$wave_arg[$n++] = 'F';
		$wave_arg[$n++] = -0.1;
		$wave_arg[$n++] = ' ';
		$start *= 4.;
	    }
	}
	mgfit_gen('zzz_mgfit.2',$mgfit_npoly,$pivot_wave,@poly_arg,$vel,'F',@wave_arg);
	
# Second-fit to refine the heights.
	
	$file = 'zzz_fit.2';
	open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
	print MOLLY "confirm\n";
	print MOLLY "mxpix 3000 sure\n";
	print MOLLY "load $molly_file 1 $nspec 1\n";  
	if($fit_type eq 'spline'){
	    print MOLLY "sfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
	}else{
	    print MOLLY "pfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
	}
	print MOLLY "load\n";
	print MOLLY "$contfit_mask\n";
	print MOLLY "quit\n";
	print MOLLY "div 1 $nspec ",$nspec+1," ",2*$nspec," 1\n";
	
# OK, spectra now divided by continuum, mgfit them.
	
	print MOLLY "mgfit 1 $nspec 0 zzz_mgfit.2 \" \" -8 8 A yes\n";
	print MOLLY "load\n";
	print MOLLY "$mgfit_mask\n";
	print MOLLY "quit\n";
	print MOLLY "quit\ny\n";
	close(MOLLY);
	
# Get revised heights
	
	open(FILE, "$file") or die "mgfit error: could not open $file!\n";
	while(<FILE>){
	    for($i=0; $i<$nline; $i++){
		for($j=0; $j<$ngauss[$i]; $j++){
		    if(/^\s*Variable: height$i$j\s*=\s*(\S*)/){
			$height[$i][$j] = $1;
		    }elsif(/GAUSSJ/){
			die "Singular matrix encountered on second fit. Look at $file.\n";
		    }
		}
	    }
	}
	close(FILE);
	for($i=0;$i<$nline;$i++){
	    for($j=0;$j<$ngauss[$i];$j++){
		defined $height[$i][$j] or die "mgfit error: height$i$j undefined!\n";
	    }
	}
	
	print STDERR "Second fit completed.\n";
	
# Re-generate the fit file (allow gaussian FWHMs to vary, fix poly apart
# from constant)
	
	$poly_arg[0] = 1.;
	$poly_arg[1] = ' ';
	for($i=1;$i<$mgfit_npoly;$i++){
	    $poly_arg[2*$i]   = 0.;
	    $poly_arg[2*$i+1] = 'F';
	}
	$n = 0;
	for($i=0;$i<$nline;$i++){
	    $wave_arg[$n++]   = $wave[$i];
	    $wave_arg[$n++]   = $ngauss[$i];
	    $start = 2.;
	    for($j=0;$j<$ngauss[$i];$j++){
		$wave_arg[$n++] = $start;
		$wave_arg[$n++] = ' ';
		$wave_arg[$n++] = $height[$i][$j];
		$wave_arg[$n++] = 'F';
		$start *= 4.;
	    }
	}
	mgfit_gen('zzz_mgfit.3',$mgfit_npoly,$pivot_wave,@poly_arg,$vel,'F',@wave_arg);
	
# Third fit to refine the FWHMs
	
	$file = 'zzz_fit.3';
	open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
	print MOLLY "confirm\n";
	print MOLLY "mxpix 3000 sure\n";
	print MOLLY "load $molly_file 1 $nspec 1\n";  
	if($fit_type eq 'spline'){
	    print MOLLY "sfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
	}else{
	    print MOLLY "pfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
	}
	print MOLLY "load\n";
	print MOLLY "$contfit_mask\n";
	print MOLLY "quit\n";
	print MOLLY "div 1 $nspec ",$nspec+1," ",2*$nspec," 1\n";
	
# OK, spectra now divided by continuum, mgfit them.
	
	print MOLLY "mgfit 1 $nspec 0 zzz_mgfit.3 \" \" -8 8 A yes\n";
	print MOLLY "load\n";
	print MOLLY "$mgfit_mask\n";
	print MOLLY "quit\n";
	print MOLLY "quit\ny\n";
	close(MOLLY);
	
# Get revised FWHMs
	
	open(FILE, "$file") or die "mgfit error: could not open $file!\n";
	while(<FILE>){
	    for($i=0; $i<$nline; $i++){
		for($j=0; $j<$ngauss[$i]; $j++){
		    if(/^\s*Variable: fwhm$i$j\s*=\s*(\S*)/){
			$fwhm[$i][$j] = $1;
		    }elsif(/GAUSSJ/){
			die "Singular matrix encountered on third fit. Look at $file.\n";
		    }
		}
	    }
	}
	close(FILE);
	for($i=0;$i<$nline;$i++){
	    for($j=0;$j<$ngauss[$i];$j++){
		defined $fwhm[$i][$j] or die "mgfit error: fwhm$i$j undefined!\n";
	    }
	}

	print STDERR "Third fit completed.\n";

    }else{
	print STDERR "Default file loaded, skipping second and third fits.\n";
    }

# Re-generate the fit file (allow gaussians FWHM and heights to vary)

    $poly_arg[0] = 1.;
    $poly_arg[1] = ' ';
    for($i=1;$i<$mgfit_npoly;$i++){
	$poly_arg[2*$i]   = 0.;
	$poly_arg[2*$i+1] = 'F';
    }
    $n = 0;
    for($i=0;$i<$nline;$i++){
	$wave_arg[$n++]   = $wave[$i];
	$wave_arg[$n++]   = $ngauss[$i];
	for($j=0;$j<$ngauss[$i];$j++){
	    $wave_arg[$n++] = $fwhm[$i][$j];
	    $wave_arg[$n++] = ' ';
	    $wave_arg[$n++] = $height[$i][$j];
	    $wave_arg[$n++] = ' ';
	}
    }
    mgfit_gen('zzz_mgfit.4',$mgfit_npoly,$pivot_wave,@poly_arg,$vel,'F',@wave_arg);

# Fourth fit to refine the FWHMs and heights

    $file = 'zzz_fit.4';
    open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
    print MOLLY "confirm\n";
    print MOLLY "mxpix 3000 sure\n";
    print MOLLY "load $molly_file 1 $nspec 1\n";  
    if($fit_type eq 'spline'){
	print MOLLY "sfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }else{
	print MOLLY "pfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }
    print MOLLY "load\n";
    print MOLLY "$contfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "div 1 $nspec ",$nspec+1," ",2*$nspec," 1\n";

# OK, spectra now divided by continuum, mgfit them.

    print MOLLY "mgfit 1 $nspec 0 zzz_mgfit.4 \" \" -8 8 A yes\n";
    print MOLLY "load\n";
    print MOLLY "$mgfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "quit\ny\n";
    close(MOLLY);

# Get revised FWHMs

    undef @fwhm;
    undef @height;

    open(FILE, "$file") or die "mgfit error: could not open $file!\n";
    while(<FILE>){
	for($i=0; $i<$nline; $i++){
	    for($j=0; $j<$ngauss[$i]; $j++){
		if(/^\s*Variable: fwhm$i$j\s*=\s*(\S*)/){
		    $fwhm[$i][$j] = $1;
		}elsif(/^\s*Variable: height$i$j\s*=\s*(\S*)/){
		    $height[$i][$j] = $1;
		}elsif(/GAUSSJ/){
		    die "Singular matrix encountered on fourth fit. Look at $file.\n";
		}
	    }
	}
    }
    close(FILE);
    for($i=0;$i<$nline;$i++){
	for($j=0;$j<$ngauss[$i];$j++){
	    defined $height[$i][$j] or die "mgfit error: height$i$j undefined!\n";
	    defined $fwhm[$i][$j] or die "mgfit error: fwhm$i$j undefined!\n";
	}
    }

    print STDERR "Fourth fit completed.\n";

# Re-generate the fit file again now to measure individual velocities

    $poly_arg[0] = 1.;
    $poly_arg[1] = ' ';
    for($i=1;$i<$mgfit_npoly;$i++){
	$poly_arg[2*$i]   = 0.;
	$poly_arg[2*$i+1] = ' ';
    }
    $n = 0;
    for($i=0;$i<$nline;$i++){
	$wave_arg[$n++]   = $wave[$i];
	$wave_arg[$n++]   = $ngauss[$i];
	for($j=0;$j<$ngauss[$i];$j++){
	    $wave_arg[$n++] = $fwhm[$i][$j];
	    $wave_arg[$n++] = 'F';
	    $wave_arg[$n++] = $height[$i][$j];
	    $wave_arg[$n++] = 'F';
	}
    }
    mgfit_gen('zzz_mgfit.5',$mgfit_npoly,$pivot_wave,@poly_arg,$vel,' ',@wave_arg);

# Third fit 

    $file = 'zzz_fit.5';
    open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
    print MOLLY "confirm\n";
    print MOLLY "mxpix 3000 sure\n";
	print MOLLY "load $molly_file 1 $nspec 1\n";  
    if($fit_type eq 'spline'){
	print MOLLY "sfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }else{
	print MOLLY "pfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }
    print MOLLY "load\n";
    print MOLLY "$contfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "div 1 $nspec ",$nspec+1," ",2*$nspec," 1\n";

# OK, spectra now divided by continuum, mgfit them.

    print MOLLY "mgfit 1 $nspec 0 zzz_mgfit.5 \" \" -5 5 I yes yes\n";
    print MOLLY "load\n";
    print MOLLY "$mgfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "quit\ny\n";
    close(MOLLY);

# Get velocities

    $n = 0;
    my @vel;
    open(FILE, "$file") or die "mgfit error: could not open $file!\n";
    while(<FILE>){
	if(/^\s*Variable: vel\s*=\s*(\S*)/){
	    $vel[$n++] = $1;
	}elsif(/GAUSSJ/){
	    die "Singular matrix encountered on fifth fit. Look at $file.\n";
	}
    }
    close(FILE);
    for($i=0;$i<$nspec;$i++){
	defined $vel[$i] or die "mgfit error: velocity for spectrum ",$i+1," is undefined!\n";
    }

# Now write a file of velocities to be shifted out in order to refine the FWHM and heights

    $file = 'zzz_velocities.1';
    open(FILE,">$file") or die "Failed to open $file!\n";
    foreach $vel (@vel){
	print FILE "$vel\n";
    }
    close(FILE);

    print STDERR "Fifth fit completed.\n";

# Re-generate the fit file yet again now to refine the FWHM and heights

    $poly_arg[0] = 1.;
    $poly_arg[1] = ' ';
    for($i=1;$i<$mgfit_npoly;$i++){
	$poly_arg[2*$i]   = 0.;
	$poly_arg[2*$i+1] = ' ';
    }
    $n = 0;
    for($i=0;$i<$nline;$i++){
	$wave_arg[$n++]   = $wave[$i];
	$wave_arg[$n++]   = $ngauss[$i];
	for($j=0;$j<$ngauss[$i];$j++){
	    $wave_arg[$n++] = $fwhm[$i][$j];
	    $wave_arg[$n++] = ' ';
	    $wave_arg[$n++] = $height[$i][$j];
	    $wave_arg[$n++] = ' ';
	}
    }
    mgfit_gen('zzz_mgfit.6',$mgfit_npoly,$pivot_wave,@poly_arg,0.,'F',@wave_arg);

# molly again

    $file = 'zzz_fit.6';
    open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
    print MOLLY "confirm\n";
    print MOLLY "mxpix 3000 sure\n";
	print MOLLY "load $molly_file 1 $nspec 1\n";  
    if($fit_type eq 'spline'){
	print MOLLY "sfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }else{
	print MOLLY "pfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }
    print MOLLY "load\n";
    print MOLLY "$contfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "div 1 $nspec ",$nspec+1," ",2*$nspec," 1\n";

# Now shift out velocities

    print MOLLY "move 1 $nspec zzz_velocities.1 v s\n";

# Fit

    print MOLLY "mgfit 1 $nspec 0 zzz_mgfit.6 \" \" -4 4 A yes\n";
    print MOLLY "load\n";
    print MOLLY "$mgfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "quit\ny\n";
    close(MOLLY);

# Get final FWHMs and heights

    undef @fwhm;
    undef @height;

    open(FILE, "$file") or die "mgfit error: could not open $file!\n";
    while(<FILE>){
	for($i=0;$i<$nline;$i++){
	    for($j=0;$j<$ngauss[$i];$j++){
		if(/^\s*Variable: fwhm$i$j\s*=\s*(\S*)/){
		    $fwhm[$i][$j] = $1;
		}elsif(/^\s*Variable: height$i$j\s*=\s*(\S*)/){
		    $height[$i][$j] = $1;
		}elsif(/GAUSSJ/){
		    die "Singular matrix encountered on second fit. Look at $file.\n";
		}
	    }
	}
    }
    close(FILE);
    for($i=0;$i<$nline;$i++){
	for($j=0;$j<$ngauss[$i];$j++){
	    defined $fwhm[$i][$j] or die "mgfit error: fwhm$i$j undefined!\n";
	    defined $height[$i][$j] or die "mgfit error: height$i$j undefined!\n";
	}
    }

    print STDERR "Sixth fit completed.\n";

# Re-generate the fit file again now to measure final individual velocities

    $poly_arg[0] = 1.;
    $poly_arg[1] = ' ';
    for($i=1;$i<$mgfit_npoly;$i++){
	$poly_arg[2*$i]   = 0.;
	$poly_arg[2*$i+1] = ' ';
    }
    $n = 0;
    for($i=0;$i<$nline;$i++){
	$wave_arg[$n++]   = $wave[$i];
	$wave_arg[$n++]   = $ngauss[$i];
	for($j=0;$j<$ngauss[$i];$j++){
	    $wave_arg[$n++] = $fwhm[$i][$j];
	    $wave_arg[$n++] = 'F';
	    $wave_arg[$n++] = $height[$i][$j];
	    $wave_arg[$n++] = 'F';
	}
    }
    mgfit_gen($mgfit_fit,$mgfit_npoly,$pivot_wave,@poly_arg,$vel,' ',@wave_arg);

# Seventh and final fit 

    $file = 'zzz_fit.7';
    open(MOLLY, "|$molly > $file") or die "Could not open a pipe to molly!\n";
    print MOLLY "confirm\n";
    print MOLLY "mxpix 3000 sure\n";
	print MOLLY "load $molly_file 1 $nspec 1\n";  
    if($fit_type eq 'spline'){
	print MOLLY "sfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }else{
	print MOLLY "pfit 1 $nspec ",$nspec+1," $contfit_ncoeff -3 3 10 0 yes\n";
    }
    print MOLLY "load\n";
    print MOLLY "$contfit_mask\n";
    print MOLLY "quit\n";
    print MOLLY "div 1 $nspec ",$nspec+1," ",2*$nspec," 1\n";

# OK, spectra now divided by continuum, mgfit them.

    print MOLLY "mgfit 1 $nspec " ,$nspec+1, " $mgfit_fit \" \" -4 4 I yes yes\n";
    print MOLLY "load\n";
    print MOLLY "$mgfit_mask\n";
    print MOLLY "quit\n";

# Plot results for first line

    $offset = 0.5;
    my $nlp;
    $nlp = 0;
    print MOLLY "ax km m\n";
    print MOLLY "offset 0.5\n";
    print MOLLY "pclose\n";
    print MOLLY "xyr -2000 2000 0 ",1.+$nspec*$offset,"\n";
    print MOLLY "dev /xs\n";
    print MOLLY "dc 2\n";
    print MOLLY "plot ",$nspec+1," ",2*$nspec," $wave[$nlp]\n";
    print MOLLY "dc 1\n";
    print MOLLY "plot 1 $nspec $wave[$nlp]\n";
    print MOLLY "close\n";

# Dumps HJDs

    print MOLLY "flis 1 $nspec zzz_hjds.dat 1 HJD\n";
    print MOLLY "quit\ny\n";
    close(MOLLY);

# Get velocities and errors

    $n = 0;
    my @verr;
    open(FILE, "$file") or die "mgfit error: could not open $file!\n";
    while(<FILE>){
	if(/^\s*Variable: vel\s*=\s*(\S*)\s\+\/-\s*(\S*)/){
	    $vel[$n]    = $1;
	    $verr[$n++] = $2;
	}elsif(/GAUSSJ/){
	    die "Singular matrix encountered on final fit. Look at $file.\n";
	}
    }
    close(FILE);
    for($i=0;$i<$nspec;$i++){
	defined $verr[$i] or die "mgfit error: velocity for spectrum ",$i+1," is undefined!\n";
    }

# Get velocities

    $n = 0;
    my ($hjd, @hjd);
    open(FILE, "zzz_hjds.dat") or die "mgfit error: could not open zzz_hjds.dat!\n";
    while(<FILE>){
	($i,$hjd)  =split;
	$hjd[$n++] = $hjd;
    }
    close(FILE);
    for($i=0;$i<$nspec;$i++){
	defined $hjd[$i] or die "mgfit error: hjd for spectrum ",$i+1," is undefined!\n";
    }

# Now write the final result to standard out.

    for($i=0;$i<$nspec;$i++){
	print "$hjd[$i] $vel[$i] $verr[$i]\n";
    }

    print STDERR "Seventh and final fit completed.\n";

}

# Generates a fit file.
#
# Arguments:
#
#  1) name of fit file
#  2) number of poly coeffs
#  3) pivot wavelength
#    For each poly cofficient:
#    i) value
#   ii) 'F' or something else if fixed or not
#  4) overall velocity
#  5) 'F' or something else if fixed or not
#  6) wavelength of line
#  7) number of gaussians. For each of these:
#     i) fwhm 
#    ii) 'F' or not
#   iii) height
#    iv) 'F' or not
#
# Repeat wavelength as necessary

sub mgfit_gen{
    
    my $mgfit_fit = shift @_;

    my $mgfit_npoly   = shift @_;
    ($mgfit_npoly > 0) or 
	die "mgfit_gen error: number of mgfit poly coefficients must be > 0\n";

    my $pivot_wave  = shift @_;
    ($pivot_wave > 100 && $pivot_wave < 100000) or 
	die "mgfit_gen error: pivot wavelength = $pivot_wave must be between 100 and 100000 A!\n";
    my $n = 0;
    my (@poly, @poly_fixed);
    while(@_ && $n < $mgfit_npoly){
	$poly[$n]         = shift @_;
	$poly_fixed[$n++] = shift @_;
    }
    ($n == $mgfit_npoly) or die "mgfit_gen error: too few poly coefficients found!\n";

    my $vel       = shift @_;
    my $vel_fixed = shift @_;

    my $nline = 0;
    my( $i, $j, @wave, @ngauss);
    my(@fwhm_fixed, @height_fixed);
    while(@_){
	$wave[$nline] = shift @_;
	($wave[$nline] > 100. && $wave[$nline] < 100000.) or
	    die "mgfit_gen error: wavelength = $wave[$nline] must be between 100 and 100000 A!\n";
	scalar(@_) or 
	    die "Each wavelength specified requires a number of gaussians too!\n"; 
	$ngauss[$nline] = shift @_;
	($ngauss[$nline] > 0 && $ngauss[$nline] < 10) or
	    die "mgfit_gen error: ngauss = $ngauss[$nline] must be from 1 to 9!\n";

	foreach($i=0; $i<$ngauss[$nline]; $i++){
	    $fwhm[$nline][$i]         = shift @_;
	    $fwhm_fixed[$nline][$i]   = shift @_;
	    $height[$nline][$i]       = shift @_;
	    $height_fixed[$nline][$i] = shift @_;
	}
	$nline++;
    }
    $nline or die "mgfit_gen error: no wavelengths were specified!\n";

# write out mgfit fit file

    open(MGFIT,">$mgfit_fit") or die "mgfit_gen error: could not open mgfit fit file = $mgfit_fit\n";

    print MGFIT "#\n";
    print MGFIT "# Fit file generated automatically by mgfit_gen\n";
    print MGFIT "#\n";
    print MGFIT "\n";

# poly stuff

    print MGFIT "poly: $pivot_wave";
    for($i=0; $i<$mgfit_npoly; $i++){
	print MGFIT " \$c$i";
    }
    print MGFIT "\n\n";

# gaussian stuff

    for($i=0; $i<$nline; $i++){
	for($j=0; $j<$ngauss[$i]; $j++){
	    print MGFIT "gaussian: $wave[$i] \$vel \$fwhm$i$j \$height$i$j\n";
	}
    }
    print MGFIT "\n";

# poly coefficient values

    for($i=0; $i<$mgfit_npoly; $i++){
	print MGFIT "\$c$i = $poly[$i]";
	if($poly_fixed[$i] eq 'F'){
	    print MGFIT " F\n";
	}else{
	    print MGFIT "\n";
	}
    }
    print MGFIT "\n";

# write gaussian coefficient values

    print MGFIT "\$vel = $vel";
    if($vel_fixed eq 'F'){
	print MGFIT " F\n\n";
    }else{
	print MGFIT "\n\n";
    }
    for($i=0; $i<$nline; $i++){
	for($j=0; $j<$ngauss[$i]; $j++){
	    print MGFIT "\$fwhm$i$j = $fwhm[$i][$j]";
	    if($fwhm_fixed[$i][$j] eq 'F'){
		print MGFIT " F\n";
	    }else{
		print MGFIT "\n";
	    }   
	    print MGFIT "\$height$i$j = $height[$i][$j]";
	    if($height_fixed[$i][$j] eq 'F'){
		print MGFIT " F\n\n";
	    }else{
		print MGFIT "\n\n";
	    }   
	}
    }
    close(MGFIT);
}

1;
