      SUBROUTINE LINFIT(Y,SIG,NDATA,A,MA,LISTA,MFIT,COVAR,NCVM,
     &CHISQ,NFIT,VECT,MXDATA,W,IFAIL)
*
* Adapted from Numerical recipes linear fit routine
*
* D Y(NDATA)   -- Y values
* D SIG(NDATA) -- Uncertainties, -ve to ignore point.
* I NDATA      -- Number of points to be fitted
* D A(MA)      -- coefficients
* I MA         -- Total number of coefficients
* I LISTA(MA)  -- Numbers of coefficients to be adjusted
* I MFIT       -- How many coefficients of MA are to be adjusted
* D COVAR(NCVM,NCVM) -- Normal matrix
* I NCVM       -- Dimension of matrix .GE. MFIT
* D CHISQ      -- Chi**2 of fit
* I NFIT       -- Actual number of points used in fit
* D VECT(MXDATA, MA) -- Fit functions evaluated at each point.
* I MXDATA     -- First dimension declared in calling routine
* D W(NDATA)   -- Work array
* I IFAIL      -- 0 if all ok
*

      IMPLICIT NONE
      INTEGER NDATA, MA, MFIT, NCVM, MXDATA
      DOUBLE PRECISION Y(NDATA), SIG(NDATA)
      INTEGER LISTA(MA)
      DOUBLE PRECISION VECT(MXDATA, MA), W(NDATA)
      DOUBLE PRECISION A(MA), COVAR(NCVM,NCVM)
      INTEGER MMAX
      PARAMETER (MMAX=100)
      DOUBLE PRECISION BETA(MMAX), SUM, CHISQ
      INTEGER I, J, K, KK, IHIT, IFAIL, NFIT
*
      IFAIL = 0
      KK = MFIT + 1
      DO J=1, MA
        IHIT = 0
        DO K = 1, MFIT
          IF(LISTA(K).EQ.J) IHIT = IHIT + 1
        END DO
        IF(IHIT.EQ.0) THEN
          LISTA(KK) = J
          KK = KK + 1
        ELSE IF(IHIT.GT.1) THEN
          IFAIL = 1
          WRITE(*,*) 'Improper set in LISTA; LFIT'
          RETURN
        END IF
      END DO
      IF(KK.NE.MA+1) THEN
        IFAIL = 1
        WRITE(*,*) 'Improper set in LISTA; LFIT'
        RETURN
      END IF
*
* Compute predicted values for coefficients that are set
*
      DO I = 1, NDATA
        W(I) = 0.
      END DO
      DO J = MFIT+1, MA
        DO I = 1, NDATA
          W(I) = W(I) + A(LISTA(J))*VECT(I,LISTA(J))
        END DO
      END DO
*
      DO J=1, MFIT
        DO K=1, MFIT
          COVAR(J,K) = 0.
        END DO
      END DO
*
* Compute vector
*
      DO J = 1, MFIT
        SUM = 0.D0
        DO I = 1, NDATA
          IF(SIG(I).GT.0.)
     &    SUM = SUM + VECT(I,LISTA(J))*(Y(I)-W(I))/SIG(I)**2
        END DO
        BETA(J) = SUM
      END DO
*
* Compute matrix
*
      DO J = 1, MFIT
        DO K = 1, J
          SUM = 0.D0
          DO I = 1, NDATA
            IF(SIG(I).GT.0.)
     &      SUM = SUM + VECT(I,LISTA(J))*VECT(I,LISTA(K))/SIG(I)**2
          END DO
          COVAR(J,K) = SUM
        END DO
      END DO
      DO J=2,MFIT
        DO K=1,J-1
          COVAR(K,J) = COVAR(J,K)
        END DO
      END DO
*
      CALL GAUSSJ(COVAR,MFIT,NCVM,BETA,1,1,IFAIL)
      IF(IFAIL.NE.0) RETURN
      DO J=1, MFIT
        A(LISTA(J)) = BETA(J)
      END DO
      DO I = 1, NDATA
        W(I) = 0.
      END DO
      DO J = 1, MA
        DO I = 1, NDATA
          W(I) = W(I) + A(J)*VECT(I,J)
        END DO
      END DO
      CHISQ = 0.
      NFIT  = 0
      DO I = 1, NDATA
        IF(SIG(I).GT.0.) THEN
          NFIT = NFIT + 1
          CHISQ = CHISQ + ((Y(I)-W(I))/SIG(I))**2
        END IF
      END DO
C     CALL COVSRT(COVAR,NCVM,MA,LISTA,MFIT)
      RETURN
      END

      SUBROUTINE SLINFIT(Y,SIG,NDATA,A,MA,LISTA,MFIT,COVAR,NCVM,
     &CHISQ,NFIT,VECT,MXDATA,W,IFAIL)
*
* Adapted from Numerical recipes linear fit routine
*
* R Y(NDATA)   -- Y values
* R SIG(NDATA) -- Uncertainties, -ve to ignore point.
* D A(MA)      -- coefficients
* I MA         -- Total number of coefficients
* I LISTA(MA)  -- Numbers of coefficients to be adjusted
* I MFIT       -- How many coefficients of MA are to be adjusted
* D COVAR(NCVM,NCVM) -- Normal matrix
* I NCVM       -- Dimension of matrix .GE. MFIT
* D CHISQ      -- Chi**2 of fit
* I NFIT       -- Actual number of points used in fit
* R VECT(MXDATA, MA) -- Fit functions evaluated at each point.
* I MXDATA     -- First dimension declared in calling routine
* R W(NDATA)   -- Work array
* I IFAIL      -- 0 if all ok
*

      IMPLICIT NONE
      INTEGER NDATA, MA, MFIT, NCVM, MXDATA
      REAL Y(NDATA), SIG(NDATA)
      INTEGER LISTA(MA)
      REAL VECT(MXDATA, MA), W(NDATA)
      DOUBLE PRECISION A(MA), COVAR(NCVM,NCVM)
      INTEGER MMAX
      PARAMETER (MMAX=100)
      DOUBLE PRECISION BETA(MMAX), SUM, CHISQ
      INTEGER I, J, K, KK, IHIT, IFAIL, NFIT
*
      IFAIL = 0
      KK = MFIT + 1
      DO J=1, MA
        IHIT = 0
        DO K = 1, MFIT
          IF(LISTA(K).EQ.J) IHIT = IHIT + 1
        END DO
        IF(IHIT.EQ.0) THEN
          LISTA(KK) = J
          KK = KK + 1
        ELSE IF(IHIT.GT.1) THEN
          IFAIL = 1
          WRITE(*,*) 'Improper set in LISTA; LFIT'
          RETURN
        END IF
      END DO
      IF(KK.NE.MA+1) THEN
        IFAIL = 1
        WRITE(*,*) 'Improper set in LISTA; LFIT'
        RETURN
      END IF
*
* Compute predicted values for coefficients that are set
*
      DO I = 1, NDATA
        W(I) = 0.
      END DO
      DO J = MFIT+1, MA
        DO I = 1, NDATA
          W(I) = W(I) + REAL(A(LISTA(J))*VECT(I,LISTA(J)))
        END DO
      END DO
*
      DO J=1, MFIT
        DO K=1, MFIT
          COVAR(J,K) = 0.
        END DO
      END DO
*
* Compute vector
*
      DO J = 1, MFIT
        SUM = 0.D0
        DO I = 1, NDATA
          IF(SIG(I).GT.0.)
     &    SUM = SUM + VECT(I,LISTA(J))*(Y(I)-W(I))/SIG(I)**2
        END DO
        BETA(J) = SUM
      END DO
*
* Compute matrix
*
      DO J = 1, MFIT
        DO K = 1, J
          SUM = 0.D0
          DO I = 1, NDATA
            IF(SIG(I).GT.0.)
     &      SUM = SUM + VECT(I,LISTA(J))*VECT(I,LISTA(K))/SIG(I)**2
          END DO
          COVAR(J,K) = SUM
        END DO
      END DO
      DO J=2,MFIT
        DO K=1,J-1
          COVAR(K,J) = COVAR(J,K)
        END DO
      END DO
*
      CALL GAUSSJ(COVAR,MFIT,NCVM,BETA,1,1,IFAIL)
      IF(IFAIL.NE.0) RETURN
      DO J=1, MFIT
        A(LISTA(J)) = BETA(J)
      END DO
      DO I = 1, NDATA
        W(I) = 0.
      END DO
      DO J = 1, MA
        DO I = 1, NDATA
          W(I) = W(I) + REAL(A(J)*VECT(I,J))
        END DO
      END DO
      CHISQ = 0.
      NFIT  = 0
      DO I = 1, NDATA
        IF(SIG(I).GT.0.) THEN
          NFIT = NFIT + 1
          CHISQ = CHISQ + ((Y(I)-W(I))/SIG(I))**2
        END IF
      END DO
C     CALL COVSRT(COVAR,NCVM,MA,LISTA,MFIT)
      RETURN
      END

      SUBROUTINE GAUSSJ(A,N,NP,B,M,MP,IFAIL)
*
* Numerical recipes Gauss-Jordan elimination
*
      IMPLICIT NONE
      INTEGER NMAX, N, NP, M, MP, I, J, K
      PARAMETER (NMAX=100)
      DOUBLE PRECISION A(NP, NP), B(NP, MP), BIG
      DOUBLE PRECISION PIVINV, DUM
      INTEGER IPIV(NMAX), INDXR(NMAX), INDXC(NMAX)
      INTEGER IFAIL, L, ICOL, IROW, LL
*
      IFAIL = 0
      DO J=1, N
        IPIV(J) = 0
      END DO
      DO I=1, N
        BIG = 0.D0
        DO J = 1, N
          IF(IPIV(J).NE.1) THEN
            DO K = 1, N
              IF(IPIV(K).EQ.0) THEN
                IF(ABS(A(J,K)).GE.BIG) THEN
                  BIG = ABS(A(J,K))
                  IROW = J
                  ICOL = K
                END IF
              ELSE IF(IPIV(K).GT.1) THEN
                WRITE(*,*) 'Matrix singular in GAUSSJ'
                IFAIL = 1
                RETURN
              END IF
            END DO
          END IF
        END DO
        IPIV(ICOL) = IPIV(ICOL) + 1
        IF(IROW.NE.ICOL) THEN
          DO L=1, N
            DUM = A(IROW,L)
            A(IROW,L) = A(ICOL,L)
            A(ICOL,L) = DUM
          END DO
          DO L = 1, M
            DUM = B(IROW, L)
            B(IROW,L) = B(ICOL,L)
            B(ICOL,L) = DUM
          END DO
        END IF
        INDXR(I) = IROW
        INDXC(I) = ICOL
        IF(A(ICOL,ICOL).EQ.0.D0) THEN
          WRITE(*,*) 'Matrix singular in GAUSSJ'
          IFAIL = 1
          RETURN
        END IF
        PIVINV = 1.D0/A(ICOL,ICOL)
        A(ICOL,ICOL) = 1.D0
        DO L=1, N
          A(ICOL,L) = A(ICOL,L)*PIVINV
        END DO
        DO L=1, M
          B(ICOL,L) = B(ICOL,L)*PIVINV
        END DO
        DO LL=1, N
          IF(LL.NE.ICOL) THEN
            DUM = A(LL,ICOL)
            A(LL,ICOL) = 0.D0
            DO L=1,N
              A(LL,L) = A(LL,L) - A(ICOL,L)*DUM
            END DO
            DO L=1, M
              B(LL,L) = B(LL,L) - B(ICOL,L)*DUM
            END DO
          END IF
        END DO
      END DO
      DO L=N,1,-1
        IF(INDXR(L).NE.INDXC(L)) THEN
          DO K=1,N
            DUM = A(K,INDXR(L))
            A(K,INDXR(L)) = A(K,INDXC(L))
            A(K,INDXC(L)) = DUM
          END DO
        END IF
      END DO
      RETURN
      END

        SUBROUTINE DLINFIT(YY,SD,X1,X2,N,NSTAR,A,VAR,IERR)
C
C SUBROUTINE TO PERFORM A WEIGHTED LEAST SQUARES FIT Y=A(1)+A(2)*X1+A(3)*X2
C TO A SET OF POINTS WITH Y,X1,X2 GIVEN AT EACH POINT
C OR TO A(1)+A(2)*X1 IF SPECIFIED (BY SETTING NSTAR=2,1 RESPECTIVELY)
C ARRAY A(3) CONTAINS THE RESULT, VAR(6) CONTAINS THE VARIANCES AS SPECIFIED
C BELOW. THESE WILL ONLY BE CORRECT IF THE ARRAY SD(N) IS CORRECTLY
C INPUT AND IF THE RESIDUALS CAN BE ASSUMED TO BE NORMALLY DISTRIBUTED.
C
C PASSED:
C         REAL*8 YY(N) ARRAY OF Y VALUES
C         REAL*8 SD(N) ARRAY OF ERROR ESTIMATES ON THE YY VALUES
C       YY VALUES ARE ASSUMED TO BE UNCORELATED (STANDARD DEVIATIONS)
C       ALL THESE VALUES MUST BE > 0. ALL 1. IF EQUAL WEIGHTS REQUIRED
C         REAL*8 X1(N),X2(N) TWO ARRAYS OF X VALUES
C         INTEGER*4 N  NUMBER OF POINTS TO FIT
C         INTEGER*4 NSTAR WHETHER A(3)*X2 IS TO BE INCLUDED
C                   NSTAR=1 THEN IT WILL BE IGNORED
C                   NSTAR=2 THEN IT WILL BE INCLUDED
C
C RETURNED:
C         REAL*8 A(3)  FITTED PARAMETERS AS ABOVE
C                NSTAR=1 THEN A(3)=0. ON EXIT
C
C         REAL*8 VAR(3) VARIANCE MATRIX. SHOULD BE 9 ELEMENTS, BUT
C                       IT IS SYMMETRIC. FOR NSTAR = 2 THEN THE
C         ELEMENTS ARE AS FOLLOWS VAR(1)=VAR( A(1) )
C                                 VAR(2)=VAR( A(2) )
C                                 VAR(3)=VAR( A(3) )
C               NSTAR=1 ELEMEMTS ARE
C                                 VAR(1)=VAR( A(1) )
C                                 VAR(2)=VAR( A(2) )
C                                 VAR(3)=0.
C WHERE ON THE RIGHT-HAND SIDE, VAR STANDS FOR VARIANCE.
C         INTEGER*4 IERR 0 IF NO ERROR, 1 OTHERWISE
C
        IMPLICIT NONE 
        INTEGER N, IERR, I, NSTAR
        DOUBLE PRECISION YY(N),SD(N),X1(N),X2(N),A(3),VAR(3)
        DOUBLE PRECISION S1, S2, S3, S4, S5, S6, V1, V2, V3
        DOUBLE PRECISION Y1, C1, C2, C3, R1, R2, R3, R4, R5
        DOUBLE PRECISION R6, ERR, DET
C
        IERR=0
C
C SUMS WHICH WILL BE FORMED SET TO ZERO INITIALLY
C
        S1=0.D0
        S2=0.D0
        S3=0.D0
        S4=0.D0
        S5=0.D0
        S6=0.D0
        V1=0.D0
        V2=0.D0
        V3=0.D0
C
C IN THE CASE WHERE ALL DATA IS ZERO,
C ERROR ESTIMATES MUST BE MADE BY CALLING ROUTINE
C ERROR RETURN
C
        DO I=1,N
          IF(YY(I).NE.0.) GOTO 5
        END DO
        IERR=1
        RETURN
5       IF(NSTAR.EQ.1) GOTO 10
C
C THE S'S ARE ELEMENTS OF A MATRIX MULTIPLYING THE A PARAMETERS GIVING
C THE VECTOR FORMED BY THE V'S
C
        DO I=1,N
          ERR=SD(I)
          IF(ERR.EQ.0.) THEN
            WRITE(*,*) CHAR(7),'** ERROR ESTIMATE EQUALS ZERO **'
            IERR=1
            RETURN
          END IF
          Y1=YY(I)/ERR
          C1=X1(I)/ERR
          C2=X2(I)/ERR
          C3=1.D0/ERR
          S1=S1+C3*C3
          S2=S2+C1*C3
          S3=S3+C2*C3
          S4=S4+C1*C1
          S5=S5+C1*C2
          S6=S6+C2*C2
          V1=V1+Y1*C3
          V2=V2+Y1*C1
          V3=V3+Y1*C2
        END DO
C
C NOW THE S MATRIX MUST BE INVERTED
C ACHIEVED HERE BY FINDING COFACTORS ETC.
C (IT'S ONLY A THREE BY THREE MATRIX)
C R1,R2 ... INVERSE ELEMENTS
C
        R1=S4*S6-S5*S5
        R2=S3*S5-S2*S6
        R3=S2*S5-S3*S4
        R4=S1*S6-S3*S3
        R5=S2*S3-S1*S5
        R6=S1*S4-S2*S2
C
C FORM DETERMINANT FROM THESE
C
        DET=S1*R1+S2*R2+S3*R3
        IF(ABS(DET).LT.1.D-10) THEN
          PRINT *,CHAR(7),
     1           '** MODULUS OF DETERMINANT LESS THAN 1.E-10 **'
          PRINT *,'NO MATRIX INVERSION DONE'
          IERR=1
          RETURN
        END IF
        A(1)=(R1*V1+R2*V2+R3*V3)/DET
        A(2)=(R2*V1+R4*V2+R5*V3)/DET
        A(3)=(R3*V1+R5*V2+R6*V3)/DET
C
C VARIANCES
        VAR(1)=R1/DET
        VAR(2)=R4/DET
        VAR(3)=R6/DET
        RETURN
C
C SCALE ACCORDING TO ERROR ESTIMATES
C AND THEN FORM SUMS
C
10      DO I=1,N
          ERR=SD(I)
          IF(ERR.EQ.0.) THEN
            PRINT *,CHAR(7),'** ERROR ESTIMATE EQUALS ZERO **'
            IERR=1
            RETURN
          END IF
          Y1=YY(I)/ERR
          C1=X1(I)/ERR
          C3=1.D0/ERR
          S1=S1+C3*C3
          S2=S2+C1*C3
          S4=S4+C1*C1
          V1=V1+Y1*C3
          V2=V2+Y1*C1
        END DO
C
C NOW THE S MATRIX MUST BE INVERTED
C FORM DETERMINANT FIRST
C
        DET=S1*S4-S2*S2
        IF(ABS(DET).LT.1.D-10) THEN
          PRINT *,CHAR(7),
     1            '** MODULUS OF DETERMINANT LESS THAN 1.E-10 **'
          PRINT *,'NO MATRIX INVERSION DONE'
          IERR=1
          RETURN
        END IF
        A(1)=(S4*V1-S2*V2)/DET
        A(2)=(-S2*V1+S1*V2)/DET
        A(3)=0.D0
C
C VARIANCES
        VAR(1)=S4/DET
        VAR(2)=S1/DET
        VAR(3)=0.D0
        RETURN
        END

        SUBROUTINE FITLINE( NDATA, XDATA, YDATA, X0, A, B )
*
*  least squares fit of a line to X-Y data pairs
*
*  Inputs:
*       NDATA   = NUMBER OF DATA PAIRS
*       XDATA   = VALUE OF INDEPENDENT VARIABLE
*       YDATA   = VALUE OF DEPENDENT VARIABLE
*  Output:
*       X0,A,B  = FUNCTION FITTED IS Y(X) = A + B*(X-X0)
*
        REAL*4 XDATA(1), YDATA(1)
        REAL*8 W, WX, WXX, WXY
*
      IF( NDATA.LE.0 ) THEN
  999   PRINT *,'** NO DATA IN ''FITLINE''.'
        RETURN
      END IF
*
*  Compute mean X-value
*
        W = 0.D0
        WX = 0.D0
      DO I=1,NDATA
        W  = W  + 1.
        WX = WX + XDATA(I)
      END DO
        IF( W.LE.0.D0 ) GOTO 999
        X0 = REAL(WX / W)
*
*  Compute sums for slope and intercept
*
        WX = 0.D0
        WY = 0.D0
        WXX = 0.D0
        WXY = 0.D0
      DO I=1,NDATA
        X = XDATA(I) - X0
        WX  = WX  + X
        WXX = WXX + X * X
        WY  = WY  + YDATA(I)
        WXY = WXY + YDATA(I) * X
      END DO
*
*  Evaluate determinant
*
        DET = REAL(W * WXX - WX * WX)
*
      IF( DET.NE.0. ) THEN
        A = REAL(( WY * WXX - WX * WXY ) / DET)
        B = REAL((  W * WXY - WX * WY ) / DET)
      ELSE
        A = REAL(WX / W)
        B = 0.
      END IF
*
        RETURN
        END


        SUBROUTINE WFITLINE( NDATA, XDATA, YDATA, YSIGMA, X0, A, B )
*
*  Weighted least squares fit of a line to X-Y data pairs
*
*  Inputs:
*       NDATA   = NUMBER OF DATA PAIRS
*       XDATA   = VALUE OF INDEPENDENT VARIABLE
*       YDATA   = VALUE OF DEPENDENT VARIABLE
*       YSIGMA  = UNCERTAINTY (1-SIGMA) IN YDATA (NEGATIVE TO IGNORE)
*  Output:
*       X0,A,B  = FUNCTION FITTED IS Y(X) = A + B*(X-X0)
*
        REAL*4 XDATA(1), YDATA(1), YSIGMA(1)
        REAL*8 W, WX, WXX, WY, WXY
*
      IF( NDATA.LE.0 ) THEN
  999   PRINT *,'** NO DATA IN ''WFITLINE''.'
        RETURN
      END IF
*
*  Compute mean X-value
*
        W = 0.D0
        WX = 0.D0
      DO I=1,NDATA
      IF( YSIGMA(I).GT.0. ) THEN
        WEIGHT = 1./(YSIGMA(I)*YSIGMA(I))
        W  = W  + WEIGHT
        WX = WX + WEIGHT * XDATA(I)
      END IF
      END DO
        IF( W.LE.0.D0 ) GOTO 999
        X0 = REAL(WX / W)
*
*  Compute sums for slope and intercept
*
        WX = 0.D0
        WXX = 0.D0
        WY = 0.D0
        WXY = 0.D0
      DO I=1,NDATA
      IF( YSIGMA(I).GT.0. ) THEN
        WEIGHT = 1. / (YSIGMA(I)*YSIGMA(I))
        X = XDATA(I) - X0
        WX  = WX  + WEIGHT * X
        WXX = WXX + WEIGHT * X * X
        WY  = WY  + WEIGHT * YDATA(I)
        WXY = WXY + WEIGHT * YDATA(I) * X
      END IF
      END DO
*
*  Evaluate determinant
*
        DET = REAL(W * WXX - WX * WX)
*
      IF( DET.NE.0. ) THEN
        A = REAL(( WY * WXX - WX * WXY ) / DET)
        B = REAL((  W * WXY - WX * WY ) / DET)
      ELSE
        A = REAL(WX / W)
        B = 0.
      END IF
*
        RETURN
        END
