*WCOMB
* WCOMB N1 N2 N3 WEXP Method -- Combines different telluric standards
*
* Parameters:
*
*     N1, N2  -- Range of spectra to combine
*     N3      -- Output spectrum
*     WEXP    -- Water absorption exponent for scaling of spectra relative
*                to one another
*     METHOD  -- Rebinning method Linear, Quadratic, Sinc
*
* In some cases you may be able to get good measurements of different regions
* of telluric absorption with different stars. You may then want to combine
* these into one telluric standard for correcting your objects. WCOMB does
* this. Since the different spectra may have been taken at different airmasses
* and have different wavelength scales, it has to know the exponent for scaling
* the spectra on the basis that the absorption in mags is proportional to the
* airmass to the power of WEXP. 
*WCOMB
      SUBROUTINE WCOMB(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, FLUX, 
     &MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, 
     &NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS, MXSLOTS, 
     &WORK1, WORK2, WORK3, WORK4, WORK5, MXWORK, IFAIL)
*
* Combines telluric line standards together
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)   -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)   -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)     -- Equivalent fluxes to COUNTS
* >  I     MXBUFF           -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
*    I     SLOTS(MXSLOTS) -- List buffer.
*    I     MXSLOTS
*    R     WORK1(MXWORK) - Work arrays
*    R     WORK2(MXWORK)
*    R     WORK3(MXWORK)
*    R     WORK4(MXWORK)
*    R     WORK5(MXWORK)
*    I     MXWORK
* <  I     IFAIL              -- Error return.
*
      IMPLICIT NONE

*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Work arrays
*
      INTEGER MXWORK
      REAL WORK1(MXWORK), WORK2(MXWORK)
      REAL WORK3(MXWORK), WORK4(MXWORK)
      REAL WORK5(MXWORK)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Slot lists
*
      INTEGER MXSLOTS, NSLOTS, SLOT
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
      DOUBLE PRECISION  WAIR, AIR
      CHARACTER*32  METHOD*3
*
* Local variables
*
      INTEGER I, J,  N
      REAL WATER, RATIO, WAT
      INTEGER SLOT1, SLOT2, SLOT3, SLT
      INTEGER MAXSPEC,  NCOM, IFAIL
      LOGICAL    DEFAULT
*
* Functions
*
      REAL GET_FLX,  CFRAT
*
      DATA SLOT1, SLOT2, SLOT3/1, 1, 2/
      DATA WATER/0.5/
      DATA METHOD/'Q'/
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL INTR_IN('First slot to combine', SPLIT, NSPLIT, 
     &MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to combine', SPLIT, NSPLIT, 
     &MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
        WRITE(*,*) 'Only 0,0 to get list option'
        GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
        IF(DEFAULT .AND. NLIST.GT.0) THEN
          CALL SETSLOT(LIST, NLIST, MXLIST, SLOTS, NSLOTS, MXSLOTS)
        ELSE
          WRITE(*,*) 'Enter list of spectra to combine'
          CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &    NSLOTS, MXSLOTS)
        END IF
        IF(NSLOTS.EQ.0) GOTO 999
      ELSE
        CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &  NSLOTS, MXSLOTS)
      END IF
*
      CALL INTR_IN('Output slot', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, SLOT3, 1, MAXSPEC, IFAIL)
      CALL REAL_IN('Water exponent', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, WATER, -1000., 1000., IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      CALL CHAR_IN('Rebinning method (L,Q,S)', SPLIT, NSPLIT, 
     &MXSPLIT, NCOM, DEFAULT, METHOD, IFAIL)
      CALL UPPER_CASE(METHOD)
      IF(METHOD.NE.'L' .AND. METHOD.NE.'Q' .AND. METHOD.NE.'S') THEN
        WRITE(*,*) 'Possible methods are:'
        WRITE(*,*) ' '
        WRITE(*,*) ' L -- Linear'
        WRITE(*,*) ' Q -- Quadratic'
        WRITE(*,*) ' S -- Sinc'
        METHOD = 'Q'
        GOTO 999
      END IF
*
* Now combine spectra. First spectrum is used to define parameters
* for output spectrum. All other spectra are rebinned onto the same
* scale
*
      DO N = 1, NSLOTS
        SLOT = SLOTS(N)
        IF(NPIX(SLOT).LE.0) THEN
          WRITE(*,*) 'Slot ',SLOT,' empty.'
          GOTO 999
        ELSE IF(NARC(SLOT).EQ.0) THEN
          WRITE(*,*) 'Slot ',SLOT,' has no wavelength scale'
          GOTO 999
        END IF
        CALL HGETD('Airmass', WAIR, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &  NDOUB, MXDOUB, IFAIL)
        IF(IFAIL.NE.0) THEN
          WRITE(*,*) 'No airmass in slot ',SLOT
          GOTO 999
        END IF                                                     
        IF(N.EQ.1) THEN
          DO I = 1, NPIX(SLOT)
            WORK1(I) = GET_FLX(COUNTS(I,SLOT), FLUX(I,SLOT))
          END DO
          AIR = WAIR
          SLT = SLOT
        ELSE
          DO I = 1, NPIX(SLOT)
            WORK2(I) = GET_FLX(COUNTS(I,SLOT), FLUX(I,SLOT))
          END DO
*
* Rebin onto same scale as first slot
*
          IF(METHOD.EQ.'L') THEN
            CALL REBIN(1, 0, WORK2, NPIX(SLOT), WORK3, 
     &      NPIX(SLT), 0., NARC(SLOT), ARC(1,SLOT), 
     &      NARC(SLT), ARC(1,SLT))
          ELSE IF(METHOD.EQ.'Q') THEN
            CALL REBIN(1, 1, WORK2, NPIX(SLOT), WORK3, 
     &      NPIX(SLT), 0., NARC(SLOT), ARC(1,SLOT), 
     &      NARC(SLT), ARC(1,SLT))
          ELSE IF(METHOD.EQ.'S') THEN
            CALL SINCBIN(1, WORK2, NPIX(SLOT), WORK3, 
     &      NPIX(SLT), 0., NARC(SLOT), ARC(1,SLOT), 
     &      NARC(SLT), ARC(1,SLT), WORK4, WORK5)
          END IF
*
* Multiply in, converting to same airmass as first slot
*
          DO J = 1, NPIX(SLT)
             WAT = REAL(EXP(LOG(ABS(WORK3(J)))*(AIR/WAIR)**WATER))
            WORK1(J) = WAT*WORK1(J)
          END DO
        END IF
      END DO
*
* Set up output
*
      DO I = 1, NPIX(SLT)
        RATIO = CFRAT(COUNTS(I,SLT),FLUX(I,SLT))
        IF(WORK1(I).EQ.0.) THEN
          FLUX(I,SLOT3)   = RATIO
          COUNTS(I,SLOT3) = 0.
        ELSE
          FLUX(I,SLOT3) = WORK1(I)
          COUNTS(I,SLOT3) = RATIO*WORK1(I)
        END IF
        ERRORS(I,SLOT3) = ERRORS(I,SLT)
      END DO
*
      CALL SET_HEAD(SLOT3, SLT, MXSPEC, NPIX, ARC, NARC, MXARC, 
     &NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &MXDOUB, MXINTR, MXREAL, IFAIL)
*
      WRITE(*,*) 'Spectra combined and stored in slot ',SLOT3
      IFAIL = 0
      RETURN
999   IFAIL = 1
      RETURN
      END
