*TABRED
* TABRED N TABLE -- reads and interpolates a table onto the wavelength scale
*                   of a slot.
*
* Parameters:
*
*    N  -- Slot to load with table
*
*    TABLE -- name of ascii tabular data file. This should be a list of
*             angstrom, mJy pairs. Linear interpolation is used with the
*             values kept constant beyond the edges of the table. The
*             angstrom values should ascend down the table.
*
* NB this is mainly for reading in filter or response type data and the earth
*    velocity will not be corrected for.
*
*TABRED
      SUBROUTINE TABRED(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, 
     &NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, 
     &NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, 
     &WTAB, FLTAB, MXTAB, IFAIL)
*
* FLX_FIT for fitting flux stars.
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
*    R     WTAB(MXTAB)  -- For tabular data
*    R     FLTAB(MXTAB)
*    I     MXTAB
* <  I     IFAIL              -- Error return.
*
      IMPLICIT NONE

*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL   FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)         
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Arrays for tabular data
*
      INTEGER MXTAB, NTAB
      REAL WTAB(MXTAB), FLTAB(MXTAB)
*
* Local variables
*
      INTEGER I, J, NCOM
      DOUBLE PRECISION  ANGST, W, W1, W2
      CHARACTER*100 STRING
      CHARACTER*64 TABLE
      INTEGER IFAIL, MAXSPEC,  SLOT, NGREAT
      LOGICAL  DEFAULT
      REAL Z, RATIO, CFRAT, FLX
*
* Functions
*
      INTEGER LENSTR
*
      DATA SLOT/1/
      DATA TABLE/'response.dat'/
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
      CALL INTR_IN('Slot to interpolate table onto', SPLIT, 
     &NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT, 1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(NPIX(SLOT).LE.0) THEN
        WRITE(*,*) 'Slot ',SLOT,' empty'
        IFAIL = 1
        GOTO 999
      END IF
      IF(NARC(SLOT).EQ.0) THEN 
        WRITE(*,*) 'Slot ',SLOT,' has no wavelength calibration'
        IFAIL = 1
        GOTO 999
      END IF
      W1 = ANGST(1., NPIX(SLOT), NARC(SLOT), 
     &         ARC(1,SLOT), MXARC)
      W2 = ANGST(REAL(NPIX(SLOT)), NPIX(SLOT), NARC(SLOT), 
     &         ARC(1,SLOT), MXARC)
      IF(W1.GT.W2) THEN
        W  = W1
        W1 = W2
        W2 = W
      END IF
*
      CALL CHAR_IN('File of angstrom/mJy pairs', SPLIT, 
     &NSPLIT, MXSPLIT, NCOM, DEFAULT, TABLE, IFAIL)
      OPEN(UNIT=47,FILE=TABLE,STATUS='OLD',IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(*,*) TABLE(:LENSTR(TABLE)),' not found.'
        GOTO 999
      END IF
*
* Read tabular data
*
      NTAB = 0
      NGREAT = 0
 100  STRING = '#'
      IFAIL = 0
      DO WHILE(IFAIL.EQ.0 .AND. (STRING(1:1).EQ.'#' 
     &.OR. STRING(1:1).EQ.'*'))
        READ(47,'(A)',IOSTAT=IFAIL) STRING
      END DO
      IF(IFAIL.EQ.0) THEN
        NTAB = NTAB + 1
        READ(STRING,*,IOSTAT=IFAIL) WTAB(NTAB), FLTAB(NTAB)
*
* Do not store more data than necessary
*
        IF(IFAIL.EQ.0) THEN
          IF(WTAB(NTAB).LT.W1) THEN
            WTAB(1)  = WTAB(NTAB)
            FLTAB(1) = FLTAB(NTAB)
            NTAB = 1
          ELSE IF(WTAB(NTAB).GT.W2) THEN
            NGREAT = NGREAT + 1
          END IF
          IF(NGREAT.LT.1) THEN
            GOTO 100
          ELSE
            WRITE(*,*) NTAB,' points read'
            CLOSE(UNIT=47)
            IFAIL = 0
          END IF
        ELSE
          WRITE(*,*) 'Failed to translate ',STRING
          GOTO 999
        END IF
      ELSE IF(IFAIL.LT.0) THEN
        WRITE(*,*) NTAB,' points read'
        CLOSE(UNIT=47)
        IF(NTAB.LE.0) THEN
          WRITE(*,*) 'Too few points'
          GOTO 999
        END IF
        IFAIL = 0
      ELSE
        WRITE(*,*) 'Error reading file'
        GOTO 999
      END IF
*
* Now set spectrum
*
      DO I = 1, NPIX(SLOT)
        W = ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &         ARC(1,SLOT), MXARC)
        Z = REAL(W)
        CALL HUNT(WTAB,NTAB,Z,J)
        IF(J.EQ.0) THEN
          FLX = FLTAB(1)
        ELSE IF(J.EQ.NTAB) THEN
          FLX = FLTAB(NTAB)
        ELSE
          FLX = ((WTAB(J+1)-Z)*FLTAB(J)+(Z-WTAB(J))*FLTAB(J+1))/
     &          (WTAB(J+1)-WTAB(J))
        END IF
        RATIO = CFRAT(COUNTS(I,SLOT), FLUX(I,SLOT))
        COUNTS(I,SLOT) = RATIO*FLX
        ERRORS(I,SLOT) = MAX(ABS(COUNTS(I,SLOT))/1000., 1.E-10)
        IF(COUNTS(I,SLOT).EQ.0.) THEN
          FLUX(I,SLOT) = RATIO
        ELSE
          FLUX(I,SLOT) = FLX
        END IF
      END DO
      WRITE(*,*) 'Table interpolated onto slot ',SLOT
      IFAIL = 0
999   CLOSE(UNIT=47)
      RETURN
      END
