CLOAD
C LOAD FILE N1 N2 M1  -- Reads MOLLY format spectra.
C
C Parameters: 
C     
C   FILE   -- Name of file to read from.
C   N1     -- Slot number to read first spectrum into. 'A' to indicate that
C             the first spectrum is loaded into the first empty slot (but be
C             careful!)
C   N2     -- Slot number to read last spectrum into.  If you have used option
C             'A' then N2 = number of spectra to load
C   M1     -- First spectrum of file to start at.      
C
C Selection criteria can be applied with SETSER. Then spectra must
C satisfy user defined requirements on header parameters. Use 'molex'
C to set/unset default file extension of .mol.
C
C Related commands: write, molex
CLOAD
      SUBROUTINE SPCRED(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &     FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC,
     &     NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &     HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &     MXREAL, NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, 
     &     NMSREAL, VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, 
     &     MXSDOUB, MXSINTR, MXSREAL, MUTE, MOLEX, NREAD, IFAIL)
C     
C     Master routine for reading spectra into MOLLY buffers
C     allowing selection criteria to be applied.
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     >  C*(*) NMSCHAR(MXSCHAR)   -- Names of character selection items.
C     >  C*(*) VSCHAR(MXSCHAR)    -- Values of character selection items.
C     >  C*(*) NMSDOUB(MXSDOUB)   -- Names of double precision selection items.
C     >  D     VSDOUB(2,MXSDOUB)  -- Values of double precision selection items.
C     >  C*(*) NMSINTR(MXSINTR)   -- Names of integer selection items.
C     >  I     VSINTR(2,MXSINTR)  -- Values of integer selection items.
C     >  C*(*) NMSREAL(MXSREAL)   -- Names of real selection items.
C     >  R     VSREAL(2,MXSREAL)  -- Values of real selection items.
C     >  I     NSCHAR             -- Number of character selection items.
C     >  I     NSDOUB             -- Number of double precsion selection items.
C     >  I     NSINTR             -- Number of integer selection items.
C     >  I     NSREAL             -- Number of real selection items.
C     >  I     MXSCHAR            -- Dimension of NMSCHAR in calling routine.
C     >  I     MXSDOUB            -- Dimension of NMSDOUB in calling routine.
C     >  I     MXSINTR            -- Dimension of NMSINTR in calling routine.
C     >  I     MXSREAL            -- Dimension of NMSREAL in calling routine.   
C     >  L     MUTE               -- Cuts out some terminal I/O
C     >  L     MOLEX              -- Default extension .mol
C     <  I     NREAD              -- Number of spectra read (for use in SPCDMP)
C     >  I     IFAIL
C     
      IMPLICIT NONE
C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL, MXSCHAR, MXSDOUB, MXSINTR
      INTEGER MXSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
      INTEGER FCODE
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
C     Search parameters, names then values.
C     
      CHARACTER*(*) NMSCHAR(MXSCHAR)
      CHARACTER*(*) NMSDOUB(MXSDOUB)
      CHARACTER*(*) NMSINTR(MXSINTR)
      CHARACTER*(*) NMSREAL(MXSREAL)
C     
      CHARACTER*(*) VSCHAR(MXSCHAR)
      DOUBLE PRECISION VSDOUB(2,MXSDOUB)
      INTEGER VSINTR(2,MXSINTR)
      REAL VSREAL(2,MXSREAL)
C     
C     Local variables
C     
      CHARACTER*210 FILE
      CHARACTER*200 FILENAME
      CHARACTER*100 STRING
      CHARACTER*16 UNITS
      CHARACTER*20 CSLOT
      INTEGER SLOT1, SLOT2, NLOAD, LSLOT, FSPEC, IFAIL, LUNIT
      INTEGER I, J,  I1, I2, NSPEC, MAXSPEC, NREAD
      LOGICAL SELECT, SAME,  WARN, DEFAULT, MUTE, OLD, MOLEX
      LOGICAL SET
      DOUBLE PRECISION ANGST, W
      REAL  Z, VLIGHT, CGS, FAC, CFAC
      INTEGER NCOM, L, LENSTR
C     
      DATA FILENAME/'spectrum'/
      DATA CSLOT/'1'/
      DATA SLOT2, NLOAD, FSPEC/1,1000, 1/
      DATA LUNIT/23/
C***********************************************************************
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
C     
      CALL CHAR_IN('File name', SPLIT, NSPLIT, MXSPLIT, NCOM,
     &     DEFAULT, FILENAME, IFAIL)
      CALL CHAR_IN('First slot to read into', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, CSLOT, IFAIL)
      IF(CSLOT.NE.'A' .AND. CSLOT.NE.'a') THEN
         READ(CSLOT,*,IOSTAT=IFAIL) SLOT1
         IF(IFAIL.NE.0) THEN
            IF(INDEX(CSLOT,'+').GT.1) THEN
               I = INDEX(CSLOT,'+')
               READ(CSLOT(:I-1),*,ERR=999) I1
               READ(CSLOT(I+1:),*,ERR=999) I2
               SLOT1 = I1 + I2
            ELSE IF(INDEX(CSLOT,'-').GT.1) THEN
               I = INDEX(CSLOT,'-')
               READ(CSLOT(:I-1),*,ERR=999) I1
               READ(CSLOT(I+1:),*,ERR=999) I2
               SLOT1 = I1 - I2
            ELSE IF(INDEX(CSLOT,'*').GT.1) THEN
               I = INDEX(CSLOT,'*')
               READ(CSLOT(:I-1),*,ERR=998) I1
               READ(CSLOT(I+1:),*,ERR=998) I2
               SLOT1 = I1 * I2
            ELSE IF(INDEX(CSLOT,'/').GT.1) THEN
               I = INDEX(CSLOT,'/')
               READ(CSLOT(:I-1),*,ERR=998) I1
               READ(CSLOT(I+1:),*,ERR=998) I2
               SLOT1 = I1 / I2
            ELSE
               GOTO 999
            END IF
            IFAIL = 0
         END IF
         IF(SLOT1.LT.1 .OR. SLOT1.GT.MAXSPEC) THEN
            WRITE(*,*) 'Slot out of range 1 to ',MAXSPEC
            IFAIL = 1
         END IF
      END IF
      IF(IFAIL.NE.0) GOTO 999
C     
      IF(CSLOT.NE.'A' .AND. CSLOT.NE.'a') THEN
         SLOT2 = MAX(SLOT1, SLOT2)
         CALL INTR_IN('Last slot to read into', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
         LSLOT = SLOT2
      ELSE
         DO I = 1, MAXSPEC
            IF(NPIX(I).LE.0) THEN
               SLOT1 = I
               GOTO 101
            END IF
         END DO
 101     CONTINUE
         NLOAD = MIN(MAXSPEC-SLOT1+1, NLOAD)
         CALL INTR_IN('Number of spectra to load', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, NLOAD, 1, MAXSPEC-SLOT1+1, IFAIL)
         LSLOT = SLOT1 + NLOAD - 1
      END IF
C     
      CALL INTR_IN('Start spectrum in file', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, FSPEC, 1, 10000000, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
      L = LENSTR(FILENAME)
      IF(MOLEX .AND. (INDEX(FILENAME,'.mol').EQ.0 .OR.
     &     INDEX(FILENAME,'.mol').NE.L-3)) THEN
         FILE = FILENAME(:L)//'.mol'
      ELSE
         FILE = FILENAME
      END IF
      L = LENSTR(FILE)
      OPEN(UNIT=LUNIT,FILE=FILE,FORM='UNFORMATTED',
     &     STATUS='OLD',IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN
         WRITE(*,*) 'Could not open ',FILE(:L)
         RETURN
      END IF
C     
C     Skip spectra until FSPEC. 5 records per spectrum
C     see WTHEAD
C     
      IF(FSPEC.GT.1) THEN
         DO I = 1, FSPEC-1
            CALL SKIP_SPEC(LUNIT, IFAIL)
            IF(IFAIL.NE.0) GOTO 998
         END DO
      END IF
C     
C     Start to read files, checking headers as we go.
C     
      NSPEC = FSPEC - 1
      WARN = .FALSE.
      NREAD  = 0
      OLD = .FALSE.
      DO I = SLOT1, LSLOT
 100     CONTINUE
         NSPEC = NSPEC + 1
         CALL RTHEAD(LUNIT, FCODE, UNITS, NPIX(I), 
     &        NARC(I), NCHAR(I), NDOUB(I), NINTR(I), NREAL(I), 
     &        MXCHAR, MXDOUB, MXINTR, MXREAL, NMCHAR(1,I),
     &        NMDOUB(1,I), NMINTR(1,I), NMREAL(1,I), HDCHAR(1,I), 
     &        HDDOUB(1,I), HDINTR(1,I), HDREAL(1,I), IFAIL)
         IF(IFAIL.NE.0) THEN
            NPIX(I) = 0
            IFAIL = MAX(0, IFAIL)
            GOTO 998
         END IF
C     
C     Check that numbers of pixels, arc coefficients etc are ok
C     
         IF(NPIX(I).GT.MAXPX) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NPIX(I),' pixels.'
            WRITE(*,*) 'compared with maximum value ',MAXPX
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NARC(I).GT.MXARC) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NARC(I),
     &           ' arc coefficients'
            WRITE(*,*) 'compared with maximum value ',MXARC
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NCHAR(I).GT.MXCHAR) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NCHAR(I),
     &           ' character items'
            WRITE(*,*) 'compared with maximum value ',MXCHAR
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NDOUB(I).GT.MXDOUB) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NDOUB(I),
     &           ' double precision items'
            WRITE(*,*) 'compared with maximum value ',MXDOUB
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NINTR(I).GT.MXINTR) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NINTR(I),
     &           ' integer items'
            WRITE(*,*) 'compared with maximum value ',MXINTR
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NREAL(I).GT.MXREAL) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NREAL(I),
     &           ' real items'
            WRITE(*,*) 'compared with maximum value ',MXREAL
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(.NOT.SAME(UNITS,'COUNTS') .AND. 
     &        .NOT.SAME(UNITS,'MILLIJANSKYS') .AND.
     &        .NOT.SAME(UNITS,'FLAMBDA')) THEN
            WRITE(*,*) 'Units = [',UNITS,']'
            WRITE(*,*) 'Not recognised'
            NPIX(I) = 0
            GOTO 998
         END IF
C     
C     Finished checks, now apply selection criteria.
C     
         IF(.NOT.SELECT(I, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &        HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &        MXCHAR, MXDOUB, MXINTR, MXREAL, MXSPEC, NMSCHAR, VSCHAR, 
     &        NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, VSREAL, NSCHAR,
     &        NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, MXSINTR, 
     &        MXSREAL)) THEN
            IF(.NOT.MUTE) WRITE(*,*) 'Spectrum ',NSPEC,' skipped.'
            NPIX(I) = 0
            CALL SKIP_DATA(LUNIT, IFAIL)
            IF(IFAIL.NE.0) GOTO 998
            GOTO 100
         END IF
C     
C     Read data
C     
         CALL RTDATA(LUNIT, COUNTS(1,I), ERRORS(1,I), FLUX(1,I), 
     &        ARC(1,I), FCODE, NPIX(I), NARC(I), IFAIL)
         IF(IFAIL.NE.0) THEN
            NPIX(I) = 0
            GOTO 998       
         END IF
         IF(UNITS.EQ.'FLAMBDA') THEN
C     
C     Convert data to milli-Janskys
C     
            VLIGHT = CGS('C')
            CFAC = VLIGHT*1.E-16
            DO J = 1, NPIX(I)
               IF(ERRORS(J,I).GT.0. .OR. FCODE.EQ.4) THEN
                  Z = REAL(J)
                  W = ANGST(Z, NPIX(I), NARC(I), ARC(1,I), MXARC)
                  FAC = REAL(W**2/CFAC)
                  FLUX(J,I)   = FAC*FLUX(J,I)
                  IF(FCODE.EQ.5) ERRORS(J,I) = FAC*ERRORS(J,I)
               END IF
            END DO
         END IF
C     
C     Put into standard counts/errors/fluxes internal
C     MOLLY format
C     
         IF(FCODE.EQ.1 .OR. FCODE.EQ.2) THEN
C     
C     Counts or counts and errors only
C     
            IF(FCODE.EQ.1) WARN = .TRUE.
            DO J = 1, NPIX(I)
               IF(FCODE.EQ.1) ERRORS(J,I) = 1.
               IF(COUNTS(J,I).NE.0) THEN
                  FLUX(J,I) = COUNTS(J,I)
               ELSE
                  FLUX(J,I) = 1.
               END IF
            END DO
         ELSE IF(FCODE.EQ.4 .OR. FCODE.EQ.5) THEN
C     
C     Fluxes or fluxes and errors on fluxes only
C     
            DO J = 1, NPIX(I)
               COUNTS(J,I) = FLUX(J,I)
               IF(COUNTS(J,I).EQ.0) FLUX(J,I) = 1.
               IF(FCODE.EQ.4) THEN
                  ERRORS(J,I) = 1.e-5
               END IF
            END DO
         END IF
C     
C     Type out 1-line on slot
C     
         IF(.NOT.MUTE) THEN
            CALL SL_INF(I, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) STRING(:LENSTR(STRING))
         END IF
C     
C     convert old style ephemeris
C     
         CALL HCONV(I, MXSPEC, NMCHAR, HDCHAR, NCHAR, MXCHAR, 
     &        NMDOUB, HDDOUB, NDOUB, MXDOUB, NMINTR, HDINTR, NINTR, 
     &        MXINTR, NMREAL, HDREAL, NREAL, MXREAL, SET, IFAIL)
         IF(SET) OLD = .TRUE.
         NREAD = NREAD + 1
      END DO
      IFAIL = 0
 998  CLOSE(UNIT=LUNIT)
      WRITE(*,*) 'Read ',NREAD,' spectra from ',FILE(:L)
      IF(WARN) THEN
         WRITE(*,*) 'At least one of the spectra read in is not'
         WRITE(*,*) 'on standard counts/errors/fluxes MOLLY'
         WRITE(*,*) 'format and a crude correction has been'
         WRITE(*,*) 'applied. SPCRED should be changed.'
      END IF
      IF(OLD) THEN
         WRITE(*,*) 'At least one of your spectra had the constant'
         WRITE(*,*) 'term of an ephemeris stored as HJD* where *'
         WRITE(*,*) 'is either O, S or B. This is now obsolete to'
         WRITE(*,*) 'account for a safer search strategy when'
         WRITE(*,*) 'editing header items that caused confusion'
         WRITE(*,*) 'with HJD. The item has been renamed to Zero*'
         WRITE(*,*) 'and nothing else is affected. However if you'
         WRITE(*,*) 'want to avoid this message every time you read'
         WRITE(*,*) 'your spectra you should dump them and delete'
         WRITE(*,*) 'the old file. Sorry!'
      END IF
 999  RETURN
      END
