CPLOT
C PLOT N1 N2 [W0] [ECORR] [APERTURE] -- Plots spectra
C
C Parameters: 
C     
C  N1      -- Slot number of first spectrum to plot.
C  N2      -- Slot number of last spectrum to plot.
C  W0      -- Central wavelength for velocity plots (Angstroms)
C  ECORR   -- Yes if you want a correction to be applied for extinction
C             in response or efficiency plots. Without this correction
C             the response and efficiency are given as observed and so
C             will get worse with airmass. With the correction, the program
C             uses the airmass, grey and rayleigh coefficients from the
C             header to correct to zero airmass. All being well with the
C             flux calibration, the result should be constant from star
C             to star. If the airmass is not found it will be taken to
C             be 0, if grey and rayleigh are not found (real items)
C             they will be taken to be 0 and 1.
C
C  APERTURE-- Aperture of telescope for efficiency plots. In metres.
C
C Routine normally picks plot ranges automatically unless you allow
C it to prompt you for parameters in which case it also allows you
C to change ranges. If you want to hold ranges fixed, use command
C 'XYR' to fix them (X or Y or both). If you want to add to a plot,
C use the command 'FLAGS' to set CLOSE to false. Plot will then not
C be closed and then reopened as usual. To alter characteristics of
C the plot such as the axis colours, etc, try the command POPT. 
C Use AXES to change the default axes.
C Selection criteria apply to the spectra plotted.
CPLOT
      SUBROUTINE SPCPLT(SPLIT, NSPLIT, MXSPLIT, XAXIS, YAXIS, 
     &     PLOT1, PLOT2, PLOT3, MXPLOT, COUNTS, ERRORS, FLUX, 
     &     MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, 
     &     NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &     HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &     MXDOUB, MXINTR, MXREAL, NMSCHAR, VSCHAR, NMSDOUB, 
     &     VSDOUB, NMSINTR, VSINTR, NMSREAL, VSREAL, NSCHAR, 
     &     NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, MXSINTR, 
     &     MXSREAL, X1, X2, Y1, Y2, LW, LS, DCOL, ECOL, LABCOL, 
     &     AXCOL, LABFON, OFF, DSIZE, LSIZE, ASIZE, ASPECT, 
     &     ERRON, STYPE, ZERO, ID, XLABEL, YLABEL, TLABEL, 
     &     CLOSE, SLOTS, MXSLOTS, IFAIL)
C     
C     Master routine for plotting spectra into MOLLY buffers
C     allowing selection criteria to be applied.
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     >  C*(*) XAXIS            -- Units for X axis plot. Recognises:
C     'PIXELS','ANGSTROMS','NANNOMETRES',
C     'MICRONS','KM/S'
C     
C     >  C*(*) YAXIS            -- Units for Y axis plot. Recognises:
C     'COUNTS','MILLIJANSKYS','FLAMBDA'
C     'AB','RESPONSE','EFFICIENCY'
C     
C     >  R     PLOT1(MXPLOT)    -- Work arrays for plotting
C     >  R     PLOT2(MXPLOT)
C     >  R     PLOT3(MXPLOT)
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     >  C*(*) NMSCHAR(MXSCHAR)   -- Names of character selection items.
C     >  C*(*) VSCHAR(MXSCHAR)    -- Values of character selection items.
C     >  C*(*) NMSDOUB(MXSDOUB)   -- Names of double precision selection items.
C     >  D     VSDOUB(2,MXSDOUB)  -- Values of double precision selection items.
C     >  C*(*) NMSINTR(MXSINTR)   -- Names of integer selection items.
C     >  I     VSINTR(2,MXSINTR)  -- Values of integer selection items.
C     >  C*(*) NMSREAL(MXSREAL)   -- Names of real selection items.
C     >  R     VSREAL(2,MXSREAL)  -- Values of real selection items.
C     >  I     NSCHAR             -- Number of character selection items.
C     >  I     NSDOUB             -- Number of double precsion selection items.
C     >  I     NSINTR             -- Number of integer selection items.
C     >  I     NSREAL             -- Number of real selection items.
C     >  I     MXSCHAR            -- Dimension of NMSCHAR in calling routine.
C     >  I     MXSDOUB            -- Dimension of NMSDOUB in calling routine.
C     >  I     MXSINTR            -- Dimension of NMSINTR in calling routine.
C     >  I     MXSREAL            -- Dimension of NMSREAL in calling routine.   
C     incremented and returned by routine.
C     >  R     X1, X2, Y1, Y2     -- Plot ranges. Automatically determined if
C     set such that X1=X2, Y1=Y2
C     >  I     LW                 -- Line width for plot.
C     >  I     DCOL, ECOL         -- Colours for data and errors
C     >  I     LABCOL, AXCOL      -- Colours for labels and axes
C     >  I     LABFON             -- Font for labels
C     >  R     OFF                -- Offset between spectra
C     >  R DSIZE,LSIZE,ASIZE,ASPECT     -- Character sizes and aspect ratio
C     >  L ERRON                  -- TRUE to plot errors
C     >  I STYPE                  -- Spectrum type
C     >  L ZERO                   -- Zero levels plotted or not
C     >  L ID                     -- Identifier added or not
C     >  C*(*) XLABEL, YLABEL, TLABEL -- Plot labels. Defaults used if set to '?'
C     >  L     CLOSE              -- TRUE means plot will be closed before exiting
C     routine
C     >  I     SLOTS(MXSLOTS)       -- List of slots
C     >  I     MXSLOTS             -- Maximum number in list
C     <  I     IFAIL              -- Error return.
C     
      IMPLICIT NONE

C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL, MXSCHAR, MXSDOUB, MXSINTR
      INTEGER MXSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
      LOGICAL CLOSE
C     
C     Plot parameters
C     
      INTEGER MXPLOT 
      REAL PLOT1(MXPLOT), PLOT2(MXPLOT), PLOT3(MXPLOT)
      CHARACTER*(*) XAXIS, YAXIS
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
C     Search parameters, names then values.
C     
      CHARACTER*(*) NMSCHAR(MXSCHAR)
      CHARACTER*(*) NMSDOUB(MXSDOUB)
      CHARACTER*(*) NMSINTR(MXSINTR)
      CHARACTER*(*) NMSREAL(MXSREAL)
C     
      CHARACTER*(*) VSCHAR(MXSCHAR)
      DOUBLE PRECISION VSDOUB(2,MXSDOUB)
      INTEGER VSINTR(2,MXSINTR)
      REAL VSREAL(2,MXSREAL)
C     
C     Slot lists
C     
      INTEGER MXSLOTS, NSLOTS, SLOT
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
C     
C     Plot parameters
C     
      REAL X1, X2, Y1, Y2, OFF, DSIZE, LSIZE, ASIZE, ASPECT
      INTEGER LW, LS, LABCOL, AXCOL, DCOL, ECOL, LABFON
      INTEGER NPLOT, STYPE
      CHARACTER*(*) XLABEL, YLABEL, TLABEL
      CHARACTER*100 XLAB, YLAB, TLAB
      LOGICAL ERRON, ZERO, ID
C     
C     Local variables
C     
      INTEGER I, J,  N,  NCOM, LENSTR
      REAL ADD,  DELTA, WCEN, XR1, XR2, YR1, YR2, XP1, XP2
      REAL YP1, YP2, DIAM, DWELL, EFAC, GREY, RAYLEIGH, CGS
      DOUBLE PRECISION   VEARTH, AIRMASS
      CHARACTER*100 STRING, OBJECT*32, CORR*5
      INTEGER SLOT1, SLOT2,  IFAIL, NL, MAXSPEC, NF
      LOGICAL SELECT, SAMECI, TRIG,  ESET
      LOGICAL RECOG, PLOK, DEFAULT, NEW
C     
      SAVE CORR, WCEN, SLOT1, SLOT2, DIAM, NLIST
      DATA CORR/'Yes'/
      DATA SLOT1, SLOT2, NLIST/1,1,0/
      DATA DIAM, WCEN/4.2, 5000./
C     
C     Check inputs
C     
      IFAIL = 1
      CALL XTYPE(XAXIS, STRING, RECOG)
      IF(.NOT.RECOG) RETURN
      IF(XLABEL.EQ.'??') THEN
         XLAB = STRING
      ELSE
         XLAB = XLABEL
      END IF
      CALL YTYPE(YAXIS, STRING, RECOG)
      IF(.NOT.RECOG) RETURN
      IF(YLABEL.EQ.'??') THEN
         YLAB = STRING
      ELSE
         YLAB = YLABEL
      END IF
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL INTR_IN('First slot to plot', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to plot', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST.GT.0) THEN
            CALL SETSLOT(LIST, NLIST, MXLIST, SLOTS, NSLOTS, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra to plot'
            CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &           NSLOTS, MXSLOTS)
         END IF
         IF(NSLOTS.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &        NSLOTS, MXSLOTS)
      END IF
C     
C     If the plot is velocity, we need a zero velocity wavelength
C     
      IF(SAMECI(XAXIS,'KM/S')) THEN
         CALL REAL_IN('Central wavelength', SPLIT, NSPLIT, MXSPLIT, 
     &        NCOM, DEFAULT, WCEN, 0., 1.E5, IFAIL)
      END IF
C     
      IF(SAMECI(YAXIS,'RESPONSE') .OR. 
     &     SAMECI(YAXIS,'EFFICIENCY')) THEN
         CALL CHAR_IN('Correct to zero airmass?', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, CORR, IFAIL)
         IF(.NOT.SAMECI(CORR,'YES') .AND. .NOT.SAMECI(CORR,'NO')) THEN
            WRITE(*,*) 'Must answer yes or no'
            GOTO 999
         END IF
      END IF
      IF(SAMECI(YAXIS,'EFFICIENCY')) THEN
         CALL REAL_IN('Aperture of telescope (m)', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, DIAM, 1.E-3, 100., IFAIL)
         EFAC = 1.E26*CGS('H')/ATAN(1.)/(100.*DIAM)**2
      END IF
      IF(IFAIL.NE.0) GOTO 999
      IF(NSPLIT.GE.NCOM) DEFAULT = .TRUE.
C     
C     Evaluate limits
C     
      TRIG = .FALSE.
      XR1 =  1.E30
      XR2 = -1.E30
      YR1 =  1.E30
      YR2 = -1.E30
      ADD = 0.
      DO I = 1, NSLOTS
C     
C     Check that it is possible to plot requested
C     figure
C     
         SLOT = SLOTS(I)
         IF(.NOT. PLOK(NPIX(SLOT),MXPLOT,NARC(SLOT),
     &        XAXIS,YAXIS)) GOTO 100
C     
         IF(.NOT.SELECT(SLOT, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &        HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &        MXCHAR, MXDOUB, MXINTR, MXREAL, MXSPEC, NMSCHAR, VSCHAR, 
     &        NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, VSREAL, NSCHAR,
     &        NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, MXSINTR, 
     &        MXSREAL)) GOTO 100
C     
         IF(SAMECI(YAXIS,'RESPONSE') .OR. 
     &        SAMECI(YAXIS,'EFFICIENCY')) THEN
            CALL HGETR('Dwell', DWELL, SLOT, MXSPEC, NMREAL, 
     &           HDREAL, NREAL, MXREAL, IFAIL)
            IF(IFAIL.NE.0) GOTO 100
         END IF
         TRIG = .TRUE.
C     
         CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
         IFAIL = 0
         CALL HGETD('Airmass', AIRMASS, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
         IFAIL = 0
         CALL HGETR('Grey', GREY, SLOT, MXSPEC, NMREAL, 
     &        HDREAL, NREAL, MXREAL, IFAIL)
         IFAIL = 0
         CALL HGETR('Rayleigh', RAYLEIGH, SLOT, MXSPEC, NMREAL, 
     &        HDREAL, NREAL, MXREAL, IFAIL)
         IF(IFAIL.NE.0) RAYLEIGH = 1.
         IFAIL = 0
C     
         IF(X1.EQ.X2 .OR. Y1.EQ.Y2) THEN
            CALL PLOAD(PLOT1, PLOT2, PLOT3, NPLOT, MXPLOT, SLOT, 
     &           COUNTS, ERRORS, FLUX, MXBUFF, MAXPX, MXSPEC, NPIX, 
     &           ARC, NARC, MXARC, VEARTH, XAXIS, YAXIS, WCEN, ADD, 
     &           DWELL, ESET, DELTA, EFAC, IFAIL)
            IF(IFAIL.NE.0) THEN
               WRITE(*,*) 'Failure in PLOAD'
               GOTO 999
            END IF
            IF(X1.EQ.X2) THEN
               DO J = 1, NPIX(SLOT)
                  IF(PLOT3(J).GT.0.) THEN
                     XR1 = MIN(XR1, PLOT1(J)-DELTA)
                     XR2 = MAX(XR2, PLOT1(J)+DELTA)
                  END IF
               END DO
            END IF
            IF(Y1.EQ.Y2) THEN
C     
C     Apply extinction correction
C     
               IF(SAMECI(CORR,'YES')) CALL ECORR(PLOT2, SLOT, 
     &              ARC, NARC, NPIX, MXARC, MXSPEC, AIRMASS, GREY, 
     &              RAYLEIGH, YAXIS, ADD)
               DO J = 1, NPIX(SLOT)
                  IF(PLOT3(J).GT.0.) THEN
                     YR1 = MIN(YR1, PLOT2(J))
                     YR2 = MAX(YR2, PLOT2(J))
                  END IF
                  IF(YR1.EQ.YR2) THEN
                     YR1 = YR1-ABS(YR1)/10.
                     YR2 = YR2+ABS(YR2)/10.
                  END IF
               END DO
            END IF
         END IF
         ADD = ADD + OFF
 100     CONTINUE
      END DO
      IF(.NOT.TRIG) THEN
         WRITE(*,*) 'No valid spectra selected'
         GOTO 999
      END IF
C     
      IF(X1.EQ.X2) THEN
         XP1 = XR1 - (XR2-XR1)/20.
         XP2 = XR2 + (XR2-XR1)/20.
      ELSE
         XP1 = X1
         XP2 = X2
      END IF
      IF(Y1.EQ.Y2) THEN
         YP1 = YR1 - (YR2-YR1)/20.
         YP2 = YR2 + (YR2-YR1)/20.
      ELSE
         YP1 = Y1
         YP2 = Y2
      END IF
      IF(.NOT.DEFAULT) THEN
         CALL SET_LIM(XP1, XP2, YP1, YP2, IFAIL)
         IF(IFAIL.NE.0) RETURN
      END IF
C     
C     Plotting section. Always try to plot to currently
C     open device if there is one.
C     
      CALL DEV_OPEN(.TRUE.,NEW,IFAIL)
      IF(IFAIL.NE.0) RETURN
      IF(NEW) THEN
         CALL VPSET(XP1, XP2, YP1, YP2, LW, AXCOL, LABFON, 
     &        ASIZE, ASPECT)
         CALL PGSCH(LSIZE)
         CALL PGSCI(LABCOL)
         IF(TLABEL.EQ.'??') THEN
            IF(NSLOTS.EQ.1) THEN
               CALL HGETC('Object', OBJECT, SLOTS(1), MXSPEC, 
     &              NMCHAR, HDCHAR, NCHAR, MXCHAR, IFAIL)
               IF(IFAIL.EQ.0) THEN
                  TLAB = OBJECT
                  TLAB(33:39) = ', slot '
                  WRITE(TLAB(40:43), '(I4)') SLOTS(1)
               ELSE
                  TLABEL = 'Spectrum 1234'
                  WRITE(TLAB(10:13), '(I4)') SLOTS(1)
               END IF
            ELSE
               TLAB = 'Spectra 1234 to 1234'
               WRITE(TLAB(9:12), '(I4)') SLOT1
               WRITE(TLAB(17:20),'(I4)') SLOT2
            END IF
         ELSE
            TLAB = TLABEL
         END IF
         CALL PGLAB(XLAB, YLAB, TLAB)
         CALL PGSCI(1)
         CALL PGSCH(1.)
         IF(ID) CALL PGIDEN
      ELSE
         CALL PGSCI(1)
      END IF
      ADD = 0.
      CALL PGSCH(DSIZE)
      DO I = 1, NSLOTS
         SLOT = SLOTS(I)
         IF(.NOT.PLOK(NPIX(SLOT),MXPLOT,NARC(SLOT),
     &        XAXIS,YAXIS)) THEN
            WRITE(*,*) 'Spectrum ',SLOT,' skipped. Invalid format'
            GOTO 200
         END IF
C     
C     Finished checks, now apply selection criteria.
C     
C     
         IF(.NOT.SELECT(SLOT, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &        HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &        NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, 
     &        MXSPEC, NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, 
     &        NMSINTR, VSINTR, NMSREAL, VSREAL, NSCHAR, 
     &        NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, 
     &        MXSINTR, MXSREAL)) THEN
            WRITE(*,*) 'Spectrum ',SLOT,' skipped.'
            GOTO 200
         END IF
         IF(SAMECI(YAXIS,'RESPONSE') .OR. 
     &        SAMECI(YAXIS, 'EFFICIENCY')) THEN
            CALL HGETR('Dwell', DWELL, SLOT, MXSPEC, NMREAL, 
     &           HDREAL, NREAL, MXREAL, IFAIL)
            IF(IFAIL.NE.0) THEN
               WRITE(*,*) 
     &              'Skipped slot ',SLOT,' as no DWELL is present.'
               GOTO 200
            END IF
         END IF
C     
C     Get necessary header items
C     
         CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
         IFAIL = 0
         CALL HGETD('Airmass', AIRMASS, SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
         IFAIL = 0
         CALL HGETR('Grey', GREY, SLOT, MXSPEC, NMREAL, 
     &        HDREAL, NREAL, MXREAL, IFAIL)
         IFAIL = 0
         CALL HGETR('Rayleigh', RAYLEIGH, SLOT, MXSPEC, NMREAL, 
     &        HDREAL, NREAL, MXREAL, IFAIL)
         IF(IFAIL.NE.0) RAYLEIGH = 1.
         IFAIL = 0
C     
         CALL PLOAD(PLOT1, PLOT2, PLOT3, NPLOT, MXPLOT, SLOT, 
     &        COUNTS, ERRORS, FLUX, MXBUFF, MAXPX, MXSPEC, NPIX, 
     &        ARC, NARC, MXARC, VEARTH, XAXIS, YAXIS, WCEN, ADD, 
     &        DWELL, ESET, DELTA, EFAC, IFAIL)
C     
         IF(SAMECI(CORR,'YES')) CALL ECORR(PLOT2, SLOT, ARC, 
     &        NARC, NPIX, MXARC, MXSPEC, AIRMASS, GREY, RAYLEIGH, 
     &        YAXIS, ADD)
C     
C     Now plot, skipping regions with negative errors except
C     for certain spectrum types.
C     
         CALL PGSLS(LS)
         CALL PGSLW(LW)
         IF(NPLOT.GT.0) THEN
            N = 1
            DO WHILE(N.LT.NPIX(SLOT))
               NF = N
               DO WHILE(N.LE.NPIX(SLOT) .AND. PLOT3(N).GT.0.)
                  N = N + 1
               END DO
               NL = N - 1
               IF(NL.GE.NF) THEN
                  CALL PGSCI(DCOL)
                  IF(STYPE.EQ.1 .OR. STYPE.EQ.4 .OR. STYPE.EQ.5) THEN
                     CALL PGBIN(NL-NF+1, PLOT1(NF), PLOT2(NF), .TRUE.)
                  ELSE IF(STYPE.EQ.2) THEN
                     CALL PGPOINT(NL-NF+1, PLOT1(NF), PLOT2(NF), 17)
                  ELSE IF(STYPE.EQ.3) THEN
                     CALL PGPOINT(NL-NF+1, PLOT1(NF), PLOT2(NF), 17)
                     CALL PGLINE(NL-NF+1, PLOT1(NF), PLOT2(NF))
                  ELSE IF(STYPE.EQ.6) THEN
                     CALL PGLINE(NL-NF+1, PLOT1(NF), PLOT2(NF))
                  END IF
                  CALL PGSCI(ECOL)
                  IF(ESET .AND. ERRON) THEN
                     IF(STYPE.EQ.1 .OR. STYPE.EQ.5) THEN
                        CALL PGBIN(NL-NF+1, PLOT1(NF), PLOT3(NF), 
     &                       .TRUE.)
                     ELSE IF(STYPE.EQ.2 .OR. STYPE.EQ.3 .OR. 
     &                       STYPE.EQ.4) THEN
                        DO J = NF, NL
                           PLOT2(J) = PLOT2(J) - PLOT3(J)
                           PLOT3(J) = PLOT2(J) + 2.*PLOT3(J)
                        END DO
                        CALL PGERRY(NL-NF+1,PLOT1(NF),PLOT2(NF),
     &                       PLOT3(NF),0.)
                     ELSE IF(STYPE.EQ.6) THEN
                        CALL PGLINE(NL-NF+1, PLOT1(NF), PLOT3(NF))
                     END IF
                  END IF
               END IF
               NF = N
               DO WHILE(N.LE.NPIX(SLOT) .AND. PLOT3(N).LE.0.)
                  N = N + 1
               END DO
               NL = N - 1
               IF(NL.GE.NF .AND. STYPE.EQ.5) THEN
                  CALL PGSCI(DCOL)
                  CALL PGSLS(2)
                  CALL PGBIN(NL-NF+1, PLOT1(NF), PLOT2(NF), .TRUE.)
                  CALL PGSCI(ECOL)
                  IF(ESET .AND. ERRON) THEN
                     CALL PGBIN(NL-NF+1, PLOT1(NF), PLOT3(NF), .TRUE.)
                  END IF
                  CALL PGSLS(LS)
               END IF
            END DO
C     
C     Plot zero level
C     
            IF(ZERO) THEN
               CALL PGSCI(1)
               CALL PGMOVE(XP1, ADD)
               CALL PGDRAW(XP2, ADD)
            END IF
C     
            CALL SL_INF(SLOT, MXSPEC, NPIX, NMCHAR, NMDOUB, 
     &           NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, 
     &           NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, 
     &           MXINTR, MXREAL, STRING, IFAIL)
            WRITE(*,*) STRING(:LENSTR(STRING))
            ADD = ADD + OFF
         END IF
         CALL PGSLS(LS)
 200     CONTINUE
      END DO
      IFAIL = 0
      IF(CLOSE) CALL DEV_CLOSE(0,IFAIL)
      RETURN
 999  IFAIL = 1
      RETURN
      END

      SUBROUTINE PLOAD(PLOT1, PLOT2, PLOT3, NPLOT, MXPLOT, 
     &     SLOT, COUNTS, ERRORS, FLUX, MXBUFF, MAXPX, MXSPEC, 
     &     NPIX, ARC, NARC, MXARC, VEARTH, XAXIS, YAXIS, WCEN, 
     &     YADD, DWELL, ESET, DELTA, EFAC, IFAIL)
C     
C     Loads up plot arrays
C     
      IMPLICIT NONE
      INTEGER MXPLOT, MAXPX, MXARC, NPLOT, MXSPEC, MXBUFF
      INTEGER NPIX(MXSPEC), NARC(MXSPEC), SLOT
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
      REAL PLOT1(MXPLOT), PLOT2(MXPLOT), PLOT3(MXPLOT)
      LOGICAL PLOK, ESET, SAMECI
      CHARACTER*(*) XAXIS, YAXIS
      REAL WCEN, VLIGHT, CGS, VFAC, YADD, DWELL, FAC
      INTEGER I, N, IFAIL
      REAL GET_FLX, GET_ERF, AB_MJY, D, EFAC, CFAC, Z, A, Z1
      REAL CFRAT, DELTA
      DOUBLE PRECISION ANGST, DISPA, VEARTH
C     

      IF(.NOT.PLOK(NPIX(SLOT), MXPLOT, NARC(SLOT), 
     &     XAXIS, YAXIS)) GOTO 999

      VLIGHT = CGS('C')/1.E5
      VFAC = 1.-REAL(VEARTH)/VLIGHT  
C     
C     First set the X axis
C     
      N = 0
      IF(SAMECI(XAXIS,'PIXELS')) THEN
         DO I = 1, NPIX(SLOT)
            IF(ERRORS(I,SLOT).GT.0.) N = N + 1
            PLOT1(I) = I
         END DO
         DELTA = 1.0
      ELSE IF(SAMECI(XAXIS,'ANGSTROMS') .OR.
     &        SAMECI(XAXIS,'MICRONS') .OR. 
     &        SAMECI(XAXIS,'NANNOMETRES')) THEN
         DELTA = REAL(VFAC*ABS(DISPA(REAL(1+NPIX(SLOT))/2., NPIX(SLOT), 
     &        NARC(SLOT), ARC(1,SLOT), MXARC))/2.)
         IF(SAMECI(XAXIS,'ANGSTROMS')) THEN
            FAC = VFAC
         ELSE IF(SAMECI(XAXIS,'MICRONS')) THEN
            FAC   = 1.E-4*VFAC
            DELTA = 1.E-4*DELTA
         ELSE IF(SAMECI(XAXIS,'NANNOMETRES')) THEN
            FAC   = 1.E-1*VFAC
            DELTA = 1.E-1*DELTA
         END IF
         DO I = 1, NPIX(SLOT)
            IF(ERRORS(I,SLOT).GT.0.) N = N + 1
            PLOT1(I) = REAL(FAC*ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC))
         END DO
         
      ELSE IF(SAMECI(XAXIS,'KM/S')) THEN
         IF(WCEN.LE.0.) GOTO 999
         DO I = 1, NPIX(SLOT)
            IF(ERRORS(I,SLOT).GT.0.) N = N + 1
            Z = REAL(ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC))
            PLOT1(I) = REAL(VLIGHT*(Z/WCEN-1.)-VEARTH)
         END DO
         DELTA = REAL(VLIGHT*ABS(DISPA(REAL(1+NPIX(SLOT))/2., 
     &        NPIX(SLOT), NARC(SLOT), ARC(1,SLOT), MXARC))/WCEN/2.)
      END IF    
      NPLOT = N
C     
C     Now the Y axis
C     
      IF(SAMECI(YAXIS,'COUNTS')) THEN
         DO I = 1, NPIX(SLOT)
            PLOT2(I) = COUNTS(I,SLOT) + YADD
            PLOT3(I) = ERRORS(I,SLOT) 
         END DO
         ESET = .TRUE.
      ELSE IF(SAMECI(YAXIS,'MILLIJANSKYS')) THEN
         DO I = 1, NPIX(SLOT)
            PLOT2(I) = GET_FLX(COUNTS(I,SLOT), FLUX(I,SLOT)) + YADD
            PLOT3(I) = GET_ERF(COUNTS(I,SLOT), ERRORS(I,SLOT), 
     &           FLUX(I,SLOT)) 
         END DO
         ESET = .TRUE.
      ELSE IF(SAMECI(YAXIS,'FLAMBDA')) THEN
         CFAC = VLIGHT*1.E-13
         DO I = 1, NPIX(SLOT)
            Z = GET_FLX(COUNTS(I,SLOT), FLUX(I,SLOT))
            A = REAL(ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC))
            PLOT2(I) = CFAC*Z/A**2 + YADD
            Z = GET_ERF(COUNTS(I,SLOT), ERRORS(I,SLOT), 
     &           FLUX(I,SLOT))
            PLOT3(I) = CFAC*Z/A**2
         END DO
         ESET = .TRUE.
      ELSE IF(SAMECI(YAXIS,'AB')) THEN
         DO I = 1, NPIX(SLOT)
            PLOT2(I) = AB_MJY(GET_FLX(COUNTS(I,SLOT), 
     &           FLUX(I,SLOT))) + YADD
            PLOT3(I) = GET_ERF(COUNTS(I,SLOT), ERRORS(I,SLOT),
     &           FLUX(I,SLOT))
         END DO
         ESET = .FALSE.
      ELSE IF(SAMECI(YAXIS,'RESPONSE')) THEN
         DO I = 1, NPIX(SLOT)
            D  = REAL(ABS(DISPA(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC)))
            A  = REAL(ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC))
            Z1 = DWELL*D/CFRAT(COUNTS(I,SLOT), FLUX(I,SLOT))
            PLOT2(I) = AB_MJY(Z1) + YADD
            PLOT3(I) = GET_ERF(COUNTS(I,SLOT), ERRORS(I,SLOT),
     &           FLUX(I,SLOT))          
         END DO
         ESET = .FALSE.
      ELSE IF(SAMECI(YAXIS,'EFFICIENCY')) THEN
         DO I = 1, NPIX(SLOT)
            D = REAL(ABS(DISPA(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC)))
            A = REAL(ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC))
            Z = EFAC*CFRAT(COUNTS(I,SLOT), FLUX(I,SLOT))/DWELL*A/D
            PLOT2(I) = Z + YADD
            PLOT3(I) = GET_ERF(COUNTS(I,SLOT), ERRORS(I,SLOT),
     &           FLUX(I,SLOT))          
         END DO
         ESET = .FALSE.
      END IF
      IFAIL = 0
      RETURN
 999  IFAIL = 1
      RETURN
      END
      
      SUBROUTINE ECORR(PLOT, SLOT, ARC, NARC, NPIX, 
     &     MXARC, MXSPEC, AIR, GREY, RAYLEIGH, YAXIS, YADD)
C     
C     Corrects response and efficiency plots to zero airmass
C     
      IMPLICIT NONE
      INTEGER MXARC, MXSPEC, SLOT, I
      INTEGER NARC(MXSPEC), NPIX(MXSPEC)
      REAL PLOT(*), YADD
      DOUBLE PRECISION ANGST, ARC(MXARC,MXSPEC)
      DOUBLE PRECISION AIR
      REAL GREY, RAYLEIGH, EXTINC, W
      LOGICAL SAMECI
      CHARACTER*(*) YAXIS
C     
      IF(SAMECI(YAXIS,'RESPONSE')) THEN
         DO I = 1, NPIX(SLOT)
            W  = REAL(ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC))
            PLOT(I) = PLOT(I) + REAL(AIR*(GREY+RAYLEIGH*EXTINC(W)))
         END DO
      ELSE IF(SAMECI(YAXIS,'EFFICIENCY')) THEN
         DO I = 1, NPIX(SLOT)
            W = REAL(ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &           ARC(1,SLOT), MXARC))
            PLOT(I) = (PLOT(I)-YADD)*
     &           10.**(SNGL(AIR)*(GREY+RAYLEIGH*EXTINC(W))/2.5) + YADD
         END DO
      END IF
      RETURN
      END
      
      LOGICAL FUNCTION PLOK(NPIX, MXPLOT, NARC, XAXIS, YAXIS)
C     
C     Checks that a spectrum is plottable in requested format
C     
      IMPLICIT NONE
      INTEGER NPIX, MXPLOT, NARC
      CHARACTER*(*) XAXIS, YAXIS
      LOGICAL SAMECI

C     
      PLOK = .NOT.(NPIX.GT.MXPLOT .OR. (NARC.EQ.0
     &     .AND. (SAMECI(XAXIS,'ANGSTROMS') .OR.
     &     SAMECI(XAXIS,'MICRONS') .OR. 
     &     SAMECI(XAXIS,'NANNOMETRES') .OR.
     &     SAMECI(XAXIS,'KM/S')))) 
C     
      PLOK = PLOK .AND. .NOT. (NARC.EQ.0
     &     .AND. (SAMECI(YAXIS,'RESPONSE') .OR.
     &     SAMECI(YAXIS,'FLAMBDA') .OR. 
     &     SAMECI(YAXIS,'EFFICIENCY')) ) 
C     
      PLOK = PLOK .AND. NPIX.GT.0
C     
      RETURN 
      END
      


