CSDEC
C SDEC N1 N2 N3 N4 HJD0 NFREQ FREQ1, FREQ2, ...  PFIX [PHASE1, ...] NPOLY -- 
C
C                    Fits sinusoidal waves as function of HJD with the same fixed
C                    frequencies at every wavelength. Option to fix phases too.
C
C If the phases are allowed to be free then five spectra are generated/period. 
C
C    These are:
C             (1) amplitude of cosine component, 
C             (2) amplitude of sine component,
C             (3) Total amplitude
C             (4) Phase (in degrees) defined by 
C                 Cos(omega*t+phi), phi = phase converted to degrees.
C             (5) Total amplitude corrected for noise bias because a 
C                 positive amplitude is always obtained even with no true 
C                 signal present. The latter is set to zero if the power 
C                 becomes < 0 during correction. 
C
C  A character header item called 'Type' is set equal to 'Cosine', 'Sine', 
C  'Amplitude', 'Phase' and 'Corr amplitude' for the various spectra. The 
C  HJD0 and frequency used are stored in double precision items CSfreq and CSHJD
C
C  If the phases are fixed then only the amplitude is fitted (Type = 'Amplitude')
C  and the sequence of output spectra are just the amplitude spectra for each 
C  frequency. The phase used is stored in the REAL header parameter CSphase
C
C Model fitted Y = A*COS(2*PI*PHI) + B*SIN(2*PI*PHI) for every period but a 
C polynomial is fitted and subtracted prior to computation to reduce power 
C leakage from low frequencies. Masked data are ignored and the fitted sinusoid
C is not subtracted from them.
C
C NB. This routine has to access the arrays in the wrong direction and may
C     therefore page fault alot. 
C
C Parameters: 
C     
C  N1    -- Slot number of first spectrum to fit
C  N2    -- Slot number of last spectrum to fit
C  N3    -- Slot of first (cosine) component spectrum
C  N4    -- Slot of first output spectrum with component subtracted (0 to
C           ignore)
C  HJD0  -- HJD0 of zero phase
C  NFREQ -- Number of frequencies
C  FREQ1, FREQ2, ... - Frequencies of waves. Enter as positive number if you want
C                    it to be subtracted or negative if you don't (only
C                    relevant if N4 is greater than 0). Units of days.
C                    Units of cycles/day.
C  PFIX   -- Fix phases? Y or N
C  PHASE1, PHASE2, ... if FIX='Y' then you will need to fix the phases.
C  NPOLY -- Number of coefficients of polynomial to subtract
C
C Related command: tpoly
C
CSDEC
      SUBROUTINE SINDEC(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &     FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, 
     &     NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, 
     &     NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, 
     &     NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, 
     &     VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, 
     &     MXSINTR, MXSREAL, DWORK1, DWORK2, DWORK3, DWORK4, DWORK5,
     &     DWORK6, WHICH, MXWORK, SLOTS, MXSLOTS, IFAIL)
C     
C     Fits sinuisoidal waves to time series spectra.
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     >  C*(*) NMSCHAR(MXSCHAR)   -- Names of character selection items.
C     >  C*(*) VSCHAR(MXSCHAR)    -- Values of character selection items.
C     >  C*(*) NMSDOUB(MXSDOUB)   -- Names of double precision selection items.
C     >  D     VSDOUB(2,MXSDOUB)  -- Values of double precision selection items.
C     >  C*(*) NMSINTR(MXSINTR)   -- Names of integer selection items.
C     >  I     VSINTR(2,MXSINTR)  -- Values of integer selection items.
C     >  C*(*) NMSREAL(MXSREAL)   -- Names of real selection items.
C     >  R     VSREAL(2,MXSREAL)  -- Values of real selection items.
C     >  I     NSCHAR             -- Number of character selection items.
C     >  I     NSDOUB             -- Number of double precsion selection items.
C     >  I     NSINTR             -- Number of integer selection items.
C     >  I     NSREAL             -- Number of real selection items.
C     >  I     MXSCHAR            -- Dimension of NMSCHAR in calling routine.
C     >  I     MXSDOUB            -- Dimension of NMSDOUB in calling routine.
C     >  I     MXSINTR            -- Dimension of NMSINTR in calling routine.
C     >  I     MXSREAL            -- Dimension of NMSREAL in calling routine.   
C     >  D     DWORK1  Work array
C     >  D     DWORK2      "
C     >  D     DWORK3      "
C     >  D     DWORK4      "
C     >  D     DWORK5      "
C     >  D     DWORK6      "
C     >  I     WHICH       "
C     >  I     MXWORK     Size of work array
C     I     SLOTS(MXSLOTS) -- Work array for list of slots
C     I     MXSLOTS
C     <  I     IFAIL              -- Error return.
C     
      IMPLICIT NONE

C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL   FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Search parameters
C     
      INTEGER MXSCHAR, MXSDOUB, MXSINTR
      INTEGER MXSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL
C     
C     Search parameters, names then values.
C     
      CHARACTER*(*) NMSCHAR(MXSCHAR)
      CHARACTER*(*) NMSDOUB(MXSDOUB)
      CHARACTER*(*) NMSINTR(MXSINTR)
      CHARACTER*(*) NMSREAL(MXSREAL)
C     
      CHARACTER*(*) VSCHAR(MXSCHAR)
      DOUBLE PRECISION VSDOUB(2,MXSDOUB)
      INTEGER VSINTR(2,MXSINTR)
      REAL VSREAL(2,MXSREAL)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
      INTEGER MXWORK
      DOUBLE PRECISION DWORK1(MXWORK), DWORK2(MXWORK)
      DOUBLE PRECISION DWORK3(MXWORK), DWORK4(2*MXWORK) 
      DOUBLE PRECISION DWORK5(2*MXWORK), DWORK6(MXWORK)
      INTEGER WHICH(MXWORK)
C     
C     Slot lists
C     
      INTEGER MXSLOTS, NSLOTS, SLOT, SLT
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
C     
C     Local variables
C     
      INTEGER I, J, K,   IND
      INTEGER   NV, NSAME
      REAL GET_FLX, GET_ERF, CFRAT, RATIO
      DOUBLE PRECISION POLY,  HJD0, f
      DOUBLE PRECISION AVE, LOW, HIGH, RANGE
      INTEGER MXPOL, NPOLY, NOK, MAXFREQ, NFREQ, NCOM
      PARAMETER (MXPOL=15,MAXFREQ=10)
      DOUBLE PRECISION XM(MXPOL,2*MXPOL+3), CHISQ, AF(MXPOL)
      DOUBLE PRECISION FREQ(MAXFREQ)
      REAL A(MAXFREQ), B(MAXFREQ), EA(MAXFREQ), EB(MAXFREQ)
      REAL RAB(MAXFREQ), P(MAXFREQ),  EP(MAXFREQ),  CP(MAXFREQ)
      REAL CEP(MAXFREQ), PH(MAXFREQ), EPH(MAXFREQ)
      INTEGER NR(MAXFREQ), NL(MAXFREQ)
      CHARACTER*100 STRING
      CHARACTER*2 REPLY
      INTEGER SLOT1, SLOT2, SLOT3, SLOT4, IFAIL, MAXSPEC, NP, LENSTR
      REAL FLX, PHS, TWOPI
      LOGICAL SELECT, RESET, PFIXED, DEFAULT, SAMECI
C     
      DATA SLOT1, SLOT2, SLOT3, SLOT4/1,3,4,0/
      DATA HJD0, FREQ/2440000., 10., 20.,30.,40.,50.,
     &     60.,70.,80.,90.,100./
      DATA NPOLY, NFREQ/2, 1/
      DATA B/0.,0.,0.,0.,0.,0.,0.,0.,0.,0./
      DATA REPLY/'Y'/
C     
      TWOPI = 8.*ATAN(1.)      
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
C     
C     First slot
C     
      CALL INTR_IN('First spectrum to fit', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
C     
C     Second slot
C     
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last spectrum to fit', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST.GT.0) THEN
            CALL SETSLOT(LIST, NLIST, MXLIST, SLOTS, NSLOTS, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra to fit'
            CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &           NSLOTS, MXSLOTS)
         END IF
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &        NSLOTS, MXSLOTS)
      END IF
      IF(NSLOTS.EQ.0) GOTO 999
C     
C     Check through for valid spectra. Store values for ease
C     of reference later
C     
      NSAME = 0
      NV    = 0
      DO I = 1, NSLOTS
         SLOT = SLOTS(I)
         IF( NPIX(SLOT).GT.0 .AND. NPIX(SLOT).LE.MXWORK .AND.
     &        SELECT(SLOT,NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &        HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &        MXCHAR, MXDOUB, MXINTR, MXREAL, MXSPEC, NMSCHAR, 
     &        VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, 
     &        VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, 
     &        MXSDOUB, MXSINTR, MXSREAL)) THEN
            IF(NSAME.EQ.0) THEN
               NSAME = NPIX(SLOT)
               NV = 1
               WHICH(NV) = SLOT
            ELSE IF(NPIX(SLOT).NE.NSAME) THEN
               WRITE(*,*) 'Slot ',SLOT,' has ',NPIX(SLOT),' pixels'
               WRITE(*,*) 'rather than ',NSAME,' as the first valid'
               WRITE(*,*) 'spectrum. Choose again.'
               GOTO 999
            ELSE
               NV = NV + 1
               WHICH(NV) = SLOT
            END IF
         END IF
      END DO
      IF(NV.LT.3) THEN
         WRITE(*,*) 'Need at least 3 valid spectra to fit'
         GOTO 999
      END IF
C     
C     First slot for output
C     
      CALL INTR_IN('First slot for components', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT3, 1, MAXSPEC-4, IFAIL)
C     
      CALL INTR_IN('First slot for subtracted spectra (0 to ignore)',
     &     SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT4, 0, 
     &     MAXSPEC-NV+1, IFAIL)
C     
C     HJD0
C     
      CALL DOUB_IN('HJD of zero phase', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, HJD0, 0.D0, 1.D10, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
C     Number of frequencies
C     
      CALL INTR_IN('Number of frequencies', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NFREQ, 1, MAXFREQ, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(NFREQ*NV.GT.2*MXWORK) THEN
         WRITE(*,*) 'There is not enough workspace for storage of'
         WRITE(*,*) 'sines and cosines. Space available = ',2*MXWORK
         WRITE(*,*) 'Space needed = ',NFREQ*NV
         WRITE(*,*) 'Either increase work array in MOLLY, or use fewer'
         WRITE(*,*) 'spectra and or frequencies.'
         GOTO 999
      END IF
C     
C     Periods
C     
      DO I = 1, NFREQ
         STRING = 'Enter frequency (cycles/day) of sinusoid'
         IF(NFREQ.GT.1) WRITE(STRING(42:43),'(I2)') I 
         CALL DOUB_IN(STRING, SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &        DEFAULT, FREQ(I),-1.D6, 1.D6, IFAIL)
         IF(IFAIL.NE.0) GOTO 999
         IF(FREQ(I).EQ.0.D0) THEN
            WRITE(*,*) 'Can''t have frequency = 0'
            GOTO 999
         END IF
      END DO

      CALL CHAR_IN('Hold phases fixed?', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, REPLY, IFAIL)
      PFIXED = SAMECI(REPLY,'Y')

      IF(PFIXED) THEN
         DO I = 1, NFREQ
            STRING = 'Enter phase (degrees) of sinusoid'
            IF(NFREQ.GT.1) WRITE(STRING(35:36),'(I2)') I 
            CALL REAL_IN(STRING, SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &           DEFAULT, B(I), -360., 360., IFAIL)
            IF(IFAIL.NE.0) GOTO 999
         END DO
      END IF
C     
      CALL INTR_IN('Order of poly to subtract', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NPOLY, 1, MXPOL, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(NPOLY.GE.NV) THEN
         WRITE(*,*) 'Too few spectra for poly order = ',NV
         GOTO 999
      END IF
C     
C     Store times 
C     
      AVE = 0.D0
      DO I = 1, NV
         SLOT = WHICH(I)
         CALL HGETD('HJD', DWORK1(I), SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &        NDOUB, MXDOUB, IFAIL)
         IF(IFAIL.NE.0) THEN
            WRITE(*,*) 'Failed to find HJD in slot ',SLOT
            GOTO 999
         END IF
         DWORK1(I) = DWORK1(I)-HJD0
         AVE = AVE + DWORK1(I)
      END DO
      AVE = AVE/DBLE(NV)
C     
C     Scale for least squares fitting
C     
      LOW  =  1.D30
      HIGH = -1.D30
      DO I = 1, NV
         DWORK6(I) = DWORK1(I) - AVE
         LOW  = MIN(LOW, DWORK6(I))
         HIGH = MAX(HIGH, DWORK6(I))
      END DO
      RANGE = HIGH-LOW
      DO I = 1, NV
         DWORK6(I) = DWORK6(I)/RANGE
      END DO
C     
C     Copy over spectra if SLOT4 > 0
C     
      IF(SLOT4.GT.0) THEN
         WRITE(*,*) 'Copying spectra to subtracted slots'
         DO I = 1, NV
            SLOT = SLOT4+I-1
            CALL SET_HEAD(SLOT, WHICH(I), MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
            DO J = 1, NSAME
               COUNTS(J,SLOT) = COUNTS(J,WHICH(I))
               ERRORS(J,SLOT) = ERRORS(J,WHICH(I))
               FLUX(J,SLOT)   = FLUX(J,WHICH(I))
            END DO
         END DO
         WRITE(*,*) 'Finished copying'
      END IF
C     
C     Loop through pixels
C     
      RESET = .TRUE.
      DO J = 1, NFREQ
         NL(J) = 0
         NR(J) = 0
      END DO
      DO J = 1, NSAME
         NOK = 0
         DO I = 1, NV
            SLOT = WHICH(I)
            DWORK2(I) = GET_FLX(COUNTS(J,SLOT), FLUX(J,SLOT))
            DWORK3(I) = GET_ERF(COUNTS(J,SLOT), ERRORS(J,SLOT), 
     &           FLUX(J,SLOT))
            IF(DWORK3(I).GT.0.D0) NOK = NOK + 1
         END DO
C     
C     Fit and subtract poly 
C     
         IF(NOK.GT.MAX(3,NPOLY)) THEN
            CALL LEASQ(DWORK6, DWORK2, DWORK3, NV, NPOLY, AF,
     &           CHISQ,XM,1)
            DO I = 1, NV
               DWORK2(I) = DWORK2(I) - POLY(AF,NPOLY,DWORK6(I))
            END DO
C     
C     Fit sinusoid + const
C     
            CALL SINCOS(DWORK1, DWORK2, DWORK3, NV, NFREQ, FREQ, A, 
     &           B, EA, EB, RAB, NP, F, DWORK4, DWORK5, PFIXED, 
     &           RESET, IFAIL)
            RESET = .FALSE.
C     
C     Subtract all sinusoids with positive period
C     
            IF(SLOT4.GT.0) THEN
               DO I = 1, NV
                  SLOT = SLOT4+I-1
                  SLT  = WHICH(I)
                  IF(ERRORS(J,SLT).GT.0.) THEN
                     RATIO = CFRAT(COUNTS(J,SLT),FLUX(J,SLT))
                     FLX   = GET_FLX(COUNTS(J,SLT),FLUX(J,SLT))
                     DO K = 1, NFREQ
                        IF(FREQ(K).GT.0.D0) THEN
                           IND = NV*(K-1)+I
                           FLX   = REAL(FLX - A(K)*DWORK4(IND) - 
     &                          B(K)*DWORK5(IND))
                        END IF
                     END DO
                     COUNTS(J,SLOT) = RATIO*FLX
                     IF(COUNTS(J,SLOT).EQ.0.) THEN
                        FLUX(J,SLOT) = RATIO
                     ELSE
                        FLUX(J,SLOT) = FLX
                     END IF
                  END IF
               END DO
            END IF
            DO I = 1, NFREQ
               IF(.NOT.PFIXED) THEN
                  P(I)  = SQRT(A(I)**2+B(I)**2)
                  EP(I) = SQRT((A(I)*EA(I))**2+(B(I)*EB(I))**2
     &                 +2.*A(I)*EA(I)*B(I)*EB(I)*RAB(I))/P(I)
                  CP(I) = SQRT(MAX(0.,P(I)**2-EA(I)**2-EB(I)**2))
                  CEP(I)= EP(I)
                  IF(A(I).NE.0. .OR. B(I).NE.0.) THEN
                     PH(I)  = 360.*ATAN2(-B(I),A(I))/TWOPI
                     IF(PH(I).GT.-90. .AND. PH(I).LT.90.) THEN
                        NR(I) = NR(I) + 1
                     ELSE
                        NL(I) = NL(I) + 1
                     END IF
                     EPH(I) = SQRT((A(I)*EB(I))**2+(B(I)*EA(I))**2
     &                    -2.*A(I)*EA(I)*B(I)*EB(I)*RAB(I))/
     &                    (A(I)**2+B(I)**2)
                     EPH(I) = 360.*EPH(I)/TWOPI
                  ELSE
                     PH(I)  = 0.
                     EPH(I) = 360.
                  END IF
               END IF
            END DO
         ELSE
            DO I = 1, NFREQ
               A(I)   =  0.
               EA(I)  = -1.
               IF(.NOT.PFIXED) B(I)   =  0.
               EB(I)  = -1.
               P(I)   =  0.
               EP(I)  = -1.
               CP(I)  =  0.
               CEP(I) = -1.
               PH(I)  =  0.
               EPH(I) = -1.
            END DO
         END IF
C     
C     Store cosine and sine amplitudes. Use first valid spectrum
C     to provide sensitivity 
C     
         RATIO = CFRAT(COUNTS(J,WHICH(1)),FLUX(J,WHICH(1)))
         DO I = 1, NFREQ
            IF(PFIXED) THEN
C     
C     Phases fixed, store only amplitudes.
C     
               SLT = SLOT3 + I - 1
               COUNTS(J,SLT) = RATIO*A(I)
               ERRORS(J,SLT) = RATIO*EA(I)
               IF(COUNTS(J,SLT).EQ.0.) THEN
                  FLUX(J,SLT) = RATIO
               ELSE
                  FLUX(J,SLT) = A(I)
               END IF
            ELSE
C     
C     Cosine
C     
               SLT = SLOT3 + 5*(I-1)
               COUNTS(J,SLT) = RATIO*A(I)
               ERRORS(J,SLT) = RATIO*EA(I)
               IF(COUNTS(J,SLT).EQ.0.) THEN
                  FLUX(J,SLT) = RATIO
               ELSE
                  FLUX(J,SLT) = A(I)
               END IF
C     
C     Sine
C     
               SLT = SLT + 1
               COUNTS(J,SLT) = RATIO*B(I)
               ERRORS(J,SLT) = RATIO*EB(I)
               IF(COUNTS(J,SLT).EQ.0.) THEN
                  FLUX(J,SLT) = RATIO
               ELSE
                  FLUX(J,SLT) = B(I)
               END IF
C     
C     Amplitude
C     
               SLT = SLT + 1
               COUNTS(J,SLT) = RATIO*P(I)
               ERRORS(J,SLT) = RATIO*EP(I)
               IF(COUNTS(J,SLT).EQ.0.) THEN
                  FLUX(J,SLT) = RATIO
               ELSE
                  FLUX(J,SLT) = P(I)
               END IF
C     
C     Phase
C     
               SLT = SLT + 1
               COUNTS(J,SLT) = RATIO*PH(I)
               ERRORS(J,SLT) = RATIO*EPH(I)
               IF(COUNTS(J,SLT).EQ.0.) THEN
                  FLUX(J,SLT) = RATIO
               ELSE
                  FLUX(J,SLT) = PH(I)
               END IF
C     
C     Amplitude corrected for noise bias
C     
               SLT = SLT + 1
               COUNTS(J,SLT) = RATIO*CP(I)
               ERRORS(J,SLT) = RATIO*CEP(I)
               IF(COUNTS(J,SLT).EQ.0.) THEN
                  FLUX(J,SLT) = RATIO
               ELSE
                  FLUX(J,SLT) = CP(I)
               END IF
            END IF
         END DO
      END DO
C     
C     Correct phases if more appear to be in lefthand quadrants
C     
      IF(.NOT.PFIXED) THEN
         DO J = 1, NFREQ
            IF(NL(J).GT.NR(J)) THEN
               SLOT = SLOT3+5*(J-1)+3
               DO I = 1, NSAME
                  RATIO = CFRAT(COUNTS(I,SLOT),FLUX(I,SLOT))
                  PHS = GET_FLX(COUNTS(I,SLOT),FLUX(I,SLOT))
                  IF(PHS.LT.0.) PHS = PHS + 360.
                  COUNTS(I,SLOT) = RATIO*PHS
                  IF(COUNTS(I,SLOT).EQ.0.) THEN
                     FLUX(I,SLOT) = RATIO
                  ELSE
                     FLUX(I,SLOT) = PHS
                  END IF
               END DO
            END IF
         END DO
      END IF
C     
C     Set headers of output spectra equal to those of first valid
C     spectrum. Set types for ease of reference later
C     
      DO J = 1, NFREQ
         IF(PFIXED) THEN
            SLT = SLOT3 + J - 1
            CALL SET_HEAD(SLT, WHICH(1), MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
            CALL HSETC('Type', 'Amplitude', SLT, MXSPEC, NMCHAR, HDCHAR,
     &           NCHAR, MXCHAR, IFAIL)
            CALL HSETD('CSfreq', FREQ(J), SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL HSETD('CSHJD', HJD0, SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL HSETR('CSphase', B(J), SLT, MXSPEC, NMREAL, HDREAL, 
     &           NREAL, MXREAL, IFAIL)
            CALL SL_INF(SLT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) 'Amplitude for frequency = ',FREQ(J),
     &           ' stored in:'
            WRITE(*,*) STRING(:LENSTR(STRING))
            WRITE(*,*) ' '
         ELSE
            SLT = SLOT3 + 5*(J-1)
            CALL SET_HEAD(SLT, WHICH(1), MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
            CALL HSETC('Type', 'Cosine', SLT, MXSPEC, NMCHAR, HDCHAR, 
     &           NCHAR, MXCHAR, IFAIL)
            CALL HSETD('CSfreq', FREQ(J), SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL HSETD('CSHJD', HJD0, SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL SL_INF(SLT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) 'Cosine amplitude for frequency = ',FREQ(J),
     &           ' stored in:'
            WRITE(*,*) STRING(:LENSTR(STRING))
            WRITE(*,*) ' '
            SLT = SLT + 1
            CALL SET_HEAD(SLT, WHICH(1), MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
            CALL HSETC('Type', 'Sine', SLT, MXSPEC, NMCHAR, HDCHAR, 
     &           NCHAR, MXCHAR, IFAIL)
            CALL HSETD('CSfreq', FREQ(J), SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL HSETD('CSHJD', HJD0, SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL SL_INF(SLT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) 'Sine amplitude for frequency = ',FREQ(J),
     &           ' stored in:'
            WRITE(*,*) STRING(:LENSTR(STRING))
            WRITE(*,*) ' '
            SLT = SLT + 1
            CALL SET_HEAD(SLT, WHICH(1), MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
            CALL HSETC('Type', 'Amplitude', SLT, MXSPEC, NMCHAR, HDCHAR,
     &           NCHAR, MXCHAR, IFAIL)
            CALL HSETD('CSfreq', FREQ(J), SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL HSETD('CSHJD', HJD0, SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL SL_INF(SLT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) 'Total amplitude for frequency = ',FREQ(J),
     &           ' stored in:'
            WRITE(*,*) STRING(:LENSTR(STRING))
            WRITE(*,*) ' '
            SLT = SLT + 1
            CALL SET_HEAD(SLT, WHICH(1), MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
            CALL HSETC('Type', 'Phase', SLT, MXSPEC, NMCHAR, HDCHAR, 
     &           NCHAR, MXCHAR, IFAIL)
            CALL HSETD('CSfreq', FREQ(J), SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL HSETD('CSHJD', HJD0, SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL SL_INF(SLT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) 'Phase for frequency = ',FREQ(J),
     &           ' stored in:'
            WRITE(*,*) STRING(:LENSTR(STRING))
            WRITE(*,*) ' '
            SLT = SLT + 1
            CALL SET_HEAD(SLT, WHICH(1), MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
            CALL HSETC('Type', 'Corr amplitude', SLT, MXSPEC, NMCHAR, 
     &           HDCHAR, NCHAR, MXCHAR, IFAIL)
            CALL HSETD('CSfreq', FREQ(J), SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL HSETD('CSHJD', HJD0, SLT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            CALL SL_INF(SLT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) 'Noise corrected amplitude for frequency = ',
     &           FREQ(J),' stored in:'
            WRITE(*,*) STRING(:LENSTR(STRING))
            WRITE(*,*) ' '
         END IF
      END DO
      IF(SLOT4.GT.0) THEN
         WRITE(*,*) 'Spectra with component(s) subtracted are stored in'
         WRITE(*,*) 'slots ',SLOT4,' to ',SLOT4+NV-1
      END IF
 999  RETURN
      END

      SUBROUTINE SINCOS(XDATA, YDATA, YERR, NUM, NFREQ, FREQ, A, B, 
     &     EA, EB, RAB, NP, F, COSINE, SINE, PFIXED, RESET, IFAIL)
C     
C     Subroutine to fit sinusoids. It has two major modes of operation set by
C     PFIXED standing for phase fixed.
C     
C     If PFIXED is .FALSE. then it fits:
C     
C     Y = Sum_I=1,NFREQ A(I)*COS(2*PI*FREQ(I)*X) + B(I)*SIN(2*PI*FREQ(I)*X)
C     
C     If PFIXED is .TRUE. then it fits:
C     
C     Y = Sum_I=1,NFREQ A(I)*COS(2*PI*FREQ(I)*X-TWOPI*B(I)/360.)
C     
C     where the B(I) are the phase in degrees.
C     
C     This has two
C     If PFIXED the B(I) are not fitted and it is assumed that the phases are
C     fixed at values entered through the array B.
C     
C     Returning A(NFREQ), B(NFREQ), and their errors EA(NFREQ), EB(NFREQ) and 
C     correlation coefficient RAB
C     
C     
C     Passed:
C     
C     D XDATA(NUM) -- Array of X values
C     D YDATA(NUM) -- Array of Y values
C     D YERR(NUM)  -- Array of errors on Y values 1-sigma
C     I NUM        -- Number of values
C     I NFREQ     -- Number of periods to fit
C     D FREQ(NFREQ) - Values of periods
C     D COSINE(NUM,NFREQ)    Work arrays containing cosine and sine evaluated
C     D SINE(NUM,NFREQ)      at correct times. Computed if RESET
C     L RESET        TRUE to start with, FALSE on subsequent calls to avoid
C     recomputing SINE and COSINE
C     
C     RETURNED: 
C     
C     R A(NFREQ)   -- Cosine amplitude
C     R B(NFREQ)   -- Sine amplitude
C     R EA(NFREQ)  -- Uncertainty in A
C     R EB(NFREQ)  -- Uncertainty in B
C     R RAB(NFREQ) -- Correlation coefficient  between A(I), B(I)
C     
C     I NP    -- Actual number of points used
C     R F     -- Value of chi-squared, correct if YERR are scaled correctly
C     I IFAIL -- =0 If no error , 1 otherwise
C     
C     To ignore any point, YERR should be set to less than or equal 0.
C     
C     
C     
      IMPLICIT NONE
      INTEGER I, J, K, NUM, NP, NFREQ, IFAIL
      LOGICAL PFIXED
      DOUBLE PRECISION XDATA(NUM), YDATA(NUM), YERR(NUM), TWOPI
      DOUBLE PRECISION COSINE(NUM,NFREQ), SINE(NUM,NFREQ), FREQ(NFREQ)
      DOUBLE PRECISION XX, SNJ, CNJ, CNK, SNK, WW, PHASE, DETERM, F
      REAL A(NFREQ), B(NFREQ), EA(NFREQ), EB(NFREQ), RAB(NFREQ)
      INTEGER MAXFREQ
      PARAMETER (MAXFREQ=10)
      DOUBLE PRECISION M(2*MAXFREQ+1,2*MAXFREQ), V(2*MAXFREQ),  YSQ
      DOUBLE PRECISION  DWK(2*MAXFREQ)
      INTEGER  IWK(2*MAXFREQ), IND
      INTEGER I1, I2, J1, J2, K1, K2
      LOGICAL RESET

      IF(NFREQ.LT.1 .OR. NFREQ.GT.MAXFREQ) THEN
         WRITE(*,*) 'Number of periods out of range 1 to ',MAXFREQ
         WRITE(*,*) 'inside SINCOS'
         IFAIL = 1
         RETURN
      END IF
      TWOPI = 8.D0*ATAN(1.D0)
      IFAIL = 0
      DO J = 1, 2*NFREQ
         V(J) = 0.D0
         DO I = 1, 2*NFREQ
            M(I,J) = 0.D0
         END DO
      END DO
      YSQ = 0.D0
      IF(RESET) THEN
         DO J = 1, NFREQ
            IF(PFIXED) PHASE = TWOPI*DBLE(B(J))/360.
            DO I = 1, NUM
               XX = TWOPI*XDATA(I)*ABS(FREQ(J))
               IF(PFIXED) THEN
                  COSINE(I,J) = COS(XX-PHASE)
                  SINE(I,J)   = 0.
               ELSE
                  COSINE(I,J) = COS(XX)
                  SINE(I,J)   = SIN(XX)
               END IF
            END DO
         END DO
      END IF
      NP = 0
      IF(PFIXED) THEN
         DO I = 1, NUM
            IF(YERR(I) .GT. 0.D0) THEN
               NP = NP + 1
               WW = 1./(YERR(I)*YERR(I))
               YSQ = YSQ + WW*YDATA(I)*YDATA(I)
               DO J = 1, NFREQ
                  CNJ = COSINE(I,J)
C     
C     Accumulate sums
C     
                  V(J) = V(J) + WW*CNJ*YDATA(I)
                  DO K = J, NFREQ
                     CNK = COSINE(I,K)
                     M(K,J) = M(K,J) + WW*CNK*CNJ
                  END DO
               END DO
            END IF
         END DO
         IF(NP.LT.2) THEN
            WRITE(*,*) '** <2 VALID POINTS FOR A 2 PARAMETER FIT'
            IFAIL = 1
            RETURN
         END IF
         DO J = 2, NFREQ
            DO I = 1, J-1
               M(I,J) = M(J,I)
            END DO
         END DO
      ELSE
         DO I = 1, NUM
            IF(YERR(I) .GT. 0.D0) THEN
               NP = NP + 1
               WW = 1./(YERR(I)*YERR(I))
               YSQ = YSQ + WW*YDATA(I)*YDATA(I)
               DO J = 1, NFREQ
                  CNJ = COSINE(I,J)
                  SNJ = SINE(I,J)
C     
C     Accumulate sums
C     
                  J1 = 2*J-1
                  J2 = J1+1
                  V(J1) = V(J1) + WW*CNJ*YDATA(I)
                  V(J2) = V(J2) + WW*SNJ*YDATA(I)
                  DO K = J, NFREQ
                     CNK = COSINE(I,K)
                     SNK = SINE(I,K)
                     K1 = 2*K-1
                     K2 = K1 + 1
                     M(K1,J1) = M(K1,J1) + WW*CNK*CNJ
                     M(K2,J1) = M(K2,J1) + WW*SNK*CNJ
                     M(K2,J2) = M(K2,J2) + WW*SNK*SNJ
                  END DO
               END DO
            END IF
         END DO
         IF(NP.LT.2) THEN
            WRITE(*,*) '** <2 VALID POINTS FOR A 2 PARAMETER FIT'
            IFAIL = 1
            RETURN
         END IF
         DO J = 2, 2*NFREQ
            DO I = 1, J-1
               M(I,J) = M(J,I)
            END DO
         END DO
      END IF
C     
C     Solve simultaneous equations to determine fit coefficients
C     
      IF(PFIXED) THEN
         CALL PDA_DGEFS(M,2*MAXFREQ+1,NFREQ,V,1,IND,DWK,IWK,IFAIL)
      ELSE
         CALL PDA_DGEFS(M,2*MAXFREQ+1,2*NFREQ,V,1,IND,DWK,IWK,IFAIL)
      END IF
      IF(IFAIL.NE.0) THEN
         WRITE(*,*) 'PDA_DGEFS. IFAIL = ',IFAIL
         WRITE(*,*) 'Failed to fit sinusoids'
         DO I = 1, NFREQ
            A(I)   =  0.
            B(I)   =  0.
            EA(I)  = -1.
            EB(I)  = -1.
            RAB(I) = -1.
         END DO
      ELSE
C     
C     Store fit coefficients
C     
         IF(PFIXED) THEN
            DO I = 1, NFREQ
               A(I) = REAL(V(I))
            END DO
C     
C     Separately invert matrix to determine covariance matrix
C     
            CALL PDA_DGEFA( M, 2*MAXFREQ+1, NFREQ, IWK, IFAIL)
            IF(IFAIL.NE.0) THEN
               WRITE(*,*) 'PDA_DGEFA. IFAIL = ',IFAIL
               WRITE(*,*) 'Matrix prob singular'
               DO I = 1, NFREQ
                  EA(I)  = 1.E4
               END DO
            ELSE
               CALL PDA_DGEDI( M, 2*MAXFREQ+1, NFREQ, IWK, DETERM, 
     &              DWK, 1)
C     
C     Store errors on A 
C     
               DO I = 1, NFREQ
                  EA(I)  = REAL(SQRT(M(I,I)))
               END DO
            END IF

C     
C     Calculate chi-squared
C     

            F = YSQ
            DO J = 1, NFREQ 
               DO I = 1, J
                  IF(I.EQ.J) THEN
                     F = F + V(I)*M(I,J)*V(J)
                  ELSE
                     F = F + 2.*V(I)*M(I,J)*V(J)
                  END IF
               END DO
            END DO

         ELSE
            DO I = 1, NFREQ
               A(I) = REAL(V(2*I-1))
               B(I) = REAL(V(2*I))
            END DO
C     
C     Separately invert matrix to determine covariance matrix
C     
            CALL PDA_DGEFA( M, 2*MAXFREQ+1, 2*NFREQ, IWK, IFAIL)
            IF(IFAIL.NE.0) THEN
               WRITE(*,*) 'PDA_DGEFA. IFAIL = ',IFAIL
               WRITE(*,*) 'Matrix prob singular'
               DO I = 1, NFREQ
                  EA(I)  = 1.E4
                  EB(I)  = 1.E4
                  RAB(I) = 0.
               END DO
            ELSE
               CALL PDA_DGEDI( M, 2*MAXFREQ+1, 2*NFREQ, IWK, DETERM, 
     &              DWK, 1)
C     
C     Store errors on A and B and correlation coefficients
C     
               DO I = 1, NFREQ
                  I1 = 2*I-1
                  I2 = 2*I
                  EA(I)  = REAL(SQRT(M(I1,I1)))
                  EB(I)  = REAL(SQRT(M(I2,I2)))
                  RAB(I) = REAL(M(I2,I1)/SQRT(M(I2,I2)*M(I1,I1)))
               END DO
            END IF
C     
C     Calculate chi-squared
C     
            F = YSQ
            DO J = 1, 2*NFREQ 
               DO I = 1, J
                  IF(I.EQ.J) THEN
                     F = F + V(I)*M(I,J)*V(J)
                  ELSE
                     F = F + 2.*V(I)*M(I,J)*V(J)
                  END IF
               END DO
            END DO
         END IF
      END IF
      RETURN
      END
