*MAG
* MAG N1 N2 COMP FILT OUT -- Computes the relative magnitude of a list of
*                        stars compared to a template star by integration
*                        over a bandpass provided by the user.
*
* Parameters:
*            N1   -- First object spectrum
*            N2   -- Last object spectrum
*            COMP -- Star to compare with
*            FILT -- Table of data with columns of wavelength and response.
*                    The wavelengths (angstroms) should monotonically
*                    increase but do not have to be uniform. Response (QE)
*                    should go to zero at ends. 
*            OUT  -- <CR> for terminal only
*                    name -- for a file name
*                    !name to store in real header with name = name
*
* Details: the magnitude is photon weighted. i.e. it is assumed that you
* would measure it with a detector that reponds to the number of incdent
* photons. This ends up as an integral over filt*fnu*dlambda/lambda.
*
* The table will be linearly interpolated between values, but it is taken to be
* constant outside the range of the wavelength. Therefore the response should 
* go to zero at either end if you want a finite bandwidth. The integration will 
* not extend beyond the ends of the spectrum. Masked regions are ignored, i.e.
* they are effectively set to zero.
*
* The output is the magnitude of the target spectra - magnitude of COMP
*MAG
      SUBROUTINE RELMAG(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, FLUX, 
     &MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, 
     &NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS, MXSLOTS, 
     &WORK1, WORK2, MXWORK, CLOBBER, IFAIL)
*
* Compute relative magnitude of spectra.
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* =  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* =  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* =  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* >  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
*    I     SLOTS(MXSLOTS)     -- Workspace for list of slots
* <  I     IFAIL            -- Error return.
*
      IMPLICIT NONE
*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX
      INTEGER MXCHAR, MXDOUB, MXREAL, MXINTR, MXARC
      INTEGER NPIX(MXSPEC), NARC(MXSPEC)
      LOGICAL CLOBBER
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Slot lists
*
      INTEGER MXSLOTS, NSLOTS, SLOT
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
*
* Work arrays
*
      INTEGER MXWORK
      REAL WORK1(MXWORK), WORK2(MXWORK)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Local variables
*
      INTEGER I, J,   JFIND, L, NFILT
      REAL GET_FLX, W, E, FILT, RMAG
      INTEGER  NCOM
      DOUBLE PRECISION ANGST, DISPA, SUM, CSUM
      DOUBLE PRECISION LAM, DLAM
      REAL Z, X
      CHARACTER*64 FILTER, NAME
      CHARACTER*100 STRING
      INTEGER SLOT1, SLOT2, SLOT3,   IFAIL
      INTEGER  MAXSPEC, OP
      LOGICAL  SAME, DEFAULT
*
* Functions
*
      INTEGER LENSTR, ISTSTR
      DATA SLOT1, SLOT2, SLOT3/1,1,2/
      DATA FILTER/'filter'/
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      NCOM    = 0
      IFAIL   = 0
*
      CALL INTR_IN('First slot to measure', SPLIT, 
     &NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to measure', SPLIT, NSPLIT, 
     &MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
        WRITE(*,*) 'Only 0,0 to get list option'
        GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
        WRITE(*,*) 'Enter list of spectra to measure magnitudes'
        CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &  NSLOTS, MXSLOTS)
        IF(NSLOTS.EQ.0) GOTO 999
      ELSE
        CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &  NSLOTS, MXSLOTS)
      END IF
*
      CALL INTR_IN('Comparison spectrum', SPLIT, NSPLIT, 
     &MXSPLIT, NCOM, DEFAULT, SLOT3, 1, MAXSPEC, IFAIL)
      IF(NPIX(SLOT3).LT.1) THEN
        WRITE(*,*) SLOT3,' is empty.'
        GOTO 999
      END IF
      CALL CHAR_IN('Filter file', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, FILTER, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      OPEN(UNIT=21,FILE=FILTER,STATUS='OLD',IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN 
        WRITE(*,*) 'Failed to open ',FILTER(:LENSTR(FILTER))
        GOTO 999
      END IF
      IFAIL = 0
      NFILT = 0
      DO WHILE(IFAIL.EQ.0 .AND. NFILT.LT.MXWORK)
        READ(21,'(A)',IOSTAT=IFAIL) STRING
        IF(IFAIL.EQ.0 .AND. STRING(1:1).NE.'#') THEN
          READ(STRING,*,IOSTAT=IFAIL) W, E
          IF(IFAIL.EQ.0) THEN
            NFILT = NFILT + 1
            WORK1(NFILT) = W
            WORK2(NFILT) = E
          END IF
        END IF
      END DO
      CLOSE(UNIT=21)
      IF(IFAIL.GT.0) THEN
        WRITE(*,*) 'Error while reading ',FILTER(:LENSTR(FILTER))
        GOTO 999
      ELSE
        IFAIL = 0
        WRITE(*,*) 'Read ',NFILT,' points.'
        IF(NFILT.EQ.MXWORK) THEN
          WRITE(*,*) 'May have been limited by workspace'
        END IF
      END IF
      NAME = ' '
      CALL CHAR_IN('O/P? <CR>=term, xxx=file, !xxx=headers', 
     &SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, NAME, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(NAME.EQ.' ') THEN
        OP = 1
      ELSE IF(SAME('!',NAME)) THEN
        OP = 2
        L = INDEX(NAME,'!')
        L = L + ISTSTR(NAME(L+1:))
      ELSE 
        OP = 3
        IF(CLOBBER) THEN
          OPEN(UNIT=21,FILE=NAME,STATUS='UNKNOWN',IOSTAT=IFAIL)
        ELSE
          OPEN(UNIT=21,FILE=NAME,STATUS='NEW',IOSTAT=IFAIL)
        END IF
        IF(IFAIL.NE.0) THEN
          WRITE(*,*) 'Could not open output file'
          GOTO 999
        END IF
        WRITE(21,*,ERR=998) '#'
        WRITE(21,*,ERR=998) '# Relative magnitude output from MAG'
        WRITE(21,*,ERR=998) '#'
      END IF
*
* Integrate over comparison spectrum
*
      CSUM = 0.D0
      JFIND = (1+NFILT)/2
      DO I = 1, NPIX(SLOT3)
        Z = REAL(I)
        DLAM = DISPA(Z, NPIX(SLOT3), NARC(SLOT3), ARC(1,SLOT3), 
     &      MXARC)
        LAM  = ANGST(Z, NPIX(SLOT3), NARC(SLOT3), ARC(1,SLOT3), 
     &      MXARC)
        IF(ERRORS(I,SLOT3).GT.0.) THEN
          X = REAL(LAM)
          CALL HUNT(WORK1,NFILT,X,JFIND)
          IF(JFIND.EQ.0) THEN
            FILT = WORK2(1)
          ELSE IF(JFIND.EQ.NFILT) THEN
            FILT = WORK2(NFILT)
          ELSE
            FILT = ((X-WORK1(JFIND  ))*WORK2(JFIND+1)+
     &              (WORK1(JFIND+1)-X)*WORK2(JFIND  ))/
     &              (WORK1(JFIND+1)-WORK1(JFIND))
          END IF
          CSUM = CSUM + FILT*LAM*DLAM*
     &    GET_FLX(COUNTS(I,SLOT3), FLUX(I,SLOT3))
        END IF
      END DO
      IF(CSUM.LE.0) THEN
        WRITE(*,*) 'Obtained non-positive counts in comparison ',CSUM
        GOTO 999
      END IF
*
* Now do same over spectra
*
      DO I = 1, NSLOTS
        SLOT = SLOTS(I)
        IF(NPIX(SLOT).LT.1) THEN
          WRITE(*,*) 'Spectrum ',slot,' empty.'
        ELSE
          SUM  = 0.D0
          JFIND = (1+NFILT)/2
          DO J = 1, NPIX(SLOT)
            Z = REAL(J)
            DLAM = DISPA(Z, NPIX(SLOT), NARC(SLOT), 
     &      ARC(1,SLOT), MXARC)
            LAM  = ANGST(Z, NPIX(SLOT), NARC(SLOT), 
     &      ARC(1,SLOT), MXARC)
            IF(ERRORS(J,SLOT).GT.0.) THEN
              X = REAL(LAM)
              CALL HUNT(WORK1,NFILT,X,JFIND)
              IF(JFIND.EQ.0) THEN
                FILT = WORK2(1)
              ELSE IF(JFIND.EQ.NFILT) THEN
                FILT = WORK2(NFILT)
              ELSE
                FILT = ((X-WORK1(JFIND  ))*WORK2(JFIND+1)+
     &                 (WORK1(JFIND+1)-X)*WORK2(JFIND  ))/
     &                 (WORK1(JFIND+1)-WORK1(JFIND))
              END IF
              SUM = SUM + FILT*LAM*DLAM*
     &        GET_FLX(COUNTS(J,SLOT), FLUX(J,SLOT))
            END IF
          END DO
          IF(SUM.LE.0.) THEN
            WRITE(*,*) 'Non-positive counts in slot ',SLOT
            WRITE(*,*) 'Rel mag set to 100'
            RMAG = 100.
          ELSE
            RMAG = REAL(2.5*(LOG10(CSUM)-LOG10(SUM)))
            WRITE(*,*) 'Mag ',SLOT,'- ',SLOT3,'=',RMAG
          END IF
          IF(OP.EQ.2) THEN
            CALL HSETR(NAME(L:), RMAG, SLOT, MXSPEC, NMREAL, 
     &      HDREAL, NREAL, MXREAL, IFAIL)
            IF(IFAIL.NE.0) THEN
              WRITE(*,*) 'Failed to store value in '//
     &        'header of slot ',SLOT
              GOTO 999
            END IF
          ELSE IF(OP.EQ.3) THEN
            WRITE(21,*,ERR=998) RMAG
          END IF
        END IF
      END DO
      CLOSE(UNIT=21)
      IFAIL = 0
      RETURN
*
998   WRITE(*,*) 'Error writing to output file'
999   CLOSE(UNIT=21)
      RETURN
      END

