      SUBROUTINE POLMOL( NDATA, XDATA, YDATA, YSIGMA, YFIT,
     &NPOLY, NCYCLE, THRHI, THRLO, WORK1, WORK2, WORK3, 
     &MAXWORK, IFAIL )
*
* Adapted from POLYFITA 
* Fits polynomial to data and performs reject cycles with
* no user interaction.
*
*  Inputs:
*      NDATA      = NUMBER OF DATA VALUES
*      XDATA      = X DATA VALUES
*      YDATA      = Y DATA VALUES
*      YSIGMA     = Y UNCERTAINTIES (1-SIGMA) (NEGATIVE TO REJECT)
*      NPOLY      = Number of terms in polynomial (0-maxpoly)
*      NCYCLE     = Max number of reject cycles
*      THRHI      = high threshold for sigma clipping
*      THRLO      = low threshold for sigma clipping
*      WORK1(MAXWORK) R*8 work array
*      WORK2(MAXWORK) R*8 work array
*      WORK3(MAXWORK) R*8 work array
*      MAXWORK     Size of work array which should be able to take all fit
*      IFAIL = -1 to suppress messages
*  Output:
*      YFIT       = Y FIT VALUES
*      IFAIL      = 0 IF SUCCESSFUL, 1 IF FAILED.
*
* Oct 1986 KDH @ STScI
* 29/01/87 TRM @ RGO eliminated statement label
*
      IMPLICIT NONE
      INTEGER NDATA, NPOLY, NCYCLE, IFAIL, LASTREJ, NREJ
      INTEGER ICYCLE, MAXWORK, i
      REAL THRHI, THRLO, RMS
      REAL XDATA(NDATA), YDATA(NDATA), YSIGMA(NDATA), YFIT(NDATA)
      DOUBLE PRECISION WORK1(MAXWORK), WORK2(MAXWORK), WORK3(MAXWORK)
      LOGICAL LOUD
*
* Test inputs
*
      LOUD = .NOT.IFAIL.EQ.-1
      IF(NDATA.LE.0) THEN
        WRITE(*,*) '** No data in POLMOL ', NDATA
        GOTO 999
      END IF
      LASTREJ = 1
      NREJ = 0      
      ICYCLE = -1   
      DO WHILE(ICYCLE.LT.NCYCLE .AND. NREJ.NE.LASTREJ)
        ICYCLE = ICYCLE + 1
        LASTREJ = NREJ      
*
* Fit the polynomial
*
        CALL POLMOL1( NDATA, XDATA, YDATA, YSIGMA, YFIT, 
     &  NPOLY, WORK1, WORK2, WORK3, MAXWORK, IFAIL)
        IF( IFAIL.NE.0 ) GOTO 999
*
        NREJ = -1
        IF( NCYCLE.EQ.0 ) THEN
*
* Evaluate the fit
* Do not restore any points
*
          CALL REJECT( NDATA,YDATA,YSIGMA,YFIT,1.E20,-1.E20,RMS,NREJ)
        ELSE
*
* Evaluate the fit and reject large outliers
*
          CALL REJECT(NDATA,YDATA,YSIGMA,YFIT,THRHI,THRLO,RMS,NREJ)
        END IF
*
* Report results of this reject
*
        IF(LOUD) WRITE(*,*) ' Cycle', ICYCLE, '  RMS =', RMS,
     &      ' (sigma)  Rejects =', NREJ
      END DO
*
* normal return
*
      IFAIL = 0
      RETURN
*
* error return
*
999   IFAIL = 1
      WRITE(*,*) 'POLMOL aborted.'
      RETURN

      END


      SUBROUTINE POLMOL1( NDATA, XDATA, YDATA, YSIGMA, 
     *            YFIT, NPOLY, W1, W2, W3, MAXFIT, IFAIL )
*
*  COMPUTES WEIGHTED LEAST-SQUARES POLY FIT TO DATA PAIRS
*
*  Input:
*      NDATA      = NUMBER OF DATA PAIRS
*      XDATA      = DATA X VALUES
*      YDATA      = DATA Y VALUES
*      YSIGMA     = UNCERTAINTY IN Y (1-SIGMA) (NEGATIVE TO REJECT)
*      NPOLY      = NUMBER OF POLY TERMS REQUESTED
*  Output:
*      YFIT       = FITTED Y VALUES
*      RMS        = RMS NORMALIZED RESIDUAL OF POINTS NOT REJECTED
*      IFAIL      = 0 IF SUCCESSFUL, 1 IF FAILED.
*
*      USES LEASQ
*
      IMPLICIT NONE
      INTEGER NDATA, MAXFIT, NPOLY, MAXPOLY
      REAL XDATA(NDATA), YDATA(NDATA), YSIGMA(NDATA), YFIT(NDATA)
      REAL RMS
      DOUBLE PRECISION W1(MAXFIT), W2(MAXFIT), W3(MAXFIT)
      PARAMETER (MAXPOLY = 20)
      DOUBLE PRECISION PFT(MAXPOLY),XM(MAXPOLY,2*MAXPOLY+3)
      DOUBLE PRECISION CHISQ, CALC, POLY, XSTART, XEND
      INTEGER   NFIT, IGET, IPUT, NPFIT, I
      INTEGER  IFAIL
      COMMON/PLYFIT0/NPFIT
      COMMON/PLYFIT1/XSTART,XEND,PFT
*
*  Check input data
*
      IF(NDATA.LE.0) THEN
        WRITE(*,*) '** No data.', NDATA
        GOTO 999
      END IF
*
* Error return if NPOLY > NFIT
*
      NFIT=0
      DO I=1,NDATA
        IF(YSIGMA(I).GT.0.) NFIT = NFIT + 1
      END DO
      IF( NPOLY.GT.NFIT) THEN
        WRITE(*,*) '** INSUFFICIENT DATA FOR POLYFIT. NPOLY=', 
     *            NPOLY, 'NFIT=', NFIT
        GOTO 999
      END IF
*
*  Load data arrays for poly fit ---------------------------------------
*
      IPUT = 0
      DO IGET= 1,NDATA
        IF( YSIGMA(IGET).GT.0.) THEN  
          IPUT = IPUT + 1
          W1(IPUT) = XDATA(IGET)
          W2(IPUT) = YDATA(IGET)
          W3(IPUT) = YSIGMA(IGET)
        END IF
      END DO
      NFIT = IPUT
*
*  Re-scale weights to their RMS value
*
      CALC = 0.D0
      DO I = 1,NFIT
        CALC = CALC + W3(I)*W3(I)
      END DO
      RMS = REAL(SQRT(CALC/NFIT))
      DO I=1,NFIT
        W3(I) = W3(I)/RMS
      END DO
*
*  Call LEASQ routine to compute poly fit --------------------------------
*
      CALL LEASQ(W1, W2, W3, NFIT,NPOLY,PFT,CHISQ,XM,1)
*
*  Evaluate poly at required points
*
      DO I=1,NDATA
        YFIT(I) = SNGL( POLY( PFT, NPOLY, DBLE(XDATA(I)) ))
      END DO
      NPFIT  = NPOLY
      XSTART = W1(1)
      XEND   = XSTART
      DO I=1,NFIT
        XSTART = MIN(XSTART,W1(I))
        XEND   = MAX(XEND,  W1(I))
      END DO
*
*  Normal return
*
      IFAIL = 0
      RETURN
*
*  Error return
*
999   WRITE(*,*) '** POLMOL1 aborted.'
      IFAIL = 1
      RETURN
*
      END

