*BBODY
* BBODY N [W1 W2 NBIN] TEMP NORM [WAVE MJY] or [ANG] -- 
*                Generates a black-body spectrum
*
* Parameters
*       N     -- Slot which will contain spectrum
*
*       If empty then you also need to specify
*
*         W1, W2 -- Wavelength range
*         NBIN  -- Number of bins
*
*       otherwise the old wavelength scale is preserved.
*
*       TEMP  -- Temperature of black-body
*       NORM  -- Method of normalisation
*                1 -- Specify a wavelength and a mJy flux at that wavelength
*                2 -- Specify an angular diameter in units defined so the
*                     Sun at 100pc = 1
*       If NORM= 1 need
*          WAVE -- Wavelength
*          MJY  -- mJy flux at WAVE
*
*          NORM=2
*          ANG  -- Angular diameter as described above.
*
* The errors are set equal to 1/10000 of the flux, with a lower limit of
* 1.E-15
*BBODY
      SUBROUTINE PLANCK(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC,
     &NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &MXREAL, IFAIL)
*
* Generate black-body spectrum
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
* >  I     IFAIL     -- O if read ok
*
      IMPLICIT NONE
*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Local variables
*
      INTEGER SLOT, IFAIL,  NBIN, NORM
      INTEGER I,  MAXSPEC
      INTEGER  NP, NCOM
      REAL W, F, E, WAVE, MJY, TEMP, W1, W2, ANG, SCALE
      REAL C
      LOGICAL EMPTY, DEFAULT
      DOUBLE PRECISION ANGST
*
* Functions
*
      REAL PLFUN
*
      DATA SLOT, W1, W2, NBIN/1,3000., 6000., 1000/
      DATA NORM, ANG, WAVE, MJY, TEMP/1, 1.,5000.,1.,10000./
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL INTR_IN('Slot for black-body', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, SLOT, 1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(NPIX(SLOT).LE.0) THEN
        EMPTY = .TRUE.
        CALL REAL_IN('First wavelength', SPLIT, NSPLIT, MXSPLIT, 
     &  NCOM, DEFAULT, W1, 0., 1.E10, IFAIL)
        CALL REAL_IN('Last wavelength', SPLIT, NSPLIT, MXSPLIT, 
     &  NCOM, DEFAULT, W2, 0., 1.E10, IFAIL)
        CALL INTR_IN('Number of bins', SPLIT, NSPLIT, MXSPLIT, 
     &  NCOM, DEFAULT, NBIN, 1, MAXPX, IFAIL)
        NP = NBIN
      ELSE
        EMPTY = .FALSE.
        NP = NPIX(SLOT)
      END IF
      CALL REAL_IN('Temperature', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, TEMP, 1., 1.E10, IFAIL)
      CALL INTR_IN('Normalisation method (1 or 2)', SPLIT, 
     &NSPLIT, MXSPLIT, NCOM, DEFAULT, NORM, 1, 2, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(NORM.EQ.1) THEN
        CALL REAL_IN('Reference wavelength', SPLIT, NSPLIT, 
     &  MXSPLIT, NCOM, DEFAULT, WAVE, 0., 1.E10, IFAIL)
        CALL REAL_IN('Flux at this wavelength', SPLIT, NSPLIT, 
     &  MXSPLIT, NCOM, DEFAULT, MJY, 1.E-10, 1.E10, IFAIL)
      ELSE IF(NORM.EQ.2) THEN
        CALL REAL_IN('Angular diameter (Sun at 100pc = 1)',
     &  SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, ANG, 1.E-20, 
     &  1.E20, IFAIL)
      END IF
      IF(IFAIL.NE.0) GOTO 999
*
      IF(NORM.EQ.1) THEN
        SCALE = LOG(ABS(MJY))-PLFUN(WAVE,TEMP)
      ELSE  
        SCALE = 16.6143+2.*LOG(ANG)
      END IF
      DO I = 1, NP
        IF(EMPTY) THEN
          W = W1 + (W2-W1)*REAL(I-1)/REAL(MAX(1,NP-1))
        ELSE
          W = REAL(ANGST(REAL(I), NPIX(SLOT), NARC(SLOT), 
     &        ARC(1,SLOT), MXARC))
        END IF
        F = EXP(SCALE+PLFUN(W,TEMP))
        C = F
        E = MAX(1.E-15, F/10000.)
        COUNTS(I,SLOT) = C
        ERRORS(I,SLOT) = E
        IF(C.EQ.0.) THEN
          FLUX(I,SLOT) = 1.
        ELSE
          FLUX(I,SLOT) = C
        END IF        
      END DO
      CALL HSETC('Object', 'Black-body', SLOT, MXSPEC, 
     &NMCHAR, HDCHAR, NCHAR, MXCHAR, IFAIL)
      CALL HSETR('Temperature', TEMP, SLOT, MXSPEC, 
     &NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
      IF(EMPTY) THEN
        NPIX(SLOT) = NBIN
        NARC(SLOT) = 2
        ARC(1,SLOT) = W1 - (W2-W1)/REAL(NBIN-1)
        ARC(2,SLOT) = REAL(NBIN)*(W2-W1)/REAL(NBIN-1)
      END IF
      WRITE(*,*) 'Set slot ',SLOT,' to a black-body spectrum.'
      RETURN
999   IFAIL = 1
      RETURN
      END

      REAL FUNCTION PLFUN(WAVE,TEMP)
*
* Given temperature, wavelength, PLFUN is the natural 
* log of the specific intensity in 
* ergs/s/cm**2/steradian/hertz
* WAVE in Angstroms, TEMP in Kelvin. Angle integrated version is
* Pi times larger than this.
*
      REAL WAVE,TEMP
      DATA EFAC,PFAC/1.4387E8,19.8009/
C
      TEST=EFAC/TEMP/WAVE
      PLFUN = PFAC-3.*LOG(WAVE)
      IF(TEST.GT.20) THEN
        PLFUN = PLFUN - TEST
        RETURN
      ELSE IF(TEST.LT.1.E-5) THEN
        PLFUN = PLFUN - LOG(TEST)
      ELSE
        PLFUN = PLFUN - LOG( EXP(TEST)-1.)
      END IF
      RETURN
      END
