CPGRAM
C PGRAM N1 N2 N3 F1 F2 NF NPOLY SET -- Computes Scargle's periodogram 
C                                      at every wavelength
C
C Results are stored as spectra which can be displayed in standard fashion.
C
C Parameters: 
C     
C  N1    -- Slot number of first spectrum for periodogram
C  N2    -- Slot number of last spectrum for periodogram
C  N3    -- First slot in which to store results
C  F1    -- First frequency (cycles/day)
C  F2    -- Last frequency (cycles/day)
C  NF    -- Number of frequencies
C  NPOLY -- Order of polynomial to subtract off prior to computation of 
C           periodogram. Removes leakage from DC to higher frequiencies
C           which can otherwise swamp signal.
C  SET   -- Yes to mask regions of no interest
CPGRAM
      SUBROUTINE PERGM(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &     FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, 
     &     NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &     HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &     MXDOUB, MXINTR, MXREAL, NMSCHAR, VSCHAR, NMSDOUB, 
     &     VSDOUB, NMSINTR, VSINTR, NMSREAL, VSREAL, NSCHAR, 
     &     NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, MXSINTR, 
     &     MXSREAL, WORK1, WORK2, WORK3, MASK, DWORK1, DWORK2, 
     &     DWORK3, DWORK4, DWORK5, DWORK6, WHICH, MXWORK, SLOTS, 
     &     MXSLOTS, CLOSE, IFAIL)
C     
C     
C     Computes Scargle's periodogram
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     >  C*(*) NMSCHAR(MXSCHAR)   -- Names of character selection items.
C     >  C*(*) VSCHAR(MXSCHAR)    -- Values of character selection items.
C     >  C*(*) NMSDOUB(MXSDOUB)   -- Names of double precision selection items.
C     >  D     VSDOUB(2,MXSDOUB)  -- Values of double precision selection items.
C     >  C*(*) NMSINTR(MXSINTR)   -- Names of integer selection items.
C     >  I     VSINTR(2,MXSINTR)  -- Values of integer selection items.
C     >  C*(*) NMSREAL(MXSREAL)   -- Names of real selection items.
C     >  R     VSREAL(2,MXSREAL)  -- Values of real selection items.
C     >  I     NSCHAR             -- Number of character selection items.
C     >  I     NSDOUB             -- Number of double precsion selection items.
C     >  I     NSINTR             -- Number of integer selection items.
C     >  I     NSREAL             -- Number of real selection items.
C     >  I     MXSCHAR            -- Dimension of NMSCHAR in calling routine.
C     >  I     MXSDOUB            -- Dimension of NMSDOUB in calling routine.
C     >  I     MXSINTR            -- Dimension of NMSINTR in calling routine.
C     >  I     MXSREAL            -- Dimension of NMSREAL in calling routine.   
C     >  L     WAVE      -- TRUE for uniform wavelength scale (else velocity)
C     >  D     DWORK1  Work array
C     >  D     DWORK2  Work array
C     >  D     DWORK3  Work array
C     >  D     DWORK4  Work array
C     >  D     DWORK5  Work array
C     >  D     DWORK6  Work array
C     >  I     WHICH      "
C     >  I     MXWORK     Size of work array
C     I     SLOTS(MXSLOTS) -- Work array for list of slots
C     I     MXSLOTS
C     <  I     IFAIL              -- Error return.
C     
      IMPLICIT NONE

C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL   FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Search parameters
C     
      INTEGER MXSCHAR, MXSDOUB, MXSINTR
      INTEGER MXSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL
C     
C     Search parameters, names then values.
C     
      CHARACTER*(*) NMSCHAR(MXSCHAR)
      CHARACTER*(*) NMSDOUB(MXSDOUB)
      CHARACTER*(*) NMSINTR(MXSINTR)
      CHARACTER*(*) NMSREAL(MXSREAL)
C     
      CHARACTER*(*) VSCHAR(MXSCHAR)
      DOUBLE PRECISION VSDOUB(2,MXSDOUB)
      INTEGER VSINTR(2,MXSINTR)
      REAL VSREAL(2,MXSREAL)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
C     Mask arrays
C     
      INTEGER MXMASK
      PARAMETER (MXMASK=50)
      INTEGER NPMASK, NWMASK, NVMASK
      INTEGER PMASK(2,MXMASK)
      REAL WMASK(2,MXMASK)
      REAL VMASK(3,MXMASK)
C     
      LOGICAL CLOSE
      INTEGER MXWORK
      DOUBLE PRECISION DWORK1(MXWORK), DWORK2(MXWORK)
      DOUBLE PRECISION DWORK3(MXWORK), DWORK4(MXWORK)
      DOUBLE PRECISION DWORK5(MXWORK), DWORK6(MXWORK)
      REAL WORK1(MXWORK), WORK2(MXWORK), WORK3(MXWORK)
      LOGICAL MASK(MXWORK)
      INTEGER WHICH(MXWORK)
C     
C     Slot lists
C     
      INTEGER MXSLOTS, NSLOTS, SLOT, NS
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
C     
C     Local variables
C     
      INTEGER I, J, K, NV, LENSTR
      REAL GET_FLX, GET_ERF, CFRAT, SCARG, FREQ, F, RATIO
      DOUBLE PRECISION POLY,  SUB, VEARTH
      DOUBLE PRECISION AVE, LOW, HIGH, RANGE
      INTEGER MXPOL, NPOLY
      PARAMETER (MXPOL=15)
      DOUBLE PRECISION XM(MXPOL,2*MXPOL+3), CHISQ, A(MXPOL)
      CHARACTER*3 SET
      CHARACTER*100 STRING
      INTEGER SLOT1, SLOT2, SLOT3, NF, MX
      INTEGER  IFAIL, NSAME, NOK
      INTEGER  MAXSPEC,  NCOM
      LOGICAL SELECT, DEFAULT, RESET
      REAL F1, F2, E
C     
C     Functions
C     
      LOGICAL SAME
C     
      DATA SLOT1, SLOT2, SLOT3/1,3,4/
      DATA F1, F2, NF/0., 100., 100/
      DATA NPOLY/2/
      DATA SET/'Y'/
      DATA NPMASK, NWMASK, NVMASK/0,0,0/
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
C     
C     First slot
C     
      CALL INTR_IN('First spectrum for periodogram', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last spectrum for periodogram', SPLIT, 
     &     NSPLIT, MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST.GT.0) THEN
            CALL SETSLOT(LIST, NLIST, MXLIST, SLOTS, NSLOTS, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra for periodogram'
            CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &           NSLOTS, MXSLOTS)
         END IF
         IF(NSLOTS.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &        NSLOTS, MXSLOTS)
      END IF
C     
C     Check through for valid spectra. Store values for ease
C     of reference later
C     
      NSAME = 0
      NV    = 0
      DO I = 1, NSLOTS
         SLOT = SLOTS(I)
         IF( NPIX(SLOT).GT.0 .AND. NPIX(SLOT).LE.MXWORK .AND.
     &        SELECT(SLOT,NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &        HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &        MXCHAR, MXDOUB, MXINTR, MXREAL, MXSPEC, NMSCHAR, 
     &        VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, 
     &        VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, 
     &        MXSDOUB, MXSINTR, MXSREAL)) THEN
            IF(NSAME.EQ.0) THEN
               NSAME = NPIX(SLOT)
               NV = 1
               WHICH(NV) = SLOT
            ELSE IF(NPIX(SLOT).NE.NSAME) THEN
               WRITE(*,*) 'Slot ',SLOT,' has ',NPIX(SLOT),' pixels'
               WRITE(*,*) 'rather than ',NSAME,' as the first valid'
               WRITE(*,*) 'spectrum. Choose again.'
               GOTO 999
            ELSE
               NV = NV + 1
               WHICH(NV) = SLOT
            END IF
         END IF
      END DO
      IF(NV.LT.3) THEN
         WRITE(*,*) 'Need at least 3 valid spectra to fit'
         GOTO 999
      ELSE IF(NV.GT.MXWORK) THEN
         WRITE(*,*) 'Too many spectra for MXWORK = ',MXWORK
         GOTO 999
      END IF
C     
      CALL INTR_IN('First output slot', SPLIT, NSPLIT, MXSPLIT, 
     &     NCOM, DEFAULT, SLOT3, 1, MAXSPEC, IFAIL)
      CALL REAL_IN('First frequency (cyc/day)', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, F1, 0., 1.E6, IFAIL)
      F2 = MAX(F1, F2)
      CALL REAL_IN('Last frequency (cyc/day)', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, F2, F1, 1.E6, IFAIL)
C     
      MX = MXSPEC-SLOT3+1
      CALL INTR_IN('Number of frequencies', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NF, 1, MX, IFAIL)
      CALL INTR_IN('Order of poly to subtract', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NPOLY, 1, MXPOL, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(NPOLY.GE.NV) THEN
         WRITE(*,*) 'Too few spectra for poly order = ',NV
         GOTO 999
      END IF
      CALL CHAR_IN('Set mask?', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &     DEFAULT, SET, IFAIL)
      CALL UPPER_CASE(SET)
      IF(SAME(SET,'YES')) THEN
         WRITE(*,*) 'Mask regions not wanted for periodogram.'
         CALL MASKIT(PMASK, WMASK, VMASK, NPMASK, NWMASK, 
     &        NVMASK, MXMASK, COUNTS, ERRORS, FLUX, MXBUFF, 
     &        MAXPX, NPIX, MXSPEC, NARC, ARC, MXARC, NMCHAR, 
     &        NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &        HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &        MXDOUB, MXINTR, MXREAL, WORK1, WORK2, WORK3, 
     &        MXWORK, MASK)
      END IF
C     
C     Get mask for first spectrum. Assumed same for all.
C     
      SLOT = WHICH(1)
      IF(NARC(SLOT).NE.0) THEN
C     
C     Search for double precision header item Vearth. If not
C     found, assume = 0
C     
         CALL HGETD('Vearth', VEARTH, SLOT, MXSPEC, 
     &        NMDOUB, HDDOUB, NDOUB, MXDOUB, IFAIL)
         IFAIL = 0
      END IF
C     
C     Get mask
C     
      CALL APPMASK(MASK,NPIX(SLOT),ARC(1,SLOT),NARC(SLOT),
     &     MXARC,PMASK,NPMASK,WMASK,NWMASK,VMASK,NVMASK,MXMASK,
     &     .FALSE.,VEARTH,IFAIL)
C     
C     Retrieve HJDs, subtract constant off them
C     
      DO I = 1, NV
         SLOT = WHICH(I)
         CALL HGETD('HJD', DWORK1(I), SLOT, MXSPEC, NMDOUB, 
     &        HDDOUB, NDOUB, MXDOUB, IFAIL)
         IF(IFAIL.NE.0) THEN
            WRITE(*,*) 'No HJD in slot ',SLOT
            GOTO 999
         END IF
      END DO
      SUB = DBLE(INT(DWORK1(1)))
      AVE = 0.D0
      DO I = 1, NV
         DWORK1(I) = DWORK1(I) - SUB
         AVE = AVE + DWORK1(I)
      END DO
      AVE = AVE/DBLE(NV)
C     
C     Scale for least squares fitting
C     
      LOW  =  1.D30
      HIGH = -1.D30
      DO I = 1, NV
         DWORK6(I) = DWORK1(I) - AVE
         LOW  = MIN(LOW, DWORK6(I))
         HIGH = MAX(HIGH, DWORK6(I))
      END DO
      RANGE = HIGH-LOW
      DO I = 1, NV
         DWORK6(I) = DWORK6(I)/RANGE
      END DO
C     
C     Loop through pixels
C     
      RESET = .TRUE.
      DO K = 1, NF
         FREQ = F1 + (F2-F1)*REAL(K-1)/REAL(MAX(1,NF-1))
         SLOT = SLOT3 + K - 1
         DO J = 1, NSAME
            IF(MASK(J)) THEN
               NOK = 0
               DO I = 1, NV
                  NS = WHICH(I)
                  DWORK2(I) = GET_FLX(COUNTS(J,NS), FLUX(J,NS))
                  DWORK3(I) = GET_ERF(COUNTS(J,NS), ERRORS(J,NS), 
     &                 FLUX(J,NS))
                  IF(DWORK3(I).GT.0.D0) NOK = NOK + 1
               END DO
C     
C     Fit and subtract poly 
C     
               IF(NOK.GT.NPOLY) THEN
                  CALL LEASQ(DWORK6, DWORK2, DWORK3, NV, NPOLY, A,
     &                 CHISQ,XM,1)
                  DO I = 1, NV
                     DWORK2(I) = DWORK2(I) - POLY(A,NPOLY,DWORK6(I))
                  END DO
C     
C     Compute periodogram
C     
                  F = SCARG(DWORK1, DWORK2, DWORK3, NV, FREQ, 
     &                 DWORK4, DWORK5, MXWORK, RESET)
                  E = MAX(F/10000.,1.E-20)
                  RESET = .FALSE.
               ELSE
                  F = 0.
                  E = -1.
               END IF
            ELSE
               F = 0.
               E = -1.
            END IF
C     
C     Store using first valid spectrum to provide sensitivity.
C     Error array arbitrary
C     
            
            RATIO = CFRAT(COUNTS(J,WHICH(1)),FLUX(J,WHICH(1)))
C     
            COUNTS(J,SLOT) = RATIO*F
            ERRORS(J,SLOT) = RATIO*E
            IF(COUNTS(J,SLOT).EQ.0.) THEN
               FLUX(J,SLOT) = RATIO
            ELSE
               FLUX(J,SLOT) = F
            END IF
         END DO
C     
C     Set headers of output spectra equal to those of first valid
C     spectrum
C     
         CALL SET_HEAD(SLOT, WHICH(1), MXSPEC, NPIX, ARC, NARC, 
     &        MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &        HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &        MXDOUB, MXINTR, MXREAL, IFAIL)
         CALL SL_INF(SLOT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &        NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &        NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &        IFAIL)
         WRITE(*,*) 'Power at frequency = ',FREQ,' stored in'
         WRITE(*,*) STRING(:LENSTR(STRING))
         CALL HSETR('Frequency', FREQ, SLOT, MXSPEC, NMREAL, 
     &        HDREAL, NREAL, MXREAL, IFAIL)
      END DO
 999  RETURN
      END

      REAL FUNCTION SCARG(TIME, DATA, SIGMA, N, FREQ, 
     &     SINE, COSINE, MXWORK, RESET)
C     
C     Computes Scargle (1982) periodogram function for N data
C     points at times TIME, value DATA
C     TIME and 1/FREQ should have same units.
C     It is advantageous in terms of speed to use this routine on the same
C     period lots of times to avoid recomputing the sines and cosines
C     
C     D TIME(N) -- Times of points
C     D DATA(N) -- Values of points
C     D SIGMA(N) -- Uncertainties on each point
C     I N - Number of points
C     R FREQ -- Frequency in cycles/unit time, with unit time being
C     the same unit as used for TIME
C     D SINE(MXWORK)    -- Workspace buffers to avoid recomputation
C     D COSINE(MXWORK)  -- of sines and cosines. 
C     I MXWORK          -- Size of the above Need MXWORK.GE.N
C     L RESET           -- Force recomputation of sines/cosines
C     Needed if TIME(N) have changed at all
C     
      IMPLICIT NONE
      INTEGER I, N
      REAL FREQ, FOLD
      DOUBLE PRECISION TIME(N)
      DOUBLE PRECISION DATA(N), SIGMA(N)
      DOUBLE PRECISION TWOPI
      DOUBLE PRECISION SUM1, SUM2, SUM3, SUM4
      DOUBLE PRECISION OMEGA, PHI, PHI0
      DOUBLE PRECISION WXS, WXC
      DOUBLE PRECISION XC, XS,  S0, C0, WGT
      INTEGER MXWORK
      DOUBLE PRECISION SINE(MXWORK), COSINE(MXWORK)
      LOGICAL RESET
      DATA FOLD/-1.D0/
C     
      TWOPI = 8.D0*ATAN(1.D0)
      OMEGA = TWOPI*FREQ
      IF(N.GT.MXWORK) THEN
         WRITE(*,*) 'Too many points for buffers in SCARG'
         SCARG = 0.
         RETURN
      END IF
      IF(RESET .OR. ABS(FREQ-FOLD).GT.1.E-6) THEN
         DO I = 1, N
            PHI       = OMEGA*TIME(I)
            SINE(I)   = SIN(PHI)
            COSINE(I) = COS(PHI)
         END DO
         FOLD = FREQ
      END IF
C     
C     Evaluate tau
C     
      SUM1 = 0.D0
      SUM2 = 0.D0
      DO I = 1, N
         IF(SIGMA(I).GT.0.) THEN
            WGT = 1./SIGMA(I)**2
            SUM1 = SUM1 + WGT*(2.*SINE(I)*COSINE(I))
            SUM2 = SUM2 + WGT*(2.*COSINE(I)**2-1.)
         END IF
      END DO
      PHI0 = ATAN2(SUM1, SUM2)/2.
      S0 = SIN(PHI0)
      C0 = COS(PHI0)
C     
      SUM1 = 0.D0
      SUM2 = 0.D0
      SUM3 = 0.D0
      SUM4 = 0.D0
      DO I = 1, N
         IF(SIGMA(I).GT.0.) THEN
            WGT  = 1./SIGMA(I)**2
            XS   = SINE(I)*C0-COSINE(I)*S0
            WXS  = XS*WGT
            XC   = COSINE(I)*C0+SINE(I)*S0
            WXC  = XC*WGT
            SUM1 = SUM1 + WXC*XC
            SUM2 = SUM2 + WXS*XS
            SUM3 = SUM3 + WXC*DATA(I)
            SUM4 = SUM4 + WXS*DATA(I)
         END IF
      END DO
      IF(FREQ.EQ.0.) THEN
         SCARG = REAL(SUM3*SUM3/SUM1/2.)
      ELSE
         SCARG = REAL((SUM3*SUM3/SUM1+SUM4*SUM4/SUM2)/2.)
      END IF
      RETURN
      END
