*MTOASC 
*   MTOASC Filein Fileout  -- Reads a MOLLY format file and dumps it out as
*                          an ASCII file to facilitate file transfer
*
* Parameters:
*
*    Filein  -- Name of input MOLLY file
*    Filiout -- Name of ASCII output file
*
*MTOASC 
      SUBROUTINE MTOASC(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC,
     &NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &MXREAL, CLOBBER, IFAIL)
*
* Converts MOLLY file to ASCII equivalent
* allowing selection criteria to be applied.
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
* >  L     CLOBBER
* >  I     IFAIL
*
      IMPLICIT NONE
*
* Integer parameters
*
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
      INTEGER FCODE
*
* Data arrays
*
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
*
* Numbers of header items
*
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
*
* Names of header items
*
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
*
* Values of header items
*
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
*
* Arc coefficients
*
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
*
* Command parameters
*
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
*
* Local variables
*
      CHARACTER*64 INFILE, OUTFILE
      INTEGER INUNIT, OUTUNIT
      CHARACTER*16 UNITS
      INTEGER IFAIL, ESLOT, NPX
      INTEGER  J, N, NSPEC, MAXSPEC
      LOGICAL  SAME,  WARN, DEFAULT, CLOBBER
      REAL  Z, VLIGHT, CGS, FAC, CFAC, AVE
      INTEGER NCOM
      DOUBLE PRECISION W, ANGST
*
* Functions
*
*
      DATA INFILE, OUTFILE/'SPECTRUM.MOL', 'SPECTRUM.ASC'/
      DATA INUNIT, OUTUNIT/23,24/
************************************************************************
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      ESLOT = 0
      N = 1
      DO WHILE(ESLOT .EQ. 0 .AND. N.LE.MAXSPEC)
        IF(NPIX(N).EQ.0) ESLOT = N
        N = N + 1
      END DO
      IF(ESLOT.EQ.0) THEN
        WRITE(*,*) 'No empty slots found. Need one for conversion.'
        GOTO 999
      END IF
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
*
      CALL CHAR_IN('MOLLY input file', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, INFILE, IFAIL)
      CALL CHAR_IN('ASCII output file', SPLIT, NSPLIT, MXSPLIT, 
     &NCOM, DEFAULT, OUTFILE, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
*
* Open files 
*
      OPEN(UNIT=INUNIT,FILE=INFILE,FORM='UNFORMATTED',
     &     STATUS='OLD',IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(*,*) 'Could not open ',INFILE
        RETURN
      END IF
      IF(CLOBBER) THEN
        OPEN(UNIT=OUTUNIT,FILE=OUTFILE,FORM='FORMATTED',
     &     STATUS='UNKNOWN',IOSTAT=IFAIL)
      ELSE 
       OPEN(UNIT=OUTUNIT,FILE=OUTFILE,FORM='FORMATTED',
     &     STATUS='NEW',IOSTAT=IFAIL)
      END IF
      IF(IFAIL.NE.0) THEN
        WRITE(*,*) 'Could not open ',OUTFILE
        RETURN
      END IF
*
* Start to read files, checking headers as we go.
*
      NSPEC = 0
      WARN = .FALSE.
      IFAIL = 0
      DO WHILE(IFAIL.EQ.0)
        NSPEC = NSPEC + 1
        CALL RTHEAD(INUNIT, FCODE, UNITS, NPX, 
     &  NARC(ESLOT), NCHAR(ESLOT), NDOUB(ESLOT), 
     &  NINTR(ESLOT), NREAL(ESLOT), MXCHAR, MXDOUB, 
     &  MXINTR, MXREAL, NMCHAR(1,ESLOT), NMDOUB(1,ESLOT), 
     &  NMINTR(1,ESLOT), NMREAL(1,ESLOT), HDCHAR(1,ESLOT), 
     &  HDDOUB(1,ESLOT), HDINTR(1,ESLOT), HDREAL(1,ESLOT), 
     &  IFAIL)
        IF(IFAIL.NE.0) THEN
          IFAIL = MAX(0, IFAIL)
          GOTO 998
        END IF
*
* Check that numbers of pixels, arc coefficients etc are ok
*
        IF(NPX.GT.MAXPX) THEN
          WRITE(*,*) 'Spectrum ', NSPEC,' has ',NPX,
     &    ' pixels.'
          WRITE(*,*) 'compared with maximum value ',MAXPX
          GOTO 998
        END IF
        IF(NARC(ESLOT).GT.MXARC) THEN
          WRITE(*,*) 'Spectrum ', NSPEC,' has ',NARC(ESLOT),
     &    ' arc coefficients'
          WRITE(*,*) 'compared with maximum value ',MXARC
          GOTO 998
        END IF
        IF(NCHAR(ESLOT).GT.MXCHAR) THEN
          WRITE(*,*) 'Spectrum ', NSPEC,' has ',NCHAR(ESLOT),
     &    ' character items'
          WRITE(*,*) 'compared with maximum value ',MXCHAR
          GOTO 998
        END IF
        IF(NDOUB(ESLOT).GT.MXDOUB) THEN
          WRITE(*,*) 'Spectrum ', NSPEC,' has ',NDOUB(ESLOT),
     &    ' double precision items'
          WRITE(*,*) 'compared with maximum value ',MXDOUB
          GOTO 998
        END IF
        IF(NINTR(ESLOT).GT.MXINTR) THEN
          WRITE(*,*) 'Spectrum ', NSPEC,' has ',NINTR(ESLOT),
     &    ' integer items'
          WRITE(*,*) 'compared with maximum value ',MXINTR
          GOTO 998
        END IF
        IF(NREAL(ESLOT).GT.MXREAL) THEN
          WRITE(*,*) 'Spectrum ', NSPEC,' has ',NREAL(ESLOT),
     &    ' real items'
          WRITE(*,*) 'compared with maximum value ',MXREAL
          GOTO 998
        END IF
        IF(.NOT.SAME(UNITS,'COUNTS') .AND. 
     &     .NOT.SAME(UNITS,'MILLIJANSKYS') .AND.
     &     .NOT.SAME(UNITS,'FLAMBDA')) THEN
          WRITE(*,*) 'Units = ',UNITS
          WRITE(*,*) 'Not recognised'
          GOTO 998
        END IF
*
* Read data
*
        CALL RTDATA(INUNIT, COUNTS(1,ESLOT), 
     &  ERRORS(1,ESLOT), FLUX(1,ESLOT), ARC(1,ESLOT), 
     &  FCODE, NPX, NARC(ESLOT), IFAIL)
        IF(IFAIL.NE.0) GOTO 998       
        IF(UNITS.EQ.'FLAMBDA') THEN
*
* Convert data to milli-Janskys
*
          VLIGHT = CGS('C')
          CFAC = VLIGHT*1.E-16
          DO J = 1, NPX
            IF(ERRORS(J,ESLOT).GT.0. .OR. FCODE.EQ.4) THEN
              Z = REAL(J)
              W = ANGST(Z, NPX, NARC(ESLOT), 
     &        ARC(1,ESLOT), MXARC)
              FAC = REAL(W**2/CFAC)
              FLUX(J,ESLOT)   = FAC*FLUX(J,ESLOT)
              IF(FCODE.EQ.5) ERRORS(J,ESLOT) = 
     &        FAC*ERRORS(J,ESLOT)
            END IF
          END DO
        END IF
*
* Put into standard counts/errors/fluxes internal
* MOLLY format
*
        IF(FCODE.EQ.1 .OR. FCODE.EQ.2) THEN
*
* Counts or counts and errors only
*
          WARN = .TRUE.
          DO J = 1, NPX
            IF(FCODE.EQ.1) ERRORS(J,ESLOT) = 1.
            IF(COUNTS(J,ESLOT).NE.0) THEN
              FLUX(J,ESLOT) = COUNTS(J,ESLOT)
            ELSE
              FLUX(J,ESLOT) = 1.
            END IF
          END DO
        ELSE IF(FCODE.EQ.4 .OR. FCODE.EQ.5) THEN
*
* Fluxes or fluxes and errors on fluxes only
*
          WARN = .TRUE.
          AVE = 0.
          DO J = 1, NPX
            AVE = AVE + FLUX(J,ESLOT)
          END DO
          AVE = AVE/REAL(NPX)/10.
          DO J = 1, NPX
            COUNTS(J,ESLOT) = FLUX(J,ESLOT)/AVE
            IF(COUNTS(J,ESLOT).EQ.0) FLUX(J,ESLOT) = 1.
            IF(FCODE.EQ.4) THEN
              ERRORS(J,ESLOT) = 1.
            ELSE
              ERRORS(J,ESLOT) = ERRORS(J,ESLOT)/AVE
            END IF
          END DO
        END IF
*
* Dump everthing to an ASCII file
*
        CALL WTHDAS(OUTUNIT, NPX, NARC(ESLOT), NCHAR(ESLOT), 
     &  NDOUB(ESLOT), NINTR(ESLOT), NREAL(ESLOT), MXCHAR, 
     &  MXDOUB, MXINTR, MXREAL, NMCHAR(1,ESLOT), NMDOUB(1,ESLOT), 
     &  NMINTR(1,ESLOT), NMREAL(1,ESLOT), HDCHAR(1,ESLOT), 
     &  HDDOUB(1,ESLOT), HDINTR(1,ESLOT), HDREAL(1,ESLOT), IFAIL)
        IF(IFAIL.NE.0) GOTO 998
*
        CALL WTDTAS(OUTUNIT, COUNTS(1,ESLOT), ERRORS(1,ESLOT), 
     &  FLUX(1,ESLOT), ARC(1,ESLOT), NPX, NARC, IFAIL)
        IF(IFAIL.NE.0) GOTO 998
      END DO
      IFAIL = 0
998   CLOSE(UNIT=INUNIT)
      CLOSE(UNIT=OUTUNIT)
      IF(WARN) THEN
        WRITE(*,*) 'At least one of the spectra read in is not'
        WRITE(*,*) 'on standard counts/errors/fluxes MOLLY'
        WRITE(*,*) 'format and a crude correction has been'
        WRITE(*,*) 'applied. MTOASC should be changed.'
      END IF
999   RETURN
      END
