/*

 Program to convert a swap bytes in a molly file to facilitate
 data transfer between SUN <---> Dec, Alpha, LINUX.

 Invoke as: <program name> <input molly file> <output molly file> [oneway]

 The program will recognize whether the file to be converted is
 'native' (i.e. written on the host computer) or foreign and will
 act accordingly. The program will not attempt to overwrite old
 files and cannot overwrite the input file (a change from before,
 but it seemed that the old method was not reliable).

 If you add "oneway" at the end, molcon will only convert in one
 direction, which is TO the native machine format. If it finds that
 the file is already in the native format, it will simply copy it with
 no conversion.

 T.R.Marsh, August 2001, ordinary C version to avoid the need for g++.

 Compile as in:

 gcc -o molcon molcon.c

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#define MAXBUFF 100000

/* 

Preprocessor switch for LINUX, Dec etc vs Sparc 
With gcc invoke using -DSPARC (for Solaris)

*/

#ifdef SPARC
#define SPARC 1
#else
#define SPARC 0
#endif

void reverse(const char *input, char *output, int nbytes, int nitems, int swap);

main(int argc, char *argv[]){
  
  long int i, fcode, npix, narc, nchar, ndoub, nintr, nreal, left, nspec, nload;
  union interp{  /* Union for understanding some parameters */
    char c[8]; 
    int i;
    float r;
    double d;
  } same, trans;
  int nobuff, nskip;
  char ibuffer[MAXBUFF], obuffer[MAXBUFF];
  enum file_type {native, foreign} ftype;
  FILE *molly_input, *molly_output;
  int oneway = 0, swap = 0;

  if(argc !=3 && argc !=4){
    fprintf(stderr,"Usage: input output [oneway]\n");
    exit(EXIT_FAILURE);
  }
  
  if(argc == 4){
    if(strcmp(argv[3], "oneway") != 0){
      fprintf(stderr,"Third argument, if specified, must say \"oneway\"\n");
      exit(EXIT_FAILURE);
    }
    oneway = 1;
  }
    
  if(!(molly_input = fopen(argv[1], "rb"))){
    fprintf(stderr,"Cannot open %s for input.\n",argv[1]);
    exit(EXIT_FAILURE);
  }

  if(!access(argv[2], F_OK)){
    fprintf(stderr,"%s already exists. Cannot overwrite it!\n",argv[2]);
    fclose(molly_input);    
    exit(EXIT_FAILURE);
  }

  if(!(molly_output = fopen(argv[2], "wb"))){
    fprintf(stderr,"Cannot open %s for output.\n",argv[2]);
    fclose(molly_input);    
    exit(EXIT_FAILURE);
  }

  /* OK, files are open */

  nspec  = 0;
  nobuff = 0;
  
  /* Each molly spectrum consists of five records with a format
     that will be explained as they are encountered.
     Fortran records start and end with 4 bytes representing the 
     number of bytes in the record. These 4 bytes must be swapped
     just as the real data does otherwise the FORTRAN program
     will not know how many bytes to read. */
  
  while(fread(ibuffer,1,4,molly_input) == 4){
    
    /* 
       On first one check that format of file passes elementary test
       and work out whether this file is native or foreign. Note the 
       sense of the test is reversed according to machine and so the 
       parameter SPARC set by a compiler option to the preprocessor
       is used. 
    */

    if(!nspec){
      if(ibuffer[0] == 0x2c && !ibuffer[1] && !ibuffer[2] && !ibuffer[3]){
	if(SPARC){
	  ftype = foreign;
	}else{
	  ftype = native;
	}
      }else if(ibuffer[3] == 0x2c && !ibuffer[2] && !ibuffer[1] && !ibuffer[0]){
	if(SPARC){
	  ftype = native;
	}else{
	  ftype = foreign;
	}
      }else{
	fprintf(stderr,"First 4 bytes not standard molly\n");
	exit(EXIT_FAILURE);
      }
    }
    
    swap = (!oneway || ftype == foreign);
 
    reverse(ibuffer,obuffer,4,1,swap);
    if(fwrite(obuffer,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write first 4 bytes.\n");
      exit(EXIT_FAILURE);
    }
    
    /* 

       The first record of a molly file consists of: 
       
       4 byte integer format code
       16 byte character string specifying units
       4 byte integer number of pixels
       4 byte integer number of arc coefficients
       4 byte integer number of character header items
       4 byte integer number of double header items
       4 byte integer number of integer header items
       4 byte integer number of real header items 

    */
    
    if(fread(same.c,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read format code.\n");
      exit(EXIT_FAILURE);
    }	

    reverse(same.c,trans.c,4,1,swap);

    if(ftype == native){
      fcode = same.i;
    }else{
      fcode = trans.i;
    }
    if(fcode < 1 || fcode > 5){
      fprintf(stderr,"Invalid format code %d\n",fcode);
      exit(EXIT_FAILURE);
    }

    if(fwrite(trans.c,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write first format code.\n");
      exit(EXIT_FAILURE);
    }
    
    /* Character strings do not need to be swapped  */
    
    if(fread(ibuffer,1,16,molly_input) != 16){
      fprintf(stderr,"Failed to read units.\n");
      exit(EXIT_FAILURE);
    }	
    if(fwrite(ibuffer,1,16,molly_output) != 16){
      fprintf(stderr,"Failed to write units.\n");
      exit(EXIT_FAILURE);
    }	
    
    if(fread(same.c,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read number of pixels.\n");
      exit(EXIT_FAILURE);
    }

    reverse(same.c,trans.c,4,1,swap);

    if(ftype == native){
      npix = same.i;
    }else{
      npix = trans.i;
    }
    if(npix < 1){
      fprintf(stderr,"Invalid value of %d\n",npix);
      exit(EXIT_FAILURE);
    }

    if(fwrite(trans.c,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write number of pixels.\n");
      exit(EXIT_FAILURE);
    }	
    
    if(fread(same.c,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read number of arc coefficients.\n");
      exit(EXIT_FAILURE);
    }

    reverse(same.c,trans.c,4,1,swap);

    if(ftype == native){
      narc = same.i;
    }else{
      narc = trans.i;
    }

    if(fwrite(trans.c,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write number of arc coefficients.\n");
      exit(EXIT_FAILURE);
    }	
      
    if(fread(same.c,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read number of character headers.\n");
      exit(EXIT_FAILURE);
    }

    reverse(same.c,trans.c,4,1,swap);

    if(ftype == native){
      nchar = same.i;
    }else{
      nchar = trans.i;
    }

    if(nchar < 0){
      fprintf(stderr,"Invalid value of %d\n",nchar);
      exit(EXIT_FAILURE);
    }
    if(fwrite(trans.c,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write number of character headers.\n");
      exit(EXIT_FAILURE);
    }	
    
    if(fread(same.c,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read number of double headers.\n");
      exit(EXIT_FAILURE);
    }	

    reverse(same.c,trans.c,4,1,swap);

    if(ftype == native){
      ndoub = same.i;
    }else{
      ndoub = trans.i;
    }
    if(ndoub < 0){
      fprintf(stderr,"Invalid value of %d\n",ndoub);
      exit(EXIT_FAILURE);
    }

    if(fwrite(trans.c,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write number of double headers.\n");
      exit(EXIT_FAILURE);
    }	

    if(fread(same.c,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read number of integer headers.\n");
      exit(EXIT_FAILURE);
    }
	
    reverse(same.c,trans.c,4,1,swap);

    if(ftype == native){
      nintr = same.i;
    }else{
      nintr = trans.i;
    }
    if(nintr < 0){
      fprintf(stderr,"Invalid value of %d\n",nintr);
      exit(EXIT_FAILURE);
    }

    if(fwrite(trans.c,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write number of integer headers.\n");
      exit(EXIT_FAILURE);
    }	
      
    if(fread(same.c,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read number of real headers.\n");
      exit(EXIT_FAILURE);
    }	

    reverse(same.c,trans.c,4,1,swap);

    if(ftype == native){
      nreal = same.i;
    }else{
      nreal = trans.i;
    }
    if(nreal < 0){
      fprintf(stderr,"Invalid value of %d\n",nreal);
      exit(EXIT_FAILURE);
    }

    if(fwrite(trans.c,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write number of real headers.\n");
      exit(EXIT_FAILURE);
    }	

    /* Cope with 4 bytes at end of first and start of second record */

    if(fread(ibuffer,1,8,molly_input) != 8){
      fprintf(stderr,"Failed to read bytes between records 1 and 2.\n");
      exit(EXIT_FAILURE);
    }	

    reverse(ibuffer,obuffer,4,2,swap);
    if(fwrite(obuffer,1,8,molly_output) != 8){
      fprintf(stderr,"Failed to write bytes between records 1 and 2.\n");
      exit(EXIT_FAILURE);
    }	
      
    /* The second record of a molly spectrum contains the header
       item names which are all 16 bytes in length. These character
       strings do not need swapping. */
      
    if(left = 16*(nchar+ndoub+nintr+nreal)){
      while(left){
	nload = MAXBUFF < left ? MAXBUFF : left;
	left -= nload;
	fread(ibuffer,1,nload,molly_input);
	fwrite(ibuffer,1,nload,molly_output);
      }
    }

    if(fread(ibuffer,1,8,molly_input) != 8){
      fprintf(stderr,"Failed to read bytes between records 2 and 3.\n");
      exit(EXIT_FAILURE);
    }	
    reverse(ibuffer,obuffer,4,2,swap);

    if(fwrite(obuffer,1,8,molly_output) != 8){
      fprintf(stderr,"Failed to write bytes between records 2 and 3.\n");
      exit(EXIT_FAILURE);
    }	
    
    /* 3rd record contains header values in order:
       nchar 32 byte character strings (no swapping needed)
       ndoub  8 byte double precision numbers
       nintr  4 byte integers
       ndoub  4 bytes reals */
    
    if(left = 32*nchar){
      while(left){
	nload = MAXBUFF < left ? MAXBUFF : left;
	left -= nload;
	fread(ibuffer,1,nload,molly_input);
	fwrite(ibuffer,1,nload,molly_output);
      }
    }
    
    if(left = ndoub){
      while(left){
	nload = MAXBUFF/8 < left ? MAXBUFF/8 : left;
	left -= nload;
	fread(ibuffer,1,8*nload,molly_input);
	reverse(ibuffer,obuffer,8,nload,swap);
	fwrite(obuffer,1,8*nload,molly_output);
      }
    }
      
    if(left = nintr+nreal){
      while(left){
	nload = MAXBUFF/4 < left ? MAXBUFF/4 : left;
	left -= nload;
	fread(ibuffer,1,4*nload,molly_input);
	reverse(ibuffer,obuffer,4,nload,swap);
	fwrite(obuffer,1,4*nload,molly_output);
      }
    }
      
    /* 4 bytes at end of 3rd, start of 4th records */

    if(fread(ibuffer,1,8,molly_input) != 8){
      fprintf(stderr,"Failed to read bytes between records 3 and 4.\n");
      exit(EXIT_FAILURE);
    }	
    reverse(ibuffer,obuffer,4,2,swap);
    if(fwrite(obuffer,1,8,molly_output) != 8){
      fprintf(stderr,"Failed to write bytes between records 3 and 4.\n");
      exit(EXIT_FAILURE);
    }	    
      
    /* 4th record consists of abs(narc) 8 byte 
       double precision arc coefficients */
      
    if(left = abs(narc)){
      while(left){
	nload = MAXBUFF/8 < left ? MAXBUFF/8 : left;
	left -= nload;
	fread(ibuffer,1,8*nload,molly_input);
	reverse(ibuffer,obuffer,8,nload,swap);
	fwrite(obuffer,1,8*nload,molly_output);
      }
    }
      
    if(fread(ibuffer,1,8,molly_input) != 8){
      fprintf(stderr,"Failed to read bytes between records 4 and 5.\n");
      exit(EXIT_FAILURE);
    }	
    reverse(ibuffer,obuffer,4,2,swap);
    if(fwrite(obuffer,1,8,molly_output) != 8){
      fprintf(stderr,"Failed to write bytes between records 4 and 5.\n");
      exit(EXIT_FAILURE);
    }	    
      
    /* 5th record contains data as array of reals
       Numbers depend on exact format code used  */
      
    if(fcode  == 1){
      left = npix;
    }else if(fcode == 2){
      left = 2*npix;
    }else if(fcode == 3){
      left = 3*npix;
    }else if(fcode == 4){
      left = npix;
    }else if(fcode == 5){
      left = 2*npix;
    }
    while(left){
      nload = MAXBUFF/4 < left ? MAXBUFF/4 : left;
      left -= nload;
      fread(ibuffer,1,4*nload,molly_input);
      reverse(ibuffer,obuffer,4,nload,swap);
      fwrite(obuffer,1,4*nload,molly_output);
    }
      
    /* 4 bytes at end of 5th record, this is the end of the spectrum */

    if(fread(ibuffer,1,4,molly_input) != 4){
      fprintf(stderr,"Failed to read bytes at end of record 5.\n");
      exit(EXIT_FAILURE);
    }	
    reverse(ibuffer,obuffer,4,2,swap);
    if(fwrite(obuffer,1,4,molly_output) != 4){
      fprintf(stderr,"Failed to write bytes at end of record 5.\n");
      exit(EXIT_FAILURE);
    }	    
  }
  fclose(molly_input);
  fclose(molly_output);
  exit(EXIT_SUCCESS);
}

/* byte swaps buffer (if swap) */

void reverse(const char *input, char *output, int nbytes, int nitems, int swap){
  
  /* Reverses byte order of nitems of nbytes a piece. output must
     be at least as big as input */
      
  int i, j, n;
  if(swap){
    for(n = 0; n < nitems; n++){
      for(i=n*nbytes,j=(n+1)*nbytes-1;i<(n+1)*nbytes;i++,j--) 
	output[i] = input[j];
    }
  }else{
    for(n = 0; n < nbytes*nitems; n++)
      output[n] = input[n];
  }
  return;
}



