      SUBROUTINE LINEPOSN( NPIX, DATA, SIGMA, PIXCEN, PIXSIG,
     *      PROFILE, P1, P2, P3, P4 )
*
*  Measures the pixel positions of spectral lines by maximizing the
*  cross-correlation between the spectrum data and a template line profile.
*
*  The first two derivatives of the template line profile are evaluated
*  by the USER-WRITTEN function PROFILE( PIXEL, IDERIV, P1, P2, ... ), where
*
*      PIXEL      = PIXEL OFFSET FROM CENTER OF LINE PROFILE
*      IDERIV     = 1 IF FIRST DERIVATIVE NEEDED
*                   2 IF SECOND DERIVATIVE IS NEEDED
*      P1,P2...= PARAMETERS DETERMINING FORM OF LINE PROFILE
*
*  The formal uncertainty in the line position is also evaluated.
*
*  Inputs:
*
*      NPTS       = Number of data points
*      DATA       = Spectrum data
*      SIGMA      = 1-sigma uncertainties
*      PIXCEN     = Initial guess at pixel position of spectrum line
*      PIXSIG     = Estimated sigma of the spectral line
*      PROFILE    = Function to evaluate template profile, its first and
*            second derivatives with respect to pixel position
*      P1,P2...= PARAMETERS USED BY PROFILE( PIXEL, IDERIV, P1, P2,...)
*
*  Output:
*
*      PIXCEN      = Position of spectrum line in pixels
*      PIXSIG      = Formal 1-sigma uncertainty in line position
*
*  Sept 1984 by Keith Horne at IOA.
*
      IMPLICIT NONE
      INTEGER NPIX, I, J, K, L, I1, I2
      INTEGER NITS, ITER
      REAL DATA(NPIX), SIGMA(NPIX)
      DOUBLE PRECISION SUM, SUM2
      DOUBLE PRECISION PROFILE
      EXTERNAL PROFILE
      DOUBLE PRECISION EPS, CARE, PIXWID, BIGSTEP, PIXCEN, PCEN, PIXSIG
      DOUBLE PRECISION OFFSET, P1, P2, P3, P4, C1, V1, C2, V2, PIXSTEP
      DOUBLE PRECISION STEP, ADD
      INTEGER MAXITER
      DATA EPS/8.E-5/
      DATA CARE/.9/
      DATA MAXITER/100/
*
      PIXWID = MAX( 5., 3*PIXSIG )      ! Half-width of pixel window
      BIGSTEP = PIXWID / 2.            ! Largest allowed step length
*
* Start with binary chop
* Find two points on either side of answer
*
        L = 0
500     CONTINUE
        J = 1
        L = L + 1
        DO K = -1, 1, 2
          PCEN = PIXCEN + BIGSTEP*REAL(K)
          I1 = MAX(1, MIN( NPIX, NINT( PCEN - PIXWID ) ))
          I2 = MAX(1, MIN( NPIX, NINT( PCEN + PIXWID ) ))
          SUM = 0.D0
          DO I = I1, I2
            OFFSET = I - PCEN
            SUM  = SUM  + DATA(I) * PROFILE( OFFSET, 1, P1, P2, P3, P4 )
          END DO
          IF(J.EQ.1) THEN
            C1 = PCEN
            V1 = SUM
          ELSE
            C2 = PCEN
            V2 = SUM
          END IF
          J = J + 1
        END DO
*
* If we fail, carry on dividing
* interval in case we were too cavalier
*
        IF(V1*V2.GT.0.) THEN
          IF(L.LT.4) THEN
            BIGSTEP = BIGSTEP/2.
            GOTO 500
          END IF
*
* Give up
*
          WRITE(*,*) 'Binary chop stage failed'
          PIXSIG = -PIXSIG
          RETURN
        END IF
*
* Basic binary chop section. We will return to this
* if we encounter trouble in the Newton stage
*
        NITS = 0
1000    CONTINUE
        NITS = NITS + 1
        PCEN = (C1+C2)/2.
        I1 = MAX(1, MIN( NPIX, NINT( PCEN - PIXWID ) ))
        I2 = MAX(1, MIN( NPIX, NINT( PCEN + PIXWID ) ))
        SUM = 0.D0
        DO I = I1, I2
          OFFSET = I - PCEN
          SUM  = SUM  + DATA(I) * PROFILE( OFFSET, 1, P1, P2, P3, P4 )
        END DO
        IF(SUM*V1.GT.0.) THEN
          C1 = PCEN
          V1 = SUM
        ELSE
          C2 = PCEN
          V2 = SUM
        END IF
        IF(C2-C1.GT.BIGSTEP/10.) GOTO 1000
        PIXCEN = (C1+C2)/2.
        BIGSTEP = (C2-C1)/2.
        IF(NITS.GT.MAXITER) THEN
          IF(BIGSTEP.LT.EPS) GOTO 20
          WRITE(*,*) 'Exceeded binary chop limit'
          PIXSIG = -PIXSIG
          RETURN
        END IF
*
*  Newton iteration to refine line position
*
      DO ITER = 1, MAXITER
        I1 = MAX(1, MIN( NPIX, NINT( PIXCEN - PIXWID ) ))
        I2 = MAX(1, MIN( NPIX, NINT( PIXCEN + PIXWID ) ))
        SUM = 0.D0
        SUM2 = 0.D0
        DO I = I1, I2
          OFFSET = I - PIXCEN
          SUM  = SUM  + DATA(I) * PROFILE( OFFSET, 1, P1, P2, P3, P4 )
          SUM2 = SUM2 + DATA(I) * PROFILE( OFFSET, 2, P1, P2, P3, P4 )
        END DO
*
        IF( SUM2.NE.0.D0 ) THEN
          PIXSTEP = SUM / SUM2
        ELSE
          GOTO 1000
        END IF
*
        STEP = ABS(PIXSTEP)
        IF( STEP.GT.BIGSTEP ) GOTO 1000
*
        PIXCEN = PIXCEN + PIXSTEP * CARE
        IF( STEP .LT. EPS ) GOTO 20
      END DO
*
      WRITE(*,*) '** WARNING: LINE POSITION FAILED TO CONVERGE:'
      WRITE(*,*) '** LAST STEP WAS', PIXSTEP, ' PIXELS.'
*
*  Evaluate formal uncertainty in line position ----------------
*
20    CONTINUE
      I1 = MAX(1, MIN( NPIX, NINT( PIXCEN - PIXWID ) ))
      I2 = MAX(1, MIN( NPIX, NINT( PIXCEN + PIXWID ) ))
      SUM = 0.D0
      SUM2= 0.D0
      DO I = I1, I2
        OFFSET = I - PIXCEN
        ADD = SIGMA(I) * PROFILE( OFFSET, 1, P1, P2, P3, P4 )
        SUM = SUM + ADD * ADD
        SUM2= SUM2+ DATA(I) * PROFILE( OFFSET, 2, P1, P2, P3, P4 )
      END DO
      PIXSIG = SQRT( SUM )
      IF( SUM2.NE.0.D0 ) PIXSIG = PIXSIG / ABS( SUM2 )
*
      RETURN
      END
*
* -------------------------------------------------------------------
*
      DOUBLE PRECISION FUNCTION COREFIT( PIXEL, IDERIV, SIGMA )
*
*  Evaluates gaussian line profile template and first two derivatives
*  needed for spectrum line position fitting by LINEPOSN.
*
*  Input:
*      PIXEL      = OFFSET IN PIXELS FROM LINE CENTER
*      IDERIV     = 0 IF PROFILE IS TO BE EVALUATED
*              1 IF FIRST DERIVATIVE
*              2 IF SECOND DERIVATIVE
*      SIGMA      = GAUSSIAN PIXEL WIDTH OF TEMPLATE ( SHOULD BE > 0.5)
*  Output:
*      COREFIT      = DESIRED VALUE
*
*  Sepy 1984 by Keith Horne at IOA
*
      IMPLICIT NONE
      DOUBLE PRECISION PIXEL, SIGMA, S, X, XX, G
      INTEGER IDERIV
*
      S = MAX( SIGMA, 0.5D0 )
      X = PIXEL / S
      IF( ABS(X) .GT. 6.D0 ) THEN
        COREFIT = 0.D0
        RETURN
      END IF
      XX = X*X
      G = EXP( -0.5D0 * XX )
      IF( IDERIV.LT.1 ) THEN
        COREFIT = G
      ELSE IF( IDERIV.EQ.1 ) THEN
        COREFIT = - G * X / S
      ELSE
        COREFIT = G * (XX-1.) / (S*S)
      END IF
*
      RETURN
      END
*
* -------------------------------------------------------------------
*
      DOUBLE PRECISION FUNCTION WINGFIT( PIXEL, IDERIV, SIGMA, SEP )
*
*  Evaluates first two derivatives of a flat-topped line profile template
*  used to fit velocity to data in the wings of the line profile.
*
*  Input:
*      PIXEL      = OFFSET IN PIXELS FROM LINE CENTER
*      IDERIV      = 1 IF FIRST DERIVATIVE
*              2 IF SECOND DERIVATIVE
*      (NOTE: this subroutine will NOT calculate the profile itself.)
*      SIGMA      = GAUSSIAN PIXEL WIDTH OF BANDPASSES ( SHOULD BE > 0.5)
*      SEP      = SEPARATION IN PIXELS
*  Output:
*      WINGFIT      = DESIRED VALUE
*
*  Sept 1984 by Keith Horne at IOA
*
      IMPLICIT NONE
      DOUBLE PRECISION PIXEL, S, X1, X2, G1, G2, SIGMA, SEP
      INTEGER IDERIV
      S = MAX( SIGMA, 0.5D0 )
      X1 = (PIXEL - 0.5D0*SEP) / S
      X2 = (PIXEL + 0.5D0*SEP) / S
      G1 = 0.D0
      G2 = 0.D0
      IF( ABS(X1) .LE. 8.D0 ) G1 = EXP( -0.5D0 * X1*X1 )
      IF( ABS(X2) .LE. 8.D0 ) G2 = EXP( -0.5D0 * X2*X2 )
*
      IF( IDERIV.LE.1 ) THEN
        WINGFIT = G1 - G2
      ELSE
        WINGFIT = ( G2 * X2 - G1 * X1 ) / S
      END IF
*
      RETURN
      END
