CLFOR
C LFOR FILE N1 N2 MODE -- Loads foreign MOLLY format spectra (i.e. written 
C                      under different operating systems). This is done by
C                      reading the file in byte by byte followed by appropriate
C                      byte swapping operations and therefore can be slow.
C                      Once you have read the data, dump it out into a local
C                      format file with WRITE. The file could have been
C                      ftp-ed over in binary mode or retrieved from a backup
C                      or tar tape.
C
C Parameters: 
C     
C   FILE   -- Name of file to read from.
C   N1     -- Slot number to read first spectrum into. 'A' to indicate that
C             the first spectrum is loaded into the first empty slot (but be
C             careful!)
C   N2     -- Slot number to read last spectrum into.  If you have used option
C             'A' then N2 = number of spectra to load
C
C   MODE   -- 1 = VMS                ---> ULTRIX, OSF, LINUX (Decstations, alphas)
C             2 = ULTRIX, OSF, LINUX ---> SUN (Solaris, maybe SUNos)
C                 and SUN            ---> ULTRIX, OSF, LINUX
C             3 = VMS                ---> SUN (Solaris, maybe SUNos)
C
C NB Option 2 only involves swapping bytes and so is reversible.
C
CLFOR
      SUBROUTINE LFOR(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &     FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC,
     &     NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &     HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &     MXREAL, MUTE, IFAIL)
C     
C     Routine for reading foreign MOLLY format data written under different
C     operating systems.
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     >  L     MUTE
C     >  I     IFAIL
C     
      IMPLICIT NONE
C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
      INTEGER FCODE
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C     Local variables
C     
      CHARACTER*64 FILENAME
      CHARACTER*100 STRING
      CHARACTER*16 UNITS
      CHARACTER*20 CSLOT
      INTEGER SLOT1, SLOT2, NLOAD, LSLOT, IFAIL, LUNIT
      INTEGER I, J,  NSPEC, MAXSPEC, MODE, LENSTR
      LOGICAL  SAME,  WARN, DEFAULT, MUTE, OLD, SET
      DOUBLE PRECISION ANGST, W
      REAL  Z, VLIGHT, CGS, FAC, CFAC, AVE
      INTEGER NR, NCOM
C     
C     Functions
C     
      INTEGER RECL, NBYTE
      COMMON/REC_LEN/RECL, NBYTE
C     
      DATA FILENAME/'spectrum.mol'/
      DATA CSLOT, MODE/'1', 1/
      DATA SLOT2, NLOAD/1,1000/
      DATA LUNIT/23/
      DATA RECL/2042/
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
C     
      CALL CHAR_IN('File name', SPLIT, NSPLIT, MXSPLIT, NCOM,
     &     DEFAULT, FILENAME, IFAIL)
      CALL CHAR_IN('First slot to read into', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, CSLOT, IFAIL)
      CALL UPPER_CASE(CSLOT)
      IF(CSLOT.NE.'A') THEN
         READ(CSLOT,*,IOSTAT=IFAIL) SLOT1
         IF(IFAIL.EQ.0 .AND. (SLOT1.LT.1 .OR. SLOT1.GT.MAXSPEC)) THEN
            WRITE(*,*) 'Slot out of range 1 to ',MAXSPEC
            IFAIL = 1
         END IF
      END IF
      IF(IFAIL.NE.0) GOTO 999
C     
      IF(CSLOT.NE.'A') THEN
         SLOT2 = MAX(SLOT1, SLOT2)
         CALL INTR_IN('Last slot to read into', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
         LSLOT = SLOT2
      ELSE
         DO I = 1, MAXSPEC
            IF(NPIX(I).LE.0) THEN
               SLOT1 = I
               GOTO 101
            END IF
         END DO
 101     CONTINUE
         NLOAD = MIN(MAXSPEC-SLOT1+1, NLOAD)
         CALL INTR_IN('Number of spectra to load', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, NLOAD, 1, MAXSPEC-SLOT1+1, IFAIL)
         LSLOT = SLOT1 + NLOAD - 1
      END IF
      CALL INTR_IN('Mode (1=V>U,O; 2=U,O,L>S or S>U,O,L; 3=V>S)', 
     &     SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, MODE, 1, 3, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
      OPEN(UNIT=LUNIT,FILE=FILENAME,FORM='UNFORMATTED',
     &     STATUS='OLD',IOSTAT=IFAIL)
      IF(IFAIL.NE.0) THEN
         WRITE(*,*) 'Could not open ',FILENAME
         RETURN
      END IF
C     
C     Read spectra
C     
      NSPEC = 0
      WARN = .FALSE.
      NR  = 0
      OLD = .FALSE.
      DO I = SLOT1, LSLOT
         NSPEC = NSPEC + 1
         IF(MODE.EQ.1) THEN
            CALL VMS_DECH(LUNIT, FCODE, UNITS, NPIX(I), 
     &           NARC(I), NCHAR(I), NDOUB(I), NINTR(I), NREAL(I), 
     &           MXCHAR, MXDOUB, MXINTR, MXREAL, NMCHAR(1,I),
     &           NMDOUB(1,I), NMINTR(1,I), NMREAL(1,I), HDCHAR(1,I), 
     &           HDDOUB(1,I), HDINTR(1,I), HDREAL(1,I), IFAIL)
         ELSE IF(MODE.EQ.2) THEN
            CALL DEC_SUNH(LUNIT, FCODE, UNITS, NPIX(I), 
     &           NARC(I), NCHAR(I), NDOUB(I), NINTR(I), NREAL(I), 
     &           MXCHAR, MXDOUB, MXINTR, MXREAL, NMCHAR(1,I),
     &           NMDOUB(1,I), NMINTR(1,I), NMREAL(1,I), HDCHAR(1,I), 
     &           HDDOUB(1,I), HDINTR(1,I), HDREAL(1,I), IFAIL)
         ELSE IF(MODE.EQ.3) THEN
            CALL VMS_SUNH(LUNIT, FCODE, UNITS, NPIX(I), 
     &           NARC(I), NCHAR(I), NDOUB(I), NINTR(I), NREAL(I), 
     &           MXCHAR, MXDOUB, MXINTR, MXREAL, NMCHAR(1,I),
     &           NMDOUB(1,I), NMINTR(1,I), NMREAL(1,I), HDCHAR(1,I), 
     &           HDDOUB(1,I), HDINTR(1,I), HDREAL(1,I), IFAIL)
         END IF
         IF(IFAIL.NE.0) THEN
            NPIX(I) = 0
            IFAIL = MAX(0, IFAIL)
            GOTO 998
         END IF
C     
C     Check that numbers of pixels, arc coefficients etc are ok
C     
         IF(NPIX(I).GT.MAXPX) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NPIX(I),' pixels.'
            WRITE(*,*) 'compared with maximum value ',MAXPX
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NARC(I).GT.MXARC) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NARC(I),
     &           ' arc coefficients'
            WRITE(*,*) 'compared with maximum value ',MXARC
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NCHAR(I).GT.MXCHAR) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NCHAR(I),
     &           ' character items'
            WRITE(*,*) 'compared with maximum value ',MXCHAR
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NDOUB(I).GT.MXDOUB) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NDOUB(I),
     &           ' double precision items'
            WRITE(*,*) 'compared with maximum value ',MXDOUB
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NINTR(I).GT.MXINTR) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NINTR(I),
     &           ' integer items'
            WRITE(*,*) 'compared with maximum value ',MXINTR
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(NREAL(I).GT.MXREAL) THEN
            WRITE(*,*) 'Spectrum ', NSPEC,' has ',NREAL(I),
     &           ' real items'
            WRITE(*,*) 'compared with maximum value ',MXREAL
            NPIX(I) = 0
            GOTO 998
         END IF
         IF(.NOT.SAME(UNITS,'COUNTS') .AND. 
     &        .NOT.SAME(UNITS,'MILLIJANSKYS') .AND.
     &        .NOT.SAME(UNITS,'FLAMBDA')) THEN
            WRITE(*,*) 'Units = ',UNITS
            WRITE(*,*) 'Not recognised'
            NPIX(I) = 0
            GOTO 998
         END IF
C     
C     Read data
C     
         IF(MODE.EQ.1) THEN
            CALL VMS_DECD(LUNIT, COUNTS(1,I), ERRORS(1,I), FLUX(1,I), 
     &           ARC(1,I), FCODE, NPIX(I), NARC(I), IFAIL)
         ELSE IF(MODE.EQ.2) THEN
            CALL DEC_SUND(LUNIT, COUNTS(1,I), ERRORS(1,I), FLUX(1,I), 
     &           ARC(1,I), FCODE, NPIX(I), NARC(I), IFAIL)
         ELSE IF(MODE.EQ.3) THEN
            CALL VMS_SUND(LUNIT, COUNTS(1,I), ERRORS(1,I), FLUX(1,I), 
     &           ARC(1,I), FCODE, NPIX(I), NARC(I), IFAIL)
         END IF
         IF(IFAIL.NE.0) THEN
            NPIX(I) = 0
            GOTO 998       
         END IF
         IF(UNITS.EQ.'FLAMBDA') THEN
C     
C     Convert data to milli-Janskys
C     
            VLIGHT = CGS('C')
            CFAC = VLIGHT*1.E-16
            DO J = 1, NPIX(I)
               IF(ERRORS(J,I).GT.0. .OR. FCODE.EQ.4) THEN
                  Z = REAL(J)
                  W = ANGST(Z, NPIX(I), NARC(I), ARC(1,I), MXARC)
                  FAC = REAL(W**2/CFAC)
                  FLUX(J,I)   = FAC*FLUX(J,I)
                  IF(FCODE.EQ.5) ERRORS(J,I) = FAC*ERRORS(J,I)
               END IF
            END DO
         END IF
C     
C     Put into standard counts/errors/fluxes internal
C     MOLLY format
C     
         IF(FCODE.EQ.1 .OR. FCODE.EQ.2) THEN
C     
C     Counts or counts and errors only
C     
            WARN = .TRUE.
            DO J = 1, NPIX(I)
               IF(FCODE.EQ.1) ERRORS(J,I) = 1.
               IF(COUNTS(J,I).NE.0) THEN
                  FLUX(J,I) = COUNTS(J,I)
               ELSE
                  FLUX(J,I) = 1.
               END IF
            END DO
         ELSE IF(FCODE.EQ.4 .OR. FCODE.EQ.5) THEN
C     
C     Fluxes or fluxes and errors on fluxes only
C     
            WARN = .TRUE.
            AVE = 0.
            DO J = 1, NPIX(I)
               AVE = AVE + FLUX(J,I)
            END DO
            AVE = AVE/REAL(NPIX(I))/10.
            DO J = 1, NPIX(I)
               COUNTS(J,I) = FLUX(J,I)/AVE
               IF(COUNTS(J,I).EQ.0) FLUX(J,I) = 1.
               IF(FCODE.EQ.4) THEN
                  ERRORS(J,I) = 1.
               ELSE
                  ERRORS(J,I) = ERRORS(J,I)/AVE
               END IF
            END DO
         END IF
C     
C     Type out 1-line on slot
C     
         IF(.NOT.MUTE) THEN
            CALL SL_INF(I, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) STRING(:LENSTR(STRING))
         END IF
C     
C     convert for old style ephemeris
C     
         CALL HCONV(I, MXSPEC, NMCHAR, HDCHAR, NCHAR, MXCHAR, 
     &        NMDOUB, HDDOUB, NDOUB, MXDOUB, NMINTR, HDINTR, NINTR, 
     &        MXINTR, NMREAL, HDREAL, NREAL, MXREAL, SET, IFAIL)
         IF(SET) OLD = .TRUE.
         NR = NR + 1
      END DO
      IFAIL = 0
 998  CLOSE(UNIT=LUNIT)
      WRITE(*,*) 'Read ',NR,' spectra from ',FILENAME
      IF(WARN) THEN
         WRITE(*,*) 'At least one of the spectra read in is not'
         WRITE(*,*) 'on standard counts/errors/fluxes MOLLY'
         WRITE(*,*) 'format and a crude correction has been'
         WRITE(*,*) 'applied. SPCRED should be changed.'
      END IF
      IF(OLD) THEN
         WRITE(*,*) 'At least one of you spectra had the constant'
         WRITE(*,*) 'term of an ephemeris stored as HJD* where *'
         WRITE(*,*) 'is either O, S or B. This is now obsolete to'
         WRITE(*,*) 'account for a safer search strategy when'
         WRITE(*,*) 'editing header items that caused confusion'
         WRITE(*,*) 'with HJD. The item has been renamed to Zero*'
         WRITE(*,*) 'and nothing else is affected. However if you'
         WRITE(*,*) 'want to avoid this message every time you read'
         WRITE(*,*) 'your spectra you should dump them and delete'
         WRITE(*,*) 'the old file. Sorry!'
      END IF
 999  RETURN
      END

