      SUBROUTINE LEASQ(XDATA,YDATA,YERR,NUMBER,N,A,CHISQ,XM,NORM)
*
*   LEASQ provides a polynomial fit of specified degree to a set of
*   data points. The fit is done in DOUBLE PRECISION.
*   uses the DOUBLE PRECISION matrix inversion subroutine 'MLSRAR'.
*
*   XDATA   =  Array containing the X-values. 
*   YDATA   =  Array containing the Y-values.
*   YERR    =  Array containing the Y-sigmas,negative to ignore.
*   NUMBER  =  Number of data points to be fitted.
*   N       =  Number of coefficients in polynomial to be fitted. The
*              order of the polynomial is (n-1).
*   A       =  Array holding coefficients of fit (REAL*8). This is arranged:
*              POLY = A(1) + A(2)*X + ... A(N)*X**(N-1)
*   CHISQ   =  Contains the Chi-square value of the fit on output. If
*              CHISQ = -1. on output then the matrix was singular, if
*              CHISQ = -2. on output then overflow or divide check occured.
*              CHISQ = -3. on output then invalid parameters input.
*              If you set chisq = 0. on input error messages will be printed.
*              (REAL*8)
*   XM      =  Working storage array (REAL*8). Dimension in the main
*              program as XM(NMAX,2*NMAX+3) where 'NMAX' is the maximum
*              value of 'N' you wish to call.
*   NORM    =  Scaling parameter. Since large powers of the input data are
*              to be taken overflows or underflows can easily occur. If
*              you set norm = 1 on input data are scaled to reduce likelihood
*              of this event. norm = 0 instructs for no scaling. Input data
*              are not destroyed by the scaling, and coefficients are
*              automatically scaled back before output.
*
      IMPLICIT NONE
      LOGICAL RITE
      INTEGER NUMBER, N
      DOUBLE PRECISION XM(N,2*N+3),XX,RR,EPS,AVE,SIGMA
      DOUBLE PRECISION XDATA(NUMBER),YDATA(NUMBER),A(N)
      DOUBLE PRECISION YERR(NUMBER),CHISQ,S,R, X1, X2, X3
      INTEGER NMAX, N21, N22, N23, ITER, NORM
      INTEGER I, J, K, L, M2, ITEST
*
*   EPS    =  Maximum allowed error in iterations on the polynomial
*             coefficients. Convergence terminates when ERROR.LT.EPS
*   NMAX   =  Maximum allowed number of coefficients (due to dimension
*             statements).
*
      DATA EPS,NMAX/1.D-6,50/
      RITE=.FALSE.      
*
*  Test input data.
*
      IF((NUMBER.LE.0.OR.N.GT.NUMBER).OR.N.LE.0) THEN
        WRITE(*,'(1X,A,I5,A,I5/A)') 'LEASQ: no of points ',NUMBER,
     &' no of coefficients ',N,
     &' invalid values (must be positive and points.ge.coefficients).'
        CHISQ=-3.
        RETURN
      ELSE IF (N.GT.NMAX) THEN
        WRITE (*,'(1X,A,I4,A,I4)') 'LEASQ: no of coefficients: ',
     &  N,' must not exceed: ',NMAX
        CHISQ=-3.
        RETURN
      END IF
*
*  If the input value of CHISQ is 0. Allow printing of error messages.  
*
      IF (CHISQ.EQ.0.) RITE=.TRUE.    
      ITER=5     
      IF (RITE) ITER=-ITER   
      N21 = 2*N + 1
      N22 = N21 + 1
      N23 = N21 + 2
*
*  Rescale the input data (for NORM.NE.0).
*
      IF (NORM.NE.0 .AND. N.NE.1) THEN
        AVE = 0.0
        SIGMA = 0.0
        DO I=1,NUMBER
          X1    = XDATA(I)
          AVE   = AVE + X1
          SIGMA = SIGMA + X1*X1    
        END DO 
        AVE = AVE/NUMBER
        SIGMA = DSQRT(SIGMA/NUMBER-AVE*AVE)
      END IF
*
*  Zero the working array.
*
      DO I = 1,N     
        DO J = N21,N23
          XM(I,J) = 0.D0
        END DO
      END DO
      X2 = 0.D0
      X3 = 0.D0
*
*   Compute the moments of the data.
*
      M2 = 2*N 
      DO I = 1, NUMBER
        IF(YERR(I).GT.0.) THEN
          RR = (1.D0/YERR(I))**2 
          X2 = X2 + RR
          XX = YDATA(I)*RR
          X3 = X3 + XX 
          IF(N.NE.1) THEN
            X1 = XDATA(I)
            DO J = 3, M2    
              IF (NORM.EQ.0) THEN
                RR = RR*X1 
              ELSE
                RR = RR*(X1-AVE)/SIGMA
              END IF
              IF(J.GT.N) THEN
                XM(J-N,N22)=XM(J-N,N22)+RR
              ELSE
                XM(J,N21)=XM(J,N21)+RR 
              END IF
            END DO
            DO J = 2, N
              IF (NORM.EQ.0) THEN
                XX = XX*X1 
              ELSE
                XX = XX*(X1-AVE)/SIGMA
              END IF
              XM(J,N23) = XM(J,N23) + XX 
            END DO
          END IF
        END IF
      END DO
      XM(2,N21) = X2
      XM(1,N23) = X3  
*
*  Compute matrix for inversion.
*
      DO I = 1, N
        DO J = 1, N
          K = I + J    
          IF(K.GT.N) THEN   
            XM(I,J) = XM(K-N,N22)
          ELSE
            XM(I,J) = XM(K,N21)
          END IF
        END DO
      END DO
*
* Call DOUBLE PRECISION matrix inversion routine.
*
      IF(N.NE.1) THEN
        CALL MLSRAR(N,XM,XM(1,N23),ITER,EPS,A,ITEST,0,XM(1,N+1))  
        IF(ITEST.GE.5) THEN
          CHISQ = - 2.0
          RETURN
        END IF
      ELSE
        XM(1,1) = 1.D0/XM(1,1)
        A(1) = XM(1,1)*XM(1,N23) 
      END IF
*
*  Compute Chi-square for resulting fit.
*
      CHISQ=0.0  
      DO I=1,NUMBER
        IF(YERR(I).GT.0.) THEN
          S = A(1)
          IF(N.NE.1) THEN     
            R=1.
            X1 = XDATA(I)
            DO J = 2, N  
              IF (NORM.EQ.0) THEN
                R=R*X1
              ELSE
                R=R*(X1-AVE)/SIGMA
              END IF
              S = S + A(J)*R
            END DO
          END IF
          CHISQ = CHISQ +((S-YDATA(I))/YERR(I))**2
        END IF
      END DO
*
* Error messages after inversion of the matrix XM (H in the write-up).
*
      IF (NORM.EQ.0 .OR. N.EQ.1) RETURN
*
*   Rescale coefficients if data scaling was requested.
*
      SIGMA = 1.D0/SIGMA
      AVE = -AVE*SIGMA
      L = N-1   
      DO I=1,L
        XM(I,1) = AVE**I
        XM(I,2) = 0.D0
      END DO
      XM(1,2) = 1.D0
      XM(N,2) = 0.D0
      DO I=1,L
        K=N-I+1
        DO J=2,K     
          XM(J,2) = XM(J,2) + XM(J-1,2)
        END DO
        K = I+1
        DO J = K,N
          A(I) = A(I) + A(J)*XM(J-I+1,2)*XM(J-I,1)
        END DO
        A(I) = A(I)*SIGMA**(I-1)
      END DO
      A(N) = A(N)*SIGMA**(N-1)
      RETURN 
      END
