      SUBROUTINE FRUIT( GCLONG,  GDLAT,   H,
     $                  EPOCH,   RAINP,   DECINP,
     $                  IYEAR,   IMONTH,  IDAY,    UTC )
C
C   FRUIT IS A GENERAL PURPOSE SUBROUTINE USED TO CALCULATE CELESTIAL OBJECT
C   COORDINATES, HOUR ANGLES, ELEVATIONS, AIRMASSES, VELOCITY CORRECTIONS,
C   HELIOCENTRIC TIME DELAYS AND JULIAN DATES. THE PERTINENT PARAMETERS ARE
C   INPUT VIA AN ARGUMENT LIST. THE DESIRED QUANTITIES CALCULATED BY FRUIT
C   ARE RETURNED IN COMMON BLOCKS. ALL ARITHMETIC IS IN REAL*8.
C   VALID TIME RANGE ARE YEARS 1800 TO 2200.
C
C   'FRUIT' IS ADAPTED FROM THE IBM 370 PROGRAM 'LOG' WRITTEN BY STEVE KENT.
C   THIS VERSION:   16 AUG 1979  BY  PETER YOUNG.
C               :   15 OCT 1980 PLACED ON ASTRONOMY VAX BY JEFF PIER.
C
C
C   GCLONG   =  LONGITUDE OF OBSERVING SITE IN DECIMAL DEGREES (REAL*8).
C   GDLAT    =  GEODETIC LATITUDE OF OBSERVING SITE IN DECIMAL DEGREES.
C               (REAL*8).
C   H        =  HEIGHT OF OBSERVING SITE IN METRES (REAL*8).
C   EPOCH    =  EPOCH (BESSELIAN DATE IN YEARS) OF COORDINATES TO BE INPUT.
C               (REAL*8).
C   RAINP    =  RIGHT ASCENSION OF OBJECT IN DECIMAL HOURS (REAL*8).
C   DECINP   =  DECLINATION OF OBJECT IN DECIMAL DEGRESS (REAL*8).
C               NOTE THAT POSITION MUST BE MEAN COORDINATES.
C   IYEAR    =  YEAR OF OBSERVATION (INTEGER*4).
C   IMONTH   =  MONTH OF OBSERVATION (INTEGER*4).
C   IDAY     =  DAY OF OBSERVATION (INTEGER*4).
C   UTC      =  COORDINATED UNIVERSAL TIME OF OBSERVATION IN DECIMAL HOURS.
C               (REAL*8).
C
      IMPLICIT REAL*8(A-H,O-Z)
      INTEGER*2 PRINTR
      REAL*8 XINP(3),XREF(3),XMEAN(3),XTRUE(3),
     $PREC(3,3),NUT(3,3),VEL(3),SUN(3)
C
C   RA0,DEC0 =  EPOCH 1950.0 POSITION OF OBJECT (DECIMAL HOURS, DEGREES).
C               THESE ARE MEAN COORDINATES (REAL*8).
C
      COMMON/FRUIT1/RA0,DEC0
C
C   RA,DEC   =  MEAN COORDINATES FOR EPOCH OF TIME OF OBSERVATION.
C   FPOCH    =  BESSELIAN DATE IN YEARS FOR THESE COORDINATES.
C
      COMMON/FRUIT2/RA,DEC,FPOCH
C
C   RAT,DECT =  TRUE COORDINATES AT TIME OF OBSERVATION, CORRECTED FOR
C               NUTATION.
C
      COMMON/FRUIT3/RAT,DECT
C
C   RAA,DECA =  ABERRATION CORRECTED COORDINATES AT TIME OF OBSERVATION.
C
      COMMON/FRUIT4/RAA,DECA
C
C   RAR,DECR =  REFRACTION CORRECTED COORDINATES FOR POSITION AND TIME OF
C               OBSERVATION. VALUES RETURNED ONLY IF ELEVATION ABOVE -35 ARCMIN.
C
      COMMON/FRUIT5/RAR,DECR
C
C   GLONG    =  GALACTIC LONGITUDE (SYSTEM II) OF OBJECT.
C   GLAT     =  GALACTIC LATITUDE (SYSTEM II) OF OBJECT.
C
      COMMON/FRUIT6/GLONG,GLAT
C
C   RJD      =  JULIAN DATE AT TIME OF OBSERVATION (DECIMAL DAYS). (REAL*8)
C   HJD      =  HELIOCENTRIC JULIAN DATE (DECIMAL DAYS).
C   TDELAY   =  TIME DELAY (RJD - HJD). (REAL*8). UNITS ARE DAYS.
C   VHELC    =  HELIOCENTRIC VELOCITY CORRECTION (SUBTRACT FROM OBSERVED
C               VELOCITY TO GET TRUE VELOCITY). (REAL*8). KM/SEC.
C
      COMMON/FRUIT7/RJD,HJD,TDELAY,VHELC
C
C   SIDTIM   =  LOCAL SIDERIAL TIME OF OBSERVATION (DECIMAL HOURS). (REAL*8)
C   HA       =  HOUR ANGLE OF OBJECT (DECIMAL HOURS). (REAL*8)
C
      COMMON/FRUIT8/SIDTIM,HA
C
C   EL       =  ELEVATION OF OBJECT (TRUE, NOT REFRACTED). DECIMAL DEGREES.
C   AZ       =  AZIMUTH OF OBJECT (DECIMAL DEGREES). BOTH ARE REAL*8.
C   ELR      =  REFRACTION CORRECTED ELEVATION (REAL*8). VALUE IS RETURNED
C               ONLY IF TRUE ELEVATION ABOVE -35 ARCMIN.
C   AIRMSS   =  AIRMASS OF OBSERVATION. VALUE RETURNED ONLY IF TRUE ELEVATION
C               OF OBJECT ABOVE -35 ARCMIN.
C
      COMMON/FRUIT9/EL,AZ,ELR,AIRMSS
C
C   GET JULIAN DATE AT TIME OF OBSERVATION.
C
      DATA PRINTR/2/
      CALL JULIAN(IYEAR,IMONTH,IDAY,UTC,RJD)
      IF (RJD.LT.2378494.D0.OR.RJD.GT.2524959.D0) THEN
      WRITE (PRINTR,200) RJD
 200  FORMAT (' FRUIT: JULIAN DATE: ',F15.6,' OUTSIDE ALLOWED RANGE')
      PRINT 200,RJD
      RETURN
      END IF
C
C   BESSELIAN DATE (YEARS) OF TIME OF OBSERVATION.
C
      FPOCH=(RJD-2433282.423378D0)/365.2421988D0+1950.D0
C
C   GET COORDINATE VECTOR OF INPUT RIGHT ASCENSION, DECLINATION.
C
      CALL VECTOR(XINP,RAINP,DECINP,1)
C
C   GET JULIAN DATE OF COORDINATE EPOCH (IN 'RJDEP').
C
      RJDEP=(EPOCH-1950.D0)*365.2421988D0+2433282.423378D0
      IF (RJDEP.LT.2378494.D0.OR.RJDEP.GT.2524959.D0) THEN
      WRITE (PRINTR,201) RJDEP
 201  FORMAT (' FRUIT: EPOCH J.D.: ',F15.6,' OUTSIDE ALLOWED RANGE')
      PRINT 201,RJDEP
      RETURN
      END IF
C
C   GET 1950.0 MEAN COORDINATES FROM INPUT COORDINATES.
C
      CALL PRECES(RJDEP,PREC,NUT,GCLONG,GDLAT,H,VEL,SUN)
      CALL ROTAT(XINP,XREF,PREC,-1)
      CALL VECTOR(XREF,RA0,DEC0,-1)
C
C   GET GALACTIC COORDINATES.
C
      CALL GALAXY(RA0,DEC0,GLAT,GLONG,1)
C
C   PRECESS 1950.0 COORDINATES TO GET MEAN COORDINATES AT TIME OF OBSERVATION.
C
      CALL PRECES(RJD,PREC,NUT,GCLONG,GDLAT,H,VEL,SUN)
      CALL ROTAT(XMEAN,XREF,PREC,1)
      CALL VECTOR(XMEAN,RA,DEC,-1)
C
C   NUTATE MEAN COORDINATES TO GET TRUE COORDINATES.
C
      CALL ROTAT(XTRUE,XMEAN,NUT,1)
      CALL VECTOR(XTRUE,RAT,DECT,-1)
C
C   CORRECT TRUE COORDINATES FOR ABERRATION.
C
      CALL ABERR(RAT,DECT,VEL,RAA,DECA)
C
C   GET SIDERIAL TIME, HOUR ANGLE, ALTITUDE, AZIMUTH OF OBJECT.
C
      CALL AZEL(RJD,GCLONG,GDLAT,RAA,DECA,SIDTIM,HA,EL,AZ,RAR,DECR,
     $ELR,AIRMSS)
C
C   HELIOCENTRIC VELOCITY CORRECTION (SUBTRACT FROM OBSERVED VELOCITY TO
C   GET TRUE VELOCITY). KM/SEC.
C
      VHELC=-(VEL(1)*XMEAN(1)+VEL(2)*XMEAN(2)+VEL(3)*XMEAN(3))
C
C   CALCULATE HELIOCENTRIC TIME DELAY (SUBTRACT FROM JULIAN DATE TO GET
C   HELIOCENTRIC JULIAN DATE). UNITS ARE DAYS.
C
      TDELAY=(SUN(1)*XMEAN(1)+SUN(2)*XMEAN(2)+SUN(3)*XMEAN(3))
      HJD=RJD-TDELAY
      RETURN
      END
 
      SUBROUTINE PRECES(RJD,PREC,NUT,GCLONG,GDLAT,H,VEL,SUN)
C
C   PRECES COMPUTES A PRECESSION MATRIX FOR A PARTICULAR TIME. THIS MATRIX
C   MAY BE APPLIED TO OBJECT COORDINATE VECTORS TO CONVERT THOSE COORDINATES
C   EITHER FROM OR TO 1950.0 COORDINATES. NOTE THAT PRECESS OPERATES ON
C   MEAN COORDINATES.
C   NUTATION MATRIX IS ALSO COMPUTED, TO BE APPLIED TO COORDINATE VECTORS
C   WHICH HAVE ALREADY BEEN PRECESSED. ONLY THE LARGEST TERMS ARE CALCULATED.
C   APPROXIMATE ERRORS ARE 0.1 ARCSEC IN 'DPSI' AND 0.05 ARCSEC IN 'DEPS'.
C   IN ADDITION THE VELOCITY VECTOR OF THE OBSERVING SITE IS COMPUTED
C   RELATIVE TO THE SUN IN KM/SEC REFERRED TO THE MEAN ECLIPTIC AND EQUUATOR
C   OF DATE (GEOCENTRIC COORDINATES). VALUE ACCURATE TO 0.04 KM/SEC (RESIDUAL
C   DUE TO PLANETARY PERTURBATIONS OF BOTH EARTH AND SUN).
C   ALSO THE RECTANGULAR COORDINATES OF THE SUN RELATIVE TO THE EARTH ARE
C   COMPUTED (IN DAYS OF LIGHT TRAVEL TIME) REFERRED TO MEAN EQUATOR AND
C   ECLIPTIC OF DATE. ACCURACY IS ABOUT 3 SECONDS (RESIDUAL DUE TO PLANETARY
C   PERTURBATIONS OF BOTH SUN AND EARTH).
C
C   RJD    =  JULIAN DATE (DECIMAL DAYS) OF DESIRED EPOCH (REAL*8).
C   PREC   =  3 BY 3 PRECESSION MATRIX FOR EPOCH RJD. IF YOU INPUT THE RJD
C             FOR EPOCH 1950.0 THEN 'PREC' IS THE IDENTITY MATRIX.
C   NUT    =  3 BY 3 NUTATION MATRIX. (REAL*8).
C   GCLONG =  LONGITUDE OF OBSERVING SITE (REAL*8) IN DECIMAL DEGREES.
C   GDLAT  =  GEODETIC LATITUDE OF SITE (REAL*8) IN DECIMAL DEGREES.
C   H      =  HEIGHT OF SITE IN METRES (REAL*8).
C   VEL    =  3-COORDINATE VECTOR (REAL*8) CONTAINING EARTH'S VELOCITY
C             VECTOR IN KM/SEC.
C   SUN    =  RECTANGULAR COORDINATES OF SUN FROM EARTH (3-D VECTOR)
C             MEASURED IN DAYS (LIGHT TRAVEL TIME). (REAL*8).
C
C   IF AN OBJECT HAS MEAN COORDINATE VECTOR  X(I) THEN:
C   TO PRECESS FROM 1950.0 TO 'RJD': XNEW(I)=PREC(I,J)*X(J) SUM ON J.
C   TO PRECESS FROM 'RJD' TO 1950.0: X(J)=XNEW(I)*PREC(I,J) SUM ON I.
C   TO NUTATE MEAN VECTOR: XTRUE(I)=XMEAN(I)*NUT(I,J)*XMEAN(J)
C
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 NUT(3,3),V1(3),SUN(3),VEL(3),PREC(3,3)
      DATA ETUT/46.D0/
      DATA PI,TWOPI/3.1415926535897932D0,6.2831853071795865D0/
      DATA CONVD,CONVDS/1.7453292519943296D-2,4.8481368110953599D-6/
      DATA CONVHS/7.2722052166430399D-5/
C
C   STATEMENT FUNCTIONS FOR ANGLES AND ORBITAL ELEMENTS.
C
C   PRECESSION CONSTANTS: X IN TROPICAL CENTURIES FROM 1950.0.
      ZETA(X)=X*(2.304948D3+X*(0.3020D0+X*0.179D-1))*CONVDS
      Z(X)=X*(2.304948D3+X*(1.093D0+X*0.192D-1))*CONVDS
      THETA(X)=X*(2.004255D3+X*(-0.426D0-X*0.416D-1))*CONVDS
C
C   MEAN OBLIQUITY OF EARTH RELATIVE TO MEAN ECLIPTIC OF DATE, X IN JULIAN
C   CENTURIES FROM 1900 JAN0.5
      OBLQM(X)=(23.452294D0+X*(-0.0130125D0+X*(-1.64D-6+X*5.03D-7)))
     $*CONVD
C
C   ORBITAL ELEMENTS OF EARTH-MOON BARYCENTER RELATIVE TO SUN REFERRED TO
C   MEAN QUINOX AND ECLIPTIC OF DATE.  X IN JULIAN DAYS SINCE 1900 JAN0.5.
C   INCLINATION AND LONGITUDE OF ASCENDING NODE ARE ZERO BY DEFINITION.
C   ECCENTRICITY:
      ES(X)=0.01675104D0+X*(-1.1444D-9-X*9.4D-17)
C   ARGUMENT OF PERIHELION (ADD 180 DEG TO THIS ONLY TO GET SUN RELATIVE
C   TO EARTH).
      WS(X)=(101.220833D0+X*(4.70684D-5+X*(3.39D-13+X*7.D-20)))*CONVD
C   MEAN ANOMALY:
      ANOMS(X)=(358.475845D0+X*(0.985600267D0+X*(-1.12D-13-X*7.D-20)))
     $*CONVD
C
C   ELEMENTS OF MOON RELATIVE TO EARTH, SAME REFERENCE AND EPOCH:
C   LONGITUDE OF ASCENDING NODE:
      OMEGAM(X)=(259.183275D0+X*(-0.0529539222D0+X*(1.557D-12+X*5.D-20))
     $)*CONVD
C   ARGUMENT OF PERIGEE:
      WM(X)=(75.146281D0+X*(0.1643580025D0+X*(-9.296D-12-X*3.1D-19)))
     $*CONVD
C   MEAN ANOMALY:
      ANOMM(X)=(-63.895392D0+X*(13.0649924465D0+X*(6.889D-12+X*2.99D-19)
     $))*CONVD
C
C   CONVERSION FROM MEAN TO TRUE ANOMALY (APPROX.):
      ATRUE(X,E)=X+(2.D0*E-0.25D0*E**3)*DSIN(X)+1.25D0*E**2*DSIN(2.D0*X)
     $+13.D0/12.D0*E**3*DSIN(3.D0*X)
C
C   CONSTANTS OF THE ORBITS
C   ECCENTRICITY OF MOON
      E2=0.054900489D0
C   INCLINATION OF MOON
      AIM=5.14539122*CONVD
C   VALUES FOR SEMI-MAJOR AXES FROM MIKE ASH
C   SUN-EARTH-MOON BARYCENTER (KM)
      AS=499.00478D0*2.99792456D5*1.00000023D0
C   DITTO (IN DAYS OF LIGHT TRAVEL TIME).
      SUNDIS=499.00478D0*1.00000023D0/86400.D0
C   EARTH-MOON
      AM=60.2665D0*6378.16D0
C   MASSES IN KM**3/SEC**2
C   EARTH
      EMASS=398601.2D0
      SMASS=EMASS*328900.D0*82.3D0/81.3D0
      AMMASS=EMASS/81.3D0
C   TEST FOR VALID TIME RANGE.
      IF (RJD.LT.2378494.D0.OR.RJD.GT.2524959.D0) RETURN
C
C   T IS EPHEMERIS TIME IN DAYS SINCE 1900 JAN0.5 ET. (RJD=2415020.0).
C   ET=UT+46 SEC (APPROX. 1975) MORE ACCURATE VALUE CAN BE FOUND IN AENA.
      T=RJD-2415020.D0+ETUT/3600.D0
C   TY=T IN JULIAN YEARS
      TY=T/365.25
C   TC=T IN CENTURIES.
      TC=TY/100.D0
C   TTP=T IN TROPICAL CENTURIES SINCE 1950.0
      TTP=(T-18262.423378)/36524.21988D0
C
C   COMPUTE PRECESSION MATRIX.
C
      Z1=ZETA(TTP)
      Z2=Z(TTP)
      Z3=THETA(TTP)
      CZT=DCOS(Z1)
      SZT=DSIN(Z1)
      CZ=DCOS(Z2)
      SZ=DSIN(Z2)
      CT=DCOS(Z3)
      ST=DSIN(Z3)
      Q1=CZ*CT
      Q2=SZ*CT
      PREC(1,1)=CZT*Q1-SZT*SZ
      PREC(1,2)=-SZT*Q1-CZT*SZ
      PREC(1,3)=-CZ*ST
      PREC(2,1)=CZT*Q2+SZT*CZ
      PREC(2,2)=CZT*CZ-SZT*Q2
      PREC(2,3)=-ST*SZ
      PREC(3,1)=CZT*ST
      PREC(3,2)=-SZT*ST
      PREC(3,3)=CT
C   LONGITUDE OF ASCENDING NODE OF MOON.
      O2=OMEGAM(T)
C   MEAN LONGITUDE OF MOON:
      SMOON=O2+WM(T)+ANOMM(T)
C   MEAN LONGITUDE OF SUN:
      SSUN=WS(T)+ANOMS(T)
C   NUTATION CONSTANTS FROM ESE:
      DPSI=-(17.233D0+0.017D0*TC)*DSIN(O2)+0.209D0*DSIN(2.D0*O2)-1.273D0
     $*DSIN(2.D0*SSUN)-0.204D0*DSIN(2.D0*SMOON)
      DEPS=(9.210D0+0.0009D0*TC)*DCOS(O2)-0.090D0*DCOS(2.D0*O2)+0.552D0
     $*DCOS(2.D0*SSUN)+0.088D0*DCOS(2.D0*SMOON)
      DPSI=DPSI*CONVDS
      DEPS=DEPS*CONVDS
C   TRUE OBLIQUITY
      OBLQ=OBLQM(TC)+DEPS
      CO=DCOS(OBLQ)
      SO=DSIN(OBLQ)
C
C   ROTATION MATRIX TO FIRST ORDER (SECOND ORDER CORRECTIONS ARE OF
C   MAGNITUDE 1.D-8 RADIANS).
C
      NUT(1,1)=1.D0
      NUT(1,2)=-DPSI*CO
      NUT(1,3)=-DPSI*SO
      NUT(2,1)=-NUT(1,2)
      NUT(2,2)=1.D0
      NUT(2,3)=-DEPS
      NUT(3,1)=-NUT(1,3)
      NUT(3,2)=DEPS
      NUT(3,3)=1.D0
C
C   CALCULATE GEOCENTRIC LATITUDE AND RADIUS ARM OF SITE.
C
      F=1.D0/298.25D0
      A=6378.16D0
      HH=H*1.D-3
      RLAT=GDLAT*CONVD
      CPHI=DCOS(RLAT)
      SPHI=DSIN(RLAT)
      C=1.D0/DSQRT(CPHI**2+((1.D0-F)*SPHI)**2)
      S=(1.D0-F)**2*C
      RAD=DSQRT(((A*S+HH)*SPHI)**2+((A*C+HH)*CPHI)**2)
C   GEOCENTRIC LATITUDE.
      GCLAT=DATAN2((A*S+HH)*DTAN(RLAT),A*C+HH)/CONVD
C   SIDEREAL TIME AT 0H UT (IN RADIANS).
      S= DBLE(NINT(RJD))-2415020.5D0
      UT=RJD-DBLE(NINT(RJD))+.5D0
      SIDTMO=1.73993589472D0+S*(1.7202791266D-2+S*5.06409D-15)
C   SFRACT=RATIO OF SIDEREAL TIME TO UTC IN RAD/SEC
      SFRACT=7.2921158546827D-5+S*1.1727115D-19
C   LOCAL SIDEREAL TIME. THIS EXPRESSION WOULD BE EXACT IF UT WERE UT1 AND
C   DPSI WAS CALCULATED MORE EXACTLY.
      OBLQ=OBLQM(TC)+DEPS

C Bugfix TRM, 06/11/2009. Where I have put 86400 it was incorrectly 3600

      SIDTIM=SIDTMO+SFRACT*UT*86400.D0+DCOS(OBLQ)*DPSI
      SIDTIM=SIDTIM-GCLONG*CONVD
      SIDTIM=DMOD(SIDTIM,TWOPI)
C   ROTATION RATE OF SITE IN KM/SEC
      VROT=RAD*DCOS(GCLAT*CONVD)*SFRACT
C   VECTOR VELOCITY REFERENCED TO TRUE EQUATOR AND ECLIPTIC.
      V1(1)=-VROT*DSIN(SIDTIM)
      V1(2)=VROT*DCOS(SIDTIM)
      V1(3)=0.D0
C   ROTATE TO MEAN EQUATOR AND ECLIPTIC
      DO 20 J=1,3
      VEL(J)=0.D0
      DO 20 I=1,3
 20   VEL(J)=VEL(J)+NUT(I,J)*V1(I)
C   VELOCITY OF E-M BARYCENTER RELATIVE TO SUN.
C   TO BE MORE ACCURATE VELOCITY OF SUN RELATIVE TO SS BARYCENTER SHOULD BE
C   COMPUTED.
      E1=ES(T)
      W1=WS(T)
      A1=ANOMS(T)
C   TRUE ANOMALY.
      AV1=ATRUE(A1,E1)
      VREV=DSQRT((EMASS+SMASS+AMMASS)/(AS*(1.D0-E1*E1)))
C   VECTOR VELOCITY REFERENCED TO ECLIPTIC.
      V1(1)=-VREV*(DSIN(AV1+W1)+E1*DSIN(W1))
      V1(2)=VREV*(DCOS(AV1+W1)+E1*DCOS(W1))
C   ELEMENTS FOR MOON
      O2=OMEGAM(T)
      W2=WM(T)
      A2=ANOMM(T)
      AV2=ATRUE(A2,E2)
      VREV=DSQRT((EMASS+AMMASS)/(AM*(1.D0-E2*E2)))/82.3D0
      CO=DCOS(O2)
      SO=DSIN(O2)
      CI=DCOS(AIM)
      SI=DSIN(AIM)
      CL=DCOS(AV2+W2)
      SL=DSIN(AV2+W2)
      CW=DCOS(W2)
      SW=DSIN(W2)
C   ADD VELOCITY OF EARTH TO V1, MAKING SURE SIGN IS CORRECT.
      V1(1)=V1(1)+VREV*(CO*(SL+E2*SW)+CI*SO*(CL+E2*CW))
      V1(2)=V1(2)+VREV*(SO*(SL+E2*SW)-CI*CO*(CL+E2*CW))
      V1(3)=-VREV*SI*(CL+E2*CW)
C   ROTATE TO MEAN EQUATOR AND ECLIPTIC
      OBLQ=OBLQM(TC)
      CM=DCOS(OBLQ)
      SM=DSIN(OBLQ)
      VEL(1)=VEL(1)+V1(1)
      VEL(2)=VEL(2)+V1(2)*CM-V1(3)*SM
      VEL(3)=VEL(3)+V1(2)*SM+V1(3)*CM
C   CURRENT ELEMENTS OF EARTH'S ORBIT.
      E1=ES(T)
      OB1=OBLQM(TC)
      W1=WS(T)
      A1=ANOMS(T)
C   TRUE ANOMALY.
      AV1=ATRUE(A1,E1)
C   TRUE ECLIPTIC LONGITUDE OF SUN.
      SLONG=AV1+W1+180.D0*CONVD
C   DISTANCE TO SUN.
      SUND=SUNDIS*(1.D0-E1*E1)/(1.D0+E1*DCOS(AV1))
C   RECTANGULAR COORDINATES OF SUN (RELATIVE TO EARTH).
      SUN(1)=SUND*DCOS(SLONG)
      SUN(2)=SUND*DSIN(SLONG)*DCOS(OB1)
      SUN(3)=SUND*DSIN(SLONG)*DSIN(OB1)
      RETURN
      END
 
      SUBROUTINE JULIAN(IYEAR,IMONTH,IDAY,UTC,RJD)
C
C   'JULIAN' COMPUTES THE DECIMAL JULIAN DATE FROM A GREGORIAN CALENDAR
C   DATE AND TIME IN UTC. WORKS FOR ANY INPUT TIME.
C
C   IYEAR   =  YEAR NUMBER (INTEGER*4). IYEAR = 0 FOR 1 B.C. ETC.
C   IMONTH  =  MONTH NUMBER (INTEGER*4).
C   IDAY    =  DAY NUMBER (INTEGER*4).
C   UTC     =  COORDINATED UNIVERSAL TIME IN DECIMAL HOURS (REAL*8).
C   RJD     =  RETURNED JULIAN DATE (DECIMAL DAYS). REAL*8.
C
C   JULIAN DATE   RJD = 2305447.0  ON 1600 JAN 0.5 UTC (GREGORIAN
C   CALENDAR) IS THE FIDUCIAL POINT USED BY 'JULIAN'. FIDUCIAL POINT MUST
C   BE A YEAR DIVISIBLE BY 400.
C
      REAL*8 RJD,UTC
      INTEGER*2 MON(12)
      DATA MON/0,31,59,90,120,151,181,212,243,273,304,334/
      DATA IFID,JDFID/1600,2305447/
C   ADD EXTRA DAY FOR LEAP YEAR.
      LYR=0
      IF ((IYEAR/4)*4.EQ.IYEAR) LYR=1
      IF ((IYEAR/100)*100.EQ.IYEAR) LYR=0
      IF ((IYEAR/400)*400.EQ.IYEAR) LYR=1
      IF (IMONTH.LE.2) LYR=0
C   NDAY IS THE DAY NUMBER OF THE YEAR.
      NDAY=MON(IMONTH)+IDAY+LYR
C   GET J.D. FOR JAN 0.5 UTC IN REQUIRED YEAR.
      IF (IYEAR.GT.IFID) THEN
      IY=IYEAR-IFID
      RJD=JDFID+IY*365.D0+(IY-1)/4+1-(IY-1)/100+(IY-1)/400
      ELSE IF (IYEAR.EQ.IFID) THEN
      RJD=JDFID
      ELSE IF (IYEAR.LT.IFID) THEN
      IY=IFID-IYEAR
      RJD=JDFID-IY*365.D0-IY/4+IY/100-IY/400
      END IF
C   ADD ON THE DAYS AND HOURS COMPONENT.
      RJD=RJD+NDAY+UTC/24.D0-.5D0
      RETURN
      END
 
      SUBROUTINE ROTAT(X,Y,ROT,IDIR)
C
C   PERFORMS ROTATION X(I)=ROT(I,J)*Y(J) SUM ON J; IDIR=1.
C   PERFORMS ROTATION Y(J)=ROT(I,J)*X(I) SUM ON I; IDIR=-1.
C
      REAL*8 RA,DEC,CONVD
      REAL*8 RDEC,RRA,CDEC
      REAL*8 X(3),Y(3),ROT(3,3),XX(3)
      DATA CONVD/1.7453292519943296D-2/
      IF (IDIR.LT.0) GO TO 10
      DO 2 I=1,3
      X(I)=0.D0
      DO 2 J=1,3
2     X(I)=X(I)+ROT(I,J)*Y(J)
      RETURN
10    CONTINUE
      DO 12 J=1,3
      Y(J)=0.D0
      DO 12 I=1,3
12    Y(J)=Y(J)+ROT(I,J)*X(I)
      RETURN
      ENTRY VECTOR(XX,RA,DEC,MODE)
C
C   CONVERTS R.A. AND DEC. TO COORDINATE VECTOR XX(I): MODE=1
C   CONVERTS COORDINATE VECTOR XX(I) TO R.A. AND DEC.: MODE=-1
C
      IF (MODE.LT.0) GO TO 17
      RRA=RA*CONVD*15.D0
      RDEC=DEC*CONVD
      CDEC=DCOS(RDEC)
      XX(1)=CDEC*DCOS(RRA)
      XX(2)=CDEC*DSIN(RRA)
      XX(3)=DSIN(RDEC)
      RETURN
17    CONTINUE
      RA=DATAN2(XX(2),XX(1))/(CONVD*15.D0)
      IF (RA.LT.0.D0) RA=RA+24.D0
      DEC=DASIN(XX(3))/CONVD
      RETURN
      END
 
      SUBROUTINE AZEL(RJD,GCLONG,GDLAT,RA,DEC,SIDTIM,HA,EL,AZ,
     $RAR,DECR,ELR,AIRMSS)
C
C   'AZEL' COMPUTES THE SIDERIAL TIME, HOUR ANGLE, TRUE (NOT REFRACTED)
C   ELEVATION AND AZIMUTH OF AN OBSERVED OBJECT. IF TRUE ELEVATION ABOVE
C   -35 ARCMIN IT ALSO COMPUTES REFRACTED COORDINATES, ELEVATION AND AIRMASS.
C
C   RJD     =  JULIAN DATE (DECIMAL DAYS) OF OBSERVATION. (REAL*8).
C   GCLONG  =  LONGITUDE (DECIMAL DEGREES) OF OBSERVING SITE (REAL*8).
C   GDLAT   =  GEODETIC LATITUDE OF SITE (DECIMAL DEGREES). (REAL*8).
C   RA      =  RIGHT ASCENSION (DECIMAL HOURS) OF OBJECT. (REAL*8).
C   DEC     =  DECLINATION (DECIMAL HOURS) OF OBJECT. (REAL*8).
C              NOTE THAT THESE SHOULD BE TRUE COORDINATES FOR THE EPOCH,
C              CORRECTED FOR ABERRATION.
C   SIDTIM  =  LOCAL SIDERIAL TIME AT SITE AT TIME 'RJD'. (REAL*8).
C   HA      =  HOUR ANGLE OF OBJECT (REAL*8). HA, SIDTIM IN DECIMAL HOURS.
C   EL      =  TRUE (UNREFRACTED) ELEVATION OF OBJECT IN DECIMAL DEGREES.
C              (REAL*8).
C   AZ      =  AZIMUTH OF OBJECT IN DECIMAL DEGREES (REAL*8).
C   RAR      =  REFRACTION CORRECTED R.A. (DECIMAL HOURS). REAL*8.
C   DECR     =  REFRACTION CORRECTED DEC. (DECIMAL DEGREES). REAL*8.
C   ELR      =  REFRACTED ELEVATION OF OBJECT.
C   AIRMSS   =  VALUE OF THE AIRMASS (REAL*8) RETURNED.
C
      IMPLICIT REAL*8 (A-H,O-Z)
      DATA PI,TWOPI/3.1415926535897932D0,6.2831853071795865D0/
      DATA CONVD,CONVDS/1.7453292519943296D-2,4.8481368110953599D-6/
      DATA CONVHS/7.2722052166430399D-5/
C   TEST FOR VALID TIME RANGE.
      IF (RJD.LT.2378494.D0.OR.RJD.GT.2524959.D0) RETURN
C   CONVERT INPUT JULIAN DATE.
      T=DBLE(NINT(RJD))-2415020.5D0
      UTC=(RJD-DBLE(NINT(RJD))+.5D0)*24.D0
C   CALCULATE LOCAL SIDERIAL TIME.
      SIDTMO=1.73993589472D0+T*(1.7202791266D-2+T*5.06409D-15)
      SFRACT=7.2921158546827D-5+T*(1.1727115D-19)
      SIDTIM=((SIDTMO+SFRACT*UTC*3600.D0)/CONVD-GCLONG)/15.D0
      SIDTIM=DMOD(SIDTIM,24.D0)
      IF (SIDTIM.LT.0.D0) SIDTIM=SIDTIM+24.D0
C   HOUR ANGLE
      HA=SIDTIM-RA
      IF (HA.LE.-12.D0) HA=HA+24.D0
      IF (HA.GT.12.D0) HA=HA-24.D0
      RHA=HA*15.D0*CONVD
      CHA=DCOS(RHA)
      SHA=DSIN(RHA)
      RDEC=DEC*CONVD
      CDEC=DCOS(RDEC)
      SDEC=DSIN(RDEC)
      RLAT=GDLAT*CONVD
      CLAT=DCOS(RLAT)
      SLAT=DSIN(RLAT)
C   COSINE OF ZENITH ANGLE.
      COSZ=SLAT*SDEC+CLAT*CDEC*CHA
      EL=DASIN(COSZ)/CONVD
      AZ=DATAN2(-CDEC*SHA,SDEC*CLAT-CDEC*CHA*SLAT)/CONVD
      IF (AZ.LT.0.D0) AZ=AZ+360.D0
      IF (EL.LT.-35.D0/60.D0) RETURN
C   COMPUTE REFRACTED ELEVATION AND AIRMASS.
      IF (EL.GT.4.D0+50.D0/60.D0) THEN
      SINZ=DSIN((90.D0-EL)*CONVD)
      TANZ=SINZ/COSZ
      ELR=EL+(58.3*TANZ-0.067*TANZ**3)/3600.D0
      SECZ=1.D0/COSZ
      CHI=SECZ-1.D0
      AIRMSS=SECZ-CHI*(.0018167D0+CHI*(.002875D0+CHI*.0008083D0))
      ELSE
      ZT=90.D0-EL
      THETA=-287.8165098D0+7.792832093D0*ZT-0.0401680058D0*ZT*ZT
      AIRMSS=1.0D0/DCOS(THETA*CONVD)
      ELR=EL+55.2053*AIRMSS/3600.D0
      END IF
      CAZ=DCOS(AZ*CONVD)
      SAZ=DSIN(AZ*CONVD)
      CELR=DCOS(ELR*CONVD)
      SELR=DSIN(ELR*CONVD)
C   COSINE OF ANGLE FROM NORTH POLE.
      CPOLE=SLAT*SELR+CLAT*CELR*CAZ
      DECR=DASIN(CPOLE)/CONVD
C   NEW HOUR ANGLE.
      HANEW=DATAN2(-CELR*SAZ,SELR*CLAT-CELR*CAZ*SLAT)/(CONVD*15.D0)
C   NEW R.A.
      RAR=DMOD(SIDTIM-HANEW,24.D0)
      IF (RAR.LT.0.D0) RAR=RAR+24.D0
      RETURN
      END
 
      SUBROUTINE GALAXY(RA,DEC,GLAT,GLONG,IDIR)
C
C   'GALAXY' CONVERTS EPOCH 1950.0 MEAN R.A. AND DEC. TO GALACTIC
C   COORDINATES, AND VICE VERSA.
C
C   RA     =  EPOCH 1950.0 R.A. (DECIMAL HOURS). (REAL*8).
C   DEC    =  EPOCH 1950.0 DEC. (DECIMAL DEGREES). (REAL*8).
C   GLAT   =  GALACTIC LATITUDE (SYSTEM II). (REAL*8).
C   GLONG  =  GALACTIC LONGITUDE (SYSTEM II). (REAL*8).
C   IDIR   =  MODE SELECTOR (INTEGER*4). IF IDIR = -1 THEN CONVERT GALACTIC
C             TO EQUATORIAL COORDINATES. OTHERWISE CONVERT EQUATORIAL TO
C             GALACTIC COORDINATES.
C
      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION R(3,3),G(3),E(3)
      DATA R/-.066988740D0,.492728466D0,-.867600811D0,-.872755766D0,
     $-.450346958D0,-.188374601D0,-.483538915D0,.744584633D0,
     $.460199785D0/
      DATA RADDEG/57.29577951D0/,RADHR/3.8197186335D0/
C
C   GALACTIC   COORDINATES PI,THETA,Z IN G(3),EQUATORIAL X,Y,Z IN E(3).
C   CONVERSION: G(I)=R(I,J)*E(J), SUM ON J.
C   INVERSE:    E(J)=R(I,J)*G(I), SUM ON I.
C
      IF (IDIR.EQ.-1) GO TO 50
      RRA=RA/RADHR
      RDEC=DEC/RADDEG
      CDEC=DCOS(RDEC)
      E(1)=CDEC*DCOS(RRA)
      E(2)=CDEC*DSIN(RRA)
      E(3)=DSIN(RDEC)
      DO 5 I=1,3
      G(I)=0.
      DO 5 J=1,3
5     G(I)=G(I)+E(J)*R(I,J)
      GLAT=DASIN(G(3))*RADDEG
      GLONG=DATAN2(G(2),G(1))*RADDEG
      IF (GLONG.LT.0.D0) GLONG=GLONG+360.D0
      RETURN
50    CONTINUE
      RGLAT=GLAT/RADDEG
      RGLONG=GLONG/RADDEG
      CLAT=DCOS(RGLAT)
      G(1)=CLAT*DCOS(RGLONG)
      G(2)=CLAT*DSIN(RGLONG)
      G(3)=DSIN(RGLAT)
      DO 55 J=1,3
      E(J)=0.
      DO 55 I=1,3
55    E(J)=E(J)+R(I,J)*G(I)
      DEC=DASIN(E(3))*RADDEG
      RA=DATAN2(E(2),E(1))*RADHR
      IF (RA.LT.0.) RA=RA+24.D0
      RETURN
      END
 
      SUBROUTINE ABERR(RA,DEC,VEL,RAA,DECA)
C
C   'ABERR' CORRECTS THE COORDINATES OF AN OBJECT FOR THE SPECIAL RELATIVISTIC
C   ABERRATION OF LIGHT.
C
C   RA     =  RIGHT ASCENSION OF OBJECT (DECIMAL HOURS). (REAL*8).
C   DEC    =  DECLINATION OF OBJECT (DECIMAL DEGREES). (REAL*8).
C   VEL    =  3-D VELOCITY VECTOR OF EARTH IN KM/SEC IN RECTANGULAR
C             GEOCENTRIC COORDINATES.
C   RAA    =  ABERRATION CORRECTED R.A. (DECIMAL HOURS). (REAL*8).
C   DECA   =  ABERRATION CORRECTED DEC. (DECIMAL DEGREES). (REAL*8).
C
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 R(3),RNEW(3),VEL(3)
      DATA C/299792.456D0/
      DATA CONVD,CONVDS/1.7453292519943296D-2,4.8481368110953599D-6/
C
C   CALCULATE OBJECT COORDINATE VECTOR FROM RA,DEC.
C
      R(1)=DCOS(DEC*CONVD)*DCOS(RA*CONVD*15.D0)
      R(2)=DCOS(DEC*CONVD)*DSIN(RA*CONVD*15.D0)
      R(3)=DSIN(DEC*CONVD)
C
C   PHOTONS FROM OBJECT HAVE VELOCITY VECTOR = -C*R(I).
C   VELOCITY VECTOR OF EARTH = VEL(I).
C   USE SPECIAL RELATIVISTIC VELOCITY TRANSFORM TO GET VELOCITY VECTOR
C   OF PHOTONS AS SEEN FROM EARTH  = RNEW(I).
C
      VE2=VEL(1)**2+VEL(2)**2+VEL(3)**2
      GAMMA2=1.D0/DSQRT(1.D0-VE2/(C*C))
      DELTA2=1.D0/(C*C*(1.D0+1.D0/GAMMA2))
      V12=-(R(1)*VEL(1)+R(2)*VEL(2)+R(3)*VEL(3))*C
      GAM12=1.D0-V12/(C*C)
      RNEW(1)=(V12*DELTA2*VEL(1)+VE2*DELTA2*C*R(1)-C*R(1)-VEL(1))/GAM12
      RNEW(2)=(V12*DELTA2*VEL(2)+VE2*DELTA2*C*R(2)-C*R(2)-VEL(2))/GAM12
      RNEW(3)=(V12*DELTA2*VEL(3)+VE2*DELTA2*C*R(3)-C*R(3)-VEL(3))/GAM12
C
C   NEW COORDINATE VECTOR = -RNEW(I)/C.
C
      RNEW(1)=-RNEW(1)/C
      RNEW(2)=-RNEW(2)/C
      RNEW(3)=-RNEW(3)/C
C
C   TRANSFORM NEW COORDINATE VECTOR INTO R.A. AND DEC. VALUES.
C
      RAA=DATAN2(RNEW(2),RNEW(1))/(CONVD*15.D0)
      IF (RAA.LT.0.D0) RAA=RAA+24.D0
      DECA=DASIN(RNEW(3))/CONVD
      RETURN
      END
