*SINE
* SINE N1 N2 GAMMA KX KY Phase -- Removes sinusoidal velocity from spectra
*
*  Parameters:
*
*           N1, N2 -- Range of spectra to treat
*           GAMMA  -- Systemic velocity to remove  (km/s)
*           KX     -- Sine amplitude               (km/s)
*           KY     -- Cosine amplitude             (km/s)
*           Phase  -- Name of phase to use (e.g. 'Orbital phase')
*
* Velocity removed is given by 
*
*       V = GAMMA - KX*COS(2*PI*PHI) + KY*SIN(2*PI*PHI) 
*
* SINE does nothing to the spectra, it just modifies the wavelength scales
* appropriately. If you want to average the results together with the 
* correct shifts, you should first rebin the results and then average them.
*
* If the (double precision) header parameter Phase is not found, the routine
* will shift out Gamma alone from the spectra.
*
* See also related command MOVE
*
*SINE                                                           
*MOVE
* MOVE N1 N2 File WHICH ADD -- Alters wavelength scales to give a 
*                              velocity, wavelength or pixel shift.
*
*  Parameters:
*
*           N1, N2 -- Range of spectra to treat
*           File   -- File of velocities (single column, one per spectrum)
*                     km/s. These are removed. If a file is not found, the
*                     program will try to translate the string given as
*                     a shift, if this fails it gives up.
*           WHICH  -- Type of shift V(elocity), W(avelength) or P(ixel)
*           ADD    -- 'A' to add the shift, 'S'  to subtract. 
*                     If WHICH=V, then ADD='A' causes the spectrum's velocity
*                     to be increased (i.e. shifted to red) 
*                     If WHICH=W, then ADD='A' causes the spectrum's 
*                     wavelengths to be increased (i.e. shifted to red) 
*                     If WHICH=P, then ADD='A' causes the spectrum to be 
*                     shifted to the right.
*                     (in each case for a positive value from the file)
*
* NB MOVE does **nothing** to the spectra, it just modifies the wavelength 
* scales appropriately. If you want to average the results together with the 
* correct shifts, you should first rebin the results and then average them.
* Note that it is not straightforward to shift a logarithmic scale by a
* constant wavelength increment and so MOVE does not attempt to do so.
* In such a case you should rebin to a normal wavelength scale first.
*
* See also related command SINE
*
*MOVE
      SUBROUTINE FIXVEL(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, 
     &NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, 
     &NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, 
     &NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, NMSINTR, VSINTR, NMSREAL, 
     &VSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL, MXSCHAR, MXSDOUB, 
     &MXSINTR, MXSREAL, DWORK, MXWORK, SLOTS, MXSLOTS, METHOD, IFAIL)
C
C Changes arc fits to remove velocity from a set of spectra. Velocities 
C can be read from a file or specified as a sine wave.
C 
C Arguments:
C
C >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C                              zero in which case defaults are used.
C >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C >  I     MXSPEC           -- Maximum number of spectra allowed by size
C                              of header buffers. 
C >  I     MAXPX            -- Maximum number of pixels/spectrum
C <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C <  I     NCHAR(MXSPEC)  -- Number of character header items.
C <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C <  I     NINTR(MXSPEC)  -- Number of integer items.
C <  I     NREAL(MXSPEC)  -- Number of real items.
C >  I     MXCHAR    -- Maximum number of character header items/spec
C >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C >  I     MXINTR    -- Maximum number of integer header items/spec
C >  I     MXREAL    -- Maximum number of real headers items/spec
C >  C*(*) NMSCHAR(MXSCHAR)   -- Names of character selection items.
C >  C*(*) VSCHAR(MXSCHAR)    -- Values of character selection items.
C >  C*(*) NMSDOUB(MXSDOUB)   -- Names of double precision selection items.
C >  D     VSDOUB(2,MXSDOUB)  -- Values of double precision selection items.
C >  C*(*) NMSINTR(MXSINTR)   -- Names of integer selection items.
C >  I     VSINTR(2,MXSINTR)  -- Values of integer selection items.
C >  C*(*) NMSREAL(MXSREAL)   -- Names of real selection items.
C >  R     VSREAL(2,MXSREAL)  -- Values of real selection items.
C >  I     NSCHAR             -- Number of character selection items.
C >  I     NSDOUB             -- Number of double precsion selection items.
C >  I     NSINTR             -- Number of integer selection items.
C >  I     NSREAL             -- Number of real selection items.
C >  I     MXSCHAR            -- Dimension of NMSCHAR in calling routine.
C >  I     MXSDOUB            -- Dimension of NMSDOUB in calling routine.
C >  I     MXSINTR            -- Dimension of NMSINTR in calling routine.
C >  I     MXSREAL            -- Dimension of NMSREAL in calling routine.   
C                                incremented and returned by routine.
C >  I     SLOTS(MXSLOTS)     -- List of slots
C >  I     MXSLOTS            -- Maximum number in list
C >  C     METHOD             -- 'F' file
C                                'S' sinusoid
C <  I     IFAIL              -- Error return.
C
      IMPLICIT NONE

C
C     Integer parameters
C
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL, MXSCHAR, MXSDOUB, MXSINTR
      INTEGER MXSREAL, NSCHAR, NSDOUB, NSINTR, NSREAL
      INTEGER MAXSPEC
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
      CHARACTER*(*) METHOD
C
C     Data arrays
C
      REAL COUNTS(MXBUFF), ERRORS(MXBUFF), FLUX(MXBUFF)
C
C     Numbers of header items
C
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C
C     Names of header items
C
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C
C     Values of header items
C
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C
C     Arc coefficients
C
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C
C     Slot lists
C
      INTEGER MXSLOTS, NSLOTS
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
C
C     Command parameters
C
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C
C     Search parameters, names then values.
C
      CHARACTER*(*) NMSCHAR(MXSCHAR)
      CHARACTER*(*) NMSDOUB(MXSDOUB)
      CHARACTER*(*) NMSINTR(MXSINTR)
      CHARACTER*(*) NMSREAL(MXSREAL)
C
      CHARACTER*(*) VSCHAR(MXSCHAR)
      DOUBLE PRECISION VSDOUB(2,MXSDOUB)
      INTEGER VSINTR(2,MXSINTR)
      REAL VSREAL(2,MXSREAL)
C
      INTEGER MXWORK
      DOUBLE PRECISION DWORK(MXWORK)
C
C     Local variables
C
      INTEGER I, J, K
      CHARACTER*64 FILENAME
      CHARACTER*3 WHICH, ADD
      INTEGER MAXVAR
      PARAMETER (MAXVAR=20)
      INTEGER SLOT1, SLOT2, IFAIL,   NCOM
      INTEGER  LUNIT,  SLOT
      LOGICAL SELECT, DEFAULT, OPEN, ESAMECI
      REAL    GAMMA, KX, KY, TWOPI
      DOUBLE PRECISION PHASE, VLIGHT, SHIFT, VFAC
      DOUBLE PRECISION SUM, FAC, BETA, PHI
      CHARACTER*32 PNAME
C
C     Functions
C
      INTEGER LENSTR
      REAL CGS
C
      DATA SLOT1, SLOT2/1,1/
      DATA FILENAME/'VEL.LIS'/
      DATA GAMMA, KX, KY/0.,0.,0./
      DATA PNAME/'Orbital phase'/
      DATA WHICH, ADD/'v', 's'/
      DATA LUNIT/29/
C
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM = 0
      CALL INTR_IN('First slot to shift', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to shift', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(DEFAULT .AND. NLIST.GT.0) THEN
            CALL SETSLOT(LIST, NLIST, MXLIST, SLOTS, NSLOTS, MXSLOTS)
         ELSE
            WRITE(*,*) 'Enter list of spectra to shift'
            CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &           NSLOTS, MXSLOTS)
         END IF
         IF(NSLOTS.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &        NSLOTS, MXSLOTS)
      END IF
C     
      IF(METHOD.EQ.'F') THEN
         CALL CHAR_IN('Shift or file of shifts', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, FILENAME, IFAIL)
         OPEN(UNIT=LUNIT, FILE=FILENAME, STATUS='OLD', IOSTAT=IFAIL)
         IF(IFAIL.NE.0) THEN
            READ(FILENAME,*,IOSTAT=IFAIL) SHIFT
            IF(IFAIL.NE.0) THEN
               WRITE(*,*) 'Failed to open ',FILENAME
               WRITE(*,*) 'and could not translate as a number'
               GOTO 999
            END IF
            OPEN = .FALSE.
         ELSE
            OPEN = .TRUE.
         END IF
         CALL CHAR_IN('Shift what ? (v,w,p)', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, WHICH, IFAIL)
         IF(.NOT.ESAMECI(WHICH,'V') .AND. .NOT.ESAMECI(WHICH,'W')
     &        .AND. .NOT.ESAMECI(WHICH,'P')) THEN
            WRITE(*,*) 'Must answer either V for velocity, W for'
            WRITE(*,*) 'wavelength or P for pixel shifts.'
            GOTO 999
         END IF
         CALL CHAR_IN('A(dd) or S(ubtract) shifts ?', SPLIT, 
     &        NSPLIT, MXSPLIT, NCOM, DEFAULT, ADD, IFAIL)
         IF(.NOT.ESAMECI(ADD,'A') .AND. .NOT.ESAMECI(ADD,'S')) THEN
            WRITE(*,*) 'Must answer either A or S'
            GOTO 999
         END IF
         IF(ESAMECI(ADD,'S') .AND. .NOT.OPEN) SHIFT = - SHIFT
      ELSE IF(METHOD.EQ.'S') THEN
         WRITE(*,*) 'Will remove velocity determined by:'
         WRITE(*,*) 'V = GAMMA - KX*COS(2*PI*PHI) + KY*SIN(2*PI*PHI)'
         WRITE(*,*) ' '
         CALL REAL_IN('Gamma (km/s)', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &        DEFAULT, GAMMA, -1.E5, 1.E5, IFAIL)
         CALL REAL_IN('Kx (km/s)', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &        DEFAULT, KX, -1.E5, 1.E5, IFAIL)
         CALL REAL_IN('Ky (km/s)', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &        DEFAULT, KY, -1.E5, 1.E5, IFAIL)
         CALL CHAR_IN('Use which phase?', SPLIT, NSPLIT, MXSPLIT, NCOM, 
     &        DEFAULT, PNAME, IFAIL)
      ELSE
         WRITE(*,*) 'Unrecognised method = ',METHOD
         IFAIL = 1
         GOTO 999
      END IF
C     
C     Now change velocities
C     
      TWOPI  = 8.*ATAN(1.)
      VLIGHT = CGS('C')/1.E5
      DO I = 1, NSLOTS
         SLOT = SLOTS(I)
         IF(NPIX(SLOT).LE.0) THEN
            WRITE(*,*) 'Slot ',SLOT,' empty.'
            GOTO 200
         END IF
         IF(.NOT.SELECT(SLOT, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &        HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &        NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, 
     &        MXSPEC, NMSCHAR, VSCHAR, NMSDOUB, VSDOUB, NMSINTR, 
     &        VSINTR, NMSREAL, VSREAL, NSCHAR, NSDOUB, NSINTR, 
     &        NSREAL, MXSCHAR, MXSDOUB, MXSINTR, MXSREAL)) THEN
            WRITE(*,*) 'Slot ',SLOT,' skipped.'
            GOTO 200
         END IF
C     
         IF(NARC(SLOT).EQ.0) THEN
            WRITE(*,*) 'Slot ',SLOT,' has no wavelength scale'
            GOTO 200
         ELSE IF(METHOD.EQ.'F' .AND. ESAMECI(WHICH,'W') .AND. 
     &           NARC(SLOT).LT.0) THEN
            WRITE(*,*) 'Slot ',SLOT,' has a logarithmic scale.'
            WRITE(*,*) 'It is not possible to shift a logarithmic'
            WRITE(*,*) 'wavelength scale by a constant wavelength'
            WRITE(*,*) 'shift. Must rebin with WBIN first.'
            GOTO 200
         END IF
         IF(METHOD.EQ.'F') THEN
            IF(OPEN) THEN
               READ(LUNIT, *, IOSTAT=IFAIL) SHIFT
               IF(ESAMECI(ADD,'S')) SHIFT = - SHIFT
            END IF
         ELSE IF(METHOD.EQ.'S') THEN
C     
C     Get phase and HJD
C     
            CALL HGETD(PNAME, PHASE, SLOT, MXSPEC, NMDOUB, HDDOUB, 
     &           NDOUB, MXDOUB, IFAIL)
            IF(IFAIL.EQ.1) THEN
               WRITE(*,*) 'No [',PNAME(:LENSTR(PNAME)),
     &              '] in slot ',SLOT
               WRITE(*,*) 'Will shift out GAMMA alone.'
               SHIFT = -GAMMA
            ELSE IF(IFAIL.EQ.2) THEN
               WRITE(*,*) 'More than one match to [',
     &              PNAME(:LENSTR(PNAME)),'] in slot ',SLOT
               WRITE(*,*) 'Will shift out GAMMA alone.'
               SHIFT = -GAMMA
            ELSE
               PHI = PHASE - DBLE(NINT(PHASE))
               SHIFT = -(GAMMA - KX*COS(TWOPI*PHI) + KY*SIN(TWOPI*PHI))
            END IF
         END IF
C     
C     Now modify arc scale appropriately.
C     
         IF(METHOD.EQ.'S' .OR. (METHOD.EQ.'F' .AND. 
     &        ESAMECI(WHICH,'V'))) THEN
            VFAC = 1.D0 + SHIFT/VLIGHT   
            IF(NARC(SLOT).GT.0) THEN
               DO J = 1, NARC(SLOT)
                  ARC(J,SLOT) = VFAC*ARC(J,SLOT)
               END DO
            ELSE IF(NARC(SLOT).LT.0) THEN
               ARC(1,SLOT) = ARC(1,SLOT) + LOG(VFAC)
            END IF
            WRITE(*,*) 'Shifted slot ',SLOT,' by ',SNGL(SHIFT),' km/s.'
         ELSE IF(ESAMECI(WHICH,'W')) THEN
            ARC(1,SLOT) = ARC(1,SLOT) + SHIFT
            WRITE(*,*) 'Shifted slot ',SLOT,' by ',
     &           SNGL(SHIFT),' Angstroms.'
         ELSE IF(ESAMECI(WHICH,'P')) THEN
            IF(ABS(NARC(SLOT)).LT.MXWORK) THEN
               BETA = SHIFT/DBLE(NPIX(SLOT))
               DO J = 1, ABS(NARC(SLOT))
                  FAC = 1.D0
                  SUM = 0.D0
                  DO K = J, ABS(NARC(SLOT))
                     SUM = SUM + FAC*ARC(K,SLOT)
                     FAC = FAC*BETA*DBLE(K)/DBLE(K-J+1)
                  END DO
                  DWORK(J) = SUM
               END DO
               DO J = 1, ABS(NARC(SLOT))
                  ARC(J,SLOT) = DWORK(J)
               END DO
               WRITE(*,*) 'Shifted slot ',SLOT,' by ',
     &              SNGL(SHIFT),' pixels.'
            END IF
         END IF
 200     CONTINUE
      END DO
 999  CLOSE(UNIT=LUNIT)
      RETURN
      END
