CFITSPEC
C FITSPEC N1 N2 N3 N4 NG File NP PARAMS[NP] AXES[NP] FULL -- Fitspec estimates 
C                best-fit parameter values of some observed spectra relative 
C                to some model spectra tabulated on a cartesian grid. It assumes 
C                that spectra between the grid points can be represented by 
C                bi-linear interpolation.
C
C Parameters:
C
C   N1, N2    -- Range of observed spectra
C
C   N3, N4    -- Range of model spectra
C
C   NG        -- Number of spectra in each group. The spectra should be 
C                thought of in groups of NG spectra. e.g. if you had 
C                observations of Halpha and Hbeta with separate spectra
C                you would set NG = 2. You would require models for each
C                of course. All spectra of the same parameter should be
C                in one block. 
C
C   FILE      -- A file of best fit values can be dumped. <CR> to ignore;
C                ! will store the values in the headers instead which can 
C                then be used with 'GENSPEC' to compute the best-fit models.
C
C   NP        -- The number of parameters . The model spectra should on a 
C                Cartesian grid (i.e. equal steps in each parameter).
C                e.g you might have a grid of log g and log T with
C                the spectra stored to change first in log g and then log T
C                This would have NP=2.
C
C   PARAMS    -- Names of the parameters, which should correspond to the 
C                storage order, so that in the example above they would be
C                "log g" and "log T".  These should be REAL variables
C                with values stored in the headers of the model spectra.
C               
C   AXES      -- The number of points on each parameter axis. e.g. if you
C                stored 10 gravities at 5 temperature points, you would
C                enter 10, 5. *** NB *** take care over both 'PARAMS' and
C                'AXES' to get reliable results.
C
C   FULL      -- gives fuller output with chi**2 and best-fit values 
C                reported for every cuboid and for each observed spectrum.
C
C Checks are made for the correct match of pixels, but not for anything
C else so the onus is on the user to get the axes etc correct. The method
C used is to go through every cuboid of the model grid and carry out a
C non-linear optimisation assuming bi-linear type interpolation from the
C corners of the cuboids. The best few results are reported, the program
C distinguishing between those in and outside their defining cuboids.
C Ideally one should get just one fit within its cuboid (aka "non-extrapolated"),
C all others being extrapolated. The best of the extrapolated values should be 
C within spitting distance of the non-extrapolated values if the model grid is
C fine enough. More than 1 non-extrapolated value is listed in case of
C multiple minima.
C
C Uncertainties are calculated from covariance matrix. To mask points, mask the
C data only. The mask on the model spectra is ignored.
C
CFITSPEC
CGENSPEC
C GENSPEC N1 N2 N3 N4 NG NP PARAMS[NP] AXES[NP] MASK -- genspec interpolates
C                spectra in a grid according to values of header items.
C
C Parameters:
C
C   N1, N2    -- Spectra to set
C
C   N3, N4    -- Model spectra
C
C   NG        -- Number of spectra in each group. The spectra should be 
C                thought of in groups of NG spectra. e.g. if you had 
C                observations of Halpha and Hbeta with separate spectra
C                you would set NG = 2. You would require models for each
C                of course. All spectra of the same parameter should be
C                in one block. 
C
C   NP        -- The number of parameters . The model spectra should on a 
C                Cartesian grid (i.e. equal steps in each parameter).
C                e.g you might have a grid of log g and temperature with
C                the spectra stored to change first in log g and then Temp.
C                This would have NP=2
C
C   PARAMS    -- Names of the parameters, which should correspond to the 
C                storage order, so that in the example above they would be
C                "log g" and "Temperature".  These should be REAL variables
C                with values stored in the headers. ALL spectra (i.e. objects
C                and models) should have these variables set.
C
C   AXES      -- The number of points on each parameter axis. e.g. if you
C                stored 10 gravities at 5 temperature points, you would
C                enter 10, 5
C
C   MASK      -- Yes to retain whatever mask has been applied to the data
C                when computing the output interpolated spectra. If not models
C                are interpolated onto all available pixels.
C
C
CGENSPEC
      SUBROUTINE FITSPEC(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, 
     &     FLUX, MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, 
     &     NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, 
     &     NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, 
     &     DWORK1, DWORK2, MXWORK, SLOTS1, SLOTS2, MXSLOTS, CLOBBER,
     &     METHOD, IFAIL)
C     
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     >  D     DWORK1  Work array
C     >  D     DWORK2  Work array
C     >  I     MXWORK     Size of work array
C     I     SLOTS1(MXSLOTS) -- Work array for list of slots
C     I     SLOTS2(MXSLOTS) -- Work array for list of slots
C     I     MXSLOTS
C     <  I     IFAIL              -- Error return.
C     
      IMPLICIT NONE

C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL   FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT, NCOM
      CHARACTER*(*) SPLIT(MXSPLIT)
      CHARACTER*(*) METHOD
C     
      INTEGER MXWORK
      DOUBLE PRECISION DWORK1(MXWORK), DWORK2(MXWORK)
C     
C     Slot lists
C     
      INTEGER SLOT1, SLOT2, SLOT3, SLOT4
      INTEGER MXSLOTS, NSLOTS1, NSLOTS2
      INTEGER SLOTS1(MXSLOTS), SLOTS2(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST1, LIST1(2,MXLIST)
      INTEGER NLIST2, LIST2(2,MXLIST)
C     
C     Local variables
C     
      INTEGER I, J, K, NP, MXGROUP, NOBS
      INTEGER NCHECK, SL1, SL2, NKEEP, OSLOT, MSLOT
      INTEGER MULT, SPOS, NCUBE, NNBEST, NBEST, N
      INTEGER ISTR, NDOF, LENSTR
      REAL OVAL, RVAL1, RVAL2, RATIO, CFRAT
      CHARACTER*100 STRING
      INTEGER NGROUP, IFAIL, MAXSPEC
      LOGICAL DEFAULT, OK, CLOBBER, FOPEN, CONTAINED
      LOGICAL SAMECI, FULL, NEXT, MASK
      INTEGER MXPAR, MXSTR
      PARAMETER (MXPAR=4,MXSTR=5)
      INTEGER AXIS(MXPAR), CORNER(MXPAR), IA(MXPAR)
      REAL PAR(MXPAR), PBEST(MXPAR,MXSTR), PNBEST(MXPAR,MXSTR)
      REAL EPAR(MXPAR), EBEST(MXPAR,MXSTR), ENBEST(MXPAR,MXSTR)
      CHARACTER*16 PARNAM(MXPAR)
      CHARACTER*64 FILENAME, REPLY*5
      DOUBLE PRECISION A(MXPAR), ALAMDA, CHISQ, CHOLD, STEP
      DOUBLE PRECISION CHBEST(MXSTR), CHNBEST(MXSTR)
      DOUBLE PRECISION Y, AOLD
C     
      DATA SLOT1, SLOT2, SLOT3, SLOT4, NGROUP/1,1,2,3,1/
      DATA NP/2/
      DATA FILENAME/' '/
      DATA AXIS/6,4,2,1/
      DATA PARNAM/'log g','Temperature','axis 3','axis 4'/
      DATA REPLY/'N'/
      DATA MASK, FULL/.FALSE.,.FALSE./
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
C     
C     First slot
C     
      IF(METHOD.EQ.'F') THEN
         CALL INTR_IN('First observed spectrum', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      ELSE IF(METHOD.EQ.'G') THEN
         CALL INTR_IN('First spectrum to set', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      ELSE
         WRITE(*,*) 'INVALID METHOD = ',METHOD
         GOTO 999
      END IF
C     
C     Second slot
C     
      SLOT2 = MAX(SLOT1, SLOT2)
      IF(METHOD.EQ.'F') THEN
         CALL INTR_IN('Last observed spectrum', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      ELSE IF(METHOD.EQ.'G') THEN
         CALL INTR_IN('Last spectrum to set', SPLIT, NSPLIT, 
     &        MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      END IF
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         IF(METHOD.EQ.'F') THEN
            WRITE(*,*) 'Enter list of observed spectra'
         ELSE IF(METHOD.EQ.'G') THEN
            WRITE(*,*) 'Enter list of spectra to set'
         END IF
         CALL GETLIS(LIST1, NLIST1, MXLIST, MAXSPEC, SLOTS1, 
     &        NSLOTS1, MXSLOTS)
         IF(NSLOTS1.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST1, NLIST1, MXLIST, SLOTS1, 
     &        NSLOTS1, MXSLOTS)
      END IF
C     
      CALL INTR_IN('First model spectrum', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT3, 0, MAXSPEC, IFAIL)
C     
C     Second slot
C     
      SLOT4 = MAX(SLOT3, SLOT4)
      CALL INTR_IN('Last model spectrum', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT4, SLOT3, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT3.EQ.0 .AND. SLOT4.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT3.EQ.0 .AND. SLOT4.EQ.0) THEN
         WRITE(*,*) 'Enter list of model spectra'
         CALL GETLIS(LIST2, NLIST2, MXLIST, MAXSPEC, SLOTS2, 
     &        NSLOTS2, MXSLOTS)
         IF(NSLOTS2.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT3, SLOT4, LIST2, NLIST2, MXLIST, 
     &        SLOTS2, NSLOTS2, MXSLOTS)
      END IF
C     
      MXGROUP = MIN(NSLOTS1, NSLOTS2)
      CALL INTR_IN('Number/group', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NGROUP, 1, MXGROUP, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(MOD(NSLOTS1,NGROUP).NE.0 .OR. MOD(NSLOTS2,NGROUP).NE.0) THEN
         WRITE(*,*) 'The group size conflicts with'//
     &        ' the numbers of spectra'
         WRITE(*,*) 'that you have chosen.'
         GOTO 999
      END IF
C     
C     Now check pixel sizes. First check each group internally.
C     
      NCHECK = NSLOTS1/NGROUP
      IF(NCHECK.GT.1) THEN
         DO J = 1, NGROUP
            SL1 = SLOTS1(J)
            NKEEP = NPIX(SL1)
            IF(NKEEP.LE.0) THEN
               WRITE(*,*) 'Slot ',SL1,' is empty.'
               GOTO 999
            END IF
            DO I = 2, NCHECK
               SL2 = SLOTS1(NGROUP*(I-1)+J)
               IF(NPIX(SL2).NE.NKEEP) THEN
                  WRITE(*,*) 'Conflicting pixel numbers in'//
     &                 ' slots ',SL1,' and ',SL2
                  GOTO 999
               END IF
            END DO
         END DO
      END IF 
C     
      NCHECK = NSLOTS2/NGROUP
      IF(NCHECK.GT.1) THEN
         DO J = 1, NGROUP
            SL1 = SLOTS2(J)
            NKEEP = NPIX(SL1)
            IF(NKEEP.LE.0) THEN
               WRITE(*,*) 'Slot ',SL1,' is empty.'
               GOTO 999
            END IF
            DO I = 2, NCHECK
               SL2 = SLOTS2(NGROUP*(I-1)+J)
               IF(NPIX(SL2).NE.NKEEP) THEN
                  WRITE(*,*) 'Conflicting pixel numbers in'//
     &                 ' slots ',SL1,' and ',SL2
                  GOTO 999
               END IF
            END DO
         END DO
      END IF 
C     
C     Now check that each group matches the other
C     
      DO I = 1, NGROUP
         SL1 = SLOTS1(I)
         SL2 = SLOTS2(I)
         IF(NPIX(SL1).LE.0) THEN
            WRITE(*,*) 'Observed slot ',SL1,' has no pixels.'
         ELSE IF(NPIX(SL2).LE.0) THEN
            WRITE(*,*) 'Model slot ',SL2,' has no pixels.'
         ELSE IF(NPIX(SL1).NE.NPIX(SL2)) THEN
            WRITE(*,*) 'Observed and models slots ',SL1,' and ',SL2
            WRITE(*,*) 'have conflicting numbers of pixels.'
            GOTO 999
         END IF
      END DO
C     
      FILENAME = ' '
      FOPEN = .FALSE.
      IF(METHOD.EQ.'F') THEN
         CALL CHAR_IN(
     &        'File of results (<CR> = ignore, ! = headers)',
     &        SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, FILENAME, IFAIL)
         IF(FILENAME.NE.' ' .AND. FILENAME.NE.'!') THEN
            IF(CLOBBER) THEN
               OPEN(UNIT=32,FILE=FILENAME,ACCESS='SEQUENTIAL',
     &              STATUS='UNKNOWN',IOSTAT=IFAIL)
            ELSE
               OPEN(UNIT=32,FILE=FILENAME,ACCESS='SEQUENTIAL',
     &              STATUS='NEW',IOSTAT=IFAIL)
            END IF
            IF(IFAIL.NE.0) THEN
               WRITE(*,*) 'Failed to open ',FILENAME
               GOTO 999
            END IF
            FOPEN = .TRUE.
         END IF
      END IF
C     
      CALL INTR_IN('Number of parameters', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, NP, 1, MXPAR, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
C     Get parameter names and axis dimensions
C     
      DO I = 1, NP
         STRING = 'Parameter name ( )'
         WRITE(STRING(17:17),'(I1)') I
         CALL CHAR_IN(STRING(:18), SPLIT, NSPLIT, MXSPLIT, NCOM,
     &        DEFAULT, PARNAM(I), IFAIL)
         STRING = 'Dimension ( )'
         WRITE(STRING(12:12),'(I1)') I
         CALL INTR_IN(STRING(:13), SPLIT, NSPLIT, MXSPLIT, NCOM,
     &        DEFAULT, AXIS(I), 2, NSLOTS2, IFAIL)
         IF(IFAIL.NE.0) GOTO 999
      END DO
C     
      IF(METHOD.EQ.'F') THEN
         CALL CHAR_IN('Full output?', SPLIT, NSPLIT, MXSPLIT, NCOM,
     &        DEFAULT, REPLY, IFAIL)
         FULL = SAMECI(REPLY,'YES')
      ELSE
         FULL = .FALSE.   
      END IF
C     
      IF(METHOD.EQ.'G') THEN
         IF(MASK) THEN
            CALL CHAR_IN(
     &           'Retain data mask in interpolated spectra? [Y]', 
     &           SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, REPLY, 
     &           IFAIL)
            MASK = .NOT.SAMECI(REPLY,'NO')
         ELSE
            CALL CHAR_IN(
     &           'Retain data mask in interpolated spectra? [N]', 
     &           SPLIT, NSPLIT, MXSPLIT, NCOM, DEFAULT, REPLY, 
     &           IFAIL)
            MASK = SAMECI(REPLY,'YES')
         END IF
      END IF
C     
C     Number of cuboids
C     
      NCUBE   = 1
      DO J = 1, NP
         NCUBE = NCUBE*(AXIS(J)-1)
      END DO
C     
C     Fit spectra
C     
      IF(METHOD.EQ.'F') THEN
C     
C     Loop through observed spectra
C     
         DO I = 1, NP
            IA(I) = I
         END DO
         DO NOBS = 1, NSLOTS1/NGROUP
C     
C     I have decided that the only fully reliable method is to minimise
C     over every possible cuboid of the model grid.
C     
            DO I = 1, MXSTR
               CHBEST(I)  = 1.D30
               CHNBEST(I) = 1.D30
            END DO
            NBEST  = 0
            NNBEST = 0
C     
            DO I = 1, NP
               CORNER(I) = 1
            END DO
C     
C     Get info line
C     
            OSLOT = SLOTS1(NGROUP*(NOBS-1)+1)
            CALL SL_INF(OSLOT, MXSPEC, NPIX, NMCHAR, NMDOUB, 
     &           NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, 
     &           HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, STRING, IFAIL)           
            IF(FULL) THEN
               WRITE(*,*) ' '
               WRITE(*,*) ' '
               WRITE(*,*) 'Observed spectrum group ',NOBS
               WRITE(*,*) 'First spectrum of group:'
               WRITE(*,*) STRING(:LENSTR(STRING))
               IF(FOPEN) THEN
                  WRITE(32,*,ERR=998) ' '
                  WRITE(32,*,ERR=998) ' '
                  WRITE(32,*,ERR=998) 'Observed spectrum group ',NOBS
                  WRITE(32,*,ERR=998) 'First spectrum of group:'
                  WRITE(32,*,ERR=998) STRING(:LENSTR(STRING))
               END IF
            END IF
C     
C     Compute number of degrees of freedom
C     
            NDOF = - NP
            DO K = 1, NGROUP
               OSLOT = SLOTS1(NGROUP*(NOBS-1)+K)
               DO N = 1, NPIX(OSLOT)
                  IF(ERRORS(N,OSLOT).GT.0.) NDOF = NDOF + 1
               END DO
            END DO
C     
            DO N = 1, NCUBE
C     
C     Start at mid-point of cuboid
C     
               DO I = 1, NP
                  A(I) = 5.D-1
               END DO
C     
C     Perform minimisation
C     
               ALAMDA = -1.D0
               STEP   = 1.D0
               CHOLD  = 1.D30
               AOLD   = ALAMDA-100.
               DO WHILE(AOLD.LT.ALAMDA .OR. STEP.GT.1.D-2)
                  AOLD = ALAMDA
                  CALL MRQMIN1(COUNTS, ERRORS, FLUX, MXBUFF, 
     &                 MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, 
     &                 NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &                 HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &                 NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &                 MXREAL, SLOTS1, NSLOTS1, SLOTS2, NSLOTS2, 
     &                 AXIS, CORNER, NGROUP, NOBS, A, IA, NP, 
     &                 DWORK1, DWORK2, NP, CHISQ, ALAMDA)
                  STEP  = CHOLD - CHISQ
                  CHOLD = CHISQ
               END DO
C     
C     Compute the covariances
C     
               ALAMDA = 0.D0
               CALL MRQMIN1(COUNTS, ERRORS, FLUX, MXBUFF, 
     &              MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, 
     &              NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, 
     &              HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &              NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, 
     &              MXREAL, SLOTS1, NSLOTS1, SLOTS2, NSLOTS2, 
     &              AXIS, CORNER, NGROUP, NOBS, A, IA, NP, 
     &              DWORK1, DWORK2, NP, CHISQ, ALAMDA)
C     
C     Compute parameter values
C     
               DO I = 1, NP
C     
C     First get slot equivalent to current corner
C     
                  SPOS = 1
                  DO J = 1, NP
                     MULT = NGROUP
                     DO K = 1, J-1
                        MULT = MULT*AXIS(K)
                     END DO
                     SPOS = SPOS + MULT*(CORNER(J)-1)
                  END DO
                  MSLOT = SLOTS2(SPOS)
                  CALL HGETR(PARNAM(I), RVAL1, MSLOT, MXSPEC, 
     &                 NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
                  IF(IFAIL.NE.0) THEN
                     WRITE(*,*) 'Failed to find ',PARNAM(I),
     &                    ' in slot ',MSLOT
                     GOTO 999
                  END IF
C     
C     Get slot of corner adjacent 1 along axis I
C     
                  MULT = NGROUP
                  DO K = 1, I-1
                     MULT = MULT*AXIS(K)
                  END DO
                  SPOS = SPOS + MULT
                  MSLOT = SLOTS2(SPOS)
                  CALL HGETR(PARNAM(I), RVAL2, MSLOT, MXSPEC, 
     &                 NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
                  IF(IFAIL.NE.0) THEN
                     WRITE(*,*) 'Failed to find ',PARNAM(I),
     &                    ' in slot ',MSLOT
                     GOTO 999
                  END IF
C     
C     Compute parameter value equivalent to best fit
C     
                  PAR(I)  = REAL(RVAL1 + (RVAL2-RVAL1)*A(I))
                  EPAR(I) = REAL(ABS(RVAL2-RVAL1)*
     &                 SQRT(DWORK1(NP*(I-1)+I)))
               END DO
C     
C     Full output
C     
               IF(FULL) THEN
                  WRITE(*,*) ' '
                  WRITE(*,*) 'Cuboid corner ',(CORNER(K),K=1,NP),
     &                 ', chi**2 = ',REAL(CHISQ)
                  WRITE(*,*) 'Chi**2/DOF (reduced) = ',
     &                 REAL(CHISQ)/REAL(NDOF)
                  WRITE(*,*) 'Best-fit fractions: ',(A(K),K=1,NP)
                  WRITE(*,*) 'Physical values: ',(PAR(K),K=1,NP) 
                  WRITE(*,*) '  Uncertainties: ',(EPAR(K),K=1,NP)
                  IF(FOPEN) THEN
                     WRITE(32,*,ERR=998) ' '
                     WRITE(32,*,ERR=998) 
     &                    'Cuboid corner ',(CORNER(K),K=1,NP),
     &                    ', chi**2 = ',REAL(CHISQ)
                     WRITE(32,*,ERR=998) 'Chi**2/DOF (reduced) = ',
     &                    REAL(CHISQ)/REAL(NDOF)
                     
                     WRITE(32,*,ERR=998) 
     &                    'Best-fit fractions: ',(A(K),K=1,NP)
                     WRITE(32,*,ERR=998) 
     &                    'Physical values: ',(PAR(K),K=1,NP)
                     WRITE(32,*,ERR=998) 
     &                    '  Uncertainties: ',(EPAR(K),K=1,NP)
                  END IF
               END IF
C     
C     See if best-fit occurs within bounds of cuboid or not.
C     
               CONTAINED = .TRUE.
               DO I = 1, NP
                  CONTAINED = CONTAINED 
     &                 .AND. (A(I).GE.0. .AND. A(I).LE.1.)
               END DO
C     
C     Store the results for the top MXSTR contained and 
C     uncontained values. Contained values should be more
C     reliable.
C     
               IF(CONTAINED) THEN
                  ISTR = 0
                  DO I = NBEST, 1, -1
                     IF(CHISQ.LT.CHBEST(I)) ISTR = I
                  END DO
                  IF(ISTR.GT.0 .AND. ISTR.LT.NBEST) THEN
                     DO I = MIN(NBEST,MXSTR-1), ISTR, -1
                        CHBEST(I+1) = CHBEST(I)
                        DO J = 1, NP
                           PBEST(J,I+1) = PBEST(J,I)
                           EBEST(J,I+1) = EBEST(J,I)
                        END DO
                     END DO
                  END IF
                  IF(NBEST.EQ.0 .AND. CHISQ.LT.CHBEST(1)) ISTR = 1
                  IF(ISTR.GT.0) THEN
                     IF(NBEST.LT.MXSTR) NBEST = NBEST + 1
                     CHBEST(ISTR) = CHISQ
                     DO J = 1, NP
                        PBEST(J,ISTR) = PAR(J)
                        EBEST(J,ISTR) = EPAR(J)
                     END DO
                  END IF
               ELSE
                  ISTR = 0
                  DO I = NNBEST, 1, -1
                     IF(CHISQ.LT.CHNBEST(I)) ISTR = I
                  END DO
                  IF(ISTR.GT.0 .AND. ISTR.LT.NNBEST) THEN
                     DO I = MIN(NNBEST,MXSTR-1), ISTR, -1
                        CHNBEST(I+1) = CHNBEST(I)
                        DO J = 1, NP
                           PNBEST(J,I+1) = PNBEST(J,I)
                           ENBEST(J,I+1) = ENBEST(J,I)
                        END DO
                     END DO
                  END IF
                  IF(NNBEST.EQ.0 .AND. CHISQ.LT.CHNBEST(1)) ISTR = 1
                  IF(ISTR.GT.0) THEN
                     IF(NNBEST.LT.MXSTR) NNBEST = NNBEST + 1
                     CHNBEST(ISTR) = CHISQ
                     DO J = 1, NP
                        PNBEST(J,ISTR) = PAR(J)
                        ENBEST(J,ISTR) = EPAR(J)
                     END DO
                  END IF
               END IF 
C     
C     Update corner indices
C     
               K = 1
               NEXT = .TRUE.
               DO WHILE(NEXT)
                  IF(CORNER(K).EQ.AXIS(K)-1) THEN
                     CORNER(K) = 1
                     K = K + 1
                  ELSE
                     CORNER(K) = CORNER(K) + 1
                     NEXT = .FALSE.
                  END IF
               END DO
            END DO
C     
C     Have now finished comparisons with models.
C     Print out results
C     
            IF(.NOT.FULL) THEN
               WRITE(*,*) ' '
               WRITE(*,*) ' '
               WRITE(*,*) 'Observed spectrum group ',NOBS
               WRITE(*,*) 'First spectrum of group:'
               WRITE(*,*,ERR=998) STRING(:LENSTR(STRING))
            END IF
            IF(NBEST.GT.0) THEN
               WRITE(*,*) ' '
               WRITE(*,*) 'Best ',NBEST,' non-extrapolated fits were:'
               DO I = 1, NBEST
                  WRITE(*,*) 'Fit ',I
                  WRITE(*,*) 'Chi**2 = ',CHBEST(I)
                  WRITE(*,*) 'Chi**2/DOF (reduced) = ',
     &                 REAL(CHBEST(I))/REAL(NDOF)
                  WRITE(*,*) '   Parameters: ',(PBEST(J,I),J=1,NP)
                  WRITE(*,*) 'Uncertainties: ',(EBEST(J,I),J=1,NP)
               END DO
            ELSE
               WRITE(*,*) 'There were NO non-extrapolated fits'
            END IF
            IF(NNBEST.GT.0) THEN
               WRITE(*,*) ' '
               WRITE(*,*) 'Best ',NNBEST,' extrapolated fits were:'
               DO I = 1, NNBEST
                  WRITE(*,*) 'Fit ',I
                  WRITE(*,*) 'Chi**2 = ',CHNBEST(I)
                  WRITE(*,*) 'Chi**2/DOF (reduced) = ',
     &                 REAL(CHNBEST(I))/REAL(NDOF)
                  WRITE(*,*) '   Parameters: ',(PNBEST(J,I),J=1,NP)
                  WRITE(*,*) 'Uncertainties: ',(ENBEST(J,I),J=1,NP)
               END DO
            ELSE
               WRITE(*,*) 'There were NO extrapolated fits'
            END IF
            IF(FOPEN) THEN
               IF(.NOT.FULL) THEN
                  WRITE(32,*,ERR=998) ' '
                  WRITE(32,*,ERR=998) ' '
                  WRITE(32,*,ERR=998) 'Observed spectrum group ',NOBS
                  WRITE(32,*,ERR=998) 'First spectrum of group:'
                  WRITE(32,*,ERR=998) STRING(:LENSTR(STRING))
               END IF
               IF(NBEST.GT.0) THEN
                  WRITE(32,*,ERR=998) ' '
                  WRITE(32,*,ERR=998) 'Best ',NBEST,
     &                 ' non-extrapolated fits were:'
                  DO I = 1, NBEST
                     WRITE(32,*,ERR=998) 'Fit ',I
                     WRITE(32,*,ERR=998) 'Chi**2 = ',CHBEST(I)
                     WRITE(32,*,ERR=998) 'Chi**2/DOF (reduced) = ',
     &                    REAL(CHBEST(I))/REAL(NDOF)
                     WRITE(32,*,ERR=998) '   Parameters: ',
     &                    (PBEST(J,I),J=1,NP)
                     WRITE(32,*,ERR=998) 'Uncertainties: ',
     &                    (EBEST(J,I),J=1,NP)
                  END DO
               ELSE
                  WRITE(32,*,ERR=998) 
     &                 'There were NO non-extrapolated fits'
               END IF
               IF(NNBEST.GT.0) THEN
                  WRITE(32,*,ERR=998) ' '
                  WRITE(32,*,ERR=998) 
     &                 'Best ',NNBEST,' extrapolated fits were:'
                  DO I = 1, NNBEST
                     WRITE(32,*,ERR=998) 'Fit ',I
                     WRITE(32,*,ERR=998) 'Chi**2 = ',CHNBEST(I)
                     WRITE(32,*,ERR=998) 'Chi**2/DOF (reduced) = ',
     &                    REAL(CHNBEST(I))/REAL(NDOF)
                     WRITE(32,*,ERR=998) '   Parameters: ',
     &                    (PNBEST(J,I),J=1,NP)
                     WRITE(32,*,ERR=998) 'Uncertainties: ',
     &                    (ENBEST(J,I),J=1,NP)
                  END DO
               ELSE
                  WRITE(32,*,ERR=998) 'There were NO extrapolated fits'
               END IF
            END IF
C     
C     Store best fit in headers
C     
            IF(FILENAME.EQ.'!' .AND. NBEST.GT.0) THEN
               DO J = 1, NGROUP
                  OSLOT = SLOTS1(NGROUP*(NOBS-1)+J)
                  CALL HSETR(PARNAM(I), PBEST(J,1), OSLOT, MXSPEC, 
     &                 NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
               END DO
            ELSE IF(FILENAME.EQ.'!') THEN
               WRITE(*,*) 'No fit values stored in headers'
            END IF
         END DO
C     
      ELSE IF(METHOD.EQ.'G') THEN
C     
         DO NOBS = 1, NSLOTS1/NGROUP
            OSLOT = SLOTS1(NGROUP*(NOBS-1)+1)
            OK = .TRUE.
            DO I = 1, NP
C     
C     Locate observed value
C     
               CALL HGETR(PARNAM(I), OVAL, OSLOT, MXSPEC, 
     &              NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
               IF(IFAIL.NE.0) THEN
                  WRITE(*,*) 'Failed to find ',PARNAM(I),
     &                 ' in slot',OSLOT
                  GOTO 999
               END IF
C     
C     Loop through model values to find appropriate cuboid (one
C     which brackets observed values)
C     
               MULT = NGROUP
               DO K = 1, I-1
                  MULT = MULT*AXIS(K)
               END DO
               CORNER(I) = 0
               K = 1
               DO WHILE(CORNER(I).EQ.0 .AND. K.LE.AXIS(I)-1)
                  MSLOT = SLOTS2(MULT*(K-1)+1)
                  CALL HGETR(PARNAM(I), RVAL1, MSLOT, MXSPEC, 
     &                 NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
                  IF(IFAIL.NE.0) THEN
                     WRITE(*,*) 'Failed to find ',PARNAM(I),
     &                    ' in slot',MSLOT
                     GOTO 999
                  END IF
                  MSLOT = SLOTS2(MULT*K+1)
                  CALL HGETR(PARNAM(I), RVAL2, MSLOT, MXSPEC, 
     &                 NMREAL, HDREAL, NREAL, MXREAL, IFAIL)
                  IF(IFAIL.NE.0) THEN
                     WRITE(*,*) 'Failed to find ',PARNAM(I),
     &                    ' in slot',MSLOT
                     GOTO 999
                  END IF
                  IF((RVAL1.LE.OVAL .AND. RVAL2.GE.OVAL) .OR.
     &                 (RVAL2.LE.OVAL .AND. RVAL1.GE.OVAL)) THEN
                     CORNER(I) = K
                     A(I) = (OVAL-RVAL1)/(RVAL2-RVAL1)
                  END IF
                  K = K + 1
               END DO
               OK = OK .AND. CORNER(I).GT.0
            END DO
            IF(OK) THEN
C     
C     Now set spectrum
C     
               DO I = 1, NGROUP
                  OSLOT = SLOTS1(NGROUP*(NOBS-1)+I)
                  DO J = 1, NPIX(OSLOT)
                     RATIO = CFRAT(COUNTS(J,OSLOT), 
     &                    FLUX(J,OSLOT))
                     CALL TABFIT(COUNTS, ERRORS, FLUX, 
     &                    MXBUFF, MXSPEC, MAXPX, NPIX, 
     &                    ARC, NARC, MXARC, NMCHAR, NMDOUB, 
     &                    NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &                    HDINTR, HDREAL, NCHAR, NDOUB, 
     &                    NINTR, NREAL, MXCHAR, MXDOUB, 
     &                    MXINTR, MXREAL, SLOTS2, NSLOTS2, 
     &                    AXIS, CORNER, J, I, NGROUP, A, 
     &                    Y, DWORK1, NP)
                     COUNTS(J,OSLOT) = REAL(RATIO*Y)
                     IF(MASK .AND. ERRORS(J,OSLOT).LE.0.) THEN
                        ERRORS(J,OSLOT) = 
     &                       -MAX(ABS(COUNTS(J,OSLOT))/1000.,1.E-5)
                     ELSE
                        ERRORS(J,OSLOT) = 
     &                       MAX(ABS(COUNTS(J,OSLOT))/1000.,1.E-5)
                     END IF
                     IF(COUNTS(J,OSLOT).EQ.0.) THEN
                        FLUX(J,OSLOT) = RATIO
                     ELSE
                        FLUX(J,OSLOT) = REAL(Y)
                     END IF              
                  END DO
               END DO
               WRITE(*,*) 'Interpolated spectrum group ',NOBS
            ELSE
               WRITE(*,*) 'Parameters of spectrum group ',NOBS,
     &              ' are out of range.'
            END IF            
         END DO
      END IF
      IFAIL = 0
 999  IF(FOPEN) CLOSE(UNIT=32)
      RETURN 
 998  WRITE(*,*) 'Failed to write to ',FILENAME
      CLOSE(UNIT=32)
      RETURN
      END

      SUBROUTINE MRQMIN1(COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, MAXPX, 
     &     NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &     HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &     MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS1, NSLOTS1, SLOTS2, 
     &     NSLOTS2, AXIS, CORNER, NGROUP, NOBS, A, IA, MA, COVAR, ALPHA,
     &     NCA, CHISQ, ALAMDA)
C     
C     Modified version of NM mrqmin specialised to molly data to allow many 
C     to be fitted simultaneously.
C     
      IMPLICIT NONE
C     
C     molly parameters
C     
C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL, NSLOTS1, NSLOTS2
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
      INTEGER MA
      INTEGER SLOTS1(NSLOTS1), SLOTS2(NSLOTS2)
      INTEGER AXIS(MA), CORNER(MA), NGROUP, NOBS
C     
      INTEGER NCA,IA(MA),MMAX, IFAIL
      DOUBLE PRECISION ALAMDA,CHISQ,A(MA),ALPHA(NCA,NCA)
      DOUBLE PRECISION COVAR(NCA,NCA)
      PARAMETER (MMAX=100)
C     U    USES COVSRT,GAUSSJ,MRQCOF
      INTEGER J,K,L,M,MFIT
      DOUBLE PRECISION OCHISQ,ATRY(MMAX),BETA(MMAX),DA(MMAX)
      SAVE OCHISQ,ATRY,BETA,DA,MFIT
C     
      IF(ALAMDA.LT.0.D0)THEN
         MFIT=0
         DO 11 J=1,MA
            IF (IA(J).NE.0) MFIT=MFIT+1
 11      CONTINUE
         ALAMDA=1.D-3
         CALL MRQCOF1(COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, MAXPX, 
     &        NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &        HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, 
     &        NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS1, NSLOTS1, 
     &        SLOTS2, NSLOTS2, AXIS, CORNER, NGROUP, NOBS, A, IA, MA, 
     &        ALPHA, BETA, NCA, CHISQ)
         OCHISQ=CHISQ
         DO 12 J=1,MA
            ATRY(J)=A(J)
 12      CONTINUE
      ENDIF
      J=0
      DO 14 L=1,MA
         IF(IA(L).NE.0) THEN
            J=J+1
            K=0
            DO 13 M=1,MA
               IF(IA(M).NE.0) THEN
                  K=K+1
                  COVAR(J,K)=ALPHA(J,K)
               ENDIF
 13         CONTINUE
            COVAR(J,J)=ALPHA(J,J)*(1.+ALAMDA)
            DA(J)=BETA(J)
         ENDIF
 14   CONTINUE
      CALL GAUSSJ(COVAR,MFIT,NCA,DA,1,1,IFAIL)
      IF(IFAIL.NE.0) RETURN
      IF(ALAMDA.EQ.0.D0)THEN
         CALL COVSRT(COVAR,NCA,MA,IA,MFIT)
         RETURN
      ENDIF
      J=0
      DO 15 L=1,MA
         IF(IA(L).NE.0) THEN
            J=J+1
            ATRY(L)=A(L)+DA(J)
         ENDIF
 15   CONTINUE
      CALL MRQCOF1(COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, MAXPX, 
     &     NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &     HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &     MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS1, NSLOTS1, SLOTS2, 
     &     NSLOTS2, AXIS, CORNER, NGROUP, NOBS, ATRY, IA, MA, COVAR, 
     &     DA, NCA, CHISQ)
      IF(CHISQ.LT.OCHISQ) THEN
         ALAMDA=1.D-1*ALAMDA
         OCHISQ=CHISQ
         J=0
         DO 17 L=1,MA
            IF(IA(L).NE.0) THEN
               J=J+1
               K=0
               DO 16 M=1,MA
                  IF(IA(M).NE.0) THEN
                     K=K+1
                     ALPHA(J,K)=COVAR(J,K)
                  ENDIF
 16            CONTINUE
               BETA(J)=DA(J)
               A(L)=ATRY(L)
            ENDIF
 17      CONTINUE
      ELSE
         ALAMDA=1.D1*ALAMDA
         CHISQ=OCHISQ
      ENDIF
      RETURN
      END

      SUBROUTINE MRQCOF1(COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, MAXPX, 
     &     NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &     HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &     MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS1, NSLOTS1, SLOTS2, 
     &     NSLOTS2, AXIS, CORNER, NGROUP, NOBS, A, IA, MA, ALPHA, BETA,
     &     NALP, CHISQ)
C     
C     Modified for molly data from mrqcof
C     
      IMPLICIT NONE
C     
C     molly parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
      INTEGER NSLOTS1, SLOTS1(NSLOTS1), NOBS, SLOT
      INTEGER NSLOTS2, SLOTS2(NSLOTS2), NGROUP
C     
      INTEGER MA, NALP, IA(MA), MMAX
      INTEGER AXIS(MA), CORNER(MA)
      DOUBLE PRECISION CHISQ,A(MA),ALPHA(NALP,NALP),BETA(MA)
      DOUBLE PRECISION Y, SIG
      REAL GET_FLX, GET_ERF
      PARAMETER (MMAX=100)
      INTEGER MFIT,KK,J,K,L,M,N
      DOUBLE PRECISION DY,SIG2I,WT,YMOD,DYDA(MMAX)
C     
      MFIT=0
      DO 11 J=1,MA
         IF (IA(J).NE.0) MFIT=MFIT+1
 11   CONTINUE
      DO 13 J=1,MFIT
         DO 12 K=1,J
            ALPHA(J,K)=0.D0
 12      CONTINUE
         BETA(J)=0.D0
 13   CONTINUE
      CHISQ=0.D0
C     
      DO KK = 1, NGROUP
         SLOT = SLOTS1(NGROUP*(NOBS-1)+KK)
         DO N = 1, NPIX(SLOT)
            IF(ERRORS(N,SLOT).GT.0.) THEN
               Y   = GET_FLX(COUNTS(N,SLOT), FLUX(N,SLOT))
               SIG = GET_ERF(COUNTS(N,SLOT), 
     &              ERRORS(N,SLOT), FLUX(N,SLOT))
C     
               CALL TABFIT(COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, MAXPX, 
     &              NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, 
     &              NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, 
     &              NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL,
     &              SLOTS2, NSLOTS2, AXIS, CORNER, N, KK, NGROUP, A, 
     &              YMOD, DYDA, MA)
C     
               SIG2I=1.D0/(SIG*SIG)
               DY=Y-YMOD
               J=0
               DO L=1,MA
                  IF(IA(L).NE.0) THEN
                     J=J+1
                     WT=DYDA(L)*SIG2I
                     K=0
                     DO M=1,L
                        IF(IA(M).NE.0) THEN
                           K=K+1
                           ALPHA(J,K)=ALPHA(J,K)+WT*DYDA(M)
                        ENDIF
                     END DO
                     BETA(J)=BETA(J)+DY*WT
                  END IF
               END DO
               CHISQ=CHISQ+DY*DY*SIG2I
            END IF
         END DO
      END DO
C     
      DO J=2,MFIT
         DO K=1,J-1
            ALPHA(K,J)=ALPHA(J,K)
         END DO
      END DO
      RETURN
      END

      SUBROUTINE TABFIT(COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, MAXPX, 
     &     NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, 
     &     HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, 
     &     MXCHAR, MXDOUB, MXINTR, MXREAL, SLOTS2, NSLOTS2, AXIS, 
     &     CORNER, IP, IG, NGROUP, A, Y, DYDA, MA)
C     
C     Produces predicted Y value and derivatives with respect to parameters
C     at pixel IP of spectrum IG (out of NGROUP)
C     
      IMPLICIT NONE
C     
C     molly parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
C     Arc coefficients
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
      INTEGER NGROUP, MSLOT, MA
      INTEGER NSLOTS2, SLOTS2(NSLOTS2), SPOS
      INTEGER AXIS(MA), CORNER(MA)
C     
      INTEGER INDX(20), IP, IG, I, J, K
      INTEGER NCORN, LEFT, MULT
      REAL FLX, GET_FLX
      DOUBLE PRECISION A(MA), Y, DYDA(MA), DFAC
C     
      IF(MA.GT.20) THEN
         WRITE(*,*) 'SOMETHING VERY WRONG IN TABFIT'
         RETURN
      END IF
C     
C     Number of corners of cuboid. 
C     
      NCORN = 2**MA
      Y = 0.D0
      DO I = 1, MA
         DYDA(I) = 0.D0
      END DO
      DO I = 1, NCORN
C     
C     Generate an array of numbers which are 0 or 1 according to whether
C     we are on the first or second point of a given axis
C     
         LEFT = I - 1
         DO J = 1, MA
            INDX(J) = MOD(LEFT,2)
            LEFT = LEFT/2
         END DO
C     
C     Work out slots of model spectrum
C     
         SPOS = 0
         DO J = 1, MA
            MULT = NGROUP
            DO K = 1, J-1
               MULT = MULT*AXIS(K)
            END DO
            SPOS = SPOS + MULT*(CORNER(J)+INDX(J)-1)
         END DO
         MSLOT = SLOTS2(SPOS+IG)
         FLX = GET_FLX(COUNTS(IP,MSLOT), FLUX(IP,MSLOT))
C     
C     Compute multipliers and add into function and derivatives
C     
         DFAC = 1.D0
         DO J = 1, MA
            IF(INDX(J).EQ.0) THEN
               DFAC = DFAC*(1.D0-A(J))
            ELSE
               DFAC = DFAC*A(J)
            END IF
         END DO
         Y = Y + DFAC*FLX
C     
         DO K = 1, MA
            DFAC = 1.D0
            DO J = 1, MA 
               IF(J.NE.K) THEN
                  IF(INDX(J).EQ.0) THEN
                     DFAC = DFAC*(1.D0-A(J))
                  ELSE
                     DFAC = DFAC*A(J)
                  END IF
               ELSE IF(INDX(J).EQ.0) THEN
                  DFAC = -DFAC
               END IF
            END DO
            DYDA(K) = DYDA(K) + DFAC*FLX
         END DO
      END DO
      RETURN
      END 
