CBCKFIL
C BCKFIL N1 N2 N3 FWHM -- Applies filter that can be used prior to 
C back-projection in order to compute filtered
C back-projection used in Doppler tomography.
C     
C The filter applied is ABS(S)*EXP(-(S/W)**2/2) where S is the frequency with
C the Nyquist frequency of one cycle/2 pixels as the maximum. W is a user
C defined parameter (see below) that suppresses high frequencies and leads
C to a blurring of the final image.
C     
C The filter is applied to the entire spectrum. You may want to rebin before 
C using it if you are only interested in a small region of your spectra,
C although it is a reasonably fast operation.
C     
C Parameters:
C     N1   -- First spectrum
C     N2   -- Last spectrum
C     N3   -- First output slot
C     FWHM -- FWHM of window to taper off high frequencies. It is
C     scaled in terms of the Nyquist frequency which is the 
C     highest useful frequency. Thus FWHM >>1 will have little
C     effect and may be OK if you have good data whereas for
C     noisy data you may wish to choose a FWHM < 1. You
C     should probably start with a high value and decrease it
C     if the result looks very noisy. Even quite noisy data 
C     does not seem to require FWHM < 0.5.
C     
C Related commands: BACK, BNDF
C     
CBCKFIL
      SUBROUTINE BCKFIL(SPLIT, NSPLIT, MXSPLIT, COUNTS, ERRORS, FLUX, 
     &     MXBUFF, MXSPEC, MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, 
     &     NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, 
     &     NCHAR, NDOUB, NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, 
     &     SLOTS, MXSLOTS, WORK1, WORK2, MXWORK, IFAIL)
C     
C     
C     Arguments:
C     
C     >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
C     >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
C     zero in which case defaults are used.
C     >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
C     =  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
C     =  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
C     =  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
C     >  I     MXBUFF            -- Size of spectrum buffers in calling routine
C     >  I     MXSPEC           -- Maximum number of spectra allowed by size
C     of header buffers. 
C     >  I     MAXPX            -- Maximum number of pixels/spectrum
C     >  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
C     <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
C     <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
C     >  I     MXARC            -- Maximum number of arc coefficients/spectrum
C     <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
C     <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
C     <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
C     <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
C     <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
C     <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
C     <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
C     <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
C     <  I     NCHAR(MXSPEC)  -- Number of character header items.
C     <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
C     <  I     NINTR(MXSPEC)  -- Number of integer items.
C     <  I     NREAL(MXSPEC)  -- Number of real items.
C     >  I     MXCHAR    -- Maximum number of character header items/spec
C     >  I     MXDOUB    -- Maximum number of double precsion header items/spec
C     >  I     MXINTR    -- Maximum number of integer header items/spec
C     >  I     MXREAL    -- Maximum number of real headers items/spec
C     I     SLOTS(MXSLOTS)     -- Workspace for list of slots
C     <  I     IFAIL            -- Error return.
C     
      IMPLICIT NONE
C     
C     Integer parameters
C     
      INTEGER MXSPEC, MXBUFF, MAXPX
      INTEGER MXCHAR, MXDOUB, MXREAL, MXINTR, MXARC
      INTEGER NPIX(MXSPEC), NARC(MXSPEC)
C     
C     Data arrays
C     
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C     
C     Numbers of header items
C     
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C     
C     Names of header items
C     
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C     
C     Values of header items
C     
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C     
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C     
C     Slot lists
C     
      INTEGER MXSLOTS, NSLOTS, SLOT
      INTEGER SLOTS(MXSLOTS)
      INTEGER MXLIST
      PARAMETER (MXLIST=50)
      INTEGER NLIST, LIST(2,MXLIST)
C     
C     Work arrays
C     
      INTEGER MXWORK
      REAL WORK1(MXWORK), WORK2(MXWORK)
C     
C     Command parameters
C     
      INTEGER MXSPLIT, NSPLIT, NCOM
      CHARACTER*(*) SPLIT(MXSPLIT)
C     
C     Local variables
C     
      INTEGER I, J, NOUT, OUT, NTOT, MAXSPEC, NBI2
      INTEGER J1, J2, LENSTR
      REAL GET_FLX,  FWHM, CFRAT, RATIO
      DOUBLE PRECISION FAC, PI, PI2, AA, CC, DD, EFAC
      INTEGER SLOT1, SLOT2, IFAIL
      LOGICAL  DEFAULT
      CHARACTER*100 STRING
C     
      DATA FWHM/10./
      DATA SLOT1, SLOT2, OUT/1,2,3/
C     
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      NCOM    = 0
      IFAIL   = 0
C     
      CALL INTR_IN('First slot to filter', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT1, 0, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
      SLOT2 = MAX(SLOT1, SLOT2)
      CALL INTR_IN('Last slot to filter', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT2, SLOT1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
      IF(SLOT1.EQ.0 .AND. SLOT2.NE.0) THEN
         WRITE(*,*) 'Only 0,0 to get list option'
         GOTO 999
      ELSE IF(SLOT1.EQ.0 .AND. SLOT2.EQ.0) THEN
         WRITE(*,*) 'Enter list of spectra to filter'
         CALL GETLIS(LIST, NLIST, MXLIST, MAXSPEC, SLOTS, 
     &        NSLOTS, MXSLOTS)
         IF(NSLOTS.EQ.0) GOTO 999
      ELSE
         CALL SETLIS(SLOT1, SLOT2, LIST, NLIST, MXLIST, SLOTS, 
     &        NSLOTS, MXSLOTS)
      END IF
      CALL INTR_IN('First output slot', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, OUT, 1, MAXSPEC-NSLOTS+1, IFAIL)
      CALL REAL_IN('FWHM of gaussian window', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, FWHM, 0., 1.E10, IFAIL)
      IF(IFAIL.NE.0) GOTO 999
C     
      PI   =  4.D0*ATAN(1.D0)
      PI2  = -1.D0/PI**2
      EFAC = 4.D0*LOG(2.D0)/DBLE(FWHM)**2
      NOUT = OUT - 1
      DO I = 1, NSLOTS
         SLOT = SLOTS(I)
         IF( NPIX(SLOT).GT.0) THEN
            DO J = 1, NPIX(SLOT)
               WORK1(J) = GET_FLX(COUNTS(J,SLOT),FLUX(J,SLOT))
            END DO
            NTOT = 2**(INT(LOG(REAL(2*NPIX(SLOT)-1))/LOG(2.))+1)
            IF(NTOT.GT.MXWORK) THEN
               WRITE(*,*) 'Require too large a buffer for FFTs'
               WRITE(*,*) 'Need ',NTOT
               GOTO 999
            END IF
            NOUT = NOUT + 1
C     
C     Set convolving function which is FT of ABS(S) truncated
C     at Nyquist frequency and sampled onto same period as
C     data. New code added here to only allow lags up to +/-(NPIX-1)
C     
            NBI2 = NTOT/2
            WORK2(1)      = 2.5D-1
            WORK2(NBI2+1) = 0.D0
C     DO J = 2, NBI2
            DO J = 2, NPIX(SLOT)
               IF(MOD(J,2).EQ.1) THEN
                  AA = 0.D0
               ELSE
                  AA = PI2/DBLE(J-1)**2
               END IF
               WORK2(J     )   = REAL(AA)
               WORK2(NTOT-J+2) = REAL(AA)
            END DO
            DO J = NPIX(SLOT)+1,NTOT-NPIX(SLOT)+1
               WORK2(J) = 0.D0
            END DO
C     
C     FFT of convolving function to get filter function
C     
            CALL REALFT(WORK2, NTOT, 1)
C     
C     Pad out data array with zeroes and FFT
C     
            DO J = NPIX(SLOT)+1, NTOT
               WORK1(J) = 0.D0
            END DO
            CALL REALFT(WORK1, NTOT, 1)
C     
C     Apply filter along with gaussian window function.
C     
            WORK1(1) = WORK1(1)*WORK2(1)
            WORK1(2) = WORK1(2)*WORK2(2)*REAL(EXP(-EFAC))
            DO J = 1, NTOT/2-1
               J1 = 2*J+1
               FAC = WORK2(J1)*EXP(-EFAC*(REAL(J)/REAL(NTOT/2))**2)
               CC = WORK1(J1)
               J2 = J1+1
               DD = WORK1(J2)
               WORK1(J1) = REAL(FAC*CC)
               WORK1(J2) = REAL(FAC*DD)
            END DO
            CALL REALFT(WORK1,NTOT,-1)
C     
C     Now store in output slot.
C     

            DO J = 1, NPIX(SLOT)
               RATIO = CFRAT(COUNTS(J,SLOT),FLUX(J,SLOT))
               FLUX(J,NOUT) = WORK1(J)
               COUNTS(J,NOUT) = RATIO*WORK1(J)
               ERRORS(J,NOUT) = ERRORS(J,SLOT)
               IF(COUNTS(J,NOUT).EQ.0.) THEN
                  FLUX(J,NOUT) = RATIO
               ELSE
                  FLUX(J,NOUT) = WORK1(J)
               END IF
            END DO
C     
C     Set headers 
C     
            CALL SET_HEAD(NOUT, SLOT, MXSPEC, NPIX, ARC, NARC, 
     &           MXARC, NMCHAR, NMDOUB, NMINTR, NMREAL, HDCHAR, HDDOUB, 
     &           HDINTR, HDREAL, NCHAR, NDOUB, NINTR, NREAL, MXCHAR, 
     &           MXDOUB, MXINTR, MXREAL, IFAIL)
C     
            CALL SL_INF(NOUT, MXSPEC, NPIX, NMCHAR, NMDOUB, NMINTR, 
     &           NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &           NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, STRING, 
     &           IFAIL)
            WRITE(*,*) STRING(:LENSTR(STRING))
         END IF
      END DO
 999  RETURN
      END

