*ARC
* ARC N -- Fits wavelength scale to an arc spectrum
*
* Parameters:
*        N -- The arc to be fitted. Can be changed inside routine.
*
* ARC is interactive in nature and has many commands inside it.
* The general aim is to identify arc lines with specific wavelengths
* and fit a polynomial to the result. This can then be interpolated
* onto object spectra later. An outline of the procedure is as follows:
*
* First use 'NEW' to measure positions of lines in the arc spectrum (uses 
* cross-correlation with a gaussian of user specified FWHM). I normally use
* a FWHM comparable to the lines, but it should not matter. However, you should
* always use the same FWHM for all lines, and sometimes smaller FWHM may be
* less affected by closely spaced lines. Another parameter here allows one to
* avoid having lines too closely spaced. Next 'IDENTIFY' 2 or 3 lines spanning 
* as much of the wavelength range as possible (not very critical though).
* Come out of 'IDENTIFY' and 'FIT' a poly (2 coefficients to start), and then
* go back into 'IDENTIFY'. The routine will then predict wavelengths for the
* lines which helps greatly. If you have a good list of lines then these can
* now be loaded. With more lines you can see if a higher order fit is needed, 
* going back into 'FIT' etc, until you have completed the process. Check for
* blends and dump your line list to disk. You should only use good lines that
* will not be present in one spectrum but not another.
*
* For similar arcs, load your line list, 'TWEAK' the positions and then refit.
* If during the tweak, all the shifts are similar, then you are OK. If one is
* obviously discrepant, the line may have been a blend or weak, and it may be
* better to discard it, in which case you should re-do the original fit for
* consistency. Sometimes by applying an appropriate guess at the shift it is
* possible to recover such lines. It is however important to use the SAME set
* of lines for all your arcs, even if it means redoing every one because you
* cannot get the position of a line in your very last arc although you could
* in all the others. 
*
* If you have taken several different arcs at the start of the night which 
* give a good scale but then only observed one of them during the night
* to save time, you can first fit many lines with a high order poly and set
* it as a 'master' polynomial inside 'FIT'. For the nighttime arcs you can then
* just fit a few low order coefficients.
* 
* Once you have finished, come out of ARC and dump the calibrated spectra. You
* can use 'DRIFT' to plot the drift of arc scale with time. Particularly useful
* if you are following a single object for a long time.
*
* Once you are underway with the TWEAK, FIT cycle, you can use 'COMB' to apply
* them to a succession of arcs, with no questions asked. If you are running this
* in a batch mode, I recommend that you set it to abort if any tweak fails ('ABORT').
* This flags the dangerous possibility that the first tweak fails without you noticing
* it. It aborts on all failures within the tweak section except failure to find a
* spectrum within the 'combination' section (which often occurs if one enters a range 
* guaranteed to exceed the number of arc spectra to ensure that you get them all).
*
* Typically I proceed as follows: first I fit as many lines as possible in one arc
* (or perhaps several arcs as long as they were taken under the same conditions).
* These define the high-order fit. You must go through every line carefully to check against
* blends; no short cuts here. I dump the final list to two files, one 'master_arc.lis',
* the other 'tweak_arc.lis'. When I calibrate everything in batch mode, I begin by loading
* the master arc list, and setting up a high order master polynomial. I then set it to
* tweak the lowest 2 or 3 terms. Then I load the tweak list, tweak it once to set up the
* tweak parameters, and then run 'combination' on all the arcs. 
*
* Very often, over the course of tens of arcs, one or more of the lines will fail during 
* a tweak. You then must delete it from the tweak list (I just do this to the disk file with
* 'emacs') and fit ALL arcs again. I do this again with a script and go through every single
* arc of a run to boil the tweak list down to the lines that are rock solid. You have to be careful
* at this stage that lines are not lost just because the arcs themselves are corrupt. Again this is
* a tedious and long drawn out process, with no short cuts.
*
* Related command: ACAL, DRIFT
*
*ARC 
      SUBROUTINE ARC_FIT(SPLIT, NSPLIT, MXSPLIT, PLOT1, PLOT2, 
     &PLOT3, KEY, MXPLOT, COUNTS, ERRORS, FLUX, MXBUFF, MXSPEC, 
     &MAXPX, NPIX, ARC, NARC, MXARC, NMCHAR, NMDOUB, NMINTR, 
     &NMREAL, HDCHAR, HDDOUB, HDINTR, HDREAL, NCHAR, NDOUB, 
     &NINTR, NREAL, MXCHAR, MXDOUB, MXINTR, MXREAL, CLOBBER,
     &IFAIL)
*
* ARC_FIT routine. Adapted from 'RUBY'
* 
* Arguments:
*
* >  C*(*) SPLIT(MXSPLIT)   -- Command parameters. See below.
* >  I     NSPLIT           -- Actual number of entries in SPLIT. Can be
*                              zero in which case defaults are used.
* >  I     MXSPLIT          -- Dimension of SPLIT in calling routine.
* <  R     WCEN             -- Central wavelength returned for velocity plots
* >  R     PLOT1(MXPLOT)    -- Work arrays for plotting
* >  R     PLOT2(MXPLOT)
* >  R     PLOT3(MXPLOT)
* >  I     KEY(MXPLOT)       -- Sorting array
* <  R     COUNTS(MXBUFF)    -- Array for spectra in counts 
* <  R     ERRORS(MXBUFF)    -- 1-sigma uncertainties in counts
* <  R     FLUX(MXBUFF)      -- Equivalent fluxes to COUNTS
* >  I     MXBUFF            -- Size of spectrum buffers in calling routine
* >  I     MXSPEC           -- Maximum number of spectra allowed by size
*                              of header buffers. 
* >  I     MAXPX            -- Maximum number of pixels/spectrum
* <  I     NPIX(MXSPEC)     -- Number of pixels for each spectrum
* <  D     ARC(MXARC,MXSPEC)-- Arc coefficient buffers
* <  I     NARC(MXSPEC)     -- Number of coefficients for each spectrum
* >  I     MXARC            -- Maximum number of arc coefficients/spectrum
* <  C*(*) NMCHAR(MXCHAR,MXSPEC) -- Names of character header items.
* <  C*(*) NMDOUB(MXDOUB,MXSPEC) -- Names of double precision header items.
* <  C*(*) NMINTR(MXINTR,MXSPEC) -- Names of integer header items.
* <  C*(*) NMREAL(MXREAL,MXSPEC) -- Names of real header items.
* <  C*(*) HDCHAR(MXCHAR,MXSPEC) -- Character header items
* <  D     HDDOUB(MXDOUB,MXSPEC) -- Double precision header items
* <  I     HDINTR(MXINTR,MXSPEC) -- Integer header items
* <  R     HDREAL(MXREAL,MXSPEC) -- Real header items
* <  I     NCHAR(MXSPEC)  -- Number of character header items.
* <  I     NDOUB(MXSPEC)  -- Number of double precision header items.
* <  I     NINTR(MXSPEC)  -- Number of integer items.
* <  I     NREAL(MXSPEC)  -- Number of real items.
* >  I     MXCHAR    -- Maximum number of character header items/spec
* >  I     MXDOUB    -- Maximum number of double precsion header items/spec
* >  I     MXINTR    -- Maximum number of integer header items/spec
* >  I     MXREAL    -- Maximum number of real headers items/spec
* <  I     IFAIL     -- Error return.
*
      IMPLICIT NONE
C
C     Integer parameters
C
      INTEGER MXSPEC, MXBUFF, MAXPX, MXARC, MXCHAR, MXDOUB
      INTEGER MXINTR, MXREAL
      INTEGER NPIX(MXSPEC)
      INTEGER NARC(MXSPEC)
C
C     Plot parameters
C
      INTEGER MXPLOT 
      REAL PLOT1(MXPLOT), PLOT2(MXPLOT), PLOT3(MXPLOT)
      INTEGER KEY(MXPLOT)
C
C     Data arrays
C
      REAL COUNTS(MAXPX,MXBUFF/MAXPX)
      REAL ERRORS(MAXPX,MXBUFF/MAXPX)
      REAL FLUX(MAXPX,MXBUFF/MAXPX)
C
C     Numbers of header items
C
      INTEGER NCHAR(MXSPEC)
      INTEGER NDOUB(MXSPEC)
      INTEGER NINTR(MXSPEC)
      INTEGER NREAL(MXSPEC)
C
C     Names of header items
C
      CHARACTER*(*) NMCHAR(MXCHAR,MXSPEC)
      CHARACTER*(*) NMDOUB(MXDOUB,MXSPEC)
      CHARACTER*(*) NMINTR(MXINTR,MXSPEC)
      CHARACTER*(*) NMREAL(MXREAL,MXSPEC)
C
C     Values of header items
C
      CHARACTER*(*)    HDCHAR(MXCHAR,MXSPEC)
      DOUBLE PRECISION HDDOUB(MXDOUB,MXSPEC)
      INTEGER          HDINTR(MXINTR,MXSPEC)
      REAL             HDREAL(MXREAL,MXSPEC)
C
C     Arc coefficients
C
      DOUBLE PRECISION ARC(MXARC,MXSPEC)
C
C     Command parameters
C
      INTEGER MXSPLIT, NSPLIT
      CHARACTER*(*) SPLIT(MXSPLIT)
C
C     Plot parameters
C
      CHARACTER*64 XLABEL, YLABEL, TLABEL
C
C     Line list arrays
C
      INTEGER MAXLIN
      PARAMETER (MAXLIN=500)
      INTEGER NLINE, MASK(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      CHARACTER*7 SOURCE(MAXLIN)
C
C     Local variables
C
      INTEGER I, J, N
      REAL XPOS, YPOS
      DOUBLE PRECISION ANGST
      REAL Z1, W1, W2, YP1, YP2, DMAX
      CHARACTER*100 STRING
      CHARACTER*10 COMMAND, REPLY, LINEID*11
      INTEGER SLOT, SLOT1, SLOT2, IFAIL, NCOM
      INTEGER MAXSPEC, KEEP, NPCHK
      LOGICAL SAMECI, DEFAULT, WAVE, CLOBBER, NEW
      LOGICAL ABORT
      REAL P1, P2
      DATA SLOT, SLOT1, SLOT2, NLINE/1, 1, 1, 0/
      DATA DMAX, ABORT/0.2, .FALSE./
      SAVE SLOT, ABORT
*
      MAXSPEC = MIN(MXBUFF/MAXPX, MXSPEC)
      DEFAULT = .FALSE.
      IFAIL = 0
      NCOM  = 0
      CALL INTR_IN('Slot containing arc', SPLIT, NSPLIT, 
     &     MXSPLIT, NCOM, DEFAULT, SLOT, 1, MAXSPEC, IFAIL)
      IF(IFAIL.NE.0) RETURN
      IF(NPIX(SLOT).LE.0) THEN
         WRITE(*,*) 'Slot empty'
        RETURN
      END IF
      IF(NARC(SLOT).NE.0) 
     &     WRITE(*,*) 'An arc fit is already present'
      P1  = 1.
      P2  = REAL(NPIX(SLOT))
      W1  = -1000.
      W2  = -1000.
      YP1 =  1.E30
      YP2 = -1.E30
      DO I = 1, NPIX(SLOT)
         YP1 = MIN(YP1, COUNTS(I,SLOT))
         YP2 = MAX(YP2, COUNTS(I,SLOT))
      END DO
*     
*     Interactive section
*
 100  WRITE(*,*) ' '
      WRITE(*,*) 'ARC commands:'
      WRITE(*,*) ' '
      WRITE(*,*) ' plot     -- Plot spectrum.'
      WRITE(*,*) ' new      -- Locate new lines.'
      WRITE(*,*) ' tweak    -- Tweak lines.'
      WRITE(*,*) ' fit      -- Fit identified lines.'
      WRITE(*,*) ' identify -- Identify lines.'
      WRITE(*,*) ' edit     -- Edit line list.'
      WRITE(*,*) ' show     -- Show lines.'
      WRITE(*,*) ' dump     -- Dump arc line list.'
      WRITE(*,*) ' load     -- Load arc line list.'
      WRITE(*,*) ' another  -- Select another arc'
      WRITE(*,*) ' comb     -- Combine tweak and fit'
      IF(ABORT) THEN
         WRITE(*,*) 
     &        ' abort    -- abort on tweak failure.' //
     &        ' Now ON.'
      ELSE
         WRITE(*,*) 
     &        ' abort    -- abort on tweak failure.' //
     &        ' Now OFF.'
      END IF
      WRITE(*,*) ' quit     -- Quit ARC'
 200  WRITE(*,*) ' '
      WRITE(*,'(A,$)') 'arc> '
      READ(*,'(A)') COMMAND
*     
      IF(SAMECI(COMMAND,'PLOT')) THEN
C
C     Plot arc spectrum
C
         WAVE = .FALSE.
         IF(NARC(SLOT).NE.0) THEN
            WRITE(*,'(A,$)') 'W(avelength) or P(ixel) ? [P] '
            READ(*,'(A)') REPLY
            CALL UPPER_CASE(REPLY)
            IF(REPLY.EQ.'W') THEN
               WAVE = .TRUE.
            ELSE IF(REPLY.EQ.'P' .OR. REPLY.EQ.' ') THEN
               WAVE = .FALSE.
            ELSE
               GOTO 200
            END IF
         END IF
         IF(WAVE) THEN
            IF(W1.LT.0.) THEN
               W1 = REAL(ANGST(1., NPIX(SLOT), NARC(SLOT), 
     &              ARC(1,SLOT), MXARC))
               W2 = REAL(ANGST(REAL(NPIX(SLOT)), NPIX(SLOT), 
     &              NARC(SLOT), ARC(1,SLOT), MXARC))
            END IF
            CALL SET_LIM(W1, W2, YP1, YP2, IFAIL)
            IF(IFAIL.NE.0) GOTO 200
         ELSE
            CALL SET_LIM(P1, P2, YP1, YP2, IFAIL)
            IF(IFAIL.NE.0) GOTO 200
         END IF
C     
         CALL DEV_OPEN(.FALSE.,NEW,IFAIL)
         IF(IFAIL.NE.0) GOTO 200
         CALL PGSCI(5)
         IF(WAVE) THEN
            CALL PGENV(W1, W2, YP1, YP2, 0, 0)
         ELSE
            CALL PGENV(P1, P2, YP1, YP2, 0, 0)
         END IF
         YLABEL = 'Counts'
         IF(WAVE) THEN
            XLABEL = 'Wavelength (\\A)'
         ELSE
            XLABEL = 'Pixels'
         END IF
         CALL PGSCI(7)
         TLABEL = 'Spectrum'
         WRITE(TLABEL(10:13), '(I4)') SLOT
         CALL PGLAB(XLABEL, YLABEL, TLABEL)
         CALL PGSCI(1)
C     
C     Load plot data in arrays.
C     
         N = 0
         DO J = 1, NPIX(SLOT)
            IF(ERRORS(J,SLOT).GT.0.) THEN
               N = N + 1
C     
C     Load X array
C     
               IF(WAVE) THEN
                  Z1 = REAL(J)
                  PLOT1(N) = REAL(ANGST(Z1, NPIX(SLOT), NARC(SLOT), 
     &                 ARC(1,SLOT), MXARC))
               ELSE 
                  PLOT1(N) = REAL(J)
               END IF
C     
C     Load Y array
C     
               PLOT2(N) = COUNTS(J,SLOT) 
C     
C     Load errors array
C     
               PLOT3(N) = ERRORS(J,SLOT)
            END IF
         END DO
         IF(N.GT.0) THEN
            CALL PGBIN(N, PLOT1, PLOT2, .TRUE.)
            CALL PGSCI(2)
            CALL PGBIN(N, PLOT1, PLOT3, .TRUE.)
            CALL PGSCI(1)
C     
C     Add line ids
C     
            CALL PGSCH(0.5)
            DO I = 1, NLINE
               IF(LAMBDA(I).GT.1.) THEN
                  IF((WAVE .AND. W1.LT.LAMBDA(I) .AND. 
     &                 W2.GT.LAMBDA(I)) .OR. (.NOT.WAVE .AND.
     &                 ((P2.GT.P1 .AND. P1.LT.PIXEL(I) .AND. 
     &                 P2.GT.PIXEL(I)) .OR.
     &                 (P2.LT.P1 .AND. P2.LT.PIXEL(I) .AND. 
     &                 P1.GT.PIXEL(I))))) THEN
                     IF(LAMBDA(I).LT.1000.) THEN
                        WRITE(LINEID,'(F8.4)') LAMBDA(I)
                     ELSE IF(LAMBDA(I).LT.10000.) THEN
                        WRITE(LINEID,'(F9.4)') LAMBDA(I)
                     ELSE IF(LAMBDA(I).LT.100000.) THEN
                        WRITE(LINEID,'(F10.4)') LAMBDA(I)
                     ELSE IF(LAMBDA(I).LT.1000000.) THEN
                        WRITE(LINEID,'(F11.4)') LAMBDA(I)
                     END IF
                     IF(WAVE) THEN
                        XPOS = REAL(LAMBDA(I))
                     ELSE
                        XPOS = PIXEL(I)
                     END IF
                     YPOS = MIN(PEAK(I),YP2)+(YP2-YP1)/100.
                     IF(PEAK(I).GT.(YP2-YP1)/100.) 
     &                    CALL PGPTEXT(XPOS,YPOS,90.,0.,LINEID)
                  END IF
               END IF
            END DO
            CALL PGIDEN
         END IF
         CALL DEV_CLOSE(0, IFAIL)
C     
C     End of plotting
C     
      ELSE IF(SAMECI(COMMAND,'NEW')) THEN
*     
*     Locate arc lines
*
         CALL ARC_FIND(COUNTS(1,SLOT), ERRORS(1,SLOT), NPIX(SLOT), 
     &        PIXEL, PISIG, PEAK, LAMBDA, SOURCE, MASK, NLINE, 
     &        MAXLIN)
*     
      ELSE IF(SAMECI(COMMAND,'TWEAK')) THEN
*     
*     Tweak positions of lines
*     
         WRITE(*,'(A,F5.2,A,$)') 
     &        'Enter maximum shift deviation [',DMAX,'] '
         READ(*,'(A)') STRING
         IF(STRING.NE.' ') THEN
            READ(STRING,*,IOSTAT=IFAIL) DMAX
            IF(IFAIL.NE.0) GOTO 200
         END IF
         IF(DMAX.LE.0.) THEN
            WRITE(*,*) 'Must be postive'
            DMAX = 1.
            GOTO 200
         END IF

         CALL ARC_TWEAK(COUNTS(1,SLOT), ERRORS(1,SLOT), NPIX(SLOT), 
     &        PIXEL, PISIG, PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN,
     &        .FALSE., PLOT1, KEY, MXPLOT, DMAX, IFAIL)
         IF(IFAIL.NE.0 .AND. ABORT) 
     &        STOP '''molly'' aborted on tweak failure'
*     
      ELSE IF(SAMECI(COMMAND,'FIT')) THEN
*     
*     Fit polynomial
*     
         CALL ARC_POLY( ARC(1,SLOT), NARC(SLOT), MXARC, NPIX(SLOT), 
     &        PIXEL, PISIG, PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN,
     &        PLOT1, PLOT2, PLOT3, MXPLOT, .FALSE.)
*     
*     Set character item to show that this is an arc spectrum
*     
         CALL HSETC('TYPE','ARC',SLOT,MXSPEC,NMCHAR,HDCHAR,NCHAR,
     &        MXCHAR,IFAIL)
*     
      ELSE IF(SAMECI(COMMAND,'IDENTIFY')) THEN
*     
*     Identify arc lines
*     
         CALL ARC_IDENT( ARC(1,SLOT), NARC(SLOT), MXARC, NPIX(SLOT), 
     &        PIXEL, PISIG, PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN)
*     
      ELSE IF(SAMECI(COMMAND, 'EDIT')) THEN
*     
*     Edit list
*     
         CALL ARC_EDIT(PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &        MASK, NLINE, MAXLIN)
*     
      ELSE IF(SAMECI(COMMAND,'SHOW')) THEN
*     
*     List arclines
*     
         CALL ARC_LIST (0, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &        MASK, NLINE, MAXLIN)
*     
      ELSE IF(SAMECI(COMMAND,'DUMP')) THEN
*     
*     Dump arc lines to disk
*     
         CALL ARC_DUMP(PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &        MASK, NLINE, MAXLIN, CLOBBER)
*     
      ELSE IF(SAMECI(COMMAND,'LOAD')) THEN
*     
*     Load arc lines from disk
*     
         CALL ARC_LOAD(PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &        MASK, NLINE, MAXLIN)
*     
      ELSE IF(SAMECI(COMMAND,'ANOTHER')) THEN
         KEEP = SLOT
         WRITE(*,'(A,I4,A,$)') 'Enter arc slot [',SLOT,'] '
         READ(*,*,IOSTAT=IFAIL) SLOT
         IF(IFAIL.EQ.0) THEN
            IF(SLOT.LT.1 .OR. SLOT.GT.MAXSPEC) THEN
               WRITE(*,*) 'Slot out of range 1 - ',MAXSPEC
               WRITE(*,*) 'Retaining old slot ',KEEP
               SLOT = KEEP
            ELSE IF(NPIX(SLOT).EQ.0) THEN
               WRITE(*,*) SLOT,' is empty.'
               WRITE(*,*) 'Retaining old slot ',KEEP
               SLOT = KEEP
            END IF
            IF(NARC(SLOT).NE.0) 
     &           WRITE(*,*) 'An arc fit is already present'
         ELSE
            WRITE(*,*) 'Retaining old slot ',KEEP
            SLOT = KEEP
         END IF

      ELSE IF(SAMECI(COMMAND,'ABORT')) THEN
         ABORT = .NOT.ABORT
         IF(ABORT) THEN
            WRITE(*,*) 'molly will be aborted if tweak fails'
         ELSE
            WRITE(*,*) 'molly will not be aborted if tweak fails'
         END IF

      ELSE IF(SAMECI(COMMAND,'COMB')) THEN
         WRITE(*,'(A,I4,A,I4,A,$)') 'Enter range of slots [',
     &        SLOT1,',',SLOT2,'] '
         READ(*,'(A)') STRING
         IF(STRING.NE.' ') THEN
            READ(STRING,*,IOSTAT=IFAIL) SLOT1, SLOT2
            IF(IFAIL.NE.0) GOTO 200
         END IF
         IF(SLOT1.LT.1 .OR. SLOT1.GT.MAXSPEC .OR.
     &        SLOT2.LT.1 .OR. SLOT2.GT.MAXSPEC) THEN
            WRITE(*,*) 'Slots out of range 1 - ',MAXSPEC
            GOTO 200
         END IF
         WRITE(*,'(A,F5.2,A,$)') 
     &        'Enter maximum shift deviation [',DMAX,'] '
         READ(*,'(A)') STRING
         IF(STRING.NE.' ') THEN
            READ(STRING,*,IOSTAT=IFAIL) DMAX
            IF(IFAIL.NE.0) GOTO 200
         END IF
         IF(DMAX.LE.0.) THEN
            WRITE(*,*) 'Must be positive'
            DMAX = 1.
            GOTO 200
         END IF
*     
         NPCHK = NPIX(SLOT1)
         DO I = SLOT1, SLOT2
            IF(NPIX(I).NE.NPCHK) THEN
               WRITE(*,*) ' '
               WRITE(*,*) 'Slot ',I,' has different number of pixels (',
     &              NPIX(I),') from slot ',SLOT1,
     &              ' (first one) which has ',NPCHK,' pixels.'
C               IF(ABORT) STOP '''molly'' aborted on tweak failure'
               GOTO 200
            END IF
            CALL ARC_TWEAK(COUNTS(1,I), ERRORS(1,I), NPIX(I), 
     &           PIXEL, PISIG, PEAK, LAMBDA, SOURCE, MASK, NLINE, 
     &           MAXLIN, .TRUE., PLOT1, KEY, MXPLOT, DMAX, IFAIL)
            IF(IFAIL.NE.0) THEN
               IF(IFAIL.EQ.1 .AND. ABORT) 
     &              STOP '''molly'' aborted on tweak failure'
               GOTO 200
            END IF
*     
            CALL ARC_POLY( ARC(1,I), NARC(I), MXARC, NPIX(I), 
     &           PIXEL, PISIG, PEAK, LAMBDA, SOURCE, MASK, NLINE, 
     &           MAXLIN, PLOT1, PLOT2, PLOT3, MXPLOT, .TRUE.)
*     
*     Set character item to show that this is an arc spectrum
*     
            CALL HSETC('TYPE','ARC', I, MXSPEC,NMCHAR,HDCHAR,
     &           NCHAR,MXCHAR,IFAIL)

         END DO
*     
      ELSE IF(SAMECI(COMMAND,'QUIT')) THEN
         RETURN
      ELSE
         GOTO 100 
      END IF
      GOTO 200
      END
      
      SUBROUTINE ARC_FIND(ASPEC, AERR, NPIX, PIXEL, PISIG, 
     &     PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN)
*     
*     Locates arc lines. Adapted from ARCFIND in RUBY
*
      IMPLICIT NONE
      INTEGER NPIX, GET, PUT
      REAL ASPEC(NPIX), AERR(NPIX)
      INTEGER MAXLIN, NLINE
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN)
*
      REAL THRESH, FWHM, AVOID, MEDVAL, RBINSIG
      REAL BOT, TOP, MED, PLACE, DIFF
      INTEGER P1, P2, I1, I2, I, J, IP, IM
      INTEGER INSERT, NFIND
      CHARACTER*100 STRING
      DATA THRESH, FWHM, AVOID/1000., 4., 2./
      DATA P1, P2/1, 1000000/
*
      P1 = MAX(1, P1)
      P2 = MIN(NPIX, P2)
*
* Find lines.
*
      WRITE(*,'(A,I4,A,I4,A,$)') 'Enter pixel range [',P1,',',P2,'] '
      READ(*,'(A)') STRING
      IF(STRING.NE.' ') THEN
        READ(STRING,*,ERR=999) I1, I2
         write(*,*) 'should not be here'
        IF(I1.LT.1 .OR. I1.GT.NPIX .OR.
     &    I1.GE.I2) THEN
          WRITE(*,*) 'Invalid range'
          RETURN
        END IF
        P1 = I1
        P2 = I2
      END IF
      BOT = 1.E30
      TOP = -1.E30
      DO I = P1, P2
        BOT = MIN(BOT, ASPEC(I))
        TOP = MAX(TOP, ASPEC(I))
      END DO
      MED = MEDVAL(ASPEC(P1),P2-P1+1)
      WRITE(*,*) 'Smallest value in range = ',BOT
      WRITE(*,*) ' Largest value in range = ',TOP
      WRITE(*,*) '           Median value = ',MED
      WRITE(*,'(A,G10.4,A,$)') 'Search threshold [',THRESH,'] '
      READ(*,'(A)') STRING
      IF(STRING.NE.' ') READ(STRING,*,ERR=999) THRESH
      WRITE(*,'(A,G10.4,A,$)') 'Gaussian FWHM [',FWHM,'] '
      READ(*,'(A)') STRING
      IF(STRING.NE.' ') THEN
        READ(STRING,*,ERR=999) FWHM
        IF(FWHM.LT.2.) THEN
          WRITE(*,*) 'Minimum value = 2'
          GOTO 999
        END IF
      END IF
      WRITE(*,'(A,G10.4,A,$)') 'Minimum separation [',AVOID,'] '
      READ(*,'(A)') STRING
      IF(STRING.NE.' ') READ(STRING,*,ERR=999) AVOID
*
* Now search for lines
*
      NFIND = 0
      DO 299 I= P1, P2
        IF(ASPEC(I).LT.THRESH) GOTO 299
        IM = MAX(I-1,1)
        IP = MIN(I+1,NPIX)
        IF(ASPEC(I).LE.ASPEC(IM)) GOTO 299
        IF(ASPEC(I).LE.ASPEC(IP)) GOTO 299
*
*  Refine the line position using gaussian cross-correlation algorithm
*
        CALL MEDIAN( ASPEC, NPIX, I, FWHM, PLACE, RBINSIG, 1)
*
*  Locate new line's position in the line list
*
        INSERT=1
        IF(NLINE.GE.1) THEN
          DO J=1,NLINE
            DIFF = PIXEL(J) - PLACE
*
*  Omit duplicate line
*
            IF(ABS(DIFF).LT.AVOID) THEN
              WRITE(*,*) '** DUPLICATE LINE AT PIXEL', PLACE, 
     &        ' OMITTED.'
              GOTO 299
            END IF
*
            IF(DIFF.GT.0.) GOTO 220
          END DO
          J = NLINE + 1
220       INSERT = J
        END IF
*
*  Make room for the new line
*
        IF(INSERT.LE.NLINE) THEN
          DO GET = NLINE,INSERT,-1
            PUT = GET + 1
            PIXEL(PUT)   = PIXEL(GET)
            PISIG(PUT)   = PISIG(GET)
            PEAK(PUT)    = PEAK(GET)
            LAMBDA(PUT)  = LAMBDA(GET)
            SOURCE(PUT)  = SOURCE(GET)
            MASK(PUT)    = MASK(GET)
          END DO
        END IF
*
*  Add the new line
*
        NLINE = NLINE + 1
        NFIND = NFIND + 1
        PIXEL(INSERT) = PLACE
        PISIG(INSERT) = RBINSIG
        PEAK(INSERT)  = ASPEC( NINT(PLACE) )
        LAMBDA(INSERT)= 0.D0
        SOURCE(INSERT)= ' '
        MASK(INSERT)  = 0
*
*  Report new line on terminal
*
        IF(NFIND.EQ.1) CALL ARC_LIST(-1, PIXEL, PISIG, PEAK, 
     &  LAMBDA, SOURCE, MASK, NLINE, MAXLIN)
        CALL ARC_LIST(INSERT, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
*
        IF(NLINE.GE.MAXLIN) THEN
          WRITE(*,*) '** LINE LIST IS FULL.', NLINE, MAXLIN
          GOTO 300
        END IF
*
299   END DO
*
300   WRITE(*,*) 'Lines found:', NFIND
999   CONTINUE
      RETURN
      END

      SUBROUTINE ARC_LIST (LINE, PIXEL, PISIG, 
     &PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN)
*
*  Summarizes arc lines on user terminal
*
*  Input:
*      LINE      = LINE NUMBER
*            = 0 FOR ENTIRE LINE LIST
*            = NEGATIVE VALUE TO TYPE COLUMN HEADING ONLY
*
*  AUGUST 1984 BY KEITH HORNE AT IOA.
*
* Adapted for MOLLY by TRM @OXVAD, 22/06/92
*
      IMPLICIT NONE
      INTEGER MAXLIN, NLINE, LINE
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN), NROWS, J
      INTEGER LINE1, LINE2, NCOLUMNS, LAST, FIRST
      CHARACTER*1 CODE(2)
      DATA CODE/' ','*'/
*
      IF(NLINE.LE.0) THEN
        WRITE(*,*) '** NO ARC LINES.'
        RETURN
      END IF
*
*  Determine range of lines to be listed
*
      IF(LINE.EQ.0) THEN
        LINE1 = 1
        LINE2 = NLINE
      ELSE
        LINE1 = LINE
        LINE2 = LINE
      END IF
      NCOLUMNS = MIN(2,LINE2-LINE1+1)
*
*  Column heading
*
      IF(LINE.LE.0) THEN
        WRITE(*,*) ' '
        IF(NCOLUMNS.EQ.1) THEN
          WRITE(*,*) ' Line     Pixel  Angstrom Source   Peak '
        ELSE IF(NCOLUMNS.EQ.2) THEN
          WRITE(*,*) ' Line     Pixel  Angstrom Source   Peak '//
     &               ' Line     Pixel  Angstrom Source   Peak '
        END IF
        IF( LINE.LT.0) RETURN
      END IF
*
*  Loop to list line summary
*
      NROWS = (LINE2-LINE1)/NCOLUMNS + 1
      DO FIRST = LINE1, LINE1+NROWS-1
        LAST = FIRST + (NCOLUMNS-1)*NROWS
        IF( LAST .GT. NLINE) LAST = NLINE
        IF(NCOLUMNS.EQ.1) THEN
          WRITE(*,'(I5,A1,F9.3,F10.3,1X,A6,F8.0)') 
     &    (J, CODE(MASK(J)+1), PIXEL(J), LAMBDA(J), 
     *                  SOURCE(J), PEAK(J), J=FIRST,LAST,NROWS)
        ELSE IF(NCOLUMNS.EQ.2) THEN
          WRITE(*,'(2(I5,A1,F9.3,F10.3,1X,A6,F8.0))') 
     &    (J, CODE(MASK(J)+1), PIXEL(J), LAMBDA(J), 
     *                  SOURCE(J), PEAK(J), J=FIRST,LAST,NROWS)
        END IF
      END DO
*
      RETURN
      END

      SUBROUTINE ARC_DUMP(PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN, CLOBBER)
*
*  Dumps arc line list to disk
*
* Adapted for MOLLY by TRM @OXVAD, 22/06/92
*
      IMPLICIT NONE
      INTEGER MAXLIN, NLINE, I
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN)                       
      CHARACTER*60 FILE
      LOGICAL CLOBBER
*
      IF(NLINE.LT.2) THEN
        WRITE(*,*) '** NO ARC LINES. IN ''ARC_DUMP''.'
        RETURN
      END IF
*
*  Open disk file
*
      WRITE(*,*) ' '
      WRITE(*,*) 'Ready to dump arc line table to a disk file.'
      WRITE(*,*) 'Enter name of disk file'
      READ(*,'(A)') FILE
      IF(CLOBBER) THEN
        OPEN(UNIT=47,FILE=FILE,STATUS='UNKNOWN',FORM='FORMATTED',
     &  ERR=30)
      ELSE
        OPEN(UNIT=47,FILE=FILE,STATUS='NEW',FORM='FORMATTED',
     &  ERR=30)
      END IF
      WRITE(*,*) 'File open successful.'
*
*  Dump arc lines to disk file
*
      DO I=1,NLINE
        WRITE(47,'(3(F10.3),1X,A6,F15.1,I4)',ERR=40) 
     *  LAMBDA(I), PIXEL(I), PISIG(I), SOURCE(I), PEAK(I), MASK(I)
      END DO
*
      WRITE(*,*) 'Arc lines dumped:', NLINE
      GOTO 50
*
*  Report trouble opening disk file
*
30    WRITE(*,*) CHAR(7),'** ERROR OCCURRED OPENING DISK FILE.'
      RETURN
*
*  Report trouble writing to disk file
*
40    WRITE(*,*) CHAR(7),'** ERROR OCCURRED WRITING TO DISK FILE.'
      GOTO 50
*
*  Close disk file
*
50    CLOSE(UNIT=47)
      WRITE(*,*) 'Disk file closed.'
      RETURN
      END

      SUBROUTINE ARC_EDIT(PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
*
*  Edits arc line list 
*
* Adapted for MOLLY by TRM @OXVAD, 22/06/92
*
      IMPLICIT NONE
      INTEGER MAXLIN, NLINE, I
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN), LINE1, LINE2, IGET, IPUT
      CHARACTER*8 COMMAND
      LOGICAL SAMECI
*
      IF(NLINE.LE.0) THEN
        WRITE(*,*) '** NO ARC LINES IN ''ARCEDIT''.'
        RETURN
      END IF
      GOTO 100
*
*  List arc lines on user terminal ------------------------------
*
100   CALL ARC_LIST(0,PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
*
*  Command node --------------------------------------------------
*
10    WRITE(*,*) ' '
      WRITE(*,'(A,$)') 'Delete, Mask, Unmask or Quit ?'
      READ(*,'(A)') COMMAND
*
      IF( SAMECI( COMMAND, 'QUIT' ) )      RETURN
      IF( SAMECI( COMMAND, 'MASK' ) )      GOTO 200
      IF( SAMECI( COMMAND, 'UNMASK'))      GOTO 300
      IF( SAMECI( COMMAND, 'DELETE'))      GOTO 400
      GOTO 100
*
*  MASK ARC LINES -----------------------------------------------
*
200   WRITE(*,'(A,I3,A)') ' Enter range of arc lines (1,',NLINE,
     &') to be MASKED : '
      READ(*,*,ERR=10) LINE1,LINE2
*
      LINE1 = MAX(LINE1,1)
      LINE2 = MIN(LINE2,NLINE)
      IF(LINE1.GT.LINE2) THEN
        WRITE(*,*) '** INVALID ARC LINE RANGE REQUESTED.'
        GOTO 10
      END IF
*
      CALL ARC_LIST(-1, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
      DO I=LINE1,LINE2
        CALL ARC_LIST(I, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
        MASK(I)=1
      END DO
*
      WRITE(*,*) 'Total number of arc lines MASKED:', LINE2 - LINE1 + 1
      GOTO 10
*
*  Unmask arc lines ---------------------------------------------
*
300   WRITE(*,'(A,I3,A)') ' Enter range of arc lines (1,',NLINE,
     &') to be UNMASKED : '
      READ(*,*,ERR=10) LINE1,LINE2
*
      LINE1 = MAX(LINE1,1)
      LINE2 = MIN(LINE2,NLINE)
      IF(LINE1.GT.LINE2) THEN
        WRITE(*,*) '** INVALID ARC LINE RANGE REQUESTED.'
        GOTO 10
      END IF
*
      CALL ARC_LIST(-1, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
      DO I=LINE1,LINE2
        CALL ARC_LIST(I, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
        MASK(I)=0
      END DO
*
      WRITE(*,*) 'Total number of arc lines UNMASKED:', 
     &LINE2 - LINE1 + 1
      GOTO 10
*
*  Delete unwanted arc lines -----------------------------
*
400   WRITE(*,*) ' '
      WRITE(*,'(A,I3,A)') ' Enter range of arc lines (1,',NLINE,
     &') to be DELETED : '
      READ(*,*,ERR=10) LINE1, LINE2
*
      LINE1 = MAX(LINE1,1)
      LINE2 = MIN(LINE2,NLINE)
      IF(LINE1.GT.LINE2) THEN
        WRITE(*,*) 'No lines deleted.'
        GOTO 10
      END IF
*
      IGET = LINE2
      WRITE(*,*) 'Arc lines deleted :'
      CALL ARC_LIST(-1, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)                  
      DO IPUT=LINE1,NLINE
        IF(IPUT.LE.LINE2) CALL ARC_LIST(IPUT, PIXEL, PISIG, PEAK, 
     &  LAMBDA, SOURCE, MASK, NLINE, MAXLIN)
        IF(IGET.LT.NLINE) THEN
          IGET = IGET+1
          PIXEL(IPUT) = PIXEL(IGET)
          PISIG(IPUT) = PISIG(IGET)
          LAMBDA(IPUT) = LAMBDA(IGET)
          SOURCE(IPUT) = SOURCE(IGET)
          PEAK(IPUT) = PEAK(IGET)
          MASK(IPUT) = MASK(IGET)
        END IF
      END DO
      NLINE = NLINE - LINE2 + LINE1 - 1
*
      IF(NLINE.LE.0) THEN
        WRITE(*,*) '** ALL LINES DELETED.'
        RETURN
      END IF
      WRITE(*,*) 'Total number of arc lines deleted :',LINE2-LINE1+1
      GOTO 10
*
      END

      SUBROUTINE ARC_IDENT( ARC, NARC, MXARC, NPIX, PIXEL, PISIG, 
     &PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN)
*
*  Idenitify arc lines
*
* Adapted for MOLLY by TRM @OXVAD, 22/06/92
*
*
*
*  Input:
*      ARC      = ARCFIT COEFFICIENTS
* 
*      NARC      = Number of arcfit coefficients (negative if log scale)
*                = 0 if no arcfit coefficients are available
*
*      MXARC     = Max number of arc coefficients
*
*      NPIX      = Number of pixels in arc spectrum
*
      IMPLICIT NONE
      INTEGER MAXLIN, NLINE, MXARC, NARC, NPIX, I
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN)                       
      DOUBLE PRECISION ARC(MXARC)
      CHARACTER*1 REPLY
      CHARACTER*7 NAME
      DOUBLE PRECISION WPOLY, WLOOKUP, DISPER
      DOUBLE PRECISION ANGST, DISPA
      INTEGER LINE1, LINE2, MODE, NMATCH
      REAL ANGNEW, SLOP
*
      IF(NLINE.LE.0) THEN
        WRITE(*,*) '** NO ARC LINES IN ''ARCIDENT''.'
        RETURN
      END IF
*
*  Introduction
*
      WRITE(*,*) ' '
      WRITE(*,*) 'ARCIDENT -- Assign an '//
     &'angstrom value to each arc line.'
*
*  Get range of arc lines to be identified
*
1     CALL ARC_LIST(0, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
      WRITE(*,'(A,I3,A,$)')
     *      'Enter range of arc lines (1,', NLINE, ') '//
     &'to be IDENTIFIED : '
      READ(*,*,ERR=1) LINE1, LINE2
*
      IF(LINE1.EQ.0 .AND. LINE2.EQ.0) RETURN
*
      LINE1 = MAX(LINE1,1)
      LINE2 = MIN(LINE2,NLINE)
      IF(LINE1.GT.LINE2) THEN
        WRITE(*,*) CHAR(7),'** INVALID RANGE REQUESTED.'
        RETURN
      END IF
*
*  Load a list of accurate wavelengths from a disk file
*
      MODE = -1
      CALL ARC_LOOKUP( WPOLY, SLOP, WLOOKUP, NAME, MODE)
*
*  Loop through the arc lines
*
      DO 100 I = LINE1, LINE2
*
*  Type information about this arc line
*
10      WRITE(*,*) ' '
        CALL ARC_LIST(-1, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
        CALL ARC_LIST(I, PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &  MASK, NLINE, MAXLIN)
        WRITE(*,'(A, F10.3, A, A)')
     *  '      Default Angstrom value =', LAMBDA(I), 
     &' Source =', SOURCE(I)
*
*  Get angstrom value from polyfit
*
        WPOLY = LAMBDA(I)
        IF(IABS(NARC).GT.1) THEN
          WPOLY = ANGST(PIXEL(I), NPIX, NARC, ARC, MXARC)
          WRITE(*,'(1X,A,F10.3)') 'Angstrom value from POLYFIT =',WPOLY
        END IF
*
*  Get angstrom value from lookup table
*
        WLOOKUP = 0.D0
        IF( WPOLY.GT.0.D0) THEN
*
          IF(IABS(NARC).GT.1) THEN
            DISPER = DISPA(PIXEL(I), NPIX, NARC, ARC, MXARC)
            SLOP = REAL(1.5*ABS(DISPER))
            SLOP = MAX(0.2, SLOP)
          ELSE
            SLOP = 0.2
          END IF
*
          NMATCH = 0
          CALL ARC_LOOKUP( WPOLY, SLOP, WLOOKUP, NAME, NMATCH)
*
          IF(NMATCH.LE.0) THEN
            WRITE(*,*) 
     *      '** NO ACCURATE ANGSTROM VALUES FOUND IN LOOKUP TABLE'
            WRITE(*,'(1X,A,F8.3,A,F10.3)')
     *      '** IN THE', 2*SLOP, ' ANGSTROM RANGE AROUND', WPOLY
          ELSE
            WRITE(*,'(A, F10.3, A, A)')
     *      ' Angstrom value from  LOOKUP =', WLOOKUP, 
     &      ' Source =', NAME
            IF(NMATCH.GT.1) THEN
              WRITE(*,'(1X,A,I3,A)')
     *        '** THIS IS THE CLOSEST OF', NMATCH,
     &        ' POSSIBLE ANGSTROM VALUES'
              WRITE(*,'(1X,A,F8.3,A,F10.3)')
     *        '** IN THE', 2*SLOP, ' ANGSTROM RANGE AROUND', WPOLY
            END IF
          END IF
*
        END IF
*
*  Command node --------------------------------------------------
*
        WRITE(*,*) ' '
        WRITE(*,*) 'Type ''E'' to ENTER new angstrom value'
        IF(WLOOKUP.GT.0.D0) 
     *      WRITE(*,*) '     ''L'' to adopt angstrom value from LOOKUP'
        IF( LAMBDA(I).GT.0.)
     *      WRITE(*,*) '     ''C'' to CANCEL angstrom value'
        WRITE(*,*) '    <CR> to use DEFAULT angstrom value'
        WRITE(*,*) '     ''A'' to ABORT'
*
        WRITE(*,'(A,$)') 'arcident> '
        READ(*,'(A)') REPLY
        CALL UPPER_CASE(REPLY)
*
        IF( REPLY.EQ.'L' .AND. WLOOKUP.GT.0.D0) GOTO 60
        IF( REPLY.EQ.'A') GOTO 1
        IF( REPLY.EQ.'C') GOTO 70
        IF( REPLY.EQ.'E') GOTO 80
        IF( REPLY.EQ.' ') GOTO 90
        GOTO 10
*
60      LAMBDA(I) = WLOOKUP
        SOURCE(I) = NAME
        GOTO 10
*
70      LAMBDA(I) = 0.D0
        SOURCE(I) = ' '
        GOTO 10
*
80      WRITE(*,*) ' '
        WRITE(*,*) 'Enter new angstrom value : '//
     &  '(0 to keep previous value)'
        READ(*,*,ERR=80) ANGNEW
        IF( ANGNEW.GT.0.) THEN
          LAMBDA(I) = ANGNEW
          SOURCE(I) = 'User'
        END IF
        GOTO 10
*
*  Proceed to next arc line
*
90      WRITE(*,'(1X,A,F10.3)') 'Adopted angstrom value:', LAMBDA(I)
*
100   END DO
*
      GOTO 1
      END

      SUBROUTINE ARC_LOAD(PIXEL, PISIG, PEAK, LAMBDA, SOURCE, 
     &MASK, NLINE, MAXLIN)
*
*  LOAD ARC LINE TABLE FROM A DISK FILE
*
*  JULY 1984 BY KEITH HORNE AT IOA
*
*
*  22/6/92 Modified for MOLLY by TRM
*
      IMPLICIT NONE
      INTEGER MAXLIN, NLINE
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN)                       
      CHARACTER*60 FILE
*
*  Open disk file
*
      WRITE(*,*) ' '
      WRITE(*,*) 'Ready to load arc line table from a disk file.'
      WRITE(*,*) 'Enter name of disk file'
      READ(*,'(A)') FILE
      OPEN( UNIT=47, FILE=FILE, STATUS='OLD', FORM='FORMATTED', ERR=30)
      WRITE(*,*) 'File open successful.'
*
*  Read arc lines from the disk file
*
      NLINE = 0
20    NLINE = NLINE + 1
      READ(47,'(3(F10.3),1X,A6,F15.1,I4)',END = 25, ERR=40) 
     &LAMBDA(NLINE), PIXEL(NLINE), PISIG(NLINE), SOURCE(NLINE), 
     &PEAK(NLINE) , MASK(NLINE)
      GOTO 20
*
*  Normal end of file encountered
*
25    NLINE = NLINE - 1
      WRITE(*,*) 'Arc lines LOADED:', NLINE
      GOTO 50
*
*  Report trouble opening disk file
*
30    WRITE(*,*) CHAR(7),'** ERROR OCCURRED OPENING DISK FILE.'
      RETURN
*
*  Report trouble reading from disk file
*
40    WRITE(*,*) CHAR(7),'** ERROR OCCURRED READING FROM DISK FILE.'
      GOTO 50
*
*  Close disk file
*
50    CLOSE(UNIT=47)
      WRITE(*,*) 'Disk file closed.'
      RETURN
      END

      SUBROUTINE ARC_LOOKUP( ANGINP, SLOP, ANGFIND, SOURCE, NFIND )
*
*  LOOKS UP ACCURATE ANGSTROM VALUES FROM A DISK FILE
*
*  Input:
*
*      ANGINP     = (REAL*8) APPROXIMATE WAVELENGTH (ANGSTROMS)
*      SLOP       = LARGEST ACCEPTABLE DISCREPANCY (ANGSTROMS)
*      NFIND      = NEGATIVE TO LOAD NEW ANGSTROM VALUES FROM A DISK FILE
*
*  Output:
*
*      ANGFIND    = (REAL*8) ACCURATE WAVELENGTH (ANGSTROMS)
*      SOURCE     = SOURCE OF ANGSTROM VALUE
*      NFIND      = NUMBER OF ANGSTROM VALUES FOUND BETWEEN
*                   ANGINP-SLOP AND ANGINP+SLOP
*
* JULY 1984 BY KEITH HORNE AT IOA.
*
      IMPLICIT NONE
      DOUBLE PRECISION ANGINP, ANGFIND
      CHARACTER*(*) SOURCE
      CHARACTER*60 FILE
      CHARACTER*10 REPLY
      INTEGER MAXTABLE
      PARAMETER (MAXTABLE=1000)
      DOUBLE PRECISION ANG(MAXTABLE)
      CHARACTER*7 NAME(MAXTABLE)
      INTEGER NFIND, NTABLE, ICHAMP, I, LINE1
      REAL SLOP, SMALLEST, TEST
      DATA ICHAMP, NTABLE/1,0/
*
      IF(NFIND.GE.0) GOTO 100
*
*  Introduction
*
      WRITE(*,*) ' '
      WRITE(*,*) 'ARCLOOKUP -- If you now load '//
     &'a list of accurate angstrom'
      WRITE(*,*) 'values from a disk file,'//
     &' you later may use approximate'
      WRITE(*,*) 'angstrom values when identifying arc lines.'
*
*  List angstrom value table -------------------------------
*
10    WRITE(*,*) ' '
      WRITE(*,*) 'Total number of angstrom values loaded:', NTABLE
*
*  Command node ------------------------------------------
*
      WRITE(*,*) ' '
      WRITE(*,*) 'Type ''C'' to CLEAR the table of angstrom values'
      WRITE(*,*) '     ''D'' to load additional '//
     &'angstrom values from a DISK file'
      WRITE(*,*) '    <CR> to continue'
      READ(*,'(A)') REPLY
      CALL UPPER_CASE(REPLY)
      IF(REPLY.EQ.'C') NTABLE = 0
      IF(REPLY.EQ.'D') GOTO 20
      IF(REPLY.EQ.' ') GOTO 100
      GOTO 10
*
*  Open disk file containing accurate angstrom values -------------------
*      
20    WRITE(*,*) ' '
      WRITE(*,*) 'Enter name of DISK FILE containing angstrom values.'
      READ(*,'(A)') FILE
      OPEN( UNIT=47, FILE=FILE, STATUS='OLD', FORM='FORMATTED',
     *      ERR=25)
      WRITE(*,*) 'File open successful.'
*
*  Read angstrom values
*
      LINE1 = NTABLE + 1
      DO NTABLE = LINE1, MAXTABLE
        READ(47,*,END=50,ERR=30 ) ANG(NTABLE), NAME(NTABLE)
      END DO
*
*  Report line table overflow
*
      WRITE(*,*)  CHAR(7), '** LINE TABLE OVERFLOW.'
      NTABLE = MAXTABLE
      GOTO 60
*
*  Report trouble opening file
*
25    WRITE(*,*)  CHAR(7), '** ERROR ON FILE OPEN'
      GOTO 10
*
*  Report trouble reading from file
*
30    WRITE(*,*)  CHAR(7), '** ERROR READING FROM DISK FILE'
      NTABLE = NTABLE - 1
      GOTO 60
*
*  Report end of file
*
50    WRITE(*,*) 'Normal end of file encountered.'
      NTABLE = NTABLE - 1
      GOTO 60
*
*  Close disk file
*
60    CLOSE (UNIT=47)
      WRITE(*,*) 'File closed.'
      WRITE(*,*) 'New angstrom values loaded:',NTABLE-LINE1+1
      WRITE(*,*) ' '
      IF(NTABLE.GT.0)
     *WRITE(*,'(4(2X,F10.3,1X,A6))') (ANG(I),NAME(I),I=1,NTABLE)
      WRITE(*,*) 'Total number of angstrom values loaded:', NTABLE
      GOTO 10
*
*  Search lookup table for accurate angstrom value --------------------
*
100   NFIND = 0
      IF( NTABLE.GT.0) THEN
        SMALLEST = REAL(ABS(ANGINP - ANG(ICHAMP)))
        DO I=1,NTABLE
          TEST = REAL(ABS( ANGINP - ANG(I)))
          IF(TEST .LT. SLOP) THEN
            NFIND = NFIND + 1
            IF(TEST .LT. SMALLEST) THEN
              ICHAMP = I
              SMALLEST = TEST
            END IF
          END IF
        END DO
        IF( NFIND.GT.1) THEN
          DO I=1,NTABLE
            TEST = REAL(ABS( ANGINP - ANG(I)))
            IF(TEST .LT. SLOP) THEN
              WRITE(*,*) ANG(I),' ', NAME(I)
            END IF
          END DO
        END IF
      END IF
*
      IF(NFIND.GT.0) THEN
        ANGFIND = ANG(ICHAMP)
        SOURCE = NAME(ICHAMP)
      ELSE
        ANGFIND = 0.D0
        SOURCE = ' '
      END IF
*
      RETURN
      END

      SUBROUTINE ARC_POLY( ARC, NARC, MXARC, NPIX, PIXEL, PISIG, 
     &     PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN, PLOT1, 
     &     PLOT2, PLOT3, MAXPLOT, COMB)
*
*
*  Polynomial fit to (angstrom-pixel) pairs
*
*  Output:
*      ARC      = ARCFIT COEFFICIENTS
*      NARC     = NUMBER OF COEFFICIENTS
*               = 0 IF NO POLYFIT WAS PERFORMED
*
*  JULY 1984 BY KEITH HORNE AT IOA
*  OCT  1990 TRM fixed bug which caused crashed if masked lines were present
*
*  Modified 22/6/92 by TRM
*
      IMPLICIT NONE
      INTEGER MAXLIN, NLINE, MXARC, NARC, NPIX, I
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN), IFAIL
      DOUBLE PRECISION ARC(MXARC)
      LOGICAL COMB, NEW
      INTEGER MAXFIT, MXPOLY, MAXPLOT
      PARAMETER (MAXFIT=200, MXPOLY=20)
      DOUBLE PRECISION ARCLINEAR(2)
      DOUBLE PRECISION XDATA(MAXFIT), YDATA(MAXFIT), YERR(MAXFIT)
      DOUBLE PRECISION WORK(MXPOLY, 2*MXPOLY+3)
      DOUBLE PRECISION WCALC, DCALC, CALC, GCALC, ANGST, DISPA
      REAL  PLOT1(MAXPLOT), PLOT2(MAXPLOT), PLOT3(MAXPLOT)
      INTEGER NGPOLY, NFIT, NGPIX, NALL, NPART, NEWTWEAK, NEWPOLY
      INTEGER NPLOT, MAXPOLY, LASTPOLY, NTWEAK
      REAL RMS, Z, RMSALL, RMSFIT, SFIT, SALL, ERROR, WEIGHT, WEG
      REAL AFAC, FFAC, PART, PLOW, PHIGH, PLOW1, PHIGH1, WAV1, WAV2
      REAL XMN, XMX, YMN, YMX, XPLOT, YPLOT, WFACTOR
      DOUBLE PRECISION GRANDARC(MXPOLY)            
      LOGICAL MASTER
      CHARACTER*7 CONFIRM, STRING*100
      CHARACTER*1 REPLY
      SAVE NGPOLY, NGPIX, LASTPOLY, NTWEAK, MASTER
      SAVE WFACTOR, PLOW, PHIGH
      DATA LASTPOLY, NTWEAK, MASTER/2, 2, .FALSE./
      DATA WFACTOR/0./
      DATA NGPOLY, NGPIX/0, 0/
      DATA PLOW, PHIGH/0.,1.E9/

C
C     Introduction
C
      MAXPOLY = MIN(MXARC, MXPOLY)
      WRITE(*,*) ' '
      WRITE(*,*) 'ARCPOLY - A polynomial giving '//
     &     'the angstrom value corresponding'
      WRITE(*,*) 'to each pixel is fitted to '//
     &     'the arc line angstrom/pixel pairs.'
C    
C     Check inputs
C
      IF(NLINE.LE.0.) THEN
         WRITE(*,*) '** NO ARC LINES IN ''ARCPOLY''.', NLINE
         GOTO 999
      END IF
      IF(NPIX.LE.0) THEN
         WRITE(*,*) '** INVALID NPIX IN ''ARCPOLY''.', NPIX
         GOTO 999
      END IF
      NARC = LASTPOLY

C     
C     Load arrays for arc poly-fit by 'LEASQ'
C     
 1    NFIT=0
      RMS = 0.
C     
C     Weight softening factor. WFACTOR = 0. uniform weights
C     WFACTOR = infinity Poisson weghts. We imagine a lower
C     limit to the accuracy of the arc line caused for example
C     by under-sampling
C     
      DO I=1,NLINE
         IF(LAMBDA(I).GT.0. .AND. MASK(I).EQ.0) THEN
            NFIT=NFIT+1
            XDATA(NFIT)=  PIXEL(I)/NPIX
            YDATA(NFIT)=  LAMBDA(I)
            YERR(NFIT) = SQRT(1.+WFACTOR/MAX(1.,PEAK(I)))
            RMS = RMS + REAL(YERR(NFIT)**2)
         END IF
      END DO
      IF(NFIT.LT.1) THEN
         WRITE(*,*) CHAR(7),'** TOO FEW ARC LINES FOR POLY-FIT.', NFIT
         GOTO 999
      END IF
C
C     Renormalise
C
      RMS = SQRT(RMS/NFIT)
      DO I=1,NFIT
         YERR(I) = YERR(I)/RMS
      END DO
C
C     Subtract master polynomial from angstrom values
C     
      IF(NGPOLY.NE.0 .AND. NPIX.NE.NGPIX) THEN
         WRITE(*,*) CHAR(7),'** NUMBER OF PIXELS INCOMPATIBLE WITH ',
     &        'NUMBER USED FOR MASTER POLY **'
         GOTO 999
      END IF
      MASTER = NGPOLY.NE.0
      IF( MASTER ) THEN
         DO I=1,NFIT
            Z = REAL(NPIX)*REAL(XDATA(I))
            GCALC = ANGST(Z, NGPIX, NGPOLY, GRANDARC, MXARC)
            YDATA(I) = YDATA(I) - GCALC
         END DO
         NARC = NTWEAK
      END IF
C     
C     Compute poly-fit
C     
      IF(NARC.GT.NFIT) THEN
         WRITE(*,*) 'Setting order of fit equal'//
     &        ' to number of lines = ',NFIT
      END IF
      NARC = MIN( NFIT, NARC )

      CALC = 0.D0
      CALL LEASQ( XDATA, YDATA, YERR, NFIT, NARC, ARC, CALC, WORK, 1)
      LASTPOLY = NARC
C     
C     Add Master arcfit poly to angstrom values and fitted poly 
C
      IF( MASTER ) THEN
         DO I=1,NFIT
            Z = REAL(NPIX)*REAL(XDATA(I))
            GCALC = ANGST(Z, NGPIX, NGPOLY, GRANDARC, MXARC)
            YDATA(I) = YDATA(I) + GCALC
         END DO
         IF( NARC.LT.NGPOLY ) THEN
            DO I=NARC+1, NGPOLY
               ARC(I) = 0.D0
            END DO
         END IF
         DO I=1,NGPOLY
            ARC(I) = ARC(I) + GRANDARC(I)
         END DO
         NARC = MAX(NARC, NGPOLY)

      END IF
C
C     Evaluate the polyfit
C
      RMSALL=0.
      RMSFIT=0.
      SFIT = 0.
      SALL = 0.
      NALL=0
      NFIT=0
      WRITE(*,*) ' '
      WRITE(*,*)  'Line     Pixel      Peak  Observed Calculated',
     *     '      O-C    Weight'
      DO I=1,NLINE
         IF(LAMBDA(I).GT.0.) THEN
            CALC = ANGST(PIXEL(I), NPIX, NARC, ARC, MXARC)
            ERROR = REAL(LAMBDA(I) - CALC)
            NALL = NALL + 1
            IF(MASK(I).NE.0) THEN
               WEIGHT = (RMS/SQRT(1.+WFACTOR/MAX(1.,PEAK(I))))**2
            ELSE
               NFIT = NFIT + 1
               WEIGHT = 1./REAL(YERR(NFIT)**2)
            END IF
            WEG = WEIGHT
            IF( MASK(I).NE.0 ) WEIGHT = 0.
            WRITE(*,'(I5, F10.2, F10.0, 4(F10.3))')
     *           I, PIXEL(I), PEAK(I), LAMBDA(I), CALC, ERROR, WEIGHT
            RMSALL = RMSALL + WEG*ERROR*ERROR
            SALL = SALL + WEG
            IF( MASK(I).EQ.0) THEN
               RMSFIT = RMSFIT + WEG*ERROR*ERROR
               SFIT = SFIT + WEG
            END IF
         END IF
      END DO
C
C     Normalisation for number of degrees of freedom
C     
      AFAC   = SQRT(REAL(NALL)/REAL(MAX(NALL-LASTPOLY,1)))
      FFAC   = SQRT(REAL(NFIT)/REAL(MAX(NFIT-LASTPOLY,1)))
      RMSALL = AFAC*SQRT( RMSALL / SALL )
      RMSFIT = FFAC*SQRT( RMSFIT / SFIT )
      IF( MASTER ) THEN
         WRITE(*,'(1X,A,I4,A,I4,A)')
     *        'Master polynomial has', NGPOLY, 
     *        ' terms.', NTWEAK,' low-order terms were tweaked.'
      ELSE
         WRITE(*,'(1X,A,I4,A)')
     *        'Wavelength scale polynomial has ', NARC, ' terms.'
      END IF
      WRITE(*,'(1X,A,I4,A,G11.4,A)')
     *     '    Lines Fitted=', NFIT, ' RMS=', RMSFIT, ' (Angstroms)'
      WRITE(*,'(1X,A,I4,A,G11.4,A)')
     *     'Lines identified=', NALL, ' RMS=', RMSALL, ' (Angstroms)'
*     
      WRITE(*,*) '     Pixel  Angstrom   Dispersion'
      NPART = 3
      DO I=1,NPART
         PART = (I-1)/FLOAT(NPART-1)
         Z = REAL(NINT( 1.+ REAL(NPIX-1)*REAL(I-1)/REAL(NPART-1)))
         WCALC = ANGST(Z, NPIX, NARC, ARC, MXARC)
         DCALC = DISPA(Z, NPIX, NARC, ARC, MXARC)
         WRITE(*,'(1X,2(F10.3),F13.3)') Z, WCALC, DCALC
      END DO
*     
*     COMMAND NODE --------------------------------------------------
*     

      IF(COMB) RETURN
 10   WRITE(*,*) ' '
      WRITE(*,*) 'Type ''P'' to PLOT the polynomial arcfit'
      WRITE(*,*) '     ''R'' to plot RESIDUALS'
      WRITE(*,*) '     ''C'' to CHANGE parameters and refit'
      WRITE(*,*) '     ''M'' to set the MASTER arcfit'
      IF( MASTER ) THEN
         WRITE(*,*) '     ''X'' to cancel the master arcfit polynomial'
      END IF
      WRITE(*,*) '     <CR> to continue.'
      READ(*,'(A)') REPLY
      CALL UPPER_CASE(REPLY)
*     
      IF( REPLY.EQ.' ') GOTO 30
      IF( REPLY.EQ.'C') GOTO 50
      IF( REPLY.EQ.'P') GOTO 100
      IF( REPLY.EQ.'R') GOTO 100
      IF( REPLY.EQ.'M') GOTO 40
      IF( MASTER ) THEN
         IF( REPLY.EQ.'X') GOTO 45
      END IF
      WRITE(*,*) '** INVALID REPLY.'
      GOTO 10
*     
*     Normal return
*     
 30   RETURN

*     
*     Set master arcfit polynomial
*     
 40   WRITE(*,*) ' '
      WRITE(*,*) 'When the MASTER arcfit polynomial is'//
     &     ' in effect, arcfits adjust'
      WRITE(*,*) 'only the low-order terms of the wavelength scale.'
      WRITE(*,*) ' '
      WRITE(*,*) 'Please type ''CONFIRM'' '//
     &     'if you wish to adopt the current'
      WRITE(*,'(A,I4,A)') ' arcfit with', NARC, 
     &     ' terms as the master arcfit polynomial.'
      READ(*,'(A)') CONFIRM
      CALL UPPER_CASE(CONFIRM)
*     
      IF( CONFIRM.NE.'CONFIRM' ) THEN
         IF( MASTER ) THEN
            WRITE(*,*)  CHAR(7), 
     &           '** PREVIOUS MASTER ARCFIT POLYNOMIAL RETAINED.'
         ELSE
            WRITE(*,*)  CHAR(7), '** Master arcfit '//
     &           'polynomial NOT in effect.'
         END IF
         GOTO 10
      END IF
*     
      NGPOLY = NARC
      NGPIX  = NPIX
      DO I=1,NARC
         GRANDARC(I) = ARC(I)
      END DO
      MASTER = .TRUE.
      GOTO 10
*     
*     Cancel master arcfit polynomial
*     
 45   WRITE(*,*) ' '
      WRITE(*,*) 'Please type ''CONFIRM'' if '//
     &     'you wish to CANCEL the current'
      WRITE(*,'(A,I4,A)')
     *     ' master arcfit polynomial with', NARC, ' terms.'
      READ(*,'(A)') CONFIRM
      CALL UPPER_CASE(CONFIRM)
*     
      IF( CONFIRM.NE.'CONFIRM' ) THEN
         WRITE(*,*) CHAR(7),'** Master arcfit '//
     &        'polynomial NOT cancelled.'
         GOTO 10
      END IF
*     
      NGPIX  = 0
      NGPOLY = 0
      MASTER = .FALSE.
      WRITE(*,*) 'Master arcfit polynomial cancelled !'
      GOTO 10
*     
*     Set parameters for poly-fit ------------------------------------
*     
 50   IF( MASTER ) THEN
*     
         WRITE(*,*) 'The master arcfit polynomial has', 
     &        NGPOLY, ' terms.'
         WRITE(*,*) 'Previous arcfit tweaked', NTWEAK, 
     &        ' low-order terms.'
         WRITE(*,*) 'How many of the low-order '//
     &        'terms do you now wish to tweak ?'
         READ(*,*,ERR=10) NEWTWEAK
         IF( NEWTWEAK.LT.0 ) THEN
            WRITE(*,*) '** INVALID REPLY.'
            GOTO 10
         ELSE IF( NEWTWEAK.GT.NFIT .OR. NEWTWEAK.GT.MAXPOLY ) THEN
            WRITE(*,*) CHAR(7),'** TOO MANY POLY TERMS.  NFIT=', NFIT,
     *           ' MAXPOLY=',MAXPOLY
            GOTO 10
         END IF
         NTWEAK = NEWTWEAK
*     
      ELSE
*     
         WRITE(*,*) 'How many polynomial terms for arcfit ?'
         READ(*,*,ERR=10) NEWPOLY
         IF(NEWPOLY.LT.2) THEN
            WRITE(*,*) CHAR(7),'** YOU NEED AT LEAST TWO TERMS.'
            GOTO 10
         ELSE IF(NEWPOLY.GT.NFIT .OR. NEWPOLY.GT.MAXPOLY) THEN
            WRITE(*,*) CHAR(7),'** TOO MANY POLY TERMS.  NFIT=', NFIT,
     *           ' MAXPOLY=',MAXPOLY
            GOTO 10
         END IF
         NARC = NEWPOLY
*     
      END IF
*     
      WRITE(*,*) 'Enter weight factor. Weight = PEAK/(A+PEAK)'
      WRITE(*,*) 'Enter A. A=0 Uniform, large A for Poisson'
      WRITE(*,*) 'weights'
      READ(*,*,ERR=10) WFACTOR
      WFACTOR = MAX(0.,MIN(1.E8,WFACTOR))
*     
      GOTO 1
C     
C     PLOT ARCFIT RESIDUALS --------------------------------------
C     
 100  WRITE(*,*) ' '
      WRITE(*,*) 'PLOTTING ARCFIT RESIDUALS'
      WRITE(*,*) ' '
      WRITE(*,*) 'What peak height range to plot ?'
      WRITE(*,'(A,G11.4,A,G11.4,A,$)') '[',PLOW,',',PHIGH,']: '
      READ(*,'(A)',ERR=100) STRING
      IF(STRING.EQ.' ') THEN
         PLOW1  = PLOW
         PHIGH1 = PHIGH
      ELSE
         READ(STRING,*,ERR=100) PLOW1, PHIGH1
      END IF
      IF(PLOW1.GE.PHIGH1) GOTO 100
      PLOW  = PLOW1
      PHIGH = PHIGH1
*     
*     Compute best-fit linear arcfit coefficients
*     
      CALC = 0.D0
      DO I=1,NFIT               ! Set uniform weights
         YERR(I) = 1.
      END DO
      CALL LEASQ( XDATA, YDATA, YERR, NFIT, 2, 
     &     ARCLINEAR, CALC, WORK, 1)
*     
*     Set X range for the plot
*     
      WCALC = ANGST(1., NPIX, NARC, ARC, MXARC)
      WAV1 = REAL(WCALC)
*     
      WCALC = ANGST(REAL(NPIX), NPIX, NARC, ARC, MXARC)
      WAV2 = REAL(WCALC)
*     
*     ensure that wavelengths ascend to the right
*     
      XMN = MIN( WAV1, WAV2 )
      XMX = MAX( WAV1, WAV2 )
*     
*     Set Y range for the plot
*     
      NPLOT = 0
      DO I=1,NLINE
         IF( LAMBDA(I).GT.0.D0 .AND. PEAK(I).GE.PLOW .AND. 
     &        PEAK(I).LE.PHIGH) THEN
            NPLOT = NPLOT + 1
            IF( REPLY.EQ.'P') THEN
               CALC = ANGST(PIXEL(I), NPIX, 2, ARCLINEAR, MXARC)
            ELSE IF( REPLY.EQ.'R' ) THEN
               CALC = ANGST(PIXEL(I), NPIX, NARC, ARC, MXARC)
            END IF
            PLOT2(NPLOT) = REAL(LAMBDA(I) - CALC)
         END IF
      END DO
      IF(NPLOT.LE.0) THEN
         WRITE(*,*) 'No points to plot'
         GOTO 10
      END IF
      CALL PGLIMIT(NPLOT, PLOT2, YMN, YMX, 0.1 )
      CALL SET_LIM(XMN, XMX, YMN, YMX, IFAIL)
      IF(IFAIL.NE.0) GOTO 10
C     
C     Open the plot
C     
      CALL DEV_OPEN(.FALSE.,NEW,IFAIL)
      IF(IFAIL.NE.0) GOTO 10
      CALL PGIDEN
C     
C     Frame the plot
C     
      CALL PGSCI(5)      
      CALL PGENV( XMN, XMX, YMN, YMX, 0, 2)
      CALL PGSCI(7)      
      IF( REPLY.EQ.'P') THEN
         CALL PGLAB('Computed Wavelength (\\A)', 
     *        '\\gl\\dexact\\u - \\gl\\dcomputed\\u (\\A)',
     *        'Polynomial Wavelength Calibration (linear term removed)')
      ELSE IF( REPLY.EQ.'R' ) THEN
         CALL PGLAB('Computed Wavelength (\\A)', 
     *        '\\gl\\dexact\\u - \\gl\\dcomputed\\u (\\A)',
     *        'Polynomial Wavelength Calibration Residuals' )
      END IF
*
*     Plot centroids of individual arc lines
*     
      DO I=1,NLINE
         IF( LAMBDA(I).GT.0.D0.AND. PEAK(I).GE.PLOW .AND. 
     &        PEAK(I).LE.PHIGH) THEN
            IF( REPLY.EQ.'P') THEN
               CALC = ANGST(PIXEL(I), NPIX, 2, ARCLINEAR, MXARC)
            ELSE IF( REPLY.EQ.'R' ) THEN
               CALC = ANGST(PIXEL(I), NPIX, NARC, ARC, MXARC)
            END IF
            XPLOT = REAL(LAMBDA(I))
            YPLOT = REAL(LAMBDA(I) - CALC)
            IF(MASK(I).EQ.0) THEN
               CALL PGSCI(1)      
               CALL PGPOINT(1, XPLOT, YPLOT, 15)
            ELSE
               CALL PGSCI(3)    
               CALL PGPOINT(1, XPLOT, YPLOT, 20)
            END IF
         END IF
      END DO
*     
*     Plot polynomial fit
*     
      IF( REPLY.EQ.'P' ) THEN
         NPLOT = MIN(MAXPLOT, NARC*10)
         DO I = 1,NPLOT
            PART = (I-1.)/(NPLOT-1.)
            Z = 1. - PART + NPIX*PART
            WCALC = ANGST(Z, NPIX, NARC, ARC, MXARC)
            PLOT1(I) = REAL(WCALC)
            WCALC = ANGST(Z, NPIX, 2, ARCLINEAR, MXARC)
            PLOT2(I) = REAL(PLOT1(I) - WCALC)
         END DO
         CALL PGSCI(2)      
         CALL PGLINE( NPLOT, PLOT1, PLOT2)
      END IF
C     
C     Close the plot
C      
      CALL DEV_CLOSE(0, IFAIL)
      GOTO 10
*
999   NARC = 0            
      RETURN  
      END

      SUBROUTINE ARC_TWEAK(ASPEC, AERR, NPIX, PIXEL, PISIG, 
     &     PEAK, LAMBDA, SOURCE, MASK, NLINE, MAXLIN, COMB, 
     &     CHANGE, KEY, MXWORK, DMAX, IFAIL)
*     
*     
*     Tweaks pixel centroids of arc lines in the table
*     
*     Input:
*     ASPEC = ARC SPECTRUM
*     NPIX      = NUMBER OF PIXELS IN ARC SPECTRUM
*     
*     JULY 1984 BY KEITH HORNE AT IOA.
*     Modified for MOLLY by TRM
*     
      IMPLICIT NONE
      INTEGER NPIX, IFAIL, N
      REAL ASPEC(NPIX), AERR(NPIX)
      INTEGER MAXLIN, NLINE, MXWORK
      REAL PIXEL(MAXLIN), PEAK(MAXLIN), PISIG(MAXLIN)
      REAL CHANGE(MXWORK)
      INTEGER KEY(MXWORK)
      DOUBLE PRECISION LAMBDA(MAXLIN)
      CHARACTER*(*) SOURCE(MAXLIN)
      INTEGER MASK(MAXLIN)
      LOGICAL COMB
      REAL FWHM, SHIFT, RBIN, DIFF, RBINSIG, DMAX, SMEAN
      REAL TDIFF
      INTEGER I1, I2, I, J, K
      INTEGER LINE1, LINE2, IPEAK, IFLIP
      CHARACTER*100 STRING
      DATA SHIFT, FWHM / 0., 2. /
*
*     Introduction
*     
      IFAIL = 0
      WRITE(*,*) ' '
      WRITE(*,*) 'ARCTWEAK -- Pixel centroids '//
     &     'of arc lines are refined by'
      WRITE(*,*) 'maximizing the cross-correlation '//
     &     'between the arc spectrum'
      WRITE(*,*) 'and a gaussian template profile.'
*     
*     Test inputs
*     
      IF( NPIX.LE.0 ) THEN
         WRITE(*,*) '** NO ARC SPECTRUM IN ''ARCTWEAK''.', NPIX
         IFAIL = 2
         GOTO 999
      END IF
      IF( NLINE.LE.0 ) THEN
         WRITE(*,*) '** NO ARC LINES IN ''ARCTWEAK''.', NLINE
         IFAIL = 1
         GOTO 999
      END IF
*     
*     Set default parameters
*     
      LINE1 = 1
      LINE2 = NLINE
      SHIFT = 0.
*     
*     Edit tweak parameters ---------------------------------------------
*     
      IF(.NOT.COMB) THEN
         WRITE(*,'(A,I4,A,I4,A,$)') 'Enter line range [',
     &        LINE1,',',LINE2,'] '
         READ(*,'(A)') STRING
         IF(STRING.NE.' ') THEN
            READ(STRING,*,ERR=999) I1, I2
            IF(I1.LT.1 .OR. I1.GT.NLINE .OR.
     &           I1.GT.I2) THEN
               WRITE(*,*) 'Invalid range'
               GOTO 999
            END IF
            LINE1 = I1
            LINE2 = MIN(I2, NLINE)
         END IF
         WRITE(*,'(A,G10.4,A,$)') 'Gaussian FWHM [',FWHM,'] '
         READ(*,'(A)') STRING
         IF(STRING.NE.' ') THEN
            READ(STRING,*,ERR=999) FWHM
            IF(FWHM.LT.2.) THEN
               WRITE(*,*) 'Minimum value = 2'
               GOTO 999
            END IF
         END IF
         WRITE(*,'(A,G10.4,A,$)') 
     &        'Shift to be applied before tweak [',SHIFT,'] '
         READ(*,'(A)') STRING
         IF(STRING.NE.' ') READ(STRING,*,ERR=999) SHIFT
      END IF
*     
*     Tweak line centroids -----------------------------------------
*     
      WRITE(*,*) ' '
      WRITE(*,*) 'Tweaking line positions. SHIFT=', SHIFT, 
     &     ' FWHM=', FWHM
      WRITE(*,*) ' Line  Old Pixel  New Pixel   Shift'
      SMEAN = 0.
      DO K = 1, 4
         N = 0
         DO J=LINE1, LINE2
            IPEAK=NINT(PIXEL(J)+SHIFT)
*     
*     Line shifted out of range
*     
            IF(IPEAK.LE.2 .OR. IPEAK.GE.NPIX-2) THEN
               WRITE(*,'(I5, F11.2, A)')
     &              J, PIXEL(J), 
     &              ' ** SHIFTED OUT OF RANGE, CENTROID NOT TWEAKED.'
               IFAIL = 1
*     
*     Refine the line position using gaussian convolution algorithm
*     
            ELSE
               IFLIP=1
               CALL MEDIAN( ASPEC, NPIX, IPEAK, FWHM, RBIN, RBINSIG, 
     &              IFLIP)
               N = MIN(N + 1, MXWORK)
               CHANGE(N) = RBIN-PIXEL(J)
               IF(K.EQ.4) THEN
                  SMEAN = SMEAN + CHANGE(N)
                  WRITE(*,'(I5, 2(F11.2), F10.2 )')
     &                 J, PIXEL(J), RBIN, RBIN-PIXEL(J)
                  PIXEL(J)= RBIN
                  PEAK(J) = ASPEC( NINT(RBIN) )
               END IF
            END IF
         END DO
         IF(K.LT.4) THEN
            IF(N.LT.1) THEN
               WRITE(*,*) 'No lines found'
               IFAIL = 1
               GOTO 999
            END IF
*     
*     Compute median shift to supply the next cycle
*     
            CALL HEAPSORT(N,CHANGE,KEY)
            IF(MOD(N,2).EQ.1) THEN
               SHIFT = CHANGE(KEY((N+1)/2))
            ELSE
               SHIFT = (CHANGE(KEY(N/2))+CHANGE(KEY(N/2+1)))/2.
            END IF
            WRITE(*,*) 'Shift revised to ',SHIFT,' on pass ',K
         END IF
      END DO

      IF(N.GT.1) THEN
         SMEAN = SMEAN/REAL(N)
         DIFF = - 1.
         DO I = 1, N
            TDIFF = ABS(CHANGE(I)-SMEAN)
            DIFF = MAX(DIFF, TDIFF)
            IF(TDIFF.GT.DMAX) 
     &           WRITE(*,*) 'Line ',I,', pixel ',PIXEL(I),
     &           ' has shifted by ',TDIFF,' from the mean.'
         END DO
         IF(DIFF.GT.DMAX) THEN
            WRITE(*,*) 'At least one line has shifted by more than'
            WRITE(*,*) 'than the defined minimum of ',DMAX
            IFAIL = 1
         END IF
      END IF
*     
 999  RETURN
      END
