package Molly;

=head1 NAME

    Molly - PDL  module for reading and writing molly format spectra.

=head1 SYNOPSIS

  use Molly;

  open(MOLLY, "test.mol);
  if(cmol(*MOLLY)){
    $spec1 = rmol(*MOLLY);
  }
  if(cmol(*MOLLY)){
    $spec2 = rmol(*MOLLY);
  }
  close(MOLLY);

  $flux = $spec->getflux;
  $wave = $spec->gettwav;
 
  line $wave, $flux;


  rmol  e.g. $spec = rmol(*FILEHANDLE);   -- reads a spectrum.
  wmol  e.g. $spec->wmol(*FILEHANDLE);    -- writes a spectrum.
  smol  e.g. smol(*FILEHANDLE);           -- skips a spectrum.
  hmol  e.g. $hdr = hmol(*FILEHANDLE);    -- reads headers.
  cmol  e.g. cmol(*FILEHANDLE);           -- checks existence of data before
                                             applying rmol, smol or hmol.
  gettwav e.g. $wave = $spec->gettwav;    -- gets wavelengths.
  gethwav e.g. $wave = $spec->gethwav;    -- gets wavelengths, correcting to
                                             heliocentric scale.
  getflux  e.g. $flux = getflux $spec;    -- gets fluxes.
  getferr  e.g. $ferr = getferr $spec;    -- gets flux uncertainties.
  getcnts  e.g. $cnts = getcnts $spec;    -- gets counts.
  getcerr  e.g. $cerr = getcerr $spec;    -- gets errors on counts.
  achar e.g. $spec->achar('Object','M31') -- adds a character header item.
  areal e.g. $spec->areal('Time','100')   -- adds a real header item.

=head1 DESCRIPTION

This module allow one to read molly spectra rapidly into PDL.

=cut

require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(rmol wmol smol hmol cmol gettwav gethwav getflux 
	     getferr getcnts getcerr achar areal);
use PDL;
use PDL::Types;

=head2 rmol(*FILEHANDLE) 

  Reads a molly spectrum. Requires a file to be open and must be used after a call to 'cmol'
  e.g.

    open(MOLLY, 'spectrum.mol')
    cmol(*MOLLY);
    $spec1 = rmol(*MOLLY); 
    cmol(*MOLLY);
    $spec2 = rmol(*MOLLY); 
    cmol(*MOLLY);
    $spec3 = rmol(*MOLLY);
   close(MOLLY);

  $spec1 etc are then npix by 3 dimesional piddles representing the 
  first 3 molly spectra. They contain the standard counts, errors and 
  flux arrays of molly. 'cmol' is needed to skip over first 4 bytes
  (and can be tested for success if number of spectra not known). This
  is to work around a problem with testing pdls because it was otherwise
  difficult to test for the end of the file.
 
  NB: At the end of each read, one is left at the start of the next spectrum.
 
  Much more information is stored in headers as a hash which can be 
  obtained via:
 
  $hdr = $spec->gethdr;
 
  Then 
 
   %{$hdr} are the headers (e.g. $$hdr{'Object'} is the object name)
 
   $$hdr{'First record'} is a reference to an array containing the
           first record i.e. fcode, units, npix, narc, nchar, ndoub, 
                             nintr, nreal
 
   $$hdr{'Arc'} is a reference to the arc coeffcients array
 
   $$hdr{'Character names'} is a reference to an array containing the
           names of character header items.
 
   $$hdr{'Double names'} is a reference to an array containing the
           names of double header items.
 
   $$hdr{'Integer names'} is a reference to an array containing the
           names of integer header items.
 
   $$hdr{'Real names'} is a reference to an array containing the
           names of real header items.
 
  The latter 4 are required in order to be able to write molly spectra.

=cut

sub rmol {PDL->rmol(@_)}

sub PDL::rmol{
    barf 'Usage: $a = rmol(FILEHANDLE)' if $#_ != 1;
    my ($class, $filehandle) = @_;

    # Setup header hash

    my $header = {};
    my ($fcode,$units,$npix,$narc,$nchar,$ndoub,$nintr,$nreal,$dref);
    my ($buf,$template,$nhead,$nbytes,$ncoeff,@header_names,@header_values);
    my ($offset);

    # Read and unpack the first record

    read($filehandle, $buf, 48) == 48
	or barf "Couldn't read first record";
    $$header{'First record'} = [unpack("ia16i6x4", $buf)];
    $fcode  = $$header{'First record'}[0];
    $fcode == 3 or 
	barf "fcode = $fcode. Can only read fcode=3 molly data\n";	
    $npix   = $$header{'First record'}[2];
    $narc   = $$header{'First record'}[3];
    $nchar  = $$header{'First record'}[4];
    $ndoub  = $$header{'First record'}[5];
    $nintr  = $$header{'First record'}[6];
    $nreal  = $$header{'First record'}[7];

    # Read second record of header item names

    $nhead  = $nchar + $ndoub + $nintr + $nreal;
    $nbytes = 16*$nhead+8;
    read($filehandle, $buf, $nbytes) == $nbytes
	or barf "Couldn't read second record";
    $template = "x4";
    for (1..$nhead) {
	$template .= "A16";
    }
    @header_names = unpack($template, $buf);

    # Remove trailing blanks.

    foreach(@header_names){
	s/ *$//;
    }

    # Read and interpret header item values
    
    $nbytes = 32*$nchar + 8*$ndoub + 4*$nintr + 4*$nreal + 8;
    read($filehandle, $buf, $nbytes) == $nbytes
	or barf "Couldn't read third record";
    $template = "x4";
    for (1..$nchar) {
	$template .= "A32";
    }
    $template .= "d$ndoub i$nintr f$nreal";
    @header_values = unpack($template, $buf);

    # Load the hash
	
    for (0..$#header_names) {
	$$header{$header_names[$_]} = $header_values[$_];
    }

    # Store the types

    $offset = 0;
    for ($i=0;$i<$nchar;$i++){
	$$header{'Character names'}[$i] = $header_names[$offset+$i];
    }
    $offset += $nchar;
    for ($i=0;$i<$ndoub;$i++){
	$$header{'Double names'}[$i] = $header_names[$offset+$i];
    }
    $offset += $ndoub;
    for ($i=0;$i<$nintr;$i++){
	$$header{'Integer names'}[$i] = $header_names[$offset+$i];
    }
    $offset += $nintr;
    for ($i=0;$i<$nreal;$i++){
	$$header{'Real names'}[$i] = $header_names[$offset+$i];
    }

    # Read arc coefficients, store as an array reference in the hash.

    $nbytes = $narc < 0 ? -8*$narc + 8: 8*$narc + 8;
    $ncoeff = $narc < 0 ? -$narc : $narc;
    read($filehandle, $buf, $nbytes) == $nbytes
	    or barf "Couldn't read arc coefficients record";
    $template = "x4d$ncoeff";
    $$header{'Arc'} = [unpack($template, $buf)];

    # Now read counts, errors, fluxes into the pdl

    my $pdl  = $class->new;
    $pdl->set_datatype($PDL_F);
    $pdl->setdims([$npix,3]);
    $dref = $pdl->get_dataref;

    read($filehandle, $buf, 4) == 4
	or barf "Couldn't read first 4 bytes of 5th record";
    $nbytes =  12*$npix;
    read($filehandle, $ {$dref}, $nbytes) == $nbytes
	or barf "Couldn't read data arrays";
    read($filehandle, $buf, 4) == 4
	or barf "Couldn't read last 4 bytes of 5th record";
    $pdl->upd_data();

    # Put header hash into pdl

    $pdl->sethdr($header);

    return $pdl;
}

=head2 wmol($spec, *FILEHANDLE) 

  Writes a molly spectrum to a file opened for output on filehandle.

  e.g.

       open(OUTPUT,">out.mol");
       wmol($spec,*OUTPUT);
       close(OUTPUT);

  or
 
        open(OUTPUT,">out.mol");
        $spec->wmol(*OUTPUT);
        close(OUTPUT);

=cut

*wmol = \&PDL::wmol;

sub PDL::wmol{
    barf 'Usage: wmol($spec,FILEHANDLE)' if $#_ != 1;
    my($spec, $filehandle) = @_;

    my ($buf, $dref, $nbytes, $hdr, $npix, $narc, $nchar, $ndoub);
    my ($nintr, $nreal, $ncoeff);

    $nbytes = 44;
    
    # Pack and write first record

    $hdr = $spec->gethdr;
    $buf = pack "i2a16i7", $nbytes, @{$$hdr{'First record'}}, $nbytes;
    print $filehandle $buf or
	barf "Failed while writing first record.\n";
    
    $npix   = $$hdr{'First record'}[2];
    $narc   = $$hdr{'First record'}[3];	
    $nchar  = $$hdr{'First record'}[4];	 
    $ndoub  = $$hdr{'First record'}[5];	 
    $nintr  = $$hdr{'First record'}[6];	 
    $nreal  = $$hdr{'First record'}[7];
    
    # Write second record of header names

    $nbytes = 16*($nchar+$ndoub+$nintr+$nreal);
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing start of second record.\n";
    if($nchar>0){
	$buf = pack "A16" x $nchar, @{$$hdr{'Character names'}};
	print $filehandle $buf or
	    barf "Failed while writing character header item names.\n";
    }
    if($ndoub>0){
	$buf = pack "A16" x $ndoub, @{$$hdr{'Double names'}};
	print $filehandle $buf or
	    barf "Failed while writing double header item names.\n";
    }
    if($nintr>0){
	$buf = pack "A16" x $nintr, @{$$hdr{'Integer names'}};
	print $filehandle $buf or
	    barf "Failed while writing integer header item names.\n";
    }
    if($nreal>0){
	$buf = pack "A16" x $nreal, @{$$hdr{'Real names'}};
	print $filehandle $buf or
	    barf "Failed while writing real header item names.\n";
    }
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing end of second record.\n";

    # Write third record of header item values
    
    $nbytes = 32*$nchar + 8*$ndoub + 4*$nintr + 4*$nreal;
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing start of third record.\n";
    if($nchar>0){

	# Next rather offputting line writes header item values
	# as a slice defined by the values of the 'Character names'
	# array of the hash $hdr

	$buf = pack "A32" x $nchar, @{$hdr}{@{$$hdr{'Character names'}}};
	print $filehandle $buf or
	    barf "Failed while writing character header item values.\n";
    }
    if($ndoub>0){
	$buf = pack "d" x $ndoub, @{$hdr}{@{$$hdr{'Double names'}}};
	print $filehandle $buf or
	    barf "Failed while writing double header item values.\n";
    }
    if($nintr>0){
	$buf = pack "i" x $nintr, @{$hdr}{@{$$hdr{'Integer names'}}};
	print $filehandle $buf or
	    barf "Failed while writing integer header item values.\n";
    }
    if($nreal>0){
	$buf = pack "f" x $nreal, @{$hdr}{@{$$hdr{'Real names'}}};
	print $filehandle $buf or
	    barf "Failed while writing real header item values.\n";
    }
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing end of third record.\n";

    # Write arc coefficients

    $nbytes = $narc < 0 ? -8*$narc: 8*$narc;
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing start of fourth record.\n";
    $ncoeff = $narc < 0 ? -$narc : $narc;
    if($ncoeff>0){
	$buf = pack "d" x $ncoeff, @{$$hdr{'Arc'}};
	print $filehandle $buf or
	    barf "Failed while writing arc coefficients.\n";
    }
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing end of fourth record.\n";

    # Write data arrays

    $dref = $spec->get_dataref;

    $nbytes = 12*$npix;
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing start of fifth record.\n";
    print $filehandle $ {$dref} or
	barf "Failed while writing data arrays.\n";
    $buf = pack "i", $nbytes;
    print $filehandle $buf or
	barf "Failed while writing end of fifth record.\n";

    return 1;
}

=head2 smol(*FILEHANDLE) 

    Skips over a molly spectrum more efficiently than using rmol to read it.
    
    Like rmol you need first to use cmol to skip first 4 bytes if possible.
    smol reads first record to work out how many bytes to skip and then skips
    them.

=cut

sub smol {PDL->smol(@_)}

sub PDL::smol{
    barf 'Usage: smol(FILEHANDLE)' if $#_ != 1;
    my ($class,$filehandle) = @_;

    my ($fcode,$units,$npix,$narc,$nchar,$ndoub,$nintr,$nreal);
    my ($buf,$nbytes);

    # Read and unpack the first record

    read($filehandle, $buf, 48) == 48
	or barf "Couldn't read first record";
    ($fcode,$units,$npix,$narc,$nchar,$ndoub,$nintr,$nreal) =
	unpack("ia16i6x4", $buf);

    # Compute number of bytes to skip:

    $nbytes  =  16*($nchar + $ndoub + $nintr + $nreal) + 8;
    $nbytes +=  32*$nchar + 8*$ndoub + 4*$nintr + 4*$nreal + 8;
    $nbytes +=  $narc < 0 ? -8*$narc + 8: 8*$narc + 8;
    $nbytes +=  12*$npix + 8;

    read($filehandle, $buf, $nbytes) == $nbytes 
	or barf "Failed to skip over molly spectrum correctly.";
}

=head2 hmol(*FILEHANDLE)	

    Reads molly headers, then skips to end of spectrum. Like 'rmol'
    and 'smol', must be used after a call to 'cmol'. Slightly more
    efficient than using 'rmol' to read everything.
    
    e.g.
    
    $hdr = hmol(*MOLLY);
    print "Object name = $$hdr{'Object'}\n";
			       
=cut

sub hmol {PDL->hmol(@_)}

sub PDL::hmol{
    barf 'Usage: $a = hmol(FILEHANDLE)' if $#_ != 1;
    my ($class, $filehandle) = @_;

    # Setup header hash

    my $header = {};
    my ($fcode,$units,$npix,$narc,$nchar,$ndoub,$nintr,$nreal);
    my ($buf,$template,$nhead,$nbytes,$ncoeff,@header_names,@header_values);

    # Read and unpack the first record

    read($filehandle, $buf, 48) == 48
	or barf "Couldn't read first record";
    $$header{'First record'} = [unpack("ia16i6x4", $buf)];
    $fcode  = $$header{'First record'}[0];
    $fcode == 3 or 
	barf "fcode = $fcode. Can only read fcode=3 molly data\n";	
    $npix   = $$header{'First record'}[2];
    $narc   = $$header{'First record'}[3];
    $nchar  = $$header{'First record'}[4];
    $ndoub  = $$header{'First record'}[5];
    $nintr  = $$header{'First record'}[6];
    $nreal  = $$header{'First record'}[7];

    # Read second record of header item names

    $nhead  = $nchar + $ndoub + $nintr + $nreal;
    $nbytes = 16*$nhead+8;
    read($filehandle, $buf, $nbytes) == $nbytes
	or barf "Couldn't read second record";
    $template = "x4";
    for (1..$nhead) {
	$template .= "A16";
    }
    @header_names = unpack($template, $buf);

    # Remove trailing blanks.

    foreach(@header_names){
	s/ *$//;
    }

    # Read and interpret header item values
    
    $nbytes = 32*$nchar + 8*$ndoub + 4*$nintr + 4*$nreal + 8;
    read($filehandle, $buf, $nbytes) == $nbytes
	or barf "Couldn't read third record";
    $template = "x4";
    for (1..$nchar) {
	$template .= "A32";
    }
    $template .= "d$ndoub i$nintr f$nreal";
    @header_values = unpack($template, $buf);

    # Load the hash
	
    for (0..$#header_names) {
	$$header{$header_names[$_]} = $header_values[$_];
    }

    # Read arc coefficients, store as an array reference in the hash.

    $nbytes = $narc < 0 ? -8*$narc + 8: 8*$narc + 8;
    $ncoeff = $narc < 0 ? -$narc : $narc;
    read($filehandle, $buf, $nbytes) == $nbytes
	    or barf "Couldn't read arc coefficients record";
    $template = "x4d$ncoeff";
    $$header{'Arc'} = [unpack($template, $buf)];

    # Now read counts, errors, fluxes into the pdl

    $nbytes =  12*$npix+8;
    read($filehandle, $buf, $nbytes) == $nbytes
	or barf "Failed to skip data";

    return $header;
}

=head2 cmol(*FILEHANDLE) 

  Skips over first 4 bytes of a molly spectrum (which contains ignorable data) 
  and returns 0 or 1 depending on success.

  cmol should be used before every call to 'rmol', 'smol' and 'hmol' in order
  to be get the correct position correctly in the file. It is also needed when
  a file has an unknown number of spectra:

  open(INPUT, 'file')
  while(cmol(*INPUT)){
     $spec = rmol(*INPUT);
 
   .
   .
   
   Process spectra
 
   .
   .
  }
  close(INPUT);

=cut

sub cmol {PDL->cmol(@_)}

sub PDL::cmol{
    barf 'Usage: cmol(FILEHANDLE)' if $#_ != 1;
    my ($class,$filehandle) = @_;
    my ($buf,$nbytes);

    # Read 4 bytes if possible

    if(read($filehandle, $buf, 4) == 4){
	my $nbytes = unpack("i", $buf);
	return 1;
    }else{			
	return 0;
    }
}

=head2 gettwav($spec) 

  returns a pdl of wavelengths generated from the arc coefficients. 
  No correction to a heliocentric scale is applied (marked by the 
  second 't') although of course the arc coefficients may already 
  represent a heliocentric scale.

  e.g. 

  $wave = $spec->gettwav;
  $wave = gettwav $spec;

=cut
			 
*gettwav = \&PDL::gettwav;

sub PDL::gettwav{
    barf 'Usage: $wave = gettwav($spec)' if $#_ > 1;
    my $spec  = shift;
    my ($wave,$hdr,$npix,$narc,$i,$tpdl1,$tpdl2,$ncoeff);
    $hdr  = $spec->gethdr;
    $npix = $$hdr{'First record'}[2];
    $narc = $$hdr{'First record'}[3];
    if($narc == 0){
	$wave = xvals($spec->slice(':,(0)')) + 1;
    }else{
	$ncoeff = $narc > 0 ? $narc : -$narc;
	$wave   = zeroes($spec->slice(':,(0)'));
	$tpdl1  = ones $wave;
	$tpdl2  = (xvals($wave) + 1)/$npix;
	for ($i=0;$i<$ncoeff;$i++){
	    $wave  += $$hdr{'Arc'}[$i]*$tpdl1;
	    $tpdl1 *= $tpdl2;
	}
	if($narc < 0){
	    $wave = exp($wave);
	}
    }
    return $wave;
}


=head2 gethwav($spec) 
  
  Returns a pdl of wavelengths. A correction is made for heliocentric offsets 
  if the header item 'Vearth' exists. Thus the h denotes heliocentric.

=cut

*gethwav = \&PDL::gethwav;

sub PDL::gethwav{
    barf 'Usage: $wave = gethwav($spec)' if $#_ > 1;
    my $spec  = shift;
    my ($wave,$hdr,$npix,$narc,$i,$tpdl1,$tpdl2,$ncoeff,$cfac);
    $hdr  = $spec->gethdr;
    $npix = $$hdr{'First record'}[2];
    $narc = $$hdr{'First record'}[3];
    if($narc == 0){
	$wave = xvals($spec->slice(':,(0)')) + 1;
    }else{
	$ncoeff = $narc > 0 ? $narc : -$narc;
	$wave   = zeroes($spec->slice(':,(0)'));
	$tpdl1  = ones $wave;
	$tpdl2  = (xvals($wave) + 1)/$npix;
	for ($i=0;$i<$ncoeff;$i++){
	    $wave  += $$hdr{'Arc'}[$i]*$tpdl1;
	    $tpdl1 *= $tpdl2;
	}
	if($narc < 0){
	    $wave = exp($wave);
	}
	if(defined $$hdr{'Vearth'}){
	    $cfac   = 1.-$$hdr{'Vearth'}/2.997925e5;
	    $wave *= $cfac;
	}
    }
    return $wave;
}

=head2 getflux($spec) 
 
  returns a pdl of fluxes. The third row of the basic pdl
  produced by rmol almost but not quite represents the
  fluxes. The "not quite" comes about because of zero count pixels.
  Use getflux to return proper fluxes.

  e.g. 
 
  $flux = $spec->getflux;
  $flux = getflux $spec;

=cut

*getflux = \&PDL::getflux;

sub PDL::getflux{
    barf 'Usage: $flux = getflux($spec)' if $#_ > 1;
    my $spec  = shift;
    my($flux,$tpdl);
    
    # Make copy of third row which contains fluxes

    $flux = $spec->slice(':,(2)')->copy;

    # Account for any zero count (first row) cases when the
    # fluxes must be set to zero explicitly.

    $tpdl = $flux->where($spec->slice(':,(0)') == 0);
    $tpdl .= 0;
    return $flux;
}

=head2 getferr($spec) 

  returns a pdl of flux uncertainties. 

  e.g. 
 
  $ferr = $spec->getferr;
  $ferr = getferr $spec;

=cut

*getferr = \&PDL::getferr;

sub PDL::getferr{
    barf 'Usage: $ferr = getferr($spec)' if $#_ > 1;
    my $spec  = shift;
    my($ferr,$tpdl,$ind,$ind_c); 
    
    ($ind,$ind_c) = which_both($spec->slice(':,(0)') == 0);

    # Generate error pdl

    $ferr = zeroes $spec->slice(':,(0)');

    $tpdl = $ferr->index($ind);
    $tpdl .= $spec->slice(':,(2)')->index($ind)*
	$spec->slice(':,(1)')->index($ind);

    $tpdl = $ferr->index($ind_c);
    $tpdl .= $spec->slice(':,(2)')->index($ind_c)*
	$spec->slice(':,(1)')->index($ind_c)/
	    $spec->slice(':,(0)')->index($ind_c);

    return $ferr;
}

=head2 getcnts($spec) 

  returns a pdl of counts. This is a simple case of returning
  the first row of the pdl.

  e.g. 
 
  $cnts = $spec->getcnts;
 
=cut

*getcnts = \&PDL::getcnts;

sub PDL::getcnts{
    barf 'Usage: $flux = getcnts($spec)' if $#_ > 1;
    my $spec  = shift;
    my($cnts);
    
    $cnts = $spec->slice(':,(0)')->copy;

    return $cnts;
}

=head2 getcerr($spec) 

  returns a pdl of errors on counts. This is a simple case 
  of returning the second row of the pdl.

  e.g. 
 
  $cerr = $spec->getcerr;

=cut

*getcerr = \&PDL::getcerr;

sub PDL::getcerr{
    barf 'Usage: $cerr = getcerr($spec)' if $#_ > 1;
    my $spec  = shift;
    my($cerr);
    
    $cerr = $spec->slice(':,(1)')->copy;

    return $cerr;
}

=head2 achar($spec,$name,$item) 
 
  adds a character header item to a molly piddle.

  The item has name $name and value $item.

=cut

*achar = \&PDL::achar;

sub PDL::achar{
    barf 'Usage: achar($spec,$name,$item)' if $#_ != 2;
    my ($spec,$name,$item)  = @_;
    my($hdr);
    $hdr = $spec->gethdr;
    if(defined $$hdr{$name}){
	$$hdr{$name} = $item;
    }else{
	$$hdr{$name} = $item;
	$$hdr{'Character names'}[$$hdr{'First record'}[4]] = $name;
	$$hdr{'First record'}[4]++;
    }
    $spec->sethdr($hdr);
    return 1;
}

=head2 areal($spec,$name,$item) 

  adds a real header item to a molly piddle.

  The item has name $name and value $item.

=cut

*areal = \&PDL::areal;

sub PDL::areal{
    barf 'Usage: areal($spec,$name,$item)' if $#_ != 2;
    my ($spec,$name,$item)  = @_;
    my($hdr);
    $hdr = $spec->gethdr;
    if(defined $$hdr{$name}){
	$$hdr{$name} = $item;
    }else{
	$$hdr{$name} = $item;
	$$hdr{'Real names'}[$$hdr{'First record'}[7]] = $name;
	$$hdr{'First record'}[7]++;
    }
    $spec->sethdr($hdr);
    return 1;
}

1;


