        SUBROUTINE FITGAUSS( N, X, Y, SIGY, A, B, WID, X0,
     *  SIGA, SIGB, SIGWID, SIGX0 )
C
C  USES LSQENP TO FIT A GAUSSIAN + BACKGROUND
C
*  INPUT:
*       N       = Number of data values
*       X(N)    = (REAL*4) Array of values of independent variable
*       Y(N)    = (REAL*4) Array of corresponding values of dependent variable
*       SIGY(N) = (REAL*4) Array of 1-sigma uncertainties in dependent variable
*       A,B,WID,XO      = DEFAULT PARAMETER VALUES
*                       ( Not used unless value is fixed)
*       SIGA, SIGB,     = 0 TO FIX PARAMETER VALUE AT DEFAULT
*       SIGWID, SIGX0   > 0 TO VARY THE PARAMETER
*
*  OUTPUT:
*       A       = Area          SIGA = FORMAL ERROR IN A
*       B       = Background    SIGB = FORMAL ERROR IN B
*       WID     = WIDTH         SIGWID = FORMAL ERROR IN WID
*       X0      = POSITION      SIGX0 = FORMAL ERROR IN X0
*
*  For best results, provide data symmetrically disposed about line centre
*  Modified Nov 1984 by Keith Horne at IOA
*  Dec 86 KDH @ STScI - switch to 1-parameter confidence limits.
* Jan 90 KDH @ STScI - reduce verbiage, eliminate type

        IMPLICIT NONE
        INTEGER N, NPARAMS
        REAL X(N),Y(N), SIGY(N)
        PARAMETER (NPARAMS=4)
        REAL PAR(NPARAMS)
        INTEGER IFIX(NPARAMS), IOUT, IMIN, IMAX, IQUT
        INTEGER I, NOBS, NPAR, NVAR, NFIX, IDER, INCI
        INTEGER IPRT, NFIT
        REAL BU, BL, RMS, A, B, WID, X0
        REAL SIGA, SIGB, SIGWID, SIGX0
        EXTERNAL FGFCODE, FGPCODE
        REAL STE, OPL, OPU, SPL, SPU, YMEAN, WTS
        COMMON/WEITS/WTS(500)           ! Data point weights used by LSQENP
        COMMON/LSQIOUT/ IOUT            ! Output unit number
        COMMON/LSQSTE/ STE(NPARAMS)     ! Standard 1-sigma uncertainty
        COMMON/LSQOPL/ OPL(NPARAMS)     ! One-parameter lower limits
        COMMON/LSQOPU/ OPU(NPARAMS)     ! One-parameter upper limits
        COMMON/LSQSPL/ SPL(NPARAMS)     ! Support plane lower limits
        COMMON/LSQSPU/ SPU(NPARAMS)     ! Support plane upper limits
        COMMON/LSQBU/ BU(NPARAMS)       ! Nonlinear upper limits
        COMMON/LSQBL/ BL(NPARAMS)       ! Nonlinear lower limits
        LOGICAL FIRSTCALL, STANDARD, ONEP  
        DATA FIRSTCALL, STANDARD, ONEP/.TRUE., .FALSE.,.TRUE./
        DATA (IFIX(I),I=1,NPARAMS) /NPARAMS*0/
* introduction


      IF( FIRSTCALL ) THEN
        FIRSTCALL = .FALSE.
        WRITE(*,*) 'A NOTE ABOUT ERROR BARS:'
        WRITE(*,*)
     #  'FITGAUSS reports approximate 1-sigma parameter uncertainties'
        WRITE(*,*) ' that in fact are 1/4 of'//
     &' the 2-sigma "single-parameter"'
        WRITE(*,*) ' symmetric confidence '//
     &'region estimated from the curvature'
        WRITE(*,*) ' tensor at the minimum of the chi-squared surface.'
      END IF

C  Compute data weights = 1/VARIANCE


        IMIN = 1
        IMAX = 1
        YMEAN = 0.
        NFIT = 0
        RMS = 0.
      DO I=1,N
      IF( SIGY(I).LE.0. ) THEN
        WTS(I) = 0.
      ELSE
        NFIT = NFIT + 1
        RMS = RMS + SIGY(I)*SIGY(I)
        WTS(I) = 1. / ( SIGY(I)*SIGY(I) )
        IF(Y(I).GT.Y(IMAX) ) IMAX = I
        IF(Y(I).LT.Y(IMIN) ) IMIN = I
        YMEAN = YMEAN + Y(I)
      END IF
      END DO
        YMEAN = YMEAN/NFIT
      IF( NFIT.LE.4 ) THEN
        WRITE(*,*) CHAR(7),'** INSUFFICIENT DATA. ''FITGAUSS'' ABORTED.'
        RETURN
      END IF
        RMS = RMS/NFIT
      DO I=1,N
        WTS(I) = WTS(I) * RMS
      END DO
        IF( NFIT.NE.N) WRITE(*,*) CHAR(7), 'REJECTED DATA:', N-NFIT

        WRITE(*,*) 'FITGAUSS - gaussian fit to', NFIT, ' data points.'

*  Initialize variables for call to LSQENP
       

        NOBS=N  ! Number of observations
        NPAR=4  ! Number of parameters
        NVAR=1  ! Number of variables/observation
        NFIX=0  ! Number of fixed parameters
        IDER=0  ! =0 for analytic derivatives
        INCI=0
        IQUT=200
        IOUT=-1 ! SUPPRESS PRINTOUT
        IPRT=-1 ! SUPPRESS PRINTOUT
      DO I=1,4          ! CANCEL PREVIOUS CONFIDENCE LIMITS
        SPL(I) = 0.
        SPU(I) = 0.
        OPL(I) = 0.
        OPU(I) = 0.
        BU(I) = 0.
        BL(I) = 0.
      END DO

*  Set parameter starting values

        PAR(1) = A      ! Area
        PAR(2) = B      ! Background
        PAR(3) = WID    ! Width
        PAR(4) = X0     ! Centre

*  Report parameter starting values

      IF( SIGA.LE.0. ) THEN
        WRITE(*,*) '  area:', A, ' FIXED'
        NFIX = NFIX + 1
        IFIX(NFIX) = 1
      ELSE
      END IF

      IF( SIGB.LE.0. ) THEN
        WRITE(*,*) '    bg:', B, ' FIXED'
        NFIX = NFIX + 1
        IFIX(NFIX) = 2
      ELSE
      END IF

      IF( SIGWID.LE.0. ) THEN
        WRITE(*,*) ' sigma:', WID, ' FIXED'
        NFIX = NFIX + 1
        IFIX(NFIX) = 3
      ELSE
      END IF

      IF( SIGX0.LE.0. ) THEN
        WRITE(*,*) 'center:', X0, ' FIXED'
        NFIX = NFIX + 1
        IFIX(NFIX) = 4
      ELSE
      END IF

*  LSQENP optimizes the parameters

        CALL LSQENP( NOBS, NPAR, NVAR, Y, X, PAR, NFIX, IFIX,
     *  IDER, INCI, IQUT, IPRT, FGFCODE, FGPCODE )

*  Move parameters

        A = PAR(1)
        B = PAR(2)
        WID = PAR(3)
        X0 = PAR(4)

*  Use 1-parameter confidence limits (CONVERT 2-sigma TO 1-SIGMA EQUIVALENTS)

      IF( STANDARD ) THEN
        IF( SIGA.GT.0. ) SIGA = STE(1)
        IF( SIGB.GT.0. ) SIGB = STE(2)
        IF( SIGWID.GT.0. ) SIGWID = STE(3)
        IF( SIGX0.GT.0. ) SIGX0 = STE(4)

      ELSE IF( ONEP ) THEN
        IF( SIGA.GT.0. ) SIGA = 0.25*( OPU(1) - OPL(1) )
        IF( SIGB.GT.0. ) SIGB = 0.25*( OPU(2) - OPL(2) )
        IF( SIGWID.GT.0. ) SIGWID = 0.25*( OPU(3) - OPL(3) )
        IF( SIGX0.GT.0. ) SIGX0 = 0.25*( OPU(4) - OPL(4) )

*  Use support plane confidence limits (CONVERT TO 1-SIGMA EQUIVALENTS)
      
      ELSE
        IF( SIGA.GT.0. ) SIGA = 0.25*( SPU(1) - SPL(1) )
        IF( SIGB.GT.0. ) SIGB = 0.25*( SPU(2) - SPL(2) )
        IF( SIGWID.GT.0. ) SIGWID = 0.25*( SPU(3) - SPL(3) )
        IF( SIGX0.GT.0. ) SIGX0 = 0.25*( SPU(4) - SPL(4) )
      END IF

*  Report results

      IF( STANDARD ) THEN
      ELSE IF( ONEP ) THEN
      ELSE
      END IF
        IF( SIGA.GT.0. ) WRITE(*,*) '  area:', A, ' +-', SIGA,
     #          ' (',BL(1)-A, BU(1)-A,')=90%'
        IF( SIGB.GT.0. ) WRITE(*,*) '    bg:', B, ' +-', SIGB,
     #          ' (',BL(2)-B, BU(2)-B,')=90%'
        IF( SIGWID.GT.0. ) WRITE(*,*) ' sigma:', WID, ' +-', SIGWID,
     #          ' (',BL(3)-WID, BU(3)-WID,')=90%'
        IF( SIGX0.GT.0. ) WRITE(*,*)  'center:', X0, ' +-', SIGX0,
     #          ' (',BL(4)-X0, BU(4)-X0,')=90%'

        RETURN
      END

        SUBROUTINE FGFCODE(Y,X,B,F,I)
C
C  LSQENP ROUTINE TO EVALUATE GAUSSIAN + BACKGROUND FUNCTION F(I)
C
C  B(1) = AREA
C  B(2) = BACKGROUND
C  B(3) = SIGMA
C  B(4) = POSITION
C
        REAL*4 Y(1),X(500,1),B(4)
        IF( TWOPI.LE.0. ) TWOPI = ATAN2(1.,1.)*8.
        ROOT = 1./SQRT(TWOPI)
        XX = X(I,1) - B(4)
        SI = 1./B(3)
         A = XX*SI
         G = SI*ROOT*EXP(-0.5*A*A)
         F = B(1)*G + B(2)
        RETURN
      END

        SUBROUTINE FGPCODE(P,X,B,F,I)
C
C  LSQUAR ROUTINE TO COMPUTE DERIVATIVES OF FUNCTION F(I) WITH RESPECT
C  TO PARAMETERS:
C
C  B(1) = AREA
C  B(2) = BACKGROUND
C  B(3) = SIGMA
C  B(4) = POSITION
C
        REAL*4 P(4),X(500,1),B(4)
        TWOPI = ATAN2(1.,1.)*8.
        ROOT = 1./SQRT(TWOPI)
        XX = X(I,1) - B(4)
        SI = 1./B(3)
        A = XX*SI
        GA = F-B(2)
      IF( B(1) .NE. 0. ) THEN
        G = GA/B(1)     ! This version is fast, but crashes if B(1) = 0.
      ELSE
        XX = X(I,1) - B(4)      ! This version does not crash.
        SI = 1./B(3)
         A = XX*SI
         G = SI*ROOT*EXP(-0.5*A*A)
      END IF
        P(1)=G
        P(2)=1.
        P(4)=GA*A*SI
        P(3)=P(4)*A - GA*SI
        RETURN
      END




        SUBROUTINE GAUSSMIN(P, N, FTOL, ITER, FRET)

* Given a starting point P that is a vector of length N,
* Fletcher-Reeves-Polak-Ribiere minimisation is performed
* on a function FGAUSS using its gradient as calculated by a
* routine DGAUSS. The convergence tolerance on the function
* value is input as FTOL. Returned  quatities are P (the
* location of the minimum), ITER the number of iterations
* that were performed), and FRET (the minimum of the function).
* The routine DGAUSSLIN is called to perform the line minimisations
*
* TRM @RGO 2/10/87 Input: ITER = Max number of iterations to do
*                  Output: returned ITER number done
*
* This is a specific implementation for Gaussian fitting
* You need to pass data in common blocks to the routines
* FGAUSS and DGAUSS so see those routines.
*
* Possibilities:
*
* Parameter METHOD passed through common
*
* METHOD=0: Y = P(1) + P(2)*EXP(-(X-P(4))**2/2/P(3)**2) is fitted
* with all parameters varied (N=4)
* METHOD=1: Y = BACK + P(1)*EXP(-(X-P(3))**2/2/P(2)**2) is fitted
* BACK is passed by common

        IMPLICIT NONE 
        INTEGER N, NMAX, ITS, ITMAX, ITER, J
        DOUBLE PRECISION EPS, FP, FGAUSS, FRET, FTOL
        DOUBLE PRECISION GG, DGG, GAM
        PARAMETER (NMAX=4,EPS=1.E-10)
        DOUBLE PRECISION P(N), G(NMAX), H(NMAX), XI(NMAX)

        FP=FGAUSS(P)
        print *,'DGAUSS(1)'
        CALL DGAUSS(P,XI)
        DO J=1,N
          G(J)=-XI(J)
          H(J)=G(J)
          XI(J)=H(J)
        END DO
        ITMAX = ITER
        DO ITS=1, ITMAX
          ITER=ITS
          print *,'DGAUSSLIN'
          CALL DGAUSSLIN(P,XI,N,FRET)
          IF(2.*ABS(FRET-FP).LE.FTOL*(ABS(FRET)+ABS(FP)+EPS)) RETURN
          FP=FGAUSS(P)
          print *,'DGAUSS(2)'
          CALL DGAUSS(P,XI)
          GG=0.
          DGG=0.
          DO J =1,N
            GG=GG+G(J)**2
            DGG=DGG+XI(J)**2
            DGG=DGG+(XI(J)+G(J))*XI(J)
          END DO
          IF(GG.EQ.0) RETURN
          GAM=DGG/GG
          DO J=1, N
            G(J)=-XI(J)
            H(J)=G(J)+GAM*H(J)
            XI(J)=H(J)
          END DO
        END DO
        RETURN
        END

        SUBROUTINE DGAUSSLIN(P,XI,N,FRET)

* Given an N dimensional point P and an N dimensional direction
* XI, moves and resets P to where the function FGAUSS(P) is
* minimum  along the direction XI from P and replaces XI by
* the actual vector displacement that P was moved. Also returns
* as FRET the value of FGAUSS at the returned location P. This
* is actually all accomplished by calling the routines MNBRAK
* and BRENT
* NUMERICAL RECIPES program

        IMPLICIT NONE 
        INTEGER NMAX, N, NCOM, J
        DOUBLE PRECISION TOL, PCOM, XICOM, AX, BX, XX
        DOUBLE PRECISION F1GAUSS, DF1GAUSS, FA, FX, FB
        DOUBLE PRECISION FRET, XMIN, DBRENT
        PARAMETER (NMAX=4, TOL=1.E-4)
        EXTERNAL F1GAUSS, DF1GAUSS
        REAL*8 P(N), XI(N)
        COMMON/F1COM/ PCOM(NMAX), XICOM(NMAX), NCOM

        NCOM = N
        DO J = 1, N
          PCOM(J)  = P(J)
          XICOM(J) = XI(J)
        END DO
        AX = 0.D0
        XX = 1.D0
        BX = 2.D0
         print *,'MNBRAK'
        CALL MNBRAK(AX, XX, BX, FA, FX, FB, F1GAUSS)
        FRET = DBRENT(AX, XX, BX, F1GAUSS, DF1GAUSS, TOL, XMIN)
        DO J= 1, N
          XI(J) = XMIN*XI(J)
          P(J) = P(J) + XI(J)
        END DO
        RETURN
        END

        DOUBLE PRECISION FUNCTION F1GAUSS(X)
        IMPLICIT NONE
        INTEGER NMAX, NCOM, I
        DOUBLE PRECISION PCOM, XICOM, FGAUSS, X
        PARAMETER (NMAX=4)
        COMMON/F1COM/ PCOM(NMAX), XICOM(NMAX), NCOM
        DOUBLE PRECISION XT(NMAX)
*
        DO I = 1, NCOM
          XT(I) = PCOM(I) + X*XICOM(I)
        END DO
        F1GAUSS = FGAUSS(XT)
        RETURN
        END

        DOUBLE PRECISION FUNCTION DF1GAUSS(X)
        IMPLICIT NONE 
        INTEGER NMAX, NCOM, I, J
        DOUBLE PRECISION PCOM, XICOM, X
        PARAMETER (NMAX=4)
        COMMON/F1COM/ PCOM(NMAX), XICOM(NMAX), NCOM
        DOUBLE PRECISION XT(NMAX), DF(NMAX)

        DO I = 1, NCOM
          XT(I) = PCOM(I) + X*XICOM(I)
        END DO
        print *,'DGAUSS(3)'
        CALL DGAUSS(XT,DF)
        DF1GAUSS = 0.D0
        DO J=1, NCOM
          DF1GAUSS = DF1GAUSS + DF(J)*XICOM(J)
        END DO
        RETURN
        END

        DOUBLE PRECISION FUNCTION FGAUSS(P)

* 1/10/87 TRM @RGO
* Function for Gaussian fitting.
*
* One component, accounting for undersampling. Arrays
* of X,Y,Uncertainties passed by common.
* Y = P(1) + P(2)*EXP(-((X-P(4))/P(3))**2/2) (Approximate; EXP term
* integrated over a bin
* Pass through common
* Number of points for fit, X binsize, X, Y and error arrays
* E.G. if integer bins, XBIN = 1.
* If error estimate is negative or zero, point will be ignored
*
* METHOD = 0 Background, strength, width position fitted
* METHOD = 1 Strength, width, position fitted


        IMPLICIT NONE 
        INTEGER METHOD, NGAUS1, NGAUS2, NFREE, I, IFAIL
        REAL XBIN, BACK
        DOUBLE PRECISION PDA_DERF, P(4), SWAP(4), FAC2, HALF
        DOUBLE PRECISION X, X1, X2, YFIT
        COMMON/NGAUSS/ METHOD, BACK, NGAUS1, NGAUS2, XBIN
        REAL PLOT1(1)
        REAL PLOT2(1)
        REAL PLOT3(1)
        COMMON/PLOT1/ PLOT1     ! X array
        COMMON/PLOT2/ PLOT2     ! Y array
        COMMON/PLOT3/ PLOT3     ! error array

        IF(METHOD.EQ.0) THEN
          DO I = 1, 4
            SWAP(I) = P(I)
          END DO
        ELSE IF(METHOD.EQ.1) THEN
          SWAP(1) = BACK
          SWAP(2) = P(1)
          SWAP(3) = P(2)
          SWAP(4) = P(3)
        END IF
        FGAUSS = 0.
        FAC2 = SWAP(2)*SWAP(3)*SQRT(8.*ATAN(1.))/XBIN
        HALF = XBIN/2./SWAP(3)
        NFREE = 0
        DO I = NGAUS1, NGAUS2
          IF(PLOT3(I).GT.0.) THEN
            X = (PLOT1(I)-SWAP(4))/SWAP(3)
            X1 = X - HALF
            X2 = X + HALF
            YFIT = SWAP(1) + 
     &      FAC2*(0.5*(PDA_DERF(-X1/SQRT(2.))-
     &           DERFC(-X2/SQRT(2.))))
            FGAUSS = FGAUSS + ((PLOT2(I)-YFIT)/PLOT3(I))**2
            NFREE = NFREE + 1
          END IF
        END DO
        FGAUSS = FGAUSS/DBLE(NFREE)
        RETURN
        END

        SUBROUTINE DGAUSS(P, DF)

* 1/10/87 TRM @RGO
* Derivative function for Gaussian fitting.
*

        IMPLICIT NONE 
        INTEGER METHOD, NGAUS1, NGAUS2, I, NFREE, IFAIL
        REAL XBIN, BACK
        DOUBLE PRECISION PDA_DERF, P(4), DF(4), SWAP(4), FAC2
        DOUBLE PRECISION AREA, HALF, X, X1, X2, FRAC, FIRST
        DOUBLE PRECISION EX1, EX2
        COMMON/NGAUSS/ METHOD, BACK, NGAUS1, NGAUS2, XBIN
        REAL PLOT1(1)
        REAL PLOT2(1)
        REAL PLOT3(1)
        COMMON/PLOT1/ PLOT1     ! X array
        COMMON/PLOT2/ PLOT2     ! Y array
        COMMON/PLOT3/ PLOT3     ! error array

        IF(METHOD.EQ.0) THEN
          DO I = 1, 4
            SWAP(I) = P(I)
          END DO
        ELSE IF(METHOD.EQ.1) THEN
          SWAP(1) = BACK
          SWAP(2) = P(1)
          SWAP(3) = P(2)
          SWAP(4) = P(3)
        END IF
        FAC2 = SWAP(3)*SWAP(2)
        DO I = 1, 4
          DF(I) = 0.D0
        END DO
        AREA = SQRT(8.*ATAN(1.))/XBIN
        HALF = XBIN/2./SWAP(3)
        NFREE = 0
        DO I = NGAUS1, NGAUS2
          IF(PLOT3(I).GT.0.) THEN
            X = (PLOT1(I)-SWAP(4))/SWAP(3)
            X1 = X - HALF
            X2 = X + HALF
            FRAC = AREA*
     &           0.5*(PDA_DERF(-X1/SQRT(2.)) - 
     &           PDA_DERF(-X1/SQRT(2.)))
            FIRST = -2.D0*(PLOT2(I)-SWAP(1)-FAC2*FRAC)/PLOT3(I)**2
            DF(1) = DF(1) + FIRST
            DF(2) = DF(2) + FIRST*FRAC*SWAP(3)
            EX1 = EXP(-X1**2/2.D0)
            EX2 = EXP(-X2**2/2.D0)
            DF(3) = DF(3) + FIRST*SWAP(2)*(X1*EX1-X2*EX2+FRAC)
            DF(4) = DF(4) + FIRST*SWAP(2)*(EX1-EX2)
            NFREE = NFREE + 1
          END IF
        END DO
        DO I = 1, 4
          DF(I) = DF(I)/DBLE(NFREE)
        END DO
        IF(METHOD.EQ.1) THEN
          DF(1) = DF(2)
          DF(2) = DF(3)
          DF(3) = DF(4)
        END IF
        RETURN
        END

        DOUBLE PRECISION FUNCTION GSIM(CEN,WID,GWID,NSIM)
C
C CALCULATES THE VALUE OF A GAUSSIAN CURVE, ACCOUNTING FOR
C UNDERSAMPLING USING SIMPSONS' RULE. USED WHEN A FINITE
C LENGTH PIXEL POSITIONED IN A GAUSSIAN AND AN AVERAGE
C VALUE IS REQUIRED OVER THE PIXEL.
C
C PASSED:
C       REAL*8 GWID GAUSSIAN REPRESENTED AS EXP(-X*X*GWID)
C                   GWID IS THEREFORE RELATED TO THE WIDTH
C                   OF THE GAUSSIAN.
C       REAL*8 NSIM -- ODD INTEGER, REPRESENTING THE NUMBER OF
C                      POINTS TO BE USED IN THE SIMPSON APPROX.
C       REAL*8 CEN,WID  THE POSITION OF THE PIXEL RELATIVE TO
C                       THE GAUSSIAN CENTRE, AND THE FULL WIDTH
C                       OF THE PIXEL.
C RETURNED:
C       REAL*8 GSIM -- THE CALCULATED VALUE
C
C *** WARNING NO CHECKS OF INPUTS ARE MADE, SO CRASHES ARE POSSIBLE
C *** GIVEN BAD INPUTS.
C
        IMPLICIT NONE 
        INTEGER NSIM, I
        DOUBLE PRECISION CEN, GWID, DIFF, WID, POS
C
        IF(NSIM.EQ.1) THEN
          GSIM=EXP(-CEN*CEN*GWID)
          RETURN
        END IF
        DIFF=WID/DBLE(NSIM-1)
        POS=CEN-WID/2.
        GSIM=EXP(-POS*POS*GWID)
        POS=CEN+WID/2.
        GSIM=GSIM+EXP(-POS*POS*GWID)
        DO I=2,NSIM-1
          POS=CEN+DBLE(I-NSIM/2-1)*DIFF
          POS=POS*POS*GWID
          IF(MOD(I,2).EQ.0) THEN
            GSIM=GSIM+4.D0*EXP(-POS)
          ELSE
            GSIM=GSIM+2.D0*EXP(-POS)
          END IF
        END DO
        GSIM=GSIM/3./DBLE(NSIM-1)
        RETURN
        END
C
C
        DOUBLE PRECISION FUNCTION DSIM(CEN,WID,GWID,NSIM)
C
C CALCULATES THE DERIVATIVE WITH RESPECT TO CEN
C ALL ARGUMENTS AS ABOVE
C
C
C
C *** WARNING NO CHECKS OF INPUTS ARE MADE, SO CRASHES ARE POSSIBLE
C *** GIVEN BAD INPUTS.
C
        IMPLICIT NONE 
        INTEGER NSIM, I
        DOUBLE PRECISION CEN, WID, GWID, DIFF, POS, PS
C
        IF(NSIM.EQ.1) THEN
          DSIM=-2.*CEN*GWID*EXP(-CEN*CEN*GWID)
          RETURN
        END IF
        DIFF=WID/DBLE(NSIM-1)
        POS=CEN-WID/2.
        DSIM=POS*EXP(-POS*POS*GWID)
        POS=CEN+WID/2.
        DSIM=DSIM+POS*EXP(-POS*POS*GWID)
        DO I=2,NSIM-1
          POS=CEN+DBLE(I-NSIM/2-1)*DIFF
          PS=POS*POS*GWID
          IF(MOD(I,2).EQ.0) THEN
            DSIM=DSIM+4.D0*POS*EXP(-PS)
          ELSE
            DSIM=DSIM+2.D0*POS*EXP(-PS)
          END IF
        END DO
        DSIM=-2.*GWID*DSIM/3./DBLE(NSIM-1)
        RETURN
        END
C
C
        DOUBLE PRECISION FUNCTION HSIM(CEN,WID,GWID,NSIM)
C
C CALCULATES THE DERIVATIVE WITH RESPECT TO GWID
C ALL ARGUMENTS AS ABOVE
C
C
C
C *** WARNING NO CHECKS OF INPUTS ARE MADE, SO CRASHES ARE POSSIBLE
C *** GIVEN BAD INPUTS.
C
        IMPLICIT NONE 
        INTEGER NSIM, I
        DOUBLE PRECISION CEN, WID, GWID, DIFF, POS, PS
C
        IF(NSIM.EQ.1) THEN
          HSIM=-CEN*CEN*EXP(-CEN*CEN*GWID)
          RETURN
        END IF
        DIFF=WID/DBLE(NSIM-1)
        POS=CEN-WID/2.
        HSIM=-POS*POS*EXP(-POS*POS*GWID)
        POS=CEN+WID/2.
        HSIM=HSIM-POS*POS*EXP(-POS*POS*GWID)
        DO I=2,NSIM-1
          POS=CEN+DBLE(I-NSIM/2-1)*DIFF
          PS=POS*POS
          IF(MOD(I,2).EQ.0) THEN
            HSIM=HSIM-4.D0*PS*EXP(-PS*GWID)
          ELSE
            HSIM=HSIM-2.D0*PS*EXP(-PS*GWID)
          END IF
        END DO
        HSIM=HSIM/3./DBLE(NSIM-1)
        RETURN
        END
