      SUBROUTINE DSTREAM(X,V,Q,STEP,ACC,DELTA)
*
* Controls Runge-Kutta integration of gas
* stream by alteration of stepsize. It loops until
* it finds a sufficiently small step.
*
* REAL*8 X(3) -- Position
* REAL*8 V(3) -- Velocity
* REAL*8 Q    -- Mass ratio
* REAL*8 STEP -- Time step to be used, and, on exit, the suggested
*
* REAL*8 ACC  -- Allowed change in Jacobi constant/unit time step
* REAL*8 DELTA - Actual time step used.
*
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 X(3), V(3), Q, STEP, ACC, DELTA
      REAL*8 XS(3), VS(3), JACOBI, XB(3), VB(3)
*
10    CONTINUE
      DO I = 1, 3
        XB(I) = X(I)
        VB(I) = V(I)
      END DO
      TSTEP = STEP
      CALL STRINT(XB,VB,Q,TSTEP)
*
* Take two half size steps
*
      TSTEP = STEP/2.
      DO I = 1, 3
        XS(I) = X(I)
        VS(I) = V(I)
      END DO
      CALL STRINT(XS,VS,Q,TSTEP)
      CALL STRINT(XS,VS,Q,TSTEP)
*
* Error
*
      ERROR = ABS(JACOBI(XS,VS,Q)-JACOBI(XB,VB,Q))/ACC*STEP
      IF(ERROR.GT.1.) THEN
        STEP = 0.9*STEP/ERROR**2.5D-1
        GOTO 10
      ELSE
        DELTA = STEP
        IF(ERROR.GT.1.D-3) THEN
          STEP = 0.9*STEP/ERROR**2.D-1
        ELSE
          STEP = 4.*STEP
        END IF
      END IF
      DO I = 1, 3
        X(I) = (16.*XS(I)-XB(I))/15.
        V(I) = (16.*VS(I)-VB(I))/15.
      END DO
      RETURN
      END

      SUBROUTINE STRINT(X, V, Q, STEP)
*
* Integrates equations of motion in Roche potential
* for one time step. 4th order Runge-Kutta.
*
* Unit defined by a = 1, P = 2*PI
*
* I = input, O = output
*
* I/O REAL*8 X(3) -- Coordinates.
* I/O REAL*8 V(3) -- Velocity.
* I   REAL*8 Q    -- Mass ratio.
* I   REAL*8 STEP -- Time step size.
*
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 X(3), V(3), Q, STEP, TX(3), TV(3), A(3)
      REAL*8 X1(3), X2(3), X3(3), X4(3)
      REAL*8 V1(3), V2(3), V3(3), V4(3)
*
* Go through 4 steps of Runge-Kutta 4th order
*
* #1
*
      CALL ROCACC(X,V,Q,A)
      DO I = 1, 3
        V1(I) = STEP*A(I)
        X1(I) = STEP*V(I)
      END DO
*
* #2
*
      DO I = 1, 3
        TX(I) = X(I) + X1(I)/2.
        TV(I) = V(I) + V1(I)/2.
      END DO
      CALL ROCACC(TX,TV,Q,A)
      DO I = 1, 3
        V2(I) = STEP*A(I)
        X2(I) = STEP*TV(I)
      END DO
*
* #3
*
      DO I = 1, 3
        TX(I) = X(I) + X2(I)/2.
        TV(I) = V(I) + V2(I)/2.
      END DO
      CALL ROCACC(TX,TV,Q,A)
      DO I = 1, 3
        V3(I) = STEP*A(I)
        X3(I) = STEP*TV(I)
      END DO
*
* #4
*
      DO I = 1, 3
        TX(I) = X(I) + X3(I)/2.
        TV(I) = V(I) + V3(I)/2.
      END DO
      CALL ROCACC(TX,TV,Q,A)
      DO I = 1, 3
        V4(I) = STEP*A(I)
        X4(I) = STEP*TV(I)
      END DO
*
* New point
*
      DO  I = 1, 3
        V(I) = V(I) + (V1(I)+2.*(V2(I)+V3(I))+V4(I))/6.
        X(I) = X(I) + (X1(I)+2.*(X2(I)+X3(I))+X4(I))/6.
      END DO
      RETURN
      END

      SUBROUTINE ROCACC(X,V,Q,A)
*
* Computes acceleration in rotating coordinates in Roche potential
*
* REAL*8 X(3) -- Coordinates (units of a)
* REAL*8 V(3) -- Velocity (units of 2*PI*a/P)
* REAL*8 Q    -- Mass ratio M2/M1
* REAL*8 A(3) -- Acceleration (units (2*PI/P)**2*A)
*
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 X(3), V(3), Q, A(3)
*
      F1 = 1./(1.+Q)
      F2 = F1*Q
*
      YZSQ = X(2)**2 + X(3)**2
      R1SQ = X(1)**2 + YZSQ
      R2SQ = (X(1)-1.)**2 + YZSQ
      FM1 = F1/(R1SQ*SQRT(R1SQ))
      FM2 = F2/(R2SQ*SQRT(R2SQ))
      FM3 = FM1+FM2
*
      A(1) = - FM3*X(1) + FM2 + 2.*V(2) + (X(1) - F2)
      A(2) = - FM3*X(2)       - 2.*V(1) + X(2)
      A(3) = - FM3*X(3)
      RETURN
      END

      REAL*8 FUNCTION JACOBI(X,V,Q)
*
* Evaluates Jacobi constant
*
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 X(3), V(3), Q
*
      F1 = 1./(1.+Q)
      F2 = F1*Q
*
      YZSQ = X(2)**2 + X(3)**2
      FM1 = F1/SQRT(X(1)**2+YZSQ)
      FM2 = F2/SQRT((X(1)-1.)**2+YZSQ)
      JACOBI = (V(1)**2+V(2)**2+V(3)**2
     &          -X(2)**2-(X(1)-F2)**2)/2.-FM1-FM2
      RETURN
      END

      SUBROUTINE GSBS(X,V,Q,STEP,ACC,DELTA)
*
* Burlisch-Stoer step with monitoring of local truncation error to
* ensure accuracy and adjust step size.
*
* X(3), V(3) are the position and velocity, updated on exit.
* Q is the mass ratio
* STEP is the step size to attempt, on exit is the suggested next step
* ACC is the accuracy in position wanted
* DELTA is the actual one carried out.
*
      IMPLICIT REAL*8 (A-H,O-Z)
      PARAMETER (IMAX=11,SHRINK=0.95,NUSE=7,GROW=1.2)
      REAL*8 X(3), V(3), Y(6), YERR(6), Q, STEP, ACC
      REAL*8 YS(6),  YSEQ(6), DELTA
      INTEGER NSEQ(IMAX)
      DATA NSEQ/2,4,6,8,12,16,24,32,48,64,96/
*
      H = STEP
      DO I = 1, 3
        YS(I)   = X(I)
        YS(I+3) = V(I)
      END DO
*
1     CONTINUE
      DO I = 1, IMAX
        CALL GSMID(YS,H,NSEQ(I),YSEQ,Q)
        TEST = (H/NSEQ(I))**2
        CALL GSEXTR(I,TEST,YSEQ,Y,YERR,NUSE)
        ERRMAX = 0.
        DO J = 1, 6
          IF(J.LE.3) THEN
            ERRMAX = MAX(ERRMAX, ABS(YERR(J)))
          ELSE
            ERRMAX = MAX(ERRMAX, H*ABS(YERR(J)))
          END IF
        END DO
        IF(ERRMAX.LT.ACC) THEN
          DELTA = H
          SK = STEP
          IF(I.EQ.NUSE) THEN
            STEP = H*SHRINK
          ELSE IF(I.EQ.NUSE-1) THEN
            STEP = H*GROW
          ELSE
            STEP = (H*NSEQ(NUSE-1))/NSEQ(I)
          END IF
          DO J = 1, 3
            X(J) = Y(J)
            V(J) = Y(J+3)
          END DO
          RETURN
        END IF
      END DO
      H = 0.25*H/2**((IMAX-NUSE)/2)
      WRITE(*,*) 'Possible step size error in GSBS'
      GOTO 1
      END

      SUBROUTINE GSMID(Y,TSTEP,NSTEP,YOUT,Q)
*
* Modified mid-point method. Advances particle
* from Y to YOUT, with a step of length
* TSTEP divided into NSTEP bits.
* YOUT contains positions and velocities
*
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 Y(6), YOUT(6)
      REAL*8 YS(6), YN(6)
      REAL*8 A(6)
*
      H = TSTEP/DBLE(NSTEP)
      DO I = 1, 3
        A(I) = Y(I+3)
      END DO
      CALL ROCACC(Y,Y(4),Q,A(4))
      DO I = 1, 6
        YS(I) = Y(I)
        YN(I) = Y(I) + H*A(I)
      END DO
      DO I = 1, 3
        A(I) = YN(I+3)
      END DO
      CALL ROCACC(YN,YN(4),Q,A(4))
      H2 = 2.*H
      DO N = 2, NSTEP
        DO I = 1, 6
          SWAP = YS(I) + H2*A(I)
          YS(I) = YN(I)
          YN(I) = SWAP
        END DO
        DO I = 1, 3
          A(I) = YN(I+3)
        END DO
        CALL ROCACC(YN,YN(4),Q,A(4))
      END DO
      DO I = 1, 6
        YOUT(I) = 0.5*(YS(I)+YN(I)+H*A(I))
      END DO
      RETURN
      END

      SUBROUTINE GSEXTR(IEST,TEST,YEST,YZ,DY,NUSE)
*
* Rational function extrapolation
* YEST(6) contains positions then velocities.
*
      IMPLICIT REAL*8 (A-H,O-Z)
      PARAMETER (IMAX=11,NCOL=7)
      REAL*8 YEST(6), YZ(6), DY(6)
      REAL*8 T(IMAX), D(6,NCOL), FX(NCOL)
*
      T(IEST) = TEST
      IF(IEST.EQ.1) THEN
        DO J = 1, 6
          YZ(J)  = YEST(J)
          D(J,1) = YEST(J)
          DY(J)  = YEST(J)
        END DO
      ELSE
        M1 = MIN(IEST, NUSE)
        DO K = 1, M1-1
          FX(K+1) = T(IEST-K)/TEST
        END DO
        DO J = 1, 6
          YY = YEST(J)
          V  = D(J,1)
          C  = YY
          D(J,1) = YY
          DO K = 2, M1
            B1 = FX(K)*V
            B = B1 - C
            IF(B.NE.0.) THEN
              B = (C-V)/B
              DDY = C*B
              C = B1*B
            ELSE
              DDY = V
            END IF
            V = D(J,K)
            D(J,K) = DDY
            YY = YY + DDY
          END DO
          DY(J) = DDY
          YZ(J) = YY
        END DO
      END IF
      RETURN
      END

      SUBROUTINE GSBS1(X,V,NU,CHI,Q,STEP,ACC,DELTA)
*
* Burlisch-Stoer step with monitoring of local truncation error to
* ensure accuracy and adjust step size.
*
* X(3), V(3) are the position and velocity, updated on exit.
* Q is the mass ratio
* STEP is the step size to attempt, on exit is the suggested next step
* ACC is the accuracy in position wanted
* DELTA is the actual one carried out.
*
      IMPLICIT REAL*8 (A-H,O-Z)
      PARAMETER (IMAX=11,SHRINK=0.95,NUSE=7,GROW=1.2)
      REAL*8 X(3), V(3), Y(8), YERR(8), Q, STEP, ACC
      REAL*8 YS(8),  YSEQ(8), DELTA, NU, CHI
      INTEGER NSEQ(IMAX)
      DATA NSEQ/2,4,6,8,12,16,24,32,48,64,96/
*
      H = STEP
      DO I = 1, 3
        YS(I)   = X(I)
        YS(I+3) = V(I)
      END DO
      YS(7) = NU
      YS(8) = CHI
*
1     CONTINUE
      DO I = 1, IMAX
        CALL GSMID1(YS,H,NSEQ(I),YSEQ,Q)
        TEST = (H/NSEQ(I))**2
        CALL GSEXTR1(I,TEST,YSEQ,Y,YERR,NUSE)
        ERRMAX = 0.
        DO J = 1, 6
          IF(J.LE.3) THEN
            ERRMAX = MAX(ERRMAX, ABS(YERR(J)))
          ELSE
            ERRMAX = MAX(ERRMAX, H*ABS(YERR(J)))
          END IF
        END DO
        IF(ERRMAX.LT.ACC) THEN
          DELTA = H
          SK = STEP
          IF(I.EQ.NUSE) THEN
            STEP = H*SHRINK
          ELSE IF(I.EQ.NUSE-1) THEN
            STEP = H*GROW
          ELSE
            STEP = (H*NSEQ(NUSE-1))/NSEQ(I)
          END IF
          DO J = 1, 3
            X(J) = Y(J)
            V(J) = Y(J+3)
          END DO
          NU  = Y(7)
          CHI = Y(8)
          RETURN
        END IF
      END DO
      H = 0.25*H/2**((IMAX-NUSE)/2)
      WRITE(*,*) 'Possible step size error in GSBS'
      GOTO 1
      END

      SUBROUTINE GSMID1(Y,TSTEP,NSTEP,YOUT,Q)
*
* Modified mid-point method. Advances particle
* from Y to YOUT, with a step of length
* TSTEP divided into NSTEP bits.
* YOUT contains positions and velocities
*
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 Y(8), YOUT(8)
      REAL*8 YS(8), YN(8)
      REAL*8 A(8)
*
      H = TSTEP/DBLE(NSTEP)
      DO I = 1, 3
        A(I) = Y(I+3)
      END DO
      CALL ROCACC(Y,Y(4),Q,A(4))
      CALL CHIACC(Y,Y(7),Q,A(7))
      DO I = 1, 8
        YS(I) = Y(I)
        YN(I) = Y(I) + H*A(I)
      END DO
      DO I = 1, 3
        A(I) = YN(I+3)
      END DO
      CALL ROCACC(YN,YN(4),Q,A(4))
      CALL CHIACC(YN,YN(7),Q,A(7))
      H2 = 2.*H
      DO N = 2, NSTEP
        DO I = 1, 8
          SWAP = YS(I) + H2*A(I)
          YS(I) = YN(I)
          YN(I) = SWAP
        END DO
        DO I = 1, 3
          A(I) = YN(I+3)
        END DO
        CALL ROCACC(YN,YN(4),Q,A(4))
        CALL CHIACC(YN,YN(7),Q,A(7))
      END DO
      DO I = 1, 8
        YOUT(I) = 0.5*(YS(I)+YN(I)+H*A(I))
      END DO
      RETURN
      END

      SUBROUTINE CHIACC(X,NC,Q,DNC)
*
      IMPLICIT REAL*8 (A-H,O-Z)
      REAL*8 X(2),NC(2),Q,DNC(2),MU
*
      MU = Q/(1.D0+Q)
      DNC(2) = -2.D0*NC(1)*NC(2)
      OMEG = (1.D0-MU)/(X(1)**2+X(2)**2)**1.5
     &             +MU/((X(1)-1.D0)**2+X(2)**2)**1.5
      DNC(1) = NC(2)-OMEG-NC(1)**2
      RETURN
      END

      SUBROUTINE GSEXTR1(IEST,TEST,YEST,YZ,DY,NUSE)
*
* Rational function extrapolation
* YEST(6) contains positions then velocities.
*
      IMPLICIT REAL*8 (A-H,O-Z)
      PARAMETER (IMAX=11,NCOL=7)
      REAL*8 YEST(8), YZ(8), DY(8)
      REAL*8 T(IMAX), D(8,NCOL), FX(NCOL)
*
      T(IEST) = TEST
      IF(IEST.EQ.1) THEN
        DO J = 1, 8
          YZ(J)  = YEST(J)
          D(J,1) = YEST(J)
          DY(J)  = YEST(J)
        END DO
      ELSE
        M1 = MIN(IEST, NUSE)
        DO K = 1, M1-1
          FX(K+1) = T(IEST-K)/TEST
        END DO
        DO J = 1, 8
          YY = YEST(J)
          V  = D(J,1)
          C  = YY
          D(J,1) = YY
          DO K = 2, M1
            B1 = FX(K)*V
            B = B1 - C
            IF(B.NE.0.) THEN
              B = (C-V)/B
              DDY = C*B
              C = B1*B
            ELSE
              DDY = V
            END IF
            V = D(J,K)
            D(J,K) = DDY
            YY = YY + DDY
          END DO
          DY(J) = DDY
          YZ(J) = YY
        END DO
      END IF
      RETURN
      END

