      SUBROUTINE   MEM (METHOD,LEVEL,AIM,RMAX,DEF,ACC)
      CALL       MEMPRM(METHOD,LEVEL,AIM,RMAX,DEF,ACC,
     *                  C,TEST,CNEW,S,RNEW,SNEW,SUMF)
      RETURN
      END
 
      SUBROUTINE MEMPRM(METHOD,LEVEL,AIM,RMAX,DEF,ACC,
     *                  C,TEST,CNEW,S,RNEW,SNEW,SUMF)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*                                                                      *
*                  --------------------------------                    *
*                  "MEMSYS"  Maximum Entropy System                    *
*                  --------------------------------                    *
*                                                                      *
*               John Skilling                                          *
*               Dept. Applied Maths. and Theoretical Physics           *
*               Silver Street, Cambridge, England                      *
*                                                                      *
*         (C)   Copyright 1985                                         *
*               Maximum Entropy Data Consultants Ltd                   *
*               33 North End, Meldreth, Royston, England               *
************************************************************************
*
* Purpose:
*   Perform one iterate of maximum entropy and update map
*
*   Enter with      <1> = map f  ,  <21> = data D
*
*   and optionally
*           <20> = default model ,  <19> = weights ,  <22> = accuracies
*
*   Exit with       <1> = new map .
*
************************************************************************
*
*   Entropy    S  :=  - 1 - total( <1> * log <1>/e<20> ) / total<20>
*
*                                         2
*   Constraint C  :=  sum (Opus<1> - <21>)  * <22> / 2
*
************************************************************************
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*
*    METHOD  I     I      -         Flags for procedural method
*    LEVEL   I     I      -         Flags for level of diagnostics
*    AIM     R     I      -         Ultimate target of constraint
*    RMAX    R     I      -         Dimensionless distance limit
*    DEF     R     I      -         Default level (if constant)
*    ACC     R     I      -         Accuracy      (if constant)
*
*    C       R       O    -         Constraint value on entry
*    TEST    R       O    -         Gradient direction misfit on entry
*    CNEW    R       O    -         Predicted constraint value on exit
*    S       R       O    -         Entropy value on entry
*    RNEW    R       O    -         Dimensionless distance moved
*    SNEW    R       O    -         Predicted entropy value on exit
*    SUMF    R       O    -         Map total on exit
*
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*
*    NJ          MECOMP     I    Number of buffers in map-space area
*    MJ          MECOMP     I    Map-space buffer length in words
*    NK          MECOMP     I    Number of buffers in data-space area
*    MK          MECOMP     I    Data-space buffer length in words
*
*    KA          MECOMP     I    Allocation pointers for areas
*    KB          MECOMP     I    Base addresses in core for buffers
*    KC          MECOMP      (O) Core addresses for current buffers
*    KD          MECOMP      (O) Disc or upper core addresses of buffers
*
*    IOUT        MECOMP     I    Fortran output stream for diagnostics
*    IIN         MECOMP     I    Fortran stream for user input (if any)
*
*    L0          MECOMP       O  Flag for progress diagnostics
*    L1          MECOMP       O  Flag for numerical diagnostics
*    M0          MECOMP       O  Flag for number of search directions
*    M10         MECOMP       O  Flag for type of entropy
*    M11         MECOMP       O  Flag for constant total(map)
*    M20         MECOMP       O  Flag for constant or variable errors
*    M21         MECOMP       O  Flag for treatment of outliers
*    M3          MECOMP       O  Flag for weighting of map cells
*    M4          MECOMP       O  Flag for constant total(map)
*
*    ST          MECOMS     I O  Storage vector
*
*    PR          MECOMC      (O) Read/write diagnostic flags
*
*
* External calls:
*            Map-Data
*    MEMOP     f d     Call user transform routine OPUS
*    MEMTR     f d     Call user transpose routine TROPUS
*
*    MEMA        d     Constraint value and gradient
*    MEMB      f       Entropy and contravariant gradients
*    MEMC        d     Scalars for first three search directions
*    MEMCC       d     Start calculating fourth search direction
*    MEMD      f       Fourth search direction
*    MEME        d     Scalars for fourth search direction
*    MEMEE       d     Start calculating fifth search direction
*    MEMP      f       Fifth search direction
*    MEMQ        d     Scalars for fifth search direction
*    MEMQQ       d     Revised fourth search direction transform
*    MEMR      f       Revised fourth search direction
*
*    MEML       -      "Control" - map increment coefficients
*
*    MEMF      f       New map
*    MEMFF     f       New map
*
*
* Key:
*
*    METHOD = 10000*M4 + 1000*M3 + 100*M2 + 10*M1 + M0
*
*        M0 = 0 is full recursive generation of search directions
*        M0 = 3 is three search directions (plus the current map)
*        M0 = 4 is four  search directions (plus the current map)
*
*        M1 = 0 is constant default value DEF
*        M1 = 1 is default given by exp(total(p log f))
*        M1 = 3 is variable default levels in <20>
*        M1 = 5 is "M1=0" with total(f) held fixed
*        M1 = 6 is "M1=1" with total(f) held fixed
*        M1 = 8 is "M1=3" with total(f) held fixed
*
*        M2 = 0 is chisquared with variable accuracies in <22>
*        M2 = 1 is chisquared with constant accuracy ACC
*        M2 = 5 is "M2=0" but with outliers suppressed
*        M2 = 6 is "M2=1" but with outliers suppressed
*
*        M3 = 0 is all cells weighted equally
*        M3 = 1 is variable weights in <19>
*
*   LEVEL = 10*L1 + L0
*
*        L0 = 0 gives no progress diagnostics
*        L0 = 1 gives names of main routines
*        L0 = 2 gives additionally read/write flowchart
*        L0 = 3 additionally calls UDIAG after each main routine
*
*        L1 = 0 gives no numerical diagnostics
*        L1 = 1 gives S,TEST,C and SNEW,RNEW,CNEW
*        L1 = 2 gives additionally SNEW,RNEW,CNEW in each minor loop
*        L1 = 3 gives additionally technical diagnostics from "control"
*        L1 = 4 gives additionally the subspace scalars
*
*
* Flowchart:            Map              Data         Scalars
*   MemEX       R...................  ..W.......
*   MemA        ....................  RRR....W..   C
*   Tropus      .....W..............  .......R..
*   MemB        RWW..R............RR  ..........   SDD11,12,13,22,23,33
*                                                  S, SDEF, SUMF, TEST
*   Opus        .R..................  ...W......
*   Opus        ..R.................  ....W.....
*   MemC        ....................  .RRRR.....   CDD11,12,13,22,23,33
*   Control                                        W1,2,3
*   MemCC       ....................  .RRRR..W..
*   Tropus      .....W..............  .......R..
*   MemD        RRRW.R............R.  ..........   SDD14,24,34,44
*   Opus        ...R................  .....W....
*   MemE        ....................  .RRRRR....   CDD14,24,34,44
*   Control                                        W1,2,3,4, CNEW
* EXIT to MemF ?
*   MemEE       ....................  .RRRRR.W..
*   Tropus      .....W..............  .......R..
*   MemP        RRRRWR............R.  ..........   SDD15,25,35,45,55
*   Opus        ....R...............  ......W...
*   MemQ        ....................  .RRRRRR...   CDD15,25,35,45,55
* REPEAT
*     Control                                      W1,2,3,4,5, CNEW
*   EXIT to MemFF?
*     MemQQ     ....................  ..RRRRRW..
*               ....................  .....W.R..
*               ....................  .R...R.W..
*     MemR      RRRRRW............R.  ..........   SDD14,24,34,44
*               ...W.R..............  ..........
*     Tropus    .....W..............  .......R..
*     MemP      RRRRWR............R.  ..........   SDD15,25,35,45,55
*     Opus      ....R...............  ......W...
*     MemQ      ....................  .RRRRRR...   CDD15,25,35,45,55
* UNTIL FALSE                                      CDD14,24,34,44
*
*   MemF        RRRR.W............R.  ..........
*               W....R..............  ..........   SUMF
* RETURN
*
*   MemFF       RRRRRW............R.  ..........
*               W....R..............  ..........   SUMF
* RETURN
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*                    5 Dec 1985     Correction to MEML22
*                    9 Dec 1985     Overflow risk reduced in MEML3
*                   10 Dec 1985     "ENTRY" removed for IBM3081 linker
*                   10 Dec 1985     UDIAG input different for IBM3081
*                   10 Dec 1985     MEML2 format altered for IBM3081
*                   10 Dec 1985     MEML8 format altered for IBM3081
*
* Notes:
*    (1)  MEM is the simpler call, but MEMPRM is provided for users who
*         wish to access the main diagnostic variables.
*    (2)  In the definition of S, "total" means "sum weighted by <19>"
*    (3)  Areas <6> and <28> are workspace
*
*-----------------------------------------------------------------------
      REAL SDD(6,6),CDD(6,6),W(6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      CALL MEINIT
* Method switches
      M0=METHOD
      M1=M0/10
      M2=M1/10
      M3=M2/10
      M4=M3/10
      M0=M0-10*M1
      M1=M1-10*M2
      M2=M2-10*M3
      M3=M3-10*M4
      IF(   (M0.NE.0 .AND. M0.NE.3 .AND. M0.NE.4)
     * .OR. (M1.NE.0 .AND. M1.NE.1 .AND. M1.NE.3 .AND.
     *       M1.NE.5 .AND. M1.NE.6 .AND. M1.NE.8)
     * .OR. (M2.NE.0 .AND. M2.NE.1 .AND. M2.NE.5 .AND. M2.NE.6)
     * .OR. (M3.NE.0 .AND. M3.NE.1)
     * .OR. (M4.NE.0 .AND. M4.NE.1) ) THEN
        WRITE(IOUT,91) METHOD
        STOP
      ENDIF
      M11=M1/5
      M10=M1-M11*5
      M21=M2/5
      M20=M2-M21*5
* Diagnostic switches
      L0=LEVEL
      L1=L0/10
      L0=L0-10*L1
      IF(   (L0.NE.0 .AND. L0.NE.1 .AND. L0.NE.2 .AND. L0.NE.3)
     * .OR. (L1.NE.0 .AND. L1.NE.1 .AND. L1.NE.2 .AND. L1.NE.3
     *               .AND. L1.NE.4) ) THEN
        WRITE(IOUT,92) LEVEL
        STOP
      ENDIF
      IF(L0.GE.1) WRITE(IOUT,5) M4,M3,M2,M1,M0
*
* Maximum entropy
      CALL MEMEX
      CALL MEMA(ACC, C)
              IF(L1.GE.1) WRITE(IOUT,6) C
      CALL MEMTR(28,6)
      CALL MEMB(DEF, SDD,S,SDEF,TEST)
              SUMF=SDD(1,1)
              IF(L1.GE.1) WRITE(IOUT,7) S,TEST
      CALL MEMOP(2,24)
      CALL MEMOP(3,25)
      CALL MEMC(ACC, CDD)
      CALL MEML(3,SDD,CDD , S,C,AIM,RMAX,SDEF , SNEW,CNEW,RNEW , W)
              SPEED0=RNEW/2.
              IF(L1.GE.2) WRITE(IOUT,8) SNEW,RNEW,CNEW
      CALL MEMCC(ACC,W)
      CALL MEMTR(28,6)
      CALL MEMD(SDD)
      CALL MEMOP(4,26)
      CALL MEME(ACC, CDD)
      CALL MEML(4,SDD,CDD , S,C,AIM,RMAX,SDEF , SNEW,CNEW,RNEW , W)
              SPEED=RNEW/3.
              IF(L1.GE.2) WRITE(IOUT,8) SNEW,RNEW,CNEW
              IF(SPEED.LT.SPEED0 .OR. M0.EQ.3) THEN
                IF(L1.EQ.1) WRITE(IOUT,8) SNEW,RNEW,CNEW
                CALL MEMF(W,SUMF)
                RETURN
              ENDIF
              SPEED0=SPEED
      CALL MEMEE(ACC,W)
      CALL MEMTR(28,6)
      CALL MEMP(SDD)
      CALL MEMOP(5,27)
      CALL MEMQ(ACC, CDD)
      NDIR=3
    9 CONTINUE
        CALL MEML(5,SDD,CDD , S,C,AIM,RMAX,SDEF , SNEW,CNEW,RNEW , W)
                NDIR=NDIR+1
                SPEED=RNEW/FLOAT(NDIR)
                IF(L1.GE.2) WRITE(IOUT,8) SNEW,RNEW,CNEW
                IF(SPEED.LT.SPEED0 .OR. M0.EQ.4) THEN
                  IF(L1.EQ.1) WRITE(IOUT,8) SNEW,RNEW,CNEW
                  CALL MEMFF(W,SUMF)
                  RETURN
                ENDIF
                SPEED0=SPEED
        CALL MEMQQ(ACC,W)
        CALL MEMR(W,SDD)
        CALL MEMTR(28,6)
        CALL MEMP(SDD)
        CALL MEMOP(5,27)
        CALL MEMQ(ACC,CDD)
        CDD(1,4)=SDD(1,5)
        CDD(2,4)=SDD(2,5)
        CDD(3,4)=SDD(3,5)
        CDD(4,4)=SDD(4,5)
      GOTO 9
*
    5 FORMAT(' MEM December 1985  METHOD = ',5I1)
    6 FORMAT(50X,'  C    === ',1PE12.5)
    7 FORMAT('      S    === ',1PE12.5,'    TEST === ',0PF8.5)
    8 FORMAT('      SNEW === ',1PE12.5,'    dist === ',0PF8.5,
     *                                 '    CNEW === ',1PE12.5)
   91 FORMAT('  METHOD =',I11,'   not implemented.       STOP')
   92 FORMAT('   LEVEL =',I11,'   not implemented.       STOP')
      END
 
 
      SUBROUTINE MEMA (ACC , C)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Constraint value and covariant gradient (data space)
*
*                  C  := ( <23> - <21> )**2 * <22> /2
*
*                <28> := ( <23> - <21> ) * <22>
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ACC     R    (I)     -         Accuracy (alternative to <22>)
*    C       R       O    -         Constraint value
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M20         MECOMP     I    Accuracy switch (0 =<22>, 1 =ACC)
*    M21         MECOMP     I    Outlier switch (0=normal, 1=suppressed)
*
* External calls:
*    MEMA0       Optional vector arithmetic for rms residual
*    MEMA1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) Constraint is chisquared, possibly corrected for outliers
*
*-----------------------------------------------------------------------
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
* Set X = twice (6 * RMSresidual )**2       (or infinity if not needed)
      IF(M21.EQ.0) THEN
        X=1.E35
      ELSE
        DATA=FLOAT(NK)*FLOAT(MK)
        X=0.
        DO 1 I=1,NK
          CALL UREAD(21)
          CALL UREAD(23)
            IF(M20.EQ.0) CALL UREAD(22)
          CALL MEMA0
     *      (MK,ST(KC(21)),ST(KC(23)),M20,ST(KC(22)),ACC, X)
    1   CONTINUE
        CALL UINIT
        X=36.*X/(36.+DATA)
      ENDIF
*
      C=0.
      DO 2 I=1,NK
        CALL UREAD(21)
        CALL UREAD(23)
          IF(M20.EQ.0) CALL UREAD(22)
        CALL MEMA1
     *    (MK,ST(KC(21)),ST(KC(23)),M20,ST(KC(22)),ACC,X, ST(KC(28)),C)
        CALL UWRITE(28)
    2 CONTINUE
      C=.5*C
      CALL UINIT
      RETURN
    5 FORMAT('   MemA')
      END
 
      SUBROUTINE MEMA0(MK,D,F,M2,E,ACC, X)
      REAL D(*),F(*),E(*)
      Z1=0.
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        R=F(I)-D(I)
        Z1=Z1+A*R*R
    1 CONTINUE
      X=X+Z1
      RETURN
      END
 
      SUBROUTINE MEMA1(MK,D,F,M2,E,ACC,X, GC,C)
      REAL D(*),F(*),E(*), GC(*)
      Z1=0.
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        R=F(I)-D(I)
        GC(I)=A*R
        Y=GC(I)*R
        IF(Y.GT.X) THEN
          GC(I)=GC(I)*SQRT(X/Y)
          Y=X
        ENDIF
        Z1=Z1+Y
    1 CONTINUE
      C=C+Z1
      RETURN
      END
 
 
      SUBROUTINE MEMB(DEF, SDD,S,SDEF,TEST)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Contravariant second and third search directions
*   ( f grad S  and   f grad C )  and initial entropy scalars
*
*   Entropy  S  :=  - 1 - total( <1> * log <1>/e<20> ) / total<20>
*
*         SDEF  :=  total<20>
*
*           <2> := ( log <20> - log <1> ) * metric
*
*           <3> := <6> * metric
*
*         SDD   := <i> * <j> / metric     for     ij = 11,12,13,22,23,33
*            ij
*
*         total<i> = sum( <i> * <19> )     ,    metric = <1> / <19>
*
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    DEF     R    (I)     -         Default level (if needed)
*    SDD     R       O   6,6        Entropy curvature scalars
*    S       R       O    -         Entropy value (dimensionless)
*    SDEF    R       O    -         Total model
*    TEST    R       O    -         1 - cos(angle(grad S, grad C))
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NJ,MJ       MECOMP     I    Sizes (map space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M10         MECOMP     I    Model switch (0=DEF, 1=free, 3=<20>)
*    M11         MECOMP     I    Total(map) switch (0 =off, 1 =on)
*    M3          MECOMP     I    Weight switch (0 for none, 1 =<19>)
*
* External calls:
*    MEMB0       Optional vector arithmetic to determine default level
*    MEMB1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1)  S is calculated dimensionless, but gradients and curvatures
*         are dimensional (multiplied by SDEF). "Control" corrects this.
*    (2)  The factor 0.99999 guards against accidentally making TEST
*         imaginary when projecting onto fixed total(map)
*
*-----------------------------------------------------------------------
      REAL SDD(6,6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
* Set DEFA = default value
      IF(M10.NE.1) THEN
        DEFA=DEF
      ELSE
        SUMF=0.
        SUMFLF=0.
        DO 1 I=1,NJ
          CALL UREAD(1)
          IF(M3.EQ.1) CALL UREAD(19)
          CALL MEMB0(MJ,ST(KC(1)),M3,ST(KC(19)),SUMF,SUMFLF)
    1   CONTINUE
        CALL UINIT
        DEFA=EXP(SUMFLF/SUMF)
      ENDIF
*
      SDEF=0.0
      SDD(1,1)=0.
      SDD(1,2)=0.
      SDD(1,3)=0.
      SDD(2,2)=0.
      SDD(2,3)=0.
      SDD(3,3)=0.
      DO 2 I=1,NJ
        CALL UREAD(1)
        CALL UREAD(6)
          IF(M10.EQ.3) CALL UREAD(20)
          IF(M3 .EQ.1) CALL UREAD(19)
        CALL MEMB1
     *   (MJ,ST(KC(1)),ST(KC(6)),M3,ST(KC(19)),M10,ST(KC(20)),DEFA,
     *    ST(KC(2)),ST(KC(3)),SDEF,
     *    SDD(1,1),SDD(1,2),SDD(1,3),SDD(2,2),SDD(2,3),SDD(3,3))
        CALL UWRITE(2)
        CALL UWRITE(3)
    2 CONTINUE
      CALL UINIT
*
      S=(SDD(1,1)+SDD(1,2))/SDEF-1.0
      IF(M11.EQ.0) THEN
        SS=SDD(2,2)
        SC=SDD(2,3)
        CC=SDD(3,3)
      ELSE
        SS=SDD(2,2)-0.99999*SDD(1,2)*SDD(1,2)/SDD(1,1)
        SC=SDD(2,3)-0.99999*SDD(1,2)*SDD(1,3)/SDD(1,1)
        CC=SDD(3,3)-0.99999*SDD(1,3)*SDD(1,3)/SDD(1,1)
      ENDIF
      TEST=1.-SC/(SQRT(SS+1.E-35)*SQRT(CC+1.E-35))
*
      RETURN
    5 FORMAT('   MemB')
      END
 
      SUBROUTINE MEMB0(MJ,F,M3,WTFILE,SUMF,SUMFLF)
      REAL F(*),WTFILE(*)
      Z1=0.
      Z2=0.
      DO 1 I=1,MJ
        IF(M3.EQ.1) THEN
          WEIGHT=WTFILE(I)
        ELSE
          WEIGHT=1.
        ENDIF
        Z1=Z1+WEIGHT*F(I)
        IF(F(I).GT.0.) Z2=Z2+WEIGHT*F(I)*LOG(F(I))
    1 CONTINUE
      SUMF  =SUMF  +Z1
      SUMFLF=SUMFLF+Z2
      RETURN
      END
 
      SUBROUTINE MEMB1(MJ,F,GC,M3,WTFILE,M1,AMODEL,DEF,
     *                 FGS,FGC,SDEF,S11,S12,S13,S22,S23,S33)
      REAL F(*),GC(*),WTFILE(*),AMODEL(*), FGS(*),FGC(*)
      Z0=0.
      Z1=0.
      Z2=0.
      Z3=0.
      Z4=0.
      Z5=0.
      Z6=0.
      DO 1 I=1,MJ
        IF(M3.EQ.1) THEN
          WEIGHT=WTFILE(I)
        ELSE
          WEIGHT=1.
        ENDIF
        IF(M1.GE.3) THEN
          DEFLT=AMODEL(I)
        ELSE
          DEFLT=DEF
        ENDIF
        IF(F(I).GT.0. .AND. DEFLT.GT.0.) THEN
          GS=WEIGHT*LOG(DEFLT/F(I))
        ELSE
          GS=0.
        ENDIF
        AMET=F(I)/WEIGHT
        X=AMET*GS
        Y=AMET*GC(I)
        Z0=Z0+WEIGHT*DEFLT
        Z1=Z1+WEIGHT*F(I)
        Z2=Z2+WEIGHT*X
        Z3=Z3+WEIGHT*Y
        Z4=Z4+X*GS
        Z5=Z5+Y*GS
        Z6=Z6+Y*GC(I)
        FGS(I)=X
        FGC(I)=Y
    1 CONTINUE
      SDEF=SDEF+Z0
      S11=S11+Z1
      S12=S12+Z2
      S13=S13+Z3
      S22=S22+Z4
      S23=S23+Z5
      S33=S33+Z6
      RETURN
      END
 
 
      SUBROUTINE MEMC (ACC , CDD)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Constraint scalars of first three search directions
*
*   CDD   := <22+i> * <22> * (22+j>      for     ij = 11,12,13,22,23,33
*      ij
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ACC     R    (I)     -         Accuracy (alternative to <22>)
*    CDD     R       O   6,6        Constraint curvature scalars
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I    Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M20         MECOMP     I    Accuracy switch (0 =<22>, 1 =ACC)
*    M21         MECOMP     I    Outlier switch (0=normal, 1=suppressed)
*
* External calls:
*    MEMC1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL CDD(6,6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      CDD(1,1)=0.
      CDD(1,2)=0.
      CDD(1,3)=0.
      CDD(2,2)=0.
      CDD(2,3)=0.
      CDD(3,3)=0.
      DO 1 I=1,NK
        CALL UREAD(23)
        CALL UREAD(24)
        CALL UREAD(25)
          IF(M20.EQ.0) CALL UREAD(22)
        CALL MEMC1
     *   (MK,ST(KC(23)),ST(KC(24)),ST(KC(25)),M20,ST(KC(22)),ACC,
     *    CDD(1,1),CDD(1,2),CDD(1,3),CDD(2,2),CDD(2,3),CDD(3,3))
    1 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemC')
      END
 
      SUBROUTINE MEMC1(MK,A1,A2,A3,M2,E,ACC, C11,C12,C13,C22,C23,C33)
      REAL A1(*),A2(*),A3(*),E(*)
      Z1=0.
      Z2=0.
      Z3=0.
      Z4=0.
      Z5=0.
      Z6=0.
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        Z1=Z1+A1(I)*A1(I)*A
        Z2=Z2+A1(I)*A2(I)*A
        Z3=Z3+A1(I)*A3(I)*A
        Z4=Z4+A2(I)*A2(I)*A
        Z5=Z5+A2(I)*A3(I)*A
        Z6=Z6+A3(I)*A3(I)*A
    1 CONTINUE
      C11=C11+Z1
      C12=C12+Z2
      C13=C13+Z3
      C22=C22+Z4
      C23=C23+Z5
      C33=C33+Z6
      RETURN
      END
 
 
      SUBROUTINE MEMCC(ACC,W)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Start generating fourth search direction
*
*           <28> := ( W1*<23> + W2*<24> + W3*<25> ) * <22>
*
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ACC     R    (I)     -         Accuracy (alternative to <22>)
*    W       R     I      3         Map increment coefficients
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M20         MECOMP     I    Accuracy switch (0 =<22>, 1 =ACC)
*    M21         MECOMP     I    Outlier switch (0=normal, 1=suppressed)
*
* External calls:
*    MEMCC1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL W(*)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      DO 1 I=1,NK
        CALL UREAD(23)
        CALL UREAD(24)
        CALL UREAD(25)
          IF(M20.EQ.0) CALL UREAD(22)
        CALL MEMCC1(MK,ST(KC(23)),ST(KC(24)),ST(KC(25)),
     *              M20,ST(KC(22)),ACC,W(1),W(2),W(3), ST(KC(28)))
        CALL UWRITE(28)
    1 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemCC')
      END
 
      SUBROUTINE MEMCC1(MK,A1,A2,A3,M2,E,ACC,W1,W2,W3, B)
      REAL A1(*),A2(*),A3(*),E(*), B(*)
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        B(I)=A*(W1*A1(I)+W2*A2(I)+W3*A3(I))
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEMD(SDD)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Contravariant fourth search direction and its entropy scalars
*
*          <4> := <6> * metric    ,    metric = <1> / <19>
*
*        SDD   := <6> * <i>      for     i = 1,2,3,4
*           i4
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    SDD     R       O   6,6        Entropy curvature scalars
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NJ,MJ       MECOMP     I    Sizes (map space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M3          MECOMP     I    Weight switch (0 for none, 1 =<19>)
*
* External calls:
*    MEMD1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL SDD(6,6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      SDD(1,4)=0.
      SDD(2,4)=0.
      SDD(3,4)=0.
      SDD(4,4)=0.
      DO 1 I=1,NJ
        CALL UREAD(1)
        CALL UREAD(2)
        CALL UREAD(3)
        CALL UREAD(6)
          IF(M3.EQ.1) CALL UREAD(19)
        CALL MEMD1(MJ,ST(KC(1)),ST(KC(2)),ST(KC(3)),ST(KC(6)),
     *    M3,ST(KC(19)), ST(KC(4)),SDD(1,4),SDD(2,4),SDD(3,4),SDD(4,4))
        CALL UWRITE(4)
    1 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemD')
      END
 
      SUBROUTINE MEMD1(MJ,F,A2,A3,B,M3,WTFILE, A4,S14,S24,S34,S44)
      REAL F(*),A2(*),A3(*),B(*),WTFILE(*), A4(*)
      Z1=0.
      Z2=0.
      Z3=0.
      Z4=0.
      DO 1 I=1,MJ
        IF(M3.EQ.1) THEN
          WEIGHT=WTFILE(I)
        ELSE
          WEIGHT=1.
        ENDIF
        AMET=F(I)/WEIGHT
        X=AMET*B(I)
        Z1=Z1+ F(I)*B(I)
        Z2=Z2+A2(I)*B(I)
        Z3=Z3+A3(I)*B(I)
        Z4=Z4+    X*B(I)
        A4(I)=X
    1 CONTINUE
      S14=S14+Z1
      S24=S24+Z2
      S34=S34+Z3
      S44=S44+Z4
      RETURN
      END
 
 
      SUBROUTINE MEME (ACC , CDD)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Constraint scalars of fourth search direction
*
*   CDD   := <22+i> * <22> * (22+j>      for     ij = 14,24,34,44
*      ij
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ACC     R    (I)     -         Accuracy (alternative to <22>)
*    CDD     R       O   6,6        Constraint curvature scalars
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I    Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M20         MECOMP     I    Accuracy switch (0 =<22>, 1 =ACC)
*    M21         MECOMP     I    Outlier switch (0=normal, 1=suppressed)
*
* External calls:
*    MEME1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL CDD(6,6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      CDD(1,4)=0.
      CDD(2,4)=0.
      CDD(3,4)=0.
      CDD(4,4)=0.
        DO 1 I=1,NK
        CALL UREAD(23)
        CALL UREAD(24)
        CALL UREAD(25)
        CALL UREAD(26)
          IF(M20.EQ.0) CALL UREAD(22)
        CALL MEME1
     *    (MK,ST(KC(23)),ST(KC(24)),ST(KC(25)),ST(KC(26)),
     *     M20,ST(KC(22)),ACC, CDD(1,4),CDD(2,4),CDD(3,4),CDD(4,4))
    1 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemE')
      END
 
      SUBROUTINE MEME1(MK,A1,A2,A3,A4,M2,E,ACC, C14,C24,C34,C44)
      REAL A1(*),A2(*),A3(*),A4(*),E(*)
      Z1=0.
      Z2=0.
      Z3=0.
      Z4=0.
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        X=A4(I)*A
        Z1=Z1+A1(I)*X
        Z2=Z2+A2(I)*X
        Z3=Z3+A3(I)*X
        Z4=Z4+A4(I)*X
    1 CONTINUE
      C14=C14+Z1
      C24=C24+Z2
      C34=C34+Z3
      C44=C44+Z4
      RETURN
      END
 
 
      SUBROUTINE MEMEE(ACC,W)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Start generating fifth search direction
*
*       <28> := ( W1*<23> + W2*<24> + W3*<25> + W4*<26> ) * <22>
*
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ACC     R    (I)     -         Accuracy (alternative to <22>)
*    W       R     I      4         Map increment coefficients
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M20         MECOMP     I    Accuracy switch (0 =<22>, 1 =ACC)
*    M21         MECOMP     I    Outlier switch (0=normal, 1=suppressed)
*
* External calls:
*    MEMEE1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL W(*)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      DO 1 I=1,NK
        CALL UREAD(23)
        CALL UREAD(24)
        CALL UREAD(25)
        CALL UREAD(26)
          IF(M20.EQ.0) CALL UREAD(22)
        CALL MEMEE1(MK,ST(KC(23)),ST(KC(24)),ST(KC(25)),ST(KC(26)),
     *              M20,ST(KC(22)),ACC ,ST(KC(28)),W(1),W(2),W(3),W(4))
        CALL UWRITE(28)
    1   CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemEE')
      END
 
      SUBROUTINE MEMEE1(MK,A1,A2,A3,A4,M2,E,ACC, B,W1,W2,W3,W4)
      REAL A1(*),A2(*),A3(*),A4(*),E(*), B(*)
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        B(I)=A*(W1*A1(I)+W2*A2(I)+W3*A3(I)+W4*A4(I))
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEMP(SDD)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Contravariant fifth search direction and its entropy scalars
*
*          <5> := <6> * metric    ,    metric = <1> / <19>
*
*        SDD   := <6> * <i>      for     i = 1,2,3,4,5
*           i5
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    SDD     R       O   6,6        Entropy curvature scalars
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NJ,MJ       MECOMP     I    Sizes (map space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M3          MECOMP     I    Weight switch (0 for none, 1 =<19>)
*
* External calls:
*    MEMP1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL SDD(6,6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      SDD(1,5)=0.
      SDD(2,5)=0.
      SDD(3,5)=0.
      SDD(4,5)=0.
      SDD(5,5)=0.
      DO 1  I=1,NJ
        CALL UREAD(1)
        CALL UREAD(2)
        CALL UREAD(3)
        CALL UREAD(4)
        CALL UREAD(6)
          IF(M3.EQ.1) CALL UREAD(19)
        CALL MEMP1(MJ,ST(KC(1)),ST(KC(2)),ST(KC(3)),
     *                ST(KC(4)),ST(KC(6)),M3,ST(KC(19)),
     *    ST(KC(5)),SDD(1,5),SDD(2,5),SDD(3,5),SDD(4,5),SDD(5,5))
        CALL UWRITE(5)
    1 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemP')
      END
 
      SUBROUTINE MEMP1
     *  (MJ,F,A2,A3,A4,B,M3,WTFILE, A5,S15,S25,S35,S45,S55)
      REAL F(*),A2(*),A3(*),A4(*),B(*),WTFILE(*), A5(*)
      Z1=0.
      Z2=0.
      Z3=0.
      Z4=0.
      Z5=0.
      DO 1 I=1,MJ
        IF(M3.EQ.1) THEN
          WEIGHT=WTFILE(I)
        ELSE
          WEIGHT=1.
        ENDIF
        AMET=F(I)/WEIGHT
        X=AMET*B(I)
        Z1=Z1+ F(I)*B(I)
        Z2=Z2+A2(I)*B(I)
        Z3=Z3+A3(I)*B(I)
        Z4=Z4+A4(I)*B(I)
        Z5=Z5+    X*B(I)
        A5(I)=X
    1 CONTINUE
      S15=S15+Z1
      S25=S25+Z2
      S35=S35+Z3
      S45=S45+Z4
      S55=S55+Z5
      RETURN
      END
 
 
      SUBROUTINE MEMQ (ACC , CDD)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Constraint scalars of fifth search direction
*
*   CDD   := <22+i> * <22> * (22+j>      for     ij = 15,25,35,45,55
*      ij
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ACC     R    (I)     -         Accuracy (alternative to <22>)
*    CDD     R       O   6,6        Constraint curvature scalars
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I    Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M20         MECOMP     I    Accuracy switch (0 =<22>, 1 =ACC)
*    M21         MECOMP     I    Outlier switch (0=normal, 1=suppressed)
*
* External calls:
*    MEMQ1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL CDD(6,6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      CDD(1,5)=0.
      CDD(2,5)=0.
      CDD(3,5)=0.
      CDD(4,5)=0.
      CDD(5,5)=0.
      DO 1 I=1,NK
        CALL UREAD(23)
        CALL UREAD(24)
        CALL UREAD(25)
        CALL UREAD(26)
        CALL UREAD(27)
          IF(M20.EQ.0) CALL UREAD(22)
        CALL MEMQ1
     *  (MK,ST(KC(23)),ST(KC(24)),ST(KC(25)),ST(KC(26)),ST(KC(27)),
     *  M20,ST(KC(22)),ACC,CDD(1,5),CDD(2,5),CDD(3,5),CDD(4,5),CDD(5,5))
    1 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemQ')
      END
 
      SUBROUTINE MEMQ1(MK,A1,A2,A3,A4,A5,M2,E,ACC, C15,C25,C35,C45,C55)
      REAL A1(*),A2(*),A3(*),A4(*),A5(*),E(*)
      Z1=0.
      Z2=0.
      Z3=0.
      Z4=0.
      Z5=0.
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        X=A5(I)*A
        Z1=Z1+A1(I)*X
        Z2=Z2+A2(I)*X
        Z3=Z3+A3(I)*X
        Z4=Z4+A4(I)*X
        Z5=Z5+A5(I)*X
    1 CONTINUE
      C15=C15+Z1
      C25=C25+Z2
      C35=C35+Z3
      C45=C45+Z4
      C55=C55+Z5
      RETURN
      END
 
 
      SUBROUTINE MEMQQ(ACC,W)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Set data-space version of revised fourth search direction
*   and start generating revised fifth search direction
*
*       <26> :=  W1*<23> + W2*<24> + W3*<25> + W4*<26> + W5*<27>
*
*       <28> :=  <26> * <22>
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ACC     R    (I)     -         Accuracy (alternative to <22>)
*    W       R     I      5         Map increment coefficients
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M20         MECOMP     I    Accuracy switch (0 =<22>, 1 =ACC)
*    M21         MECOMP     I    Outlier switch (0=normal, 1=suppressed)
*
* External calls:
*    MEMQQ1       Vector arithmetic for new <26>
*    MEMK         Copy vector
*    MEMQQ2       Vector arithmetic for new <28>
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL W(*)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      DO 1 I=1,NK
        CALL UREAD(23)
        CALL UREAD(24)
        CALL UREAD(25)
        CALL UREAD(26)
        CALL UREAD(27)
        CALL MEMQQ1
     *    (MK,ST(KC(23)),ST(KC(24)),ST(KC(25)),ST(KC(26)),ST(KC(27)),
     *     W(1),W(2),W(3),W(4),W(5),  ST(KC(28)))
        CALL UWRITE(28)
    1 CONTINUE
      CALL UINIT
      CALL MEMK(28,26)
      DO 2 I=1,NK
        CALL UREAD(26)
          IF(M20.EQ.0) CALL UREAD(22)
        CALL MEMQQ2(MK,ST(KC(26)),M20,ST(KC(22)),ACC, ST(KC(28)))
        CALL UWRITE(28)
    2 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemQQ')
      END
 
      SUBROUTINE MEMQQ1(MK,A1,A2,A3,A4,A5,W1,W2,W3,W4,W5, B)
      REAL A1(*),A2(*),A3(*),A4(*),A5(*), B(*)
      DO 1 I=1,MK
        B(I)=W1*A1(I)+W2*A2(I)+W3*A3(I)+W4*A4(I)+W5*A5(I)
    1 CONTINUE
      RETURN
      END
 
      SUBROUTINE MEMQQ2(MK,A4,M2,E,ACC, B)
      REAL A4(*),E(*), B(*)
      DO 1 I=1,MK
        IF(M2.EQ.0) THEN
          A=E(I)
        ELSE
          A=ACC
        ENDIF
        B(I)=A*A4(I)
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEMR(W,SDD)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Revised fourth search direction and its entropy scalars
*
*          <4> := W1*<1> + W2*<2> + W3*<3> + W4*<4> +W5*<5>
*
*        SDD   := <4> * <i> / metric     for     i = 1,2,3,4
*           i4
*                             metric = <1> / <19>
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    W       R     I      5         Map increment coefficients
*    SDD     R       O   6,6        Entropy curvature scalars
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NJ,MJ       MECOMP     I    Sizes (map space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M3          MECOMP     I    Weight switch (0 for none, 1 =<19>)
*
* External calls:
*    MEMR1       Vector arithmetic
*    MEMJ        Copy vector
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      REAL W(*),SDD(6,6)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      SDD(1,4)=0.
      SDD(2,4)=0.
      SDD(3,4)=0.
      SDD(4,4)=0.
      DO 1 I=1,NJ
        CALL UREAD(1)
        CALL UREAD(2)
        CALL UREAD(3)
        CALL UREAD(4)
        CALL UREAD(5)
          IF(M3.EQ.1) CALL UREAD(19)
        CALL MEMR1(MJ,ST(KC(1)),ST(KC(2)),ST(KC(3)),ST(KC(4)),
     *    ST(KC(5)),M3,ST(KC(19)),W(1),W(2),W(3),W(4),W(5),
     *    ST(KC(6)),SDD(1,4),SDD(2,4),SDD(3,4),SDD(4,4))
        CALL UWRITE(6)
    1 CONTINUE
      CALL UINIT
      CALL MEMJ(6,4)
      RETURN
    5 FORMAT('   MemR')
      END
 
      SUBROUTINE MEMR1
     * (MJ,A1,A2,A3,A4,A5,M3,WTFILE,W1,W2,W3,W4,W5, A6,S14,S24,S34,S44)
      REAL A1(*),A2(*),A3(*),A4(*),A5(*),WTFILE(*), A6(*)
      Z1=0.
      Z2=0.
      Z3=0.
      Z4=0.
      DO 1 I=1,MJ
        X=W1*A1(I)+W2*A2(I)+W3*A3(I)+W4*A4(I)+W5*A5(I)
        IF(M3.EQ.1) THEN
          WEIGHT=WTFILE(I)
        ELSE
          WEIGHT=1.
        ENDIF
        IF(A1(I).GT.0.) THEN
          ZMET=WEIGHT/A1(I)
          Z1=Z1+X*WEIGHT
          Z2=Z2+X*A2(I)*ZMET
          Z3=Z3+X*A3(I)*ZMET
          Z4=Z4+X*X*ZMET
        ENDIF
        A6(I)=X
    1 CONTINUE
      S14=S14+Z1
      S24=S24+Z2
      S34=S34+Z3
      S44=S44+Z4
      RETURN
      END
 
 
      SUBROUTINE MEMF(W,SUMF)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   New map from four search directions (including old map)
*
*             <1>  :=  W1*<1) + W2*<2> + W3*<3> + W4*<4> ,
*                      possibly rescaled to constant total(map)
*
*             SUMF :=  total(map)
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    W       R     I      4         Map increment coefficients
*    SUMF    R       O    -         Map total
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I    Storage vector
*    NJ,MJ       MECOMP     I    Sizes (map space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M11         MECOMP     I    Total(map) switch (0 =off, 1 =on)
*    M3          MECOMP     I    Weight switch (0 for none, 1 =<19>)
*
* External calls:
*    MEMF1       Vector arithmetic
*    MEMF2       Copy (and rescale)
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) Map components f may not decrease by more than 80 percent,
*        or below 1.E-12
*
*-----------------------------------------------------------------------
      REAL W(*)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      SUM=0.
      DO 1 I=1,NJ
        CALL UREAD(1)
        CALL UREAD(2)
        CALL UREAD(3)
        CALL UREAD(4)
          IF(M3.EQ.1) CALL UREAD(19)
        CALL MEMF1(MJ,ST(KC(1)),ST(KC(2)),ST(KC(3)),ST(KC(4)),
     *    M3,ST(KC(19)),W(1),W(2),W(3),W(4), ST(KC(6)),SUM)
        CALL UWRITE(6)
    1 CONTINUE
      CALL UINIT
      IF(M11.EQ.0) THEN
        SUMF=SUM
        SCALE=1.
      ELSE
        SCALE=SUMF/SUM
      ENDIF
      DO 2 I=1,NJ
        CALL UREAD(6)
        CALL MEMF2(MJ,SCALE,ST(KC(6)), ST(KC(1)))
        CALL UWRITE(1)
    2 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemF')
      END
 
      SUBROUTINE MEMF1(MJ,A1,A2,A3,A4,M3,WTFILE,W1,W2,W3,W4, F,SUM)
      REAL A1(*),A2(*),A3(*),A4(*),WTFILE(*), F(*)
      Z1=0.
      DO 2 I=1,MJ
        IF(M3.EQ.1) THEN
          WEIGHT=WTFILE(I)
        ELSE
          WEIGHT=1.
        ENDIF
        X=A1(I)
        D=W1*A1(I)+W2*A2(I)+W3*A3(I)+W4*A4(I)
        IF(X.GT.0.) X=MAX(0.2*X,X+D,1.E-12)
        Z1=Z1+X*WEIGHT
        F(I)=X
    2 CONTINUE
      SUM=SUM+Z1
      RETURN
      END
 
      SUBROUTINE MEMF2(MJ,SCALE,OLD, F)
      REAL OLD(*),F(*)
      DO 1  I=1,MJ
        F(I)=OLD(I)*SCALE
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEMFF(W,SUMF)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   New map from five search directions (including old map)
*
*         <1>  :=  W1*<1) + W2*<2> + W3*<3> + W4*<4> + W5*<5> ,
*                  possibly rescaled to constant total(map)
*
*         SUMF :=  total(map)
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    W       R     I      5         Map increment coefficients
*    SUMF    R       O    -         Map total
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NJ,MJ       MECOMP     I    Sizes (map space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*    IOUT,L0     MECOMP     I    Diagnostics
*    M11         MECOMP     I    Total(map) switch (0 =off, 1 =on)
*    M3          MECOMP     I    Weight switch (0 for none, 1 =<19>)
*
* External calls:
*    MEMFF1      Vector arithmetic
*    MEMFF2      Copy (and rescale)
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) Map components f may not decrease by more than 80 percent,
*        or below 1.E-12
*
*-----------------------------------------------------------------------
      REAL W(*)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      IF(L0.GE.1) WRITE(IOUT,5)
      SUM=0.
      DO 1 I=1,NJ
        CALL UREAD(1)
        CALL UREAD(2)
        CALL UREAD(3)
        CALL UREAD(4)
        CALL UREAD(5)
          IF(M3.EQ.1) CALL UREAD(19)
        CALL MEMFF1
     *   (MJ,ST(KC(1)),ST(KC(2)),ST(KC(3)),ST(KC(4)),ST(KC(5)),
     *    M3,ST(KC(19)),W(1),W(2),W(3),W(4),W(5), ST(KC(6)),SUM)
        CALL UWRITE(6)
    1 CONTINUE
      CALL UINIT
      IF(M11.EQ.0) THEN
        SUMF=SUM
        SCALE=1.
      ELSE
        SCALE=SUMF/SUM
      ENDIF
      DO 2 I=1,NJ
        CALL UREAD(6)
        CALL MEMFF2(MJ,SCALE,ST(KC(6)), ST(KC(1)))
        CALL UWRITE(1)
    2 CONTINUE
      CALL UINIT
      RETURN
    5 FORMAT('   MemFF')
      END
 
      SUBROUTINE MEMFF1
     * (MJ,A1,A2,A3,A4,A5,M3,WTFILE,W1,W2,W3,W4,W5, F,SUM)
      REAL A1(*),A2(*),A3(*),A4(*),A5(*),WTFILE(*), F(*)
      Z1=0.
      DO 2 I=1,MJ
        IF(M3.EQ.1) THEN
          WEIGHT=WTFILE(I)
        ELSE
          WEIGHT=1.
        ENDIF
        X=A1(I)
        D=W1*A1(I)+W2*A2(I)+W3*A3(I)+W4*A4(I)+W5*A5(I)
        IF(X.GT.0.) X=MAX(0.2*X,X+D,1.E-12)
        Z1=Z1+X*WEIGHT
        F(I)=X
    2 CONTINUE
      SUM=SUM+Z1
      RETURN
      END
 
      SUBROUTINE MEMFF2(MJ,SCALE,OLD, F)
      REAL OLD(*),F(*)
      DO 1  I=1,MJ
        F(I)=OLD(I)*SCALE
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEMJ(M,N)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Copy map-space file.          <N> := <M>
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    M       I     I      -         Input  area number
*    N       I     I      -         Output area number
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NJ,MJ       MECOMP     I    Sizes (map space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*
* External calls:
*    MEMJ1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1)  No diagnostics produced
*
*-----------------------------------------------------------------------
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      DO 1  I=1,NJ
        CALL UREAD(M)
        CALL MEMJ1(MJ,ST(KC(M)),ST(KC(N)))
        CALL UWRITE(N)
    1 CONTINUE
      CALL UINIT
      RETURN
      END
 
      SUBROUTINE MEMJ1(M,A, B)
      REAL A(*),B(*)
      DO 1 I=1,M
        B(I)=A(I)
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEMK(M,N)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Copy data-space file.          <N> := <M>
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    M       I     I      -         Input  area number
*    N       I     I      -         Output area number
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I O  Storage vector
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA,KB,KC,KD MECOMP     I    Initialised pointers
*
* External calls:
*    MEMK1       Vector arithmetic
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1)  No diagnostics produced
*
*-----------------------------------------------------------------------
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      DO 1  I=1,NK
        CALL UREAD(M)
        CALL MEMK1(MK,ST(KC(M)),ST(KC(N)))
        CALL UWRITE(N)
    1 CONTINUE
      CALL UINIT
      RETURN
      END
 
      SUBROUTINE MEMK1(M,A, B)
      REAL A(*),B(*)
      DO 1 I=1,M
        B(I)=A(I)
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEMEX
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Model the ACTUAL behaviour of the experiment.
*
*                 <23> :=  Actual OPUS transform <1>
*
* LINEAR experiments are treated by a call to the differential response
*   routine OPUS.
*
* NONLINEAR experiments should transform an image on area 1 to mock data
*   on file 23, and may possibly need to adjust area 21, area 22,
*   and the required value of C.
*
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*     -
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*     -
*
* External calls:
*    MEMOP        Driver routine for OPUS
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) This implementation is for linear experiments.
*
*-----------------------------------------------------------------------
      CALL MEMOP(1,23)
      RETURN
      END
 
 
      SUBROUTINE MEMOP(K,L)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Call user's transform routine OPUS and re-initialise pointers
*
*                 <L> :=  OPUS <K>
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    K       I     I      -         Input  area number  (map space)
*    L       I     I      -         Output area number (data space)
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    IOUT,L0     MECOMP     I    Diagnostic stream and flag
*    PR          MECOMC       O  Read/Write flags
*
* External calls:
*    OPUS        User routine
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      CHARACTER*1 PR
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMC/ PR(40)
      IF(L0.GE.1) WRITE(IOUT,5)
      PR(K)='R'
      PR(L)='W'
* Protect OPUS against over-writing its arguments
      KK=K
      LL=L
      CALL OPUS(KK,LL)
* Re-initialise to protect the user
      CALL UINIT
      RETURN
    5 FORMAT('   Opus')
      END
 
 
      SUBROUTINE MEMTR(K,L)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Call user's transform routine TROPUS and re-initialise pointers
*
*                 <L> :=  TROPUS <K>
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    K       I     I      -         Input  area number (data space)
*    L       I     I      -         Output area number  (map space)
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    IOUT,L0     MECOMP     I    Diagnostic stream and flag
*    PR          MECOMC       O  Read/Write flags
*
* External calls:
*    TROPUS        User routine
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      CHARACTER*1 PR
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMC/ PR(40)
      IF(L0.GE.1) WRITE(IOUT,5)
      PR(K)='R'
      PR(L)='W'
* Protect TROPUS against over-writing its arguments
      KK=K
      LL=L
      CALL TROPUS(KK,LL)
* Re-initialise to protect the user
      CALL UINIT
      RETURN
    5 FORMAT('   Tropus')
      END
 
 
      SUBROUTINE MEML
     * (NSRCH,SDDA,CDDA ,SA,CA,AIMA,RMAXA,SDEFA ,SNEWA,CNEWA,RNEWA ,WA)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*                          " CONTROL "
*
*      Maximise quadratic model of entropy S (=objective function)
*      over the target value (or an approach to it) of the quadratic
*      model of the constraint C within a given distance limit.
*      Enter with models
*       S(w) = S + w.GSD - w.SDD.w/2  ,  C(w) = C + w.GCD + w.CDD.w/2
*      Exit with map increments w and diagnostics.
*
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of search directions
*    SDDA    R     I   NSRCH,NSRCH  Entropy metric
*    CDDA    R     I   NSRCH,NSRCH  Constraint matrix
*    SA      R     I      -         Current value of entropy
*    CA      R     I      -         Current value of constraint
*    AIMA    R     I      -         Ultimate target of constraint
*    RMAXA   R     I      -         Dimensionless distance limit
*    SDEFA   R     I      -         Scale factor for entropy
*    SNEWA   R       O    -         Predicted value of entropy
*    CNEWA   R       O    -         Predicted value of constraint
*    RNEWA   R       O    -         Dimensionless distance moved
*    WA      R       O    -         Map increment components
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    M11       MECOMP       I    Total(map) switch (0 =off, 1 =on)
*    IOUT      MECOMP       I    Output stream for diagnostics
*    L1        MECOMP       I    Level of numerical diagnostics
*
* External calls:
*    MEML1     Copy to internal (double precision) variables
*    MEML2       Optional diagnostics of input model "scalars"
*    MEML22      Optional orthogonalisation if total(map) held fixed
*    MEML3     Simultaneous diagonalisation of entropy and constraint
*    MEML4     Transform to eigenvector coords
*    MEML5     * Central Control * Find alpha and distance penalty *
*    MEML6     Transform back to original coordinates
*    MEML66      Optional coefficient of "map" if total(map) held fixed
*    MEML7     Copy to external (single precision) parameters
*    MEML8       Optional numerical diagnostics
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*   (1) There is some protection against a non-convex
*       constraint function, but this version of control
*       is not designed to handle such problems robustly
*   (2) Input parameters are preserved
*   (3) Only the upper triangles j>=i of SDDA and CDDA are read
*   (4) The map total SUMF and the gradients of S and C are extracted
*       from SDDA, which involves the following assumptions:
*      a) Scalar products in map-space have been calculated using
*         the entropy curvature -gradgradS
*      b) The first  search direction is f=map
*      c) The second search direction is contravariant gradS
*      d) The third  search direction is contravariant gradC
*   (5) If total(map) is held fixed, the central control routines use
*       only directions 2,3,... , orthogonalised to direction 1
*   (6) The distance limit is   w.SDD.w /SUMF  <  RMAX**2
*   (7) All matrices and vectors are stored in arrays dimensioned 6
*
* Notation:
*       S = entropy
*       C = constraint (e.g. chisquared)
*       Prefix G = gradient, as in GS and GC
*       Suffix D = Downstairs covariant index, as in GSD, GCD, SDD, CDD
*       Suffix U = Upstairs contravariant index, as in XU
*       Suffix A = single-precision parameter for external use only
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
* Call parameters
      REAL SDDA(6,6),CDDA(6,6),WA(6)
      REAL SA,CA,AIMA,RMAXA,SDEFA,SNEWA,CNEWA,RNEWA
* Internal variables
      REAL*8 GSD0(6),GCD0(6),VECU(6,6),EVAL(6),GSD(6),GCD(6)
      REAL*8 GQD(6),XU(6),SDD(6,6),CDD(6,6),W(6)
      REAL*8 S,C,AIM,SDEF,CAIM,AL0,PEN0,RATES,AAA,PP
      REAL*8 SNEW,CNEW,CPEN,D2S
*
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      IF(L0.GE.1) WRITE(IOUT,5)
      IF( NSRCH.LT.1 .OR. NSRCH.GT.6 ) THEN
        WRITE(IOUT,6) NSRCH
        STOP
      ENDIF
* K = first relevant search direction
      K=M11
* Copy to internal (double precision) variables
      CALL MEML1
     * (NSRCH,SDDA,CDDA,SA,CA,AIMA,RMAXA,SDEFA,
     *  SDD,CDD,GSD0,GCD0,S,C,AIM,SDEF,RATES,AL0)
* Optional diagnostics of input model "scalars"
      IF(L1.GE.4) CALL MEML2
     * (NSRCH,SDD,CDD,GSD0,GCD0,IOUT)
* Orthogonalise?
      IF(K.EQ.1) CALL MEML22(NSRCH,SDD,CDD,GSD0,GCD0)
* Diagonalise
      CALL MEML3
     * (NSRCH-K,SDD(1+K,1+K),CDD(1+K,1+K), NDIM,EVAL,VECU,PEN0)
* Transform to eigenvector coords
      CALL MEML4
     * (NSRCH-K,NDIM,GSD0(1+K),GCD0(1+K),EVAL,VECU,C,AIM,GSD,GCD,CAIM)
*
* Central Control * Find alpha and distance penalty *
      CALL MEML5
     * (NDIM,GSD,GCD,EVAL,C,CAIM,PEN0,RATES,AL0, AAA,PP)
*
* Transform back to original coordinates
      CALL MEML6
     * (NSRCH-K,NDIM,GSD,GCD,EVAL,VECU,S,C,PP,AAA,
     *  GQD,XU,W(1+K),SNEW,CNEW,CPEN,D2S)
* Orthogonalised?
      IF(K.EQ.1) CALL MEML66(NSRCH,SDD,W)
* Copy to external (single precision) parameters
      CALL MEML7
     * (NSRCH,W,SNEW,CNEW,D2S,SDEFA, WA,SNEWA,CNEWA,RNEWA)
* Optional numerical diagnostics
      IF(L1.GE.3) CALL MEML8
     * (NSRCH,NDIM,SDD,EVAL,GSD,GCD,GQD,XU,WA,
     *  RNEWA,SA,C,CAIM,SNEWA,CNEW,CPEN,AAA,PP,PEN0, IOUT)
      RETURN
    5 FORMAT('   Control')
    6 FORMAT(' Invalid NSRCH =',I10)
      END
 
 
      SUBROUTINE MEML1
     * (NSRCH,SDDA,CDDA ,SA,CA,AIMA,RMAXA,SDEFA,
     *  SDD,CDD,GSD0,GCD0,S,C,AIM,SDEF,RATES,AL0)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Copy to internal (double precision) variables
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    SDDA    R     I   NSRCH,NSRCH  Entropy metric
*    CDDA    R     I   NSRCH,NSRCH  Constraint matrix
*    SA      R     I      -         Current value of entropy
*    CA      R     I      -         Current value of constraint
*    AIMA    R     I      -         Ultimate target of constraint
*    RMAXA   R     I      -         Dimensionless distance limit
*    SDEFA   R     I      -         Scale factor for entropy
*    SDD     R*8     O NSRCH,NSRCH  Entropy metric
*    CDD     R*8     O NSRCH,NSRCH  Constraint matrix
*    GSD0    R*8     O   NSRCH      Components of GSD
*    GCD0    R*8     O   NSRCH      Components of GCD
*    S       R*8     O    -         Current value of entropy
*    C       R*8     O    -         Current value of constraint
*    AIM     R*8     O    -         Ultimate target of constraint
*    SDEF    R*8     O    -         Scale factor for entropy
*    RATES   R*8     O    -         Maximum distance-squared (*SDEF)
*    AL0     R*8     O    -         Suggested first choice of alpha
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) Internal calculations use S scaled by SDEFA
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL SDDA(6,6),CDDA(6,6),SA,CA,AIMA,RMAXA,SDEFA
      REAL*8 SDD(6,6),CDD(6,6),GSD0(6),GCD0(6),S,C,AIM,SDEF,RATES,AL0
      S=DBLE(SA*SDEFA)
      C=DBLE(CA)
      AIM=DBLE(AIMA)
      SDEF=DBLE(SDEFA)
      RATES=SDEF*DBLE(RMAXA**2)
      DO 2 I=1,NSRCH
        DO 1 J=I,NSRCH
          SDD(I,J)=DBLE(SDDA(I,J))
          CDD(I,J)=DBLE(CDDA(I,J))
          SDD(J,I)=DBLE(SDDA(I,J))
          CDD(J,I)=DBLE(CDDA(I,J))
    1     CONTINUE
        GSD0(I)=SDD(I,2)
        GCD0(I)=SDD(I,3)
    2   CONTINUE
      AL0=SQRT(SDD(3,3))/SQRT(SDD(2,2)+RATES)
      RETURN
      END
 
 
      SUBROUTINE MEML2
     * (NSRCH,SDD,CDD,GSD0,GCD0,IOUT)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Optional diagnostics of input "scalars"
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    SDD     R*8   I   NSRCH,NSRCH  Entropy metric
*    CDD     R*8   I   NSRCH,NSRCH  Constraint matrix
*    GSD0    R*8   I     NSRCH      Components of GSD
*    GCD0    R*8   I     NSRCH      Components of GCD
*    IOUT    I     I      -         Output stream for diagnostics
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*                   10 Dec 1985     Format altered for IBM3081
*
* Notes:
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 GSD0(6),GCD0(6),SDD(6,6),CDD(6,6)
        WRITE(IOUT,10) (GSD0(J),J=1,NSRCH)
        WRITE(IOUT,11) (GCD0(J),J=1,NSRCH)
        WRITE(IOUT,14)
        DO 3 I=1,NSRCH
          WRITE(IOUT,12) (SDD(I,J),J=1,NSRCH)
    3   CONTINUE
        WRITE(IOUT,14)
        DO 4 I=1,NSRCH
          WRITE(IOUT,13) (CDD(I,J),J=1,NSRCH)
    4   CONTINUE
        WRITE(IOUT,14)
      RETURN
   10 FORMAT(/'  GSD  ',1PE14.6,5E14.6)
   11 FORMAT(/'  GCD  ',1PE14.6,5E14.6)
   12 FORMAT('  SDD  ',1PE14.6,5E14.6)
   13 FORMAT('  CDD  ',1PE14.6,5E14.6)
   14 FORMAT('  ')
      END
 
 
      SUBROUTINE MEML22(NSRCH,SDD,CDD,GSD0,GCD0)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*    Orthogonalise search directions 2,3,...,NSRCH to direction 1 (=map)
*    if total(map) held fixed.
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    SDD     R*8   I O NSRCH,NSRCH  Entropy metric
*    CDD     R*8   I O NSRCH,NSRCH  Constraint matrix
*    GSD0    R*8   I O   NSRCH      Components of GSD
*    GCD0    R*8   I O   NSRCH      Components of GCD
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*                    5 Dec 1985     Correction
*
* Notes:
*    (1) The factors of 0.99999 guard against accidentally making
*        matrices non-positive
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 SDD(6,6),CDD(6,6),GSD0(6),GCD0(6),X,Y
      DO 2 I=2,NSRCH
        X=0.99999*SDD(1,I)/SDD(1,1)
        DO 1 J=2,NSRCH
          Y=0.99999*SDD(1,J)/SDD(1,1)
          SDD(I,J)=SDD(I,J)-Y*SDD(1,I)
          CDD(I,J)=CDD(I,J)-X*CDD(1,J)-Y*CDD(1,I)+X*Y*CDD(1,1)
    1   CONTINUE
        GSD0(I)=GSD0(I)-X*GSD0(1)
        GCD0(I)=GCD0(I)-X*GCD0(1)
    2 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEML3
     * (NSRCH,SDD,CDD, NDIM,EVAL,VECU,PEN)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Simultaneous diagonalisation of entropy and constraint matrices
*                             k       k                     k
*   Solves  CDD(i,j) * VECU(j)  = EVAL  * SDD(i,j) * VECU(j)
*   for eigenvalues EVAL and eigenvectors VECU, for k=1,2,...,NDIM
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    SDD     R*8   I   NSRCH,NSRCH  Entropy metric    (symmetric)
*    CDD     R*8   I   NSRCH,NSRCH  Constraint matrix (symmetric)
*    NDIM    I       O    -         Number usefully independent dirns
*    EVAL    R*8     O   NDIM       Eigenvalues of CDD over SDD metric
*    VECU    R*8     O NSRCH,NDIM   Eigenvectors of CDD over SDD metric
*    PEN     R*8     O    -         Uncertainty in lowest EVAL
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    MEML33    Diagonalise symmetric matrix
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*                    9 Dec 1985     Overflow risk reduced
*
* Notes:
*   (1) Eigenvalues EVAL are returned in increasing order
*   (2) Eigenvectors VECU are normalised to  V.SDD.V=1
*   (3) Input matrices SDD and CDD are preserved
*   (4) NDIM usually returns as the full dimension NSRCH of the metric
*       SDD, but if SDD is ill-conditioned the routine will use only
*       the subspace spanned by its larger eigenvalues, and NDIM will
*       return with an appropriately smaller value.
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 SDD(6,6),CDD(6,6),EVAL(6),VECU(6,6)
      REAL*8 AMET(6,6),AMAT(6,6),W1(6,6),W2(6,6),SVEC(6,6)
      REAL*8 SVAL(6),CVAL(6),ERR,EPS,X,Y,PEN
* EPS is related to machine accuracy
      PARAMETER (EPS=3.0E-16)
      PARAMETER (ERR=3.0E-5)
* Copy normalised input matrices to AMAT=CDD, AMET=SDD
      DO 2 I=1,NSRCH
        DO 1 J=1,NSRCH
          X=SQRT(SDD(I,I))*SQRT(SDD(J,J))
          AMAT(I,J)=CDD(I,J)/(X+1.0E-18)
          AMET(I,J)=SDD(I,J)/(X+1.0E-18)
    1   CONTINUE
    2 CONTINUE
* Entropy eigenvalues SVAL and eigenvectors SVEC
* NDIM = number of usefully independent directions
      CALL MEML33(NSRCH,AMET,SVAL,SVEC)
      X=ERR*SVAL(NSRCH)
      K=0
    3 CONTINUE
        K=K+1
      IF( SVAL(K).LT.X )  GOTO 3
      NDIM=NSRCH-K+1
* Rotate CDD to SDD eigenvector space
      DO 6 I=1,NSRCH
        DO 5 J=1,NDIM
          M=J+K-1
          X=0.0
          DO 4 L=1,NSRCH
            X=X+AMAT(I,L)*SVEC(L,M)
    4     CONTINUE
          W2(I,J)=X
    5   CONTINUE
    6 CONTINUE
      DO 9 I=1,NDIM
        M=I+K-1
        DO 8 J=1,NDIM
          X=0.0
          DO 7  L=1,NSRCH
            X=X+SVEC(L,M)*W2(L,J)
    7     CONTINUE
          W1(I,J)=X
    8   CONTINUE
    9 CONTINUE
* CDD is now W1, SDD is now Diag(SVAL)
* Squeeze CDD=W1 to isotropise SDD
      DO 11 I=1,NDIM
        DO 10 J=1,NDIM
          W1(I,J)=W1(I,J)/(SQRT(SVAL(I+K-1))*SQRT(SVAL(J+K-1)))
   10   CONTINUE
   11 CONTINUE
* CDD=W1 eigenvalues CVAL and eigenvectors W2
      CALL MEML33(NDIM,W1,CVAL,W2)
* Complete squeeze of W2 back to SDD eigenvector space
      DO 13 I=1,NDIM
        DO 12 J=1,NDIM
          W2(I,J)=W2(I,J)/SQRT(SVAL(I+K-1))
   12   CONTINUE
   13 CONTINUE
* Rotate W2 to original space
      DO 16 J=1,NDIM
        DO 15 I=1,NSRCH
          X=0.0
          DO 14 L=1,NDIM
            M=L+K-1
            X=X+SVEC(I,M)*W2(L,J)
   14     CONTINUE
        W1(I,J)=X/(SQRT(SDD(I,I))+1.0E-35)
   15   CONTINUE
   16 CONTINUE
* Polish eigenvalues EVAL and recover eigenvectors VECU
      DO 20 J=1,NDIM
        X=0.0
        Y=0.0
        DO 18 L=1,NSRCH
          DO 17 M=1,NSRCH
            X=X+W1(L,J)*CDD(L,M)*W1(M,J)
            Y=Y+W1(L,J)*SDD(L,M)*W1(M,J)
   17     CONTINUE
   18   CONTINUE
        EVAL(J)=X/Y
        DO 19 L=1,NSRCH
          VECU(L,J)=W1(L,J)/SQRT(Y)
   19   CONTINUE
   20 CONTINUE
* Uncertainty in EVAL(1)  (care with overflow risk)
      X=0.0
      DO 22 L=1,NSRCH
        DO 21 M=1,NSRCH
          Y=VECU(L,1)*CDD(L,M)
          X=X+ABS(Y*VECU(M,1))
          Y=VECU(L,1)*SDD(L,M)
          X=X+ABS(EVAL(1)*Y*VECU(M,1))
   21   CONTINUE
   22 CONTINUE
      PEN=ERR*X/NSRCH
      RETURN
      END
 
 
      SUBROUTINE MEML33(ND,AMAT,VAL,VEC)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Diagonalisation of symmetric matrix
*                                 k      k         k
*   Solves      AMAT(i,j) * VEC(j)  = VAL  * VEC(i)
*   for eigenvalues VAL and eigenvectors VEC, for k=1,2,...,ND
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    ND      I     I      -         Number of dimensions
*    AMAT    R*8   I     ND,ND      Matrix (from 6*6 array)
*    VAL     R*8     O    ND        Eigenvalues
*    VEC     R*8     O   ND,ND      Eigenvectors
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
*
* Notes:
*   (1) Eigenvalues VAL are returned in increasing order
*   (2) Eigenvectors VEC are normalised to V.AMAT.V=1
*   (3) Input matrix AMAT is preserved
*   (4) Only the upper triangle j>=i of AMAT(i,j) is read
*
*   (5) Algorithm is to repeatedly square AMAT until the largest
*       eigenvalue dominates, then to subtract off that eigenvector,
*       and repeat until all the eigenvectors have been found.
*       This ND**4 algorithm would be too slow for large matrices,
*       but is robust for small.
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 AMAT(6,6),VAL(6),VEC(6,6)
      REAL*8 Z(6,6),P(6,6),Q(6,6),X(6),Y(6),T,A,B,C,EPS
* EPS is related to machine accuracy
      PARAMETER (EPS=3.0E-16)
* Copy AMAT to a positive definite matrix Z (by adding C to eigenvals)
      A=0.0
      B=0.0
      T=0.0
      DO 2 I=1,ND
        A=A+AMAT(I,I)
        DO 1  J=1,ND
          C=AMAT(I,J)
          IF(ABS(C).LT.1.0E-18) C=0.0
          B=B+C**2
    1   CONTINUE
        T=T+1.0
    2 CONTINUE
      C=MAX(B-A*A/T,0.0D0)
      C=A/T-SQRT(C)-EPS*SQRT(B)
      DO 4 I=1,ND
        DO 3 J=I,ND
          Z(I,J)=AMAT(I,J)
          Z(J,I)=AMAT(I,J)
    3   CONTINUE
        Z(I,I)=Z(I,I)-C
    4 CONTINUE
* LMAX = maximum number of inner loop iterates
      T=-LOG(EPS)/(EPS*LOG(2.0))
      T=LOG(T)/LOG(2.0)
      LMAX=IFIX(SNGL(T+2.0))
*
      N=ND
* Outer loop over N for successively smaller eigenvalues
    5 CONTINUE
        T=0.0
        DO 7 I=1,ND
          DO 6 J=1,ND
            P(I,J)=Z(I,J)
    6     CONTINUE
          T=T+P(I,I)
    7   CONTINUE
*
        L=0
* Inner loop over L for squaring P and setting T=trace
    8   CONTINUE
          L=L+1
          T=1.0/T
          DO 10 I=1,ND
            DO  9 J=1,ND
              Q(I,J)=P(I,J)*T
              IF(ABS(Q(I,J)).LT.1.0E-18) Q(I,J)=0.0
    9       CONTINUE
   10     CONTINUE
          T=0.0
          DO 13 I=1,ND
            DO 12 J=I,ND
              A=0.0
              DO 11 K=1,ND
                A=A+Q(I,K)*Q(K,J)
   11         CONTINUE
              IF(ABS(A).LT.1.0E-18) A=0.0
              P(I,J)=A
              P(J,I)=A
   12       CONTINUE
            T=T+P(I,I)
   13     CONTINUE
        IF( T.LT.1.0-EPS*10.0 .AND. L.LE.LMAX )  GOTO 8
* End inner loop when P is dyadic
*
* K = largest column = estimate of current largest eigenvector
        K=1
        A=0.0
        DO 14 I=1,ND
          IF(P(I,I).GT.A) THEN
            A=P(I,I)
            K=I
          ENDIF
   14   CONTINUE
*
* P(P(largest column)) = better estimate X of eigenvector
        DO 16 I=1,ND
          A=0.0
          DO 15  J=1,ND
            A=A+P(I,J)*P(J,K)
   15     CONTINUE
          IF(ABS(A).LT.1.0E-18) A=0.0
          Y(I)=A
   16   CONTINUE
        T=0.0
        DO 18 I=1,ND
          A=0.0
          DO 17 J=1,ND
            A=A+P(I,J)*Y(J)
   17     CONTINUE
          IF(ABS(A).LT.1.0E-18) A=0.0
          T=T+A*A
          X(I)=A
   18   CONTINUE
* Repeat..  Orthogonalise X to previous eigenvectors
        K=ND
        IF(K.EQ.N) GOTO 22
   19     A=0.0
          DO 20 I=1,ND
            A=A+X(I)*VEC(I,K)
   20     CONTINUE
          DO 21 I=1,ND
            B=X(I)-A
            IF(ABS(B).LT.1.0E-18) B=0.0
            X(I)=B
   21     CONTINUE
          K=K-1
          IF(K.GT.N) GOTO 19
* ..while
* Normalise eigenvector X
   22   A=0.0
        DO 23 I=1,ND
          A=A+X(I)*X(I)
   23   CONTINUE
        A=1.0/SQRT(A)
* Copy eigenvector X into output array VEC
        DO 24 I=1,ND
          X(I)=A*X(I)
          VEC(I,N)=X(I)
   24   CONTINUE
* Set eigenvalue VAL directly from the eigenvector
        A=0.0
        DO 26 I=1,ND
          DO 25 J=1,ND
            B=X(I)*X(J)
            IF(ABS(B).LT.1.0E-18) B=0.0
            A=A+Z(I,J)*B
   25     CONTINUE
   26   CONTINUE
        VAL(N)=A+C
* Finish ?   (if full set of eigenvectors has been found)
        N=N-1
        IF(N.LE.0) RETURN
* Otherwise, remove dyadic  X.Xtranspose  from matrix Z
        DO 28 I=1,ND
          DO 27 J=I,ND
            B=X(I)*X(J)
            IF(ABS(B).LT.1.0E-18) B=0.0
            Z(I,J)=Z(I,J)-A*B
            IF(ABS(Z(I,J)).LT.1.0E-18) Z(I,J)=0.0
            Z(J,I)=Z(I,J)
   27     CONTINUE
   28   CONTINUE
      GOTO 5
* End outer loop
*
      END
 
 
      SUBROUTINE MEML4
     * (NSRCH,NDIM,GSD0,GCD0,EVAL,VECU,C,AIM, GSD,GCD,CAIM)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Transfer problem to eigenvector coordinates
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    NDIM    I     I      -         Number of independent eigenvectors
*    GSD0    R*8   I     NSRCH      Components of GSD in original coords
*    GCD0    R*8   I     NSRCH      Components of GCD in original coords
*    EVAL    R*8   I     NDIM       Eigenvalues of C matrix
*    VECU    R*8   I   NSRCH,NDIM   Eigenvector matrix from (6,6) array
*    C       R*8   I      -         Current value of Constraint
*    AIM     R*8   I      -         Ultimate target of Constraint
*    GSD     R*8     O   NDIM       Components of GSD in diagonal coords
*    GCD     R*8     O   NDIM       Components of GCD in diagonal coords
*    CAIM    R*8     O    -         Realistic target value of Constraint
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*   (1) CAIM is set not more than 4/5 of the way down to the lowest
*       value of constraint accessible in the entire subspace.
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 GSD0(6),GCD0(6),VECU(6,6),EVAL(6),GSD(6),GCD(6)
      REAL*8 C,AIM,CAIM,CMIN
* Rewrite vectors GSD,GCD in eigenvector coordinates
      DO 2 J=1,NDIM
        GCD(J)=0.0
        GSD(J)=0.0
        DO 1 K=1,NSRCH
          GCD(J)=GCD(J)+GCD0(K)*VECU(K,J)
          GSD(J)=GSD(J)+GSD0(K)*VECU(K,J)
    1   CONTINUE
    2 CONTINUE
* Metric SDD is now unit, Matrix CDD is now Diagonal(EVAL)
* Set CAIM
      CMIN=C
      DO 3 K=1,NDIM
        CMIN=CMIN-0.5*GCD(K)*GCD(K)/MAX(EVAL(K),1.0D-12)
    3 CONTINUE
      CAIM=MAX((4.0*CMIN+C)/5.0 , AIM)
      RETURN
      END
 
 
      SUBROUTINE MEML5
     * (NDIM,GSD,GCD,EVAL,C,CAIM,PEN0,RATES,AL0, AAA,PP)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Find smallest acceptable distance penalty PP and its alpha AAA
*   of the final subspace solution which maximises S over the target C
*   subject to the distance constraint
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NDIM    I     I      -         Number of dimensions in subspace
*    GSD     R*8   I     NDIM       Components of GSD in diagonal coords
*    GCD     R*8   I     NDIM       Components of GCD in diagonal coords
*    EVAL    R*8   I     NDIM       Eigenvalues of C matrix (now diag.)
*    C       R*8   I      -         Current value of Constraint
*    CAIM    R*8   I      -         Realistic target value of Constraint
*    PEN0    R*8   I      -         Penalty offset (usually small)
*    RATES   R*8   I      -         Maximum limit of distance**2 moved
*    AL0     R*8   I      -         Suggested starting value of alpha
*    AAA     R*8     O    -         Alpha of final subspace solution
*    PP      R*8     O    -         Penalty of final subspace solution
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    MEML55    Alpha-loop
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*   (1) There is some protection against negative eigenvales EVAL of the
*       constraint function, but this version of control is not
*       designed to handle such non-convex problems robustly
*   (2) Distance penalty P increases the effective constraint value CPEN
*       and not the entropy
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 GSD(6),GCD(6),EVAL(6),C,CAIM,PEN0,RATES,AL0,EE,AAA,PP
      REAL*8 PEN,P1,P2,P,EPS,EPS4,A1,AL,A2,AA
* EPS related to machine accuracy
      PARAMETER (EPS=3.0D-16)
      EPS4=SQRT(SQRT(EPS))
* EE is eigenvalue norm
      EE=1.0E-35
      DO 4 K=1,NDIM
        EE=EE+ABS(EVAL(K))/FLOAT(K)
    4 CONTINUE
* Initialise penalty loop
      PEN=0.0
      P1=0.0
      P2=1.0E30
      PP=0.0
      AAA=0.0
      LOOP=0
*
* Begin penalty loop
    1 CONTINUE
        IF(LOOP.EQ.1)  PEN=(2.0*P1+1.0+P1/P2)/(1.0+(2.0+P1)/P2)
        LOOP=1
* Initialise alpha loop
        AL=AL0
        A1=AL*EPS
        A2=AL/EPS
        AA=0.0
        P=PEN0+0.1*PEN*EE
* Call alpha-loop routine to find correct value of alpha
        CALL MEML55(NDIM,GSD,GCD,EVAL,C,CAIM,P,RATES,A1,AL,A2,AA)
* Check acceptability of alpha returned from its loop
        IF( AA.EQ.A2 )  THEN
* Penalty large enough. Store it and prepare to decrease it
          AAA=AA
          PP=PEN
          P2=PEN
        ELSE
* Penalty too small. Prepare to increase it
          P1=PEN
        ENDIF
      IF( P2-P1.GT.EPS4*PEN )  GOTO 1
*
* Exit with smallest acceptable distance penalty and its alpha
      PP=PEN0+0.1*PP*EE
      RETURN
      END
 
 
      SUBROUTINE MEML55
     * (NDIM,GSD,GCD,EVAL,C,CAIM,P,RATES,A1,AL,A2,AA)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*      Find the value of Alpha which maximises S over the target value
*      of C within the distance limit in the diagonalised subspace.
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NDIM    I     I      -         Number of dimensions in subspace
*    GSD     R*8   I     NDIM       Components of GSD in diagonal coords
*    GCD     R*8   I     NDIM       Components of GCD in diagonal coords
*    EVAL    R*8   I     NDIM       Eigenvalues of C matrix (now diag.)
*    C       R*8   I      -         Current value of Constraint
*    CAIM    R*8   I      -         Target value (within distance limit)
*    P       R*8   I      -         Distance penalty
*    RATES   R*8   I      -         Maximum limit of distance**2 moved
*    A1      R*8   I O    -         Lower limit on alpha in binary chop
*    AL      R*8   I O    -         Alpha
*    A2      R*8   I O    -         Upper limit on alpha in binary chop
*    AA      R*8   I O    -         Smallest A2 within distance limit
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*   (1) Alpha is found to full machine accuracy
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 GSD(6),GCD(6),EVAL(6),C,CAIM,P,RATES,A1,A2,AA
      REAL*8 E(6),GQD,XU,CMIN,CMAX,DA,AL,D2S,CPEN,CNEW
* Set extreme values
      CMIN=MIN(C,CAIM)
      CMAX=MAX(C,CAIM)
* Revise eigenvalues by distance penalty
      DO 1 K=1,NDIM
        E(K)=MAX( EVAL(K)+P ,0.0D0)
    1 CONTINUE
*
* Begin central alpha-chop loop
    2 CONTINUE
* Set alpha between A1 and A2 (allowing for large dynamic range)
*     and DA for use in terminating the loop
        AL=(2.0*A1*(A2/AL)+A1+A2)/(2.0+(A1+A2)/AL)
        DA=A2-A1
* Solve B(..)X(.)=GQ(.) for X(.) and
*  get corresponding new statistics CNEW, CPEN and squared-distance D2S
        D2S=0.0
        CPEN=C
        CNEW=C
        DO 3 K=1,NDIM
          GQD=AL*GSD(K)-GCD(K)
          XU=GQD/(E(K)+AL)
          D2S=D2S+XU*XU
          CPEN=CPEN+XU*(GCD(K)+0.5*E(K)*XU)
          CNEW=CNEW+XU*(GCD(K)+0.5*EVAL(K)*XU)
   3   CONTINUE
* Alpha control
        IF( CPEN.GT.CMAX .OR.
     *    (CNEW.GT.CMIN .AND. (D2S-RATES)*(C-CAIM).LE.0.0)) THEN
*   Reduce alpha and store upper limit in A2 if distance is acceptable
          A2=AL
          IF(D2S.LE.RATES) AA=AL
        ELSE
*   Increase alpha
          A1=AL
        ENDIF
      IF( A2-A1.LT.DA )  GOTO 2
* End alpha loop when full machine accuracy is reached
*  and DA can no longer decrease
*
* Exit with AA=A2 if the result is within the distance limit
      RETURN
      END
 
 
      SUBROUTINE MEML6
     * (NSRCH,NDIM,GSD,GCD,EVAL,VECU,S,C,PDIST,ALPHA,
     *  GQD,XU,W,SNEW,CNEW,CPEN,D2S)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Transfer solution back to original coordinates
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    NDIM    I     I      -         Number of independent eigenvectors
*    GSD     R*8   I     NDIM       Components of GSD in diagonal coords
*    GCD     R*8   I     NDIM       Components of GCD in diagonal coords
*    EVAL    R*8   I     NDIM       Eigenvalues of C matrix
*    VECU    R*8   I   NSRCH,NDIM   Eigenvector matrix from (6,6) array
*    S       R*8   I      -         Current value of Entropy
*    C       R*8   I      -         Current value of Constraint
*    PDIST   R*8   I      -         Distance penalty
*    ALPHA   R*8   I      -         Alpha
*    GQD     R*8     O   NDIM       Alpha*GSD - GCD
*    XU      R*8     O   NDIM       Map increment in diagonal coords
*    W       R*8     O   NSRCH      Map increment in original coords
*    SNEW    R*8     O    -         Predicted value of Entropy
*    CNEW    R*8     O    -         Predicted value of Constraint
*    CPEN    R*8     O    -         CNEW modified by distance penalty
*    D2S     R*8     O    -         Distance**2 actually moved  (*SUMF)
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*   (1) The output parameters GQD, XU, CPEN, CNEW, D2S
*       are only returned for diagnostics
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 GSD(6),GCD(6),EVAL(6),VECU(6,6),GQD(6),XU(6),W(6)
      REAL*8 S,C,PDIST,ALPHA,SNEW,CNEW,CPEN,D2S, E
* Set map increment in diagonal coords and set scalar diagnostics
      D2S=0.0
      CPEN=C
      CNEW=C
      SNEW=S
      DO 1 K=1,NDIM
        E=MAX( EVAL(K)+PDIST ,0.0D0)
        GQD(K)=ALPHA*GSD(K)-GCD(K)
        XU(K)=GQD(K)/(E+ALPHA)
        D2S=D2S+XU(K)*XU(K)
        CPEN=CPEN+XU(K)*(GCD(K)+0.5*E*XU(K))
        CNEW=CNEW+XU(K)*(GCD(K)+0.5*EVAL(K)*XU(K))
        SNEW=SNEW+XU(K)*(GSD(K)-0.5*XU(K))
    1 CONTINUE
* Rewrite map increment in original coords
      DO 3 J=1,NSRCH
        W(J)=0.0
        DO 2 K=1,NDIM
          W(J)=W(J)+VECU(J,K)*XU(K)
    2   CONTINUE
    3 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEML66(NSRCH,SDD,W)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*          W(1) := Coefficient of "map" if total(map) held fixed
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    SDD     R*8   I   NSRCH,NSRCH  Entropy metric
*    W       R*8   I O   NSRCH      Map increment in original coords
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 SDD(6,6),W(6),X
      X=0.
      DO 1 I=2,NSRCH
        X=X+SDD(1,I)*W(I)
    1 CONTINUE
      W(1)=-X/SDD(1,1)
      RETURN
      END
 
 
      SUBROUTINE MEML7
     * (NSRCH,W,SNEW,CNEW,D2S,SDEFA, WA,SNEWA,CNEWA,RNEWA)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Copy to external (single precision) parameters
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    W       R*8   I     NSRCH      Map increment in original coords
*    SNEW    R*8   I      -         Predicted value of entropy
*    CNEW    R*8   I      -         Predicted value of constraint
*    D2S     R*8   I      -         Distance moved (*SUMF)
*    SDEFA   R     I      -         Scale factor for entropy
*    WA      R       O    -         Map increment components
*    SNEWA   R       O    -         Predicted value of entropy
*    CNEWA   R       O    -         Predicted value of constraint
*    RNEWA   R       O    -         Dimensionless distance moved
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) Internal calculations use S scaled by SUMDEF
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 W(6),SNEW,CNEW,D2S
      REAL WA(6),SNEWA,CNEWA,RNEWA,SDEFA
      SNEWA=SNGL(SNEW)/SDEFA
      CNEWA=SNGL(CNEW)
      RNEWA=SQRT(SNGL(D2S)/SDEFA)
      DO 1 I=1,NSRCH
        WA(I)=SNGL(W(I))
    1 CONTINUE
      RETURN
      END
 
 
      SUBROUTINE MEML8(NSRCH,NDIM,SDD,EVAL,GSD,GCD,GQD,XU,W,
     *           RNEWA,S,C,CAIM,SNEW,CNEW,CPEN,ALPHA,PDIST,PEN0, IOUT)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Optional diagnostics
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    NSRCH   I     I      -         Number of original search dirns
*    NDIM    I     I      -         Number of independent eigenvectors
*    SDD     R*8   I   NSRCH,NSRCH  Entropy metric in original coords
*    EVAL    R*8   I     NDIM       Eigenvalues of C matrix
*    GSD     R*8   I     NDIM       Components of GSD in diagonal coords
*    GCD     R*8   I     NDIM       Components of GCD in diagonal coords
*    GQD     R*8   I     NDIM       Alpha*GSD - GCD
*    XU      R*8   I     NDIM       Map increment in diagonal coords
*    W       R     I     NSRCH      Map increment in original coords
*    RNEWA   R     I      -         Distance moved (dimensionless)
*    S       R     I      -         Current value of Entropy
*    C       R*8   I      -         Current value of Constraint
*    CAIM    R*8   I      -         Realistic target value of Constraint
*    SNEW    R     I      -         Predicted value of Entropy
*    CNEW    R*8   I      -         Predicted value of Constraint
*    CPEN    R*8   I      -         CNEW modified by distance penalty
*    ALPHA   R*8   I      -         Alpha
*    PDIST   R*8   I      -         Distance penalty
*    PEN0    R*8   I      -         Penalty offset
*    IOUT    I     I      -         Output stream for diagnostics
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    -
*
* External calls:
*    -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*                   10 Dec 1985     Format altered for IBM3081
*
* Notes:
*
*-----------------------------------------------------------------------
      IMPLICIT CHARACTER*1 (A-H,O-Z)
      REAL*8 SDD(6,6),EVAL(6),GSD(6),GCD(6),GQD(6),XU(6)
      REAL*8 C,CAIM,CNEW,CPEN,ALPHA,PDIST,PEN0, CMIN,CMAX
      REAL W(6),RNEWA,S,SNEW
      CMIN=MIN(C,CAIM)
      CMAX=MAX(C,CAIM)
      WRITE(IOUT,1) RNEWA,CNEW,ALPHA,PDIST-PEN0
      WRITE(IOUT,2) CMIN,CNEW,CPEN,CMAX,C
      WRITE(IOUT,3) SNEW,S
      WRITE(IOUT,4) (SDD(K,K),K=1,NSRCH)
      WRITE(IOUT,5) (EVAL(K),K=1,NDIM)
      WRITE(IOUT,6) (GSD(K),K=1,NDIM)
      WRITE(IOUT,7) (GCD(K),K=1,NDIM)
      WRITE(IOUT,8) (GQD(K),K=1,NDIM)
      WRITE(IOUT,9) (XU(K),K=1,NDIM)
      WRITE(IOUT,10) (W(K),K=1,NSRCH)
      WRITE(IOUT,11)
      RETURN
    1 FORMAT(' RCAP   ',1PE13.5,4E13.5)
    2 FORMAT(' <<<<C  ',1PE13.5,4E13.5)
    3 FORMAT(' NEW S  ',1PE26.5,E39.5)
    4 FORMAT('  A.A  ',1PE10.2,5E10.2)
    5 FORMAT('  EVAL ',1PE10.2,5E10.2)
    6 FORMAT('  GSD  ',1PE10.2,5E10.2)
    7 FORMAT('  GCD  ',1PE10.2,5E10.2)
    8 FORMAT('  GQD  ',1PE10.2,5E10.2)
    9 FORMAT('  XU   ',1PE10.2,5E10.2)
   10 FORMAT('  W    ',1PE10.2,5E10.2)
   11 FORMAT('  ')
      END
 
 
      SUBROUTINE UREAD(I)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Read a buffer from given area and increment pointers appropriately
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    I       I     I      -         Input area number
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS       O  Storage vector
*    MJ,MK       MECOMP     I    Buffer sizes (map and data space)
*    KA          MECOMP     I    Allocation pointers
*    KB          MECOMP      -   Base addresses
*    KC          MECOMP     I O  Core addresses
*    KD          MECOMP     I O  Disc or upper core addresses
*    PR          MECOMC       O  Read/Write flags
*
* External calls:
*    UGET        Get buffer from disc
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) Use UPDATE instead of UREAD if area to be modified in-place
*
*-----------------------------------------------------------------------
      CHARACTER*1 PR
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      COMMON /MECOMC/ PR(40)
      PR(I)='R'
      IF(I.LE.20) THEN
        M=MJ
      ELSE
        M=MK
      ENDIF
      IF(KA(I).GT.0) THEN
* Disc
        CALL UGET(I,ST(KC(I)),M)
      ELSE
* Core
        KC(I)=KD(I)
        KD(I)=KD(I)+M
      ENDIF
      RETURN
      END
 
      SUBROUTINE UPDATE(I)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Read a buffer from given area, ready for in-place writing
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    I       I     I      -         Input area number
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS       O  Storage vector
*    MJ,MK       MECOMP     I    Buffer sizes (map and data space)
*    KA          MECOMP     I    Allocation pointers
*    KB          MECOMP      -   Base addresses
*    KC          MECOMP     I    Core addresses
*    KD          MECOMP      -   Disc or upper core addresses
*    PR          MECOMC       O  Read/Write flags
*
* External calls:
*    UGET        Get buffer from disc
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1) UPDATE will not work on sequential access disc files
*    (2) Use UREAD instead of UPDATE if area is read-only
*
*-----------------------------------------------------------------------
      CHARACTER*1 PR
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      COMMON /MECOMC/ PR(40)
      PR(I)='U'
      IF(I.LE.20) THEN
        M=MJ
      ELSE
        M=MK
      ENDIF
* Disc
      IF(KA(I).GT.0) CALL UGET(I,ST(KC(I)),M)
* Core
      RETURN
      END
 
      SUBROUTINE UWRITE(I)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Write a buffer to given area and increment pointers appropriately
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    I       I     I      -         Output area number
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I    Storage vector
*    MJ,MK       MECOMP     I    Buffer sizes (map and data space)
*    KA          MECOMP     I    Allocation pointers
*    KB          MECOMP      -   Base addresses
*    KC          MECOMP     I O  Core addresses
*    KD          MECOMP     I O  Disc or upper core addresses
*    PR          MECOMC       O  Read/Write flags
*
* External calls:
*    UPUT        Put buffer to disc
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      CHARACTER*1 PR
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
      COMMON /MECOMC/ PR(40)
      PR(I)='W'
      IF(I.LE.20) THEN
        M=MJ
      ELSE
        M=MK
      ENDIF
      IF(KA(I).GT.0) THEN
* Disc
        CALL UPUT(I,ST(KC(I)),M)
      ELSE
* Core
        KD(I)=KD(I)+M
        KC(I)=KD(I)
      ENDIF
      RETURN
      END
 
      SUBROUTINE UINIT
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Re-initialise all pointers, with optional diagnostics
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*     -
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    IOUT,L0     MECOMP     I    Diagnostic stream and flag
*    PR          MECOMC     I    Read/Write flags
*
* External calls:
*    MEINIT      Re-initialise all pointers
*    UDIAG       User's diagnostic routine
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      CHARACTER*1 PR
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMC/ PR(40)
      IF(L0.GE.2) WRITE(IOUT,4) PR
      CALL MEINIT
      IF(L0.GE.3) CALL UDIAG
      RETURN
    4 FORMAT(11X,20A1,2X,20A1)
      END
 
      SUBROUTINE MEINIT
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Re-initialise all pointers
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*     -
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    PR          MECOMC       O  Read/Write flags
*
* External calls:
*    URESET      Reset pointers for a given area
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      CHARACTER*1 PR
      COMMON /MECOMC/ PR(40)
      DO 1 I=1,40
        PR(I)='.'
        CALL URESET(I)
    1 CONTINUE
      RETURN
      END
 
      SUBROUTINE URESET(I)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Reset pointers for given area
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    I       I     I      -         Area number
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    KA          MECOMP     I    Allocation pointers
*    KB          MECOMP     I    Base addresses
*    KC          MECOMP       O  Core addresses
*    KD          MECOMP       O  Disc or upper core addresses
*
* External calls:
*    UWIND       Rewind disc file
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*
*-----------------------------------------------------------------------
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      KC(I)=KB(I)
      IF(KA(I).GT.0) THEN
* Disc
        CALL UWIND(I)
      ELSE
* Core
        KD(I)=KB(I)
      ENDIF
      RETURN
      END
 
      SUBROUTINE UGET(I,X,M)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Get a buffer from disc file
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    I       I     I      -         Input area number
*    X       R       O    M         Buffer's destination in core
*    M       I     I      -         Length of buffer in words
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    KA          MECOMP     I    Allocation pointers to disc files
*
* External calls:
*     -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1)  This implementation uses unformatted sequential Fortran I/O
*    (2)  For direct access files, pointers KD are available for use
*
*-----------------------------------------------------------------------
      REAL X(M)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      K=KA(I)
      READ(K) X
      RETURN
      END
 
      SUBROUTINE UPUT(I,X,M)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   Put a buffer to disc file
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    I       I     I      -         Output area number
*    X       R     I      M         Buffer's source in core
*    M       I     I      -         Length of buffer in words
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    KA          MECOMP     I    Allocation pointers to disc files
*
* External calls:
*     -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1)  This implementation uses unformatted sequential Fortran I/O
*    (2)  For direct access files, pointers KD are available for use
*
*-----------------------------------------------------------------------
      REAL X(M)
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      K=KA(I)
      WRITE(K) X
      RETURN
      END
 
      SUBROUTINE UWIND(I)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*    Rewind disc file
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    I       I     I      -         Area number
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    KA          MECOMP     I    Allocation pointers to disc files
*
* External calls:
*     -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1)  This implementation uses unformatted Fortran I/O
*    (2)  For direct access files, pointers KD are available for use
*
*-----------------------------------------------------------------------
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      K=KA(I)
      REWIND K
      RETURN
      END
 
 
      SUBROUTINE URAND(X,Y)
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*        Generate random number from    Normal( mean=0 , variance=1 )
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    X       R         O  -         Output random variable
*    Y       R       I O  -         Initialisation value, to be set to
*                                   0 <= Y < 16777216.0 and thereafter
*                                   preserved between calls during a run
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*     -
*
* External calls:
*     -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*
* Notes:
*    (1)  Repeat cycle = 2**21 = 2097152 calls
*
*-----------------------------------------------------------------------
      REAL*8 Z
      X=0.
      Z=Y
      Z=2.0*Y+1.0
      DO 1 K=1,13
        Z=MOD(Z*19487171.D0,33554432.D0)
        X=REAL(X+Z)
    1 CONTINUE
      X=X*2.863314E-8 -6.244998
      Z=(Z-1.0)/2.0
      Y=REAL(Z)
      RETURN
      END
 
 
      SUBROUTINE UDIAG
*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*
* Purpose:
*   User's diagnostic routine, to inspect any part of any area in core.
*
* Parameters:
*   ARGUMENT TYPE  I/O  DIMENSION   DESCRIPTION
*    -
*
* Globals:
*   VARIABLE  COMMON BLOCK  I/O  DESCRIPTION
*    ST          MECOMS     I    Storage vector
*    NJ,MJ       MECOMP     I    Sizes  (map space)
*    NK,MK       MECOMP     I    Sizes (data space)
*    KA          MECOMP     I    Initialised allocation pointers
*    KB          MECOMP     I    Initialised base addresses
*
* External calls:
*     -
*
* History:
*   John Skilling    8 Nov 1985     Initial release
*                   10 Dec 1985     Modified for IBM3081
*
* Notes:
*    (1)  This implementation is numerical, and interrogates the user
*         via the default Fortran stream (*)
*
*-----------------------------------------------------------------------
      COMMON /MECOMP/ NJ,MJ,NK,MK,KA(40),KB(40),KC(40),KD(40),
     * IOUT,IIN,L0,L1,M0,M10,M11,M20,M21,M3
      COMMON /MECOMS/ ST(1)
    1 CONTINUE
        WRITE(*,*) ' Which area would you like?  (0 exits)  '
        READ(*,*,END=1,ERR=1) N
        IF(N.LT.0 .OR. N.GT.40) GOTO 1
        IF(N.EQ.0) RETURN
        IF(N.LE.20) THEN
          M=MJ
          IF(KA(N).EQ.0) M=M*NJ
        ELSE
          M=MK
          IF(KA(N).EQ.0) M=M*NK
        ENDIF
        IF(KA(N).EQ.0) THEN
          WRITE(*,5) KA(N),KB(N)
        ELSE
          WRITE(*,6) KA(N),KB(N)
        ENDIF
        WRITE(*,7) KB(N)+M-1
        WRITE(*,*) ' First point ?  '
        READ(*,*) J
        J=KB(N)+J-1
        WRITE(*,*) '  Last point ?  '
        READ(*,*) K
        K=KB(N)+K-1
        WRITE(*,8) (ST(I),I=J,K)
      GOTO 1
    5 FORMAT(' KA =',I3,'  (core file)              ',
     *       ' KB =',I7,' = base address')
    6 FORMAT(' KA =',I3,'  (disc file)              ',
     *       ' KB =',I7,' = base address')
    7 FORMAT(    40X,I7,' = last address')
    8 FORMAT(1PE14.4,3E14.4)
      END
 
