

      SUBROUTINE SUB_SIN(IMAGE, ERRORS, HJD, NX, NY, XLO, XHI, 
     &PERIOD, METHOD, CTIME, TSTEP, AMP, PHI0, NPOLY)
*
* Fits sine wave of period PERIOD to data in Y direction and subtracts
* the fit from the original data.
*
* R*4 IMAGE(NX,NY)    -- Data (NX spectrum points, NY spectra)
*
* R*4 ERRORS(NX, NY) -- 1-sigma uncertainties
*
* R*8 HJD(NY)        -- HJDs of NY spectra
*
* I*4 NX, NY -- Dimensions
*
* I*4 XLO, XHI -- Range of X for subtraction
*
* R*4 PERIOD   -- Period to fit.
*
* I*4 METHOD   -- 1 sine waves are fitted and removed; 2 sine waves are
*                 fitted and then data is replaced by sine wave of same
*                 amplitude and phase as fit plus noise; 3, as 2 but with
*                 no noise added; 4  1/f noise simulation; 5 data replaced
*                 by sine wave; 6 sine wave added; 7 polynomial subtracted.
*
* R*4 CTIME    -- Correlation time for 1/f noise simulation. At this lag,
*                 auto-correlation function has fallen by a factor of 2
*
* R*4 TSTEP    -- Time step for for 1/f noise simulation. 
*
* R*4 AMP      -- Amplitude of RMS for METHOD=4, of sine wave for METHOD=5,6
*
* R*4 PHI0     -- Phase of sine wave for METHOD=5,6 
*
* I*4 NPOLY    -- Order of polynomial if one is to be subtracted METHOD=7
*
      INTEGER NX, NY, XLO, XHI, METHOD
      REAL IMAGE(NX, NY), ERRORS(NX, NY), PERIOD
      REAL AMP, PHI0, CTIME, TSTEP
      REAL*8 HJD(NY)
      PARAMETER (MAXPOLY=20)
      REAL*8 ARC(MAXPOLY), XM(MAXPOLY,2*MAXPOLY+3)
      REAL*8 CHISQ, POLY
*
      PARAMETER (MAXPNT = 2000)
      REAL XDATA(MAXPNT), YDATA(MAXPNT), YERR(MAXPNT)
      REAL*8 DATA(3, MAXPNT)
      INTEGER IRANK(MAXPNT), NPOLY
      REAL COV(6), GAMMA, KVEL, GAUSS2
      REAL*8 AVERAGE, ALPHA, Z
      CHARACTER*80 INFORM
*
      IF(NY.GT.MAXPNT) THEN
        CALL PAR_WRUSER('Too many time points. Must increase', IFAIL)
        CALL PAR_WRUSER('MAXPNT inside SUB_SIN', IFAIL)
        RETURN
      END IF
      IF(METHOD.EQ.7 .AND. NPOLY.GT.MAXPOLY) THEN
        CALL PAR_WRUSER('Too high a poly order', IFAIL)
        RETURN
      END IF     
*
      AVERAGE = 0.D0
      DO IY = 1, NY
        AVERAGE = AVERAGE + HJD(IY)
      END DO
      AVERAGE = AVERAGE/DBLE(NY)
      DO IY = 1, NY
        XDATA(IY) = HJD(IY) - AVERAGE 
        DATA(1,IY) = HJD(IY) - AVERAGE
        XDATA(IY) = DATA(1,IY)
      END DO
*
      NCOUNT = 10000/(NY*(XHI-XLO+1))
      NCOUNT = MAX(1, NCOUNT)
      ISEED = -12345
      TWOPI = 8.*ATAN(1.)
      IF(METHOD.EQ.4) THEN
        CALL HEAPSORT(NY, XDATA, IRANK)
        ALPHA = EXP(TSTEP*LOG(0.5)/CTIME)
        SIGMA = SQRT(1.D0-ALPHA**2)*AMP
        DO IX = XLO, XHI
          IF(MOD(IX-XLO+1,NCOUNT).EQ.0) THEN
            WRITE(INFORM,'(A,I4)') ' Reached X = ',IX
            CALL PAR_WRUSER(INFORM, IFAIL)
          END IF
*
          DO IY = 1, NY
            IF(IY.EQ.1) THEN
              Z = GAUSS2(0.,AMP,ISEED)
              Y = Z
              T = XDATA(IRANK(1))
            ELSE
*
* Take enough time steps to get to next point
*
              DIFF = XDATA(IRANK(IY))-T
              IF(DIFF.GT.10.*CTIME) THEN
                Z = GAUSS2(0.,AMP,ISEED)
                Y = Z
                T = XDATA(IRANK(IY))
              ELSE
                NSTEP = INT(DIFF/TSTEP)
                NN = 0
 100            CONTINUE
                ZOLD = Z
                Z = ALPHA*Z + GAUSS2(0.,SIGMA,ISEED)
                NN = NN + 1
                IF(NN.LT.NSTEP+1) GOTO 100
                T1 = REAL(NSTEP)*TSTEP
                T2 = T1 + TSTEP
                Y = ((DIFF-T1)*Z+(T2-DIFF)*ZOLD)/TSTEP
                T = T + T2
              END IF
            END IF
            IMAGE(IX,IRANK(IY)) = Y
          END DO
        END DO
        RETURN
      ELSE IF(METHOD.EQ.5) THEN
*
* Replace data by a sine wave 
*
        DO IX = XLO, XHI
          IF(MOD(IX-XLO+1,NCOUNT).EQ.0) THEN
            WRITE(INFORM,'(A,I4)') ' Reached X = ',IX
            CALL PAR_WRUSER(INFORM, IFAIL)
          END IF
          DO IY = 1, NY
            IMAGE(IX,IY) = AMP*SIN(TWOPI*(XDATA(IY)/PERIOD-PHI0))
          END DO
        END DO
        RETURN
      ELSE IF(METHOD.EQ.6) THEN
*
* Add sine wave to data 
*
        DO IX = XLO, XHI
          IF(MOD(IX-XLO+1,NCOUNT).EQ.0) THEN
            WRITE(INFORM,'(A,I4)') ' Reached X = ',IX
            CALL PAR_WRUSER(INFORM, IFAIL)
          END IF
          DO IY = 1, NY
            IMAGE(IX,IY) = IMAGE(IX,IY) + 
     &      AMP*SIN(TWOPI*(XDATA(IY)/PERIOD-PHI0))
          END DO
        END DO
        RETURN
      ELSE IF(METHOD.EQ.7) THEN
*
* Fit and subtract polynomial
*
        DO IX = XLO, XHI
          IF(MOD(IX-XLO+1,NCOUNT).EQ.0) THEN
            WRITE(INFORM,'(A,I4)') ' Reached X = ',IX
            CALL PAR_WRUSER(INFORM, IFAIL)
          END IF
          DO IY = 1, NY
            DATA(2,IY) = IMAGE(IX,IY) 
            DATA(3,IY) = ERRORS(IX,IY)
          END DO
          CALL LSQUAR(DATA, NY, NPOLY, ARC, CHISQ, XM, 1)
          DO IY = 1, NY
            IMAGE(IX,IY) = IMAGE(IX,IY) - POLY(ARC, NPOLY, DATA(1,IY))
          END DO
        END DO
        RETURN
      END IF
*
      DO IX = XLO, XHI
        IF(MOD(IX-XLO+1,NCOUNT).EQ.0) THEN
          WRITE(INFORM,'(A,I4)') ' Reached X = ',IX
          CALL PAR_WRUSER(INFORM, IFAIL)
        END IF
*
* Load data
*
        DO IY= 1, NY
          YDATA(IY) = IMAGE(IX,IY)
          YERR(IY) = ERRORS(IX,IY)
        END DO
*
* Fit sine wave (y = GAMMA + KVEL*SIN(TWOPI*(X/PERIOS-PHI0)))
*
        CALL SINFIT(XDATA, YDATA, YERR, NY, PERIOD, GAMMA, 
     &  KVEL, PHI0, COV, NP, F, IFAIL)
*
        IF(IFAIL.EQ.0) THEN
          DO IY = 1, NY
            Y = GAMMA + KVEL*SIN(TWOPI*(XDATA(IY)/PERIOD-PHI0))
            IF(METHOD.EQ.1) THEN
              IMAGE(IX,IY) = IMAGE(IX,IY) - Y
            ELSE IF(METHOD.EQ.2) THEN
              IMAGE(IX,IY) = Y + GAUSS2(0.,YERR(IY),ISEED)
            ELSE IF(METHOD.EQ.3) THEN
              IMAGE(IX,IY) = Y
            END IF
          END DO
        END IF
      END DO
      RETURN
      END
