*RADPOT
*
* RADPOT -- Plots radial profile of an image.
*
* This uses the same routine as IMSYMM to generate a median at a series
* of radii in an image. It is useful for seeing how the emissivity varies
* with radius.
*
* FIGARO parameters:
*
* IMAGE     -- The image file to load
* NIMAGE    -- The particular image to treat (if there is more than 1)
* STORE     -- .TRUE. to store profile
* PROFILE   -- File to store profile
* XCEN      -- X coordinate of centre of symmetry (km/s)
* YCEN      -- Y coordinate of centre of symmetry (km/s)
* VPLOT     -- TRUE if velocity plot wanted, else position plot
* DISTANCE  -- Distance to system in pc
* KW        -- White dwarf K velocity
* KR        -- Red dwarf K velocity
* ANGLE     -- Inclination angle
* PERIOD    -- Orbital period minutes
* XLOG      -- TRUE for logarithmic X axis
* YLOG      -- TRUE for logarithmic Y axis
* HARDCOPY  -- Hard copy? else soft
*
*RADPOT
      SUBROUTINE RADPOT
*
* R A D P O T
*
* Written by T.R. Marsh, plots radial profile of an image
*
*
      IMPLICIT NONE
*
* Functions
*
      INTEGER DYN_ELEMENT, DYN_INCREMENT
*
* Local variables
*
      INTEGER STATUS, IFAIL, NDIM, DIMS(3), NRAD
      INTEGER NELM, NX, NY, NIMAGE, NPIX
      REAL XCEN, YCEN, KW, VPIX, RIMAGE
      REAL ANGLE, PERIOD, DISTANCE, KR
      INCLUDE 'DYNAMIC_MEMORY'
      INTEGER ADDRESS, SLOT, PPTR, XPTR, DPTR
      LOGICAL FAULT, STORE, VPLOT, XLOG, YLOG, HARD
      CHARACTER*64 DTA_NAME, XAXIS(2), ZAXIS(2)
      CHARACTER*64 DEVICE
      INTEGER MAXPROF
      PARAMETER (MAXPROF = 1000)
      REAL XDATA(MAXPROF), YDATA(MAXPROF)
*

      FAULT = .TRUE.
*
* Get image region input file 
*
      CALL DSA_OPEN(STATUS)
      CALL DSA_INPUT('IMAGE', 'IMAGE', STATUS)
      CALL DSA_DATA_SIZE('IMAGE',3,NDIM,DIMS,NELM,STATUS)
      IF (STATUS.NE.0) GOTO 999
      IF(NDIM.NE.2 .AND. NDIM.NE.3) THEN
        CALL PAR_WRUSER('Invalid number of dimensions',STATUS)
        GOTO 999
      END IF
      NX = DIMS(1)
      NY = DIMS(2)
      IF(NDIM.EQ.3) THEN
        IF(DIMS(3).GT.1) THEN
          CALL PAR_RDVAL('NIMAGE',1.,REAL(DIMS(3)),1.,' ',RIMAGE)
          NIMAGE = NINT(RIMAGE)
        END IF
      ELSE
        NIMAGE = 1
      END IF      
*
* Get centre of symmetry. Centre of image = 0,0
*
      CALL DSA_READ_STRUCT_DEF('doppler',STATUS)
      CALL DSA_ELEMENT_NAME('IMAGE','VPIX',DTA_NAME,STATUS)
      CALL DTA_RDVARF(DTA_NAME,1,VPIX,IFAIL)
      IF(IFAIL.NE.0) THEN
        CALL PAR_WRUSER('Could not read VPIX. Assume = 1', STATUS)
        VPIX = 1.
      END IF
      CALL PAR_RDKEY('STORE', .FALSE., STORE)
      IF(STORE) THEN
        CALL DSA_OUTPUT('PROFILE','PROFILE','IMAGE',1,1,STATUS)
      END IF      
      CALL PAR_RDVAL('XCEN',-1.E10,1.E10,0.,' ',XCEN)
      CALL PAR_RDVAL('YCEN',-1.E10,1.E10,0.,' ',YCEN)
      CALL PAR_RDKEY('VPLOT', .TRUE., VPLOT)
      CALL PAR_RDVAL('DISTANCE', 0., 1.E10, 100., ' ', DISTANCE)
      IF(.NOT. VPLOT) THEN
        CALL PAR_RDVAL('KW', 0., 1.E10, 100., ' ', KW)
        CALL PAR_RDVAL('KR', 0., 1.E10, 400., ' ', KR)
        CALL PAR_RDVAL('ANGLE', 0., 90., 80., ' ', ANGLE)
        CALL PAR_RDVAL('PERIOD', 0., 1.E10, 200., ' ', PERIOD)
      END IF
      CALL PAR_RDKEY('XLOG', .FALSE., XLOG)
      CALL PAR_RDKEY('YLOG', .FALSE., YLOG)
      CALL PAR_RDKEY('HARDCOPY',.FALSE.,HARD)
      IF(HARD) THEN
        CALL VAR_GETCHR('HARD',0,0,DEVICE,STATUS)
      ELSE
        CALL VAR_GETCHR('SOFT',0,0,DEVICE,STATUS)
      END IF
      IF(STATUS.NE.0) THEN
        CALL PAR_WRUSER('Device not defined. Use SOFT or HARD', STATUS)
        GOTO 999
      END IF
*
* Convert coordinates to those used by subroutine
*
      XCEN = XCEN/VPIX + REAL(NX+1)/2.
      YCEN = YCEN/VPIX + REAL(NY+1)/2.
*
* Map input data
*
      CALL DSA_MAP_DATA('IMAGE','READ','FLOAT',ADDRESS,SLOT,STATUS)
      IF(STATUS.NE.0) GOTO 999
      DPTR = DYN_ELEMENT(ADDRESS) 
      NPIX = NX*NY*(NIMAGE-1)
      DPTR = DYN_INCREMENT(DPTR,'FLOAT',NPIX)
*
* Compute symmetric part of image
*
      NRAD = MAXPROF
      CALL FIGPROF(DYNAMIC_MEM(DPTR), NX, NY, XCEN, YCEN, VPLOT,
     &KW, KR, VPIX, DISTANCE, ANGLE, PERIOD, XLOG, YLOG, XDATA, 
     &YDATA, NRAD, DEVICE)
      IF(STORE) THEN
        CALL DSA_SIMPLE_OUTPUT('PROFILE','DATA, AXIS','FLOAT',1,
     &  NRAD,STATUS)
*
* Set profile data
*
        CALL DSA_MAP_DATA('PROFILE','WRITE','FLOAT',ADDRESS,SLOT,STATUS)
        IF(STATUS.NE.0) GOTO 999
        PPTR = DYN_ELEMENT(ADDRESS) 
        CALL TRANSF(YDATA,DYNAMIC_MEM(PPTR),NRAD)
*
* Set X axis data
*
        CALL DSA_MAP_AXIS_DATA('PROFILE',1,'WRITE','FLOAT',
     &  ADDRESS,SLOT,STATUS)
        IF(STATUS.NE.0) GOTO 999
        XPTR = DYN_ELEMENT(ADDRESS) 
        CALL TRANSF(XDATA,DYNAMIC_MEM(XPTR),NRAD)
*
* Set labels
*       
        XAXIS(1) = ' '
        ZAXIS(1) = ' ' 
        IF(VPLOT) THEN
          IF(XLOG) THEN
            XAXIS(2) = 'Log\d10\u(Velocity (km/s))'
          ELSE
            XAXIS(2) = 'Velocity (km/s)'
          END IF
          IF(YLOG) THEN
            ZAXIS(2) = 'Log\d10\u(I ergs/s/sr/(km/s)\u2\d)'
          ELSE
            ZAXIS(2) = 'I ergs/s/sr/(km/s)\u2\d'
          END IF
        ELSE
          IF(XLOG) THEN
            XAXIS(2) = 'Log\d10\u(Radius (cm))'
          ELSE
            XAXIS(2) = 'Radius (cm)'
          END IF
          IF(YLOG .AND. XLOG) THEN
            ZAXIS(2) = 'Log\d10\u(R\u2\d I (ergs/s/sr))'
          ELSE IF(.NOT.YLOG .AND. XLOG) THEN
            ZAXIS(2) = 'R\u2\d I (ergs/s/sr)'
          ELSE IF(YLOG .AND. .NOT.XLOG) THEN
            ZAXIS(2) = 'Log\d10\u(R I (ergs/s/sr/cm))'
          ELSE
            ZAXIS(2) = 'R I (ergs/s/sr/cm)'
          END IF
        END IF
* 
        CALL DSA_SET_DATA_INFO('PROFILE',2,ZAXIS,0,0.D0,STATUS)
        CALL DSA_SET_AXIS_INFO('PROFILE',1,2,XAXIS,0,0.D0,STATUS)
      END IF
      FAULT = .FALSE.
*
*     Tidy up
*
999   CONTINUE
      IF(FAULT) CALL FIG_SETERR
      CALL DSA_CLOSE(STATUS)
      RETURN
      END	

      SUBROUTINE TRANSF(INPUT, OUTPUT, N)
      INTEGER N, I
      REAL INPUT(N), OUTPUT(N)
      DO I = 1, N
        OUTPUT(I) = INPUT(I)
      END DO
      RETURN
      END 

      SUBROUTINE FIGPROF(IMAGE, NX, NY, XCEN, YCEN, VPLOT,
     &KW, KR, VPIX, DISTANCE, ANGLE, PERIOD, XLOG, YLOG, 
     &XDATA, YDATA, NRAD, DEVICE)
*
* Subroutine computes radial profile and then plots it
* DISTANCE in parsecs, ANGLE in degrees, PERIOD in minutes
*
      REAL IMAGE(NX,NY), XDATA(NRAD), YDATA(NRAD)
      REAL XCEN, YCEN, KW, KR, VPIX, DISTANCE, ANGLE, PERIOD
      LOGICAL VPLOT, XLOG, YLOG
      CHARACTER*(*) DEVICE
      CHARACTER*64 XLABEL, YLABEL
*
* Compute profile
*
      CALL PROFMED( IMAGE, NX, NY, XCEN, YCEN,
     &     RMIN, RMAX, YDATA, NRAD)
*
      VMIN = RMIN*VPIX
      VMAX = RMAX*VPIX
      TWOPI = 8.*ATAN(1.)
*
* Convert distance to cm, divide by 1.E10 to account for
* standard image units of 1.E-20 ergs/(km/s)**2/cm**2
*
      DISTANCE = DISTANCE*1.48E13*3600.*360./TWOPI/1.E10
      FACTOR = DISTANCE*DISTANCE
      DO I = 1, NRAD
        YDATA(I) = FACTOR*YDATA(I)
      END DO
      IF(.NOT. VPLOT) THEN
*
* Compute separation in cm
*
        SEPAR = (KW+KR)*PERIOD*6.E6/TWOPI/SIND(ANGLE)
        RFAC  = KR*(KR+KW)*SEPAR
      END IF
      DO I = 1, NRAD
        V = VMIN + (VMAX-VMIN)*REAL(I-1)/REAL(NRAD-1)
        IF(VPLOT) THEN
          XDATA(I) = V
        ELSE
          V  = MAX(1., V)
          XDATA(I) = RFAC/V**2
          IF(XLOG) THEN
            YDATA(I) = YDATA(I)*V**2
          ELSE
            YDATA(I) = YDATA(I)*V**4/RFAC
          END IF
        END IF
      END DO
      IF(XLOG) THEN
        DO I = 1, NRAD
          XDATA(I) = LOG10(MAX(1.E-30,XDATA(I)))
        END DO
      END IF
      IF(YLOG) THEN
        DO I = 1, NRAD
          YDATA(I) = LOG10(MAX(1.E-30,YDATA(I)))
        END DO
      END IF
      IF(VPLOT) THEN
        IF(XLOG) THEN
          XLABEL = 'Log\d10\u(Velocity (km/s))'
        ELSE
          XLABEL = 'Velocity (km/s)'
        END IF
        IF(YLOG) THEN
          YLABEL = 'Log\d10\u(Flux (ergs/s/sr/(km/s)\u2\d))'
        ELSE
          YLABEL = 'Flux (ergs/s/sr/(km/s)\u2\d)'
        END IF
      ELSE
        IF(XLOG) THEN
          XLABEL = 'Log\d10\u(Radius (cm))'
        ELSE
          XLABEL = 'Radius (cm)'
        END IF
        IF(YLOG .AND. XLOG) THEN
          YLABEL = 'Log\d10\u(R\u2\d I (ergs/s/sr))'
        ELSE IF(YLOG) THEN
          YLABEL = 'Log\d10\u(R I (ergs/s/sr/cm))'
        ELSE IF(XLOG) THEN
          YLABEL = 'R\u2\d I (ergs/s/sr)'
        ELSE
          YLABEL = 'R I (ergs/s/sr/cm)'
        END IF
      END IF
*
* Set plot limits
*
      XMIN = XDATA(1)
      XMAX = XMIN
      DO I = 1, NRAD
        XMIN = MIN(XMIN, XDATA(I))
        XMAX = MAX(XMAX, XDATA(I))
      END DO
      X1 = XMIN - (XMAX-XMIN)/20.
      X2 = XMAX + (XMAX-XMIN)/20.
      YMIN = YDATA(1)
      YMAX = YMIN
      DO I = 1, NRAD
        YMIN = MIN(YMIN, YDATA(I))
        YMAX = MAX(YMAX, YDATA(I))
      END DO
      Y1 = YMIN - (YMAX-YMIN)/20.
      Y2 = YMAX + (YMAX-YMIN)/20.
      CALL SETLIMS(X1,X2,Y1,Y2)
*
* Select device
*
      IFAIL = 0
      DO WHILE(IFAIL.NE.1)
        IFAIL = PGBEGIN(0,DEVICE,1,1)
        IF(IFAIL.NE.1) DEVICE = '?'
      END DO
      IF(XLOG .AND. YLOG) THEN
        CALL PGENV(X1,X2,Y1,Y2, 1, 0)
      ELSE
        CALL PGENV(X1,X2,Y1,Y2,0,0)
      END IF
      CALL PGLABEL(XLABEL, YLABEL, ' ')
      CALL PGLINE(NRAD, XDATA, YDATA)
      CALL PGEND
99    RETURN
      END
