*OPTGAM
*
* OPTGAM -- optimises systemic velocity of system.
*
* After you have iterated close to the final image you can use optgam to
* find the best gamma velocity by chi**2 minimisation in 1D. The revised
* value is stored in the image which is otherwise unchanged. You should
* re-iterate the image and in general a couple of times round the cycle 
* are needed.
*
* Parameters:
*
*   IMAGE     -- Input image file to load and which will contain the
*                modified systemic velocity.
*   DATA      -- Data file containing data and errors used to generate IMAGE
*   STEP      -- Initial step size to take (km/s)
*   DELTA     -- Accuracy wanted in final answer (km/s)
*
*OPTGAM
      SUBROUTINE OPTGAM(STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'DAT_PAR'
      INCLUDE 'doppler.inc'
      INCLUDE 'CNF_PAR'
      INTEGER STATUS, LBND(2), UBND(2), NDIM, DI
     :M(3)
      INTEGER IMAGE, EL, I, DATA, DWPTR, PPTR, HPTR
      INTEGER PLACE, PHASES, WIDTHS, PWPTR, IPTR, DPTR
      INTEGER EPTR, WORK, WPTR
      REAL STEP, DELTA
      DOUBLE PRECISION HJD0, PERIOD
      CHARACTER*16 USE
      CHARACTER*(DAT__SZLOC) LOC
      LOGICAL SAMECI, BASE
C
      IF(STATUS.NE.SAI__OK) RETURN
      CALL NDF_BEGIN
C
C     Open image file 
C
      CALL NDF_ASSOC('IMAGE','UPDATE',IMAGE, STATUS)
      CALL NDF_ISBAS(IMAGE, BASE, STATUS)
      IF(.NOT.BASE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','memit cannot handle NDF sections.',
     &        STATUS)
      END IF
      CALL NDF_DIM(IMAGE,3,DIM,NDIM,STATUS)
      IF(STATUS.EQ.SAI__OK) THEN
         IF(NDIM.EQ.1) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ',
     &           'Images must be 2 or 3D', STATUS)
         ELSE IF(DIM(1).NE.DIM(2)) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Image(s) not square', STATUS)
         ELSE
            NSIDE  = DIM(1)
            NIMAGE = DIM(3)
         END IF
      END IF
C
C     Get header items
C
      CALL NDF_XLOC(IMAGE,'DOPPLER_MAP','READ',LOC,STATUS)
C
      CALL CMP_GET0R(LOC,'VPIX',VPIX,STATUS)
      CALL CMP_GET1R(LOC, 'WAVE',MAXWV,WAVE,NWAVE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NWAVE.LT.NIMAGE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Too few wavelengths found', STATUS)
      END IF
C
C     Read scale factors and correspondence arrays.
C     
      IF(STATUS.EQ.SAI__OK .AND. NIMAGE.NE.NWAVE) THEN
         IF(NIMAGE.GT.1) THEN
            CALL CMP_GET1I(LOC, 'NWHICH',NWAVE,NWHICH,EL,STATUS)
            IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
               STATUS = SAI__ERROR
               CALL ERR_REP(' ','NWHICH/WAVE: conflicting sizes.',
     &              STATUS)
            END IF
         ELSE
            DO I = 1, NWAVE
               NWHICH(I) = 1
            END DO
         END IF
         CALL CMP_GET1R(LOC, 'SCALE',NWAVE,SCALE,EL,STATUS)
         IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','SCALE/WAVE: conflicting sizes.',
     &           STATUS)
         END IF
      ELSE IF(STATUS.EQ.SAI__OK) THEN
         DO I = 1, NWAVE
            SCALE(I)  = 1.
            NWHICH(I) = I
         END DO
      END IF
C
      CALL CMP_GET0R(LOC,'FWHM',FWHM,STATUS)
      CALL CMP_GET0R(LOC,'GAMMA',GAMMA,STATUS)
      CALL CMP_GET0R(LOC,'OFFSET',OFFSET,STATUS)
      CALL CMP_GET0I(LOC,'NDIV',NDIV,STATUS)
      CALL CMP_GET0C(LOC,'USE',USE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. (.NOT.SAMECI(USE,'Phases') 
     &     .AND. .NOT.SAMECI(USE,'HJDs')) ) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','USE has unrecognised value',STATUS)
      END IF
      CALL CMP_GET0I(LOC,'NSUB',NSUB,STATUS)
      CALL DAT_ANNUL(LOC, STATUS)
C
C     Open data file to provide phases, wavelength scale etc
C     for computed data
C
      CALL NDF_ASSOC('DATA', 'READ', DATA, STATUS)
      CALL NDF_ISBAS(DATA, BASE, STATUS)
      IF(.NOT.BASE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','memit cannot handle NDF sections.',
     &        STATUS)
      END IF
      CALL NDF_DIM(DATA,2,DIM,NDIM,STATUS)
      NPIX  = DIM(1)
      NSPEC = DIM(2)
C     
C     Read header
C     
      CALL NDF_XLOC(DATA,'DOPPLER_DATA','READ',LOC,STATUS)
C     
      CALL CMP_GET0I(LOC,'NARC',NARC,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NARC.NE.-2) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ',
     &        'Doppler imaging programs only use'//
     &        ' log wavelength scales.', STATUS)
      END IF
      CALL CMP_GET1D(LOC, 'ARC', 2, ARC, EL, STATUS)
      IF(STATUS.EQ.SAI__OK .AND. EL.NE.2) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Could not read 2 arc coefficients',
     &        STATUS)
      END IF
C     
C     Get orbital period and dwells
C     
      CALL CMP_GET0D(LOC,'PERIOD',PERIOD,STATUS)
      CALL CMP_MAPV(LOC, 'DWELL', '_REAL', 'READ', DWPTR, 
     &     EL, STATUS)
C     
C     Get phase either from phases array or HJDs
C     
      IF(SAMECI(USE,'Phases')) THEN
         CALL CMP_MAPV(LOC,'PHASES','_DOUBLE','READ',PPTR, 
     &        EL, STATUS)
      ELSE IF(SAMECI(USE,'HJDs')) THEN
         CALL CMP_GET0D(LOC,'HJD0',HJD0,STATUS)
         CALL CMP_MAPV(LOC, 'HJD', '_DOUBLE', 'READ', HPTR, 
     &        EL, STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_DOUBLE',1,1,NSPEC,PLACE,PHASES,STATUS)
         CALL NDF_MAP(PHASES,'Data','_DOUBLE','WRITE',PPTR,
     &        EL,STATUS)
         CALL PHCOMP(%VAL(CNF_PVAL(HPTR)), NSPEC, %VAL(CNF_PVAL(PPTR)),
     &        HJD0, PERIOD,STATUS)
      END IF
      CALL DAT_ANNUL(LOC, STATUS)
      CALL NDF_TEMP(PLACE,STATUS)
      CALL NDF_NEW('_REAL',1,1,NSPEC,PLACE,WIDTHS,STATUS)
      CALL NDF_MAP(WIDTHS,'Data','_REAL','WRITE',PWPTR,
     &     EL,STATUS)
      CALL PWCOMP(%VAL(CNF_PVAL(DWPTR)),NSPEC,%VAL(CNF_PVAL(PWPTR)),
     :            PERIOD,STATUS)
C
      CALL PAR_GDR0R('STEP',50.,0.,1.E3,.FALSE.,STEP,STATUS)
      CALL PAR_GDR0R('DELTA',1.,0.,1.E3,.FALSE.,DELTA,STATUS)
C
C     Map
C
      CALL NDF_MAP(IMAGE,'Data','_REAL','READ',IPTR,EL,STATUS)
      CALL NDF_MAP(DATA,'Data','_REAL','READ',DPTR,EL,STATUS)
      CALL NDF_MAP(DATA,'Errors','_REAL','READ',EPTR,EL,STATUS)
C
C     Get workspace equal in size to data array
C
      CALL NDF_TEMP(PLACE,STATUS)
      CALL NDF_BOUND(DATA,2,LBND,UBND,NDIM,STATUS)
      CALL NDF_NEW('_REAL',NDIM,LBND,UBND,PLACE,WORK,STATUS)
      CALL NDF_MAP(WORK,'Data','_REAL','WRITE',WPTR,EL,STATUS)
C
C     Compute optimum gamma
C
      OPFIR = .TRUE.
      CALL OPT_GAM(%VAL(CNF_PVAL(IPTR)), %VAL(CNF_PVAL(DPTR)), 
     :             %VAL(CNF_PVAL(EPTR)),
     &     %VAL(CNF_PVAL(WPTR)), %VAL(CNF_PVAL(PPTR)), 
     :     %VAL(CNF_PVAL(PWPTR)), STEP,
     &     DELTA, STATUS)
C
C     Store new gamma velocity
C     
      CALL NDF_XPT0R(GAMMA,IMAGE,'DOPPLER_MAP','GAMMA',STATUS)      
      CALL NDF_END(STATUS)
      RETURN
      END	

      SUBROUTINE OPT_GAM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &PWIDTH,STEP, DELTA, STATUS)
C
C Subroutine to optimise gamma velocity by minimising chi**2
C This uses a golden section search which is slow but reliable
C
C R*4 IMAGE(NSIDE*NISDE*NIMAGE) -- NIMAGE images of NSIDE*NSIDE dimension
C R*4 DATA(NPIX*NSPEC)   -- Data NPIX in X by NSPEC in Y
C R*4 ERRORS(NPIX*NSPEC) -- 1-sigma uncertainties on DATA
C
C R*4 WORK(NPIX*NSPEC)     Work space array for computed data
C R*4 PHASES(NSPEC) -- Phases and phase widths of
C R*4 PWIDTH(NSPEC)    exposures
C R*4 STEP  -- Initial step size to take.
C R*4 DELTA -- Accuracy of result. Minimum guaranteed to be within DELTA km/s
C              of the value of GAMMA returned via common block.
C R*4 IFAIL = 0, OK. Otherwise error.
C
C All other arguments (e.g. NIMAGE) are passed by common blocks inside
C doppler.inc)
C
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'doppler.inc'
      INTEGER STATUS
      REAL IMAGE(NSIDE,NSIDE,NIMAGE), DATA(NPIX,NSPEC)
      REAL ERRORS(NPIX,NSPEC), WORK(NPIX,NSPEC)
      REAL STEP, DELTA, PHASES(NSPEC), PWIDTH(NSPEC)
      REAL G1, G2, G3, CHICOM, C1, C2, C3
      REAL CNEW, GOLD, ADD
C
      IF(STATUS.NE.SAI__OK) RETURN
C
C     First locate two points which bracket minimum
C
      GOLD  = (3.-SQRT(5.))/2.
      ADD   = STEP
      GAMMA = GAMMA - ADD
      G1    = GAMMA
      C1    = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &     PWIDTH, STATUS)
      IF(STATUS.NE.SAI__OK) RETURN
      GAMMA = GAMMA + ADD
      G2    = GAMMA
      C2    = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &     PWIDTH, STATUS)
      IF(STATUS.NE.SAI__OK) RETURN
      IF(C1.LT.C2) THEN
         GAMMA = GAMMA - 2.*ADD
         ADD   = ADD/GOLD
         C3    = C2
         G3    = G2
         C2    = C1
         G2    = G1
         G1    = GAMMA
         C1    = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &        PWIDTH, STATUS)
         IF(STATUS.NE.SAI__OK) RETURN
      ELSE
         GAMMA = GAMMA + ADD
         ADD   = ADD/GOLD
         G3    = GAMMA
         C3    = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &        PWIDTH, STATUS)
         IF(STATUS.NE.SAI__OK) RETURN
      END IF
10    CONTINUE
      IF(C1.LE.C2 .AND. C2.LE.C3) THEN
         GAMMA = GAMMA - ADD
         ADD   = ADD/GOLD
         C3    = C2
         G3    = G2
         C2    = C1
         G2    = G1
         G1    = GAMMA
         C1    = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &        PWIDTH, STATUS)
         IF(STATUS.NE.SAI__OK) RETURN
         GOTO 10
      ELSE IF(C1.GE.C2 .AND. C2.GE.C3) THEN
         GAMMA = GAMMA + ADD
         ADD   = ADD/GOLD
         C1    = C2
         G1    = G2
         C2    = C3
         G2    = G3
         G3    = GAMMA
         C3    = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &        PWIDTH, STATUS)
         IF(STATUS.NE.SAI__OK) RETURN
         GOTO 10
      END IF
C
C     We now have three increasing values of Gamma with the minimum Chi**2 
C     for the central value. Golden section search until two outer points
C     are separated by less than 2*DELTA
C
 20   CONTINUE
      CALL MSG_OUT(' ','Minimum bracketed', STATUS)
      CALL MSG_SETR('GAMMA',G1)
      CALL MSG_SETR('CHISQ',C1)
      CALL MSG_OUT(' ',
     &     'Point 1, gamma = ^GAMMA, Chi squared = ^CHISQ',STATUS)
      CALL MSG_SETR('GAMMA',G2)
      CALL MSG_SETR('CHISQ',C2)
      CALL MSG_OUT(' ',
     &     'Point 2, gamma = ^GAMMA, Chi squared = ^CHISQ',STATUS)
      CALL MSG_SETR('GAMMA',G3)
      CALL MSG_SETR('CHISQ',C3)
      CALL MSG_OUT(' ',
     &     'Point 3, gamma = ^GAMMA, Chi squared = ^CHISQ',STATUS)
C     
      IF(G3-G2.GT.G2-G1) THEN
         GAMMA = G2 + GOLD*(G3-G2)
         CNEW = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &        PWIDTH, STATUS)
         IF(STATUS.NE.SAI__OK) RETURN
         IF(CNEW.GT.C2) THEN
            G3 = GAMMA
            C3 = CNEW
         ELSE
            G1 = G2
            C1 = C2
            G2 = GAMMA
            C2 = CNEW
         END IF
      ELSE
         GAMMA = G2 - GOLD*(G2-G1)
         CNEW = CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &        PWIDTH, STATUS)
         IF(STATUS.NE.SAI__OK) RETURN
         IF(CNEW.GT.C2) THEN
            G1 = GAMMA
            C1 = CNEW
         ELSE
            G3 = G2
            C3 = C2
            G2 = GAMMA
            C2 = CNEW
         END IF
      END IF
      IF(G3-G1.GT.2.*DELTA) GOTO 20
      CALL MSG_SETR('GAMMA',G2)
      CALL MSG_SETR('CHISQ',C2)
      CALL MSG_OUT(' ',
     &     'Finished, gamma = ^GAMMA, Chi squared = ^CHISQ',STATUS)
      GAMMA = G2
      RETURN
      END
      
      REAL FUNCTION CHICOM(IMAGE, DATA, ERRORS, WORK, PHASES, PWIDTH,
     &     STATUS)
C
C     Computes chi**2
C
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'doppler.inc'
      INTEGER STATUS
      REAL IMAGE(NSIDE,NSIDE,NIMAGE), DATA(NPIX,NSPEC)
      REAL ERRORS(NPIX,NSPEC), WORK(NPIX,NSPEC), CHISQRD
      REAL PHASES(NSPEC), PWIDTH(NSPEC) 
*
      IF(STATUS.NE.SAI__OK) RETURN
      CALL OP(IMAGE, WORK, PHASES, PWIDTH, STATUS)
      IF(STATUS.NE.SAI__OK) RETURN
      CHICOM = CHISQRD(DATA,WORK,ERRORS,NPIX*NSPEC,STATUS)
      RETURN
      END
