*ECLIPSE
*
* ECLIPSE -- like 'COMDAT' computes data equivalent to a particular image,
*            but also accounts for eclipses and therefore requires you to
*            specify parameters that allow it to convert from velocity to
*            position (it assumes a "keplerian" flow).
*
* Parameters:
*
*   IMAGE     -- The image file to load
*   TEMPLATE  -- TRUE if template data file is to be used otherwise the user
*                must define pixels sizes etc.
*   If TEMPLATE
*
*     DATA      -- Data file containing example format data
*
*   Else
*
*     NPIX      -- Number of pixels/spectrum        
*     NSPEC     -- Number of spectra
*     VDATA     -- Velocity km/s/pixel
*     WCEN      -- Central wavelength (Angstroms)
*     WIDTH     -- Phase width of exposures
*
*   OUTPUT    -- Calculated data file name
*   KW        -- White dwarf's semi-amplitude
*   KR        -- Red star's semi-amplitude
*   IANGLE    -- Inclination angle
*   OVER      -- Over sampling factor. This makes the program
*                over-sample the velocity grid to produce
*                smoother time dependence. You should be very careful
*                with this parameter as the computation time scales as
*                OVER**2.
*   RMAX      -- maximum distance in terms of RL1 to attempt computation
*                of eclipse. Any emission that translates to a place
*                beyond this radius is assumed not eclipsed. You could
*                allow this to extend beyond 1, but I don't know how
*                it will cope with points that end up inside the mass
*                donor.
*
*   If NADD
*
*     CONT      -- Effective continuum for noise computation
*     NOISE     -- Signal-to-noise in effective continuum
*     SEED      -- Seed integer to start noise generator if NADD
*
*ECLIPSE
      SUBROUTINE ECLIPSE(STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'DAT_PAR'
      INCLUDE 'doppler.inc'
      INCLUDE 'CNF_PAR'
      INTEGER STATUS, DIM(3), NDIM, IMAGE, EL, I
     :, DATA
      INTEGER LBND(2), UBND(2), DWPTR, PPTR, HPTR, PLACE
      INTEGER PHASES, WIDTHS, PWPTR, OUTPUT, SEED
      INTEGER CEPTR, CDPTR, IPTR, DPTR, EPTR, OVER
      REAL VDATA, WIDTH, VLIGHT, CGS, WCEN, CONT, NOISE
      REAL CHISQ, CHISQRD, KW, KR, IANGLE, RMAX
      DOUBLE PRECISION PERIOD, HJD0
      CHARACTER*(DAT__SZLOC) LOC
      CHARACTER*16 USE
      LOGICAL SAMECI, TEMPLATE, DERROR, NADD
C
      IF(STATUS.NE.SAI__OK) RETURN
      CALL NDF_BEGIN
C
C     Open image file 
C
      CALL NDF_ASSOC('IMAGE','READ',IMAGE, STATUS)
C
      CALL NDF_DIM(IMAGE,3,DIM,NDIM,STATUS)
      IF(STATUS.EQ.SAI__OK) THEN
         IF(NDIM.EQ.1) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ',
     &           'Images must be 2 or 3D', STATUS)
         ELSE IF(DIM(1).NE.DIM(2)) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Image(s) not square', STATUS)
         ELSE
            NSIDE  = DIM(1)
            NIMAGE = DIM(3)
         END IF
      END IF
C
C     Get header items
C
      CALL NDF_XLOC(IMAGE,'DOPPLER_MAP','READ',LOC,STATUS)
C
      CALL CMP_GET0R(LOC,'VPIX',VPIX,STATUS)
      CALL CMP_GET1R(LOC, 'WAVE',MAXWV,WAVE,NWAVE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NWAVE.LT.NIMAGE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Too few wavelengths found', STATUS)
      END IF
C     
C     Read scale factors and correspondence arrays.
C     
      IF(STATUS.EQ.SAI__OK .AND. NIMAGE.NE.NWAVE) THEN
         IF(NIMAGE.GT.1) THEN
            CALL CMP_GET1I(LOC, 'NWHICH',NWAVE,NWHICH,EL,STATUS)
            IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
               STATUS = SAI__ERROR
               CALL ERR_REP(' ','NWHICH/WAVE: conflicting sizes.',
     &              STATUS)
            END IF
         ELSE
            DO I = 1, NWAVE
               NWHICH(I) = 1
            END DO
         END IF
         CALL CMP_GET1R(LOC, 'SCALE',NWAVE,SCALE,EL,STATUS)
         IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','SCALE/WAVE: conflicting sizes.',
     &           STATUS)
         END IF
      ELSE IF(STATUS.EQ.SAI__OK) THEN
         DO I = 1, NWAVE
            SCALE(I)  = 1.
            NWHICH(I) = I
         END DO
      END IF
C     
      CALL CMP_GET0R(LOC,'FWHM',FWHM,STATUS)
      CALL CMP_GET0R(LOC,'GAMMA',GAMMA,STATUS)
      CALL CMP_GET0R(LOC,'OFFSET',OFFSET,STATUS)
      CALL CMP_GET0I(LOC,'NDIV',NDIV,STATUS)
      CALL CMP_GET0C(LOC,'USE',USE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. (.NOT.SAMECI(USE,'Phases') 
     &     .AND. .NOT.SAMECI(USE,'HJDs')) ) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','USE has unrecognised value',STATUS)
      END IF
      CALL CMP_GET0I(LOC,'NSUB',NSUB,STATUS)
      CALL DAT_ANNUL(LOC, STATUS)
C     
C     Open data file to provide phases, wavelength scale etc
C     for computed data
C     
      CALL PAR_GET0L('TEMPLATE', TEMPLATE, STATUS)
      IF(TEMPLATE) THEN
         CALL NDF_ASSOC('DATA', 'READ', DATA, STATUS)
         CALL NDF_DIM(DATA,2,DIM,NDIM,STATUS)
         NPIX  = DIM(1)
         NSPEC = DIM(2)
         CALL NDF_STATE(DATA,'VARIANCE',DERROR,STATUS)
C     
C     Read header
C     
         CALL NDF_XLOC(DATA,'DOPPLER_DATA','READ',LOC,STATUS)
C     
         CALL CMP_GET0I(LOC,'NARC',NARC,STATUS)
         IF(STATUS.EQ.SAI__OK .AND. NARC.NE.-2) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ',
     &           'Doppler imaging programs only use'//
     &           ' log wavelength scales.', STATUS)
         END IF
         CALL CMP_GET1D(LOC, 'ARC', 2, ARC, EL, STATUS)
         IF(STATUS.EQ.SAI__OK .AND. EL.NE.2) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Could not read 2 arc coefficients',
     &           STATUS)
         END IF
C     
C     Get orbital period and dwells
C     
         CALL CMP_GET0D(LOC,'PERIOD',PERIOD,STATUS)
         CALL CMP_MAPV(LOC, 'DWELL', '_REAL', 'READ', DWPTR, 
     &        EL, STATUS)
C     
C     Get phase either from phases array or HJDs
C     
         IF(SAMECI(USE,'Phases')) THEN
            CALL CMP_MAPV(LOC,'PHASES','_DOUBLE','READ',PPTR, 
     &           EL, STATUS)
         ELSE IF(SAMECI(USE,'HJDs')) THEN
            CALL CMP_GET0D(LOC,'HJD0',HJD0,STATUS)
            CALL CMP_MAPV(LOC, 'HJD', '_DOUBLE', 'READ', HPTR, 
     &           EL, STATUS)
            CALL NDF_TEMP(PLACE,STATUS)
            CALL NDF_NEW('_DOUBLE',1,1,NSPEC,PLACE,PHASES,STATUS)
            CALL NDF_MAP(PHASES,'Data','_DOUBLE','WRITE',PPTR,
     &           EL,STATUS)
            CALL PHCOMP(%VAL(CNF_PVAL(HPTR)), NSPEC, 
     :                  %VAL(CNF_PVAL(PPTR)),
     &           HJD0, PERIOD,STATUS)
         END IF
         CALL DAT_ANNUL(LOC, STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_REAL',1,1,NSPEC,PLACE,WIDTHS,STATUS)
         CALL NDF_MAP(WIDTHS,'Data','_REAL','WRITE',PWPTR,
     &        EL,STATUS)
         CALL PWCOMP(%VAL(CNF_PVAL(DWPTR)),NSPEC,%VAL(CNF_PVAL(PWPTR)),
     :               PERIOD,STATUS)
         CALL NDF_PROP(DATA,' ','OUTPUT', OUTPUT, STATUS)
      ELSE 
C
C     Get enough parameters to fake up data 
C
         CALL PAR_GDR0I('NPIX',NSIDE,1,10000,.FALSE.,NPIX,STATUS)
         CALL PAR_GDR0I('NSPEC',50,1,10000,.FALSE.,NSPEC,STATUS)
         CALL PAR_GDR0R('VDATA',VPIX,1.E-4,1.E4,.FALSE.,VDATA,STATUS)
         CALL PAR_GDR0R('WCEN',WAVE(1),1.E-4,1.E6,.FALSE.,WCEN,STATUS)
         CALL PAR_GDR0R('WIDTH',0.1,0.,1.,.FALSE.,WIDTH,STATUS)
         IF(STATUS.EQ.SAI__OK) THEN
            VLIGHT = CGS('C')/1.E5
            ARC(2) = DBLE(NPIX)*VDATA/VLIGHT
            ARC(1) = LOG(WCEN)-ARC(2)*DBLE(NPIX+1)/2.D0/DBLE(NPIX)
            NARC   = -2
            HJD0   = 2448600.
            PERIOD = 0.1
            LBND(1) = 1
            UBND(1) = NPIX
            IF(NSPEC.GT.1) THEN
               NDIM    = 2
               LBND(2) = 1
               UBND(2) = NSPEC
            ELSE
               NDIM = 1
            END IF
            DERROR = .FALSE.
         END IF
C
C     Create structure OUTPUT
C
         CALL NDF_CREAT('OUTPUT','_REAL',NDIM,LBND,UBND,OUTPUT,STATUS)
C     
C     Create doppler data extension
C     
         CALL NDF_XNEW(OUTPUT,'DOPPLER_DATA','Ext',0,0,LOC,STATUS)
         CALL CMP_MOD(LOC, 'HJD0',  '_DOUBLE', 0, 0, STATUS)
         CALL CMP_PUT0D(LOC,'HJD0',HJD0,STATUS)
         CALL CMP_MOD(LOC, 'PERIOD',  '_DOUBLE', 0, 0, STATUS)
         CALL CMP_PUT0D(LOC,'PERIOD',PERIOD,STATUS)
         CALL CMP_MOD(LOC, 'NARC',  '_INTEGER', 0, 0, STATUS)
         CALL CMP_PUT0I(LOC,'NARC',NARC,STATUS)
         CALL CMP_MOD(LOC, 'ARC',  '_DOUBLE', 1, ABS(NARC), STATUS)
         CALL CMP_PUT1D(LOC,'ARC',ABS(NARC),ARC,STATUS)
C
         CALL CMP_MOD(LOC, 'HJD',  '_DOUBLE', 1, NSPEC, STATUS)
         CALL CMP_MAPV(LOC, 'HJD', '_DOUBLE', 'WRITE', HPTR, 
     &        EL, STATUS)
         CALL CMP_MOD(LOC, 'PHASES',  '_DOUBLE', 1, NSPEC, STATUS)
         CALL CMP_MAPV(LOC, 'PHASES', '_DOUBLE', 'WRITE', PPTR, 
     &        EL, STATUS)
         CALL CMP_MOD(LOC, 'DWELL',  '_REAL', 1, NSPEC, STATUS)
         CALL CMP_MAPV(LOC, 'DWELL', '_REAL', 'WRITE', DWPTR, 
     &        EL, STATUS)
         CALL DAT_ANNUL(LOC, STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_REAL',1,1,NSPEC,PLACE,WIDTHS,STATUS)
         CALL NDF_MAP(WIDTHS,'Data','_REAL','WRITE',PWPTR,
     &        EL,STATUS)
C
C     Write out
C
         DO I = 1, NSPEC
            CALL DMOVE(DBLE(I)/DBLE(NSPEC), 
     &           %VAL(CNF_PVAL(PPTR)), I, NSPEC,STATUS)
            CALL DMOVE(HJD0 + PERIOD*DBLE(I)/DBLE(NSPEC), 
     &           %VAL(CNF_PVAL(HPTR)), I, NSPEC,STATUS)
            CALL RMOVE(SNGL(WIDTH*PERIOD*24.*3600.), 
     &           %VAL(CNF_PVAL(DWPTR)), I, NSPEC,STATUS)
            CALL RMOVE(WIDTH, %VAL(CNF_PVAL(PWPTR)), I, NSPEC,STATUS)
        END DO
      END IF
C
C     Get binary parameters
C
      CALL PAR_GDR0R('KW',100.,0.,10000.,.FALSE.,KW,STATUS)
      CALL PAR_GDR0R('KR',300.,0.,10000.,.FALSE.,KR,STATUS)
      CALL PAR_GDR0R('IANGLE',75.,0.,90.,.FALSE.,IANGLE,STATUS)
      CALL PAR_GDR0I('OVER',1,1,100,.FALSE.,OVER,STATUS)
      CALL PAR_GDR0R('RMAX',1.,0.,100.,.FALSE.,RMAX,STATUS)
C
C     Should noise be added or not?, then map files.
C     
      CALL PAR_GET0L('NADD',NADD,STATUS)
      IF(NADD) THEN
         IF(.NOT.DERROR) THEN
            CALL PAR_GDR0R('CONT',1.,0.,1.E20,.FALSE.,CONT,STATUS)
            CALL PAR_GDR0R('NOISE',10.,1.E-10,1.E20,.FALSE.,
     &           NOISE,STATUS)
         END IF
         CALL PAR_GDR0I('SEED',12345,1,9999999,.FALSE.,SEED,STATUS)
         SEED = - SEED
         CALL NDF_MAP(OUTPUT,'Errors','_REAL','WRITE',CEPTR,EL,STATUS)
      END IF
      CALL NDF_MAP(OUTPUT,'Data','_REAL','WRITE',CDPTR,EL,STATUS)
      CALL NDF_MAP(IMAGE,'Data','_REAL','READ',IPTR,EL,STATUS)
C
C     Compute data
C
      CALL EDATA_COMP(%VAL(CNF_PVAL(IPTR)),%VAL(CNF_PVAL(CDPTR)),
     :                %VAL(CNF_PVAL(PPTR)),%VAL(CNF_PVAL(PWPTR)),
     &     KW,KR,IANGLE,OVER,RMAX,STATUS)
C
C     Compute and report Chi**2 if possible
C
      IF(DERROR) THEN
         CALL NDF_MAP(DATA,'Data','_REAL','READ',DPTR,EL,STATUS)
         CALL NDF_MAP(DATA,'Errors','_REAL','READ',
     &        EPTR,EL,STATUS)
         CHISQ = CHISQRD(%VAL(CNF_PVAL(DPTR)),%VAL(CNF_PVAL(CDPTR)),
     :                   %VAL(CNF_PVAL(EPTR)),
     &        NSPEC*NPIX,STATUS)
         CALL MSG_SETR('CHISQ',CHISQ)
         CALL MSG_OUT(' ','Chi-squared = ^CHISQ',STATUS)
      END IF
C
C     Add noise if wanted.
C
      IF(NADD) THEN
         IF(DERROR) THEN
            CALL ADD_NS1(%VAL(CNF_PVAL(CDPTR)), %VAL(CNF_PVAL(EPTR)),
     &           %VAL(CNF_PVAL(CEPTR)), NPIX, NSPEC, SEED, STATUS)
         ELSE
            CALL ADD_NS2(%VAL(CNF_PVAL(CDPTR)), %VAL(CNF_PVAL(CEPTR)),
     &           CONT, NOISE, NPIX, NSPEC, SEED, STATUS)
         END IF
      END IF
      CALL NDF_END(STATUS)
      RETURN
      END	

      
      SUBROUTINE EDATA_COMP(DISC,DATA,PHASES,PWIDTH,KW,KR,
     &     IANGLE,OVER,RMAX,STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'doppler.inc'
      INTEGER STATUS, IFAIL, I, J, K, L, M, N, NFPIX, NADD
      INTEGER OVER
      REAL DISC(NSIDE,NSIDE,NIMAGE), DATA(NPIX,NSPEC)
      REAL PWIDTH(NSPEC), VLIGHT, VDPIX, KW, KR, IANGLE
      DOUBLE PRECISION SUM, TWOPI, PHASES(NSPEC), DQ, RL1
      DOUBLE PRECISION WZERO, PHASE
      REAL EARTH(3), PVEC(3), Q, RMAX, VOVER, CGS, EFAC
      REAL AREA, VFPIX, VFZER, VZERO, VSSQ, COSI, SINI
      REAL COSP, SINP, PWGT, VREST, SCARE, VX, VY, R, SCL
      LOGICAL BLINK, VIS
C     
      IF(STATUS.NE.SAI__OK) RETURN
      
      DQ = KW/KR
      Q = REAL(DQ)
      CALL INLAG(DQ, RL1, IFAIL)
      VLIGHT = CGS('C')/1.E5
      TWOPI = 8.D0*ATAN(1.D0)
      DO J = 1, NSPEC
         DO I = 1, NPIX
            DATA(I,J) = 0.
         END DO
      END DO
C
C     Compute smoothing array.
C     
      VDPIX   = REAL(VLIGHT*ARC(2)/DBLE(NPIX))
      NSMOOTH = INT(REAL(NDIV)*DEVMAX*FWHM/2.3548/VDPIX)+1
      IF(NSMOOTH.GT.MAXSM) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Smoothing buffer too small.',
     &        STATUS)
         RETURN
      END IF
      SUM = 0.D0
      EFAC = REAL(4.D0*LOG(2.D0)/(DBLE(NDIV)*DBLE(FWHM/VDPIX))**2)
      DO I = -NSMOOTH, NSMOOTH
         SMOOTH(I) = EXP(-EFAC*DBLE(I)**2)
         SUM = SUM + SMOOTH(I)
      END DO
      DO I = -NSMOOTH, NSMOOTH
         SMOOTH(I) = SMOOTH(I)/SUM
      END DO
C     
C     WZERO = Central wavelength
C     AREA  = (km/s)**2/pixel with other scaling factors
C     
      WZERO = ARC(1)+ARC(2)*DBLE(NPIX+1)/2/DBLE(NPIX)
      AREA  = REAL(VPIX*VPIX*1.E-7*EXP(WZERO)/VDPIX/OVER/OVER)
C
C     Fine version of data array NDIV times more pixels
C
      VFPIX = VDPIX/REAL(NDIV)
      NFPIX = NDIV*NPIX
      IF(NFPIX.GT.MAXFBF) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','FAST buffer too small',STATUS)
         RETURN
      END IF
      VFZER = - VFPIX*REAL(NFPIX+1)/2.
C
C     velocity in image calculated as VZERO+VOVER*I 
C     with I from 1 to NSIDE*OVER where OVER is an
C     integer allowing finer sampling
C
      VOVER   = VPIX/OVER
      VZERO   = -VPIX*REAL(NSIDE)/2.-VOVER/2.
      VSSQ    = KR*(KW+KR)
      PVEC(3) = 0.
      COSI    = REAL(COS(TWOPI*IANGLE/360.))
      SINI    = REAL(SIN(TWOPI*IANGLE/360.))
C     
C     Loop through each spectrum
C  
      DO N = 1, NSPEC
C     
C     Initialise projection buffer
C     
         DO I = 1, NFPIX
            FBF(I) = 0.D0
         END DO
C     
C     Loop through each sub-division of phase bin
C     
         DO M = 1, MAX(1,NSUB)
            PHASE = PHASES(N) + DBLE(OFFSET) + 
     &           DBLE(PWIDTH(N))*(DBLE(M)-DBLE(1+NSUB)/2.)/
     &           DBLE(MAX(1,NSUB-1))                
            COSP  = SNGL(COS(TWOPI*PHASE))
            SINP  = SNGL(SIN(TWOPI*PHASE))
            EARTH(1) =  COSP*SINI
            EARTH(2) = -SINP*SINI
            EARTH(3) =  COSI
C     
C     Trapezoidal weights for projections to average over
C     phase bin.
C     
            IF(NSUB.LE.1) THEN
               PWGT = AREA
            ELSE IF(M.EQ.1 .OR. M.EQ.NSUB) THEN
               PWGT = 0.5*AREA/REAL(NSUB-1)
            ELSE
               PWGT = AREA/REAL(NSUB-1)
            END IF
C     
C     Loop through each line. Compute the velocity offset
C     to add to the radial velocities computed for each image.
C     
            DO K = 1, NWAVE
               VREST = REAL(VLIGHT*(LOG(WAVE(K))-WZERO)+GAMMA)
               SCARE = PWGT*SCALE(K)
               DO I = 1, NFPIX
                  DBF(I) = 0.D0
               END DO
C
C     Loop over disc. Compute VX, VY and then convert
C     to X, Y and then calculate eclipse
C
               DO J = 1, NSIDE*OVER
                  VY   = VZERO + VOVER*REAL(J)
                  DO I = 1, NSIDE*OVER
                     VX   = VZERO + VOVER*REAL(I)
                     R    = VSSQ/(VX*VX + (VY+KW)**2)
                     VIS  = .TRUE.
                     IF(R*RL1 .LT. RMAX) THEN
                        SCL       =  R*SQRT(R/VSSQ)
                        PVEC(1)   =  REAL(SCL*(VY+KW)/RL1)
                        PVEC(2)   =  REAL(-SCL*VX/RL1)
                        VIS = .NOT.BLINK(PVEC,EARTH,Q)
                     END IF
                     IF(VIS) THEN                        
                        NADD  = NINT((VREST-VX*COSP+VY*SINP
     &                       -VFZER)/VFPIX)
                        IF(NADD.GE.1 .AND. NADD.LE.NFPIX)
     &                       DBF(NADD) = DBF(NADD) + 
     &                       DISC((I-1)/OVER+1,(J-1)/OVER+1,
     &                       NWHICH(K))
                     END IF
                  END DO  
               END DO
C         
C     Multiply disc element by (scaled) area in (km/s)**2, the inverse
C     frequency factor and the scale factor for the line in question.
C     
               DO I = 1, NFPIX
                  FBF(I) = FBF(I) + SCARE*DBF(I)
               END DO
            END DO
         END DO
C     
C     Now blurr and bin
C     
         DO I = 1, NPIX
            SUM = 0.D0
C     
C     Add in contribution from all the pixels in the fine array
C     that will bin into pixel I
C     
            DO J = NDIV*(I-1)+1, NDIV*I
C
C     blurr the fine array pixels out before binning
C     
               DO K = -NSMOOTH, NSMOOTH
                  L = J + K
                  IF(L.GE.1 .AND. L.LE.NFPIX) 
     &                 SUM = SUM + SMOOTH(K)*FBF(L)
               END DO
            END DO
            DATA(I,N) = REAL(SUM)
         END DO
      END DO
      RETURN
      END

      







