*OPTSCL
*
* OPTSCL -- scales images to reach minimum Chi**2
*
* This routine should be run immediately after a fake image has been set
* up, and prior to running 'memit'. Its purpose is to correct for a poor
* choice of initial image. The reason is that inside 'memit' the image is
* only allowed to change by a small fraction each iterattion and if it
* is far out the first iterations are spent just trying to get the scaling
* right. This can be done much more quickly with 'optscl' because it is
* just a linear least-squares problem.
*
* Parameters:
*
*   IMAGE     -- Image to optimise
*
*   DATA      -- Data file containing data and errors which will be used
*                to rescale the image.
*
*   GLOBAL    -- TRUE then a single parameter is optimised for all the images.
*                FALSE then each image is separately re-scaled. If there
*                is only one wavelength then GLOBAL = TRUE automatically.
*                If you can you should first try separate rescaling (GLOBAL
*                = FALSE) which gives the lowest chi**2. However it can also
*                produce negative scalings which are rejected because 
*                they cannot be handled by 'memit'. If this happens then 
*                rerun with GLOBAL = TRUE.
*
*OPTSCL
      SUBROUTINE OPTSCL(STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'DAT_PAR'
      INCLUDE 'doppler.inc'
      INCLUDE 'CNF_PAR'
      INTEGER STATUS, NDIM, DIM(3), LBND(3), UBN
     :D(3)
      INTEGER IMAGE, EL, I, DATA, DWPTR, PPTR, HPTR
      INTEGER PLACE, PHASES, WIDTHS, PWPTR
      INTEGER IPTR, DPTR, EPTR, WORK, WPTR
      DOUBLE PRECISION HJD0, PERIOD
      LOGICAL GLOBAL, SAMECI, BASE
      CHARACTER*16 USE
      CHARACTER*(DAT__SZLOC) LOC1, LOC2
C
      IF(STATUS.NE.SAI__OK) RETURN
      CALL NDF_BEGIN
C
C     Open image file 
C
      CALL NDF_ASSOC('IMAGE','UPDATE',IMAGE, STATUS)
      CALL NDF_ISBAS(IMAGE, BASE, STATUS)
      IF(.NOT.BASE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','memit cannot handle NDF sections.',
     &        STATUS)
      END IF
      CALL NDF_DIM(IMAGE,3,DIM,NDIM,STATUS)
      IF(STATUS.EQ.SAI__OK) THEN
         IF(NDIM.EQ.1) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ',
     &           'Images must be 2 or 3D', STATUS)
         ELSE IF(DIM(1).NE.DIM(2)) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Image(s) not square', STATUS)
         ELSE
            NSIDE  = DIM(1)
            NIMAGE = DIM(3)
         END IF
      END IF
C
C     Get header items
C
      CALL NDF_XLOC(IMAGE,'DOPPLER_MAP','READ',LOC1,STATUS)
C
      CALL CMP_GET0R(LOC1,'VPIX',VPIX,STATUS)
      CALL CMP_GET1R(LOC1, 'WAVE',MAXWV,WAVE,NWAVE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NWAVE.LT.NIMAGE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Too few wavelengths found', STATUS)
      END IF
C
C     Read scale factors and correspondence arrays.
C     
      IF(STATUS.EQ.SAI__OK .AND. NIMAGE.NE.NWAVE) THEN
         IF(NIMAGE.GT.1) THEN
            CALL CMP_GET1I(LOC1, 'NWHICH',NWAVE,NWHICH,EL,STATUS)
            IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
               STATUS = SAI__ERROR
               CALL ERR_REP(' ','NWHICH/WAVE: conflicting sizes.',
     &              STATUS)
            END IF
         ELSE
            DO I = 1, NWAVE
               NWHICH(I) = 1
            END DO
         END IF
         CALL CMP_GET1R(LOC1, 'SCALE',NWAVE,SCALE,EL,STATUS)
         IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','SCALE/WAVE: conflicting sizes.',
     &           STATUS)
         END IF
      ELSE IF(STATUS.EQ.SAI__OK) THEN
         DO I = 1, NWAVE
            SCALE(I)  = 1.
            NWHICH(I) = I
         END DO
      END IF
C
      CALL CMP_GET0R(LOC1,'FWHM',FWHM,STATUS)
      CALL CMP_GET0R(LOC1,'GAMMA',GAMMA,STATUS)
      CALL CMP_GET0R(LOC1,'OFFSET',OFFSET,STATUS)
      CALL CMP_GET0I(LOC1,'NDIV',NDIV,STATUS)
      CALL CMP_GET0C(LOC1,'USE',USE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. (.NOT.SAMECI(USE,'Phases') 
     &     .AND. .NOT.SAMECI(USE,'HJDs')) ) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','USE has unrecognised value',STATUS)
      END IF
      CALL CMP_GET0I(LOC1,'NSUB',NSUB,STATUS)
C
C     Open data file to provide phases, wavelength scale etc
C     for computed data
C
      CALL NDF_ASSOC('DATA', 'READ', DATA, STATUS)
      CALL NDF_ISBAS(DATA, BASE, STATUS)
      IF(.NOT.BASE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','memit cannot handle NDF sections.',
     &        STATUS)
      END IF
      CALL NDF_DIM(DATA,2,DIM,NDIM,STATUS)
      NPIX  = DIM(1)
      NSPEC = DIM(2)
C     
C     Read header
C     
      CALL NDF_XLOC(DATA,'DOPPLER_DATA','READ',LOC2,STATUS)
C     
      CALL CMP_GET0I(LOC2,'NARC',NARC,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NARC.NE.-2) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ',
     &        'Doppler imaging programs only use'//
     &        ' log wavelength scales.', STATUS)
      END IF
      CALL CMP_GET1D(LOC2, 'ARC', 2, ARC, EL, STATUS)
      IF(STATUS.EQ.SAI__OK .AND. EL.NE.2) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Could not read 2 arc coefficients',
     &        STATUS)
      END IF
C     
C     Get orbital period and dwells
C     
      CALL CMP_GET0D(LOC2,'PERIOD',PERIOD,STATUS)
      CALL CMP_MAPV(LOC2, 'DWELL', '_REAL', 'READ', DWPTR, 
     &     EL, STATUS)
C     
C     Get phase either from phases array or HJDs
C     
      IF(SAMECI(USE,'Phases')) THEN
         CALL CMP_MAPV(LOC2,'PHASES','_DOUBLE','READ',PPTR, 
     &        EL, STATUS)
      ELSE IF(SAMECI(USE,'HJDs')) THEN
         CALL CMP_GET0D(LOC2,'HJD0',HJD0,STATUS)
         CALL CMP_MAPV(LOC2, 'HJD', '_DOUBLE', 'READ', HPTR, 
     &        EL, STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_DOUBLE',1,1,NSPEC,PLACE,PHASES,STATUS)
         CALL NDF_MAP(PHASES,'Data','_DOUBLE','WRITE',PPTR,
     &        EL,STATUS)
         CALL PHCOMP(%VAL(CNF_PVAL(HPTR)), NSPEC, %VAL(CNF_PVAL(PPTR)),
     &        HJD0, PERIOD,STATUS)
      END IF
      CALL DAT_ANNUL(LOC2, STATUS)
      CALL NDF_TEMP(PLACE,STATUS)
      CALL NDF_NEW('_REAL',1,1,NSPEC,PLACE,WIDTHS,STATUS)
      CALL NDF_MAP(WIDTHS,'Data','_REAL','WRITE',PWPTR,
     &     EL,STATUS)
      CALL PWCOMP(%VAL(CNF_PVAL(DWPTR)),NSPEC,%VAL(CNF_PVAL(PWPTR)),
     :            PERIOD,STATUS)
C
C     Type of re-scaling to do
C
      GLOBAL = .TRUE.
      IF(NWAVE.NE.1) CALL PAR_GET0L('GLOBAL', GLOBAL, STATUS)
C
C     Map
C
      CALL NDF_MAP(IMAGE,'Data','_REAL','UPDATE',IPTR,EL,STATUS)
      CALL NDF_MAP(DATA,'Data','_REAL','READ',DPTR,EL,STATUS)
      CALL NDF_MAP(DATA,'Errors','_REAL','READ',EPTR,EL,STATUS)
C
C     Get workspace
C
      CALL NDF_BOUND(DATA,2,LBND,UBND,NDIM,STATUS)
      CALL NDF_TEMP(PLACE,STATUS)
      IF(GLOBAL) THEN
         CALL NDF_NEW('_REAL',2,LBND,UBND,PLACE,WORK,STATUS)
      ELSE
         LBND(3) = 1
         UBND(3) = NWAVE
         CALL NDF_NEW('_REAL',3,LBND,UBND,PLACE,WORK,STATUS)
      END IF
      CALL NDF_MAP(WORK,'Data','_REAL','WRITE',WPTR,EL,STATUS)         
C     
C     Compute optimum scaling factors
C
      OPFIR = .TRUE.
      CALL OPT_SCL(%VAL(CNF_PVAL(IPTR)), %VAL(CNF_PVAL(DPTR)), 
     :             %VAL(CNF_PVAL(EPTR)),
     &     %VAL(CNF_PVAL(WPTR)), %VAL(CNF_PVAL(PPTR)), 
     :     %VAL(CNF_PVAL(PWPTR)), GLOBAL,
     &     STATUS)
C
C     Write out new scale factors
C
      IF(.NOT.GLOBAL .AND. NIMAGE.NE.NWAVE)
     &     CALL CMP_PUT1R(LOC1,'SCALE',NWAVE,SCALE,STATUS)
      CALL DAT_ANNUL(LOC1,STATUS)
      CALL NDF_END(STATUS)
      RETURN
      END	

      SUBROUTINE OPT_SCL(IMAGE, DATA, ERRORS, WORK, PHASES, 
     &     PWIDTH, GLOBAL, STATUS)
C
C Subroutine to optimise scale factors of images by minimising chi**2
C
C R*4 IMAGE(NSIDE,NSIDE,NIMAGE) -- NIMAGE images of NSIDE*NSIDE dimension
C R*4 DATA(NPIX,NSPEC)   -- Data NPIX in X by NSPEC in Y
C R*4 ERRORS(NPIX,NSPEC) -- 1-sigma uncertainties on DATA
C
C R*4 WORK(NPIX,NSPEC)       (GLOBAL=.TRUE.)  Work space array for 
C R*4 WORK(NWAVE,NPIX,NSPEC) (GLOBAL=.FALSE.) least-squares vectors.
C
C R*8 PHASES(NSPEC) -- Phases
C R*4 PWIDTH(NSPEC) -- Phase widths of exposures
C
C LOGICAL GLOBAL -- If true then only a global re-scaling is computed in which
C                   all images are rescaled by the same factor. If false, the
C                   rescaling is done individually.
C I*4 STATUS = 0, OK. Otherwise error.
C
C All other arguments (e.g. NIMAGE) are passed by common blocks inside
C doppler.inc)
C
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'doppler.inc'
      INTEGER STATUS, I, J, K, L, JF, NCALC, NOK, IFAIL
      REAL IMAGE(NSIDE,NSIDE,NIMAGE), DATA(NPIX,NSPEC)
      REAL ERRORS(NPIX,NSPEC), WORK(NPIX,NSPEC,*)
      REAL PWIDTH(NSPEC), CH1, CH2, SC
      DOUBLE PRECISION PHASES(NSPEC)
      LOGICAL GLOBAL
      INTEGER INDX(MAXWV)
      DOUBLE PRECISION A(MAXWV,MAXWV), B(MAXWV), D
      DOUBLE PRECISION ASAVE(MAXWV,MAXWV), BSAVE(MAXWV)
      DOUBLE PRECISION SUM, SUM1, SUM2
      REAL SCOLD(MAXWV)
C
      IF(STATUS.NE.SAI__OK) RETURN
      NOK = 0
      DO J = 1, NSPEC
         DO I = 1, NPIX
            IF(ERRORS(I,J).GT.0.) NOK = NOK + 1
         END DO
      END DO
      IF(NOK.EQ.0) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','There are no good data',STATUS)
         RETURN
      END IF
C
C     Store initial scale factors
C
      DO I = 1, NWAVE
         SCOLD(I) = SCALE(I)
      END DO
C
C     GLOBAL = .TRUE.
C     
C     Compute data with each scale factor held at current value.
C     
C     GLOBAL = .FALSE.
C     
C     Compute data for each line with scale factor set to 1 with all other
C     lines' scale factors set = 0. The results are stored in a temporary
C     array and will be used to compute the least-squares solution.
C     
      IF(GLOBAL) THEN
         CALL OP(IMAGE, WORK, PHASES, PWIDTH, STATUS)
      ELSE
         DO I = 1, NWAVE
            DO J = 1, NWAVE
               SCALE(J) = 0.
            END DO
            SCALE(I) = 1.
            CALL OP(IMAGE, WORK(1,1,I), PHASES, PWIDTH, STATUS)
         END DO
      END IF
C
C     Compute least-squares vector (a single number if GLOBAL)
C     otherwise NWAVE numbers
C
      IF(GLOBAL) THEN
         NCALC = 1
      ELSE
         NCALC = NWAVE
      END IF
      DO I = 1, NCALC
         SUM = 0.D0
         DO K = 1, NSPEC
            DO J = 1, NPIX
               IF(ERRORS(J,K).GT.0.)
     &              SUM = SUM + DATA(J,K)*WORK(J,K,I)/ERRORS(J,K)**2
            END DO
         END DO
         B(I)     = SUM
         BSAVE(I) = SUM
      END DO
C
C     Compute least-squares matrix (a single number if GLOBAL)
C     The matrix is symmetric which is used to save some computation.
C
      DO I = 1, NCALC
         DO J = I, NCALC
            SUM = 0.D0
            DO L = 1, NSPEC
               DO K = 1, NPIX
                  IF(ERRORS(K,L).GT.0.)
     &                 SUM = SUM + 
     &                 WORK(K,L,I)*WORK(K,L,J)/ERRORS(K,L)**2
               END DO
            END DO
            A(I,J)     = SUM
            A(J,I)     = SUM
            ASAVE(I,J) = SUM
            ASAVE(J,I) = SUM
         END DO
      END DO
C
C     Solve NCALC simultaneous equations to find optimum scale factors.
C
      CALL LUDCMP(A,NCALC,MAXWV,INDX,D,IFAIL)
      IF(IFAIL.NE.0) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','LUDCMP failed.',STATUS)
         RETURN
      END IF
      CALL LUBKSB(A,NCALC,MAXWV,INDX,B)
      IF(GLOBAL) THEN
         IF(B(1).LE.0.) THEN
            CALL MSG_OUT(' ','Scale factor less than zero',STATUS)
            CALL MSG_OUT(' ','No rescaling will be done',STATUS)
            RETURN
         END IF
      ELSE
         DO I = 1, NCALC
            IF(B(I).LE.0.) THEN
               CALL MSG_SETI('I',I)
               CALL MSG_SETR('WAVE',WAVE(I))
               CALL MSG_OUT(' ','Optimum for line ^I, '//
     &              'wavelength ^WAVE is less than zero',STATUS)
               CALL MSG_OUT(' ','No rescaling will be done',STATUS)
               RETURN
            END IF
         END DO
      END IF
C     
C     Compute and report reduced Chi-squared before and after
C
      IF(GLOBAL) SCOLD(1) = 1.
      SUM1 = 0.
      SUM2 = 0.
      DO J = 1, NCALC
         DO I = 1, NCALC
            SUM1 = SUM1 + SCOLD(I)*ASAVE(I,J)*SCOLD(J)
            SUM2 = SUM2 + B(I)*ASAVE(I,J)*B(J)
         END DO
         SUM1 = SUM1 - 2.*BSAVE(J)*SCOLD(J)
         SUM2 = SUM2 - 2.*BSAVE(J)*B(J)
      END DO
      SUM = 0.D0
      DO J = 1, NSPEC
         DO I = 1, NPIX
            IF(ERRORS(I,J).GT.0.)
     &           SUM = SUM + (DATA(I,J)/ERRORS(I,J))**2
         END DO
      END DO
      CH1 = REAL((SUM1+SUM)/REAL(NOK))
      CH2 = REAL((SUM2+SUM)/REAL(NOK))
      CALL MSG_SETR('CH',CH1)
      CALL MSG_OUT(' ','Initial reduced Chi-squared = ^CH',STATUS)
      CALL MSG_SETR('CH',CH2)
      CALL MSG_OUT(' ','  Final reduced Chi-squared = ^CH',STATUS)
C
C     All ok, now rescale.
C
      IF(GLOBAL .OR. NWAVE.EQ.NIMAGE) THEN
C
C     1 line/image
C     In this case multiply images by correct scale
C     factors so that SCALE(I) = 1 is ok
C     
         DO I = 1, NIMAGE
            IF(GLOBAL) THEN
               SC = REAL(B(1))
            ELSE
               SC = REAL(B(I)/SCOLD(I))
            END IF
            DO J = 1, NSIDE
               DO K = 1, NSIDE
                  IMAGE(K,J,I) = SC*IMAGE(K,J,I)
               END DO
            END DO
            CALL MSG_SETI('I',I)
            CALL MSG_SETR('SC',SC)
            CALL MSG_OUT(' ',' Image ^I scaled by ^SC',STATUS)
         END DO
      ELSE
C
C     More than 1 line/image, GLOBAL = .FALSE.
C     Rescale image so that first line corresponding
C     to a particular image is scaled to 1
C
         DO I = 1, NIMAGE
            DO J = NWAVE,1,-1
               IF(NWHICH(J).EQ.I) JF = J
            END DO
            SC = REAL(B(JF))
            DO J = 1, NSIDE
               DO K = 1, NSIDE
                  IMAGE(K,J,I) = SC*IMAGE(K,J,I)
               END DO
            END DO
            CALL MSG_SETI('I',I)
            CALL MSG_SETR('SC',SC)
            CALL MSG_OUT(' ',' Image ^I scaled by ^SC',STATUS)
            DO J = 1, NWAVE
               IF(NWHICH(J).EQ.I) SCALE(J) = REAL(B(J)/SC)
            END DO
         END DO
      END IF
      RETURN
      END
