*MEMIT
*
* MEMIT -- iterates Doppler images. 
*
* You can run this as soon as you have made a starting image with 'makimg'
* however, I advise scaling with 'optscl' first to get a good start.
*
* Parameters:
*
*  IMAGE     -- The image to be iterated. Typically created by makimg.
*
*  DATA      -- Data file to iterate on.
*
*  CAIM      -- Reduced chi-squared to aim for. Ultimately this controls the
*               noise/resolution tradeoff in the image. If you aim for too low
*               a chi**2 your image will be corrupted by noise. Often I start
*               too low, see where the image gets to and then relax it a bit.
*
*  NITS      -- Number of iterations.
*
*  RMAX      -- Maximum step size. This limits the amount MEM tries to change
*               the image and will only apply early on when you will see the
*               step taken limited by this number.
*
*  NDEF      -- Default option. The entropy can be measured relative to 
*               different defaults. Normally I start with NDEF = 1 and then
*               switch to NDEF = 2 when I am nearer image. I don't use the
*               others at all and they are not guaranteed. Using NDEF = 2 
*               means that the entropy becomes insensitive to large scale
*               variations which is a desireable feature as we don't expect
*               uniformity in the image which NDEF = 1 implies. However we
*               do hope for smoothness on small scales.
* 
*               1 = Uniform
*               2 = Gaussian blurr
*               3 = Power law
*               4 = Gaussian blurr + radial profile
*
*  If NDEF = 2 or 4
*
*    BLURR     -- Amount of blurring for Gaussian default option
*
*  Else if NDEF = 3
*
*    EXPON     -- Exponent of power law
*    XCEN      -- X coordinate of centre for power law
*    YCEN      -- Y coordinate of centre for power law
*    RADMIN    -- Inner radius of power law. (Constant inside)
*
*MEMIT
      SUBROUTINE MEMIT(STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'DAT_PAR'
      INCLUDE 'doppler.inc'
      INCLUDE 'CNF_PAR'
      INTEGER STATUS, NDIM, DIM(3), LBND(3), UBN
     :D(3)
      INTEGER IMAGE, EL, I, DATA, DWPTR, PPTR
      INTEGER PLACE, HPTR, PHASES, WIDTHS, PWPTR, WORK
      INTEGER WPTR, OPTR, DPTR, EPTR, NITS, NDEF
      REAL CAIM, BLURR, EXPON, XCEN, YCEN, RMAX, RADMIN
      DOUBLE PRECISION HJD0, PERIOD
      LOGICAL SAMECI, BASE
      CHARACTER*16 USE
      CHARACTER*(DAT__SZLOC) LOC
C
C     Common to communicate with opus and tropus
C 
      COMMON/POINTER/PPTR, PWPTR     
C
      IF(STATUS.NE.SAI__OK) RETURN
      CALL NDF_BEGIN
C
C     Get name of image file
C
      CALL NDF_ASSOC('IMAGE','UPDATE',IMAGE,STATUS)
      CALL NDF_ISBAS(IMAGE, BASE, STATUS)
      IF(.NOT.BASE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','memit cannot handle NDF sections.',
     &        STATUS)
      END IF
      CALL NDF_DIM(IMAGE,3,DIM,NDIM,STATUS)
      IF(STATUS.EQ.SAI__OK) THEN
         IF(NDIM.EQ.1) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ',
     &           'Images must be 2 or 3D', STATUS)
         ELSE IF(DIM(1).NE.DIM(2)) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Image(s) not square', STATUS)
         ELSE
            NSIDE  = DIM(1)
            NIMAGE = DIM(3)
         END IF
      END IF
C
C     Get header items
C
      CALL NDF_XLOC(IMAGE,'DOPPLER_MAP','READ',LOC,STATUS)
C
      CALL CMP_GET0R(LOC,'VPIX',VPIX,STATUS)
      CALL CMP_GET1R(LOC, 'WAVE',MAXWV,WAVE,NWAVE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NWAVE.LT.NIMAGE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Too few wavelengths found', STATUS)
      END IF
C
C     Read scale factors and correspondence arrays.
C     
      IF(STATUS.EQ.SAI__OK .AND. NIMAGE.NE.NWAVE) THEN
        IF(NIMAGE.GT.1) THEN
            CALL CMP_GET1I(LOC, 'NWHICH',NWAVE,NWHICH,EL,STATUS)
            IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
               STATUS = SAI__ERROR
               CALL ERR_REP(' ','NWHICH/WAVE: conflicting sizes.',
     &              STATUS)
            END IF
         ELSE
            DO I = 1, NWAVE
               NWHICH(I) = 1
            END DO
         END IF
         CALL CMP_GET1R(LOC, 'SCALE',NWAVE,SCALE,EL,STATUS)
         IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','SCALE/WAVE: conflicting sizes.',
     &           STATUS)
         END IF
      ELSE IF(STATUS.EQ.SAI__OK) THEN
         DO I = 1, NWAVE
            SCALE(I)  = 1.
            NWHICH(I) = I
         END DO
      END IF
C
      CALL CMP_GET0R(LOC,'FWHM',FWHM,STATUS)
      CALL CMP_GET0R(LOC,'GAMMA',GAMMA,STATUS)
      CALL CMP_GET0R(LOC,'OFFSET',OFFSET,STATUS)
      CALL CMP_GET0I(LOC,'NDIV',NDIV,STATUS)
      CALL CMP_GET0C(LOC,'USE',USE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. (.NOT.SAMECI(USE,'Phases') 
     &     .AND. .NOT.SAMECI(USE,'HJDs')) ) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','USE has unrecognised value',STATUS)
      END IF
      CALL CMP_GET0I(LOC,'NSUB',NSUB,STATUS)
      CALL DAT_ANNUL(LOC, STATUS)
C
C     Open data file to provide phases, wavelength scale etc
C     for computed data
C
      CALL NDF_ASSOC('DATA', 'READ', DATA, STATUS)
      CALL NDF_ISBAS(DATA, BASE, STATUS)
      IF(.NOT.BASE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','memit cannot handle NDF sections.',
     &        STATUS)
      END IF
      CALL NDF_DIM(DATA,2,DIM,NDIM,STATUS)
      NPIX  = DIM(1)
      NSPEC = DIM(2)
C     
C     Read header
C     
      CALL NDF_XLOC(DATA,'DOPPLER_DATA','READ',LOC,STATUS)
C     
      CALL CMP_GET0I(LOC,'NARC',NARC,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NARC.NE.-2) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ',
     &        'Doppler imaging programs only use'//
     &        ' log wavelength scales.', STATUS)
      END IF
      CALL CMP_GET1D(LOC, 'ARC', 2, ARC, EL, STATUS)
      IF(STATUS.EQ.SAI__OK .AND. EL.NE.2) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Could not read 2 arc coefficients',
     &        STATUS)
      END IF
C     
C     Get orbital period and dwells
C     
      CALL CMP_GET0D(LOC,'PERIOD',PERIOD,STATUS)
      CALL CMP_MAPV(LOC, 'DWELL', '_REAL', 'READ', DWPTR, 
     &     EL, STATUS)
C     
C     Get phase either from phases array or HJDs
C     
      IF(SAMECI(USE,'Phases')) THEN
         CALL CMP_MAPV(LOC,'PHASES','_DOUBLE','READ',PPTR, 
     &        EL, STATUS)
      ELSE IF(SAMECI(USE,'HJDs')) THEN
         CALL CMP_GET0D(LOC,'HJD0',HJD0,STATUS)
         CALL CMP_MAPV(LOC, 'HJD', '_DOUBLE', 'READ', HPTR, 
     &        EL, STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_DOUBLE',1,1,NSPEC,PLACE,PHASES,STATUS)
         CALL NDF_MAP(PHASES,'Data','_DOUBLE','WRITE',PPTR,
     &        EL,STATUS)
         CALL PHCOMP(%VAL(CNF_PVAL(HPTR)), NSPEC, %VAL(CNF_PVAL(PPTR)),
     &        HJD0, PERIOD,STATUS)
      END IF
      CALL DAT_ANNUL(LOC, STATUS)
      CALL NDF_TEMP(PLACE,STATUS)
      CALL NDF_NEW('_REAL',1,1,NSPEC,PLACE,WIDTHS,STATUS)
      CALL NDF_MAP(WIDTHS,'Data','_REAL','WRITE',PWPTR,
     &     EL,STATUS)
      CALL PWCOMP(%VAL(CNF_PVAL(DWPTR)),NSPEC,%VAL(CNF_PVAL(PWPTR)),
     :            PERIOD,STATUS)
C
C     Get iteration parameters
C
      CALL PAR_GDR0R('CAIM',1.,0.,1.E10,.FALSE.,CAIM,STATUS)
      CALL PAR_GDR0I('NITS',1,1,99999,.FALSE.,NITS,STATUS)
      CALL PAR_GDR0R('RMAX',0.2,0.,1.,.FALSE.,RMAX,STATUS)
      CALL PAR_GDR0I('NDEF',1,1,4,.FALSE.,NDEF,STATUS)
      IF(NDEF.EQ.2) THEN
         CALL PAR_GDR0R('BLURR',10.,0.,1.E5,.FALSE.,BLURR,STATUS)
      ELSE IF(NDEF.EQ.3) THEN
         CALL PAR_GDR0R('EXPON',0.,-100.,100.,.FALSE.,EXPON,STATUS)
         CALL PAR_GDR0R('XCEN',0.,-1.E5,1.E5,.FALSE.,XCEN,STATUS)
         CALL PAR_GDR0R('YCEN',0.,-1.E5,1.E5,.FALSE.,YCEN,STATUS)
         XCEN = XCEN/VPIX + REAL(1+NSIDE)/2.
         YCEN = YCEN/VPIX + REAL(1+NSIDE)/2.
         CALL PAR_GDR0R('RADMIN',0.,0.,1.E5,.FALSE.,RADMIN,STATUS)
      ELSE IF(NDEF.EQ.4) THEN
         CALL PAR_GDR0R('BLURR',10.,0.,1.E5,.FALSE.,BLURR,STATUS)
         CALL PAR_GDR0R('XCEN',0.,-1.E5,1.E5,.FALSE.,XCEN,STATUS)
         CALL PAR_GDR0R('YCEN',0.,-1.E5,1.E5,.FALSE.,YCEN,STATUS)
         XCEN = XCEN/VPIX + REAL(1+NSIDE)/2.
         YCEN = YCEN/VPIX + REAL(1+NSIDE)/2.
      END IF
C
C     Get workspace for defaults
C
      IF(NDEF.EQ.3 .OR. NDEF.EQ.4) THEN
         CALL NDF_BOUND(IMAGE,3,LBND,UBND,NDIM,STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_REAL',2,LBND,UBND,PLACE,WORK,STATUS)
         CALL NDF_MAP(WORK,'Data','_REAL','WRITE',WPTR,EL,STATUS)
      END IF
C
C     Map
C
      CALL NDF_MAP(IMAGE,'Data','_REAL','UPDATE',OPTR,EL,STATUS)
      CALL NDF_MAP(DATA,'Data','_REAL','READ',DPTR,EL,STATUS)
      CALL NDF_MAP(DATA,'Errors','_REAL','READ',EPTR,EL,STATUS)
C
C     Now start iterations
C
      OPFIR = .TRUE.
      CALL FIGMEM(%VAL(CNF_PVAL(OPTR)), NSIDE, NSIDE, NIMAGE,
     &     %VAL(CNF_PVAL(WPTR)), %VAL(CNF_PVAL(DPTR)), 
     :     %VAL(CNF_PVAL(EPTR)),
     &     NPIX, NSPEC, NITS, CAIM, RMAX, NDEF, BLURR, 
     &     EXPON, XCEN, YCEN, RADMIN, STATUS)
      CALL NDF_END(STATUS)
      RETURN
      END	

      SUBROUTINE FIGMEM(IMAGE, NX, NY, NZ, WORK, DATA, 
     &     ERRORS, NDX, NDY, NITS, CAIM, RMAX, NDEF, 
     &     BLURR, EXPON, XCEN, YCEN, RADMIN, STATUS)
C
C Performs MEMSYS iterations
C
C  Inputs:
C      IMAGE(NX,NY,NZ)      = starting image
C      NX         = first image dimension
C      NY         = second image dimension
C      NZ         = third image dimension
C      WORK(NX,NZ)= Workspace for default computation
C      DATA(NDX,NDY)       = observations
C      ERRORS(NDX,NDY)     = 1-sigma errors
C      NDX      = first data dimension 
C      NDY      = second data dimensions
C
C Next parameters control iteration strategy:
C
C      CAIM       = Chi-square to aim for
C      NITS       = Number of iterations
C      RMAX       = Max step size
C      NDEF       = Default option 1 = Uniform
C                                  2 = Gaussian
C                                  3 = Power law
C      BLURR      = FWHM of gaussian blurr for default
C      EXPON      = Exponent of power law
C      XCEN       = X coordinate of centre for power law
C      YCEN       = Y coordinate of centre for power law
C      RADMIN     = Minimum radius for application of power law
C      
C  Outputs:
C
C      IMAGE      = final image
C
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER NX, NY, NZ, NDX, NDY, NDAT, NITS, NDEF, STATUS
      INTEGER NPIX, MODE, I, IBAD1, IT, IADD, LEVEL, IER
      REAL IMAGE(NX*NY*NZ), WORK(NX*NY)
      REAL DATA(NDX*NDY), ERRORS(NDX*NDY)
      REAL CAIM, RMAX, BLURR, EXPON, XCEN, YCEN, RADMIN
      REAL COLD, CNEW, TEST, SOLD, RNEW, SNEW, SUMF, ACC
      INCLUDE 'mem.inc'
      DATA IOUT/6/          
C
C     Initial iteration parameters 
C
      DATA  LEVEL/20/
C
C     Trap invalid dimensions
C
      IF(STATUS.NE.SAI__OK) RETURN
      NPIX = NX*NY*NZ
      NDAT = NDX*NDY
      CALL MSG_SETI('NPIX',NPIX)
      CALL MSG_SETI('NDAT',NDAT)
      CALL MSG_OUT(' ',
     &     'Image pixels = ^NPIX, data values = ^NDAT',STATUS)
      IF(NPIX.LE.1) THEN
         CALL MSG_SETI('NX',NX)
         CALL MSG_SETI('NY',NY)
         CALL MSG_SETI('NZ',NZ)
         STATUS = SAI__ERROR
         CALL ERR_REP(' ',
     &        'Bad image dimensions ^NX ^NY ^NZ',STATUS)
         RETURN
      ELSE IF(NDAT.LE.0) THEN
         CALL MSG_SETI('NDX',NDX)
         CALL MSG_SETI('NDY',NDY)
         STATUS = SAI__ERROR
         CALL ERR_REP(' ',
     &        'Bad data dimensions ^NDX ^NDY',STATUS)
         RETURN
      END IF
C     
C     Restricted variety of applications
C     
      IF(NDEF.EQ.1) THEN
         MODE = 10
      ELSE
         MODE = 30
      END IF
C     
C     Format MEM core storage for images and data
C     
      CALL MEMCORE(MAXPIX, NX, NY, NZ, NDAT, IER)
      IF(IER.NE.0) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ',
     &        'Failed to format MEMSYS image/data store.',
     &        STATUS)
         RETURN
      END IF
C     
C     Move starting image to image vector 1
C
      CALL MSG_OUT(' ','Loading initial image.',STATUS)
      CALL MOVER( IMAGE, ST(KB(1)), NPIX)
C
C     check for bad pixels
C
      IBAD1 = 0
      DO I=1,NPIX
         IF( IMAGE(I).LE.0. ) IBAD1=IBAD1+1
      END DO
      IF( IBAD1.GT.0 ) THEN
         CALL MSG_SETI('BAD',IBAD1)
         CALL MSG_OUT(' ','^BAD bad pixels in starting image.',
     &        STATUS)
      END IF
C
C     Move observed fluxes to data vector 21
C
      CALL MSG_OUT(' ','Loading observations.',STATUS)
      CALL MOVER( DATA, ST(KB(21)), NDAT)
C
C     move twice inverse variances to data vector 22
C
      CALL MSG_OUT(' ','Loading inverse variances.',STATUS)
      IADD = KB(22)-1
      DO I=1,NDAT
         IF(ERRORS(I).GT.0.) THEN
            ST(IADD+I) = 2./(ERRORS(I)*ERRORS(I) * NDAT)
         ELSE
            CALL MSG_SETI('I',I)
            CALL MSG_OUT(' ',
     &           ' Warning: sigma not positive at data point ^I',
     &           STATUS)
         END IF
      END DO
C
C     MEM iteration loop
C     
      DO IT = 1, NITS
         CALL MSG_BLANK(STATUS)
         CALL MSG_SETI('IT',IT)
         CALL MSG_OUT(' ',' MEM iteration: ^IT',STATUS)
         IF(NDEF.GT.1) THEN
C     
C     Compute default image
C
            CALL FIGDEF(NDEF, ST(KB(1)), ST(KB(20)), WORK, 
     &           BLURR, EXPON, XCEN, YCEN, RADMIN, NX, NY, 
     &           NZ, STATUS)
            IF(STATUS.NE.SAI__OK) RETURN
         END IF
C
C     perform the MEM iteration
C     
         CALL MEMPRM( MODE, LEVEL, CAIM, RMAX, 1., ACC,
     &        COLD, TEST, CNEW, SOLD, RNEW, SNEW, SUMF )
C     
C     move final images to return buffer. This is done at the end
C     of each iteration because it costs little but allows the user
C     to break iterations without losing all previous results if
C     %VAL is being used for IMAGE
C     
         CALL MOVER( ST(KB(1)), IMAGE, NPIX )
      END DO
      RETURN
      END

      SUBROUTINE FIGDEF(NDEF, IMAGE, DEFAULT, WORK, BLURR, EXPON,
     &     XCEN, YCEN, RADMIN, NX, NY, NZ, STATUS)
C
C Computes default.
C
C TRM 31/05/90 Changed gaussian blurring to an FFT based routine
C 
C I*4 NDEF              -- Default option
C R*4 IMAGE(NX,NY,NZ)   -- Image, input.
C R*4 DEFAULT(NX,NY,NZ) -- Default, returned.
C R*4 WORK(NX,NY)       -- Work space
C R*4 BLURR             -- FWHM blurr for gaussian default
C R*4 EXPON             -- Exponent for power law default
C R*4 XCEN, YCEN        -- Centre for power law and radial profile default
C R*4 RADMIN            -- Minimum radius for operation of power law
C R*4 NX, NY, NZ        -- Dimensions of image.
C
C Options NDEF = 1 return
C         NDEF = 2 default is blurred version of image
C         NDEF = 3 power law default
C         NDEF = 4 image is blurred and then radial profile is used for
C                  default
C
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER NDEF, NX, NY, NZ, STATUS
      REAL IMAGE(NX,NY,NZ), DEFAULT(NX,NY,NZ)
      REAL BLURR, EXPON, XCEN ,YCEN, RADMIN, WORK(NX,NY)
      INTEGER I, J, NRAD, IY1, IY2, IX1, IX2, IX, IY, JI, II
      INTEGER KD, JD, NMAX, J1, J2, IFAIL, K, ID, I1, I2
      REAL DX1, DXN, DY1, DYN, D2X1, D2XN, RMAX, RMIN, ADD
      REAL RADIUS, PIXEL, WEIGHT, EFAC, D2Y1, D2YN, WET, ADJ
      DOUBLE PRECISION SUM1, SUM2
      INTEGER NBLURR
      PARAMETER (NBLURR=500)
      REAL BARRAY(NBLURR)
C
      IF(STATUS.NE.SAI__OK) RETURN
      IF(NDEF.EQ.1) THEN
         RETURN
      ELSE IF(NDEF.EQ.2) THEN
         CALL MSG_OUT(' ','Gaussian default being computed',STATUS)
         DO K = 1, NZ
            DO J = 1, NY
               DO I = 1, NX
                  DEFAULT(I,J,K) = IMAGE(I,J,K)
               END DO
            END DO
            CALL GBLURR(DEFAULT(1,1,K),NX,NY,BLURR,BLURR,IFAIL)
            IF(IFAIL.NE.0) THEN
               STATUS = SAI__ERROR
               CALL ERR_REP(' ','Gaussian blurr failed.', STATUS)
               RETURN
            END IF
         END DO
      ELSE IF(NDEF.EQ.3) THEN
         CALL MSG_OUT(' ','Power law default being computed',STATUS)
C     
C     Determine min and max pixel distance from origin to four image corners
C     
         DX1  = 1. - XCEN       
         DXN  = REAL(NX) - XCEN
         DY1  = 1. - YCEN
         DYN  = REAL(NY) - YCEN
         D2X1 = DX1*DX1
         D2XN = DXN*DXN
         D2Y1 = DY1*DY1
         D2YN = DYN*DYN
         RMAX = SQRT( MAX( D2X1, D2XN ) + MAX( D2Y1, D2YN ) )
         IF(DX1.GT.0. .OR. DXN.LT.0. .OR. DY1.GT.0. .OR. DYN.LT.0.) THEN
            RMIN = SQRT( MIN( D2X1, D2XN ) + MIN( D2Y1, D2YN ) )
         ELSE
            RMIN = 0.
         END IF
C     
C     Use approx two points per pixel
C     
         NRAD = MAX(10, NINT(2.*(RMAX-RMIN)))
         IF(NRAD.GT.NBLURR) THEN
            CALL MSG_OUT(' ','Radial profile will be undersampled',
     &           STATUS)
            NRAD = NBLURR
         END IF
C
C     Set profile
C     
         DO I = 1, NRAD
            RADIUS = RMIN + (RMAX-RMIN)*REAL(I-1)/REAL(NRAD-1)
            IF(RADIUS.LE.RADMIN) THEN
               BARRAY(I) = RADMIN**EXPON
            ELSE
               BARRAY(I) = RADIUS**EXPON
            END IF
         END DO
C     
C     Set default(s)
C     
         DO K = 1, NZ
            CALL PROFADD( WORK, NX, NY, XCEN, YCEN,
     &           RMIN, RMAX, BARRAY, NRAD)
C     
C     Blurr a little to smooth discontinuities
C     
            DO J = 1, NY
               DO I = 1, NX
                  DEFAULT(I,J,K) = 0.
               END DO
            END DO
C     
C     Reduce edge effects by pretending array a little
C     larger than it really is.
C     
            DO J = 0, NY+1
               IY1 = MAX( 1, J-1)
               IY2 = MIN(NY, J+1)
               DO I = 0, NX+1
                  IX1 = MAX( 1, I-1)
                  IX2 = MIN(NY, I+1)
                  IX  = MAX(1, MIN(NX, I))
                  IY  = MAX(1, MIN(NY, J))
                  PIXEL = WORK(IX,IY)
                  DO IY = IY1, IY2
                     IF(IY.EQ.J) THEN
                        WET = PIXEL
                     ELSE
                        WET = 0.5*PIXEL
                     END IF
                     DO IX = IX1, IX2
                        IF(IX.EQ.I) THEN
                           WEIGHT = WET
                        ELSE
                           WEIGHT = 0.5*WET
                        END IF
                        DEFAULT(IX,IY,K) = DEFAULT(IX,IY,K) + WEIGHT
                     END DO
                  END DO
               END DO
            END DO
         END DO
C     
      ELSE IF(NDEF.EQ.4) THEN
         CALL MSG_OUT(' ',
     &        'Radial profile default being computed',STATUS)
C     
C     First blurr as for gaussian default
C     
         EFAC = 4.*LOG(2.)/BLURR**2.
         NMAX = MIN(2 + INT(SQRT(8./EFAC)), NBLURR)
         DO I = 1, NMAX
            BARRAY(I) = EXP(-EFAC*REAL(I-1)**2)
         END DO
         DO KD = 1, NZ
            DO JD = 1, NY
               J1 = MAX(1, MIN(JD - NMAX + 1, NY))
               J2 = MAX(1, MIN(JD + NMAX - 1, NY))
               DO ID = 1, NX
                  I1   = MAX(1, MIN(ID - NMAX + 1, NX))
                  I2   = MAX(1, MIN(ID + NMAX - 1, NX))
                  SUM1 = 0.D0
                  SUM2 = 0.D0
                  DO JI = J1, J2
                     ADJ = BARRAY(ABS(JI-JD)+1)
                     DO II = I1, I2
                        ADD = ADJ*BARRAY(ABS(II-ID)+1)
                        SUM1 = SUM1 + ADD
                        SUM2 = SUM2 + ADD*IMAGE(II,JI,KD)
                     END DO
                  END DO
                  DEFAULT(ID,JD,KD) = REAL(SUM2/SUM1)
               END DO
            END DO
C
C     Compute radial profile, generate symmetric image
C     with same profile
C     
            NRAD = NBLURR
            CALL PROFMED(DEFAULT(1,1,KD), NX, NY, XCEN, YCEN,
     &           RMIN, RMAX, BARRAY, NRAD)
            CALL PROFADD( WORK, NX, NY, XCEN, YCEN,
     &           RMIN, RMAX, BARRAY, NRAD)
C     
C     Blurr a little to smooth discontinuities
C     
            DO J = 1, NY
               DO I = 1, NX
                  DEFAULT(I,J,KD) = 0.
               END DO
            END DO
C     
C     Reduce edge effects by pretending array a little
C     larger than it really is.
C     
            DO J = 0, NY+1
               IY1 = MAX( 1, J-1)
               IY2 = MIN(NY, J+1)
               DO I = 0, NX+1
                  IX1 = MAX( 1, I-1)
                  IX2 = MIN(NY, I+1)
                  IX = MAX(1, MIN(NX, I))
                  IY = MAX(1, MIN(NY, J))
                  PIXEL = WORK(IX,IY)
                  DO IY = IY1, IY2
                     IF(IY.EQ.J) THEN
                        WET = PIXEL
                     ELSE
                        WET = 0.5*PIXEL
                     END IF
                     DO IX = IX1, IX2
                        IF(IX.EQ.I) THEN
                           WEIGHT = WET
                        ELSE
                           WEIGHT = 0.5*WET
                        END IF
                        DEFAULT(IX,IY,KD) = DEFAULT(IX,IY,KD) + WEIGHT
                     END DO
                  END DO
               END DO
            END DO
         END DO   
      ELSE
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Default not recognised', STATUS)
         RETURN
      END IF
      IF(NDEF.GT.1) THEN
C
C     Normalise default to same average value as image
C     
         DO K = 1, NZ
            SUM1 = 0.D0
            SUM2 = 0.D0
            DO J = 1, NY
               DO I = 1, NX
                  SUM1 = SUM1 + IMAGE(I,J,K)
                  SUM2 = SUM2 + DEFAULT(I,J,K)
               END DO
            END DO
            SUM1 = SUM1/SUM2
            DO J = 1, NY
               DO I = 1, NX
                  DEFAULT(I,J,K) = REAL(SUM1*DEFAULT(I,J,K))
               END DO
            END DO
         END DO
      END IF
      CALL MSG_OUT(' ','Finished setting default', STATUS)
      RETURN
      END

      SUBROUTINE OPUS(J,K)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER STATUS, J, K
      INCLUDE 'mem.inc'
      INCLUDE 'CNF_PAR'
      INTEGER PPTR, PWPTR
      COMMON/POINTER/PPTR, PWPTR
C
      WRITE(*,*) '     OPUS ',J,' ---> ',K
      STATUS = SAI__OK
      CALL OP(ST(KB(J)),ST(KB(K)),%VAL(CNF_PVAL(PPTR)),
     :        %VAL(CNF_PVAL(PWPTR)),STATUS)
      IF(STATUS.NE.SAI__OK) STOP 'OPUS aborted'
      RETURN
      END

      SUBROUTINE TROPUS(K,J)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER STATUS, J, K
      INCLUDE 'mem.inc'
      INCLUDE 'CNF_PAR'
      INTEGER PPTR, PWPTR
      COMMON/POINTER/PPTR, PWPTR
C
      WRITE(*,*) '   TROPUS ',J,' <--- ',K
      STATUS = SAI__OK
      CALL TR(ST(KB(K)),ST(KB(J)),%VAL(CNF_PVAL(PPTR)),
     :        %VAL(CNF_PVAL(PWPTR)),
     &     STATUS)
      IF(STATUS.NE.SAI__OK) STOP 'TROPUS aborted'
      RETURN
      END

