*MAKIMG
*
* MAKIMG -- make a starting image for doppler tomography or for simulating
*           data.
*
* Parameters:
*
*   NSIDE     -- Size of images (which are square)
*
*   NIMAGE    -- Number of images. Blended lines can be dealt with by 
*                assigning an image for each line. It is also possible 
*                for the same image (bar a scaling factor) to represent 
*                more than one line.
*
*   IMAGE     -- The name of the image file to create 
*
*   VPIX      -- Size of pixels in km/s. The product of NPIX and VPIX should
*                more than cover the width of the line of interest. VPIX should
*                be less than or equal to the km/s/pixel of the data. Note 
*                there is no problem if VPIX is much smaller, except in the 
*                extra CPU and memory required. If you have blended lines
*                the images need only extend over each line, not the whole
*                blend.
*
*   NWAVE     -- Number of lines. e.g you could map the Paschen series and 
*                the CaII triplet with 2 images, one for CaII and one for the
*                hydrogen lines, but you would have to specify a wavelength for
*                each line to be mapped. NWAVE >= NIMAGE
*
*   WAVE(NWAVE) -- Central wavelength of each line in Angstroms
*
*   If (NWAVE .NE. NIMAGE)
*
*     NWHICH(NWAVE) -- Defines which image corresponds to each line. e.g.
*                      If you had two image accounting for three lines
*                      you might set NWHICH = 1,2,2 which means that the
*                      first image corresponds to the first wavelength while
*                      the second image corresponds to the other 2 lines.
*
*     SCALE(NWAVE)  -- Defines scale factor to apply to each line (allows for
*                      different lines to have same image but a different
*                      scaling.) These values can be optimised outside MEM
*                      using optscl. e.g. on the above example you might
*                      have SCALE = 1.,1.,0.5
*
*   FWHM      -- Full width half maximum of instrument, km/s. Don't make too 
*                large or it will be attempting an impossible deconvolution.
*                Too small does not matter in this respect, but a finite 
*                FWHM is good for smoothing out numerical noise.
*
*   GAMMA     -- Systemic velocity of system km/s.
*
*   NSUB      -- Subdivision factor to account for phase smearing from finite 
*                exposure length. Each exposure will be divided into NSUB 
*                exposures uniformly spread through the exposure and then 
*                trapezoidally averaged. If your exposures are short compared 
*                to the orbital period, set = 1. If you do use it, only
*                apply towards the end of the iterations because it slows 
*                things down (by a factor NSUB).
*
*   OFFSET    -- Constant to add to all phases to account for zero point
*                error in the ephemeris. This rotates the entire image but
*                the image must be re-iterated after any change to OFFSET
*                for this to be seen.
*
*   NDIV      -- The projections of model to data is done first onto a
*                fine grid which is then blurred and binned onto the data 
*                array. This is usually faster than blurring each pixel 
*                directly onto the coarse grid. NDIV is the factor by 
*                which each data pixel is split to give the fine array. 
*                NDIV must be large enough so that the FWHM is well 
*                sampled. e.g if FWHM=2, then NDIV>2 should be used.
*
*                There is no reason not to change NDIV as you iterate
*                and you could for example start with NDIV = 1 and only raise
*                it for the final few iterations. Note that large NDIV values
*                will slow things down so be careful.
*                (Use 'setobj 5 image.more.doppler.ndiv' to set ndiv = 5 for
*                 example)
*
*   USE       -- This specifies whether the HJDs or phases should be used
*                to compute the phases during iterations. The option of
*                HJDs is provided to allow one to account for changes in the
*                ephemeris, however the method of choice should normally be
*                'phases'. The reason is that if the data comes from a
*                phase-folding operation (pbin inside molly) while the
*                phases are correct, the HJDs are not. e.g. say you average
*                two spectra at phase 0.5 but 1 cycle apart. The HJD will be
*                set half-way between i.e. phase 0!. The phase on the other
*                hand is correctly dealt with. You will be OK with either
*                option if you are dealing with straight spectra.
*
*                The two options are:
*
*                   HJDs   -- use HJDs
*                   Phases -- use phases
*
*                and you can change them by editing the NDF with setobj.
*
*
* Now parameters defining shape of image:
* 
*   FANCY     -- .FALSE. you just get a constant image. .TRUE. then you 
*                are prompted for many more parameters to define a gaussian
*                spot, a spiral pattern and a power law in radius.            
*
*   If(FANCY)
*
*     XCEN      -- X centre of gaussian spot (km/s)
*     YCEN      -- Y centre of gaussian spot (km/s)
*     WIDTH     -- FWHM of gaussian spot (km/s)
*     PGAUSS    -- Peak intensity of gaussian
*     XSP       -- X centre of spiral pattern and power law disc (km/s)
*     YSP       -- Y centre of spiral pattern and power law disc (km/s)
*     VLOW      -- Lowest disc velocity (outer disc) (km/s)
*     VHIGH     -- Highest disk velocity (inner disc) (km/s)
*     EXPON     -- exponent between them (intensity scales as V**EXPON)
*     ANGLE     -- Opening angle of two armed spiral pattern (degrees)
*     CURVE     -- Difference in angle between outer and inner velocities
*     PPOWER    -- Peak intensity of power law background
*     PSPIRAL   -- Peak intensity of spiral shock
*
*
*  Else
*
*     CONST     -- Constant per pixel
*
* Don't worry about the levels you set; these can be tweaked with optscl.
*
*MAKIMG
      SUBROUTINE MAKIMG(STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'DAT_PAR'
      INCLUDE 'CNF_PAR'
      INTEGER STATUS, NX, NY, NZ, LBND(3), UBND(
     :3)
      INTEGER NDIM, IMAGE, NSIDE, NIMAGE, NWAVE, I
      INTEGER NSKIP, J, NSUB, NDIV, DPTR, EL, XPTR
      INTEGER YPTR
      REAL VPIX, FWHM, GAMMA, XCEN, YCEN, WIDTH, OFFSET
      REAL PGAUSS, XSP, YSP, EXPON, VLOW, VHIGH, ANGLE
      REAL CURVE, PPOWER, PSPIRAL, CONST
      INTEGER MXWAV
      PARAMETER (MXWAV=20)
      REAL WAVE(MXWAV), SCALE(MXWAV)
      INTEGER NWHICH(MXWAV)
      LOGICAL OK, FANCY
      CHARACTER*16 USE
      CHARACTER*(DAT__SZLOC) LOC
C
C Start
C
      IF(STATUS.NE.SAI__OK) RETURN
      CALL NDF_BEGIN
C
C     Get parameters defining image
C
      CALL PAR_GDR0I('NSIDE',100,1,10000,.FALSE.,NSIDE,STATUS)
      CALL PAR_GDR0I('NIMAGE',1,1,MXWAV,.FALSE.,NIMAGE,STATUS)
      LBND(1) = 1
      UBND(1) = NSIDE
      LBND(2) = 1
      UBND(2) = NSIDE
      NX      = NSIDE
      NY      = NSIDE
      NZ      = NIMAGE
      IF(NIMAGE.EQ.1) THEN
        NDIM = 2
      ELSE
        NDIM = 3
        LBND(3) = 1
        UBND(3) = NIMAGE
      END IF
C
C     Create data file 
C
      CALL NDF_CREAT('IMAGE','_REAL',NDIM,LBND,UBND,IMAGE,STATUS)
C
C     Get velocity per pixel
C
      CALL PAR_GDR0R('VPIX',50.,0.,1000.,.FALSE.,VPIX,STATUS)
C
C     Number of lines
C
      CALL PAR_GDR0I('NWAVE',NIMAGE,NIMAGE,MXWAV,.FALSE.,NWAVE,STATUS)
C     
C     Create doppler map extension
C     
      CALL NDF_XNEW(IMAGE,'DOPPLER_MAP','Ext',0,0,LOC,STATUS)
      CALL CMP_MOD(LOC, 'VPIX',  '_REAL', 0, 0, STATUS)
      CALL CMP_MOD(LOC, 'FWHM' , '_REAL', 0, 0, STATUS)
      CALL CMP_MOD(LOC, 'GAMMA', '_REAL', 0, 0, STATUS)
      CALL CMP_MOD(LOC, 'WAVE',  '_REAL', 1, NWAVE, STATUS)
      IF(NIMAGE.NE.NWAVE) THEN
         CALL CMP_MOD(LOC, 'SCALE',  '_REAL', 1, NWAVE, STATUS)
         IF(NIMAGE.GT.1) 
     &        CALL CMP_MOD(LOC,'NWHICH','_INTEGER',1,NWAVE,STATUS)
      END IF
      CALL CMP_MOD(LOC, 'NSUB',  '_INTEGER', 0, 0, STATUS)
      CALL CMP_MOD(LOC, 'OFFSET', '_REAL', 0, 0, STATUS)
      CALL CMP_MOD(LOC, 'NDIV', '_INTEGER', 0, 0, STATUS)
      CALL CMP_MOD(LOC, 'USE', '_CHAR*16', 0, 0, STATUS)
C     
C     Set values
C     
      CALL CMP_PUT0R(LOC,'VPIX',VPIX,STATUS)
      DO I = 1, NWAVE
         WAVE(I) = 5000.+100.*REAL(I-1)
      END DO
      CALL PAR_GDR1R('WAVE',NWAVE,WAVE,100.,100000.,.FALSE.,
     &     WAVE,STATUS)
      CALL CMP_PUT1R(LOC,'WAVE',NWAVE,WAVE,STATUS)
C
C     Unequal numbers of images and lines, then we need to specify
C     which image each wavelength corresponds to.
C     
      IF(NIMAGE.NE.NWAVE) THEN
         DO I = 1, NWAVE
            NWHICH(I) = MIN(I, NIMAGE)
         END DO
         IF(NIMAGE.GT.1) THEN
            CALL PAR_GDR1I('NWHICH',NWAVE,NWHICH,1,NIMAGE,.FALSE.,
     &           NWHICH,STATUS)
C
C     Check that every image is included
C
            NSKIP = 0
            DO I = 1, NIMAGE
               OK = .FALSE.
               DO J = 1, NWAVE
                  IF(NWHICH(J).EQ.I) OK = .TRUE.
               END DO
               IF(NSKIP.EQ.0 .AND. .NOT.OK) NSKIP = I
            END DO
            IF(.NOT.OK) THEN
               STATUS = SAI__ERROR
               CALL MSG_SETC('NSKIP',NSKIP)
               CALL ERR_REP(' ',
     &              'You have not specified which wavelength'//
     &              ' corresponds to image ^NSKIP', STATUS)
            END IF
            CALL CMP_PUT1I(LOC,'NWHICH',NWAVE,NWHICH,STATUS)
         END IF
         DO I = 1, NWAVE
            SCALE(I) = 1.
         END DO
         CALL PAR_GDR1R('SCALE',NWAVE,SCALE,1.E-20,1.E20,.FALSE.,
     &        SCALE,STATUS)
         CALL CMP_PUT1R(LOC,'SCALE',NWAVE,SCALE,STATUS)
      END IF
C
      CALL PAR_GDR0R('FWHM',2.*VPIX,1.E-2,5.E4,.FALSE.,FWHM,STATUS)
      CALL CMP_PUT0R(LOC,'FWHM',FWHM,STATUS)
      CALL PAR_GDR0R('GAMMA',0.,-1.E5,1.E5,.FALSE.,GAMMA,STATUS)
      CALL CMP_PUT0R(LOC,'GAMMA',GAMMA,STATUS)
      CALL PAR_GDR0I('NSUB',1,1,1001,.FALSE.,NSUB,STATUS)
      CALL CMP_PUT0I(LOC,'NSUB',NSUB,STATUS)
      CALL PAR_GDR0R('OFFSET',0.,-1.E8,1.E8,.FALSE.,OFFSET,STATUS)
      CALL CMP_PUT0R(LOC,'OFFSET',OFFSET,STATUS)
      CALL PAR_GDR0I('NDIV',5,1,100,.FALSE.,NDIV,STATUS)
      CALL CMP_PUT0I(LOC,'NDIV',NDIV,STATUS)
      CALL PAR_CHOIC('USE','Phases','Phases,HJDs',.FALSE.,USE,STATUS)
      IF(USE(1:1).EQ.'P') THEN
         CALL CMP_PUT0C(LOC,'USE','Phases',STATUS)
      ELSE
         CALL CMP_PUT0C(LOC,'USE','HJDs',STATUS)
      END IF
      CALL DAT_ANNUL(LOC, STATUS)
C
C     Get parameters defining image
C
      CALL PAR_GET0L('FANCY', FANCY, STATUS)
      IF(FANCY) THEN
         CALL PAR_GDR0R('XCEN',0.,-1.E5,1.E5,.FALSE.,XCEN,STATUS)
         CALL PAR_GDR0R('YCEN',0.,-1.E5,1.E5,.FALSE.,YCEN,STATUS)
         CALL PAR_GDR0R('WIDTH',200.,1.E-2,1.E5,.FALSE.,
     &        WIDTH,STATUS)
         CALL PAR_GDR0R('PGAUSS',1.,0.,1.E10,.FALSE.,PGAUSS,STATUS)
         CALL PAR_GDR0R('XSP',0.,-1.E5,1.E5,.FALSE.,XSP,STATUS)
         CALL PAR_GDR0R('YSP',0.,-1.E5,1.E5,.FALSE.,YSP,STATUS)
         CALL PAR_GDR0R('EXPON',0.,-100.,100.,.FALSE.,EXPON,STATUS)
         CALL PAR_GDR0R('VLOW',0.,0.,1.E5,.FALSE.,VLOW,STATUS)
         CALL PAR_GDR0R('VHIGH',3000.,VLOW,1.E5,.FALSE.,VHIGH,STATUS)
         CALL PAR_GDR0R('ANGLE',2.5,0.,180.,.FALSE.,ANGLE,STATUS)
         CALL PAR_GDR0R('CURVE',120.,-1.E5,1.E5,.FALSE.,CURVE,STATUS)
         CALL PAR_GDR0R('PPOWER',1.,0.,1.E10,.FALSE.,PPOWER,STATUS)
         CALL PAR_GDR0R('PSPIRAL',1.,0.,1.E10,.FALSE.,PSPIRAL,STATUS)
      ELSE
         CALL PAR_GDR0R('CONST',1.,0.,1.E20,.FALSE.,CONST,STATUS)
      END IF
C
C     Convert to pixel coordinates
C
      IF(STATUS.EQ.SAI__OK) THEN
         XCEN = XCEN/VPIX + REAL(1+NSIDE)/2.
         YCEN = YCEN/VPIX + REAL(1+NSIDE)/2.
         XSP  = XSP/VPIX  + REAL(1+NSIDE)/2.
         YSP  = YSP/VPIX  + REAL(1+NSIDE)/2.
         VLOW = VLOW/VPIX
         VHIGH= VHIGH/VPIX
         WIDTH= WIDTH/VPIX
      END IF
C     
C     Map data and axis arrays. Set axis labels and units
C     
      CALL NDF_MAP(IMAGE,'Data','_REAL','WRITE',DPTR,EL,STATUS)
      CALL NDF_AMAP(IMAGE,'Centre',1,'_REAL','WRITE',XPTR,EL,STATUS)
      CALL SET_AX(%VAL(CNF_PVAL(XPTR)), NSIDE, VPIX, STATUS)
      CALL NDF_AMAP(IMAGE,'Centre',2,'_REAL','WRITE',YPTR,EL,STATUS)
      CALL SET_AX(%VAL(CNF_PVAL(YPTR)), NSIDE, VPIX, STATUS)
      CALL NDF_ACPUT('km/s',IMAGE,'Units',1,STATUS)
      CALL NDF_ACPUT('km/s',IMAGE,'Units',2,STATUS)
      CALL NDF_ACPUT('K\\dX\\u',IMAGE,'Label',1,STATUS)
      CALL NDF_ACPUT('K\\dY\\u',IMAGE,'Label',2,STATUS)
C
C     Set image
C
      CALL SET_IMG(%VAL(CNF_PVAL(DPTR)), NX, NY, NZ, FANCY, XCEN, YCEN,
     &     WIDTH, PGAUSS, XSP, YSP, EXPON, VLOW, VHIGH, 
     &     ANGLE, CURVE, PPOWER, PSPIRAL, CONST, STATUS)
      CALL NDF_END(STATUS)
      RETURN
      END	

      SUBROUTINE SET_AX(AXIS, N, VPIX, STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER N, STATUS, I
      REAL AXIS(N), VPIX
      IF(STATUS.NE.SAI__OK) RETURN
      DO I = 1, N
        AXIS(I) = VPIX*(REAL(I)-REAL(1+N)/2.)
      END DO
      RETURN
      END

      SUBROUTINE SET_IMG(DATA,NX,NY,NZ,FANCY,XCEN,YCEN,WIDTH,
     &     PGAUSS,XSP,YSP,EXPON,VLOW,VHIGH,ANGLE,CURVE,
     &     PPOWER,PSPIRAL,CONST,STATUS)
*
* Subroutine sets up initial image. If FANCY is FALSE, the image is
* a constant with CONST per pixel. If FANCY is TRUE, a gaussian of FWHM
* WIDTH at XCEN, YCEN is added to a power law running from VLOW to VHIGH
* proportional to V**EXPON, with a two arm spiral pattern. Each spiral arm
* has an opening angle of ANGLE, curves by a total of CURVE degrees and 
* runs as an equiangular spiral from VLOW to VHIGH. 
* The power law and the spiral pattern are centred on XSP, YSP. The 
* gaussian peaks at PGAUSS, while the power law and spiral peak at
* PPOWER and PSPIRAL
*
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER STATUS, NX, NY, NZ, I, J, K, NRAD, N
      REAL DATA(NX,NY,NZ), CONST, XCEN, YCEN, WIDTH, PGAUSS
      REAL XSP, YSP, EXPON, VLOW, VHIGH, ANGLE, CURVE, PPOWER
      REAL PSPIRAL, PI, PMID, DX, DY, RMAX, RMIN, SFAC
      REAL FLUX, DELTHE, OPE, VY, YSQ, VX, VEL, THETA, PHI
      REAL DELTA, DIST, DLIM, ADD, FAC, RSQ, RAD, EFAC
      INTEGER MAXRAD
      PARAMETER (MAXRAD=1000)
      REAL PROF(MAXRAD)
      LOGICAL FANCY
C
      IF(STATUS.NE.SAI__OK) RETURN
      DO J = 1, NY
         DO I = 1, NX
            DATA(I,J,1) = 0.D0
         END DO
      END DO
      IF(FANCY) THEN
         PI   = 4.*ATAN(1.)
C
C     Compute desired radial profile. Power law exponent
C     EXPON from VLOW to VHIGH, with a linear cutoff outside these
C     radii.
C     
         PMID = REAL(1+NX)/2.
         IF(XSP.GT.PMID) THEN
            DX = XSP - 0.5
         ELSE
            DX = REAL(NX)+0.5-XSP
         END IF
         PMID = REAL(1+NY)/2.
         IF(YSP.GT.PMID) THEN
            DY = YSP - 0.5
         ELSE
            DY = REAL(NY)+0.5-YSP
         END IF
         RMAX = SQRT(DX**2+DY**2)
         RMIN = 0.
         NRAD = INT(2*RMAX)
         IF(NRAD.GT.MAXRAD) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Too many radii in SET_IMG', STATUS)
            RETURN
         END IF

         IF(EXPON.GT.0.) THEN
            SFAC = PPOWER/VHIGH**EXPON
         ELSE
            SFAC = PPOWER/VLOW**EXPON
         END IF
         DO N = 1, NRAD
            RAD = RMIN + (RMAX-RMIN)*REAL(N-1)/REAL(NRAD-1)
            IF(RAD.LT.VLOW) THEN
               FLUX = SFAC*(VLOW**EXPON)*(5.*RAD/VLOW-4.)
               PROF(N) = MAX(0., FLUX)
            ELSE IF(RAD.GT.VHIGH) THEN
               FLUX = SFAC*(VHIGH**EXPON)*(5.-4.*RAD/VHIGH)
               PROF(N) = MAX(0., FLUX)
            ELSE
               PROF(N) = SFAC*(RAD**EXPON)
            END IF
         END DO
C     
C     Add in radial power law
C
         IF(STATUS.NE.SAI__OK) RETURN
         CALL PROFADD(DATA,NX,NY,XSP,YSP,RMIN,RMAX,PROF,NRAD)
C
C     Now the spiral arms
C
         DELTHE = CURVE/180.*PI/LOG(VHIGH/VLOW)
         OPE    = ANGLE/360.*PI
         IF(EXPON.GT.0.) THEN
            SFAC = PSPIRAL/VHIGH**EXPON
         ELSE
            SFAC = PSPIRAL/VLOW**EXPON
         END IF
         DO J = 1, NY
            VY  = REAL(J)-YSP
            YSQ = VY**2
            DO I = 1, NX
               VX = REAL(I)-XSP
               VEL = SQRT(YSQ+VX**2)
               IF(VEL.GT.VLOW .AND. VEL.LT.VHIGH) THEN
                  THETA = DELTHE*LOG(VEL/VLOW)
                  PHI   = ATAN2(VY, VX)
                  DELTA = PHI-THETA
                  DELTA = DELTA - PI*REAL(NINT(DELTA/PI))
                  DIST  = ABS(DELTA*VEL)
                  DLIM  = OPE*VEL
                  IF(DIST.LT.DLIM+0.5) THEN
                     ADD = SFAC*(VEL**EXPON)
                     FAC = MAX(DLIM+0.5-DIST,1.)
                     DATA(I,J,1) = DATA(I,J,1) + FAC*ADD
                  END IF
               END IF
            END DO
         END DO
C
C     Add in gaussian
C
         EFAC = LOG(2.)/WIDTH**2
         DO J = 1, NY
            YSQ = (REAL(J)-YCEN)**2
            DO I = 1, NX
               RSQ = EFAC*((REAL(I)-XCEN)**2+YSQ)
               IF(RSQ.LT.70.)
     &              DATA(I,J,1)=DATA(I,J,1)+PGAUSS*EXP(-RSQ)
            END DO
         END DO
      ELSE 
         DO J = 1, NY
            DO I = 1, NX
               DATA(I,J,1) = CONST
            END DO
         END DO
      END IF
      DO K = 2, NZ
         DO J = 1, NY
            DO I = 1, NX
               DATA(I,J,K) = DATA(I,J,1)
            END DO
         END DO
      END DO
      RETURN
      END
