      SUBROUTINE KWLOC(DATA,NX,NY,RAD,VX,VY,AVE,IFAIL)
*
* Subroutine to locate point VX,VY in an image from which
* a circle radius RAD has no sinusoidal modulation. All
* dimensions in pixels. This point is effectively a measure
* of the centre of symmetry of the image. Method uses Powell's
* method to minise amplitude of sine wave.
*
* Input:
*
* R*4 DATA(NX,NY) -- Image
* I*4 NX, NY      -- Dimensions of image
* R*4 RAD         -- Radius in pixels
* R*4 VX, VY      -- Initial guess at centre of symmetry in pixel
*                    coordinates (i.e. centre in X is (1+NX)/2
*
* Output:
*
* R*4 VX, VY      -- Refined centre of symmetry
* R*4 AVE         -- The average value along the circle.
* I*4 IFAIL       -- Error return, 0 Ok, 1 internal buffer too small,
*                    2 too few points, 3 sine fit failed.
*
      INTEGER NX, NY, IFAIL
      REAL DATA(NX,NY), VX, VY, RAD, COV(6), P(2), AVE
      PARAMETER (MAXAZ=3000)
      PARAMETER (MAXIT=40)
      REAL WORK1(MAXAZ), WORK2(MAXAZ), WORK3(MAXAZ)
      REAL FGOOD, DV(2,2)
*
      DV(1,1) = 1.
      DV(2,1) = 0.
      DV(1,2) = 0.
      DV(2,2) = 1.
      FRET = FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,0.)
      ITER = 0
      VX0 = VX
      VY0 = VY
100   CONTINUE
      ITER = ITER + 1
      FP = FRET
      IBIG = 0
      DEL = 0.
      DO I = 1, 2
        DVX = DV(1,I)
        DVY = DV(2,I)
*
* Minimise along line
*
        CALL MINLIN(DATA,NX,NY,RAD,VX,VY,DVX,DVY,STEP,
     &  FP,IFAIL)
        IF(IFAIL.LT.0) THEN
          IFAIL = 0
          GOTO 999
        ELSE IF(IFAIL.GT.0) THEN
          RETURN
        END IF
        IF(ABS(FP-FRET).GT.DEL) THEN
          DEL = ABS(FP-FRET)
          IBIG = I
        END IF
      END DO
      VXN = 2.*VX - VX0
      VYN = 2.*VY - VY0
      DVX = VX-VX0
      DVY = VY-VY0
      VX0 = VX
      VY0 = VY
      FPT = FGOOD(DATA,NX,NY,RAD,VXN,VYN,DVX,DVY,0.)
      IF(FPT.GE.FP) GOTO 100
      T = 2.*(FP-2.*FRET+FPT)*(FP-FRET-DEL)**2-DEL*(FP-FPT)**2
      IF(T.GE.0.) GOTO 100
      CALL MINLIN(DATA,NX,NY,RAD,VX,VY,DVX,DVY,STEP,
     &FRET,IFAIL)
      IF(IFAIL.LT.0) THEN
        IFAIL = 0
        GOTO 999
      ELSE IF(IFAIL.GT.0) THEN
        RETURN
      END IF
      DV(1,IBIG) = DVX
      DV(2,IBIG) = DVY
      IF(ITER.GE.MAXIT) THEN
        IFAIL = 0
        GOTO 999
      END IF
      GOTO 100
999   CALL FSINE(DATA,NX,NY,RAD,VX,VY,AVE,A,B,IFAIL)
      RETURN
      END

      SUBROUTINE MINLIN(DATA,NX,NY,RAD,VX,VY,DVX,DVY,STEP,
     &FP,IFAIL)
*
* Subroutine carries out line minimisation
* Returns with new position VX, VY
*
      INTEGER NX, NY
      REAL DATA(NX,NY), RAD, VX, VY, DVX, DVY, STEP
      REAL FGOOD, AVE
      PARAMETER (R=0.61803399,C=1.-R,TOL=0.01)
      DATA ACC/2.E-4/
*
* Build up 3 points bracketing a minimum in the
* fitted amplitude
*
      AX = 0.
      CALL FSINE(DATA,NX,NY,RAD,VX,VY,AVE,A,B,IFAIL)
      AC = A*A + B*B
      IF(IFAIL.NE.0) RETURN
*
* This is the only return point if all works
*
      IF(SQRT(AC).LT.ACC*ABS(AVE)) THEN
        IFAIL = -1
        RETURN
      END IF
*
* Set first point and direction for one-D search
*
      STEP = REAL(MIN(NX,NY))/20.
*
* Second point 
*
      BX = STEP
      BC = FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,BX)
      IF(BC.GT.AC) THEN
        STEP = - STEP
        ICASE = 1
      ELSE
        STEP = 1.5*STEP
        ICASE = 2
      END IF
*
* Third point. Need to bounce around until middle point
* is lower than other two
*
200   CONTINUE
      CX = STEP
      CC = FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,STEP)
      IF(ICASE.EQ.2) THEN
        IF(CC.LT.BC) THEN
          AX = BX
          AC = BC
          BX = CX
          BC = CC
          STEP = 1.5*STEP
          GOTO 200
        END IF
      ELSE
        IF(CC.GT.AC) THEN
          SC = AC
          SX = AX
          AC = CC
          AX = CX
          CC = BC
          CX = BX
          BC = SC
          BX = SX
        ELSE
          BC = CC 
          BX = CX
          STEP = 1.5*STEP
          ICASE = 2
          GOTO 200
        END IF
      END IF
*
* Now have 3 points with central one lower
* then the other two
*
      X0=AX
      X3=CX
      IF(ABS(CX-BX).GT.ABS(BX-AX))THEN
        X1=BX
        X2=BX+C*(CX-BX)
      ELSE
        X2=BX
        X1=BX-C*(BX-AX)
      ENDIF
      F1=FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,X1)
      F2=FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,X2)
1     IF(ABS(X3-X0).GT.TOL)THEN
        IF(F2.LT.F1)THEN
          X0=X1
          X1=X2
          X2=R*X1+C*X3
          F0=F1
          F1=F2
          F2=FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,X2)
        ELSE
          X3=X2
          X2=X1
          X1=R*X2+C*X0
          F3=F2
          F2=F1
          F1=FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,X1)
        ENDIF
        GOTO 1
      ENDIF
      IF(F1.LT.F2)THEN
        VX = VX + DVX*X1
        VY = VY + DVY*X1
        STEP = X1
        FP = F1
      ELSE
        VX = VX + DVX*X2
        VY = VY + DVY*X2
        STEP = X2
        FP = F2
      ENDIF
      RETURN
      END

      REAL FUNCTION FGOOD(DATA,NX,NY,RAD,VX,VY,DVX,DVY,STEP)
*
* Workhorse routine to compute mean, cosine and sine terms
* at a radius RAD
*
      INTEGER NX, NY
      REAL DATA(NX,NY), VX, VY, RAD, DVX, DVY
*
      VXC = VX + STEP*DVX
      VYC = VY + STEP*DVY
      CALL FSINE(DATA,NX,NY,RAD,VXC,VYC,AVE,A,B,IFAIL)
      FGOOD = A*A+B*B
      RETURN
      END

      SUBROUTINE FSINE(DATA,NX,NY,RAD,VX,VY,AVE,A,B,IFAIL)
*
* Workhorse routine to compute mean, cosine and sine terms
* at a radius RAD
*
      INTEGER NX, NY, IFAIL
      REAL DATA(NX,NY), VX, VY, RAD, COV(6), AVE, A, B
      PARAMETER (MAXAZ=3000)
      REAL WORK1(MAXAZ), WORK2(MAXAZ), WORK3(MAXAZ)
*
      TWOPI = 8.*ATAN(1.)
      NAZ = MAX(16, NINT(2.*TWOPI*RAD))
      IF(NAZ.GT.MAXAZ) THEN
        IFAIL = 1
        RETURN
      END IF
*
* Load up work arrays
*
      N = 0
      DO I = 1, NAZ
        PHASE = REAL(I-1)/REAL(NAZ)
        THETA = TWOPI*PHASE
        X = VX+RAD*COS(THETA)
        Y = VY+RAD*SIN(THETA)
        IF(X.GT.1. .AND. X.LT.REAL(NX) .AND.
     &     Y.GT.1. .AND. Y.LT.REAL(NY)) THEN
          N = N + 1
          WORK1(N) = PHASE
*
* Bi-linear interpolation
*
          I1 = INT(X)
          I2 = I1 + 1
          J1 = INT(Y)
          J2 = J1 + 1
          T = X-REAL(I1)
          U = Y-REAL(J1)
          WORK2(N) = (1.-T)*((1.-U)*DATA(I1,J1)+U*DATA(I1,J2))+
     &               T*((1.-U)*DATA(I2,J1)+U*DATA(I2,J2))
          WORK3(N) = 1.
        END IF
      END DO
      IF(N.LT.10) THEN
        IFAIL = 2
        RETURN
      END IF
*
* Fit sine wave.
*
      CALL SINFIT(WORK1, WORK2, WORK3, N, 1., AVE, AMP, PHI0, 
     &COV, NP, F, IFAIL)
      IF(IFAIL.NE.0.) THEN
        IFAIL = 3
        RETURN
      END IF
      A = AMP*SIN(TWOPI*PHI0)
      B = AMP*COS(TWOPI*PHI0)
      IFAIL = 0
      RETURN
      END
