*COMDAT
*
* COMDAT -- computes data equivalent to a particular image. This can be
*           subtracted from real data to give data residuals. comdat is
*           also useful for simulating data. comdat uses exactly the same
*           image to data routine as optscl and memit and should therefore
*           give the same chi**2 (for the same image and data). Noise can
*           optionally be added.
*
*
* Parameters:
*
*   IMAGE     -- The image file to load
*   TEMPLATE  -- TRUE if template data file is to be used otherwise the user
*                must define pixels sizes etc
*   If TEMPLATE
*
*     DATA      -- Data file containing example format data
*
*   Else
*
*     NPIX      -- Number of pixels/spectrum        
*     NSPEC     -- Number of spectra
*     VDATA     -- Velocity km/s/pixel
*     WCEN      -- Central wavelength (Angstroms)
*     WIDTH     -- Phase width of exposures
*
*   OUTPUT    -- Calculated data file name
*   NADD      -- TRUE if noise is to added to data
*
*   If NADD
*
*     CONT      -- Effective continuum for noise computation
*     NOISE     -- Signal-to-noise in effective continuum
*     SEED      -- Seed integer to start noise generator if NADD
*
*COMDAT 
      SUBROUTINE COMDAT(STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INCLUDE 'DAT_PAR'
      INCLUDE 'doppler.inc'
      INCLUDE 'CNF_PAR'
      INTEGER STATUS, DIM(3), NDIM, IMAGE, EL, I
     :, DATA
      INTEGER LBND(2), UBND(2), DWPTR, PPTR, HPTR, PLACE
      INTEGER PHASES, WIDTHS, PWPTR, OUTPUT, SEED
      INTEGER CEPTR, CDPTR, IPTR, DPTR, EPTR
      REAL VDATA, WIDTH, VLIGHT, CGS, WCEN, CONT, NOISE
      REAL CHISQ, CHISQRD
      DOUBLE PRECISION PERIOD, HJD0
      CHARACTER*(DAT__SZLOC) LOC
      CHARACTER*16 USE
      LOGICAL SAMECI, TEMPLATE, DERROR, NADD
C
      IF(STATUS.NE.SAI__OK) RETURN
      CALL NDF_BEGIN
C
C     Open image file 
C
      CALL NDF_ASSOC('IMAGE','READ',IMAGE, STATUS)
C
      CALL NDF_DIM(IMAGE,3,DIM,NDIM,STATUS)
      IF(STATUS.EQ.SAI__OK) THEN
         IF(NDIM.EQ.1) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ',
     &           'Images must be 2 or 3D', STATUS)
         ELSE IF(DIM(1).NE.DIM(2)) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Image(s) not square', STATUS)
         ELSE
            NSIDE  = DIM(1)
            NIMAGE = DIM(3)
         END IF
      END IF
C
C     Get header items
C
      CALL NDF_XLOC(IMAGE,'DOPPLER_MAP','READ',LOC,STATUS)
C
      CALL CMP_GET0R(LOC,'VPIX',VPIX,STATUS)
      CALL CMP_GET1R(LOC, 'WAVE',MAXWV,WAVE,NWAVE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. NWAVE.LT.NIMAGE) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','Too few wavelengths found', STATUS)
      END IF
C     
C     Read scale factors and correspondence arrays.
C     
      IF(STATUS.EQ.SAI__OK .AND. NIMAGE.NE.NWAVE) THEN
         IF(NIMAGE.GT.1) THEN
            CALL CMP_GET1I(LOC, 'NWHICH',NWAVE,NWHICH,EL,STATUS)
            IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
               STATUS = SAI__ERROR
               CALL ERR_REP(' ','NWHICH/WAVE: conflicting sizes.',
     &              STATUS)
            END IF
         ELSE
            DO I = 1, NWAVE
               NWHICH(I) = 1
            END DO
         END IF
         CALL CMP_GET1R(LOC, 'SCALE',NWAVE,SCALE,EL,STATUS)
         IF(STATUS.EQ.SAI__OK .AND. EL.NE.NWAVE) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','SCALE/WAVE: conflicting sizes.',
     &           STATUS)
         END IF
      ELSE IF(STATUS.EQ.SAI__OK) THEN
         DO I = 1, NWAVE
            SCALE(I)  = 1.
            NWHICH(I) = I
         END DO
      END IF
C     
      CALL CMP_GET0R(LOC,'FWHM',FWHM,STATUS)
      CALL CMP_GET0R(LOC,'GAMMA',GAMMA,STATUS)
      CALL CMP_GET0R(LOC,'OFFSET',OFFSET,STATUS)
      CALL CMP_GET0I(LOC,'NDIV',NDIV,STATUS)
      CALL CMP_GET0C(LOC,'USE',USE,STATUS)
      IF(STATUS.EQ.SAI__OK .AND. (.NOT.SAMECI(USE,'Phases') 
     &     .AND. .NOT.SAMECI(USE,'HJDs')) ) THEN
         STATUS = SAI__ERROR
         CALL ERR_REP(' ','USE has unrecognised value',STATUS)
      END IF
      CALL CMP_GET0I(LOC,'NSUB',NSUB,STATUS)
      CALL DAT_ANNUL(LOC, STATUS)
C     
C     Open data file to provide phases, wavelength scale etc
C     for computed data
C     
      CALL PAR_GET0L('TEMPLATE', TEMPLATE, STATUS)
      IF(TEMPLATE) THEN
         CALL NDF_ASSOC('DATA', 'READ', DATA, STATUS)
         CALL NDF_DIM(DATA,2,DIM,NDIM,STATUS)
         NPIX  = DIM(1)
         NSPEC = DIM(2)
         CALL NDF_STATE(DATA,'VARIANCE',DERROR,STATUS)
C     
C     Read header
C     
         CALL NDF_XLOC(DATA,'DOPPLER_DATA','READ',LOC,STATUS)
C     
         CALL CMP_GET0I(LOC,'NARC',NARC,STATUS)
         IF(STATUS.EQ.SAI__OK .AND. NARC.NE.-2) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ',
     &           'Doppler imaging programs only use'//
     &           ' log wavelength scales.', STATUS)
         END IF
         CALL CMP_GET1D(LOC, 'ARC', 2, ARC, EL, STATUS)
         IF(STATUS.EQ.SAI__OK .AND. EL.NE.2) THEN
            STATUS = SAI__ERROR
            CALL ERR_REP(' ','Could not read 2 arc coefficients',
     &           STATUS)
         END IF
C     
C     Get orbital period and dwells
C     
         CALL CMP_GET0D(LOC,'PERIOD',PERIOD,STATUS)
         CALL CMP_MAPV(LOC, 'DWELL', '_REAL', 'READ', DWPTR, 
     &        EL, STATUS)
C     
C     Get phase either from phases array or HJDs
C     
         IF(SAMECI(USE,'Phases')) THEN
            CALL CMP_MAPV(LOC,'PHASES','_DOUBLE','READ',PPTR, 
     &           EL, STATUS)
         ELSE IF(SAMECI(USE,'HJDs')) THEN
            CALL CMP_GET0D(LOC,'HJD0',HJD0,STATUS)
            CALL CMP_MAPV(LOC, 'HJD', '_DOUBLE', 'READ', HPTR, 
     &           EL, STATUS)
            CALL NDF_TEMP(PLACE,STATUS)
            CALL NDF_NEW('_DOUBLE',1,1,NSPEC,PLACE,PHASES,STATUS)
            CALL NDF_MAP(PHASES,'Data','_DOUBLE','WRITE',PPTR,
     &           EL,STATUS)
            CALL PHCOMP(%VAL(CNF_PVAL(HPTR)), NSPEC, 
     :                  %VAL(CNF_PVAL(PPTR)),
     &           HJD0, PERIOD,STATUS)
         END IF
         CALL DAT_ANNUL(LOC, STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_REAL',1,1,NSPEC,PLACE,WIDTHS,STATUS)
         CALL NDF_MAP(WIDTHS,'Data','_REAL','WRITE',PWPTR,
     &        EL,STATUS)
         CALL PWCOMP(%VAL(CNF_PVAL(DWPTR)),NSPEC,%VAL(CNF_PVAL(PWPTR)),
     :               PERIOD,STATUS)
         CALL NDF_PROP(DATA,' ','OUTPUT', OUTPUT, STATUS)
      ELSE 
C
C     Get enough parameters to fake up data 
C
         CALL PAR_GDR0I('NPIX',NSIDE,1,10000,.FALSE.,NPIX,STATUS)
         CALL PAR_GDR0I('NSPEC',50,1,10000,.FALSE.,NSPEC,STATUS)
         CALL PAR_GDR0R('VDATA',VPIX,1.E-4,1.E4,.FALSE.,VDATA,STATUS)
         CALL PAR_GDR0R('WCEN',WAVE(1),1.E-4,1.E6,.FALSE.,WCEN,STATUS)
         CALL PAR_GDR0R('WIDTH',0.1,0.,1.,.FALSE.,WIDTH,STATUS)
         IF(STATUS.EQ.SAI__OK) THEN
            VLIGHT = CGS('C')/1.E5
            ARC(2) = DBLE(NPIX)*VDATA/VLIGHT
            ARC(1) = LOG(WCEN)-ARC(2)*DBLE(NPIX+1)/2.D0/DBLE(NPIX)
            NARC   = -2
            HJD0   = 2448600.
            PERIOD = 0.1
            LBND(1) = 1
            UBND(1) = NPIX
            IF(NSPEC.GT.1) THEN
               NDIM    = 2
               LBND(2) = 1
               UBND(2) = NSPEC
            ELSE
               NDIM = 1
            END IF
            DERROR = .FALSE.
         END IF
C
C     Create structure OUTPUT
C
         CALL NDF_CREAT('OUTPUT','_REAL',NDIM,LBND,UBND,OUTPUT,STATUS)
C     
C     Create doppler data extension
C     
         CALL NDF_XNEW(OUTPUT,'DOPPLER_DATA','Ext',0,0,LOC,STATUS)
         CALL CMP_MOD(LOC, 'HJD0',  '_DOUBLE', 0, 0, STATUS)
         CALL CMP_PUT0D(LOC,'HJD0',HJD0,STATUS)
         CALL CMP_MOD(LOC, 'PERIOD',  '_DOUBLE', 0, 0, STATUS)
         CALL CMP_PUT0D(LOC,'PERIOD',PERIOD,STATUS)
         CALL CMP_MOD(LOC, 'NARC',  '_INTEGER', 0, 0, STATUS)
         CALL CMP_PUT0I(LOC,'NARC',NARC,STATUS)
         CALL CMP_MOD(LOC, 'ARC',  '_DOUBLE', 1, ABS(NARC), STATUS)
         CALL CMP_PUT1D(LOC,'ARC',ABS(NARC),ARC,STATUS)
C
         CALL CMP_MOD(LOC, 'HJD',  '_DOUBLE', 1, NSPEC, STATUS)
         CALL CMP_MAPV(LOC, 'HJD', '_DOUBLE', 'WRITE', HPTR, 
     &        EL, STATUS)
         CALL CMP_MOD(LOC, 'PHASES',  '_DOUBLE', 1, NSPEC, STATUS)
         CALL CMP_MAPV(LOC, 'PHASES', '_DOUBLE', 'WRITE', PPTR, 
     &        EL, STATUS)
         CALL CMP_MOD(LOC, 'DWELL',  '_REAL', 1, NSPEC, STATUS)
         CALL CMP_MAPV(LOC, 'DWELL', '_REAL', 'WRITE', DWPTR, 
     &        EL, STATUS)
         CALL DAT_ANNUL(LOC, STATUS)
         CALL NDF_TEMP(PLACE,STATUS)
         CALL NDF_NEW('_REAL',1,1,NSPEC,PLACE,WIDTHS,STATUS)
         CALL NDF_MAP(WIDTHS,'Data','_REAL','WRITE',PWPTR,
     &        EL,STATUS)
C
C     Write out
C
         DO I = 1, NSPEC
            CALL DMOVE(DBLE(I)/DBLE(NSPEC), 
     &           %VAL(CNF_PVAL(PPTR)), I, NSPEC,STATUS)
            CALL DMOVE(HJD0 + PERIOD*DBLE(I)/DBLE(NSPEC), 
     &           %VAL(CNF_PVAL(HPTR)), I, NSPEC,STATUS)
            CALL RMOVE(WIDTH*PERIOD*24.*3600., 
     &           %VAL(CNF_PVAL(DWPTR)), I, NSPEC,STATUS)
            CALL RMOVE(WIDTH, %VAL(CNF_PVAL(PWPTR)), I, NSPEC,STATUS)
        END DO
      END IF
C
C     Should noise be added or not?, then map files.
C     
      CALL PAR_GET0L('NADD',NADD,STATUS)
      IF(NADD) THEN
         IF(.NOT.DERROR) THEN
            CALL PAR_GDR0R('CONT',1.,0.,1.E20,.FALSE.,CONT,STATUS)
            CALL PAR_GDR0R('NOISE',10.,1.E-10,1.E20,.FALSE.,
     &           NOISE,STATUS)
         END IF
         CALL PAR_GDR0I('SEED',12345,1,9999999,.FALSE.,SEED,STATUS)
         SEED = - SEED
         CALL NDF_MAP(OUTPUT,'Errors','_REAL','WRITE',CEPTR,EL,STATUS)
      END IF
      CALL NDF_MAP(OUTPUT,'Data','_REAL','WRITE',CDPTR,EL,STATUS)
      CALL NDF_MAP(IMAGE,'Data','_REAL','READ',IPTR,EL,STATUS)
C
C     Compute data
C
      OPFIR = .TRUE.
      CALL OP(%VAL(CNF_PVAL(IPTR)),%VAL(CNF_PVAL(CDPTR)),
     :        %VAL(CNF_PVAL(PPTR)),%VAL(CNF_PVAL(PWPTR)),
     &     STATUS)
C
C     Compute and report Chi**2 if possible
C
      IF(DERROR) THEN
         CALL NDF_MAP(DATA,'Data','_REAL','READ',DPTR,EL,STATUS)
         CALL NDF_MAP(DATA,'Errors','_REAL','READ',
     &        EPTR,EL,STATUS)
         CHISQ = CHISQRD(%VAL(CNF_PVAL(DPTR)),%VAL(CNF_PVAL(CDPTR)),
     :                   %VAL(CNF_PVAL(EPTR)),
     &        NSPEC*NPIX,STATUS)
         CALL MSG_SETR('CHISQ',CHISQ)
         CALL MSG_OUT(' ','Chi-squared = ^CHISQ',STATUS)
      END IF
C
C     Add noise if wanted.
C
      IF(NADD) THEN
         IF(DERROR) THEN
            CALL ADD_NS1(%VAL(CNF_PVAL(CDPTR)), %VAL(CNF_PVAL(EPTR)),
     &           %VAL(CNF_PVAL(CEPTR)), NPIX, NSPEC, SEED, STATUS)
         ELSE
            CALL ADD_NS2(%VAL(CNF_PVAL(CDPTR)), %VAL(CNF_PVAL(CEPTR)),
     &           CONT, NOISE, NPIX, NSPEC, SEED, STATUS)
         END IF
      END IF
      CALL NDF_END(STATUS)
      RETURN
      END	

      SUBROUTINE ADD_NS1(DATA, ERRORS, EOUT, NX, NY, SEED, STATUS)
C
C     Adds noise according to 1-sigma error bars in ERRORS using
C     GAUSS1. SEED must be negative to start with
C
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER STATUS, NX, NY, SEED, IX, IY
      REAL DATA(NX, NY), ERRORS(NX, NY), EOUT(NX, NY), GAUSS1
C
      IF(STATUS.NE.SAI__OK) RETURN
      DO IY = 1, NY
         DO IX = 1, NX
            DATA(IX,IY) = GAUSS1(DATA(IX,IY),ERRORS(IX,IY),SEED)
            EOUT(IX,IY) = ERRORS(IX,IY)
        END DO
      END DO
      RETURN
      END

      SUBROUTINE ADD_NS2(DATA,ERRORS,CONT,NOISE,NX,NY,SEED,STATUS)
C
C     Adds noise according to parameters CONT, NOISE. Stores noise
C     in array ERRORS
C     1-sigma error defined as SQRT(CONT*(CONT + DATA))/NOISE 
C     GAUSS1 used to fake gaussian noise. SEED must be 
C     negative to start with
C     
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER NX, NY, STATUS, SEED, IX, IY
      REAL DATA(NX,NY), ERRORS(NX,NY), CONT, NOISE, GAUSS1, SIGMA
C     
      IF(STATUS.NE.SAI__OK) RETURN
      DO IY = 1, NY
         DO IX = 1, NX
            SIGMA = SQRT(MAX(1.E-30,CONT*(CONT+DATA(IX,IY))))/NOISE
            ERRORS(IX,IY) = SIGMA
            DATA(IX,IY) = GAUSS1(DATA(IX,IY),SIGMA,SEED)
         END DO
      END DO
      RETURN
      END

      SUBROUTINE DMOVE(SOURCE,DEST,I,N,STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER I, N, STATUS
      DOUBLE PRECISION SOURCE, DEST(N)
      IF(STATUS.NE.SAI__OK) RETURN
      DEST(I) = SOURCE
      RETURN
      END

      SUBROUTINE RMOVE( SOURCE, DEST, I, N, STATUS)
      IMPLICIT NONE
      INCLUDE 'SAE_PAR'
      INTEGER I, N, STATUS
      REAL SOURCE, DEST(N)
      IF(STATUS.NE.SAI__OK) RETURN
      DEST(I) = SOURCE
      RETURN
      END
